/********************************************************************************
*                                                                               *
*                  Generic socket I/O handling                                  *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#ifndef FXSOCKET_H
#define FXSOCKET_H

#ifndef FXIOHANDLE_H
#include "FXIOHandle.h"
#endif
namespace FXEX {
class FXResolver;

/**
 * This is the base class of the socket server and socket client classes.
 * It can be instantiated on its own if the user wants to have more control over
 * the connection process.
 * 
 * It is intend to be used in these scenarios:
 * a) Application creates an FXSocket object and calls connect(...) to connect to the remote host.
 *    All socket reads are forwarded to the application using a SEL_COMMAND with data of
 *    FXSocketData*
 * b) Applicaiton creates an FXSocket object and calls listen(...) to listen for any
 *    incoming connections.   The application must handle the SEL_OPENED message, with
 *    data FXSocket*, and manage a list of incoming connections.
 * c) App creats an FXSocketClient which connect to a socket server.
 *
 * Important: if you are using this class on Win32, then the FXInputHandle is in reality
 *            a struct win32socket_t* - this is the only way I could think of to implement
 *            it reasonably cleanly...  As such you shouldn't directly manipulate the
 *            FXInputHandle...  In most cases, you wont need to care about this...
 *
 * For an example, see the socketserver/socketclient demos in the tests directory
 */
class FXAPI FXSocket : public FXIOHandle {
  FXDECLARE (FXSocket)

  protected:
    FXString         hostname;          // hostname of remote connection or "localhost" for listener
    FXint            port;              // socket port
    FXSocketFamily   family;            // socket family (internet/unix)
    FXSocketType     type;              // socket type (stream/datagram/raw)
    FXResolver      *resolver;          // asynchronous hostname resolver

  private:
  
    /// helper used to send data
    FXbool writeAgain();

    /// helper to add this socket to the FXApp watch list
    virtual FXbool addSocketInput();

    /// setup various socket options
    FXbool setSocketOptions();

  public:
    enum {
      ID_SOCKET=FXIOHandle::ID_LAST,
      ID_CONNECT,
      ID_RESOLVE,
      ID_LAST
      };

  public:
    long onRead(FXObject*,FXSelector,void*);
    long onExcept(FXObject*,FXSelector,void*);
    long onConnect(FXObject*,FXSelector,void*);
    long onOpened(FXObject*,FXSelector,void*);
    long onClose(FXObject*,FXSelector,void*);
    long onConnectResolved(FXObject*,FXSelector,void*);

  protected:

    /// for serialisation
    FXSocket();

    /// helper routine - returns the filesystem path of the local domain socket
    FXString getLocalDomainPath();

    /// close the underlying operating system handle
    virtual void closehandle();

    /// instantiates this class
    virtual FXSocket* newInstance(FXInputHandle h);

    /// Create a socket using a specific port
    FXSocket(FXApp *a,FXint port,FXObject *tgt=NULL,FXSelector sel=0,
             FXSocketFamily family=FXSocketFamilyInet, FXSocketType type=FXSocketTypeStream);

    /// Create a socket using a specific service
    FXSocket(FXApp *a,const FXString& service,FXObject *tgt=NULL,FXSelector sel=0,
             FXSocketFamily family=FXSocketFamilyInet, FXSocketType type=FXSocketTypeStream);

  public:

    /// Use an existing socket, adding it to the FXApp event loop
    FXSocket(FXInputHandle s,FXApp *a,FXObject *tgt=NULL,FXSelector sel=0);

    /// create resource
    virtual void create();

    /// return the file descriptor / event handle
    virtual FXInputHandle getHandle();

    /**
     * Set the hostname to a different one; you cannot change this once the connection is
     * open - doing so just silently fails ;-)
     */
    void setHostname(const FXString& host);

    /**
     * Gets the hostname, either the name you specified (for an outbound connection),
     * or the hostname of the remote end _if it can be determined_ for inbound connection.
     */
    FXString getHostname() { return hostname; }

    /**
     * Set the port to a different one; you cannot change this once the connection is
     * open - doing so just silently fails ;-)
     */
    void setPort(FXint port);

    /// gets the port
    FXint getPort() { return port; }

    /// set the socket family
    void setSocketFamily(FXSocketFamily family);

    /// return the family used in this socket
    FXSocketFamily getSocketFamily() { return family; }

    /// set the socket type
    void setSocketType(FXSocketType type);

    /// return the scoket type for this socket
    FXSocketType getSocketType() { return type; }

    /// Open-up/create a socket
    virtual FXbool open();

    /// Close down the socket
    virtual void close();

    /**
     * Send a file down the socket.
     * Since most operating systems provide some form of native implementation for doing
     * this in an efficient manner, we provide an interface here.  This call could
     * be implemented better, by sending the file in another thread; avoids the need
     * for the GUI to block.
     */
    FXbool sendfile(const FXString& file);

    /**
     * Get some data from the socket
     * - a wrapper for the recv() syscall
     */
    FXint read(FXuchar *data,FXint len,FXbool outOfBand=FALSE);

    /**
     * Send some data over the socket - can send out of band data if required
     * - a wrapper for the send() syscall
     */
    FXint write(FXuchar *data,FXint n,FXbool outOfBand=FALSE);

    /// read the socket and generate FOX events
    FXbool readData(FXbool outOfBand=FALSE);

    /**
     * Connect socket to this host
     * Note: This can return FALSE for a 'not-yet-connected' socket
     *       ie the app should handle the SEL_OPENED event
     */
    FXbool connect(const FXString& host);

    /// setup socket to listen for incoming connections
    FXbool listen(FXint concurrent);

    /// accept an incoming socket connection
    FXSocket* accept();

    /// get the hostname of the remote machine
    static FXString getRemoteHostname(FXInputHandle s);

    /// get the port number of the remote port
    static FXint getRemotePort(FXInputHandle s);

    /// get socket family for a socket
    static FXSocketFamily getSocketFamily(FXInputHandle s);

    /// get the socket type (stream, datagram, raw...)
    static FXSocketType getSocketType(FXInputHandle s);
    
    /// duplicate this socket onto another socket - defaults to OS assigned descriptor
    FXSocket* duplicate(FXInputHandle newHandle=INVALID_HANDLE);

    /// save object to stream
    virtual void save(FXStream& store) const;

    /// load object from stream
    virtual void load(FXStream& store);

    /// dtor
    virtual ~FXSocket();
  };

} // namespace FXEX
#endif // FXSOCKET_H
