/********************************************************************************
*                                                                               *
*                  Child process object/interface                               *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#ifndef FXPROCESS_H
#define FXPROCESS_H

#include "FXBaseObject.h"
namespace FXEX {
class FXProcessManager;

/*
 * FXProcess handles starting/stopping and communicating with child processes.
 *
 * If no command string is supplied, the current process is copied.
 * If you supplied a command string, that program is executed.
 *
 * The stdio streams, stdout, stderr of the child process, are connected to this
 * object, which then forwards the events to the app, via FOX events.  Any data written
 * to this object, is forwarded to the stdin of the child process, which ends up as a FOX
 * IOHandle read event.
 */
class FXAPI FXProcess : public FXBaseObject {
  FXDECLARE(FXProcess)

private:
  FXint         exitcode;   // exit code of the child process
  FXint         code;       // status of this child process handler
  FXint         pid;        // process ID of process (-1 if no process)
  FXbool	child;      // indicates if this instance is a child or parent
  FXString      command;    // command string
  FXStringList  arguments;  // arguments to the command
  FXIOHandle   *in;         // stdin of the child process
  FXIOHandle   *out;        // stdout of the child process
  FXIOHandle   *err;        // stderr of the child process

protected:
  /// serialisation
  FXProcess();

  /// Return handle to the process manager
  FXProcessManager* getProcessManager();

public:
  enum {
    ID_PROCESS=FXBaseObject::ID_LAST,
    ID_STDIN,
    ID_STDOUT,
    ID_STDERR,
    ID_LAST
    };

public:
  long onChildDeath(FXObject*,FXSelector,void*);
  long onTerminate(FXObject*,FXSelector,void*);
  long onStdin(FXObject*,FXSelector,void*);
  long onStdout(FXObject*,FXSelector,void*);
  long onStderr(FXObject*,FXSelector,void*);

public:

  /// Create a child instance of this process
  FXProcess(FXApp* a,FXObject* tgt=NULL,FXSelector sel=0);

  /// Create a child process which starts an external program
  FXProcess(FXApp* a,const FXString& cmd,const FXStringList& args,FXObject* tgt=NULL,FXSelector sel=0);

  /// start the child process
  FXbool start();

  /// Get PID of started process (-1 otherwise)
  FXint PID() const { return pid; }

  /// return the exit status
  FXint exitStatus() const { return exitcode; }

  /// return the status
  FXint status() const { return code; }

  /// Test if child process is running.
  FXbool isRunning() const;

  /// Show if this instance a child process
  FXbool isChild() const { return child; }

  /// Try to terminate the child process.
  FXbool terminate(FXint code=15);

  /**
   * Either:
   * - when in parent context
   *   - send some data to the child process via this child's stdin channel (in parent context)
   *   - outOfBand is ignored
   * - when in child context
   *   - send some data to parent via the child's stdout channel
   *     (this is equivalent to writing to stdout)
   *   - if outOfBand=TRUE, write data to child's error channel
   *     (this is equivalent to writing to stderr, such as using fxmessage())
   */
  void write(FXuchar *data,FXuint size,FXbool outOfBand=FALSE);

  /**
   * Either:
   * - when in parent context
   *   - read data from stdout of child
   *   - if outOfBand=TRUE, read data from stderr of child
   * - when in child context
   *   - read data from write() of parent
   *   - outOfBand is ignored
   */
  FXint read(FXuchar* data,FXuint size,FXbool outOfBand=FALSE);

  /// send some text to the child process
  void setText(const FXString& message);

  /// read some text from stdout from child
  FXString getText();

  /// destructor
  virtual ~FXProcess();
  };

} // namespace FXEX
#endif // FXPROCESS
