/********************************************************************************
*                                                                               *
*                      Database field handling                                  *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.       All Rights Reserved.            *
* Copyright (C) 2003 by Giancarlo Formicuccia.  All Rights Reserved.            *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#ifndef FXDATABASEFIELD_H
#define FXDATABASEFIELD_H


#include "FXBaseObject.h"
#include "FXVariant.h"
#include <fox/FXObjectList.h>

namespace FXEX {

class FXDatabaseInterface;
class FXDatabaseQuery;
class FXDatabaseField;



/* State of current FXDatabaseQuery */
typedef enum {
  rsInvalid = -1,   /* Invalid (i.e. FXDatabaseInterface shut down) */
  rsClose,          /* Not open */
  rsRead,           /* Open and in read state */
  rsAddNew,         /* Open and in add-new state */
  rsModify,         /* Open and modify state */
  rsDelete          /* unused */
} FXrsState;


/* Type of event */
typedef enum {
  dbRefresh,        /* Field value may have changed */
  dbUpdate,         /* Update successfully performed */
  dbCancelUpdate,   /* Update aborted */
  dbAddNew,         /* Entering add-new state */
  dbEdit,           /* Entering modify state */
  dbDelete,         /* Record has been deleted */
  dbConnect,        /* FXDatabaseInterface notified a new connection */
  dbDisconnect,     /* FXDatabaseInterface shut down */
  dbOpen,           /* FXDatabaseQuery just open */
  dbClose,          /* FXDatabaseQuery closed */
  dbDestroy,        /* FXDatabaseQuery is going to be destroyed */
  dbAskData,        /* FXDatabaseField is asking his target for (modified?) data */
} FXdbEvType;


/* FXDatabase* event (aka void * field of message handlers) */
typedef struct {
  FXdbEvType type;            /* Event type */
  FXrsState state;            /* Current FXDatabaseQuery state */
  FXDatabaseInterface *dbi;   /* !=NULL if FXDatabaseInterface is in event chain */
  FXDatabaseQuery *dbq;       /* !=NULL if FXDatabaseQuery is in event chain */
  FXDatabaseField *dbf;       /* !=NULL if FXDatabaseField is in event chain */
  FXVariant data;             /* i.e. dbAskData, dbRefresh */
} FXdbEvent;


/* This struct describes the fox properties of a field */
typedef struct {
  FXString name;    /* Field name */
  FXint index;      /* Field index (base 0) */
  FXbool nullable;  /* Field is nullable */
  FXint fxType;     /* Field fox type (FXVariant::DT_*) as reported from low-level driver */
  FXint maxSize;    /* including '\0' in strings */
  FXbool varSize;   /* <info> variable-sized field */
  FXbool counter;   /* <hint> auto-increment non-null field (not always reported) */
  FXbool readOnly;  /* <hint> Field value cannot be changed */
} FXFieldType;





class FXAPI FXDatabaseField: public FXBaseObject {
  FXDECLARE(FXDatabaseField)
private:
  FXVariant oldValue;
  FXVariant value;
  FXDatabaseQuery *qry;
  FXFieldType fType;

  void checkBind() const;
  FXVariant convertData(const FXVariant &src);
protected:
  FXDatabaseField() {}
public:
  enum {
    ID_EVENT = FXBaseObject::ID_LAST,
    ID_SETFIELDTYPE,
    ID_SETDATA,
    ID_GETDATA,
    ID_REFRESHOLD, /* hack - set tgt with *old* value */
    ID_LAST
  };

  long onSetFieldType(FXObject *, FXSelector, void *);
  long onSetData(FXObject *, FXSelector, void *);
  long onGetData(FXObject *, FXSelector, void *);
  long onDispatch(FXObject *, FXSelector, void *);
  long onRefreshOld(FXObject *, FXSelector, void *);
public:
  FXDatabaseField(FXDatabaseQuery *qry);
  virtual ~FXDatabaseField() {}
  virtual void save(FXStream& store) const;
  virtual void load(FXStream& store);
  operator FXVariant(); /* *not* const :-) */
  const FXVariant &getValue() const;
  FXDatabaseField &operator=(const FXVariant &v);
  FXint getType() const { checkBind(); return fType.fxType; }
  FXbool isNullable() const { checkBind(); return fType.nullable; }
  FXString name() const { checkBind(); return fType.name; }
  const FXFieldType &getFieldType() const { checkBind(); return fType; }
};

typedef FXObjectListOf<FXDatabaseField> FXDatabaseFieldList;

}

#endif /* FXDATABASEFIELD_H */
