/********************************************************************************
*                                                                               *
*               Dynamically linked library loader (DLL loader)                  *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXFile.h>
using namespace FX;
#include "exincs.h"
#include "fxexdefs.h"
#include "FXDLL.h"
using namespace FXEX;
namespace FXEX {

/*
 * Notes:
 *
 * - A reference to macro creation for supporting object loading:
 *  http://msdn.microsoft.com/library/default.asp?url=/library/en-us/dndllpro/html/msdn_scalabil.asp
 *
 */

// used to load the library initializer and filanizer functions
typedef void (*FXDLL_func)();

// ctor
FXDLL::FXDLL(const FXString& filename){
  file=filename;
  handle=NULL;
  refcount=0;
  }

// dtor - unload library on delete if necessary
FXDLL::~FXDLL(){
  while (refcount) close();
  handle=(FXDLLHandle)-1;
  }

// opens the file using the OS specific implementation
FXDLLHandle FXDLL::loadLibrary(const FXString& filename){
#ifndef WIN32
  return ::dlopen(filename.text(),RTLD_NOW|RTLD_GLOBAL);
#else
  return LoadLibrary(filename.text());
#endif
  }

// closes the library using the OS specific implementation
FXDLLHandle FXDLL::unloadLibrary(){
#ifndef WIN32
  ::dlclose(handle);
#else
  FreeLibrary(handle);
#endif
  return NULL;
  }

// set to new filename...
void FXDLL::setFilename(const FXString& filename){
  if (refcount) fxerror("FXDLL: cannot change filename while library loaded\n");
  else file=filename;
  }

// open the library file
// Increment refcount if file is already loaded
FXbool FXDLL::open(){
  if(refcount) { refcount++; return TRUE; }
  FXString name=FXFile::name(file);
  if (!name.length()) fxerror("FXDLL: no library filename defined.\n");
  FXString path=FXFile::directory(file);
  FXbool useAltExt=TRUE;
  FXbool absolute= (file.left(1) == FXString(PATHSEP,1)) ? TRUE : FALSE;
  if (FXFile::extension(file).length()) useAltExt=FALSE;
  if (path.length()) path += PATHSEP;
#ifndef WIN32
  FXString pathEnv=FXFile::getEnvironment("LD_LIBRARY_PATH");
  FXString extalt=".so";
#else
  absolute= (absolute || (FXFile::drive(file)).length()) ? TRUE : FALSE;
  FXString pathEnv=FXFile::getEnvironment("PATH");
  FXString extalt=".dll";
#endif

  // This is a pretty crude way of selecting functionality, but it is
  // reasonably easy to read... and scales reasonbly easily...
  //
  // 1. if the filename (without directory) contains an extension
  //    do not try any other extensions (libXXX, XXX.lib, XXX.dll, etc)
  // 2. if the filename begins with a drive letter / leading slash,
  //    do not try the environment library paths
  // 3. otherwise, try variants ie try each extension, for each path

  FXint pathCount=0;
  FXString f(file);  // first try as specified...
  while(1) {

    // try alternate extensions
    FXint extSelect=1;
    while(extSelect) {

      // try loading dll
      if (FXFile::exists(f) && (handle=loadLibrary(f))!=NULL) {
        loadedFile=f;
        refcount++;
        FXDLL_func func = (FXDLL_func) getFunction("FXDLL_load");
        if (func!=NULL) (*func)();
        return TRUE;
        }

      // if we got here, then we should try alternate extensions
      if(useAltExt) {
        switch (extSelect++) {
          case 1: f=path + name + extalt; break;
          case 2: f=path + "lib" + name + extalt; break;
          case 3: f=path + name + "dll" + extalt; break;
          case 4: f=path + name + ".lib"; break;
          default: extSelect=0;
          }
        }
      else extSelect=0;
      }

    // test abolute path, or no more paths from environment variable
    if (absolute || pathCount > pathEnv.length()) break;

    // try prepending something from the library path environmental variable
    path=pathEnv.section(PATHSEP,pathCount);
    pathCount+=path.length()+1;
    if(path.right(1) != FXString(PATHSEP,1)) path+=PATHSEP;
    }
  return FALSE;
  }

// unload the library
void FXDLL::close(){
  if (!refcount) return;
  refcount--;
  if (!refcount) {
    FXDLL_func func = (FXDLL_func) getFunction("FXDLL_unload");
    if (func!=NULL) (*func)();
    handle=unloadLibrary();
    }
  }

// indicate if the library is already loaded
FXbool FXDLL::isOpen(){
  return refcount?TRUE:FALSE;
  }

// return a pointer to a function in the library
void* FXDLL::getFunction(const FXString& function){
  if (!function.length() || !handle) return NULL;
#ifndef WIN32
  FXDLLHandle h=::dlsym(handle,function.text());
  if ( dlerror() != NULL ) return NULL;
  return h;
#else
  return (void*)GetProcAddress(handle,function.text());
#endif
  }

}

