/********************************************************************************
*                                                                               *
*                   Atomic Variables                                            *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#ifndef FXATOMIC_H
#define FXATOMIC_H

namespace FXEX {
class FXFastMutex;

// declaration
template<class TYPE> class FXAtomic;

/// Support an atomic boolean type
typedef FXAtomic<FXbool> FXAtomicBool;

/// Support an atomic integer type
typedef FXAtomic<FXint> FXAtomicInt;

/// Support an atomic floating point type
typedef FXAtomic<FXfloat> FXAtomicFloat;

/// Support an atomic FXString type
typedef FXAtomic<FXString> FXAtomicString;

/*********** must declare operators for template definition ************/
/// check for equality
template<class TYPE> FXAPI FXbool operator==(FXAtomic<TYPE>& v1,FXAtomic<TYPE>& v2);
template<class TYPE> FXAPI FXbool operator==(FXAtomic<TYPE>& v1,const TYPE v2);
template<class TYPE> FXAPI FXbool operator==(const TYPE v1,FXAtomic<TYPE>& v2);
/// check for in-equality
template<class TYPE> FXAPI FXbool operator!=(FXAtomic<TYPE>& v1,FXAtomic<TYPE>& v2);
template<class TYPE> FXAPI FXbool operator!=(FXAtomic<TYPE>& v1,const TYPE v2);
template<class TYPE> FXAPI FXbool operator!=(const TYPE v1,FXAtomic<TYPE>& v2);
/// less than
template<class TYPE> FXAPI FXbool operator<(FXAtomic<TYPE>& v1,FXAtomic<TYPE>& v2);
template<class TYPE> FXAPI FXbool operator<(FXAtomic<TYPE>& v1,const TYPE v2);
template<class TYPE> FXAPI FXbool operator<(const TYPE v1,FXAtomic<TYPE>& v2);
/// greater than
template<class TYPE> FXAPI FXbool operator>(FXAtomic<TYPE>& v1,FXAtomic<TYPE>& v2);
template<class TYPE> FXAPI FXbool operator>(FXAtomic<TYPE>& v1,const TYPE v2);
template<class TYPE> FXAPI FXbool operator>(const TYPE v1,FXAtomic<TYPE>& v2);
/// less than-equal
template<class TYPE> FXAPI FXbool operator<=(FXAtomic<TYPE>& v1,FXAtomic<TYPE>& v2);
template<class TYPE> FXAPI FXbool operator<=(FXAtomic<TYPE>& v1,const TYPE v2);
template<class TYPE> FXAPI FXbool operator<=(const TYPE v1,FXAtomic<TYPE>& v2);
/// greater than-equal
template<class TYPE> FXAPI FXbool operator>=(FXAtomic<TYPE>& v1,FXAtomic<TYPE>& v2);
template<class TYPE> FXAPI FXbool operator>=(FXAtomic<TYPE>& v1,const TYPE v2);
template<class TYPE> FXAPI FXbool operator>=(const TYPE v1,FXAtomic<TYPE>& v2);
/// Stream operators
template<class TYPE> FXAPI FXStream& operator<<(FXStream& store,const FXAtomic<TYPE>& v);
template<class TYPE> FXAPI FXStream& operator>>(FXStream& store,FXAtomic<TYPE>& v);



/**
 * FXAtomic objects provide atomic access to an a value, for use by multiple threads.
 * This is achieved by providing mutex-locking each function call to set or get the
 * actual value.
 */
template<class TYPE>
class FXAPI FXAtomic {

private:
  FXFastMutex *mutex;  // used to control access to the value
  TYPE         value;  // actual value

public:
  /// Create an atomic value
  FXAtomic(const TYPE v=0);

  /// set the value
  void setValue(const TYPE v);

  /// get the value
  TYPE getValue();

  /// return a TYPE, say for use in a 'return' or an 'if'
  operator TYPE();

  /// allow assignment to new value of type FXAtomic<TYPE>
  FXAtomic<TYPE>& operator=(const FXAtomic<TYPE>& v);

  /// allow assignment to new value of type TYPE
  FXAtomic<TYPE>& operator=(const TYPE v);

  /// increment value (prefix and postfix)
  TYPE operator++();
  TYPE operator++(TYPE);

  /// decrement value (prefix and postfix)
  TYPE operator--();
  TYPE operator--(TYPE);

  /// addition
  TYPE operator+(const TYPE v);

  /// subtraction
  TYPE operator-(const TYPE v);

  /// multiplication
  TYPE operator*(const TYPE v);

  /// divisiion
  TYPE operator/(const TYPE v);

  /// increment by some value
  TYPE operator+=(const TYPE v);

  /// decrement by some value
  TYPE operator-=(const TYPE v);

  /// multiply by some value
  TYPE operator*=(const TYPE v);

  /// divide by some value
  TYPE operator/=(const TYPE v);

  /// check for equality
  friend FXAPI FXbool operator== <TYPE>(FXAtomic<TYPE>& v1,FXAtomic<TYPE>& v2);
  friend FXAPI FXbool operator== <TYPE>(FXAtomic<TYPE>& v1,const TYPE v2);
  friend FXAPI FXbool operator== <TYPE>(const TYPE v1,FXAtomic<TYPE>& v2);

  /// check for in-equality
  friend FXAPI FXbool operator!= <TYPE>(FXAtomic<TYPE>& v1,FXAtomic<TYPE>& v2);
  friend FXAPI FXbool operator!= <TYPE>(FXAtomic<TYPE>& v1,const TYPE v2);
  friend FXAPI FXbool operator!= <TYPE>(const TYPE v1,FXAtomic<TYPE>& v2);

  /// less than
  friend FXAPI FXbool operator<  <TYPE>(FXAtomic<TYPE>& v1,FXAtomic<TYPE>& v2);
  friend FXAPI FXbool operator<  <TYPE>(FXAtomic<TYPE>& v1,const TYPE v2);
  friend FXAPI FXbool operator<  <TYPE>(const TYPE v1,FXAtomic<TYPE>& v2);

  /// greater than
  friend FXAPI FXbool operator>  <TYPE>(FXAtomic<TYPE>& v1,FXAtomic<TYPE>& v2);
  friend FXAPI FXbool operator>  <TYPE>(FXAtomic<TYPE>& v1,const TYPE v2);
  friend FXAPI FXbool operator>  <TYPE>(const TYPE v1,FXAtomic<TYPE>& v2);

  /// less than-equal
  friend FXAPI FXbool operator<= <TYPE>(FXAtomic<TYPE>& v1,FXAtomic<TYPE>& v2);
  friend FXAPI FXbool operator<= <TYPE>(FXAtomic<TYPE>& v1,const TYPE v2);
  friend FXAPI FXbool operator<= <TYPE>(const TYPE v1,FXAtomic<TYPE>& v2);

  /// greater than-equal
  friend FXAPI FXbool operator>= <TYPE>(FXAtomic<TYPE>& v1,FXAtomic<TYPE>& v2);
  friend FXAPI FXbool operator>= <TYPE>(FXAtomic<TYPE>& v1,const TYPE v2);
  friend FXAPI FXbool operator>= <TYPE>(const TYPE v1,FXAtomic<TYPE>& v2);

  /// Saving to a stream
  friend FXAPI FXStream& operator<< <TYPE>(FXStream& store,const FXAtomic<TYPE>& v);

  /// Load from a stream
  friend FXAPI FXStream& operator>> <TYPE>(FXStream& store,FXAtomic<TYPE>& v);

  /// dtor
  ~FXAtomic();
  };

/**********************  I m p l e m e n t a t i o n  ************************/
// when you define a template class, the implementation must exist in the same file
// as the definition

// ctor
template<class TYPE>
FXAtomic<TYPE>::FXAtomic(const TYPE v) : value(v) {
  mutex=new FXFastMutex();
  }

template<class TYPE>
FXAtomic<TYPE>::~FXAtomic(){
  delete mutex;
  }

// set the value
template<class TYPE>
void FXAtomic<TYPE>::setValue(const TYPE v){
  mutex->lock();
  value=v;
  mutex->unlock();
  }

// return the value
template<class TYPE>
TYPE FXAtomic<TYPE>::getValue(void) {
  mutex->lock();
  TYPE t(value);
  mutex->unlock();
  return t;
  }

// return an TYPE
template<class TYPE>
FXAtomic<TYPE>::operator TYPE(void){
  return getValue();
  }

// set to new FXAtomic value
template<class TYPE>
FXAtomic<TYPE>& FXAtomic<TYPE>::operator=(const FXAtomic<TYPE>& v){
  setValue(v.getValue());
  return *this;
  }

// set to new TYPE value
template<class TYPE>
FXAtomic<TYPE>& FXAtomic<TYPE>::operator=(const TYPE v){
  setValue(v);
  return *this;
  }

// increment
template<class TYPE>
TYPE FXAtomic<TYPE>::operator++(void){
  mutex->lock();
  TYPE t(++value);
  mutex->unlock();
  return t;
  }

// increment
template<class TYPE>
TYPE FXAtomic<TYPE>::operator++(TYPE){
  mutex->lock();
  TYPE t(value++);
  mutex->unlock();
  return t;
  }

// decrement
template<class TYPE>
TYPE FXAtomic<TYPE>::operator--(void){
  mutex->lock();
  TYPE t(--value);
  mutex->unlock();
  return t;
  }

// decrement
template<class TYPE>
TYPE FXAtomic<TYPE>::operator--(TYPE){
  mutex->lock();
  TYPE t(value--);
  mutex->unlock();
  return t;
  }

// addition
template<class TYPE>
TYPE FXAtomic<TYPE>::operator+(const TYPE v){
  mutex->lock();
  TYPE t(value);
  mutex->unlock();
  return t+v;
  }

// subtraction
template<class TYPE>
TYPE FXAtomic<TYPE>::operator-(const TYPE v){
  mutex->lock();
  TYPE t(value);
  mutex->unlock();
  return t-v;
  }

// multiplication
template<class TYPE>
TYPE FXAtomic<TYPE>::operator*(const TYPE v){
  mutex->lock();
  TYPE t(value);
  mutex->unlock();
  return t*v;
  }

// division
template<class TYPE>
TYPE FXAtomic<TYPE>::operator/(const TYPE v){
  mutex->lock();
  TYPE t(value);
  mutex->unlock();
  return t/v;
  }

// increment by some value
template<class TYPE>
TYPE FXAtomic<TYPE>::operator+=(const TYPE v){
  mutex->lock();
  value+=v;
  TYPE t(value);
  mutex->unlock();
  return t;
  }

// decrement by some value
template<class TYPE>
TYPE FXAtomic<TYPE>::operator-=(const TYPE v){
  mutex->lock();
  value-=v;
  TYPE t(value);
  mutex->unlock();
  return t;
  }

// multiply by some value
template<class TYPE>
TYPE FXAtomic<TYPE>::operator*=(const TYPE v){
  mutex->lock();
  value*=v;
  TYPE t(value);
  mutex->unlock();
  return t;
  }

// divide by some value
template<class TYPE>
TYPE FXAtomic<TYPE>::operator/=(const TYPE v){
  mutex->lock();
  value/=v;
  TYPE t(value);
  mutex->unlock();
  return t;
  }

// test equality
template<class TYPE>
FXbool operator==(FXAtomic<TYPE>& v1,FXAtomic<TYPE>& v2){
  return v1.getValue()==v2.getValue();
  }

// test equality
template<class TYPE>
FXbool operator==(FXAtomic<TYPE>& v1,const TYPE v2){
  return v1.getValue()==v2;
  }

// test equality
template<class TYPE>
FXbool operator==(const TYPE v1,FXAtomic<TYPE>& v2){
  return v1==v2.getValue();
  }

// test in-equality
template<class TYPE>
FXbool operator!=(FXAtomic<TYPE>& v1,FXAtomic<TYPE>& v2){
  return v1.getValue()!=v2.getValue();
  }

// test in-equality
template<class TYPE>
FXbool operator!=(FXAtomic<TYPE>& v1,const TYPE v2){
  return v1.getValue()!=v2;
  }

// test in-equality
template<class TYPE>
FXbool operator!=(const TYPE v1,FXAtomic<TYPE>& v2){
  return v1!=v2.getValue();
  }

// test greater than
template<class TYPE>
FXbool operator>(FXAtomic<TYPE>& v1,FXAtomic<TYPE>& v2){
  return v1.getValue()>v2.getValue();
  }

// test greater than
template<class TYPE>
FXbool operator>(FXAtomic<TYPE>& v1,const TYPE v2){
  return v1.getValue()>v2;
  }

// test greater than
template<class TYPE>
FXbool operator>(const TYPE v1,FXAtomic<TYPE>& v2){
  return v1>v2.getValue();
  }

// test less than
template<class TYPE>
FXbool operator<(FXAtomic<TYPE>& v1,FXAtomic<TYPE>& v2){
  return v1.getValue()<v2.getValue();
  }

// test less than
template<class TYPE>
FXbool operator<(FXAtomic<TYPE>& v1,const TYPE v2){
  return v1.getValue()<v2;
  }

// test less than
template<class TYPE>
FXbool operator<(const TYPE v1,FXAtomic<TYPE>& v2){
  return v1<v2.getValue();
  }

// test greater than-equal
template<class TYPE>
FXbool operator>=(FXAtomic<TYPE>& v1,FXAtomic<TYPE>& v2){
  return v1.getValue()>=v2.getValue();
  }

// test greater than-equal
template<class TYPE>
FXbool operator>=(FXAtomic<TYPE>& v1,const TYPE v2){
  return v1.getValue()>=v2;
  }

// test greater than-equal
template<class TYPE>
FXbool operator>=(const TYPE v1,FXAtomic<TYPE>& v2){
  return v1>=v2.getValue();
  }

// test less than-equal
template<class TYPE>
FXbool operator<=(FXAtomic<TYPE>& v1,FXAtomic<TYPE>& v2){
  return v1.getValue()<=v2.getValue();
  }

// test less than-equal
template<class TYPE>
FXbool operator<=(FXAtomic<TYPE>& v1,const TYPE v2){
  return v1.getValue()<=v2;
  }

// test less than-equal
template<class TYPE>
FXbool operator<=(const TYPE v1,FXAtomic<TYPE>& v2){
  return v1<=v2.getValue();
  }

// save to stream
template<class TYPE>
FXStream& operator<<(FXStream& store,const FXAtomic<TYPE>& v){
  store << v.getValue();
  return store;
  }

// load from stream
template<class TYPE>
FXStream& operator>>(FXStream& store,FXAtomic<TYPE>& v){
  TYPE t;
  store >> t;
  v.setValue(t);
  return store;
  }

} // namespace FXEX
#endif // FXATOMIC_H
