/********************************************************************************
*                                                                               *
*                             S o c k e t  S t r e a m                          *
*                                                                               *
*********************************************************************************
* Copyright (C) 2000 David Tyree.(celer@ipro.lug.usf.edu) All Rights Reserved.  *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id$                                                                          *
********************************************************************************/

#include "fx.h"

#ifndef FX_NATIVE_WIN32
  #include <sys/socket.h> 
  #include <netinet/in.h>
  #include <arpa/inet.h> 
  #include <netdb.h>
  #include <unistd.h>
  #include <errno.h>
#else
  #include <winsock.h>
  #define socklen_t int
#endif 


#include <sys/types.h>
#include <string.h>
#include <fcntl.h>


#include "FXSocketStream.h"

#ifndef INVALID_SOCKET
  #define INVALID_SOCKET -1
#endif


/* Service Stuff */

FXNetworkService::FXNetworkService(const FXString &serv,const FXString &prot){
  if(prot!="")
    service=getservbyname(serv.text(),prot.text());
  else
    service=getservbyname(serv.text(),NULL);  
}



/* Address stuff */

FXNetworkAddress::FXNetworkAddress(const FXString &host,FXNetworkService *service,FXuint family){
  struct in_addr *addr=NULL;
  memset(&address,0,sizeof(sockaddr));
  address.sin_family=family; 
  address.sin_port=htons(service->getPort());
  addr=hostname(host);
  if(addr)
    memcpy(&address.sin_addr,addr,sizeof(struct in_addr));
  length=sizeof(address);
}

FXNetworkAddress::FXNetworkAddress(const FXString &host,FXuint port,FXuint family){
  struct in_addr *addr=NULL;
  memset(&address,0,sizeof(sockaddr));
  address.sin_family=family; 
  address.sin_port=htons(port);
  addr=hostname(host);
  if(addr)
    memcpy(&address.sin_addr,addr,sizeof(struct in_addr));  
  length=sizeof(address);
}

FXNetworkAddress::FXNetworkAddress(FXuint port){
  memset(&address,0,sizeof(sockaddr));  
  address.sin_port=htons(port);
  address.sin_addr.s_addr=htonl(INADDR_ANY);
  length=sizeof(address);
}

FXuint FXNetworkAddress::getPort(void){
  return ntohs(address.sin_port);
}

void FXNetworkAddress::setPort(FXuint port){
  address.sin_port=htons(port);
}

struct in_addr *FXNetworkAddress::hostname(const FXString &hostname){
  struct hostent *host;
  host=gethostbyname(hostname.text());
  if(!host)
    return NULL;
  return (struct in_addr *) *host->h_addr_list;
}

FXbool FXNetworkAddress::setHostname(const FXString &host){
  struct in_addr *addr=NULL;
  addr=hostname(host);
  if(addr){
    memcpy(&address.sin_addr,addr,sizeof(struct in_addr)); 
    length=sizeof(address);
  }
  else 
    return FALSE;
  return TRUE;
}

FXString FXNetworkAddress::getAddress(void){
  return inet_ntoa(address.sin_addr);
}

void FXNetworkAddress::setAddress(FXString addr){ 
  address.sin_addr.s_addr=inet_addr(addr.text());
}


FXString FXNetworkAddress::getHostname(FXuint family){
  FXString name="";
  struct hostent *host;
  host=gethostbyaddr((char *) &address.sin_addr,sizeof(struct in_addr),family);
  if(host)
    name=host->h_name;
  return name;
}

void FXNetworkAddress::setSockAddr(struct sockaddr_in *s){
  memcpy(&address,s,sizeof(s));
  address.sin_addr=s->sin_addr;
}


/* Socket stream stuff */

FXbool FXSocketStream::createSocket(FXint family,FXint type,FXint protocol){
  sockfd=socket(family,type,protocol);
  if(INVALID_SOCKET==sockfd){
    code=FXSocketStreamError;
    errornumber=errno;
    return FALSE;  
  }  
  code=FXStreamOK;
  return TRUE;
}

FXbool FXSocketStream::connectSocket(FXNetworkAddress *server){
  if(-1==connect((SOCKET) sockfd, (struct sockaddr *) server->getSockAddr(),(socklen_t) server->getLength())){
    code=FXSocketStreamError;
    errornumber=errno;
    return FALSE;
  }
  code=FXStreamOK;
  return TRUE;
}

FXbool FXSocketStream::bindSocket(FXNetworkAddress *myaddr){
  if(-1==bind((SOCKET) sockfd,(struct sockaddr *) myaddr->getSockAddr(),(socklen_t) myaddr->getLength())){
    code=FXSocketStreamError;
    errornumber=errno;
    return FALSE;
  }
  code=FXStreamOK;
  return TRUE;
}


FXbool FXSocketStream::listenSocket(FXint backlog){
  if(-1==listen((SOCKET) sockfd,backlog)){
    code=FXSocketStreamError;
    errornumber=errno;
    return FALSE;
  }
  code=FXStreamOK;
  return TRUE;
}

FXSocketStream *FXSocketStream::acceptSocket(FXNetworkAddress *address){
  struct sockaddr_in client;
#ifndef FX_NATIVE_WIN32  
  FXuint len=sizeof(client);
#else
  FXint len=sizeof(client);
#endif
  FXSocketStream *s=new FXSocketStream;
  SOCKET clientfd;
  clientfd=accept((SOCKET) sockfd,(struct sockaddr *) &client,&len);
  if(clientfd==INVALID_SOCKET)
  {
    errornumber=errno;
    code=FXSocketStreamError;
    delete s;
    return NULL;
  }
  if(address){
    address->setSockAddr(&client);
  }
  s->setSocket(clientfd);  
  code=FXStreamOK;
  return s;
}


FXSocketStream::FXSocketStream(const FXObject* cont):FXStream(cont){
#ifdef FX_NATIVE_WIN32  
  WSADATA wsdata; 
  WORD    wVersionRequested; 
  wVersionRequested = MAKEWORD(1,1); 
  WSAStartup(wVersionRequested, &wsdata); 
#endif  
  sockfd=INVALID_SOCKET;
  blocking=TRUE;
}


void FXSocketStream::saveItems(const void *buf,FXuint n){
  if(-1==send((SOCKET) sockfd,(const char *) buf,n,0)){
    errornumber=errno;
    code=FXSocketStreamFailedRead;
  }    
  code=FXStreamOK;
}

void FXSocketStream::loadItems(void *buf,FXuint n){
  if(-1==recv((SOCKET) sockfd,(char *) buf,n,0)){
    errornumber=errno;   
    code=FXSocketStreamFailedWrite;
  }
  code=FXStreamOK;
}


FXbool FXSocketStream::getBlocking(void){
  return blocking;
}

FXbool FXSocketStream::setBlocking(FXbool b){
  unsigned long ioctl_opt=1;
  
  if(b==FALSE){
#ifdef FX_NATIVE_WIN32
    ioctlsocket(sockfd,FIONBIO,&ioctl_opt); 
#else
    if(-1==fcntl((SOCKET) sockfd,F_SETFL,O_NONBLOCK)){  
      errornumber=errno;
      return FALSE;
    }
#endif
    blocking=FALSE;
  }
  else{
#ifdef FX_NATIVE_WIN32
    ioctl_opt=0;
    ioctlsocket(sockfd,FIONBIO,&ioctl_opt);
#else    
    if(-1==fcntl((SOCKET) sockfd,F_SETFL,~O_NONBLOCK)){
      errornumber=errno;
      return FALSE;
    }
#endif
    blocking=TRUE;
  }
  return TRUE;
}

void FXSocketStream::closeSocket(){
  if(-1==shutdown((SOCKET) sockfd,2))
    errornumber=errno;
}

FXSocketStream::~FXSocketStream(){
  shutdown((SOCKET) sockfd,2);
}
