/********************************************************************************
*                                                                               *
*                   BinaryLog File Reader                                       *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This program is free software; you can redistribute it and/or modify          *
* it under the terms of the GNU General Public License as published by          *
* the Free Software Foundation; either version 2 of the License, or             *
* (at your option) any later version.                                           *
*                                                                               *
* This program is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
* GNU General Public License for more details.                                  *
*                                                                               *
* You should have received a copy of the GNU General Public License             *
* along with this program; if not, write to the Free Software                   *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#include "fxex.h"
#include "EnterMessageBox.h"

#define ERRORLIST   "errorlist.cfg"

/**
 * This demo is used to display a log file that has been written with the FXBinaryLogger class.
 * Update the errorlist.cfg file to support more logged values.
 */

class MainWindow : public FXMainWindow {
  FXDECLARE(MainWindow)
private:
  FXBinaryLogReader *reader;
  FXList            *datelist;
  FXList            *codelist;
  FXList            *valuelist;
  FXSettings         settings;
protected:
  MainWindow(){}
public:
  enum {
    ID_MAINWINDOW=FXMainWindow::ID_LAST,
    ID_LOAD_LOG_FILE,
    ID_SELECT_DATE,
    ID_SELECT_CODE,
    ID_SELECT_ERROR,
    ID_QUIT
    };
public:
  long onQuit(FXObject*,FXSelector,void*);
  long onMainWindow(FXObject*,FXSelector,void*);
  long onSelectEntry(FXObject*,FXSelector,void*);
  long onSelectCode(FXObject*,FXSelector,void*);
  long onSelectError(FXObject*,FXSelector,void*);
  long onLoadLogFile(FXObject*,FXSelector,void*);
public:
  MainWindow(FXApp *a);
  virtual void create();
  void loadFile(const FXString& file="");
  FXString lookupCode(FXshort code);
  FXString lookupValue(FXshort value);
  ~MainWindow();
  };


// map
FXDEFMAP(MainWindow) MainWindowMap[]={
  FXMAPFUNC(SEL_SIGNAL,MainWindow::ID_QUIT,MainWindow::onQuit),
  FXMAPFUNC(SEL_CLOSE,MainWindow::ID_QUIT,MainWindow::onQuit),
  FXMAPFUNC(SEL_COMMAND,MainWindow::ID_QUIT,MainWindow::onQuit),
  FXMAPFUNC(SEL_COMMAND,MainWindow::ID_MAINWINDOW,MainWindow::onMainWindow),
  FXMAPFUNC(SEL_CLICKED,MainWindow::ID_SELECT_DATE,MainWindow::onSelectEntry),
  FXMAPFUNC(SEL_CLICKED,MainWindow::ID_SELECT_CODE,MainWindow::onSelectEntry),
  FXMAPFUNC(SEL_CLICKED,MainWindow::ID_SELECT_ERROR,MainWindow::onSelectEntry),
  FXMAPFUNC(SEL_COMMAND,MainWindow::ID_SELECT_CODE,MainWindow::onSelectCode),
  FXMAPFUNC(SEL_COMMAND,MainWindow::ID_SELECT_ERROR,MainWindow::onSelectError),
  FXMAPFUNC(SEL_COMMAND,MainWindow::ID_LOAD_LOG_FILE,MainWindow::onLoadLogFile),
  };
FXIMPLEMENT(MainWindow,FXMainWindow,MainWindowMap,ARRAYNUMBER(MainWindowMap))

// build the main window
MainWindow::MainWindow(FXApp *a) : FXMainWindow(a,"Binary Log Reader",NULL,NULL,DECOR_ALL){
  setTarget(this);
  setSelector(ID_MAINWINDOW);
  new FXToolTip(getApp());
  FXToolBar *toolbar=new FXToolBar(this,this,LAYOUT_FILL_X|FRAME_RAISED);
  new FXButton(toolbar,"Quit",NULL,this,ID_QUIT,BUTTON_TOOLBAR|FRAME_RAISED);
  new FXButton(toolbar,"Load file",NULL,this,ID_LOAD_LOG_FILE,BUTTON_TOOLBAR|FRAME_RAISED);

  FXSplitter *mainframe=new FXSplitter(this,SPLITTER_TRACKING|FRAME_SUNKEN|LAYOUT_FILL);
  mainframe->setBarSize(1);
  datelist=new FXList(mainframe,1,this,ID_SELECT_DATE,LIST_BROWSESELECT|LAYOUT_FIX_X|LAYOUT_FILL_Y, 0,0,100,0);
  FXSplitter *dateframe=new FXSplitter(mainframe,SPLITTER_TRACKING|LAYOUT_FILL);
  dateframe->setBarSize(1);

  codelist=new FXList(dateframe,1,this,ID_SELECT_CODE,LIST_BROWSESELECT|LAYOUT_FIX_X|LAYOUT_FILL_Y, 0,0,100,0);
  valuelist=new FXList(dateframe,1,this,ID_SELECT_ERROR,LIST_BROWSESELECT|LAYOUT_FILL);

  reader=new FXBinaryLogReader();
  }

MainWindow::~MainWindow(){
  delete reader;
  // FIXME: should really free the memory allocated in the FXMEMDUP's
  }

void MainWindow::create(){
  FXMainWindow::create();
  FXint fontwidth;
  FXString s;
  FXFont *font;
  font=datelist->getFont();
  s = ".88/88/8888 88:88:88.";
  fontwidth=font->getTextWidth(s.text(),s.length());
  datelist->setWidth(fontwidth);
  font=codelist->getFont();
  s = "WWWWWWWW";
  fontwidth=font->getTextWidth(s.text(),s.length());
  codelist->setWidth(fontwidth);
  settings.parseFile(ERRORLIST,FALSE);
  show(PLACEMENT_MAXIMIZED);
  }

long MainWindow::onQuit(FXObject*,FXSelector,void*){
  if(settings.isModified()) settings.unparseFile(ERRORLIST);
  getApp()->exit(0);
  return 1;
  }

long MainWindow::onMainWindow(FXObject*,FXSelector,void*){
  fxmessage("MainWindow event\n");
  return 1;
  }

// user clicked on entry
long MainWindow::onSelectEntry(FXObject*,FXSelector sel,void*){
  switch (FXSELID(sel)) {
    case ID_SELECT_DATE: {
      FXint item=datelist->getCurrentItem();
      codelist->setCurrentItem(item);
      valuelist->setCurrentItem(item);
      } break;
    case ID_SELECT_CODE: {
      FXint item=codelist->getCurrentItem();
      datelist->setCurrentItem(item);
      valuelist->setCurrentItem(item);
      } break;
    case ID_SELECT_ERROR: {
      FXint item=valuelist->getCurrentItem();
      datelist->setCurrentItem(item);
      codelist->setCurrentItem(item);
      } break;
    }
  return 1;
  }

// user double clicked on code
long MainWindow::onSelectCode(FXObject*,FXSelector,void*){
  FXListItem *item=codelist->getItem(codelist->getCurrentItem());
  FXString msg=item->getText();
  EnterMessageBox w(this,msg);
  if(w.execute()) {
    msg=w.getText();
    }
  if(msg.length()){
    FXshort *val=(FXshort*)item->getData();
    FXString code=FXStringVal((FXint)(*val));
    settings.writeStringEntry("Codes",code.text(),msg.text());
    }
  return 1;
  }

// user double clicked on error
long MainWindow::onSelectError(FXObject*,FXSelector,void*){
  FXListItem *item=valuelist->getItem(valuelist->getCurrentItem());
  FXString msg=item->getText();
  EnterMessageBox w(this,msg);
  if(w.execute()) {
    msg=w.getText();
    }
  if(msg.length()){
    FXshort *val=(FXshort*)item->getData();
    FXString value=FXStringVal((FXint)(*val));
    settings.writeStringEntry("Values",value.text(),msg.text());
    }
  return 1;
  }

// show a file open dialog
long MainWindow::onLoadLogFile(FXObject*,FXSelector,void*){
  loadFile();
  return 1;
  }

// show a file open dialog
void MainWindow::loadFile(const FXString& file){
  FXString filename=file;
  if (!filename.length()) {
    FXFileDialog w(this,"Load a log file");
    w.setPatternList("Binary Log Files (*.binlog)\nLog Files (*.log)\nAll Files (*)");
    if (w.execute()) {
      filename=w.getFilename();
      }
    }
  if(filename.length()){
    fxmessage("Load log file: %s\n",filename.text());
    if (filename.length() && reader->name(filename)) {
      FXString s;
      FXBinaryLogData *d;
      if (d->subsecond) {
        FXFont *font=datelist->getFont();
  	FXString s = ".88/88/8888 88:88:88.000000.";
        FXint fontwidth=font->getTextWidth(s.text(),s.length());
        datelist->setWidth(fontwidth);
        }
      for(d=reader->read(); d!=NULL; d=reader->read()){
        s=FXDateTime::convert(d->seconds);
        if (d->subsecond) {
	  FXString m=FXStringVal(d->microseconds);
	  m.prepend('0',6-m.length());
	  s+="." + m;
	  }
	FXshort *val;
        datelist->appendItem(s);
	FXMEMDUP(&val,FXshort,&(d->code),1);
        codelist->appendItem(lookupCode(d->code),NULL,val);
	FXMEMDUP(&val,FXshort,&(d->value),1);
	valuelist->appendItem(lookupValue(d->value),NULL,val);
        }
      }
    }
  }

// lookup code - convert to text
FXString MainWindow::lookupCode(FXshort code){
  return settings.readStringEntry("Codes",FXStringVal((FXint)code).text(),FXStringVal(code).text());
  }

// lookup value - convert to text
FXString MainWindow::lookupValue(FXshort value){
  return settings.readStringEntry("Values",FXStringVal((FXint)value).text(),FXStringVal(value).text());
  }

/* start it up */
int main(int argc, char *argv[]){
  FXApp app("Binary Log Reader","FoxExTest");
  app.init(argc,argv);
  MainWindow *w=new MainWindow(&app);
  app.create();
  if (argc > 1) {
    w->loadFile(argv[1]);
    }
  return app.run();
  }

