package filesystem

// Copyright (c) Microsoft and contributors.  All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Code generated by Microsoft (R) AutoRest Code Generator.
// Changes may cause incorrect behavior and will be lost if the code is regenerated.

import (
	"github.com/Azure/go-autorest/autorest"
	"github.com/Azure/go-autorest/autorest/azure"
	"github.com/Azure/go-autorest/autorest/validation"
	uuid "github.com/satori/go.uuid"
	"io"
	"net/http"
)

// GroupClient is the creates an Azure Data Lake Store filesystem client.
type GroupClient struct {
	ManagementClient
}

// NewGroupClient creates an instance of the GroupClient client.
func NewGroupClient() GroupClient {
	return GroupClient{New()}
}

// Append used for serial appends to the specified file. NOTE: The target must not contain data added by
// ConcurrentAppend. ConcurrentAppend and Append cannot be used interchangeably; once a target file has been modified
// using either of these append options, the other append option cannot be used on the target file.
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the file to which to append. streamContents is the file contents to include when
// appending to the file. streamContents will be closed upon successful return. Callers should ensure closure when
// receiving an error.appendParameter is flag to skip redirection. When append=false or not specified, the request is
// redirected. Submit another HTTP PUT request using the URL in the Location header with the file data to be written.
// When append=true, this redirection is skipped. op is the constant value for the operation. offset is the optional
// offset in the stream to begin the append operation. Default is to append at the end of the stream. syncFlag is
// optionally indicates what to do after completion of the concurrent append. DATA indicates that more data will be
// sent immediately by the client, the file handle should remain open/locked, and file metadata (including file length,
// last modified time) should NOT get updated. METADATA indicates that more data will be sent immediately by the
// client, the file handle should remain open/locked, and file metadata should get updated. CLOSE indicates that the
// client is done sending data, the file handle should be closed/unlocked, and file metadata should get updated.
// leaseID is optional unique GUID per file to ensure single writer semantics, meaning that only clients that append to
// the file with the same leaseId will be allowed to do so. fileSessionID is optional unique GUID per file indicating
// all the appends with the same fileSessionId are from the same client and same session. This will give a performance
// benefit when syncFlag is DATA or METADATA.
//
// Deprecated: Please use github.com/Azure/azure-sdk-for-go/services/datalake/store/2016-11-01/filesystem
func (client GroupClient) Append(accountName string, pathParameter string, streamContents io.ReadCloser, appendParameter string, op string, offset *int64, syncFlag SyncFlag, leaseID *uuid.UUID, fileSessionID *uuid.UUID) (result autorest.Response, err error) {
	req, err := client.AppendPreparer(accountName, pathParameter, streamContents, appendParameter, op, offset, syncFlag, leaseID, fileSessionID)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "Append", nil, "Failure preparing request")
		return
	}

	resp, err := client.AppendSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "Append", resp, "Failure sending request")
		return
	}

	result, err = client.AppendResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "Append", resp, "Failure responding to request")
	}

	return
}

// AppendPreparer prepares the Append request.
func (client GroupClient) AppendPreparer(accountName string, pathParameter string, streamContents io.ReadCloser, appendParameter string, op string, offset *int64, syncFlag SyncFlag, leaseID *uuid.UUID, fileSessionID *uuid.UUID) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"append":      autorest.Encode("query", appendParameter),
		"op":          autorest.Encode("query", op),
	}
	if offset != nil {
		queryParameters["offset"] = autorest.Encode("query", *offset)
	}
	if len(string(syncFlag)) > 0 {
		queryParameters["syncFlag"] = autorest.Encode("query", syncFlag)
	}
	if leaseID != nil {
		queryParameters["leaseId"] = autorest.Encode("query", *leaseID)
	}
	if fileSessionID != nil {
		queryParameters["fileSessionId"] = autorest.Encode("query", *fileSessionID)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithFile(streamContents),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// AppendSender sends the Append request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) AppendSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// AppendResponder handles the response to the Append request. The method always
// closes the http.Response Body.
func (client GroupClient) AppendResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// CheckAccess checks if the specified access is available at the given path.
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the file or directory for which to check access. fsaction is file system operation
// read/write/execute in string form, matching regex pattern '[rwx-]{3}' op is the constant value for the operation.
func (client GroupClient) CheckAccess(accountName string, pathParameter string, fsaction string, op string) (result autorest.Response, err error) {
	req, err := client.CheckAccessPreparer(accountName, pathParameter, fsaction, op)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "CheckAccess", nil, "Failure preparing request")
		return
	}

	resp, err := client.CheckAccessSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "CheckAccess", resp, "Failure sending request")
		return
	}

	result, err = client.CheckAccessResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "CheckAccess", resp, "Failure responding to request")
	}

	return
}

// CheckAccessPreparer prepares the CheckAccess request.
func (client GroupClient) CheckAccessPreparer(accountName string, pathParameter string, fsaction string, op string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"fsaction":    autorest.Encode("query", fsaction),
		"op":          autorest.Encode("query", op),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// CheckAccessSender sends the CheckAccess request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) CheckAccessSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// CheckAccessResponder handles the response to the CheckAccess request. The method always
// closes the http.Response Body.
func (client GroupClient) CheckAccessResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// Concat concatenates the list of source files into the destination file, removing all source files upon success.
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the destination file resulting from the concatenation. sources is a list of comma
// separated Data Lake Store paths (starting with '/') of the files to concatenate, in the order in which they should
// be concatenated. op is the constant value for the operation.
func (client GroupClient) Concat(accountName string, pathParameter string, sources []string, op string) (result autorest.Response, err error) {
	if err := validation.Validate([]validation.Validation{
		{TargetValue: sources,
			Constraints: []validation.Constraint{{Target: "sources", Name: validation.Null, Rule: true, Chain: nil}}}}); err != nil {
		return result, validation.NewErrorWithValidationError(err, "filesystem.GroupClient", "Concat")
	}

	req, err := client.ConcatPreparer(accountName, pathParameter, sources, op)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "Concat", nil, "Failure preparing request")
		return
	}

	resp, err := client.ConcatSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "Concat", resp, "Failure sending request")
		return
	}

	result, err = client.ConcatResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "Concat", resp, "Failure responding to request")
	}

	return
}

// ConcatPreparer prepares the Concat request.
func (client GroupClient) ConcatPreparer(accountName string, pathParameter string, sources []string, op string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", op),
		"sources":     autorest.Encode("query", sources, ","),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// ConcatSender sends the Concat request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) ConcatSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ConcatResponder handles the response to the Concat request. The method always
// closes the http.Response Body.
func (client GroupClient) ConcatResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// ConcurrentAppend appends to the specified file, optionally first creating the file if it does not yet exist. This
// method supports multiple concurrent appends to the file. NOTE: The target must not contain data added by Create or
// normal (serial) Append. ConcurrentAppend and Append cannot be used interchangeably; once a target file has been
// modified using either of these append options, the other append option cannot be used on the target file.
// ConcurrentAppend does not guarantee order and can result in duplicated data landing in the target file.
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the file to which to append using concurrent append. streamContents is the file
// contents to include when appending to the file. streamContents will be closed upon successful return. Callers should
// ensure closure when receiving an error.op is the constant value for the operation. transferEncoding is indicates the
// data being sent to the server is being streamed in chunks. appendMode is indicates the concurrent append call should
// create the file if it doesn't exist or just open the existing file for append syncFlag is optionally indicates what
// to do after completion of the concurrent append. DATA indicates that more data will be sent immediately by the
// client, the file handle should remain open/locked, and file metadata (including file length, last modified time)
// should NOT get updated. METADATA indicates that more data will be sent immediately by the client, the file handle
// should remain open/locked, and file metadata should get updated. CLOSE indicates that the client is done sending
// data, the file handle should be closed/unlocked, and file metadata should get updated.
func (client GroupClient) ConcurrentAppend(accountName string, pathParameter string, streamContents io.ReadCloser, op string, transferEncoding string, appendMode AppendModeType, syncFlag SyncFlag) (result autorest.Response, err error) {
	req, err := client.ConcurrentAppendPreparer(accountName, pathParameter, streamContents, op, transferEncoding, appendMode, syncFlag)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "ConcurrentAppend", nil, "Failure preparing request")
		return
	}

	resp, err := client.ConcurrentAppendSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "ConcurrentAppend", resp, "Failure sending request")
		return
	}

	result, err = client.ConcurrentAppendResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "ConcurrentAppend", resp, "Failure responding to request")
	}

	return
}

// ConcurrentAppendPreparer prepares the ConcurrentAppend request.
func (client GroupClient) ConcurrentAppendPreparer(accountName string, pathParameter string, streamContents io.ReadCloser, op string, transferEncoding string, appendMode AppendModeType, syncFlag SyncFlag) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", op),
	}
	if len(string(appendMode)) > 0 {
		queryParameters["appendMode"] = autorest.Encode("query", appendMode)
	}
	if len(string(syncFlag)) > 0 {
		queryParameters["syncFlag"] = autorest.Encode("query", syncFlag)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/WebHdfsExt/{path}", pathParameters),
		autorest.WithFile(streamContents),
		autorest.WithQueryParameters(queryParameters),
		autorest.WithHeader("Transfer-Encoding", autorest.String(transferEncoding)))
	return preparer.Prepare(&http.Request{})
}

// ConcurrentAppendSender sends the ConcurrentAppend request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) ConcurrentAppendSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ConcurrentAppendResponder handles the response to the ConcurrentAppend request. The method always
// closes the http.Response Body.
func (client GroupClient) ConcurrentAppendResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// Create creates a file with optionally specified content. NOTE: If content is provided, the resulting file cannot be
// modified using ConcurrentAppend.
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the file to create. write is flag to skip redirection. When write=false or not
// specified, the request is redirected. Submit another HTTP PUT request using the URL in the Location header with the
// file data to be written. When write=true, this redirection is skipped. op is the constant value for the operation.
// streamContents is the file contents to include when creating the file. This parameter is optional, resulting in an
// empty file if not specified. streamContents will be closed upon successful return. Callers should ensure closure
// when receiving an error.overwrite is the indication of if the file should be overwritten. syncFlag is optionally
// indicates what to do after completion of the create. DATA indicates that more data will be sent immediately by the
// client, the file handle should remain open/locked, and file metadata (including file length, last modified time)
// should NOT get updated. METADATA indicates that more data will be sent immediately by the client, the file handle
// should remain open/locked, and file metadata should get updated. CLOSE indicates that the client is done sending
// data, the file handle should be closed/unlocked, and file metadata should get updated. leaseID is optional unique
// GUID per file to ensure single writer semantics, meaning that only clients that append to the file with the same
// leaseId will be allowed to do so. permission is the octal representation of the unnamed user, mask and other
// permissions that should be set for the file when created. If not specified, it inherits these from the container.
func (client GroupClient) Create(accountName string, pathParameter string, write string, op string, streamContents io.ReadCloser, overwrite *bool, syncFlag SyncFlag, leaseID *uuid.UUID, permission *int32) (result autorest.Response, err error) {
	req, err := client.CreatePreparer(accountName, pathParameter, write, op, streamContents, overwrite, syncFlag, leaseID, permission)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "Create", nil, "Failure preparing request")
		return
	}

	resp, err := client.CreateSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "Create", resp, "Failure sending request")
		return
	}

	result, err = client.CreateResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "Create", resp, "Failure responding to request")
	}

	return
}

// CreatePreparer prepares the Create request.
func (client GroupClient) CreatePreparer(accountName string, pathParameter string, write string, op string, streamContents io.ReadCloser, overwrite *bool, syncFlag SyncFlag, leaseID *uuid.UUID, permission *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", op),
		"write":       autorest.Encode("query", write),
	}
	if overwrite != nil {
		queryParameters["overwrite"] = autorest.Encode("query", *overwrite)
	}
	if len(string(syncFlag)) > 0 {
		queryParameters["syncFlag"] = autorest.Encode("query", syncFlag)
	}
	if leaseID != nil {
		queryParameters["leaseId"] = autorest.Encode("query", *leaseID)
	}
	if permission != nil {
		queryParameters["permission"] = autorest.Encode("query", *permission)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	if streamContents != nil {
		preparer = autorest.DecoratePreparer(preparer,
			autorest.WithJSON(streamContents))
	}
	return preparer.Prepare(&http.Request{})
}

// CreateSender sends the Create request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) CreateSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// CreateResponder handles the response to the Create request. The method always
// closes the http.Response Body.
func (client GroupClient) CreateResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK, http.StatusCreated),
		autorest.ByClosing())
	result.Response = resp
	return
}

// Delete deletes the requested file or directory, optionally recursively.
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the file or directory to delete. op is the constant value for the operation.
// recursive is the optional switch indicating if the delete should be recursive
func (client GroupClient) Delete(accountName string, pathParameter string, op string, recursive *bool) (result FileOperationResult, err error) {
	req, err := client.DeletePreparer(accountName, pathParameter, op, recursive)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "Delete", nil, "Failure preparing request")
		return
	}

	resp, err := client.DeleteSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "Delete", resp, "Failure sending request")
		return
	}

	result, err = client.DeleteResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "Delete", resp, "Failure responding to request")
	}

	return
}

// DeletePreparer prepares the Delete request.
func (client GroupClient) DeletePreparer(accountName string, pathParameter string, op string, recursive *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", op),
	}
	if recursive != nil {
		queryParameters["recursive"] = autorest.Encode("query", *recursive)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsDelete(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// DeleteSender sends the Delete request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) DeleteSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// DeleteResponder handles the response to the Delete request. The method always
// closes the http.Response Body.
func (client GroupClient) DeleteResponder(resp *http.Response) (result FileOperationResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetACLStatus gets Access Control List (ACL) entries for the specified file or directory.
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the file or directory for which to get the ACL. op is the constant value for the
// operation. tooID is an optional switch to return friendly names in place of object ID for ACL entries. tooid=false
// returns friendly names instead of the AAD Object ID. Default value is true, returning AAD object IDs.
func (client GroupClient) GetACLStatus(accountName string, pathParameter string, op string, tooID *bool) (result ACLStatusResult, err error) {
	req, err := client.GetACLStatusPreparer(accountName, pathParameter, op, tooID)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "GetACLStatus", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetACLStatusSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "GetACLStatus", resp, "Failure sending request")
		return
	}

	result, err = client.GetACLStatusResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "GetACLStatus", resp, "Failure responding to request")
	}

	return
}

// GetACLStatusPreparer prepares the GetACLStatus request.
func (client GroupClient) GetACLStatusPreparer(accountName string, pathParameter string, op string, tooID *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", op),
	}
	if tooID != nil {
		queryParameters["tooId"] = autorest.Encode("query", *tooID)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// GetACLStatusSender sends the GetACLStatus request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) GetACLStatusSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetACLStatusResponder handles the response to the GetACLStatus request. The method always
// closes the http.Response Body.
func (client GroupClient) GetACLStatusResponder(resp *http.Response) (result ACLStatusResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetContentSummary gets the file content summary object specified by the file path.
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the file for which to retrieve the summary. op is the constant value for the
// operation.
func (client GroupClient) GetContentSummary(accountName string, pathParameter string, op string) (result ContentSummaryResult, err error) {
	req, err := client.GetContentSummaryPreparer(accountName, pathParameter, op)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "GetContentSummary", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetContentSummarySender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "GetContentSummary", resp, "Failure sending request")
		return
	}

	result, err = client.GetContentSummaryResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "GetContentSummary", resp, "Failure responding to request")
	}

	return
}

// GetContentSummaryPreparer prepares the GetContentSummary request.
func (client GroupClient) GetContentSummaryPreparer(accountName string, pathParameter string, op string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", op),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// GetContentSummarySender sends the GetContentSummary request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) GetContentSummarySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetContentSummaryResponder handles the response to the GetContentSummary request. The method always
// closes the http.Response Body.
func (client GroupClient) GetContentSummaryResponder(resp *http.Response) (result ContentSummaryResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// GetFileStatus get the file status object specified by the file path.
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the file or directory for which to retrieve the status. op is the constant value
// for the operation. tooID is an optional switch to return friendly names in place of owner and group. tooid=false
// returns friendly names instead of the AAD Object ID. Default value is true, returning AAD object IDs.
func (client GroupClient) GetFileStatus(accountName string, pathParameter string, op string, tooID *bool) (result FileStatusResult, err error) {
	req, err := client.GetFileStatusPreparer(accountName, pathParameter, op, tooID)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "GetFileStatus", nil, "Failure preparing request")
		return
	}

	resp, err := client.GetFileStatusSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "GetFileStatus", resp, "Failure sending request")
		return
	}

	result, err = client.GetFileStatusResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "GetFileStatus", resp, "Failure responding to request")
	}

	return
}

// GetFileStatusPreparer prepares the GetFileStatus request.
func (client GroupClient) GetFileStatusPreparer(accountName string, pathParameter string, op string, tooID *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", op),
	}
	if tooID != nil {
		queryParameters["tooId"] = autorest.Encode("query", *tooID)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// GetFileStatusSender sends the GetFileStatus request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) GetFileStatusSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// GetFileStatusResponder handles the response to the GetFileStatus request. The method always
// closes the http.Response Body.
func (client GroupClient) GetFileStatusResponder(resp *http.Response) (result FileStatusResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// ListFileStatus get the list of file status objects specified by the file path, with optional pagination parameters
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the directory to list. op is the constant value for the operation. listSize is
// gets or sets the number of items to return. Optional. listAfter is gets or sets the item or lexographical index
// after which to begin returning results. For example, a file list of 'a','b','d' and listAfter='b' will return 'd',
// and a listAfter='c' will also return 'd'. Optional. listBefore is gets or sets the item or lexographical index
// before which to begin returning results. For example, a file list of 'a','b','d' and listBefore='d' will return
// 'a','b', and a listBefore='c' will also return 'a','b'. Optional. tooID is an optional switch to return friendly
// names in place of owner and group. tooid=false returns friendly names instead of the AAD Object ID. Default value is
// true, returning AAD object IDs.
func (client GroupClient) ListFileStatus(accountName string, pathParameter string, op string, listSize *int32, listAfter string, listBefore string, tooID *bool) (result FileStatusesResult, err error) {
	req, err := client.ListFileStatusPreparer(accountName, pathParameter, op, listSize, listAfter, listBefore, tooID)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "ListFileStatus", nil, "Failure preparing request")
		return
	}

	resp, err := client.ListFileStatusSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "ListFileStatus", resp, "Failure sending request")
		return
	}

	result, err = client.ListFileStatusResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "ListFileStatus", resp, "Failure responding to request")
	}

	return
}

// ListFileStatusPreparer prepares the ListFileStatus request.
func (client GroupClient) ListFileStatusPreparer(accountName string, pathParameter string, op string, listSize *int32, listAfter string, listBefore string, tooID *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", op),
	}
	if listSize != nil {
		queryParameters["listSize"] = autorest.Encode("query", *listSize)
	}
	if len(listAfter) > 0 {
		queryParameters["listAfter"] = autorest.Encode("query", listAfter)
	}
	if len(listBefore) > 0 {
		queryParameters["listBefore"] = autorest.Encode("query", listBefore)
	}
	if tooID != nil {
		queryParameters["tooId"] = autorest.Encode("query", *tooID)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// ListFileStatusSender sends the ListFileStatus request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) ListFileStatusSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ListFileStatusResponder handles the response to the ListFileStatus request. The method always
// closes the http.Response Body.
func (client GroupClient) ListFileStatusResponder(resp *http.Response) (result FileStatusesResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// Mkdirs creates a directory.
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the directory to create. op is the constant value for the operation. permission is
// optional octal permission with which the directory should be created.
func (client GroupClient) Mkdirs(accountName string, pathParameter string, op string, permission *int32) (result FileOperationResult, err error) {
	req, err := client.MkdirsPreparer(accountName, pathParameter, op, permission)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "Mkdirs", nil, "Failure preparing request")
		return
	}

	resp, err := client.MkdirsSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "Mkdirs", resp, "Failure sending request")
		return
	}

	result, err = client.MkdirsResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "Mkdirs", resp, "Failure responding to request")
	}

	return
}

// MkdirsPreparer prepares the Mkdirs request.
func (client GroupClient) MkdirsPreparer(accountName string, pathParameter string, op string, permission *int32) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", op),
	}
	if permission != nil {
		queryParameters["permission"] = autorest.Encode("query", *permission)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// MkdirsSender sends the Mkdirs request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) MkdirsSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// MkdirsResponder handles the response to the Mkdirs request. The method always
// closes the http.Response Body.
func (client GroupClient) MkdirsResponder(resp *http.Response) (result FileOperationResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// ModifyACLEntries modifies existing Access Control List (ACL) entries on a file or folder.
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the file or directory with the ACL being modified. aclspec is the ACL
// specification included in ACL modification operations in the format '[default:]user|group|other::r|-w|-x|-' op is
// the constant value for the operation.
func (client GroupClient) ModifyACLEntries(accountName string, pathParameter string, aclspec string, op string) (result autorest.Response, err error) {
	req, err := client.ModifyACLEntriesPreparer(accountName, pathParameter, aclspec, op)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "ModifyACLEntries", nil, "Failure preparing request")
		return
	}

	resp, err := client.ModifyACLEntriesSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "ModifyACLEntries", resp, "Failure sending request")
		return
	}

	result, err = client.ModifyACLEntriesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "ModifyACLEntries", resp, "Failure responding to request")
	}

	return
}

// ModifyACLEntriesPreparer prepares the ModifyACLEntries request.
func (client GroupClient) ModifyACLEntriesPreparer(accountName string, pathParameter string, aclspec string, op string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"aclspec":     autorest.Encode("query", aclspec),
		"api-version": APIVersion,
		"op":          autorest.Encode("query", op),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// ModifyACLEntriesSender sends the ModifyACLEntries request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) ModifyACLEntriesSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// ModifyACLEntriesResponder handles the response to the ModifyACLEntries request. The method always
// closes the http.Response Body.
func (client GroupClient) ModifyACLEntriesResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// MsConcat concatenates the list of source files into the destination file, deleting all source files upon success.
// This method accepts more source file paths than the Concat method. This method and the parameters it accepts are
// subject to change for usability in an upcoming version.
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the destination file resulting from the concatenation. streamContents is a list of
// Data Lake Store paths (starting with '/') of the source files. Must be a comma-separated path list in the format:
// sources=/file/path/1.txt,/file/path/2.txt,/file/path/lastfile.csv streamContents will be closed upon successful
// return. Callers should ensure closure when receiving an error.op is the constant value for the operation.
// deleteSourceDirectory is indicates that as an optimization instead of deleting each individual source stream, delete
// the source stream folder if all streams are in the same folder instead. This results in a substantial performance
// improvement when the only streams in the folder are part of the concatenation operation. WARNING: This includes the
// deletion of any other files that are not source files. Only set this to true when source files are the only files in
// the source directory.
func (client GroupClient) MsConcat(accountName string, pathParameter string, streamContents io.ReadCloser, op string, deleteSourceDirectory *bool) (result autorest.Response, err error) {
	req, err := client.MsConcatPreparer(accountName, pathParameter, streamContents, op, deleteSourceDirectory)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "MsConcat", nil, "Failure preparing request")
		return
	}

	resp, err := client.MsConcatSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "MsConcat", resp, "Failure sending request")
		return
	}

	result, err = client.MsConcatResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "MsConcat", resp, "Failure responding to request")
	}

	return
}

// MsConcatPreparer prepares the MsConcat request.
func (client GroupClient) MsConcatPreparer(accountName string, pathParameter string, streamContents io.ReadCloser, op string, deleteSourceDirectory *bool) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", op),
	}
	if deleteSourceDirectory != nil {
		queryParameters["deleteSourceDirectory"] = autorest.Encode("query", *deleteSourceDirectory)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPost(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithFile(streamContents),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// MsConcatSender sends the MsConcat request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) MsConcatSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// MsConcatResponder handles the response to the MsConcat request. The method always
// closes the http.Response Body.
func (client GroupClient) MsConcatResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// Open opens and reads from the specified file.
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the file to open. read is flag to skip redirection. When read=false or not
// specified, the request is redirected. Submit another HTTP PUT request using the URL in the Location header with the
// file data to be read. When read=true, this redirection is skipped. op is the constant value for the operation.
// length is the number of bytes that the server will attempt to retrieve. It will retrieve <= length bytes. offset is
// the byte offset to start reading data from. fileSessionID is optional unique GUID per file indicating all the reads
// with the same fileSessionId are from the same client and same session. This will give a performance benefit.
func (client GroupClient) Open(accountName string, pathParameter string, read string, op string, length *int64, offset *int64, fileSessionID *uuid.UUID) (result ReadCloser, err error) {
	req, err := client.OpenPreparer(accountName, pathParameter, read, op, length, offset, fileSessionID)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "Open", nil, "Failure preparing request")
		return
	}

	resp, err := client.OpenSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "Open", resp, "Failure sending request")
		return
	}

	result, err = client.OpenResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "Open", resp, "Failure responding to request")
	}

	return
}

// OpenPreparer prepares the Open request.
func (client GroupClient) OpenPreparer(accountName string, pathParameter string, read string, op string, length *int64, offset *int64, fileSessionID *uuid.UUID) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", op),
		"read":        autorest.Encode("query", read),
	}
	if length != nil {
		queryParameters["length"] = autorest.Encode("query", *length)
	}
	if offset != nil {
		queryParameters["offset"] = autorest.Encode("query", *offset)
	}
	if fileSessionID != nil {
		queryParameters["fileSessionId"] = autorest.Encode("query", *fileSessionID)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsGet(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// OpenSender sends the Open request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) OpenSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// OpenResponder handles the response to the Open request. The method always
// closes the http.Response Body.
func (client GroupClient) OpenResponder(resp *http.Response) (result ReadCloser, err error) {
	result.Value = &resp.Body
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK))
	result.Response = autorest.Response{Response: resp}
	return
}

// RemoveACL removes the existing Access Control List (ACL) of the specified file or directory.
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the file or directory with the ACL being removed. op is the constant value for the
// operation.
func (client GroupClient) RemoveACL(accountName string, pathParameter string, op string) (result autorest.Response, err error) {
	req, err := client.RemoveACLPreparer(accountName, pathParameter, op)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "RemoveACL", nil, "Failure preparing request")
		return
	}

	resp, err := client.RemoveACLSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "RemoveACL", resp, "Failure sending request")
		return
	}

	result, err = client.RemoveACLResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "RemoveACL", resp, "Failure responding to request")
	}

	return
}

// RemoveACLPreparer prepares the RemoveACL request.
func (client GroupClient) RemoveACLPreparer(accountName string, pathParameter string, op string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", op),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// RemoveACLSender sends the RemoveACL request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) RemoveACLSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RemoveACLResponder handles the response to the RemoveACL request. The method always
// closes the http.Response Body.
func (client GroupClient) RemoveACLResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// RemoveACLEntries removes existing Access Control List (ACL) entries for a file or folder.
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the file or directory with the ACL being removed. aclspec is the ACL spec included
// in ACL removal operations in the format '[default:]user|group|other' op is the constant value for the operation.
func (client GroupClient) RemoveACLEntries(accountName string, pathParameter string, aclspec string, op string) (result autorest.Response, err error) {
	req, err := client.RemoveACLEntriesPreparer(accountName, pathParameter, aclspec, op)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "RemoveACLEntries", nil, "Failure preparing request")
		return
	}

	resp, err := client.RemoveACLEntriesSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "RemoveACLEntries", resp, "Failure sending request")
		return
	}

	result, err = client.RemoveACLEntriesResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "RemoveACLEntries", resp, "Failure responding to request")
	}

	return
}

// RemoveACLEntriesPreparer prepares the RemoveACLEntries request.
func (client GroupClient) RemoveACLEntriesPreparer(accountName string, pathParameter string, aclspec string, op string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"aclspec":     autorest.Encode("query", aclspec),
		"api-version": APIVersion,
		"op":          autorest.Encode("query", op),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// RemoveACLEntriesSender sends the RemoveACLEntries request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) RemoveACLEntriesSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RemoveACLEntriesResponder handles the response to the RemoveACLEntries request. The method always
// closes the http.Response Body.
func (client GroupClient) RemoveACLEntriesResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// RemoveDefaultACL removes the existing Default Access Control List (ACL) of the specified directory.
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the directory with the default ACL being removed. op is the constant value for the
// operation.
func (client GroupClient) RemoveDefaultACL(accountName string, pathParameter string, op string) (result autorest.Response, err error) {
	req, err := client.RemoveDefaultACLPreparer(accountName, pathParameter, op)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "RemoveDefaultACL", nil, "Failure preparing request")
		return
	}

	resp, err := client.RemoveDefaultACLSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "RemoveDefaultACL", resp, "Failure sending request")
		return
	}

	result, err = client.RemoveDefaultACLResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "RemoveDefaultACL", resp, "Failure responding to request")
	}

	return
}

// RemoveDefaultACLPreparer prepares the RemoveDefaultACL request.
func (client GroupClient) RemoveDefaultACLPreparer(accountName string, pathParameter string, op string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", op),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// RemoveDefaultACLSender sends the RemoveDefaultACL request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) RemoveDefaultACLSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RemoveDefaultACLResponder handles the response to the RemoveDefaultACL request. The method always
// closes the http.Response Body.
func (client GroupClient) RemoveDefaultACLResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// Rename rename a file or directory.
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the file or directory to move/rename. destination is the path to move/rename the
// file or folder to op is the constant value for the operation.
func (client GroupClient) Rename(accountName string, pathParameter string, destination string, op string) (result FileOperationResult, err error) {
	req, err := client.RenamePreparer(accountName, pathParameter, destination, op)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "Rename", nil, "Failure preparing request")
		return
	}

	resp, err := client.RenameSender(req)
	if err != nil {
		result.Response = autorest.Response{Response: resp}
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "Rename", resp, "Failure sending request")
		return
	}

	result, err = client.RenameResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "Rename", resp, "Failure responding to request")
	}

	return
}

// RenamePreparer prepares the Rename request.
func (client GroupClient) RenamePreparer(accountName string, pathParameter string, destination string, op string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"destination": autorest.Encode("query", destination),
		"op":          autorest.Encode("query", op),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// RenameSender sends the Rename request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) RenameSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// RenameResponder handles the response to the Rename request. The method always
// closes the http.Response Body.
func (client GroupClient) RenameResponder(resp *http.Response) (result FileOperationResult, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByUnmarshallingJSON(&result),
		autorest.ByClosing())
	result.Response = autorest.Response{Response: resp}
	return
}

// SetACL sets the Access Control List (ACL) for a file or folder.
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the file or directory on which to set the ACL. aclspec is the ACL spec included in
// ACL creation operations in the format '[default:]user|group|other::r|-w|-x|-' op is the constant value for the
// operation.
func (client GroupClient) SetACL(accountName string, pathParameter string, aclspec string, op string) (result autorest.Response, err error) {
	req, err := client.SetACLPreparer(accountName, pathParameter, aclspec, op)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "SetACL", nil, "Failure preparing request")
		return
	}

	resp, err := client.SetACLSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "SetACL", resp, "Failure sending request")
		return
	}

	result, err = client.SetACLResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "SetACL", resp, "Failure responding to request")
	}

	return
}

// SetACLPreparer prepares the SetACL request.
func (client GroupClient) SetACLPreparer(accountName string, pathParameter string, aclspec string, op string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"aclspec":     autorest.Encode("query", aclspec),
		"api-version": APIVersion,
		"op":          autorest.Encode("query", op),
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// SetACLSender sends the SetACL request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) SetACLSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// SetACLResponder handles the response to the SetACL request. The method always
// closes the http.Response Body.
func (client GroupClient) SetACLResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// SetFileExpiry sets or removes the expiration time on the specified file. This operation can only be executed against
// files. Folders are not supported.
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the file on which to set or remove the expiration time. expiryOption is indicates
// the type of expiration to use for the file: 1. NeverExpire: ExpireTime is ignored. 2. RelativeToNow: ExpireTime is
// an integer in milliseconds representing the expiration date relative to when file expiration is updated. 3.
// RelativeToCreationDate: ExpireTime is an integer in milliseconds representing the expiration date relative to file
// creation. 4. Absolute: ExpireTime is an integer in milliseconds, as a Unix timestamp relative to 1/1/1970 00:00:00.
// op is the constant value for the operation. expireTime is the time that the file will expire, corresponding to the
// ExpiryOption that was set.
func (client GroupClient) SetFileExpiry(accountName string, pathParameter string, expiryOption ExpiryOptionType, op string, expireTime *int64) (result autorest.Response, err error) {
	req, err := client.SetFileExpiryPreparer(accountName, pathParameter, expiryOption, op, expireTime)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "SetFileExpiry", nil, "Failure preparing request")
		return
	}

	resp, err := client.SetFileExpirySender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "SetFileExpiry", resp, "Failure sending request")
		return
	}

	result, err = client.SetFileExpiryResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "SetFileExpiry", resp, "Failure responding to request")
	}

	return
}

// SetFileExpiryPreparer prepares the SetFileExpiry request.
func (client GroupClient) SetFileExpiryPreparer(accountName string, pathParameter string, expiryOption ExpiryOptionType, op string, expireTime *int64) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version":  APIVersion,
		"expiryOption": autorest.Encode("query", expiryOption),
		"op":           autorest.Encode("query", op),
	}
	if expireTime != nil {
		queryParameters["expireTime"] = autorest.Encode("query", *expireTime)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/WebHdfsExt/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// SetFileExpirySender sends the SetFileExpiry request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) SetFileExpirySender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// SetFileExpiryResponder handles the response to the SetFileExpiry request. The method always
// closes the http.Response Body.
func (client GroupClient) SetFileExpiryResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// SetOwner sets the owner of a file or directory.
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the file or directory for which to set the owner. op is the constant value for the
// operation. owner is the AAD Object ID of the user owner of the file or directory. If empty, the property will remain
// unchanged. group is the AAD Object ID of the group owner of the file or directory. If empty, the property will
// remain unchanged.
func (client GroupClient) SetOwner(accountName string, pathParameter string, op string, owner string, group string) (result autorest.Response, err error) {
	req, err := client.SetOwnerPreparer(accountName, pathParameter, op, owner, group)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "SetOwner", nil, "Failure preparing request")
		return
	}

	resp, err := client.SetOwnerSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "SetOwner", resp, "Failure sending request")
		return
	}

	result, err = client.SetOwnerResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "SetOwner", resp, "Failure responding to request")
	}

	return
}

// SetOwnerPreparer prepares the SetOwner request.
func (client GroupClient) SetOwnerPreparer(accountName string, pathParameter string, op string, owner string, group string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", op),
	}
	if len(owner) > 0 {
		queryParameters["owner"] = autorest.Encode("query", owner)
	}
	if len(group) > 0 {
		queryParameters["group"] = autorest.Encode("query", group)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// SetOwnerSender sends the SetOwner request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) SetOwnerSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// SetOwnerResponder handles the response to the SetOwner request. The method always
// closes the http.Response Body.
func (client GroupClient) SetOwnerResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}

// SetPermission sets the permission of the file or folder.
//
// accountName is the Azure Data Lake Store account to execute filesystem operations on. pathParameter is the Data Lake
// Store path (starting with '/') of the file or directory for which to set the permission. op is the constant value
// for the operation. permission is a string representation of the permission (i.e 'rwx'). If empty, this property
// remains unchanged.
func (client GroupClient) SetPermission(accountName string, pathParameter string, op string, permission string) (result autorest.Response, err error) {
	req, err := client.SetPermissionPreparer(accountName, pathParameter, op, permission)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "SetPermission", nil, "Failure preparing request")
		return
	}

	resp, err := client.SetPermissionSender(req)
	if err != nil {
		result.Response = resp
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "SetPermission", resp, "Failure sending request")
		return
	}

	result, err = client.SetPermissionResponder(resp)
	if err != nil {
		err = autorest.NewErrorWithError(err, "filesystem.GroupClient", "SetPermission", resp, "Failure responding to request")
	}

	return
}

// SetPermissionPreparer prepares the SetPermission request.
func (client GroupClient) SetPermissionPreparer(accountName string, pathParameter string, op string, permission string) (*http.Request, error) {
	urlParameters := map[string]interface{}{
		"accountName":             accountName,
		"adlsFileSystemDnsSuffix": client.AdlsFileSystemDNSSuffix,
	}

	pathParameters := map[string]interface{}{
		"path": autorest.Encode("path", pathParameter),
	}

	const APIVersion = "2016-11-01"
	queryParameters := map[string]interface{}{
		"api-version": APIVersion,
		"op":          autorest.Encode("query", op),
	}
	if len(permission) > 0 {
		queryParameters["permission"] = autorest.Encode("query", permission)
	}

	preparer := autorest.CreatePreparer(
		autorest.AsPut(),
		autorest.WithCustomBaseURL("https://{accountName}.{adlsFileSystemDnsSuffix}", urlParameters),
		autorest.WithPathParameters("/webhdfs/v1/{path}", pathParameters),
		autorest.WithQueryParameters(queryParameters))
	return preparer.Prepare(&http.Request{})
}

// SetPermissionSender sends the SetPermission request. The method will close the
// http.Response Body if it receives an error.
func (client GroupClient) SetPermissionSender(req *http.Request) (*http.Response, error) {
	return autorest.SendWithSender(client,
		req,
		autorest.DoRetryForStatusCodes(client.RetryAttempts, client.RetryDuration, autorest.StatusCodesForRetry...))
}

// SetPermissionResponder handles the response to the SetPermission request. The method always
// closes the http.Response Body.
func (client GroupClient) SetPermissionResponder(resp *http.Response) (result autorest.Response, err error) {
	err = autorest.Respond(
		resp,
		client.ByInspecting(),
		azure.WithErrorUnlessStatusCode(http.StatusOK),
		autorest.ByClosing())
	result.Response = resp
	return
}
