/* rbinfo.c
 *
 * The main() routine for the rbinfo program.  This takes one or more .rb
 * files and outputs some information about each one.
 */
/* This software is copyrighted as detailed in the LICENSE file. */

#include <config.h>
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif
#include <ctype.h>
#include <rbmake/rbfile.h>

/* Various options */

static bool summarizeMarkup;
static char *libraryDir, *rocketID;

/* Some file-scoped data */

static char opts[] = "ad:D:hmMr:V";

/* Our local functions */

static const char *maybePlural(unsigned int val);
static void usage(void);

int
main(int argc, char *argv[])
{
    int rboFlags = RB_OPENFLAG_INCLUDE_IMAGES | RB_OPENFLAG_INCLUDE_AUDIO;
    bool allInfo = false;
    int ch;

    while ((ch = getopt(argc, argv, opts)) != EOF) {
	switch (ch) {
	  case 'a':
	    allInfo = true;
	    break;
	  case 'd':
	  case 'D': /* Remove soon */
	    libraryDir = optarg;
	    break;
	  case 'm':
	    summarizeMarkup = true;
	    break;
	  case 'M':
	    summarizeMarkup = true;
	    libraryDir = "";
	    rocketID = NULL;
	    break;
	  case 'r':
	    rocketID = optarg;
	    summarizeMarkup = true;
	    break;
	  case 'V':
	    RbError_warn("rbinfo v%s\n", RBMAKE_VERSION);
	    exit(0);
	  case 'h':
	  default:
	    usage();
	}
    }
    if (optind >= argc) {
	RbError_exit("You didn't specify any .rb files to query.  "
		     "Use the -h option to get help.\n");
    }

    for ( ; optind < argc; optind++) {
	RbFile *rf = RbFile_open(optarg = argv[optind], rboFlags);
	if (rf) {
	    ToC *toc;
	    RbInfoHash *ih = RbInfoHash_newFromRbFile(rf);
	    const char *title = RbInfoHash_asciiFetch(ih, "TITLE");
	    const char *author = RbInfoHash_asciiFetch(ih, "AUTHOR");
	    const char *genre = RbInfoHash_asciiFetch(ih, "GENRE");
	    const char *hkey = RbInfoHash_fetch(ih, "HKEY");
	    const char *menumark = RbInfoHash_fetch(ih, "MENUMARK");
	    const char *generator = RbInfoHash_fetch(ih, "GENERATOR");
	    int i, count[RB_PAGETYPE_COUNT], length[RB_PAGETYPE_COUNT];
	    int ulength[RB_PAGETYPE_COUNT];
	    bool encrypted = false;
	    memset(count, 0, sizeof count);
	    memset(length, 0, sizeof length);
	    memset(ulength, 0, sizeof ulength);
	    for (toc = RbFile_getTocHead(rf); (toc = ToC_getNext(toc))!=NULL;) {
		int type = ToC_getType(toc);
		if (type == RB_PAGETYPE_HTML && *ToC_getName(toc) == 'm'
		 && strnEQ(ToC_getName(toc), "menumark", 8))
		    continue;
		count[type]++;
		length[type] += ToC_getLength(toc);
		ulength[type] += RbFile_uncompressedLength(rf, toc);
		if (ToC_getFlags(toc) & RB_TOCFLAG_ENCRYPTED)
		    encrypted = true;
	    }
	    printf("Info for %s:\n\n", optarg);
	    printf("Title: %s\n", title? title : "[NONE]");
	    printf("Author: %s\n", author? author : "[NONE]");
	    if (title)
		Mem_free((void*)title);
	    if (author)
		Mem_free((void*)author);
	    if (genre) {
		printf("Genre: %s\n", genre);
		Mem_free((void*)genre);
	    }
	    if (encrypted)
		printf("Encrypted: yes\n");
	    printf("Generator: %s\n", generator? generator : "[UNKNOWN]");
	    if (hkey)
		printf("Dictionary Support: yes\n");
	    if (menumark)
		printf("Custom Go-To-Menu: yes\n");
	    for (i = RB_PAGETYPE_HTML; i < RB_PAGETYPE_COUNT; i++) {
		if (count[i]) {
		    printf("%d %s file%s ", count[i], RbFile_getTypeName(i),
			   count[i] == 1? "" : "s");
		    if (length[i] == ulength[i])
			printf("(%dK)\n", (length[i] + 512) / 1024);
		    else {
			printf("(%dK compressed to %dK)\n",
			       (ulength[i] + 512) / 1024,
			       (length[i] + 512) / 1024);
		    }
		}
	    }
	    if (summarizeMarkup) {
		char *dir = RbFile_getLibDir(rf, libraryDir, rocketID);
		unsigned int x;
		RbFile_readMarkup(rf, dir);
		Mem_free(dir);
		printf("Markup:");
		i = 0;
		if ((x = RbFile_getUnderlineCnt(rf)) != 0) {
		    printf(" %d underline%s", x, maybePlural(x));
		    i = 1;
		}
		if ((x = RbFile_getNoteCnt(rf)) != 0) {
		    if (i) putchar(',');
		    printf(" %d note%s", x, maybePlural(x));
		    i = 1;
		}
		if ((x = RbFile_getBookmarkCnt(rf)) != 0) {
		    if (i) putchar(',');
		    printf(" %d bookmark%s", x, maybePlural(x));
		    i = 1;
		}
		if (!i)
		    printf(" none");
		printf(".\n");
	    }
	    putchar('\n');
	    if (allInfo) {
		MBuf *mb = RbInfoHash_toMBuf(ih);
		char *bp = MBuf_dataPtr(mb, NULL);
		if (bp) {
		    unenhancePunctuation(bp);
		    printf("%s\n", bp);
		}
		MBuf_delete(mb);
	    }
	    RbInfoHash_delete(ih);
	}
    }
    return 0;
}

static const char *
maybePlural(unsigned int val)
{
    return val == 1? "" : "s";
}

static void
usage()
{
    printf("\
Usage: rbinfo [-OPTIONS] FILE.rb [FILE.rb ...]\n\
\n\
-a  Output all the items from the book's .info page.\n\
-d* Specify the library directory (alternate: $RB_LIB_DIR).\n\
-h  Output this help message.\n\
-m  Summarize markup that is found in the RocketLibrary's directory structure.\n\
    Use the -d and -r options (or set the environment variables) to setup\n\
    where that is.\n\
-M  Summarize markup from the same directory and same name as the .rb file.\n\
    This is useful if you've copied the .ra, .rh, & .rn files into a single\n\
    place.  E.g. \"rbinfo -M path/foo.rb\" would look for path/foo.ra, etc.\n\
-r* The rocket-ID of the ReB that generated the markup (alternate: $RB_ID).\n\
-V  Output the version of rbinfo.\n\
");
    exit(0);
}
