/* rbburst.c
 *
 * The main() routine for the rbburst program.  This takes one or more .rb
 * files and outputs the individual pages that were used to create it.
 */
/* This software is copyrighted as detailed in the LICENSE file. */

#include <config.h>
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif
#include <ctype.h>
#include <rbmake/rbfile.h>

/* Various options */

static bool includeMarkup, rawHtml;
static char *libraryDir, *rocketID;

/* Some file-scoped data */

static char opts[] = "ad:D:hilmMr:RuV";

/* This include defines normalSubst[] and unjoinSubst[] */

#include "rbburst.subst"

MArray *normalRules, *unjoinRules;
MBuf *pageBuf;
char *markupFileName;
FILE *markupFP;
int markupCnt;

/* Our local functions */

static const char *maybePlural(unsigned int val);
void burstPage(RbFile *rf, const char *safeFn, ToC *toc);
void burstNotes(RbFile *rf, const char *safeFn, ToC *toc);
static char *preContext(const char *bp, unsigned int pos);
static char *postContext(const char *bp);
static void usage(void);

int
main(int argc, char *argv[])
{
    bool listFiles = false;
    int rboFlags = RB_OPENFLAG_INCLUDE_IMAGES | RB_OPENFLAG_INCLUDE_AUDIO;
    int ch;

    while ((ch = getopt(argc, argv, opts)) != EOF) {
	switch (ch) {
	  case 'a':
	    rboFlags |= RB_OPENFLAG_INCLUDE_HIDDEN;
	    break;
	  case 'd':
	  case 'D': /* Remove soon */
	    libraryDir = optarg;
	    break;
	  case 'i':
	    rboFlags &= ~(RB_OPENFLAG_INCLUDE_IMAGES | RB_OPENFLAG_INCLUDE_AUDIO);
	    break;
	  case 'l':
	    listFiles = true;
	    break;
	  case 'm':
	    includeMarkup = true;
	    break;
	  case 'M':
	    includeMarkup = true;
	    libraryDir = "";
	    rocketID = NULL;
	    break;
	  case 'r':
	    rocketID = optarg;
	    includeMarkup = true;
	    break;
	  case 'R':
	    rawHtml = true;
	    break;
	  case 'u':
	    rboFlags |= RB_OPENFLAG_UNJOIN;
	    break;
	  case 'V':
	    RbError_warn("rbburst v%s\n", RBMAKE_VERSION);
	    exit(0);
	  case 'h':
	  default:
	    usage();
	}
    }
    if (optind >= argc) {
	RbError_exit("You didn't specify any .rb files to extract.  "
		     "Use the -h option to get help.\n");
    }

    if (!listFiles) {
	char *err = NULL;

	if (!rawHtml) {
	    normalRules = MArray_new(2, 0);
	    err = Subst_parseRules(normalRules, normalSubst);
	}

	if (!err && !rawHtml && (rboFlags & RB_OPENFLAG_UNJOIN)) {
	    unjoinRules = MArray_new(2, 0);
	    err = Subst_parseRules(unjoinRules, unjoinSubst);
	}

	if (err)
	    RbError_exit("Internal error with Subst_parseRules():\n%s\n", err);

	pageBuf = MBuf_new(8192, 0);
    }

    for ( ; optind < argc; optind++) {
	RbFile *rf = RbFile_open(optarg = argv[optind], rboFlags);
	if (rf) {
	    ToC *toc;
	    markupFileName = NULL;
	    markupFP = NULL;
	    markupCnt = 0;
	    printf("Processing %s\n", optarg);
	    if (includeMarkup) {
		char *dir = RbFile_getLibDir(rf, libraryDir, rocketID);
		RbFile_readMarkup(rf, dir);
		Mem_free(dir);
		if (RbFile_getUnderlineCnt(rf) || RbFile_getBookmarkCnt(rf)
		 || RbFile_getNoteCnt(rf)) {
		    char *rbfn, *suf, *arg = Mem_strdup(optarg);
		    MBuf *tmp;
#ifdef DOS_FILENAMES
		    for (suf = arg; *suf; suf++) {
			if (*suf == '\\')
			    *suf = '/';
		    }
#endif
		    if (!(rbfn = strrchr(arg, '/')))
			rbfn = arg;
		    else
			rbfn++;
		    suf = rbGetFileSuffix(rbfn);
		    tmp = MBuf_new(128, 0);
		    MBuf_vwrite(tmp, "markup_",7, rbfn,suf-rbfn-1, ".html",5,
				NULL);
		    markupFileName = MBuf_toBuffer(tmp, NULL);
		    if (!listFiles) {
			if (!(markupFP = fopen(markupFileName, "w"))) {
			    RbError_exit("Unable to create markup file: %s\n",
					 markupFileName);
			}
			fprintf(markupFP, "<HTML><HEAD><TITLE>Markup for %s</TITLE></HEAD><BODY><H1>Markup for %s</H1><OL>\n",
				rbfn, rbfn);
		    }
		}
	    }
	    for (toc = RbFile_getTocHead(rf); toc; toc = ToC_getNext(toc)) {
		if (listFiles) {
		    printf("  %s (length %d, type %d)", ToC_getName(toc),
			   ToC_getLength(toc), ToC_getFlags(toc));
		}
		else {
		    char *t, safeFn[RB_TOC_NAMELEN+1];
		    const char *f;

		    for (t = safeFn, f = ToC_getName(toc); *f; f++) {
			if (ISALNUM(*f) || strchr("@#-+.", *f) != NULL)
			    *t++ = *f;
			else
			    *t++ = '_';
		    }
		    *t = '\0';
		    if (ToC_getFlags(toc) & RB_TOCFLAG_ENCRYPTED) {
			printf("Skipping %s (length %d, type %d)",
			       safeFn, ToC_getLength(toc), ToC_getFlags(toc));
			if (ToC_getMarkupHead(toc))
			    burstNotes(rf, safeFn, toc);
		    }
		    else {
			printf("Writing %s (length %d, type %d)",
			       safeFn, ToC_getLength(toc), ToC_getFlags(toc));
			burstPage(rf, safeFn, toc);
		    }
		}
		if (ToC_getMarkupHead(toc)) {
		    int h = ToC_getUnderlineCnt(toc);
		    int n = ToC_getNoteCnt(toc);
		    int a = ToC_getBookmarkCnt(toc); 
		    printf(" [%d ul%s, %d note%s, and %d bm%s]",
			   h, maybePlural(h), n, maybePlural(n),
			   a, maybePlural(a));
		}
		putchar('\n');
	    }
	    if (markupFileName) {
		int h = RbFile_getUnderlineCnt(rf);
		int n = RbFile_getNoteCnt(rf);
		int a = RbFile_getBookmarkCnt(rf); 
		printf("%s: %d underline%s, %d note%s, and %d bookmark%s\n",
		       listFiles? "Total" : "Wrote",
		       h, maybePlural(h), n, maybePlural(n), a, maybePlural(a));
		if (!listFiles) {
		    fprintf(markupFP, "</OL></BODY></HTML>\n");
		    fclose(markupFP);
		    printf("into %s.\n", markupFileName);
		}
	    }
	}
    }
    return 0;
}

void
burstPage(RbFile *rf, const char *safeFn, ToC *toc)
{
    char type, buf[1024];
    FILE *fp;
    char *openAtts = rbPageTypeIsBinary(ToC_getType(toc))? "wb" : "w";

    if ((fp = fopen(safeFn, openAtts)) == NULL)
	RbError_exit("Unable to write %s\n", safeFn);

    MBuf_truncate(pageBuf, 0);
    RbFile_readPage(rf, toc, pageBuf, NULL);
    if (ToC_getMarkupHead(toc)) {
	int alen, blen = MBuf_getLength(pageBuf);
	char *bp = MBuf_toBuffer(pageBuf, &alen);
	MArray *endUnderline = MArray_new(64, 1024);
	int j, x, pos, endUnderPos = 0;
	RbMarkup *mu;

	RbMarkup_fixMarkups(toc, bp);
	Subst_initChangeset(true);
	for (mu = ToC_getMarkupHead(toc); mu; mu = RbMarkup_getNext(mu)) {
	    while (endUnderPos < MArray_itemCnt(endUnderline)) {
		pos = MArray_fetchAt(endUnderline, endUnderPos);
		if (pos > RbMarkup_getStart(mu))
		    break;
		Subst_addChange(bp, pos, 0, "</U>", 4);
		endUnderPos++;
	    }

	    switch (type = RbMarkup_getType(mu)) {
	      case RB_MARKUP_UNDERLINE:
		markupCnt++;
		sprintf(buf, "<A NAME='%c%d' HREF='%s#%c%d' Rel=%c>[%d]</A> <U>",
			type, markupCnt, markupFileName, type, markupCnt, type,
			markupCnt);
		Subst_addChange(bp, RbMarkup_getStart(mu), 0, buf, strlen(buf));
		pos = RbMarkup_getEnd(mu)+1;
		MArray_append(endUnderline, pos);
		for (j = MArray_itemCnt(endUnderline); j > endUnderPos; j--) {
		    if ((x = MArray_fetchAt(endUnderline, j-2)) <= pos)
			break;
		    MArray_storeAt(endUnderline, j-1, x);
		    MArray_storeAt(endUnderline, j-2, pos);
		}
		fprintf(markupFP, "<LI><A NAME='%c%d' HREF='%s#%c%d'><B>Underline</B></A> (for %s)<BR>%s<U>%.*s</U>%s<BR><HR>\n",
			type, markupCnt, safeFn, type, markupCnt,
			safeFn, preContext(bp, RbMarkup_getStart(mu)),
			(int)(RbMarkup_getEnd(mu) - RbMarkup_getStart(mu)) + 1,
			bp + RbMarkup_getStart(mu),
			postContext(bp + RbMarkup_getEnd(mu)+1));
		break;
	      case RB_MARKUP_NOTE:
		markupCnt++;
		sprintf(buf, " <A NAME='%c%d' HREF='%s#%c%d' Rel=%c>[%d: Note]</A>",
			type, markupCnt, markupFileName, type, markupCnt,
			type, markupCnt);
		Subst_addChange(bp, RbMarkup_getStart(mu), 0, buf, strlen(buf));
		fprintf(markupFP, "<LI><A NAME='%c%d' HREF='%s#%c%d'><B>Note</B></A> (for %s)<BR>%s <U><B>[</B>%s<B>]</B></U> %s<BR><HR>\n",
			type, markupCnt, safeFn, type, markupCnt,
			safeFn, preContext(bp, RbMarkup_getStart(mu)),
			RbMarkup_getText(mu),
			postContext(bp + RbMarkup_getStart(mu)));
		break;
	      case RB_MARKUP_BOOKMARK:
		markupCnt++;
		sprintf(buf, " <A NAME='%c%d' HREF='%s#%c%d' Rel=%c>[%d: Bookmark]</A>",
			type, markupCnt, markupFileName, type, markupCnt,
			type, markupCnt);
		Subst_addChange(bp, RbMarkup_getStart(mu), 0, buf, strlen(buf));
		fprintf(markupFP, "<LI><A NAME='%c%d' HREF='%s#%c%d'><B>Bookmark</B></A> (for %s)<BR><B>\"</B>%s<B>\"</B><BR><HR>\n",
			type, markupCnt, safeFn, type, markupCnt, safeFn,
			RbMarkup_getText(mu));
		break;
	    }
	}
	MArray_setFetchPos(endUnderline, endUnderPos);
	while ((pos = MArray_fetch(endUnderline)) != 0)
	    Subst_addChange(bp, pos, 0, "</U>", 4);
	MArray_delete(endUnderline);
	Subst_applyChangeset(&bp, &blen, &alen);
	pageBuf = MBuf_new(8192, 0);
	MBuf_appendBuffer(pageBuf, bp, blen, alen);
    }

    if (unjoinRules)
	pageBuf = Subst_runRules(unjoinRules, ToC_getName(toc), pageBuf);

    if (normalRules)
	pageBuf = Subst_runRules(normalRules, ToC_getName(toc), pageBuf);

    fwrite(MBuf_dataPtr(pageBuf, NULL), 1, MBuf_getLength(pageBuf), fp);
    fclose(fp);
}

/* This routine is used when we can't access the page's content. */
void
burstNotes(RbFile *rf, const char *safeFn, ToC *toc)
{
    RbMarkup *mu;
    const char *txt;

    for (mu = ToC_getMarkupHead(toc); mu; mu = RbMarkup_getNext(mu)) {
	txt = RbMarkup_getText(mu);
	switch (RbMarkup_getType(mu)) {
	  case RB_MARKUP_UNDERLINE:
	  case RB_MARKUP_UNDERLINE2:
	    fprintf(markupFP, "<LI><B>Underline from pos %d to %d</B> (for %s)<BR><U>%s</U><BR><HR>\n",
		    RbMarkup_getStart(mu), RbMarkup_getEnd(mu), safeFn,
		    txt? txt : "[Not specified]");
	    break;
	  case RB_MARKUP_NOTE:
	  case RB_MARKUP_NOTE2:
	    fprintf(markupFP, "<LI><B>Note at pos %d</B> (for %s)<BR><U><B>[</B>%s<B>]</B></U><BR><HR>\n",
		    RbMarkup_getStart(mu), safeFn, txt);
	    break;
	  case RB_MARKUP_BOOKMARK:
	    fprintf(markupFP, "<LI><B>Bookmark at pos %d</B> (for %s)<BR><B>\"</B>%s<B>\"</B><BR><HR>\n",
		    RbMarkup_getStart(mu), safeFn, txt);
	    break;
	}
    }
}

static char *
preContext(const char *bp, unsigned int pos)
{
    static char buf[512];
    char *t, *cp;
    int len = sizeof buf > pos? pos : sizeof buf - 1, want = 64, put = 0;

    for (t = buf, bp += pos - len; *bp && len > 0; bp++, len--) {
	switch (*bp) {
	  case '&':
	    if (!(cp = strchr(bp, ';')) || cp - bp > len)
		goto finish;
	    while (*bp != ';') *t++ = *bp++, len--;
	    *t++ = ';';
	    put++;
	    break;
	  case '<':
	    if (!(cp = strchr(bp, '>')))
		goto finish;
	    if ((strncaseEQ(bp, "<P", 2) && !ISALPHA(bp[2]))
	     || (strncaseEQ(bp, "<H", 2) && ISDIGIT(bp[2]))
	     || strncaseEQ(bp, "<BR", 3)
	     || strncaseEQ(bp, "<DIV", 4))
		t = buf, put = 0;
	    len -= cp - bp;
	    bp = cp;
	    break;
	  default:
	    *t++ = *bp;
	    put++;
	    break;
	}
    }

  finish:
    *t = '\0';

    for (cp = buf; put > want; cp++) {
	switch (*cp) {
	  case '&':
	    cp = strchr(cp, ';');
	    put--;
	    break;
	  case '<':
	    cp = strchr(cp, '>');
	    break;
	  default:
	    put--;
	    break;
	}
    }

    return cp;
}

static char *
postContext(const char *bp)
{
    static char buf[512];
    char *t, *cp;
    int len, want = 64;

    for (t = buf, len = sizeof buf - 1; *bp && want>0 && len>0; bp++, len--) {
	switch (*bp) {
	  case '&':
	    if (!(cp = strchr(bp, ';')) || cp - bp > len)
		goto finish;
	    while (*bp != ';') *t++ = *bp++, len--;
	    *t++ = ';';
	    want--;
	    break;
	  case '<':
	    if (!(cp = strchr(bp, '>')))
		goto finish;
	    if (bp[1] == '/') {
		if ((strncaseEQ(bp, "</P", 3) && !ISALPHA(bp[3]))
		 || strncaseEQ(bp, "</DIV", 5))
		    goto finish;
		if (strncaseEQ(bp, "</H", 3) && ISDIGIT(bp[3])) {
		    if (cp - bp > len)
			goto finish;
		    while (*bp != '>') *t++ = *bp++, len--;
		    *t++ = '>';
		    goto finish;
		}
		if (strncaseEQ(bp, "<BR", 3))
		    goto finish;
	    }
	    bp = cp;
	    break;
	  default:
	    *t++ = *bp;
	    want--;
	    break;
	}
    }

  finish:
    *t = '\0';

    return buf;
}

static const char *
maybePlural(unsigned int val)
{
    return val == 1? "" : "s";
}

static void
usage()
{
    printf("\
Usage: rbburst [-OPTIONS] FILE.rb [FILE.rb ...]\n\
\n\
-a  Extract even the hidden files from the .rb file.\n\
-d* Specify the library directory (alternate: $RB_LIB_DIR).\n\
-h  Output this help message.\n\
-i  Ignore the images and audio files.\n\
-l  Just list the files (doesn't write anything).\n\
-m  Include markup that is found in the RocketLibrary's directory structure.\n\
    Use the -d and -r options (or set the environment variables) to setup\n\
    where that is.  The markup is merged into the HTML pages, a summary page\n\
    is produced, and everything is hyper-linked together.\n\
-M  Include markup from the same directory and same name as the .rb file.\n\
    This is useful if you've copied the .ra, .rh, & .rn files into a single\n\
    place.  E.g. \"rbburst -M path/foo.rb\" would look for path/foo.ra, etc.\n\
-r* The rocket-ID of the ReB that generated the markup (alternate: $RB_ID).\n\
-R  Output the raw HTML (e.g. don't try to restore page-break tags, etc.).\n\
-u  Unjoin files that rbmake previously joined together.\n\
-V  Output the version of rbburst.\n\
");
    exit(0);
}
