/* rbutil.c
 *
 * Some simple utility routines.
 */
/* This software is copyrighted as detailed in the LICENSE file. */

#include <config.h>
#include <rbmake/rbfile.h>

static MArray *ignoreSufs;

static void ignoreSufsInit();

/* This does not handle any query or fragment URL bits, just pathnames. */
char *
rbGetFileSuffix(const char *fn)
{
    const char *cp = fn + strlen(fn);
    while (cp > fn && *--cp != '/' && *cp != ':') {
	if (*cp == '.')
	    return (char*)cp + 1;
    }
    return NULL;
}

char *
rbGetUrlSuffix(const char *url)
{
    static char buf[32];
    const char *cp, *end;
    int len;

    if ((end = strchr(url, '?')) == NULL && (end = strchr(url, '#')) == NULL)
	end = url + strlen(url);
    for (cp = end; cp > url && *--cp != '/' && *cp != ':'; ) {
	if (*cp == '.') {
	    cp++;
	    if (!*end)
		return (char*)cp;
	    if ((len = end - cp) >= sizeof buf)
		len = sizeof buf - 1;
	    strncpy(buf, cp, len);
	    buf[len] = '\0';
	    return buf;
	}
    }
    return NULL;
}

bool
rbIsHtmlSuf(const char *suf)
{
    return suf && strncaseEQ(suf, "htm", 3)
	&& (suf[3] == '\0' || strcaseEQ(suf+3, "l"));
}

bool
rbIsHidxSuf(const char *suf)
{
    return suf && strcaseEQ(suf, "hidx");
}

bool
rbIsHkeySuf(const char *suf)
{
    return suf && strcaseEQ(suf, "hkey");
}

bool
rbIsTextSuf(const char *suf)
{
    return suf && strcaseEQ(suf, "txt");
}

bool
rbIsImageSuf(const char *suf)
{
    if (suf
     && (strcaseEQ(suf, "jpeg") || strcaseEQ(suf, "jpg")
      || strcaseEQ(suf, "gif")  || strcaseEQ(suf, "png")))
	return 1;
    return 0;
}

bool
rbIsAudioSuf(const char *suf)
{
    return suf && strcaseEQ(suf, "wav");
}

bool
rbIsInfoSuf(const char *suf)
{
    return suf && strcaseEQ(suf, "info");
}

bool
rbIsRbSuf(const char *suf)
{
    return suf && strcaseEQ(suf, "rb");
}

int
rbSuffixToPageType(const char *suf)
{
    if (suf) {
	char *isuf;
	if (!ignoreSufs)
	    ignoreSufsInit();
	MArray_setFetchPos(ignoreSufs, 0);
	while ((isuf = MArray_fetchPtr(ignoreSufs)) != NULL) {
	    if (strcaseEQ(suf, isuf))
		return RB_PAGETYPE_IGNORE;
	}
    }
    if (!suf || !*suf)
	return RB_PAGETYPE_MAYBE_HTML;
    if (rbIsHtmlSuf(suf))
	return RB_PAGETYPE_HTML;
    if (rbIsHidxSuf(suf))
	return RB_PAGETYPE_HIDX;
    if (rbIsImageSuf(suf))
	return RB_PAGETYPE_IMAGE;
    if (rbIsTextSuf(suf))
	return RB_PAGETYPE_TEXT;
    if (rbIsAudioSuf(suf))
	return RB_PAGETYPE_AUDIO;
    if (rbIsInfoSuf(suf))
	return RB_PAGETYPE_INFO;
    if (rbIsHkeySuf(suf))
	return RB_PAGETYPE_HKEY;
    if (rbIsHkeySuf(suf))
	return RB_PAGETYPE_HKEY;
    if (rbIsRbSuf(suf))
	return RB_PAGETYPE_RB;
    return RB_PAGETYPE_UNKNOWN;
}

int
rbFileNameToPageType(const char *fn)
{
    return rbSuffixToPageType(rbGetFileSuffix(fn));
}

int
rbUrlToPageType(const char *url)
{
    return rbSuffixToPageType(rbGetUrlSuffix(url));
}

void
rbAddIgnoreSuf(const char *suf)
{
    if (!ignoreSufs)
	ignoreSufsInit();
    MArray_appendPtr(ignoreSufs, suf);
}

static char *ignoreSufsDefault[] = {
    "zip", "rpm", "deb", "gz", "Z", "bz2", "exe", "com", "doc", "rtf",
    "mp3", "mov", "mpeg", "mpg", "au",
    NULL
};

static void
ignoreSufsInit()
{
    char **cpp;
    ignoreSufs = MArray_new(32, 1024);
    for (cpp = ignoreSufsDefault; *cpp; cpp++)
	MArray_appendPtr(ignoreSufs, *cpp);
}

bool
rbPageTypeIsBinary(int type)
{
    switch (type) {
      case RB_PAGETYPE_HTML:
      case RB_PAGETYPE_TEXT:
      case RB_PAGETYPE_RAW_TEXT:
      case RB_PAGETYPE_INFO:
      case RB_PAGETYPE_HIDX:
      case RB_PAGETYPE_HKEY:
	return false;
    }
    return true;
}

static char b64Chars[]
    = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

void
rbBase64Encode(char *bp, int len)
{
    int b64len = RB_BASE64_LENGTH(len);
    char *t = bp + b64len;
    char *f = bp + b64len / 4 * 3;

    switch (len % 3) {
      case 1:
	f[-2] = '\0';
	/* FALL THROUGH */
      case 2:
	f[-1] = '\0';
	break;
    }
    while (t != bp) {
	unsigned char ch1, ch2, ch3;
	ch3 = *--f; ch2 = *--f; ch1 = *--f;
	*--t = b64Chars[ch3 & 0x3F];
	*--t = b64Chars[(ch2 << 2 | ch3 >> 6) & 0x3F];
	*--t = b64Chars[(ch1 << 4 | ch2 >> 4) & 0x3F];
	*--t = b64Chars[ch1 >> 2];
    }
    t = bp + b64len - 1;
    switch (len % 3) {
      case 1:
	*t-- = '=';
	/* FALL THROUGH */
      case 2:
	*t = '=';
	break;
    }
}
