/* rbinfohash.c
 *
 * This "class" creates a special kind of HashTable that contains the
 * NAME=VALUE pairs that go into a .info file.  These routines are used
 * to translate various sources of .info data into a hash, to merge other
 * sources of .info data into the hash, and to turn the hash back into
 * an MBuf for output.
 */
/* This software is copyrighted as detailed in the LICENSE file. */

#include <config.h>
#include <ctype.h>
#include <rbmake/rbfile.h>
#include "rbfile.h"
#include "hashtable.h"
#include "mbuf.h"

RbInfoHash *
RbInfoHash_newFromMBuf(MBuf *mb)
{
    RbInfoHash *ih = HashTable_new(101, true, true);

    if (mb) {
	char *cp, buf[1024];
	int len;

	MBuf_setReadPos(mb, 0, 0);
	while ((len = MBuf_gets(mb, buf, sizeof buf)) > 0) {
	    for (cp = buf + len; cp != buf && ISSPACE(cp[-1]); cp--) {}
	    if (cp == buf)
		continue;
	    *cp = '\0';
	    if ((cp = strchr(buf, '=')) == NULL)
		continue;
	    *cp = '\0';
	    RbInfoHash_store(ih, buf, cp+1);
	}
    }
    return ih;
}

RbInfoHash *
RbInfoHash_newFromRbFile(RbFile *rb)
{
    MBuf *mb = MBuf_new(rb->tocHead->length + 1, 0);
    RbFile_readPage(rb, rb->tocHead, mb, NULL);
    return RbInfoHash_newFromMBuf(mb);
}

void
RbInfoHash_delete(RbInfoHash *me)
{
    HashTable_delete((HashTable*)me);
}

static int
joinInfo(void *userPtr, const char *key, void *obj)
{
    MBuf *mb = userPtr;
    if (obj)
	MBuf_vwrite(mb, key, -1, "=", 1, obj, -1, "\n", 1, NULL);
    return 1;
}

/* A list of the items that we want in a certain order at the start. */
static char *joinOrder[] = {
    "COMMENT",
    "TITLE",
    "AUTHOR",
    "BODY",
    "URL",
    "GENERATOR",
};

#define ORDER_LEN	((sizeof joinOrder) / (sizeof (char*)))

MBuf *
RbInfoHash_toMBuf(RbInfoHash *me)
{
    char *save[ORDER_LEN];
    MBuf *mb = MBuf_new(1024, 0);
    int i;
    /* Output a few of the items specially */
    me->autoDupData = false;
    me->autoFreeData = false;
    for (i = 0; i < ORDER_LEN; i++) {
	save[i] = RbInfoHash_fetch(me, joinOrder[i]);
	if (save[i]) {
	    MBuf_vwrite(mb, joinOrder[i],-1, "=",1, save[i],-1, "\n",1, NULL);
	    RbInfoHash_store(me, joinOrder[i], NULL);
	}
    }
    RbInfoHash_walk(me, mb, joinInfo);
    for (i = 0; i < ORDER_LEN; i++) {
	if (save[i])
	    RbInfoHash_store(me, joinOrder[i], save[i]);
    }
    me->autoDupData = true;
    me->autoFreeData = true;
    return mb;
}

void *
RbInfoHash_store(RbInfoHash *me, const char *key, void *data)
{
    return HashTable_store((HashTable*)me, key, data);
}

bool
RbInfoHash_maybeStore(RbInfoHash *me, const char *key, void *data)
{
    if (RbInfoHash_fetch(me, key))
	return false;
    RbInfoHash_store(me, key, data);
    return true;
}

void *
RbInfoHash_remove(RbInfoHash *me, const char *key)
{
    return HashTable_remove((HashTable*)me, key);
}

char *
RbInfoHash_fetch(RbInfoHash *me, const char *key)
{
    return (char*)HashTable_fetch((HashTable*)me, key);
}

void
RbInfoHash_walk(RbInfoHash *me, void *userPtr,
		int(*nodeFunc)(void *userPtr, const char *key, void *data))
{
    HashTable_walk((HashTable*)me, userPtr, nodeFunc);
}

const char **
RbInfoHash_keys(RbInfoHash *me)
{
    return HashTable_keys((HashTable*)me);
}

unsigned
RbInfoHash_itemCnt(HashTable *me)
{
    return HashTable_itemCnt((HashTable*)me);
}

static int
mergeInfo(void *userPtr, const char *key, void *obj)
{
    RbInfoHash *me = userPtr;
    if (!RbInfoHash_fetch(me, key)) {
	RbInfoHash_store(me, key, obj);
	return -1;
    }
    return 0;
}

/* This destroys the second RbInfoHash */
void
RbInfoHash_mergeFromHash(RbInfoHash *me, RbInfoHash *ih)
{
    /* Temporarily disable memory dup/free flags so we can transfer data */
    me->autoDupData = false;
    ih->autoFreeData = false;
    RbInfoHash_walk(ih, me, mergeInfo);
    me->autoDupData = true;
    ih->autoFreeData = true;
    RbInfoHash_delete(ih);
}

void
RbInfoHash_mergeFromMBuf(RbInfoHash *me, MBuf *mb)
{
    RbInfoHash *ih = RbInfoHash_newFromMBuf(mb);
    RbInfoHash_mergeFromHash(me, ih);
}

void
RbInfoHash_mergeFromRbFile(RbInfoHash *me, RbFile *rb)
{
    MBuf *mb = MBuf_new(rb->tocHead->length + 1, 0);
    RbInfoHash *ih;
    RbFile_readPage(rb, rb->tocHead, mb, NULL);
    ih = RbInfoHash_newFromMBuf(mb);
    RbInfoHash_mergeFromHash(me, ih);
}

/* This returns memory that needs to be freed by the caller. */
char *
RbInfoHash_asciiFetch(RbInfoHash *me, const char *key)
{
    char *bp, *val = RbInfoHash_fetch(me, key);
    if (!val)
	return NULL;
    bp = Mem_strdup(val);
    unenhancePunctuation(bp);
    return bp;
}

void
unenhancePunctuation(char *bp)
{
    for ( ; *bp; bp++) {
	switch (uc(bp,0)) {
	  case RB_CH_ELIPS:
	    *bp = '.';
	    break;
	  case RB_CH_LSQUO:
	    *bp = '`';
	    break;
	  case RB_CH_RSQUO:
	    *bp = '\'';
	    break;
	  case RB_CH_LDQUO:
	  case RB_CH_RDQUO:
	    *bp = '"';
	    break;
	  case RB_CH_NDASH:
	  case RB_CH_MDASH:
	    *bp = '-';
	    break;
	}
    }
}
