/* rbimage.c
 *
 * These routines turn a file or an MBuf into a black-and-white PNG image
 * (which is output in an MBuf).
 */
/* This software is copyrighted as detailed in the LICENSE file. */

#include <config.h>
#ifdef HAVE_LIBPNG
#include <png.h>
#endif
#include <rbmake/rbmake.h>
#include "rbimage.h"

static const char *err;

extern bool gifCheckSig(char *bp);
extern bool jpgCheckSig(char *bp);

extern RbImage *gifToImage(MBuf *mb);
extern RbImage *pngToImage(MBuf *mb);
extern RbImage *jpgToImage(MBuf *mb);

extern MBuf *imageToPng(RbImage *img);

bool
rbLibHasImageSupport()
{
#ifdef HAVE_LIBPNG
    return true;
#else
    return false;
#endif
}

static RbImageFunc imageCallback = NULL;

void
RbImage_init(RbImageFunc imageFunc)
{
    imageCallback = imageFunc;
}

/* This replaces all the content in the supplied MBuf with the new image.
 * Values for "resizeIt": 0 = current size, 1 = full-page portrait,
 * 2 = full-page landscape. */
MBuf *
RbImage_turnMBufIntoPngMBuf(MBuf *mb, const char *url, int resizeIt)
{
#ifdef HAVE_LIBPNG
    RbImage *img;
    char *bp = MBuf_dataPtrAt(mb, 0, NULL);
    if (!bp)
	return NULL;
    if (gifCheckSig(bp))
	img = gifToImage(mb);
# ifdef HAVE_LIBJPEG
    else if (jpgCheckSig(bp))
	img = jpgToImage(mb);
# endif
    else if (png_check_sig((png_bytep)bp, 8))
	img = pngToImage(mb);
    else {
	err = "Unknown image type";
	return NULL;
    }
    if (!img)
	return NULL;
    switch (resizeIt) {
      case RB_IMAGE_FULLPAGE_PORTRAIT:
	RbImage_resize(img, 312, 472);
	break;
      case RB_IMAGE_FULLPAGE_LANDSCAPE:
	RbImage_resize(img, 472, 312);
	break;
    }
    if (imageCallback)
	imageCallback(img, url);
    RbImage_toGray(img);
    RbImage_toMono(img);
    return imageToPng(img);
#else
    return NULL;
#endif
}

void
RbImage_setError(const char *msg)
{
    err = msg;
}

const char *
RbImage_lastErrorMessage()
{
    return err;
}
