/* pngimage.c
 */
/* This software is copyrighted as detailed in the LICENSE file. */

#include <config.h>
#ifdef HAVE_LIBPNG
#include <png.h>
#include <rbmake/rbmake.h>
#include "rbimage.h"

static void readFromMBuf(png_structp png_ptr, png_bytep data, png_size_t len);
static void writeToMBuf(png_structp png_ptr, png_bytep data, png_size_t len);
static void flushMBuf(png_structp png_ptr);

RbImage *
pngToImage(MBuf *mb)
{
    png_structp png_ptr;
    png_infop info_ptr;
    png_uint_32 width, height, row;
    int bit_depth, color_type, interlace_type;
    RbImage *img;
    int cnt;

    png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
    if (!png_ptr) {
	RbImage_setError("out of memory");
	return NULL;
    }

    info_ptr = png_create_info_struct(png_ptr);
    if (!info_ptr) {
	png_destroy_read_struct(&png_ptr, NULL, NULL);
	RbImage_setError("out of memory");
	return NULL;
    }

    if (setjmp(png_ptr->jmpbuf)) {
	png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
	RbImage_setError("setjmp returns error condition");
	return NULL;
    }

    png_set_read_fn(png_ptr, mb, readFromMBuf);

    png_read_info(png_ptr, info_ptr);
    png_get_IHDR(png_ptr, info_ptr, &width, &height, &bit_depth,
		 &color_type, &interlace_type, NULL, NULL);

    png_set_expand(png_ptr);
    if (bit_depth == 16)
	png_set_strip_16(png_ptr);
    else if (bit_depth < 8)
	png_set_packing(png_ptr);
    if (color_type & PNG_COLOR_MASK_ALPHA)
	png_set_strip_alpha(png_ptr);

    png_read_update_info(png_ptr, info_ptr);
    png_get_IHDR(png_ptr, info_ptr, &width, &height, &bit_depth,
		 &color_type, &interlace_type, NULL, NULL);

    img = Mem_alloc(sizeof *img);
    img->width = width;
    img->height = height;
    img->row_pointers = Mem_alloc(height * sizeof (char*));
    img->bit_depth = bit_depth;
    img->color_type = color_type & ~PNG_COLOR_MASK_ALPHA;

    cnt = png_get_rowbytes(png_ptr, info_ptr);
    for (row = 0; row < height; row++)
	img->row_pointers[row] = Mem_alloc(cnt);

    png_read_image(png_ptr, (png_bytep*)img->row_pointers);
    png_read_end(png_ptr, NULL);

    MBuf_delete(mb);

    png_destroy_read_struct(&png_ptr, &info_ptr, NULL);

    return img;
}

MBuf *
imageToPng(RbImage *img)
{
    png_structp png_ptr;
    png_infop info_ptr;
    png_uint_32 height, row;
    MBuf *mb = MBuf_new(4096, 16384);

    png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
    if (!png_ptr) {
	RbImage_setError("out of memory");
	return NULL;
    }

    info_ptr = png_create_info_struct(png_ptr);
    if (!info_ptr) {
	png_destroy_write_struct(&png_ptr, NULL);
	RbImage_setError("out of memory");
	return NULL;
    }

    if (setjmp(png_ptr->jmpbuf)) {
	png_destroy_write_struct(&png_ptr, &info_ptr);
	RbImage_setError("setjmp returns error condition");
	return NULL;
    }

    png_set_write_fn(png_ptr, mb, writeToMBuf, flushMBuf);

    png_set_IHDR(png_ptr, info_ptr, img->width, img->height, img->bit_depth,
		 img->color_type, PNG_INTERLACE_NONE,
		 PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);

    png_write_info(png_ptr, info_ptr);

    png_set_packing(png_ptr);

    png_write_image(png_ptr, (png_bytep*)img->row_pointers);
    png_write_end(png_ptr, info_ptr);

    height = img->height;
    for (row = 0; row < height; row++)
	Mem_free(img->row_pointers[row]);
    Mem_free(img->row_pointers);

    png_destroy_write_struct(&png_ptr, &info_ptr);

    return mb;
}

static void
readFromMBuf(png_structp png_ptr, png_bytep data, png_size_t len)
{
    MBuf *mb = png_get_io_ptr(png_ptr);
    MBuf_read(mb, (char*)data, len);
}

static void
writeToMBuf(png_structp png_ptr, png_bytep data, png_size_t len)
{
    MBuf *mb = png_get_io_ptr(png_ptr);
    MBuf_write(mb, (char*)data, len);
}

static void
flushMBuf(png_structp png_ptr)
{
    /* Do nothing */
}

#endif
