/* mbuf.c
 *
 * A malloc'ed, multi-chunk-supporting, memory-buffer "class".  When
 * appending to an object (which can only be done to end of the last chunk
 * of the MBuf), you can either have the class reallocate the current
 * buffer, add more buffer chunks, or both.  You can also join other
 * buffers and MBuf objects into a single MBuf.
 */
/* This software is copyrighted as detailed in the LICENSE file. */

#include <config.h>
#include <stdarg.h>
#include <limits.h>
#include <rbmake/rbfile.h>
#include "mbuf.h"

static MBufChunk *findBufPos(MBuf *me, int *posp, bool allowEOF);

MBuf *
MBuf_new(int growBy, int maxChunkSize)
{
    MBuf *me = Mem_calloc(1, sizeof *me);
    me->growBy = growBy? growBy : 128;
    me->maxChunkSize = maxChunkSize? maxChunkSize : INT_MAX;
    return me;
}

void
MBuf_delete(MBuf *me)
{
    MBufChunk *bc, *next;
    for (bc = me->head; bc; bc = next) {
	next = bc->next;
	if (bc->data)
	    Mem_free(bc->data);
	Mem_free(bc);
    }
    Mem_free(me);
}

int
MBuf_truncate(MBuf *me, int len)
{
    MBufChunk *bc;
    int tailStart;

    me->upcomingLen = 0;

    if (len > me->totalLen)
	return -1;
    if (!me->tail)
	return 0;

    tailStart = me->totalLen - me->tail->len;
    me->totalLen = len;
    if (me->readPos > len) {
	me->readPos = me->rbufStart = 0;
	me->rbuf = me->head;
    }

    if (len >= tailStart) {
	len -= tailStart;
	bc = me->tail;
    }
    else
	bc = me->head;

    for ( ; bc; bc = bc->next) {
	if (len <= bc->len)
	    break;
	len -= bc->len;
    }
    /* If this is about to become the last buffer, set allocLen. */
    if (len < bc->len && bc->next)
	me->allocLen = me->totalLen + bc->len - len;
    bc->len = len;
    bc->data[len] = '\0';

    if (bc->next) {
	MBufChunk *next = bc->next;
	bc->next = NULL;
	me->tail = bc;
	while ((bc = next) != NULL) {
	    next = bc->next;
	    Mem_free(bc);
	}
    }
    return 0;
}

void
MBuf_prependBuffer(MBuf *me, char *bp, int len)
{
    MBufChunk *bc = Mem_alloc(sizeof *bc);
    bc->data = bp;
    bc->len = len;
    bc->next = me->head;
    me->head = bc;
    if (!me->tail)
	me->tail = bc;
    me->totalLen += len;
    me->allocLen += len;
    if (me->readPos) {
	me->readPos += len;
	me->rbufStart += len;
    }
    else
	me->rbuf = bc;
}

static void
appendChunk(MBuf *me, MBufChunk *bc)
{
    if (me->allocLen > me->totalLen)
	me->tail->data = Mem_realloc(me->tail->data, me->tail->len + 1);

    bc->next = NULL;
    if (!me->head)
	me->head = me->rbuf = bc;
    else
	me->tail->next = bc;
    me->tail = bc;
    me->allocLen = me->totalLen += bc->len;
}

/* This appends all of "mb" onto "me" and destroys "mb" in the process. */
void
MBuf_appendMBuf(MBuf *me, MBuf *mb)
{
    MBufChunk *bc, *next;
    for (bc = mb->head; bc; bc = next) {
	next = bc->next;
	appendChunk(me, bc);
    }
    me->allocLen += mb->allocLen - mb->totalLen;
    mb->head = NULL;
    MBuf_delete(mb);
}

void
MBuf_appendBuffer(MBuf *me, char *bp, int len, int allocLen)
{
    MBufChunk *bc = Mem_alloc(sizeof *bc);
    bc->data = bp;
    bc->len = len;
    appendChunk(me, bc);
    if (allocLen > len)
	me->allocLen += allocLen-1 - len;
}

static int
appendNewChunk(MBuf *me, int sizeHint)
{
    int extra = me->allocLen - me->totalLen;
    if (me->upcomingLen) {
	int hint = me->upcomingLen - me->allocLen;
	if (hint > sizeHint)
	    sizeHint = hint;
	else
	    me->upcomingLen = 0;
    }
    if (sizeHint > me->maxChunkSize)
	sizeHint = me->maxChunkSize;
    else
	sizeHint = sizeHint-1 - ((sizeHint-1) % me->growBy) + me->growBy;
    /* NOTE: we allocate an extra byte to leave room for a trailing '\0'. */
    if (!me->tail || me->tail->len + me->growBy > me->maxChunkSize)
	MBuf_appendBuffer(me, Mem_alloc(sizeHint + 1), 0, 0);
    else {
	if (me->tail->len + extra + sizeHint > me->maxChunkSize)
	    sizeHint = me->maxChunkSize - me->tail->len - extra;
	me->tail->data = Mem_realloc(me->tail->data,
				 me->tail->len + extra + sizeHint + 1);
    }
    me->allocLen += sizeHint;
    return 0;
}

int
MBuf_vwrite(MBuf *me, ...)
{
    int rem, oldLen;
    va_list ap;
    rem = me->allocLen - me->totalLen;
    oldLen = me->totalLen;
    va_start(ap, me);
    while (1) {
	char *bp = va_arg(ap, char*);
	int len, cnt;
	if (!bp)
	    break;
	if ((len = va_arg(ap, int)) < 0) {
	    if (!(len = strlen(bp)))
		continue;
	}
	while (1) {
	    if (rem == 0) {
		appendNewChunk(me, len);
		rem = me->allocLen - me->totalLen;
	    }
	    cnt = len > rem? rem : len;
	    memcpy(me->tail->data + me->tail->len, bp, cnt);
	    me->tail->len += cnt;
	    rem -= cnt;
	    me->totalLen += cnt;
	    if ((len -= cnt) == 0)
		break;
	    bp += cnt;
	    me->tail->data[me->tail->len] = '\0';
	}
    }
    va_end(ap);
    me->tail->data[me->tail->len] = '\0';
    return me->totalLen - oldLen;
}

int
MBuf_write(MBuf *me, const char *bp, int len)
{
    return MBuf_vwrite(me, bp,len, NULL);
}

int
MBuf_puts(MBuf *me, const char *str)
{
    return MBuf_vwrite(me, str,-1, NULL);
}

int
MBuf_putc(MBuf *me, char ch)
{
    char *data;
    int len;
    if (me->allocLen == me->totalLen)
	appendNewChunk(me, 1);
    data = me->tail->data;
    len = me->tail->len++;
    me->totalLen++;
    data[len] = ch;
    data[len+1] = '\0';
    return 1;
}

int
MBuf_extend(MBuf *me, int len)
{
    int rem = me->allocLen - me->totalLen;
    int i = len;
    while (i > rem) {
	if (rem) {
	    me->tail->len += rem;
	    me->totalLen += rem;
	    i -= rem;
	    me->tail->data[me->tail->len] = '\0';
	}
	appendNewChunk(me, i);
	rem = me->allocLen - me->totalLen;
    }
    me->tail->len += i;
    me->totalLen += i;
    me->tail->data[me->tail->len] = '\0';
    return len;
}

void
MBuf_setUpcomingLength(MBuf *me, int len)
{
    me->upcomingLen = len > me->allocLen? len : 0;
}

/* This allows you to write data to a particular spot in the buffer, but
 * will not allow you to extend the buffer past its current size. */
int
MBuf_overwrite(MBuf *me, int pos, char *bp, int len)
{
    MBufChunk *bc;
    int origLen = len;
    if (len <= 0)
	return len;
    if (!(bc = findBufPos(me, &pos, false)))
	return -1;
    while (1) {
	int minlen = len + pos > bc->len? bc->len - pos : len;
	memcpy(bc->data + pos, bp, minlen);
	if (!(len -= minlen) || !(bc = bc->next))
	    break;
	bp += minlen;
	pos = 0;
    }
    return origLen - len;
}

/* Like memcpy, only using offsets.  This works even if one or both spans
 * of characters crosses one or more chunk boundaries. */
int
MBuf_memcpy(MBuf *me, int tpos, int fpos, int len)
{
    MBufChunk *tbc, *fbc;
    int origLen = len;
    if (len <= 0)
	return len;
    if (!(tbc = findBufPos(me, &tpos, false))
     || !(fbc = findBufPos(me, &fpos, false)))
	return -1;
    while (1) {
	int minlen = len > fbc->len - fpos? fbc->len - fpos : len;
	if (minlen > tbc->len - tpos)
	    minlen = tbc->len - tpos;
	memcpy(tbc->data + tpos, fbc->data + fpos, minlen);
	if (!(len -= minlen))
	    break;
	if ((tpos += minlen) == tbc->len) {
	    if (!(tbc = tbc->next))
		break;
	    tpos = 0;
	}
	if ((fpos += minlen) == fbc->len) {
	    if (!(fbc = fbc->next))
		break;
	    fpos = 0;
	}
    }
    return origLen - len;
}

int
MBuf_read(MBuf *me, char *to, int len)
{
    int pos, got = 0;
    if (!me->rbuf)
	return -1;
    if (len <= 0)
	return 0;
    pos = me->readPos - me->rbufStart;
    while (pos == me->rbuf->len) {
	if (!me->rbuf->next)
	    return 0;
	me->rbufStart += pos;
	me->rbuf = me->rbuf->next;
	pos = 0;
    }
    while (1) {
	int rem = me->rbuf->len - pos;
	int cnt = len > rem? rem : len;
	memcpy(to, me->rbuf->data + pos, cnt);
	me->readPos += cnt;
	got += cnt;
	if ((len -= cnt) == 0 || !me->rbuf->next)
	    break;
	to += cnt;
	me->rbufStart += me->rbuf->len;
	me->rbuf = me->rbuf->next;
	pos = 0;
    }
    return got;
}

int
MBuf_gets(MBuf *me, char *buf, int tlen)
{
    int ch, pos, rem = 0;
    char *bp, *to = buf;
    bool sawCR = false;

    if (tlen < 1 || !me->rbuf)
	return -1;
    pos = me->readPos - me->rbufStart;
    while (pos == me->rbuf->len) {
	if (!me->rbuf->next)
	    goto all_done;
	me->rbufStart += pos;
	me->rbuf = me->rbuf->next;
	pos = 0;
    }
    while (1) {
	rem = me->rbuf->len - pos;
	bp = me->rbuf->data + pos;
	while (rem > 0) {
	    if (tlen == 1) {
		pos = bp - me->rbuf->data;
		goto all_done;
	    }
	    if (sawCR && *bp != '\n') {
		*to++ = '\n';
		pos = bp - me->rbuf->data;
		goto all_done;
	    }
	    ch = *to++ = *bp++;
	    rem--;
	    if (ch == '\n') {
		pos = bp - me->rbuf->data;
		goto all_done;
	    }
	    if (ch == '\r') {
		to--;
		sawCR = true;
	    }
	    else
		tlen--;
	}
	if (!me->rbuf->next) {
	    pos = bp - me->rbuf->data;
	    break;
	}
	me->rbufStart += me->rbuf->len;
	me->rbuf = me->rbuf->next;
	pos = 0;
    }
  all_done:
    *to = '\0';
    me->readPos = me->rbufStart + pos;
    return to - buf;
}

int
MBuf_getc(MBuf *me)
{
    int pos;
    if (!me->rbuf)
	return EOF;
    pos = me->readPos - me->rbufStart;
    while (pos == me->rbuf->len) {
	if (!me->rbuf->next)
	    return EOF;
	me->rbufStart += pos;
	me->rbuf = me->rbuf->next;
	pos = 0;
    }
    return me->rbuf->data[pos];
}

int
MBuf_getLength(MBuf *me)
{
    return me->totalLen;
}

int
MBuf_getReadPos(MBuf *me)
{
    return me->readPos;
}

int
MBuf_setReadPos(MBuf *me, int pos, int from)
{
    MBufChunk *bc;
    int rpos;
    switch (from) {
      case 0:
	break;
      case 1:
	pos += me->readPos;
	break;
      case 2:
	pos += me->totalLen;
	break;
      default:
	return -1;
    }
    rpos = pos;
    if (!(bc = findBufPos(me, &pos, true)))
	return -1;
    me->rbuf = bc;
    me->rbufStart = rpos - pos;
    return me->readPos = rpos;
}

/* This returns a pointer to the current read position.  If lenp is
 * non-NULL, it points to an int with the length of the data you wish to
 * handle, it is updated with a shorter length if the buffer data at this
 * spot is not long enough, and the read position is bumped forward to
 * point past the returned length (ready to reference the next chunk). */
char *
MBuf_dataPtr(MBuf *me, int *lenp)
{
    int pos = me->readPos;
    MBufChunk *bc = findBufPos(me, &pos, false);
    char *bp;
    if (!bc)
	return NULL;
    bp = bc->data + pos;
    if (lenp) {
	int len = *lenp;
	int rem = bc->len - pos;
	if (len > rem)
	    *lenp = len = rem;
	me->readPos += len;
	if (len == rem && bc->next) {
	    bc = bc->next;
	    pos = 0;
	}
	else
	    pos += len;
	me->rbuf = bc;
	me->rbufStart = me->readPos - pos;
    }
    return bp;
}

/* This doesn't affect the read position at all.  If lenp is non-null, it
 * contains a length of data that you'd like to access, and will be
 * updated with the actual length that you can access. */
char *
MBuf_dataPtrAt(MBuf *me, int pos, int *lenp)
{
    MBufChunk *bc = findBufPos(me, &pos, false);
    if (!bc)
	return NULL;
    if (lenp) {
	int rem = bc->len - pos;
	if (*lenp > rem)
	    *lenp = rem;
    }
    return bc->data + pos;
}

static MBufChunk *
findBufPos(MBuf *me, int *posp, bool allowEOF)
{
    int tailStart, pos = *posp;
    MBufChunk *bc;
    if (pos < 0)
	return NULL;
    if (pos >= me->totalLen) {
	if (!allowEOF || !me->head)
	    return NULL;
	pos = me->totalLen;
    }
    tailStart = me->totalLen - me->tail->len;
    if (pos >= tailStart) {
	bc = me->tail;
	pos -= tailStart;
    }
    else if (pos >= me->rbufStart) {
	bc = me->rbuf;
	pos -= me->rbufStart;
    }
    else
	bc = me->head;
    while (pos >= bc->len && bc->next) {
	pos -= bc->len;
	bc = bc->next;
    }
    *posp = pos;
    return bc;
}

/* This destroys the MBuf and puts all the data into a single, malloced
 * buffer.  If allocLenPtr is non-NULL, leave any extra memory in-place and
 * pass back the amount of currently-alloced memory (including the room
 * we allocated for a final null-terminator that is not counted in the
 * class's allocLen value). */
char *
MBuf_toBuffer(MBuf *me, int *allocLenPtr)
{
    MBufChunk *bc = me->head;
    char *bp;
    if (!bc) {
	bp = Mem_alloc(1);
	*bp = '\0';
	if (allocLenPtr)
	    *allocLenPtr = 1;
    }
    else {
	bp = bc->data;
	bc->data = NULL;
	if (bc->next) {
	    char *t;
	    bp = Mem_realloc(bp, me->totalLen+1);
	    for (t = bp + bc->len; (bc = bc->next) != NULL; t += bc->len)
		memcpy(t, bc->data, bc->len);
	    *t = '\0';
	    if (allocLenPtr)
		*allocLenPtr = me->totalLen+1;
	}
	else if (allocLenPtr)
	    *allocLenPtr = me->allocLen+1;
	else if (me->allocLen != me->totalLen)
	    bp = Mem_realloc(bp, me->totalLen+1);
    }
    MBuf_delete(me);
    return bp;
}
