#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "copy.h"
#include "makedir.h"
#include "rummgmt.h"

extern int		errno;

/*
 * rummgmt: functions to assist in enabling and disabling
 * the RadmindUpdateMonitor daemon
 */

static char		*sbindir = "/usr/local/sbin";
static char		*launchddir = "/Library/LaunchDaemons";
static char		*launchctl = "/bin/launchctl";
static char		*launchdplist = "edu.umich.rumd.plist";
static char		*sidir = "/Library/StartupItems";
static char		*siname = "RadmindUpdateMonitor";
static char		*siplist = "rumd-StartupParameters.plist";

    int
rum_enable( int type, char *rsrc_path )
{
    struct stat		st;
    char		src[ MAXPATHLEN ], dst[ MAXPATHLEN ];
    pid_t		pid;
    int			status;

    makedir( "/usr/local" );
    makedir( sbindir );

    if ( snprintf( dst, MAXPATHLEN, "%s/rumd", sbindir ) >= MAXPATHLEN ) {
	fprintf( stderr, "%s/rumd: path too long\n", sbindir );
	return( 2 );
    }

    /* copy rumd to /usr/local/sbin if necessary */
    if ( stat( dst, &st ) != 0 ) {
	if ( errno == ENOENT ) {
	    if ( snprintf( src, MAXPATHLEN, "%s/rumd",
			rsrc_path ) >= MAXPATHLEN ) {
		fprintf( stderr, "%s/rumd: path too long\n", rsrc_path );
		return( 2 );
	    }
	    if ( stat( src, &st ) != 0 ) {
		perror( src );
		return( 2 );
	    }

	    copy( src, dst, 0755 );
	    if ( chown( dst, 0, 0 ) != 0 ) {
		fprintf( stderr, "chown %s: %s\n", dst, strerror( errno ));
		return( 2 );
	    }
	} else {
	    perror( dst );
	    return( 2 );
	}
    }

    memset( src, 0, sizeof( src ));
    memset( dst, 0, sizeof( dst ));

    if ( type == LAUNCHD_SYSTEM ) {
	/* use launchd plist and launchctl */
	if ( stat( launchddir, &st ) != 0 ) {
	    perror( launchddir );
	    return( 2 );
	}
	if ( stat( launchctl, &st ) != 0 ) {
	    perror( launchctl );
	    return( 2 );
	}

	/* copy launchd plist if necessary */
	if ( snprintf( dst, MAXPATHLEN, "%s/%s",
			launchddir, launchdplist ) >= MAXPATHLEN ) {
	    fprintf( stderr, "%s/%s: path too long\n",
			launchddir, launchdplist );
	    return( 2 );
	}
	if ( stat( dst, &st ) != 0 ) {
	    if ( errno == ENOENT ) {
		if ( snprintf( src, MAXPATHLEN, "%s/%s",
			rsrc_path, launchdplist ) >= MAXPATHLEN ) {
		    fprintf( stderr, "%s/%s: path too long\n",
			rsrc_path, launchdplist );
		    return( 2 );
		}

		copy( src, dst, 0644 );
		if ( chown( dst, 0, 0 ) != 0 ) {
		    fprintf( stderr, "chown %s: %s\n", dst, strerror( errno ));
		    return( 2 );
		}
	    } else {
		perror( dst );
		return( 2 );
	    }
	}

	switch ( fork()) {
	case 0:
	    /* load rumd, removing disabled key (-w) from its plist */
	    execl( launchctl, launchctl, "load", "-w", dst, NULL );
	    fprintf( stderr, "execl %s: %s\n", launchctl, strerror( errno ));
	    (void)fflush( stderr );
	    _exit( 2 );

	case -1:
	    perror( "fork" );
	    exit( 2 );

	default:
	    break;
	}
    } else if ( type == PRELAUNCHD_SYSTEM ) {	/* use StartupItem script */
	makedir( sidir );
	if ( chown( sidir, 0, 0 ) != 0 ) {
	    fprintf( stderr, "chown %s: %s\n", sidir, strerror( errno ));
	    return( 2 );
	}		    

	if ( snprintf( dst, MAXPATHLEN, "%s/%s",
		sidir, siname ) >= MAXPATHLEN ) {
	    fprintf( stderr, "%s/%s: path too long\n", sidir, siname );
	    return( 2 );
	}
	if ( stat( dst, &st ) != 0 ) {
	    if ( errno == ENOENT ) {
		makedir( dst );
		if ( chown( dst, 0, 0 ) != 0 ) {
		    fprintf( stderr, "chown %s: %s\n", dst, strerror( errno ));
		    return( 2 );
		}
	    } else {
		perror( dst );
		return( 2 );
	    }
	}
		
	memset( dst, 0, sizeof( dst ));

	if ( snprintf( dst, MAXPATHLEN, "%s/%s/StartupParameters.plist",
			sidir, siname ) >= MAXPATHLEN ) {
	    fprintf( stderr, "%s/%s/StartupParameters.plist: path too long\n",
			sidir, siname );
	    return( 2 );
	}
	if ( stat( dst, &st ) != 0 ) {
	    if ( errno == ENOENT ) {
		if ( snprintf( src, MAXPATHLEN, "%s/%s",
			rsrc_path, siplist ) >= MAXPATHLEN ) {
		    fprintf( stderr, "%s/%s: path too long\n",
			rsrc_path, siplist );
		    return( 2 );
		}

		copy( src, dst, 0755 );
		if ( chown( dst, 0, 0 ) != 0 ) {
		    fprintf( stderr, "chown %s: %s\n", dst, strerror( errno ));
		    return( 2 );
		}
	    } else {
		perror( dst );
		return( 2 );
	    }
	}

	memset( dst, 0, sizeof( dst ));

	if ( snprintf( dst, MAXPATHLEN, "%s/%s/%s",
			sidir, siname, siname ) >= MAXPATHLEN ) {
	    fprintf( stderr, "%s/%s/%s: path too long\n",
			sidir, siname, siname );
	    return( 2 );
	}
	if ( stat( dst, &st ) != 0 ) {
	    if ( errno == ENOENT ) {
		if ( snprintf( src, MAXPATHLEN, "%s/%s",
			rsrc_path, siname ) >= MAXPATHLEN ) {
		    fprintf( stderr, "%s/%s: path too long\n",
			rsrc_path, siname );
		    return( 2 );
		}

		copy( src, dst, 0755 );
		if ( chown( dst, 0, 0 ) != 0 ) {
		    fprintf( stderr, "chown %s: %s\n", dst, strerror( errno ));
		    return( 2 );
		}
	    } else {
		perror( dst );
		return( 2 );
	    }
	}

	switch ( fork()) {
	case 0:
	    /* load startup item */
	    execl( dst, dst, "start", NULL );
	    fprintf( stderr, "execl %s: %s\n", dst, strerror( errno ));
	    (void)fflush( stderr );
	    _exit( 2 );

	case -1:
	    perror( "fork" );
	    exit( 2 );

	default:
	    break;
	}
    }

    pid = wait( &status );

    return( WEXITSTATUS( status ));
}

    int
rum_disable( int type )
{
    char		buf[ MAXPATHLEN ];
    pid_t		pid;
    int			status;

    if ( type == LAUNCHD_SYSTEM ) {		/* use launchctl */
	if ( snprintf( buf, MAXPATHLEN, "%s/%s",
			launchddir, launchdplist ) >= MAXPATHLEN ) {
	    fprintf( stderr, "%s/%s: path too long\n",
			launchddir, launchdplist );
	    return( 2 );
	}

	switch( fork()) {
	case 0:
	    execl( launchctl, launchctl, "unload", "-w", buf, NULL );
	    fprintf( stderr, "execl %s: %s\n", launchctl, strerror( errno ));
	    (void)fflush( stderr );
	    _exit( 2 );

	case -1:
	    perror( "fork" );
	    exit( 2 );

	default:
	    break;
	}
    } else if ( type == PRELAUNCHD_SYSTEM ) {	/* use StartupItem script */
	if ( snprintf( buf, MAXPATHLEN, "%s/%s/%s",
			sidir, siname, siname ) >= MAXPATHLEN ) {
		fprintf( stderr, "%s/%s/%s: path too long\n",
			sidir, siname, siname );
		return( 2 );
	}

	switch( fork()) {
	case 0:
	    execl( buf, buf, "stop", NULL );
	    fprintf( stderr, "execl %s: %s\n", buf, strerror( errno ));
	    (void)fflush( stderr );
	    _exit( 2 );

	case -1:
	    perror( "fork" );
	    exit( 2 );

	default:
	    break;
	}
    }

    pid = wait( &status );

    return( WEXITSTATUS( status ));
}
