/*
 * radmond: run ktcheck -n every N minutes to check server
 * for updates. post a notification if updates are available.
 *
 */
#include <CoreFoundation/CoreFoundation.h>
#include <SystemConfiguration/SystemConfiguration.h>

#include <sys/types.h>
#include <sys/param.h>
#include <sys/select.h>
#include <sys/wait.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <syslog.h>
#include <unistd.h>

#include "argcargv.h"
#include "pidforname.h"

#define RADMIND_PREFS	CFSTR( "/Library/Preferences/edu.umich.radmind" )
#define OLD_PREFS	CFSTR("/Library/Preferences/edu.umich.radmindassistant")
#define HOST_PREF	CFSTR( "radmindhost" )
#define TLS_PREF	CFSTR( "tlslevel" )
#define PATH_PREF	CFSTR( "RUMKtcheckPath" )
#define CHECK_PREF	CFSTR( "RUMCheckInterval" )

void			hup( int );
void			term( int );
int			inprogress( void );
char			*readline( int, FILE * );

int			hup_signal = 0;
time_t			interval = 3600; 	/* sane interval */
char			*radmind_server = NULL;
char			*ktcheck_path = NULL;
char			*tls_level = NULL;
char			*version = "1.0.0";

    void
term( int sig )
{
}

    void
hup( int sig )
{
    /* re-read defaults and generate new argv */
    hup_signal = 1;
}

    int
inprogress( void )
{
    char			*tools[] = { "fsdiff", "ktcheck", "lapply" };
    int				i;

    for ( i = 0; i < 3; i++ ) {
	if ( pidforname( tools[ i ] ) > 0 ) {
	    syslog( LOG_NOTICE,
		"radmind session in progress (%s running)", tools[ i ] );
	    return( 1 );
	}
    }

    return( 0 );
}

    char *
readline( int fd, FILE *of )
{
    static char			buf[ LINE_MAX ];
    fd_set			readmask;
    int				len;

    memset( buf, '\0', sizeof( buf ));

    setbuf( of, NULL );

    /* read the first line of output, if any, from ktcheck */
    for ( ;; ) {
	struct timeval		tv = { 0, 1 };

	memset( buf, '\0', sizeof( buf ));

	FD_ZERO( &readmask );
	FD_SET( fd, &readmask );

	switch ( select(( fd + 1 ), &readmask, NULL, NULL, &tv )) {
	case -1:
	    syslog( LOG_ERR, "select: %m" );
	    break;

	case 0:	/* timeout */
	    continue;

	default:
	    break;
	}

	if ( !FD_ISSET( fd, &readmask )) {
	    break;
	}

	if ( fgets( buf, LINE_MAX, of ) == NULL ) {
	    if ( ferror( of )) {
		syslog( LOG_ERR, "fgets: %m" );
		return( NULL );
	    }
	}

	/* we only want the first line */
	break;
    }

    if ( *buf != '\0' ) {
	len = strlen( buf );
	buf[ len - 1 ] = '\0';
    }

    return( buf );
}

    static void
read_prefs()
{
    CFArrayRef			keys = NULL;
    CFDictionaryRef		dict = NULL;
    CFIndex			len;
    CFNumberRef			level, check;
    CFStringRef			path;
    CFStringRef			global_id = RADMIND_PREFS;
    const CFStringRef		user = kCFPreferencesCurrentUser;
    const CFStringRef		host = kCFPreferencesAnyHost;
    const CFStringRef		pref_keys[] = {
						HOST_PREF, TLS_PREF,
						PATH_PREF, CHECK_PREF
					      };
    int				lvl;

    if (( keys = CFArrayCreate( kCFAllocatorDefault,
		(const void **)&pref_keys, 4, NULL )) == NULL ) {
	syslog( LOG_ERR, "CFArrayCreate failed" );
	exit( 2 );
    }

    dict = CFPreferencesCopyMultiple( keys, global_id, user, host );
    if ( CFDictionaryGetCount( dict ) == 0 || dict == NULL ) {
	/* try old prefs instead */
	CFRelease( dict );
	global_id = OLD_PREFS;
	dict = CFPreferencesCopyMultiple( keys, global_id, user, host );
	if ( CFDictionaryGetCount( dict ) == 0 || dict == NULL ) {
	    /* use default values */
	    syslog( LOG_ERR, "CFPreferencesCopyMultiple failed" );
	}
    }

    if ( dict ) {
	/* radmind server */
	if (( CFDictionaryGetValueIfPresent( dict, HOST_PREF,
		    (const void **)&host )) == true ) {
	    CFRetain( host );
	    len = CFStringGetLength( host );
	    if ( radmind_server != NULL ) {
		free( radmind_server );
	    }
	    if (( radmind_server = ( char * )malloc((int)len + 1 )) == NULL ) {
		syslog( LOG_ERR, "malloc: %m" );
		exit( 2 );
	    }
	    memset( radmind_server, 0, ( len + 1 ));
	    if ( CFStringGetCString( host, radmind_server, len + 1,
			CFStringGetSystemEncoding()) == false ) {
		syslog( LOG_ERR, "CFStringGetCString failed" );
		free( radmind_server );
		if (( radmind_server = strdup( "radmind" )) == NULL ) {
		    syslog( LOG_ERR, "strdup %s: %m", "radmind" );
		    exit( 2 );
		}
	    }
	    CFRelease( host );
	}
	/* TLS level */
	if (( CFDictionaryGetValueIfPresent( dict, TLS_PREF,
		    (const void **)&level )) == true ) {
	    CFRetain( level );
	    if ( CFNumberGetValue( level, kCFNumberIntType, &lvl ) == false ) {
		syslog( LOG_NOTICE, "CFNumberGetValue failed, using default" );
		lvl = 0;
	    }
	    CFRelease( level );
	    if ( tls_level != NULL ) {
		free( tls_level );
	    }
	    if (( tls_level = ( char * )malloc( MAXPATHLEN )) == NULL ) {
		syslog( LOG_ERR, "malloc: %m" );
		exit( 2 );
	    }
	    if ( snprintf( tls_level, MAXPATHLEN, "%d", lvl ) >= MAXPATHLEN ) {
		syslog( LOG_NOTICE, "%d: too long. Using default\n", lvl );
		strcpy( tls_level, "0" );
	    }
	}
	/* ktcheck path */
	if (( CFDictionaryGetValueIfPresent( dict, PATH_PREF,
		    (const void **)&path )) == true ) {
	    CFRetain( path );
	    len = CFStringGetLength( path );
	    if ( ktcheck_path != NULL ) {
		free( ktcheck_path );
	    }
	    if (( ktcheck_path = ( char * )malloc((int)len + 1 )) == NULL ) {
		syslog( LOG_ERR, "malloc: %m" );
		exit( 2 );
	    }
	    memset( ktcheck_path, 0, ( len + 1 ));
	    if ( CFStringGetCString( host, ktcheck_path, len + 1,
			CFStringGetSystemEncoding()) == false ) {
		syslog( LOG_ERR, "CFStringGetCString failed" );
		free( ktcheck_path );
		if (( ktcheck_path =
			strdup( "/usr/local/bin/ktcheck" )) == NULL ) {
		    syslog( LOG_ERR, "strdup: %m", "radmind" );
		    exit( 2 );
		}
	    }
	    CFRelease( path );
	}
	/* check interval */
	if (( CFDictionaryGetValueIfPresent( dict, CHECK_PREF,
		    (const void **)&check )) == true ) {
	    CFRetain( check );
	    if ( CFNumberGetValue( check, kCFNumberIntType,
			&interval ) == false ) {
		syslog( LOG_ERR, "CFNumberGetValue failed, using default" );
		interval = (time_t)3600;
	    }
	    CFRelease( check );
	}
    }

    if ( keys ) {
	CFRelease( keys );
    }
    if ( dict ) {
	CFRelease( dict );
    }
}

    int
main( int ac, char *av[] )
{
    SCNetworkConnectionFlags	flags;
    CFDictionaryRef		userInfo = NULL;
    CFStringRef			output = NULL;
    CFStringRef			consoleUser = NULL;
    FILE			*of = NULL;
    struct sigaction		sa, osahup;
    pid_t			pid;
    int				status, c, err = 0;
    int				i, dt, tac, fg = 0;
    int				debug = 0, logopts = LOG_PID;
    int				ofd[ 2 ];
    extern int			optind;
    extern char			*optarg;
    char			args[ LINE_MAX ];
    char			*line = NULL;
    char			**tav = NULL;

    if ( getuid() != 0 ) {
	syslog( LOG_ERR, "%s must be run as root\n", av[ 0 ] );
	exit( 2 );
    }

    /* read preferences first, allow switches to override */
    read_prefs();

    while (( c = getopt( ac, av, "dfh:k:t:Vw:" )) != EOF ) {
	switch ( c ) {
	case 'd':	/* debug */
	    debug = 1;
	    logopts = LOG_PERROR | LOG_PID;
	    break;

	case 'f':	/* run in foreground */
	    fg = 1;
	    break;

	case 'h':	/* radmind server */
	    if ( radmind_server != NULL ) {
		free( radmind_server );
	    }
	    if (( radmind_server = strdup( optarg )) == NULL ) {
		syslog( LOG_ERR, "strdup %s: %m", optarg );
		exit( 2 );
	    }
	    break;

	case 'k':	/* alternative path to ktcheck */
	    if ( ktcheck_path != NULL ) {
		free( ktcheck_path );
	    }
	    if (( ktcheck_path = strdup( optarg )) == NULL ) {
		syslog( LOG_ERR, "strdup %s: %m", optarg );
		exit( 2 );
	    }
	    break;

	case 't':	/* interval in seconds between checks */
	    errno = 0;
	    interval = (time_t)strtol( optarg, NULL, 10 );
	    if ( errno ) {
		syslog( LOG_NOTICE, "strtol: %s. Using 3600 sec. interval\n" );
		interval = (time_t)3600;
	    }
	    break;

	case 'V':	/* version */
	    printf( "%s\n", version );
	    exit( 0 );

	case 'w':	/* tls level */
	    if ( tls_level != NULL ) {
		free( tls_level );
	    }
	    if (( tls_level = strdup( optarg )) == NULL ) {
		syslog( LOG_ERR, "strdup %s: %m", optarg );
		exit( 2 );
	    }
	    break;

	default:
	    err++;
	    break;
	}
    }

    if ( err ) {
	fprintf( stderr, "Usage: %s [ -df ] [ -h radmind_server ] "
		"[ -k ktcheck_path ] [ -t check_interval ] "
		"[ -w auth-level ]\n", av[ 0 ] );
	exit( 1 );
    }

#ifdef notdef
    if ( interval < 3600 ) {	/* play nice */
	interval = (time_t)3600;
    }
#endif notdef

    if ( radmind_server == NULL ) {
	if (( radmind_server = strdup( "radmind" )) == NULL ) {
	    syslog( LOG_ERR, "strdup %s: %m", optarg );
	    exit( 2 );
	}
    }
    if ( ktcheck_path == NULL ) {
	if (( ktcheck_path = strdup( "/usr/local/bin/ktcheck" )) == NULL ) {
	    syslog( LOG_ERR, "strdup %s: %m", optarg );
	    exit( 2 );
	}
    }
    if ( tls_level == NULL ) {
	if (( tls_level = strdup( "0" )) == NULL ) {
	    syslog( LOG_ERR, "strdup 0: %m" );
	    exit( 2 );
	}
    }

    if ( snprintf( args, LINE_MAX, "%s -c sha1 -w %s -n -h %s",
		ktcheck_path, tls_level, radmind_server ) >= LINE_MAX ) {
	syslog( LOG_ERR, "%s -c sha1 -w %s -n -h %s: too long",
		ktcheck_path, tls_level, radmind_server );
	exit( 2 );
    }

    if (( tac = argcargv( args, &tav )) < 0 ) {
	syslog( LOG_ERR, "argcargv: %m" );
	exit( 2 );
    }
    if ( tac == 0 ) {
	syslog( LOG_ERR, "no arguments, exiting" );
	exit( 2 );
    }

    if ( !debug && !fg ) {
	switch ( fork()) {
	case 0:
	    if ( setsid() < 0 ) {
		syslog( LOG_ERR, "setsid: %m" );
		exit( 2 );
	    }
	    dt = getdtablesize();
	    for ( i = 0; i < dt; i++ ) {
		( void )close( i );
	    }
	    if (( i = open( "/", O_RDONLY, 0 )) == 0 ) {
		dup2( i, 1 );
		dup2( i, 2 );
	    }
	    break;

	case -1:
	    syslog( LOG_ERR, "fork: %m" );
	    exit( 2 );

	default:
	    if ( radmind_server ) {
		free( radmind_server );
	    }

	    exit( 0 );
	}
    }

    openlog( av[ 0 ], logopts, LOG_LOCAL7 );

    /* catch SIGHUP */
    memset( &sa, 0, sizeof( struct sigaction ));
    sa.sa_handler = hup;
    if ( sigaction( SIGHUP, &sa, &osahup ) < 0 ) {
        syslog( LOG_ERR, "sigaction: %m" );
        exit( 1 );
    }

    syslog( LOG_NOTICE, "restart" );
    syslog( LOG_NOTICE, "radmind server: %s", radmind_server );

    for ( ;; ) {
	/* sleep specific time between update checks */
	sleep( interval );

	/* if there's no console user, skip the check */
	if (( consoleUser = SCDynamicStoreCopyConsoleUser( NULL,
		NULL, NULL )) == NULL ) {
	    syslog( LOG_NOTICE, "No console user, skipping update check." );
	    continue;
	}
	CFRelease( consoleUser );

	/* re-read prefs, if required */
	if ( hup_signal ) {
	    hup_signal = 0;

	    syslog( LOG_NOTICE, "Caught SIGHUP, re-reading preferences" );

	    read_prefs();
	    
	    memset( args, '\0', sizeof( args ));
	    if ( snprintf( args, LINE_MAX, "%s -c sha1 -w %s -n -h %s",
			ktcheck_path, tls_level,
			radmind_server ) >= LINE_MAX ) {
		syslog( LOG_ERR, "%s -c sha1 -w %s -n -h %s: too long",
			ktcheck_path, tls_level, radmind_server );
		exit( 2 );
	    }

	    if (( tac = argcargv( args, &tav )) < 0 ) {
		syslog( LOG_ERR, "argcargv: %m" );
		exit( 2 );
	    }
	    if ( tac == 0 ) {
		syslog( LOG_ERR, "no arguments, exiting" );
		exit( 2 );
	    }
	}

	/* check for network availability */
	if ( SCNetworkCheckReachabilityByName( radmind_server,
		&flags ) == FALSE ) {
	    syslog( LOG_ERR, "SCNetworkCheckReachabilityByName failed\n" );
	    continue;
	}
	if ( ! ( flags & kSCNetworkFlagsReachable )) {
	    syslog( LOG_ERR, "Network not available\n" );
	    continue;
	}

	/* check to see that none of the radmind client tools are running */
	if ( inprogress()) {
	    continue;
	}

	/* tell the status item we're checking for updates */
	CFNotificationCenterPostNotificationWithOptions(
		    CFNotificationCenterGetDistributedCenter(),
		    CFSTR( "RUMUpdatesCheckingNotification" ),
		    NULL, NULL, kCFNotificationDeliverImmediately |
				kCFNotificationPostToAllSessions );

	/* create pipe to read ktcheck's output. */
	if ( pipe( ofd ) != 0 ) {
	    syslog( LOG_ERR, "pipe: %m" );
	    continue;
	}

	switch (( pid = fork())) {
	case 0:
	    /* dup stdout and stderr to same descriptor */
	    if ( close( ofd[ 0 ] ) != 0 ) {
		syslog( LOG_ERR, "close: %m" );
		_exit( 2 );
	    }
	    if ( dup2( ofd[ 1 ], 1 ) < 0 ) {
		syslog( LOG_ERR, "dup2: %m" );
		_exit( 2 );
	    }
	    if ( dup2( ofd[ 1 ], 2 ) < 0 ) {
		syslog( LOG_ERR, "dup2: %m" );
		_exit( 2 );
	    }
	    if ( close( ofd[ 1 ] ) != 0 ) {
		syslog( LOG_ERR, "close: %m" );
		_exit( 2 );
	    }

	    execve( tav[ 0 ], tav, NULL );
	    syslog( LOG_ERR, "execve: %m\n" );
	    _exit( 2 );

	case -1:
	    syslog( LOG_ERR, "fork: %m" );
	    ( void )close( ofd[ 0 ] );
	    ( void )close( ofd[ 1 ] );
	    exit( 2 );

	default:
	    break;
	}

	if ( signal( SIGPIPE, SIG_IGN ) == SIG_ERR ) {
	    syslog( LOG_ERR, "signal: %m" );
	}

	if ( close( ofd[ 1 ] ) != 0 ) {
	    syslog( LOG_ERR, "close: %m" );
	}

	if (( of = fdopen( ofd[ 0 ], "r" )) == NULL ) {
	    syslog( LOG_ERR, "fdopen: %m" );
	    if ( close( ofd[ 0 ] ) != 0 ) {
		syslog( LOG_ERR, "close: %m" );
	    }
	} else {
	    line = readline( ofd[ 0 ], of );
	    if ( fclose( of ) != 0 ) {
		syslog( LOG_ERR, "fclose: %m" );
	    }
	}

	/* create the dictionary for the notification */
	if ( line != NULL ) {
	    void		*keys[ 1 ];
	    void		*values[ 1 ];

	    if (( output = CFStringCreateWithBytes( kCFAllocatorDefault,
			( const UInt8 * )line, ( CFIndex )strlen( line ),
			CFStringGetSystemEncoding(), false )) == NULL ) {
		syslog( LOG_ERR, "CFStringCreateWithBytes failed" );
	    } else {
		keys[ 0 ] = ( void * )CFSTR( "RUMServerOutput" );
		values[ 0 ] = ( void * )output;
		
		userInfo = CFDictionaryCreate( kCFAllocatorDefault,
				( const void ** )keys, ( const void ** )values,
				1, &kCFTypeDictionaryKeyCallBacks,
				&kCFTypeDictionaryValueCallBacks );
		if ( userInfo == NULL ) {
		    syslog( LOG_ERR, "CFDictionaryCreate failed" );
		}
	    }
	}

	pid = wait( &status );
	switch( WEXITSTATUS( status )) {
	case 0:
	    syslog( LOG_DEBUG, "no updates on %s", radmind_server );
	    CFNotificationCenterPostNotificationWithOptions(
			CFNotificationCenterGetDistributedCenter(),
			CFSTR( "RUMUpdatesNoneNotification" ),
			NULL, userInfo, kCFNotificationDeliverImmediately |
					kCFNotificationPostToAllSessions );
	    break;

	case 1:
	    syslog( LOG_DEBUG, "updates available on %s", radmind_server );
	    CFNotificationCenterPostNotificationWithOptions(
			CFNotificationCenterGetDistributedCenter(),
			CFSTR( "RUMUpdatesAvailableNotification" ),
			NULL, userInfo, kCFNotificationDeliverImmediately |
					kCFNotificationPostToAllSessions );
	    break;

	default:
	    syslog( LOG_NOTICE, "Update check on %s exited with %d",
			radmind_server, WEXITSTATUS( status ));
	    CFNotificationCenterPostNotificationWithOptions(
			CFNotificationCenterGetDistributedCenter(),
			CFSTR( "RUMUpdatesErrorNotification" ),
			NULL, userInfo, kCFNotificationDeliverImmediately |
					kCFNotificationPostToAllSessions );
	    break;
	}

	if ( output != NULL ) {
	    CFRelease( output );
	    output = NULL;
	}
	if ( userInfo != NULL ) {
	    CFRelease( userInfo );
	    userInfo = NULL;
	}
    }

    closelog();

    return( 0 );
}
