/*
Copyright (c) 2005 Regents of The University of Michigan.
All Rights Reserved.

Permission to use, copy, modify, and distribute this software and
its documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appears in all copies and
that both that copyright notice and this permission notice appear
in supporting documentation, and that the name of The University
of Michigan not be used in advertising or publicity pertaining to
distribution of the software without specific, written prior
permission. This software is supplied as is without expressed or
implied warranties of any kind.

Research Systems Unix Group
The University of Michigan
c/o Wesley Craig
4251 Plymouth Road B1F2, #2600
Ann Arbor, MI 48105-2785

http://rsug.itd.umich.edu/software/radmind
radmind@umich.edu
*/

#include <Security/Authorization.h>

#include <sys/types.h>
#include <sys/param.h>
#include <sys/wait.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#include "copy.h"
#include "rteauthexec.h"
#include "selfrepair.h"

extern int	errno;

typedef void	(*sighandler_t)( int );
void		term( int sig );

/* actions */
int		rte_readfile( int, char *[] );
int		rte_writefile( int, char *[] );
int		rte_exec( int, char *[] );
int		rte_killpg( int, char *[] );

char		user[ MAXPATHLEN ] = { 0 };

struct rte_action	actions[] = {
    { RTE_READ,		rte_readfile },
    { RTE_WRITE,	rte_writefile },
    { RTE_EXEC,		rte_exec },
    { RTE_KILLPG,	rte_killpg },
};

    void
term( int sig )
{
    syslog( LOG_INFO, "Caught signal %d, exiting.\n", sig );
    exit( sig );
}

    int
rte_exec( int ac, char *av[] )
{
    int         status;
    pid_t       pid;

    switch ( fork()) {
    case 0:
        execve( av[ 0 ], av, NULL );
        syslog( LOG_ERR, "execve failed: %s\n", strerror( errno ));
        _exit( 2 );

    case -1:
        fprintf( stderr, "fork failed: %s\n", strerror( errno ));
        exit( 2 );

    default:
        break;
    }

    pid = wait( &status );      
    return( WEXITSTATUS( status ));
}

    int
rte_killpg( int ac, char *av[] )
{
    pid_t       pid;

    if (( pid = strtol( av[ 0 ], NULL, 10 )) == 0 ) {
        fprintf( stderr, "strtol %s: %s", av[ 0 ], strerror( errno ));
        return( 2 );
    }

    if ( pid == -1 || pid == 0 || pid == 1 ) {
        fprintf( stderr, "Will not kill process with pid %d", pid );
        return( 1 );
    }

    if ( killpg( pid, SIGTERM ) < 0 ) {
        fprintf( stderr, "kill %d: %s\n", pid, strerror( errno ));
        return( 2 );
    }
    syslog( LOG_INFO, "killed %d", pid );
    fprintf( stderr, "Process with pid %d cancelled\n", pid );

    return( 0 );
}

    int
rte_readfile( int ac, char *av[] )
{
    struct passwd	*pw;
    char		*from, *to;

    if ( ac != 2 ) {
	fprintf( stderr, "Wrong arguments\n" );
	return( 1 );
    }

    if (( pw = getpwnam( user )) == NULL ) {
	fprintf( stderr, "%s: no such user. Aborting...\n", user );
	return( 1 );
    }

    from = av[ 0 ];
    to = av[ 1 ];

    copy( from, to, 1 );
    if ( chown( to, pw->pw_uid, pw->pw_gid ) != 0 ) {
	fprintf( stderr, "chown %d:%d %s: %s\n",
		pw->pw_uid, pw->pw_gid, to, strerror( errno ));
	return( 1 );
    }

    return( 0 );
}

    int
rte_writefile( int ac, char *av[] )
{
    char	*tmppath, *path;

    if ( ac != 2 ) {
	fprintf( stderr, "Wrong arguments\n" );
	return( 2 );
    }

    tmppath = av[ 0 ];
    path = av[ 1 ];

    if ( rename( tmppath, path ) < 0 ) {
	if ( errno == EXDEV ) {
	    syslog( LOG_INFO, "Falling back to a copy..." );
	    copy( tmppath, path, 1 );
	    if ( unlink( tmppath ) != 0 ) {
		fprintf( stderr, "unlink %s: %s\n", tmppath, strerror( errno ));
		exit( 2 );
	    }
	} else {
	    fprintf( stderr, "rename %s to %s: %s", tmppath, path,
                    strerror( errno ));
	    exit( 2 );
	}
    }

    return( 0 );
}

    int
main( int argc, char *argv[] )
{
    int				rc, err = 0;
    int				status, c, i;
    int				nactions;
    char			action[ MAXPATHLEN ] = { 0 };
    const char			*rightname = "edu.umich.rteauthexec";
    AuthorizationRef		authref;
    AuthorizationExternalForm	extauth;
    AuthorizationItem   	right = { rightname, 0, NULL, 0 };
    AuthorizationRights 	rights = { 1, &right };
    AuthorizationFlags  	flags = kAuthorizationFlagDefaults |
				    kAuthorizationFlagInteractionAllowed |
				    kAuthorizationFlagExtendRights;
    extern int			optind;
    extern char			*optarg;

    if ( argc < 2 ) {
        fprintf( stderr, "Usage: %s transcript_path\n", argv[ 0 ] );
        exit( 1 );
    }

    if (( rc = read( 0, &extauth, sizeof( extauth ))) != sizeof( extauth )) {
        syslog( LOG_INFO, "read %d bytes: %s", &extauth );
	exit( rc );
    }

    if ( AuthorizationCreateFromExternalForm( &extauth, &authref ) != 0 ) {
        fprintf( stderr, "Couldn't create authref from external form.\n" );
	exit( 2 );
    }

    if ( geteuid() != 0 ) {
	exit( selfrepair( argc, argv, rightname, authref, extauth ));
    }

    if ( AuthorizationCopyRights( authref, &rights,
	    kAuthorizationEmptyEnvironment,
	    flags, NULL ) != 0 ) {
	fprintf( stderr, "AuthorizationCopyRights failed in %s\n",
		    argv[ 0 ] );
	exit( 2 );
    }

    while (( c = getopt( argc, argv, "A:d:U:" )) != EOF ) {
	switch ( c ) {
	case 'A':		/* action */
	    if ( strlen( optarg ) >= MAXPATHLEN ) {
		fprintf( stderr, "%s: too long\n", optarg );
		exit( 2 );
	    }
	    strcpy( action, optarg );
	    break;

	case 'U':		/* RTE user */
	    if ( strlen( optarg ) >= MAXPATHLEN ) {
		fprintf( stderr, "%s: too long\n", optarg );
		exit( 2 );
	    }
	    strcpy( user, optarg );
	    break;		    

	default:
	    err++;
	}
    }

    if ( err || ( argc = ( argc - optind )) == 0 ) {
	fprintf( stderr, "Usage: %s -A action -U user args ...\n", argv[ 0 ] );
	exit( 1 );
    }
    argv += optind;

    if ( *action == NULL || *user == NULL ) {
	fprintf( stderr, "-A and -U arguments required\n" );
	exit( 1 );
    }

    nactions = sizeof( actions ) / sizeof( actions[ 0 ] );
    for ( i = 0; i < nactions; i++ ) {
	if ( strcmp( action, actions[ i ].rte_name ) == 0 ) {
	    break;
	}
    }
    if ( i >= nactions ) {
	fprintf( stderr, "Action %s unrecognized\n", action );
	exit( 1 );
    }

    rc = (*(actions[ i ].rte_func))( argc, argv );

    return( rc );
}
