/*
 * Copyright (c) 2005 Regents of The University of Michigan.
 * All Rights Reserved.  See COPYRIGHT.
 */

#include <CoreFoundation/CoreFoundation.h>
#include <Security/Authorization.h>

#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <dirent.h>
#include <fcntl.h>
#include <libgen.h>
#include <pwd.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <syslog.h>
#include <unistd.h>

#include "addspecial.h"
#include "copy.h"
#include "genplist.h"
#include "makedir.h"
#include "radpaths.h"
#include "rmdirs.h"
#include "rsmauthtool.h"
#include "selfrepair.h"

extern int	errno;
char		*radmind_path = "/var/radmind";
char		*tmpdir = "/tmp";
uid_t		uid;
gid_t		gid;

AuthorizationRef	authref = NULL;

typedef void	(*sighandler_t)( int );
void		term( int sig );

/* actions */
int		a_newdir( int, char *[] );
int		a_cpfile( int, char *[] );
int		a_mvfile( int, char *[] );
int		a_krefresh( int, char *[] );
int		a_newkfile( int, char *[] );
int		a_mvloadset( int, char *[] );
int		a_rmloadset( int, char *[] );
int		a_rm( int, char *[] );
int		a_trefresh( int, char *[] );
int		a_exec( int, char *[] );
int		a_killpg( int, char *[] );
int		a_addspecial( int, char *[] );
int		a_refreshall( int, char *[] );
int		a_cfgrefresh( int, char *[] );

struct rsm_action	actions[] = {
    { RSM_NEWDIR,	a_newdir },
    { RSM_CPFILE,	a_cpfile },
    { RSM_MVFILE,	a_mvfile },
    { RSM_NEWKFILE,	a_newkfile },
    { RSM_KREFRESH,	a_krefresh },
    { RSM_MVLOADSET,	a_mvloadset },
    { RSM_RMLOADSET,	a_rmloadset },
    { RSM_RM,		a_rm },
    { RSM_TREFRESH,	a_trefresh },
    { RSM_CFGREFRESH,	a_cfgrefresh },
    { RSM_REFRESHALL,	a_refreshall },
    { RSM_EXEC,		a_exec },
    { RSM_KILLPG,	a_killpg },
    { RSM_ADDSPECIAL,	a_addspecial },
};

struct rsm_action	refresh[] = {
    { RSM_NEWDIR,	a_newdir },
    { RSM_CPFILE,	a_cpfile },
    { RSM_MVFILE,	a_mvfile },
    { RSM_NEWKFILE,	a_newkfile },
    { RSM_MVLOADSET,	a_mvloadset },
    { RSM_RMLOADSET,	a_rmloadset },
    { RSM_RM,		a_rm },
    { RSM_ADDSPECIAL,	a_addspecial },
};

struct rsm_action	xrefresh[] = {
    { RSM_EXEC,		a_exec },
};

    void
term( int sig )
{
    syslog( LOG_INFO, "Caught signal %d, exiting.\n", sig );
    exit( sig );
}

    int
a_newdir( int ac, char *av[] )
{
    if ( ac == 0 ) {
	return( 1 );
    }

    /*
     * if the path is in the T directory,
     * also create the directory in file
     */
    if ( strstr( av[ 0 ], "transcript" ) != NULL ) {
	return( newsubdir( av[ 0 ] ));
    }

    /* otherwise, just make the folder */
    makedir( av[ 0 ] );

    return( 0 );
}

/* copy file to tmpdir */
    int
a_cpfile( int ac, char *av[] )
{
    copy( av[ 0 ], av[ 1 ], 1 );
    if ( chown( av[ 1 ], uid, gid ) != 0 ) {
	fprintf( stderr, "chown %d:%d %s: %s\n", uid, gid,
		av[ 1 ], strerror( errno ));
	return( 1 );
    }

    return( 0 );
}

    int
a_mvfile( int ac, char *av[] )
{
    char	*tmppath = av[ 0 ];
    char	*path = av[ 1 ];

    if ( rename( tmppath, path ) < 0 ) {
	if ( errno == EXDEV ) {
	    syslog( LOG_INFO, "renaming across device failed, "
		"resorting to a copy..." );
	    copy( tmppath, path, 1 );
	} else {
	    fprintf( stderr, "rename %s to %s: %s\n", tmppath,
			path, strerror( errno ));
	    return( 2 );
	}
    }

    /* XXX ownership and permissions */

    return( 0 );
}

    int
a_newkfile( int ac, char *av[] )
{
    int		fd;

    if ( ac == 1 ) {
	if (( fd = open( av[ 0 ], O_CREAT | O_EXCL, 0600 )) < 0 ) {
            fprintf( stderr, "create %s: %s", av[ 0 ], strerror( errno ));
            return( 2 );
        }
        if ( close( fd ) != 0 ) {
            fprintf( stderr, "close: %s\n", strerror( errno ));
            return( 2 );
        }
    } else if ( ac == 2 ) {
	copy( av[ 0 ], av[ 1 ], 0 );
    } else {
	fprintf( stderr, "Wrong arguments\n" );
	return( 1 );
    }

    return( 0 );
}

    int
a_krefresh( int ac, char *av[] )
{
    char	kdir[ MAXPATHLEN ];
    char	kplist[ MAXPATHLEN ];

    if ( snprintf( kdir, MAXPATHLEN, "%s/command", radmind_path )
		>= MAXPATHLEN ) {
	fprintf( stderr, "%s/command: path too long\n", radmind_path );
	return( 1 );
    }
    if ( snprintf( kplist, MAXPATHLEN, "%s/command_files.plist", tmpdir )
		>= MAXPATHLEN ) {
	fprintf( stderr, "%s/command_files.plist: path too long\n", tmpdir );
	return( 1 );
    }

    genplist( kdir, kplist, kdir );

    return( 0 );
}

    int
a_mvloadset( int ac, char *av[] )
{
    char	*src, *dest;
    char	fsrc[ MAXPATHLEN ], fdest[ MAXPATHLEN ];
    char	tdest[ MAXPATHLEN ];
    struct stat	st;

    if ( ac != 2 ) {
	fprintf( stderr, "Wrong arguments\n" );
	return( 1 );
    }

    src = av[ 0 ];
    dest = av[ 1 ];

    if ( strlen( dest ) >= MAXPATHLEN ) {
	fprintf( stderr, "%s: too long\n", dest );
	return( 1 );
    }
    strcpy( tdest, dest );

    if ( lstat( dest, &st ) != 0 ) {
	if ( errno != ENOENT ) {
	    fprintf( stderr, "lstat %s: %s\n", dest, strerror( errno ));
	    return( 1 );
	}
    }

    /* if the destination is a directory, append the filename */
    if ( S_ISDIR( st.st_mode )) {
	if (( strlen( tdest ) + strlen( basename( src )) + 2 ) >= MAXPATHLEN ) {
	    fprintf( stderr, "%s/%s: too long\n", tdest, basename( src ));
	    return( 1 );
	}
	strcat( tdest, "/" );
	strcat( tdest, basename( src ));
    }

    if ( tpath2fpath( src, fsrc, MAXPATHLEN ) != 0 ) {
	return( 1 );
    }
    if ( tpath2fpath( tdest, fdest, MAXPATHLEN ) != 0 ) {
	return( 1 );
    }

    if ( rename( src, tdest ) < 0 ) {
	fprintf( stderr, "rename %s to %s: %s\n", src,
				tdest, strerror( errno ));
	return( 1 );
    }
    
    /*
     * if the associated folder in the file dir isn't there,
     * don't try to move it.
     */
    if ( lstat( fsrc, &st ) != 0 ) {
	if ( errno == ENOENT ) {
	    return( 0 );
	}
	return( 1 );
    }
    if ( rename( fsrc, fdest ) < 0 ) {
	rename( tdest, src );
	fprintf( stderr, "rename %s to %s: %s\n", fsrc, fdest,
						strerror( errno ));
	return( 1 );
    }

    return( 0 );
}

    int
a_rmloadset( int ac, char *av[] )
{
    char		fpath[ MAXPATHLEN ];
    char		*tpath = av[ 0 ];
    struct stat		st;

    if ( tpath2fpath( tpath, fpath, MAXPATHLEN ) != 0 ) {
	return( 2 );
    }

    if ( lstat( tpath, &st ) != 0 ) {
	fprintf( stderr, "lstat %s: %s\n", tpath, strerror( errno ));
	return( 2 );
    }
			
    if ( S_ISDIR( st.st_mode )) {
	if ( rmdirs( tpath ) != 0 ) {
	    fprintf( stderr, "delete %s: %s\n", tpath, strerror( errno ));
	    return( 2 );
	}
    } else {
	if ( unlink( tpath ) < 0 ) {
	    fprintf( stderr, "unlink %s: %s\n", tpath, strerror( errno ));
	    return( 2 );
	}
    }

    if ( lstat( fpath, &st ) == 0 ) {
	if ( S_ISDIR( st.st_mode )) {
	    if ( rmdirs( fpath ) != 0 ) {
		fprintf( stderr, "delete %s: %s\n", fpath, strerror( errno ));
		return( 2 );
	    }
	}
    } else if ( errno != ENOENT ) {
	fprintf( stderr, "lstat %s: %s\n", fpath, strerror( errno ));
	return( 2 );
    }
    
    return( 0 );
}

    int
a_rm( int ac, char *av[] )
{
    struct stat	st;

    if ( lstat( av[ 0 ], &st ) != 0 ) {
	fprintf( stderr, "lstat %s: %s\n", av[ 0 ], strerror( errno ));
	return( 2 );
    }

    if ( S_ISDIR( st.st_mode )) {
	if ( rmdirs( av[ 0 ] ) != 0 ) {
	    fprintf( stderr, "delete %s: %s\n", av[ 0 ], strerror( errno ));
	    return( 2 );
	}
    } else if ( unlink( av[ 0 ] ) < 0 ) {
	fprintf( stderr, "delete %s: %s\n", av[ 0 ], strerror( errno ));
	return( 2 );
    }

    return( 0 );
}

    int
a_trefresh( int ac, char *av[] )
{
    char	tdir[ MAXPATHLEN ], tmptdir[ MAXPATHLEN ];
    char	tplist[ MAXPATHLEN ], tmptplist[ MAXPATHLEN ];

    if ( snprintf( tdir, MAXPATHLEN, "%s/transcript", radmind_path )
		>= MAXPATHLEN ) {
        fprintf( stderr, "%s/transcript: path too long\n", radmind_path );
        exit( 2 );
    }
    if ( snprintf( tmptdir, MAXPATHLEN, "%s/tmp/transcript", radmind_path )
		>= MAXPATHLEN ) {
        fprintf( stderr, "%s/tmp/transcript: path too long\n", radmind_path );
        exit( 2 );
    }
    if ( snprintf( tplist, MAXPATHLEN, "%s/transcripts.plist", tmpdir )
                        >= MAXPATHLEN ) {
        fprintf( stderr, "%s/transcripts.plist: path too long\n", tmpdir );
        exit( 2 );
    }
    if ( snprintf( tmptplist, MAXPATHLEN, "%s/tmp_transcripts.plist", tmpdir )
                        >= MAXPATHLEN ) {
        fprintf( stderr, "%s/tmp_transcripts.plist: path too long\n", tmpdir );
        exit( 2 );
    }

    genplist( tdir, tplist, tdir );
    genplist( tmptdir, tmptplist, tmptdir );

    return( 0 );
}

    int
a_cfgrefresh( int ac, char *av[] )
{
    char	src[ MAXPATHLEN ], dst[ MAXPATHLEN ];

    /* copy the config file to the tmpdir */
    if ( snprintf( src, MAXPATHLEN, "%s/config", radmind_path )
		>= MAXPATHLEN ) {
	fprintf( stderr, "%s/config: path too long\n", radmind_path );
	return( 2 );
    }
    if ( snprintf( dst, MAXPATHLEN, "%s/config", tmpdir ) >= MAXPATHLEN ) {
	fprintf( stderr, "%s/config: path too long\n", tmpdir );
	return( 2 );
    }
    copy( src, dst, 1 );
    if ( chown( dst, uid, gid ) != 0 ) {
	fprintf( stderr, "chown %d:%d %s: %s\n",
		uid, gid, dst, strerror( errno ));
	return( 2 );
    }

    return( 0 );
}

/* refresh contents of tmpdir with info from server directories */
    int
a_refreshall( int ac, char *av[] )
{
    struct stat	st;
    int		rc;
    char	src[ MAXPATHLEN ], dst[ MAXPATHLEN ];

    if (( rc = a_trefresh( ac, av )) != 0 ) {
	return( rc );
    }
    if (( rc = a_krefresh( ac, av )) != 0 ) {
	return( rc );
    }
    if (( rc = a_cfgrefresh( ac, av )) != 0 ) {
	return( rc );
    }

    /* if the allow file is there, copy it over, too */
    if ( snprintf( src, MAXPATHLEN, "%s/allow", radmind_path )
                >= MAXPATHLEN ) {
        fprintf( stderr, "%s/config: path too long\n", radmind_path );
        return( 2 );
    }
    if ( stat( src, &st ) == 0 ) {
	if ( snprintf( dst, MAXPATHLEN, "%s/allow", tmpdir ) >= MAXPATHLEN ) {
	    fprintf( stderr, "%s/allow: path too long\n", tmpdir );
	    return( 2 );
	}
	copy( src, dst, 1 ); 
	if ( chown( dst, uid, gid ) != 0 ) {
	    fprintf( stderr, "chown %d:%d %s: %s\n",
		    uid, gid, dst, strerror( errno ));
	    return( 2 );
	}
    }

    /* XXX certificates */

    return( 0 );
}

    int
a_exec( int ac, char *av[] )
{
    int		status;
    pid_t	pid;

    switch ( fork()) {
    case 0:
	execve( av[ 0 ], av, NULL );
	fprintf( stderr, "execve failed: %s\n", strerror( errno ));
	fflush( stderr );
	_exit( 2 );

    case -1:
	fprintf( stderr, "fork failed: %s\n", strerror( errno ));
	fflush( stderr );
	exit( 2 );  

    default:
	break;
    }

    pid = wait( &status );
    return( WEXITSTATUS( status ));
}

    int
a_killpg( int ac, char *av[] )
{
    pid_t	pid;

    if (( pid = strtol( av[ 0 ], NULL, 10 )) == 0 ) {
	fprintf( stderr, "strtol %s: %s", av[ 0 ], strerror( errno ));
	return( 2 );
    }

    if ( pid == -1 || pid == 0 || pid == 1 ) {
	fprintf( stderr, "Will not kill process with pid %d", pid );
	return( 1 );
    }

    if ( killpg( pid, SIGTERM ) < 0 ) {
	fprintf( stderr, "kill %d: %s\n", pid, strerror( errno ));
	return( 2 );
    }
    syslog( LOG_INFO, "killed %d", pid );

    return( 0 );
}

    int
a_addspecial( int ac, char *av[] )
{
    addspecial( av[ 0 ], av[ 1 ], av[ 2 ] );

    return( 0 );
}

    int
main( int argc, char *argv[] )
{
    int				rc = 0, i;
    int				nactions;
    int				c;
    const char			*rightname = "edu.umich.radmind.generic";
    char			action[ MAXPATHLEN ];
    char			*user = NULL;
    extern int			optind;
    extern char			*optarg;
    struct sigaction		sa, osaterm;
    struct passwd		*pw;
    struct stat			st;
    AuthorizationExternalForm	extauth;
    AuthorizationItem   	right = { rightname, 0, NULL, 0 };
    AuthorizationRights 	rights = { 1, &right };
    AuthorizationFlags  	flags = kAuthorizationFlagInteractionAllowed |
				    kAuthorizationFlagExtendRights;

    if ( argc < 2 ) {
	fprintf( stderr, "Usage: %s args\n", argv[ 0 ] );
        exit( 1 );
    }

    if (( rc = read( 0, &extauth, sizeof( extauth ))) != sizeof( extauth )) {
        fprintf( stderr, "read %d bytes: %s", rc, strerror( errno ));
	exit( rc );
    }

    if ( AuthorizationCreateFromExternalForm( &extauth, &authref ) != 0 ) {
	exit( 2 );
    }

    if ( geteuid() != 0 ) {
	exit( selfrepair( argc, argv, rightname, authref, extauth ));
    }
    if ( setuid( 0 ) != 0 ) {
	fprintf( stderr, "setuid root failed: %s.\n", strerror( errno ));
	exit( 2 );
    }

    /* catch SIGTERM */
    memset( &sa, 0, sizeof( struct sigaction ));
    sa.sa_handler = term;
    if ( sigaction( SIGTERM, &sa, &osaterm ) < 0 ) {
	fprintf( stderr, "sigaction: %s", strerror( errno ));
	exit( 1 );
    }

    if ( AuthorizationCopyRights( authref, &rights,
	    kAuthorizationEmptyEnvironment,
	    flags, NULL ) != errAuthorizationSuccess ) {
	fprintf( stderr, "AuthCopyRights failed.\n" );
	exit( 1 );
    }

    makedir( radmind_path );

    while (( c = getopt( argc, argv, "A:D:d:U:" )) != EOF ) {
	switch ( c ) {
	case 'A':	/* action */
	    if ( strlen( optarg ) >= MAXPATHLEN ) {
		fprintf( stderr, "%s: too long\n", optarg );
		exit( 2 );
	    }
	    strcpy( action, optarg );
	    break;

	case 'D':	/* radmind path */
	    radmind_path = optarg;
	    break;

	case 'd':	/* tmp dir for session */
	    tmpdir = optarg;
	    break;

	case 'U':	/* run as user */
	    user = optarg;
	    break;

	default:	/* this should never happen */
	    fprintf( stderr, "Usage: %s args\n", argv[ 0 ] );
	    exit( 1 );
	}
    }

    if ( user != NULL ) {
	/* become user for the session, if specified */
	if (( pw = getpwnam( user )) == NULL ) {
	    fprintf( stderr, "%s: no such user. Aborting...\n", user );
	    exit( 2 );
	}

	if ( seteuid( pw->pw_uid ) != 0 ) {
	    fprintf( stderr, "seteuid %s: %s\n", user, strerror( errno ));
	    exit( 2 );
	}
    }

    if (( argc = ( argc - optind )) == 0 ) {
	fprintf( stderr, "No arguments\n" );
	exit( 1 );
    }
    argv += optind;

    /* get owner of tmpdir so we know what chown to make */
    if ( stat( tmpdir, &st ) != 0 ) {
	fprintf( stderr, "stat %s: %s\n", tmpdir, strerror( errno ));
	exit( 2 );
    }
    if ( ! S_ISDIR( st.st_mode )) {
	fprintf( stderr, "%s: not a directory\n", tmpdir );
	exit( 1 );
    }
    uid = st.st_uid;
    gid = st.st_gid;

    nactions = sizeof( actions ) / sizeof( actions[ 0 ] );
    for ( i = 0; i < nactions; i++ ) {
	if ( strcmp( action, actions[ i ].ra_name ) == 0 ) {
	    break;
	}
    }
    if ( i >= nactions ) {
	fprintf( stderr, "Action %s unrecognized\n", action );
	exit( 1 );
    }

    rc = (*(actions[ i ].ra_func))( argc, argv );
    if ( rc ) {
	return( rc );
    }

    /* determine if we need to refresh plists */
    nactions = sizeof( refresh ) / sizeof( refresh[ 0 ] );
    for ( i = 0; i < nactions; i++ ) {
	if ( strcmp( action, refresh[ i ].ra_name ) == 0 ) {
	    rc = a_refreshall( argc, argv );
	    break;
	}
    }

    /* see if we need to refresh loadset listing */
    if ( strcmp( action, xrefresh[ 0 ].ra_name ) == 0 ) {
	for ( i = 0; i < argc; i++ ) {
	    if ( strstr( argv[ i ], "lmerge" ) != NULL ) {
		continue;
	    }
	    rc = a_trefresh( argc, argv );
	}
    }
  
    return( rc );
}
