/*
 * Copyright (c) 2003 Regents of The University of Michigan.
 * All Rights Reserved.  See COPYRIGHT.
 */
 
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <errno.h>
#include <libgen.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include "addspecial.h"
#include "argcargv.h"
#include "code.h"
#include "copy.h"
#include "makedir.h"

extern int		errno;
extern char		*radmind_path;

char			specialdir[ MAXPATHLEN ];
char			filedir[ MAXPATHLEN ];

struct slist {
    char		*s_path;
    char		*s_tline;
    struct slist	*s_next;
};

    void
listadd( struct slist *new, struct slist **shead )
{
    struct slist	**cur = shead;

    for ( ; *cur != NULL; cur = &( *cur )->s_next ) {
	/* add at end of list */
    }

    new->s_next = *cur;
    *cur = new;
}

    void
listfree( struct slist **shead )
{
    struct slist	*tmp;

    for ( ; *shead != NULL; *shead = tmp ) {
	tmp = ( *shead )->s_next;
	free(( *shead )->s_path );
	free(( *shead )->s_tline );
	free( *shead );
    }
}

    void
createspecialdirs( char *client, struct slist **shead )
{
    char		specialdir[ MAXPATHLEN ];
    char		cspdir[ MAXPATHLEN ];
    struct slist	**cur = shead;

    if ( snprintf( specialdir, MAXPATHLEN, "%s/special", radmind_path )
		>= MAXPATHLEN ) {
	fprintf( stderr, "%s/special: path too long\n", radmind_path );
	exit( 2 );
    }
    if ( snprintf( cspdir, MAXPATHLEN, "%s/%s", specialdir, client )
		>= MAXPATHLEN ) {
	fprintf( stderr, "%s/%s: path too long\n", specialdir, client );
	exit( 2 );
    }
    makedir( cspdir );

    for ( ; *cur != NULL; cur = &( *cur )->s_next ) {
	char		dirpath[ MAXPATHLEN ]; 
	char		*c, *p;
	int		end = 0;

	if ( snprintf( dirpath, MAXPATHLEN, "%s/%s", cspdir,
		dirname(( *cur )->s_path )) >= MAXPATHLEN ) {
	    fprintf( stderr, "%s/%s: path too long\n", cspdir,
				dirname(( *cur )->s_path ));
	    exit( 2 );
	}

	p = dirpath;

	if ( *p == '.' ) {
	    p++;
	}

	c = p;

	while ( ! end ) {
	    c += strspn( c, "/" );
	    c += strcspn( c, "/" );

	    if ( *c == '\0' ) {
		end++;
	    }
	    *c = '\0';

	    if ( mkdir( p, 0700 ) < 0 ) {
		if ( errno != EEXIST ) {
		    fprintf( stderr, "mkdir %s: %s\n", p,
				strerror( errno )); 
		    exit( 2 );
		}
	    }

	    *c = '/';
	}
    }
}

    static void
copyfiles( char *loadset, char *client, struct slist **shead )
{
    struct slist	**cur = shead;
    char		srcpath[ MAXPATHLEN ], destpath[ MAXPATHLEN ];
    FILE		*f;

    for ( ; *cur != NULL; cur = &( *cur )->s_next ) {
	if ( snprintf( destpath, MAXPATHLEN, "%s/%s/%s", specialdir,
			client, ( *cur )->s_path ) >= MAXPATHLEN ) {
	    fprintf( stderr, "%s/%s/%s: path too long\n", specialdir,
			client, ( *cur )->s_path );
	    exit( 2 );
	}

	if ( snprintf( srcpath, MAXPATHLEN, "%s/%s/%s", filedir,
			loadset, ( *cur )->s_path ) >= MAXPATHLEN ) {
	    fprintf( stderr, "%s/file/%s/%s: path too long\n", radmind_path,
			loadset, ( *cur )->s_path );
	    exit( 2 );
	}
	copy( srcpath, destpath, 0 );

	/* create individual special transcripts for the items */
	if ( strlcat( destpath, ".T", MAXPATHLEN ) >= MAXPATHLEN ) {
	    fprintf( stderr, "%s.T: path too long\n", destpath );
	    exit( 2 );
	}

	if (( f = fopen( destpath, "a" )) == NULL ) {
	    fprintf( stderr, "fopen %s: %s\n", destpath, strerror( errno ));
	    exit( errno );
	}

	fprintf( f, "%s", ( *cur )->s_tline );

	( void )fclose( f );

	memset( srcpath, '\0', MAXPATHLEN );
	memset( destpath, '\0', MAXPATHLEN );
    }
}

    static void
appendtokfile( char *kpath, struct slist **shead )
{
    FILE		*f;
    char		buf[ MAXPATHLEN ], **av;
    struct slist	**cur = shead;
    int			ac;

    if (( f = fopen( kpath, "r+" )) == NULL ) {
	fprintf( stderr, "fopen %s: %s", kpath, strerror( errno ));
	exit( 2 );
    }

    while ( fgets( buf, MAXPATHLEN, f ) != NULL ) {
	struct slist	**l = cur;
	int		len = strlen( buf );
	
	if ( buf[ len - 1 ] != '\n' ) {
	    fprintf( stderr, "%s: line too long\n", kpath );
	    ( void )fclose( f );
	    exit( 2 );
	}

	if ( *buf != 's' ) {
	    continue;
	}

	if (( ac = argcargv( buf, &av )) != 2 ) {
	    continue;
	}

	for ( cur = shead ; *cur != NULL; cur = &( *cur )->s_next ) {
	    if ( strcmp( av[ 1 ], encode(( *cur )->s_path )) == 0 ) {
		struct slist	*tmp = *cur;
		*cur = ( *cur )->s_next;
		( *l )->s_next = *cur;
		free( tmp->s_path );
		free( tmp );
		break;
	    }
	    l = cur;
	}

    }

    for ( cur = shead; *cur != NULL; cur = &( *cur )->s_next ) {
	fprintf( f, "s %s\n", encode(( *cur )->s_path ));
    }

    ( void )fclose( f );
}

    void
addspecial( char *kfile, char *client, char *loadset )
{
    FILE		*f = NULL;
    char		buf[ MAXPATHLEN ];
    char		ltpath[ MAXPATHLEN ];
    char		**col;
    int			colcount;
    struct slist	*head = NULL, *new = NULL;

    if ( snprintf( specialdir, MAXPATHLEN, "%s/special", radmind_path )
		>= MAXPATHLEN ) {
	fprintf( stderr, "%s/special: path too long\n", radmind_path );
	exit( 2 );
    }
    makedir( specialdir );

    if ( snprintf( filedir, MAXPATHLEN, "%s/file", radmind_path )
		>= MAXPATHLEN ) {
	fprintf( stderr, "%s/file: path too long\n", radmind_path );
	exit( 2 );
    }
    makedir( filedir );

    if ( snprintf( ltpath, MAXPATHLEN, "%s/transcript/%s",
		radmind_path, loadset ) >= MAXPATHLEN ) {
	fprintf( stderr, "%s/transcript/%s: path too long\n",
		radmind_path, loadset );
	exit( 2 );
    }

    if (( f = fopen( ltpath, "r" )) == NULL ) {
	fprintf( stderr, "fopen %s: %s\n", ltpath, strerror( errno ));
	exit( 2 );
    }

    while ( fgets( buf, MAXPATHLEN, f ) != NULL ) {
	char		*line;
	int		len = strlen( buf );
	
	if ( buf[ len - 1 ] != '\n' ) {
	    fprintf( stderr, "%s: line too long\n", loadset );
	    ( void )fclose( f );
	    exit( 2 );
	}

	if (( line = strdup( buf )) == NULL ) {
	    fprintf( stderr, "strdup %s: %s\n", buf, strerror( errno ));
	    ( void )fclose( f );
	    exit( errno );
	}

	if (( colcount = argcargv( buf, &col )) != 8 ||
		( *col[ 0 ] != 'f' && *col[ 0 ] != 'a' )) {
	    free( line );
	    continue;
	}

	if (( new = ( void * )malloc( sizeof( struct slist ))) == NULL ) {
	    perror( "malloc" );
	    ( void )fclose( f );
	    exit( 2 );
	}
	if (( new->s_path = strdup( decode( col[ 1 ] ))) == NULL ) {
	    fprintf( stderr, "strdup %s: %s\n", col[ 1 ], strerror( errno ));
	    ( void )fclose( f );
	    exit( errno );
	}
	if (( new->s_tline = strdup( line )) == NULL ) {
	    fprintf( stderr, "strdup %s: %s\n", line, strerror( errno ));
	    ( void )fclose( f );
	    exit( errno );
	}
	free( line );

	listadd( new, &head );
    }
    ( void )fclose( f );

    createspecialdirs( client, &head );
    copyfiles( loadset, client, &head );

    appendtokfile( kfile, &head );
    listfree( &head );
}
