#include <CoreFoundation/CoreFoundation.h>

#include <sys/types.h>
#include <sys/param.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#include "automator.h"
#include "copy.h"
#include "makedir.h"

#define ETCDIR		"/etc"
#define HOOKDIR		"/etc/hooks"
#define HOOKDIR_UNUSED	"/etc/hooks/unused"
#define STARTUPITEMDIR	"/Library/StartupItems"
#define SIRADMINDDIR	"/Library/StartupItems/RadmindClient"

    void
installitem( char *srcname, char *srcdir, char *destdir, char *destname )
{
    char		srcpath[ MAXPATHLEN ], destpath[ MAXPATHLEN ];
    char		unusedpath[ MAXPATHLEN ];

    ( void )strlcpy( destpath, destdir, sizeof( destpath ));
    ( void )strlcat( destpath, "/", sizeof( destpath ));
    ( void )strlcat( destpath, destname, sizeof( destpath ));

    if ( access( destpath, F_OK ) == 0 && 
		strcmp( destdir, ETCDIR ) != 0 ) {
	/* put old hook in unused directory */
	makedir( HOOKDIR_UNUSED );

	( void )strlcpy( unusedpath, HOOKDIR_UNUSED, sizeof( unusedpath ));
	( void )strlcat( unusedpath, "/", sizeof( unusedpath ));
	( void )strlcat( unusedpath, destname, sizeof( unusedpath ));

	if ( rename( destpath, unusedpath ) < 0 ) {
	    fprintf( stderr, "rename %s to %s: %s\n", destpath,
				unusedpath, strerror( errno ));
	    syslog( LOG_ERR, "rename %s to %s: %s\n", destpath,
				unusedpath, strerror( errno ));
	    exit( errno );
	}
    } else if ( errno != ENOENT && errno != EEXIST ) {
	fprintf( stderr, "access %s: %s\n", destpath, strerror( errno ));
	syslog( LOG_ERR, "access %s: %s", destpath, strerror( errno ));
	exit( errno );
    }

    ( void )strlcpy( srcpath, srcdir, sizeof( srcpath ));
    ( void )strlcat( srcpath, "/", sizeof( srcpath ));
    ( void )strlcat( srcpath, srcname, sizeof( srcpath ));
	
    copy( srcpath, destpath, 0700 );
}

    void
removeitem( char *name, char *dirname )
{
    char		path[ MAXPATHLEN ];

    ( void )strlcpy( path, dirname, sizeof( path ));
    ( void )strlcat( path, "/", sizeof( path ));
    ( void )strlcat( path, name, sizeof( path ));

    if ( unlink( path ) < 0 ) {
	if ( errno != ENOENT ) {
	    fprintf( stderr, "unlink %s: %s\n", path, strerror( errno ));
	    syslog( LOG_ERR, "unlink %s: %s\n", path, strerror( errno ));
	    exit( errno );
	}
    }
}

    void
installhooks( char *rsrcdirpath )
{
    Boolean		continueOnReboot, validKey;
    int			i;
    char		*etchooks[] = { "login.hook", "logout.hook",
					"radmind.defaults" };
    char		*hooks[] = { "LI00CheckUsername.hook",
					"masterlogin.sh",
					"masterlogout.sh",
					"radmind.hook" };

    makedir( HOOKDIR );

    /* copy scripts to /etc/hooks */
    for ( i = 0; i < 4; i++ ) {
	installitem( hooks[ i ], rsrcdirpath, HOOKDIR, hooks[ i ] );
    }

    /* copy certain scripts to /etc */
    for ( i = 0; i < 3; i++ ) {
	installitem( etchooks[ i ], rsrcdirpath, ETCDIR, etchooks[ i ] );
    }

    /* create LORadmind.hook symlink */
    if ( symlink( "/etc/hooks/radmind.hook",
			"/etc/hooks/LO99Radmind.hook" ) < 0 ) {
	if ( errno != EEXIST ) {
	    fprintf( stderr, "symlink failed: %s\n", strerror( errno ));
	    exit( 2 );
	}
    }

    /* install RadmindClient startup item, if requested */
    continueOnReboot = CFPreferencesGetAppBooleanValue(
		    CFSTR( "RAContinueOnReboot" ),
		    CFSTR( "/Library/Preferences/edu.umich.radmindassistant" ),
			&validKey );

    if ( ! continueOnReboot || ! validKey ) {	/* remove the startup item */
	removeitem( "RadmindClient", SIRADMINDDIR );
	removeitem( "StartupParameters.plist", SIRADMINDDIR );
	if ( rmdir( SIRADMINDDIR ) < 0 ) {
	    if ( errno != ENOENT ) {
		fprintf( stderr, "unlink %s: %s", SIRADMINDDIR,
						strerror( errno ));
		syslog( LOG_ERR, "unlink %s: %s", SIRADMINDDIR,
						strerror( errno ));
		exit( errno );
	    }
	}
    } else {	/* create the startup item */
	makedir( STARTUPITEMDIR );
	makedir( SIRADMINDDIR );

	installitem( "RadmindClient", rsrcdirpath,
		    SIRADMINDDIR, "RadmindClient" );
	installitem( "RadmindClient-StartupParameters.plist", rsrcdirpath,
		    SIRADMINDDIR, "StartupParameters.plist" );
    }
}

    void
installperiodic( int periodictype, char *rsrcdirpath )
{
    char	*daily = "/etc/periodic/daily/600.radmind";
    char	*weekly = "/etc/periodic/weekly/600.radmind";
    char	*monthly = "/etc/periodic/monthly/600.radmind";
    char	*dest = NULL, src[ MAXPATHLEN ];

    /* remove any installed radmind periodic scripts first */
    if ( unlink( daily ) < 0 ) {
	if ( errno != ENOENT ) {
	    fprintf( stderr, "Failed to remove %s: %s\n",
		    daily, strerror( errno ));
	    exit( errno );
	}
    }
    if ( unlink( weekly ) < 0 ) {
	if ( errno != ENOENT ) {
	    fprintf( stderr, "Failed to remove %s: %s\n",
		    weekly, strerror( errno ));
	    exit( errno );
	}
    }
    if ( unlink( monthly ) < 0 ) {
	if ( errno != ENOENT ) {
                fprintf( stderr, "Failed to remove %s: %s\n",
                        monthly, strerror( errno ));
                exit( errno );
        }    
    }

    switch ( periodictype ) {
    case 0:		/* don't run periodic radmind script */
	return;

    case 1:		/* daily */
	dest = daily;
	break;

    case 2:		/* weekly */
	dest = weekly;
	break;

    case 3:		/* monthly */
	dest = monthly;
	break;
    }

    ( void )strlcpy( src, rsrcdirpath, sizeof( src ));
    ( void )strlcat( src, "/600.radmind", sizeof( src ));

    copy( src, dest, 0700 );
}

    void
installumscripts( char *rsrcdirpath )
{
    int		i;
    char	*umscripts[] = { "LI10SetupUserEnvironment.hook",
				"LO00EjectDisks.hook",
				"LO10CreateTemplateHomeCopy.hook",
				"LO20UserAccounting.hook",
				"ejectexclude" };

    for ( i = 0; i < 5; i++ ) {
	installitem( umscripts[ i ], rsrcdirpath, HOOKDIR, umscripts[ i ] );
    }
}

    void
removeumscripts()
{
    int		i;
    char	*umscripts[] = { "LI10SetupUserEnvironment.hook",
				"LO00EjectDisks.hook",
				"LO10CreateTemplateHomeCopy.hook",
				"LO20UserAccounting.hook",
				"ejectexclude" };

    for ( i = 0; i < 5; i++ ) {
	removeitem( umscripts[ i ], HOOKDIR );
    }
}
