/*
 * Copyright (c) 2005 Regents of The University of Michigan.
 * All Rights Reserved.  See COPYRIGHT.
 */

#import "UMLoginItem.h"

#include <Carbon/Carbon.h>
#include <CoreFoundation/CoreFoundation.h>

@implementation UMLoginItem

static UMLoginItem	    *_umSharedLoginItem = nil;

+ ( UMLoginItem * )sharedLoginItem
{
    if ( _umSharedLoginItem == nil ) {
	_umSharedLoginItem = [[ UMLoginItem alloc ] init ];
    }
    
    return( _umSharedLoginItem );
}

- ( BOOL )addLoginItem: ( NSString * )itemPath hide: ( BOOL )hide
{
    NSArray		    *prefArray = nil;
    NSMutableArray	    *autoLaunchedApps = nil;
    NSData		    *aliasData = nil;
    NSDictionary	    *loginItem = nil;
    FSRef		    ref;
    AliasHandle		    handle = NULL;
    OSStatus		    err;
    
    if ( itemPath == nil || [ itemPath length ] == 0 ) {
	return( NO );
    }
    
    /* remove it from the loginwindow prefs if it's already there */
    ( void )[ self removeLoginItem: itemPath ];
    
    prefArray = ( NSArray * )CFPreferencesCopyValue(
			    CFSTR( "AutoLaunchedApplicationDictionary" ),
			    CFSTR( "loginwindow" ),
			    kCFPreferencesCurrentUser,
			    kCFPreferencesAnyHost );
    if ( prefArray == nil ) {
	autoLaunchedApps = [[[ NSMutableArray alloc ] init ] autorelease ];
    } else {
	/* get a mutable copy of the array */
	autoLaunchedApps = [[ prefArray mutableCopy ] autorelease ];
	[ prefArray autorelease ];
    }
    
    err = FSPathMakeRef(( const UInt8 * )[ itemPath fileSystemRepresentation ],
		    &ref, NULL );
    if ( err != noErr ) {
	NSLog( @"FSPathMakeRef %@ returned %d", itemPath, ( int )err );
	return( NO );
    }
    
    err = FSNewAlias( NULL, &ref, &handle );
    if ( err != noErr && handle == NULL ) {
	NSLog( @"FSNewAlias returned %d", ( int )err );
	return( NO );
    }
    
    aliasData = [ NSData dataWithBytes: *handle
		    length: GetHandleSize(( Handle )handle ) ];
    
    loginItem = [ NSDictionary dictionaryWithObjectsAndKeys:
		    aliasData, @"AliasData",
		    [ NSNumber numberWithBool: hide ], @"Hide",
		    itemPath, @"Path", nil ];
		    
    if ( handle != NULL ) {
	DisposeHandle(( Handle )handle );
    }

    [ autoLaunchedApps addObject: loginItem ];
    
    CFPreferencesSetValue( CFSTR( "AutoLaunchedApplicationDictionary" ),
			    autoLaunchedApps,
			    CFSTR( "loginwindow" ),
			    kCFPreferencesCurrentUser,
			    kCFPreferencesAnyHost );
			    
    if ( CFPreferencesSynchronize( CFSTR( "loginwindow" ),
			    kCFPreferencesCurrentUser,
			    kCFPreferencesAnyHost ) != true ) {
	NSLog( @"CFPreferencesSynchronize failed" );
	return( NO );
    }
    
    return( YES );
}

- ( BOOL )removeLoginItem: ( NSString * )itemPath
{
    NSArray		    *prefArray = nil;
    NSMutableArray	    *autoLaunchedApps = nil;
    NSDictionary	    *dict = nil;
    int			    i;
    
    if ( itemPath == nil || [ itemPath length ] == 0 ) {
	return( NO );
    }
    
    prefArray = ( NSArray * )CFPreferencesCopyValue(
			    CFSTR( "AutoLaunchedApplicationDictionary" ),
			    CFSTR( "loginwindow" ),
			    kCFPreferencesCurrentUser,
			    kCFPreferencesAnyHost );
			
    if ( prefArray == nil ) {
	/* nothing there to remove, consider this successful */
	return( YES );
    } else {
	/* get a mutable copy to modify */
	autoLaunchedApps = [[ prefArray mutableCopy ] autorelease ];
	[ prefArray autorelease ];
    }
    
    for ( i = 0; i < [ autoLaunchedApps count ]; i++ ) {
	dict = [ autoLaunchedApps objectAtIndex: i ];
	if ( [[ dict objectForKey: @"Path" ] isEqualToString: itemPath ] ) {
	    break;
	}
    }
    if ( i >= [ autoLaunchedApps count ] ) {
	NSLog( @"%@ not found in login items array", itemPath );
	return( NO );
    }
    
    [ autoLaunchedApps removeObjectAtIndex: i ];
    
    CFPreferencesSetValue( CFSTR( "AutoLaunchedApplicationDictionary" ),
			    autoLaunchedApps,
			    CFSTR( "loginwindow" ),
			    kCFPreferencesCurrentUser,
			    kCFPreferencesAnyHost );
    
    if ( CFPreferencesSynchronize( CFSTR( "loginwindow" ),
			    kCFPreferencesCurrentUser,
			    kCFPreferencesAnyHost ) != true ) {
	NSLog( @"CFPreferencesSynchronize failed" );
	return( NO );
    }
    
    return( YES );
}

- ( void )dealloc
{
    if ( _umSharedLoginItem != nil ) {
	[ _umSharedLoginItem release ];
	_umSharedLoginItem = nil;
    }
    
    [ super dealloc ];
}

@end
