/*
 * Copyright (c) 2003 Regents of The University of Michigan.
 * All Rights Reserved.  See COPYRIGHT.
 */

#import "RCMPreferences.h"
#import "RAAuthExec.h"
#import "RAVersionCompare.h"
#import "NSPanel(Resizing).h"
#import "UMLoginItem.h"

#define RADMIND_CASE_INSENSITIVITY_VERSION    151

@implementation RCMPreferences

- ( void )toolbarSetup
{
    NSToolbar *preftbar = [[[ NSToolbar alloc ] initWithIdentifier: 
				    @"RAPrefToolbar" ] autorelease ];
    
    [ preftbar setAllowsUserCustomization: NO ];
    [ preftbar setAutosavesConfiguration: NO ];
    [ preftbar setDisplayMode: NSToolbarDisplayModeIconAndLabel ];
    
    [ preftbar setDelegate: self ];
    [ prefPanel setToolbar: preftbar ];
}

- ( void )awakeFromNib
{		
    radmindServerList = nil;
    certificateManager = nil;
    scriptManager = nil;
    
    [ prefRadmindHost removeAllItems ];
    [ prefRadmindServerTable setDataSource: self ];
    [ prefRadmindServerTable setDelegate: self ];
    [ self toolbarSetup ];
    [ self showGeneralPreferences: nil ];
    
    /* select the General prefs (10.3 only) */
    if ( [ prefPanel toolbar ] != nil &&
                [[ prefPanel toolbar ] respondsToSelector:
                @selector( setSelectedItemIdentifier: ) ] ) {
        [[ prefPanel toolbar ] setSelectedItemIdentifier:
                            RAPrefToolbarGeneralIdentifier ];
    }
    
    [ prefPanel center ];
}

- ( NSPanel * )prefPanel
{
    return( prefPanel );
}

/**/
/* required toolbar delegate methods */
/**/

- ( NSToolbarItem * )toolbar: ( NSToolbar * )toolbar
	itemForItemIdentifier: ( NSString * )itemIdent
	willBeInsertedIntoToolbar: ( BOOL )flag
{
    NSToolbarItem *preftbarItem = [[[ NSToolbarItem alloc ]
                                    initWithItemIdentifier: itemIdent ] autorelease ];
    
    if ( [ itemIdent isEqualToString: RAPrefToolbarGeneralIdentifier ] ) {
        [ preftbarItem setLabel:
                NSLocalizedStringFromTable( @"General", @"RAPrefToolbar",
                                            @"General" ) ];
        [ preftbarItem setPaletteLabel:
                NSLocalizedStringFromTable( @"General", @"RAPrefToolbar",
                                            @"General" ) ];
        [ preftbarItem setToolTip:
                NSLocalizedStringFromTable( @"Show General Preferences", @"RAPrefToolbar",
                                            @"Show General Preferences" ) ];
        [ preftbarItem setImage: [ NSImage imageNamed: @"generalprefs.png" ]];
        [ preftbarItem setAction: @selector( showGeneralPreferences: ) ];
        [ preftbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RAPrefToolbarSecurityIdentifier ] ) {
        [ preftbarItem setLabel:
                NSLocalizedStringFromTable( @"Security", @"RAPrefToolbar",
                                            @"Security" ) ];
        [ preftbarItem setPaletteLabel:
                NSLocalizedStringFromTable( @"Security", @"RAPrefToolbar",
                                            @"Security" ) ];
        [ preftbarItem setToolTip:
                NSLocalizedStringFromTable( @"Show Security Preferences", @"RAPrefToolbar",
                                            @"Show Security Preferences" ) ];
        [ preftbarItem setImage: [ NSImage imageNamed: @"security.png" ]];
        [ preftbarItem setAction: @selector( showSecurityPreferences: ) ];
        [ preftbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RAPrefToolbarTranscriptIdentifier ] ) {
	[ preftbarItem setLabel:
                NSLocalizedStringFromTable( @"Transcripts", @"RAPrefToolbar",
                                            @"Transcripts" ) ];
        [ preftbarItem setPaletteLabel:
                NSLocalizedStringFromTable( @"Transcripts", @"RAPrefToolbar",
                                            @"Transcripts" ) ];
        [ preftbarItem setToolTip:
                NSLocalizedStringFromTable( @"Show Transcript Preferences", @"RAPrefToolbar",
                                            @"Show Transcripts Preferences" ) ];
        [ preftbarItem setImage: [ NSImage imageNamed: @"transcript.png" ]];
        [ preftbarItem setAction: @selector( showTranscriptPreferences: ) ];
        [ preftbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RAPrefToolbarAutomationIdentifier ] ) {
        [ preftbarItem setLabel:
                NSLocalizedStringFromTable( @"Automation", @"RAPrefToolbar",
                                            @"Automation" ) ];
        [ preftbarItem setPaletteLabel:
                NSLocalizedStringFromTable( @"Automation", @"RAPrefToolbar",
                                            @"Automation" ) ];
        [ preftbarItem setToolTip:
                NSLocalizedStringFromTable( @"Show Automation Preferences", @"RAPrefToolbar",
                                            @"Show Automation Preferences" ) ];
        [ preftbarItem setImage: [ NSImage imageNamed: @"automationprefs.png" ]];
        [ preftbarItem setAction: @selector( showAutomationPreferences: ) ];
        [ preftbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RAPrefToolbarUserManagmentIdentifier ] ) {
        [ preftbarItem setLabel:
                NSLocalizedStringFromTable( @"User Management", @"RAPrefToolbar",
                                            @"User Management" ) ];
        [ preftbarItem setPaletteLabel:
                NSLocalizedStringFromTable( @"User Management", @"RAPrefToolbar",
                                            @"User Management" ) ];
        [ preftbarItem setToolTip:
                NSLocalizedStringFromTable( @"Show User Management Preferences", @"RAPrefToolbar",
                                            @"Show User Management Preferences" ) ];
        [ preftbarItem setImage: [ NSImage imageNamed: @"usermanagement.png" ]];
        [ preftbarItem setAction: @selector( showUserManagementSettings: ) ];
        [ preftbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RAPrefToolbarAdvancedIdentifier ] ) {
        [ preftbarItem setLabel:
                NSLocalizedStringFromTable( @"Advanced", @"RAPrefToolbar",
                                            @"Advanced" ) ];
        [ preftbarItem setPaletteLabel:
                NSLocalizedStringFromTable( @"Advanced", @"RAPrefToolbar",
                                            @"Advanced" ) ];
        [ preftbarItem setToolTip:
                NSLocalizedStringFromTable( @"Show Advanced Preferences", @"RAPrefToolbar",
                                            @"Show Advanced Preferences" ) ];
        [ preftbarItem setImage: [ NSImage imageNamed: @"advancedprefs.png" ]];
        [ preftbarItem setAction: @selector( showAdvancedPrefs: ) ];
        [ preftbarItem setTarget: self ];
    }
                
    return( preftbarItem );
}

- ( BOOL )validateToolbarItem: ( NSToolbarItem * )tItem
{
    return( YES );
}

- ( NSArray * )toolbarDefaultItemIdentifiers: ( NSToolbar * )toolbar
{
    NSArray	*tmp = [ NSArray arrayWithObjects:
                            RAPrefToolbarGeneralIdentifier,
                            RAPrefToolbarSecurityIdentifier,
			    RAPrefToolbarTranscriptIdentifier,
                            RAPrefToolbarAutomationIdentifier,
                            RAPrefToolbarUserManagmentIdentifier,
                            RAPrefToolbarAdvancedIdentifier, nil ];
                            
    return( tmp );
}

- ( NSArray * )toolbarAllowedItemIdentifiers: ( NSToolbar * )toolbar
{
    NSArray	*tmp = [ NSArray arrayWithObjects:
                            RAPrefToolbarGeneralIdentifier,
                            RAPrefToolbarSecurityIdentifier,
			    RAPrefToolbarTranscriptIdentifier,
                            RAPrefToolbarAutomationIdentifier,
                            RAPrefToolbarUserManagmentIdentifier,
                            RAPrefToolbarAdvancedIdentifier, nil ];
                            
    return( tmp );
}

    /* 10.3+ only */
- ( NSArray * )toolbarSelectableItemIdentifiers: ( NSToolbar * )toolbar
{
    NSArray     *tmp = [ NSArray arrayWithObjects:
                            RAPrefToolbarGeneralIdentifier,
                            RAPrefToolbarSecurityIdentifier,
                            RAPrefToolbarTranscriptIdentifier,
                            RAPrefToolbarAutomationIdentifier,
                            RAPrefToolbarUserManagmentIdentifier,
                            RAPrefToolbarAdvancedIdentifier, nil ];
    
    return( tmp );
}
/* end required toolbar delegate methods */

/* update preferences from array of strings to array of dictionaries */
- ( void )updateServerListPreferences: ( NSArray * )serverList
{
    NSMutableArray	*newServerArray = nil;
    NSDictionary	*serverInfo = nil;
    int			i;
    
    if ( [ serverList count ] &&
	    [[ serverList objectAtIndex: 0 ] isKindOfClass: [ NSString class ]] ) {
	newServerArray = [ NSMutableArray array ];
	
	for ( i = 0; i < [ serverList count ]; i++ ) {
	    serverInfo = [ NSDictionary dictionaryWithObjectsAndKeys:
			    [ serverList objectAtIndex: i ],
				@"RadmindServerName",
			    @"",
				@"RadmindServerPort",
			    [ NSNumber numberWithInt: 0 ],
				@"RadmindServerCompressionLevel",
			    nil ];
	    [ newServerArray addObject: serverInfo ];
	}
	
	[[ NSUserDefaults standardUserDefaults ]
		setObject: newServerArray forKey: @"RadmindServers" ];
    }
    
    [[ NSUserDefaults standardUserDefaults ]
	    removeObjectForKey: @"radmindhosts" ];
    [[ NSUserDefaults standardUserDefaults ] synchronize ];
}

- ( void )prefsReload
{
    NSUserDefaults	*defaults =
                            [ NSUserDefaults standardUserDefaults ];
    NSString		*fsdiffpath = [ defaults objectForKey:
                                        @"runfrompath" ];
    NSString		*radmind = [ defaults objectForKey:
                                    @"radmindhost" ];
    NSArray		*defaultHosts = [ defaults objectForKey: @"radmindhosts" ];
    NSString		*editor = [ defaults objectForKey:
                                    @"transeditor" ];
    BOOL		userauth = [ defaults boolForKey:
                                        @"userauth" ];
    BOOL		dontremovelocks = [ defaults boolForKey: @"DontRemoveFileLocks" ];
    BOOL		errors = [ defaults boolForKey: @"DieOnErrorsDuringUpload" ];
    BOOL		preapply = [ defaults boolForKey: @"RAAlwaysPreProcess" ];
    BOOL		postapply = [ defaults boolForKey: @"RAAlwaysPostProcess" ];
    BOOL		caseInsensitive = [ defaults boolForKey: @"RACaseInsensitive" ];
    BOOL		rum = [ defaults boolForKey: @"RadmindUpdateMonitorEnabled" ];
    NSString		*username = [ defaults stringForKey:
                                        @"username" ];
    NSArray		*alteditors = [ defaults objectForKey: @"alteditors" ];
    NSNumber		*tlsLevel = [ defaults objectForKey:
                                    @"tlslevel" ];
    NSDictionary	*serverInfo = nil;
    int			i = 0;
    
    if ( tlsLevel != nil ) {
        [ prefTLSLevel selectItemAtIndex: [ tlsLevel intValue ]];
    }
    
    [ prefRadmindHost removeAllItems ];
    [ prefRadmindHost setEnabled: NO ];
    
    if ( defaultHosts != nil ) {
	[ self updateServerListPreferences: defaultHosts ];
    }
    defaultHosts = [[ NSUserDefaults standardUserDefaults ]
			objectForKey: @"RadmindServers" ];
    
    if ( radmindServerList == nil ) {
        radmindServerList = [[ NSMutableArray alloc ] init ];
    }
    [ radmindServerList removeAllObjects ];
    [ radmindServerList addObjectsFromArray: defaultHosts ];
    
    for ( i = 0; i < [ radmindServerList count ]; i++ ) {
        [ prefRadmindHost insertItemWithTitle:
                        [[ radmindServerList objectAtIndex: i ]
				objectForKey: @"RadmindServerName" ]
                        atIndex: 0 ];
	[ prefRadmindHost setEnabled: YES ];
	if ( serverInfo == nil ) {
	    if ( [[[ radmindServerList objectAtIndex: i ]
			objectForKey: @"RadmindServerName" ]
			isEqualToString: radmind ] ) {
		serverInfo = [ radmindServerList objectAtIndex: i ];
	    }
	}
    }
    
    [ prefServerPortSummary setStringValue: @"" ];
    [ prefServerCompressionSummary setStringValue: @"" ];
    
    if ( serverInfo != nil ) {
	NSString	*portString = @"";
	NSString	*compressionString = @"";
	int		port, zlevel;
	
	if ( [ serverInfo objectForKey: @"RadmindServerPort" ] != nil ) {
	    port = [[ serverInfo objectForKey: @"RadmindServerPort" ] intValue ];
	    if ( port > 0 ) {
		portString = [ NSString stringWithFormat:
				NSLocalizedString( @"Port %d", @"Port %d" ),
				port ];
		[ prefServerPortSummary setStringValue: portString ];
	    }
	}
	zlevel = [[ serverInfo objectForKey:
		    @"RadmindServerCompressionLevel" ] intValue ];
	if ( zlevel > 1 && zlevel < 10 ) {
	    compressionString = [ NSString stringWithFormat:
				NSLocalizedString( @"Compression level %d",
				    @"Compression level %d" ), zlevel ];
	    [ prefServerCompressionSummary setStringValue: compressionString ];
	}
    }
    
    if ( radmind != nil ) {
        if ( ! [[ prefRadmindHost itemTitles ] containsObject: radmind ] ) {
            [ prefRadmindHost insertItemWithTitle: radmind atIndex: 0 ];
        }
        [ prefRadmindHost selectItemWithTitle: radmind ];
    }
    
    if ( rum ) {
	[ prefEnableRadmindUpdateMonitor setState: NSOnState ];
    } else {
	[ prefEnableRadmindUpdateMonitor setState: NSOffState ];
    }
    
    if ( userauth ) {
        [ prefEnableUserAuth setState: NSOnState ];
        [ prefUserAuthName setEnabled: YES ];
    }
    if ( username != nil ) {
	[ prefUserAuthName setStringValue: username ];
    }
    
    if ( fsdiffpath != nil ) {
        [ prefFsdiffPath selectItemWithTitle: fsdiffpath ];
    }
    if ( alteditors != nil ) {
        for ( i = 0; i < [ alteditors count ]; i++ ) {
            [ prefTransEditor insertItemWithTitle:
                        [[ alteditors objectAtIndex: i ] objectForKey: @"name" ]
                            atIndex: 0 ];
        }
    }
    if ( editor != nil ) {
        [ prefTransEditor selectItemWithTitle: editor ];
    }
    if ( caseInsensitive == NO ) {
	[ prefCaseSensitivityPopUp selectItemAtIndex: 1 ];
    }
    
    if ( [ defaults boolForKey: @"cksum" ] ) {
        [ prefGlobalCksum selectCellAtRow: 1 column: 0 ];
    } else {
        [ prefGlobalCksum selectCellAtRow: 0 column: 0 ];
    }
    
    if ( dontremovelocks == NO ) {
        [ prefLapplyForce setState: NSOnState ];
    }
    if ( errors == NO ) {
        [ prefIgnoreLcreateErrors setState: NSOnState ];
    }
    
    if ( preapply == YES ) {
	[ prefAlwaysPreApply setState: NSOnState ];
    }
    if ( postapply == YES ) {
	[ prefAlwaysPostApply setState: NSOnState ];
    }
    
    [ defaults synchronize ];
}

- ( void )showGeneralPreferences: ( id )sender
{
    if ( [[ prefPanel contentView ] isEqualTo: generalPrefsView ] ) {
	return;
    }

    [ contentBox setContentView: nil ];
    [ prefPanel resizeForContentView: generalPrefsView ];
    [ contentBox setContentView: generalPrefsView ];
    [ prefPanel setTitle: @"Radmind Assistant: General Preferences" ];
}

- ( void )showSecurityPreferences: ( id )sender
{
    if ( [[ prefPanel contentView ] isEqualTo: securityPrefsView ] ) {
	return;
    }
    
    [ contentBox setContentView: nil ];
    [ prefPanel resizeForContentView: securityPrefsView ];
    [ contentBox setContentView: securityPrefsView ];
    [ prefPanel setTitle: @"Radmind Assistant: Security Preferences" ];
}

- ( void )showTranscriptPreferences: ( id )sender
{
    int			    version;
    
    if ( [[ prefPanel contentView ] isEqualTo: transcriptPrefsView ] ) {
	return;
    }
    
    version = [[ RAVersionCompare sharedComparison ] radmindVersion ];

    if ( version < RADMIND_CASE_INSENSITIVITY_VERSION ) {
	[ prefCaseSensitivityPopUp setEnabled: NO ];
    }
    
    [ contentBox setContentView: nil ];
    [ prefPanel resizeForContentView: transcriptPrefsView ];
    [ contentBox setContentView: transcriptPrefsView ];
    [ prefPanel setTitle: @"Radmind Assistant: Transcript Preferences" ];
}

- ( void )showAdvancedPrefs: ( id )sender
{
    NSString                *lastPrefSetDate = nil;
    
    if ( [[ prefPanel contentView ] isEqualTo: advancedPrefsView ] ) {
        return;
    }
    
    lastPrefSetDate = [[ NSUserDefaults standardUserDefaults ]
                            objectForKey: @"RAGlobalPreferencesLastSetDate" ];
    
    if ( lastPrefSetDate != nil ) {
        [ prefGlobalPreferencesDateField setStringValue:
                [ NSString stringWithFormat:
                    NSLocalizedString( @"Global preferences last set on %@.",
                                        @"Global preferences last set on %@." ),
                [[ NSUserDefaults standardUserDefaults ]
                    objectForKey: @"RAGlobalPreferencesLastSetDate" ]]];
    }
    
    [ contentBox setContentView: nil ];
    [ prefPanel resizeForContentView: advancedPrefsView ];
    [ contentBox setContentView: advancedPrefsView ];
    [ prefPanel setTitle: NSLocalizedString(
            @"Radmind Assistant: Advanced Preferences",
            @"Radmind Assistant: Advanced Preferences" ) ];
}

- ( void )showUserManagementSettings: ( id )sender
{
    NSString                *templateUserName = nil;
    int                     maxlife, maxusage;
    
    if ( [[ prefPanel contentView ] isEqualTo: userManagementView ] ) {
        return;
    }
    
    [ prefUserManagementSwitch setState:
            [[ NSUserDefaults standardUserDefaults ]
                                boolForKey: @"RAUserManagementEnabled" ]];
                
    if (( templateUserName = [[ NSUserDefaults standardUserDefaults ]
                            objectForKey: @"UMTemplateUserName" ] ) != nil ) {
        [ prefTemplateUserNameField setStringValue: templateUserName ];
    }
    
    if (( maxlife = [[[[[ NSUserDefaults standardUserDefaults ]
                            objectForKey: @"UMMaxHomeLife" ]
                            componentsSeparatedByString: @" " ]
                            objectAtIndex: 0 ] intValue ] ) == 0 ) {
        maxlife = 24;
    }
    
    if (( maxusage = [[ NSUserDefaults standardUserDefaults ]
                            integerForKey: @"UMMaxDiskUsage" ] ) == 0 ) {
        maxusage = 75;
    }
    
    [ prefMaxHomeLifeField setStringValue:
            [ NSString stringWithFormat: @"%d", maxlife ]];
    [ prefMaxDiskUsageField setStringValue:
            [ NSString stringWithFormat: @"%d", maxusage ]];
    
    [ contentBox setContentView: nil ];
    [ prefPanel resizeForContentView: userManagementView ];
    [ contentBox setContentView: userManagementView ];
    [ prefPanel setTitle: NSLocalizedString(
            @"Radmind Assistant: User Management Settings",
            @"Radmind Assistant: User Management Settings" ) ];
            
    if ( ! [[ NSUserDefaults standardUserDefaults ]
                boolForKey: @"RAHideUserManagementWarning" ] ) {
        NSBeginAlertSheet( NSLocalizedString( @"Warning: Setting up user management "
                                                @"involves the installation of several "
                                                @"files, the use of login and logout hooks, "
                                                @"the setting of several system-wide preferences "
                                                @"and the creation of a new user.",
                                            @"Warning: Setting up user management "
                                                @"involves the installation of several "
                                                @"files, the use of login and logout hooks, "
                                                @"the setting of several system-wide preferences "
                                                @"and the creation of a new user." ),
                NSLocalizedString( @"OK", @"OK" ),
                NSLocalizedString( @"Don\'t Show Again", @"Don\'t Show Again" ), @"",
                prefPanel, self, @selector( userMgmtSheetDidEnd:returnCode:contextInfo: ),
                NULL, NULL,
                NSLocalizedString( @"User Management will effectively handle the creation "
                                    @"and accounting of temporary users in a lab environment. "
                                    @"You should not use the User Management scripts if you "
                                    @"are managing a personal desktop machine.", 
                                    @"User Management will effectively handle the creation "
                                    @"and accounting of temporary users in a lab environment. "
                                    @"You should not use the User Management scripts if you "
                                    @"are managing a personal desktop machine." ));
    }
}

- ( void )userMgmtSheetDidEnd: ( NSPanel * )sheet returnCode: ( int )rc
            contextInfo: ( void * )contextInfo
{
    switch ( rc ) {
    case NSAlertDefaultReturn:
        break;
        
    case NSAlertAlternateReturn:
        [[ NSUserDefaults standardUserDefaults ] setBool: YES
                            forKey: @"RAHideUserManagementWarning" ];
        [[ NSUserDefaults standardUserDefaults ] setBool: YES
                            forKey: @"RAHideAutomationWarning" ];
        break;
    }
    
    [ sheet orderOut: nil ];
    [ NSApp endSheet: sheet ];
    [ prefPanel makeKeyAndOrderFront: nil ];    
}

- ( void )showAutomationPreferences: ( id )sender
{
    NSUserDefaults	*defaults = [ NSUserDefaults standardUserDefaults ];
    
    [ autoEnableScheduleSwitch setState:
            [ defaults boolForKey: @"RAEnablePeriodic" ]];
    if ( [ autoEnableScheduleSwitch state ] == NSOnState ) {
        [ autoSchedulePopUpButton setEnabled: YES ];
    } else {
        [ autoSchedulePopUpButton setEnabled: NO ];
    }
    
    [ autoSchedulePopUpButton selectItemAtIndex:
            [ defaults integerForKey: @"RAPeriodicType" ]];
            
    [ autoEnableLoginHookSwitch setState:
            [ defaults boolForKey: @"RAEnableLoginHook" ]];
    [ autoEnableLogoutHookSwitch setState:
            [ defaults boolForKey: @"RAEnableLogoutHook" ]];
    [ autoContinueIfInterruptedSwitch setState:
            [ defaults boolForKey: @"RAContinueOnReboot" ]];
    
    if ( [[ prefPanel contentView ] isEqualTo: automationPrefsView ] ) {
        return;
    }
    
    if ( [ prefPanel toolbar ] != nil &&
                [[ prefPanel toolbar ] respondsToSelector:
                @selector( setSelectedItemIdentifier: ) ] ) {
        [[ prefPanel toolbar ] setSelectedItemIdentifier:
                            RAPrefToolbarAutomationIdentifier ];
    }
    
    [ contentBox setContentView: nil ];
    [ prefPanel resizeForContentView: automationPrefsView ];
    [ contentBox setContentView: automationPrefsView ];
    [ prefPanel setTitle: NSLocalizedString(
            @"Radmind Assistant: Automation Preferences",
            @"Radmind Assistant: Automation Preferences" ) ];
            
    if ( ! [ defaults boolForKey: @"RAHideAutomationWarning" ] ) {
        NSBeginAlertSheet( NSLocalizedString( @"Warning: Setting up Radmind automation "
                                                @"involves installing several components "
                                            @"and setting some system-wide preferences.",
                                        @"Warning: Setting up Radmind automation "
                                            @"involves installing several components "
                                            @"and setting some system-wide preferences." ),
                    NSLocalizedString( @"OK", @"OK" ), @"",
                    NSLocalizedString( @"Don\'t Show Again", @"Don\'t Show Again" ),
                    prefPanel, self, @selector( warningSheetDidEnd:returnCode:contextInfo:), NULL, NULL,
                    NSLocalizedString( @"Clicking \"Set Configuration\" will install the following items:\n\n"
                                            @"/etc/hooks\n/etc/hooks/LI00CheckUsername.hook\n"
                                            @"/etc/hooks/LORadmind.hook\n/etc/hooks/masterlogin.hook\n"
                                            @"/etc/hooks/masterlogout.hook\n/etc/hooks/radmind.hook\n"
                                            @"/etc/login.hook\n/etc/logout.hook\n/etc/radmind.defaults\n\n"
                                            @"Previous installations will be placed in /etc/hooks/unused.",
                                        @"Clicking \"Set Configuration\" will install the following items:\n\n"
                                            @"/etc/hooks\n/etc/hooks/LI00CheckUsername.hook\n"
                                            @"/etc/hooks/LORadmind.hook\n/etc/hooks/masterlogin.hook\n"
                                            @"/etc/hooks/masterlogout.hook\n/etc/hooks/radmind.hook\n"
                                            @"/etc/login.hook\n/etc/logout.hook\n/etc/radmind.defaults\n\n"
                                            @"Previous installations will be placed in /etc/hooks/unused." ));
    }
}

- ( void )warningSheetDidEnd: ( NSPanel * )sheet returnCode: ( int )rc
            contextInfo: ( void * )contextInfo
{
    [ sheet orderOut: nil ];
    [ NSApp endSheet: sheet ];
    
    switch ( rc ) {
    case NSAlertDefaultReturn:
    case NSAlertAlternateReturn:
        break;
        
    case NSAlertOtherReturn:
        [[ NSUserDefaults standardUserDefaults ] setBool: YES
                        forKey: @"RAHideAutomationWarning" ];
        [[ NSUserDefaults standardUserDefaults ] synchronize ];
        break;
    }
    
    [ prefPanel makeKeyAndOrderFront: nil ];
}

- ( IBAction )toggleUserAuth: ( id )sender
{
    if ( ! [ prefUserAuthName isEnabled ] ) {
        [ prefUserAuthName setEnabled: YES ];
	[[ NSUserDefaults standardUserDefaults ] setBool: YES forKey: @"userauth" ];
    } else {
        [ prefUserAuthName setEnabled: NO ];
	[[ NSUserDefaults standardUserDefaults ] setBool: NO forKey: @"userauth" ];
    }
}

- ( IBAction )toggleForceLockRemoval: ( id )sender
{
    BOOL		force = NO;
    
    if ( [ prefLapplyForce state ] == NSOnState ) {
	force = NO;
    } else {
	force = YES;
    }
    
    [[ NSUserDefaults standardUserDefaults ] setBool: force forKey: @"DontRemoveFileLocks" ];
}

- ( IBAction )toggleIgnoreUploadErrors: ( id )sender
{
    BOOL		errors = NO;
    
    if ( [ prefIgnoreLcreateErrors state ] == NSOnState ) {
        errors = NO;
    } else {
        errors = YES;
    }
    
    [[ NSUserDefaults standardUserDefaults ] setBool: errors forKey: @"DieOnErrorsDuringUpload" ];
}

- ( IBAction )toggleAlwaysPreApplySwitch: ( id )sender
{
    BOOL		always = YES;
    
    if ( [ prefAlwaysPreApply state ] == NSOnState ) {
	always = NO;
    }
    
    [[ NSUserDefaults standardUserDefaults ] setBool: always
	    forKey: @"RAAlwaysPreProcess" ];
}

- ( IBAction )toggleAlwaysPostApplySwitch: ( id )sender
{
    BOOL		always = YES;
    
    if ( [ prefAlwaysPostApply state ] == NSOnState ) {
	always = NO;
    }
    
    [[ NSUserDefaults standardUserDefaults ] setBool: always
	    forKey: @"RAAlwaysPostProcess" ];
}

- ( IBAction )manageScripts: ( id )sender
{
    if ( scriptManager == nil ) {
	scriptManager = [[ RAScriptManager alloc ] init ];
	[ NSBundle loadNibNamed: @"RAScriptManager"
				owner: scriptManager ];
    }
    
    [ scriptManager showScriptManagementPanelForWindow: prefPanel ];
}

- ( IBAction )setUsername: ( id )sender
{
    [[ NSUserDefaults standardUserDefaults ] setObject: [ prefUserAuthName stringValue ]
						forKey: @"username" ];
}

- ( IBAction )setTLSLevel: ( id )sender
{
    [[ NSUserDefaults standardUserDefaults ]
		setObject: [ NSNumber numberWithInt: [ prefTLSLevel indexOfSelectedItem ]]
		forKey: @"tlslevel" ];
}

- ( IBAction )manageCertificates: ( id )sender
{
    if ( certificateManager == nil ) {
	certificateManager = [[ RACertificateManager alloc ] init ];
	[ NSBundle loadNibNamed: @"RACertificateManager"
				owner: certificateManager ];
    }
    
    [ certificateManager showCertificateManagementPanelForWindow: prefPanel ];
}

- ( IBAction )setUseChecksums: ( id )sender
{
    BOOL		useCksums = ( BOOL )[ sender selectedRow ];
    
    [[ NSUserDefaults standardUserDefaults ] setBool: useCksums
						forKey: @"cksum" ];
}

- ( IBAction )editRadmindServerList: ( id )sender
{
    [ prefRadmindServerTable reloadData ];
    [ prefDeleteRadmindServerButton setEnabled: NO ];
    [ prefPanel makeKeyAndOrderFront: nil ];
    [ prefRadmindServerSettingCompressionLevel setEnabled: NO ];
    [ prefRadmindServerSettingPort setStringValue: @"" ];
    [ NSApp beginSheet: prefEditRadmindListSheet
	    modalForWindow: prefPanel
	    modalDelegate: self
	    didEndSelector: NULL
	    contextInfo: nil ];
}

- ( IBAction )setDefaultServer: ( id )sender
{
    NSDictionary		*serverInfo = nil;
    NSNumber			*port, *zlevel;
    int				i;
    
    [[ NSUserDefaults standardUserDefaults ] setObject: [ sender title ]
						forKey: @"radmindhost" ];
	
    /* remove old settings */
    [[ NSUserDefaults standardUserDefaults ]
	    removeObjectForKey: @"RadmindServerPort" ];
    [[ NSUserDefaults standardUserDefaults ]
	    removeObjectForKey: @"RadmindServerCompressionLevel" ];
	    
    for ( i = 0; i < [ radmindServerList count ]; i++ ) {
	if ( [[[ radmindServerList objectAtIndex: i ]
		    objectForKey: @"RadmindServerName" ]
		    isEqualToString: [ sender title ]] ) {
	    serverInfo = [ radmindServerList objectAtIndex: i ];
	    break;
	}
    }
    if ( serverInfo != nil ) {
	port = [ serverInfo objectForKey: @"RadmindServerPort" ];
	if ( port != nil ) {
	    [[ NSUserDefaults standardUserDefaults ]
		    setInteger: [ port intValue ] forKey: @"RadmindServerPort" ];
	}

	zlevel = [ serverInfo objectForKey: @"RadmindServerCompressionLevel" ];
	if ( zlevel == nil ) {
	    zlevel = [ NSNumber numberWithInt: 0 ];
	}
	if ( [ zlevel intValue ] > 1 && [ zlevel intValue ] < 10 ) {
	    [[ NSUserDefaults standardUserDefaults ]
		    setInteger: [ zlevel intValue ]
		    forKey: @"RadmindServerCompressionLevel" ];
	}
    }
    
    [ self prefsReload ];
}

- ( IBAction )setPort: ( id )sender
{
    NSMutableDictionary		*serverInfo = nil;
    NSString			*portString = [ sender stringValue ];
    int				port, row;
    
    row = [ prefRadmindServerTable selectedRow ];
    if ( row < 0 || row > [ radmindServerList count ] ) {
	return;
    }
    
    serverInfo = [ NSMutableDictionary dictionaryWithDictionary:
		    [ radmindServerList objectAtIndex: row ]];
    
    if ( portString == nil || [ portString length ] == 0 ) {
	[ serverInfo removeObjectForKey: @"RadmindServerPort" ];
    } else if (( port = [ portString intValue ] ) < 1 || port > 65535 ) {
	NSLog( @"%@: invalid port number", portString );
	return;
    } else {
	[ serverInfo setObject: [ NSNumber numberWithInt: port ]
		forKey: @"RadmindServerPort" ];
    }
    
    [ radmindServerList replaceObjectAtIndex: row
	    withObject: serverInfo ];
    [[ NSUserDefaults standardUserDefaults ]
	    setObject: radmindServerList forKey: @"RadmindServers" ];
}

- ( IBAction )toggleRadmindUpdateMonitor: ( id )sender
{
    BOOL			enabled = NO;
    
    if ( [ sender state ] == NSOnState ) {
        enabled = YES;
    }
    
    if ( enabled ) {
	[ self enableRadmindUpdateMonitor ];
    } else {
	[ self disableRadmindUpdateMonitor ];
    }
}

- ( void )enableRadmindUpdateMonitor
{
    NSString			*rumPath;
    NSString			*ctl = @"Unsupported";
    RAAuthExec			*rae;
    SInt32			sysVersion;
    int				rc;
    
    rumPath = [[ NSBundle mainBundle ]
		pathForResource: @"RadmindUpdateMonitor"
		ofType: @"app" ];
    if ( rumPath == nil ) {
	NSLog( @"Failed to find RadmindUpdateMonitor!" );
	return;
    }
    
    if ( [[ UMLoginItem sharedLoginItem ]
	    addLoginItem: rumPath hide: NO ] == NO ) {
	NSLog( @"failed to add %@ to AutoLaunchedApplicationDictionary",
		rumPath );
	return;
    }
    
    Gestalt( gestaltSystemVersion, &sysVersion );
     
    if ( sysVersion >= 0x00001040 ) {
	ctl = @"launchd";
    } else if ( sysVersion >= 0x00001030 && sysVersion < 0x00001040 ) {
	ctl = @"StartupItem";
    } else {
	NSLog( @"Unsupported version 0x%x", sysVersion );
	return;
    }
    
    rae = [[ RAAuthExec alloc ] init ];
    rc = [ rae executeTool: -1
		withArgs: [ NSArray arrayWithObjects: @"-A",
			    @"EnableRUM", @"--", ctl,
			    [[ NSBundle mainBundle ] resourcePath ], nil ]
		controller: nil ];
    [ rae release ];
    rae = nil;
    
    if ( rc != 0 ) {
        NSLog( @"Failed to enable RadmindUpdateMonitor" );
    }
    
    if ( [[ NSWorkspace sharedWorkspace ] openFile: rumPath ] == NO ) {
	NSLog( @"Failed to launch RadmindUpdateMonitor" );
    }
    
    [[ NSUserDefaults standardUserDefaults ]
	    setBool: YES forKey: @"RadmindUpdateMonitorEnabled" ];
}

- ( void )disableRadmindUpdateMonitor
{
    NSAppleScript		*as = nil;
    NSDictionary		*asErrorDictionary = nil;
    NSString			*rumPath;
    NSString			*ctl = @"Unsupported";
    NSString			*asSource = nil;
    RAAuthExec			*rae;
    SInt32			sysVersion;
    int				rc;
    
    rumPath = [[ NSBundle mainBundle ]
		pathForResource: @"RadmindUpdateMonitor"
		ofType: @"app" ];
    if ( rumPath == nil ) {
	NSLog( @"Failed to find RadmindUpdateMonitor!" );
	return;
    }
    
    if ( [[ UMLoginItem sharedLoginItem ]
	    removeLoginItem: rumPath ] == NO ) {
	NSLog( @"Failed to remove login item %@" );
	return;
    }
    
    Gestalt( gestaltSystemVersion, &sysVersion );
     
    if ( sysVersion >= 0x00001040 ) {
	ctl = @"launchd";
    } else if ( sysVersion >= 0x00001030 && sysVersion < 0x00001040 ) {
	ctl = @"StartupItem";
    } else {
	NSLog( @"Unsupported version 0x%x", sysVersion );
	return;
    }
     
    rae = [[ RAAuthExec alloc ] init ];
    rc = [ rae executeTool: -1
		withArgs: [ NSArray arrayWithObjects: @"-A", @"DisableRUM", @"--",
			    ctl, [[ NSBundle mainBundle ] resourcePath ], nil ]
		controller: nil ];
    [ rae release ];
    rae = nil;
    
    if ( rc != 0 ) {
	NSLog( @"Failed to disable RadmindUpdateMonitor" );
    }
    
    asSource = @"tell application \"RadmindUpdateMonitor\"\r"
		    @"quit\r"
		@"end tell\r";
    as = [[ NSAppleScript alloc ] initWithSource: asSource ];
    if ( [ as executeAndReturnError: &asErrorDictionary ] == nil ) {
	NSLog( @"NSAppleScript failed to quit RadmindUpdateMonitor: %@",
		[ asErrorDictionary objectForKey: NSAppleScriptErrorMessage ] );
    }
    
    [[ NSUserDefaults standardUserDefaults ]
	    setBool: YES forKey: @"RadmindUpdateMonitorEnabled" ];
}

- ( IBAction )setFsdiffPath: ( id )sender
{
    NSString			*path = [ sender title ];
    
    [[ NSUserDefaults standardUserDefaults ] setObject: path
	    forKey: @"runfrompath" ];
}

- ( IBAction )setTranscriptEditor: ( id )sender
{
    NSString			*editor = nil;
    
    if ( [[ sender title ] isEqualToString: @"Other..." ] ) {
	[ self chooseAlternateEditor: sender ];
	return;
    }
    
    editor = [ sender title ];
    [[ NSUserDefaults standardUserDefaults ] setObject: editor
	    forKey: @"transeditor" ];
}

- ( IBAction )setCaseSensitivity: ( id )sender
{
    BOOL			caseInsensitive;
    int				index;
    
    index = [ sender indexOfSelectedItem ];
    caseInsensitive = ( BOOL )( ! index );
    
    [[ NSUserDefaults standardUserDefaults ]
	    setBool: caseInsensitive forKey: @"RACaseInsensitive" ];
}

/* automation configuration actions */
- ( IBAction )toggleEnableScheduleSwitch: ( id )sender
{
    BOOL			enabled = NO;
    
    if ( [ sender state ] == NSOnState ) {
        enabled = YES;
    }

    [ autoSchedulePopUpButton setEnabled: enabled ];
}

- ( IBAction )visitiHookHomepage: ( id )sender
{
#define IHOOK_HOMEPAGE	@"http://www.ihook.org"

    [[ NSWorkspace sharedWorkspace ] openURL:
            [ NSURL URLWithString: IHOOK_HOMEPAGE ]];
}

- ( IBAction )setAutomationConfiguration: ( id )sender
{
    BOOL			enableSchedule = [ autoEnableScheduleSwitch state ];
    BOOL			enableLoginHook = [ autoEnableLoginHookSwitch state ];
    BOOL			enableLogoutHook = [ autoEnableLogoutHookSwitch state ];
    BOOL			continueOnReboot = [ autoContinueIfInterruptedSwitch state ];
    int				periodicType = 0;
    int				rc;
    NSString			*li = @"RemoveLoginHook", *lo = @"RemoveLogoutHook";
    RAAuthExec			*rae = nil;
    
    /* always make sure the scripts are in place */
    rae = [[ RAAuthExec alloc ] init ];
    rc = [ rae executeTool: -1
		withArgs: [ NSArray arrayWithObjects: @"-A",
			    @"InstallHooks", @"--",
			    [[ NSBundle mainBundle ] resourcePath ], nil ]
		controller: nil ];
		
    if ( rc != 0 ) {
        goto AUTOMATION_CONFIG_FAILED;
    }

    /* set or clear LoginHook */
    if ( enableLoginHook ) {
        li = @"SetLoginHook";
        [[ NSUserDefaults standardUserDefaults ] setBool: YES
                                            forKey: @"RAEnableLoginHook" ];
    } else {
        [[ NSUserDefaults standardUserDefaults ] setBool: NO
                                            forKey: @"RAEnableLoginHook" ];
    }
    
     rc = [ rae executeTool: -1
		withArgs: [ NSArray arrayWithObjects:
			    @"-A", li, @"--", @".", nil ]
		controller: nil ];
		
    if ( rc != 0 ) {
        goto AUTOMATION_CONFIG_FAILED;
    }
                
    /* set or clear LogoutHook */
    if ( enableLogoutHook ) {
        lo = @"SetLogoutHook";
        [[ NSUserDefaults standardUserDefaults ] setBool: YES
                                            forKey: @"RAEnableLogoutHook" ];
    } else {
        [[ NSUserDefaults standardUserDefaults ] setBool: NO
                                            forKey: @"RAEnableLogoutHook" ];
    }
    
    rc = [ rae executeTool: -1
		withArgs: [ NSArray arrayWithObjects:
			    @"-A", lo, @"--", @".", nil ]
		controller: nil ];
		
    if ( rc != 0 ) {
        goto AUTOMATION_CONFIG_FAILED;
    }
    
    /* set system-wide default to enable/disable radmind on reboot */
    [[ NSUserDefaults standardUserDefaults ] setBool: continueOnReboot
                                        forKey: @"RAContinueOnReboot" ];
    [[ NSUserDefaults standardUserDefaults ] synchronize ];
    
    rc = [ rae executeTool: -1
		withArgs: [ NSArray arrayWithObjects:
			    @"-A", @"SetGlobalDefaults",
			    @"--", @".", nil ]
		controller: nil ];
            
    if ( rc != 0 ) {
        goto AUTOMATION_CONFIG_FAILED;
    }
                
    /* install or remove scripts in periodic dir */
    if ( enableSchedule ) {
        periodicType = ( [ autoSchedulePopUpButton indexOfSelectedItem ] + 1 );
        [[ NSUserDefaults standardUserDefaults ] setBool: YES
                                            forKey: @"RAEnablePeriodic" ];
        [[ NSUserDefaults standardUserDefaults ]
                setInteger: [ autoSchedulePopUpButton indexOfSelectedItem ]
                forKey: @"RAPeriodicType" ];
    } else {
        [[ NSUserDefaults standardUserDefaults ] setBool: NO
                                            forKey: @"RAEnablePeriodic" ];
        [[ NSUserDefaults standardUserDefaults ]
                setInteger: 0
                forKey: @"RAPeriodicType" ];
    }
    
    rc = [ rae executeTool: -1
		withArgs: [ NSArray arrayWithObjects:
			    @"-A", @"InstallPeriodicScript", @"--",
			    [ NSString stringWithFormat: @"%d", periodicType ], 
			    [[ NSBundle mainBundle ] resourcePath ], nil ]
		controller: nil ];
		
    if ( rc != 0 ) {
        goto AUTOMATION_CONFIG_FAILED;
    }
    
    [ rae release ];
    
    NSLog( @"automation configuration set successfully." );
    
    return;
    
AUTOMATION_CONFIG_FAILED:
    switch ( rc ) {
    case -2:
        NSRunAlertPanel( NSLocalizedString( @"Authorization failed.",
                                            @"Authorization failed." ),
                            NSLocalizedString( @"You must provide a "
                                            @"valid administrator name and password "
                                            @"to complete this action.",
                                    @"You must provide a valid administrator "
                                    @"name and password to complete this action." ),
                            NSLocalizedString( @"OK", @"OK" ), @"", @"" );
        break;
        
    default:
        NSRunAlertPanel( NSLocalizedString( @"Failed to save automation configuration.",
                                            @"Failed to save automation configuration." ),
                            NSLocalizedString( @"Check the system log for details.",
                                            @"Check the system log for details." ),
                            NSLocalizedString( @"OK", @"OK" ), @"", @"" );
        break;
    }
    
    if ( rae != nil ) {
	[ rae release ];
    }
}

- ( IBAction )toggleUserManagement: ( id )sender
{
    BOOL        userManagement = ( BOOL )[ sender state ];
    
    [[ NSUserDefaults standardUserDefaults ] setBool: userManagement
                                                forKey: @"RAUserManagementEnabled" ];
    [[ NSUserDefaults standardUserDefaults ] synchronize ];
}

- ( IBAction )saveUserManagementSettings: ( id )sender
{
    NSString                    *templateUserName = nil;
    unsigned int                maxlife, maxusage;
    int				rc;
    NSArray			*array = nil;
    NSString                    *scriptpath = nil;
    RAAuthExec			*rae = nil;
    
    if ( ! [[ prefTemplateUserNameField stringValue ] length ] ) {
        [ prefTemplateUserNameField setStringValue: @"template" ];
    }
    
    if ( ! [[ prefMaxHomeLifeField stringValue ] length ] ||
                [ prefMaxHomeLifeField intValue ] == 0 ) {
        [ prefMaxHomeLifeField setStringValue: @"24" ];
    }
    
    if ( ! [[ prefMaxDiskUsageField stringValue ] length ] ||
                [ prefMaxDiskUsageField intValue ] < 25 ) {
        [ prefMaxHomeLifeField setStringValue: @"75" ];
    }
    
    templateUserName = [ prefTemplateUserNameField stringValue ];
    maxlife = [ prefMaxHomeLifeField intValue ];
    maxusage = [ prefMaxDiskUsageField intValue ];
    
    if ( [[ NSUserDefaults standardUserDefaults ]
                boolForKey: @"RAUserManagementEnabled" ] ) {
        [[ NSUserDefaults standardUserDefaults ] setObject: templateUserName
                                                    forKey: @"UMTemplateUserName" ];
        [[ NSUserDefaults standardUserDefaults ] setObject: [ NSString stringWithFormat:
                                                            @"/var/%@", templateUserName ]
                                                    forKey: @"UMTemplateHome" ];
        [[ NSUserDefaults standardUserDefaults ] setObject: [ NSString stringWithFormat:
                                                            @"%d hours ago", maxlife ]
                                                    forKey: @"UMMaxHomeLife" ];
        [[ NSUserDefaults standardUserDefaults ] setInteger: maxusage
                                                    forKey: @"UMMaxDiskUsage" ];
                                                    
        /* make sure scripts ignore current user */
        [[ NSUserDefaults standardUserDefaults ] setObject: NSUserName()
                                                    forKey: @"UMIgnoredUser" ];
                                                    
        [[ NSUserDefaults standardUserDefaults ] synchronize ];
        
        /* create user and install scripts */
        [ autoEnableLoginHookSwitch setState: NSOnState ];
        [ autoEnableLogoutHookSwitch setState: NSOnState];
        [ autoEnableScheduleSwitch setState: NSOnState ];
        [ self setAutomationConfiguration: nil ];
    
	rae = [[ RAAuthExec alloc ] init ];
        scriptpath = [[ NSBundle mainBundle ] pathForResource: @"useradd" ofType: @"sh" ];
        array = [ NSArray arrayWithObjects: @"-A",
		    @"ExecuteCommand", @"--", scriptpath, templateUserName, nil ];
    
	rc = [ rae executeTool: -1
		withArgs: array
		controller: nil ];
        if ( rc != 0 ) {
            goto USER_MANAGEMENT_SETTINGS_FAILED;
        }
        
        array = [ NSArray arrayWithObjects: @"-A",
		    @"InstallUserManagementScripts", @"--",
		    [[ NSBundle mainBundle ] resourcePath ], nil ];
        
	rc = [ rae executeTool: -1
		withArgs: array
		controller: nil ];
        if ( rc != 0 ) {
            goto USER_MANAGEMENT_SETTINGS_FAILED;
        }
        
        rc = NSRunAlertPanel( NSLocalizedString( @"Template user creation was successful.",
                                                    @"Template user creation was successful." ),
                            NSLocalizedString( @"You need to change the password for your template user. "
                                                @"Would you like to open the User Account "
                                                @"Preferences now?",
                                                @"You need to change the password for your template user. "
                                                @"Would you like to open the User Account "
                                                @"Preferences now?" ),
                            NSLocalizedString( @"Open Preferences", @"Open Preferences" ),
                            NSLocalizedString( @"Cancel", @"Cancel" ), @"" );
                            
        switch ( rc ) {
        case NSAlertDefaultReturn:
            [[ NSWorkspace sharedWorkspace ] openFile:
                    @"/System/Library/PreferencePanes/Accounts.prefPane" ];
            break;
        
        case NSAlertAlternateReturn:
            break;
        }
        
        rc = NSRunAlertPanel( NSLocalizedString( @"User management setup complete.",
                                                    @"User management setup complete." ),
                            NSLocalizedString( @"The template user's home folder is in space "
                                                @"managed by Radmind, /var/%@. To set up the template "
                                                @"home directory, login as the template user "
                                                @"and make your modifications. You can then "
                                                @"use Radmind to create a loadset containing your "
                                                @"template user and the user management scripts. "
                                                @"When lab users log in, they will get a copy of "
                                                @"the template home folder to use. When necessary, "
                                                @"e.g., after a certain period of time has passed, "
                                                @"the copy will be removed.",
                                            @"The template user's home folder is in space "
                                                @"managed by Radmind, /var/%@. To set up the template "
                                                @"home directory, login as the template user "
                                                @"and make your modifications. You can then "
                                                @"use Radmind to create a loadset containing your "
                                                @"template user and the user management scripts. "
                                                @"When lab users log in, they will get a copy of "
                                                @"the template home folder to use. When necessary, "
                                                @"e.g., after a certain period of time has passed, "
                                                @"the copy will be removed." ),
                            NSLocalizedString( @"OK", @"OK" ), @"", @"", templateUserName );
    } else {
        [[ NSUserDefaults standardUserDefaults ] removeObjectForKey: @"UMTemplateUserName" ];
        [[ NSUserDefaults standardUserDefaults ] removeObjectForKey: @"UMTemplateHome" ];
        [[ NSUserDefaults standardUserDefaults ] removeObjectForKey: @"UMMaxHomeLife" ];
        [[ NSUserDefaults standardUserDefaults ] removeObjectForKey: @"UMMaxDiskUsage" ];
        
        /* delete user and remove scripts */
        [ self setAutomationConfiguration: nil ];
        
        scriptpath = [[ NSBundle mainBundle ] pathForResource: @"userdelete" ofType: @"sh" ];
        array = [ NSArray arrayWithObjects: @"-x", scriptpath, templateUserName, nil ];
        
	rc = [ rae executeTool: -1
		withArgs: array
		controller: nil ];
	if ( rc != 0 ) {
            goto USER_MANAGEMENT_SETTINGS_FAILED;
        }
    }
    
    [ rae release ];
    
    return;
    
USER_MANAGEMENT_SETTINGS_FAILED:
    NSBeginAlertSheet( NSLocalizedString( @"Error", @"Error" ),
            NSLocalizedString( @"OK", @"OK" ), @"", @"", prefPanel,
            self, NULL, NULL, NULL,
            NSLocalizedString( @"Save of user management settings failed.",
                                @"Save of user management settings failed." ));
				
    if ( rae != nil ) {
	[ rae release ];
    }
}

- ( IBAction )setGlobalPreferences: ( id )sender
{
    int				rc;
    NSArray			*array = [ NSArray arrayWithObjects:
					    @"-A", @"SetGlobalDefaults",
					    @"--", @".", nil ];
    RAAuthExec			*rae = nil;

    rae = [[ RAAuthExec alloc ] init ];
    
    rc = [ rae executeTool: -1
		withArgs: array
		controller: nil ];
		
    [ rae release ];
    
    switch ( rc ) {
    case 0:
        break;
    
    case -2:
        NSRunAlertPanel( NSLocalizedString( @"Authorization failed.",
                                            @"Authorization failed." ),
                            NSLocalizedString( @"You must provide a "
                                            @"valid administrator name and password "
                                            @"to complete this action.",
                                    @"You must provide a valid administrator "
                                    @"name and password to complete this action." ),
                            NSLocalizedString( @"OK", @"OK" ), @"", @"" );
        return;
        
    default:
        NSLog( @"task exited with %d", rc );
        NSRunAlertPanel( NSLocalizedString( @"Failed to set preferences globally.",
                                            @"Failed to set preferences globally." ),
                            NSLocalizedString( @"Check the system log for details.",
                                            @"Check the system log for details." ),
                            NSLocalizedString( @"OK", @"OK" ), @"", @"" );
        return;
    }
    
    [ prefGlobalPreferencesDateField setStringValue:
                [ NSString stringWithFormat:
                    NSLocalizedString( @"Global preferences last set on %@.",
                                        @"Global preferences last set on %@." ),
                    [[ NSDate date ] description ]]];
                    
    [[ NSUserDefaults standardUserDefaults ] setObject: [[ NSDate date ] description ]
                                                forKey: @"RAGlobalPreferencesLastSetDate" ];

    [ prefPanel makeKeyAndOrderFront: nil ];
}

- ( IBAction )addRadmindServerToTable: ( id )sender
{
    [ radmindServerList addObject:
	    [ NSDictionary dictionaryWithObjectsAndKeys:
		@"newserver.radmind.edu", @"RadmindServerName",
		[ NSNumber numberWithInt: 0 ], @"RadmindServerCompressionLevel",
		nil ]];
    [ prefRadmindServerTable reloadData ];
    [ prefRadmindServerTable selectRow: ( [ radmindServerList count ] - 1 )
                                byExtendingSelection: NO ];
    [ prefRadmindServerTable editColumn: 0
                                row: ( [ radmindServerList count ] - 1 )
                                withEvent: nil
                                select: YES ];
}

- ( IBAction )deleteRadmindServerFromTable: ( id )sender
{
    int				row;
    
    if (( row = [ prefRadmindServerTable selectedRow ] ) < 0 ) {
	return;
    }

    [ radmindServerList removeObjectAtIndex: row ];
    
    [[ NSUserDefaults standardUserDefaults ]
	    setObject: radmindServerList
	    forKey: @"RadmindServers" ];
    [[ NSUserDefaults standardUserDefaults ] synchronize ];
	
    [ prefRadmindServerTable reloadData ];
    [ self prefsReload ];
}

- ( IBAction )setCompressionLevel: ( id )sender
{
    NSMutableDictionary	*serverInfo = nil;
    int			zlevel = [ sender indexOfSelectedItem ];
    int			row = [ prefRadmindServerTable selectedRow ];
    
    if ( row < 0 || row >= [ radmindServerList count ] ) {
	return;
    }
    
    serverInfo = [ NSMutableDictionary dictionaryWithDictionary:
		    [ radmindServerList objectAtIndex: row ]];
    
    if ( zlevel < 0 || zlevel > 9 ) {
	zlevel = 0;
    }
    
    [ serverInfo setObject: [ NSNumber numberWithInt: zlevel ]
	    forKey: @"RadmindServerCompressionLevel" ];
    [ radmindServerList replaceObjectAtIndex: row
	    withObject: serverInfo ];
	    
    [[ NSUserDefaults standardUserDefaults ]
	    setObject: radmindServerList
	    forKey: @"RadmindServers" ];
}

- ( IBAction )okEditServerTable: ( id )sender
{
    if ( radmindServerList != nil ) {
        [[ NSUserDefaults standardUserDefaults ] setObject: radmindServerList
                                                forKey: @"RadmindServers" ];
	if ( [ radmindServerList count ] == 1 ) {
	    NSDictionary	*serverInfo = nil;
	    
	    serverInfo = [ radmindServerList objectAtIndex: 0 ];
	    
	    [[ NSUserDefaults standardUserDefaults ]
		    setObject: [ serverInfo objectForKey: @"RadmindServerName" ]
		    forKey: @"radmindhost" ];
	    [[ NSUserDefaults standardUserDefaults ]
		    setObject: [ serverInfo objectForKey: @"RadmindServerPort" ]
		    forKey: @"RadmindServerPort" ];
	    [[ NSUserDefaults standardUserDefaults ]
		    setInteger: [[ serverInfo objectForKey:
				    @"RadmindServerCompressionLevel" ]
				    intValue ]
		    forKey: @"RadmindServerCompressionLevel" ];
	}
    }
    [ prefEditRadmindListSheet orderOut: nil ];
    [ NSApp endSheet: prefEditRadmindListSheet ];
    [ prefPanel makeKeyAndOrderFront: nil ];
    [ self prefsReload ];
}


- ( IBAction )chooseAlternateEditor: ( id )sender
{
    NSOpenPanel		*op = [ NSOpenPanel openPanel ];
    NSUserDefaults	*defaults = [ NSUserDefaults standardUserDefaults ];
    NSString		*lastDir = [ defaults objectForKey:
                                    @"NSDefaultOpenDirectory" ];
                                
    if ( lastDir == nil ) {
        lastDir = NSHomeDirectory();
    }
    
    [ op setTitle: @"Choose an Alternate Transcript Editor" ];
    [ op setPrompt: @"Choose" ];
    [ op setCanChooseDirectories: NO ];
    [ op setAllowsMultipleSelection: NO ];
    
    [ op beginSheetForDirectory: lastDir file: nil types: nil
            modalForWindow: prefPanel modalDelegate: self
            didEndSelector: @selector( altEditorOpenPanelDidEnd:returnCode:contextInfo: )
            contextInfo: nil ];
}

- ( void )altEditorOpenPanelDidEnd: ( NSOpenPanel * )sheet returnCode: ( int )rc
            contextInfo: ( void * )contextInfo
{
    NSDictionary	*dict;
    NSMutableArray	*altarray;
    NSString		*alted;
    NSUserDefaults	*defaults;

    switch ( rc ) {
    case NSOKButton:
        alted = [[ sheet filenames ] objectAtIndex: 0 ];
                            
        break;
        
    default:
    case NSCancelButton:
        return;
    }
    
    dict = [ NSDictionary dictionaryWithObjectsAndKeys:
                alted, @"path",
                [[ NSFileManager defaultManager ] displayNameAtPath: alted ],
                        @"name", nil ];
                
    defaults = [ NSUserDefaults standardUserDefaults ];
    altarray = [ defaults objectForKey: @"alteditors" ];
    if ( altarray == nil ) {
        altarray = [ NSMutableArray array ];
    }
    [ altarray addObject: dict ];
    [ defaults setObject: altarray forKey: @"alteditors" ];
    [ defaults setObject: [ dict objectForKey: @"name" ]
                forKey: @"transeditor" ];
    [ self prefsReload ];
}

- ( int )numberOfRowsInTableView: ( NSTableView * )tv
{
    return( [ radmindServerList count ] );
}

- ( id )tableView: ( NSTableView * )tv
        objectValueForTableColumn: ( NSTableColumn * )tc
        row: ( int )row
{
    return( [[ radmindServerList objectAtIndex: row ]
		objectForKey: @"RadmindServerName" ] );
}

- ( void )tableView: ( NSTableView * )tv setObjectValue: ( id )obj
        forTableColumn: ( NSTableColumn * )tc
        row: ( int )row
{
    NSMutableDictionary    *serverInfo = nil;
    
    if ( ! [ obj isKindOfClass: [ NSString class ]] ) {
	return;
    }
    if ( ! [ ( NSString * )obj length ] ) {
	return;
    }
    if ( row < 0 || row > [ radmindServerList count ] ) {
	return;
    }
    
    serverInfo = [ NSMutableDictionary dictionaryWithDictionary:
		    [ radmindServerList objectAtIndex: row ]];
    [ serverInfo setObject: obj forKey: @"RadmindServerName" ];
    
    [ radmindServerList replaceObjectAtIndex: row
	    withObject: serverInfo ];
    [ prefRadmindServerTable reloadData ];
}

- ( void )tableViewSelectionDidChange: ( NSNotification * )notification
{
    NSArray	    *servers = nil;
    NSDictionary    *serverInfo = nil;
    NSNumber	    *port;
    NSString	    *serverName;
    BOOL	    enabled = NO;
    int		    row, zlevel = 0;
    
    if (( row = [ prefRadmindServerTable selectedRow ] ) >= 0 ) {
	enabled = YES;
    }
    
    [ prefDeleteRadmindServerButton setEnabled: enabled ];
    
    servers = [[ NSUserDefaults standardUserDefaults ]
		    objectForKey: @"RadmindServers" ];
    
    [ prefRadmindServerSettingBox setTitle:
	    NSLocalizedString( @"Server Settings", @"Server Settings" ) ];
    [ prefRadmindServerSettingCompressionLevel setEnabled: NO ];
    [ prefRadmindServerSettingCompressionLevel selectItemAtIndex: 0 ];
    [ prefRadmindServerSettingPort setStringValue: @"" ];
    [ prefRadmindServerSettingPort setEnabled: NO ];
		    
    if ( row < 0 || row >= [ servers count ] ) {
	return;
    }
    
    [ prefRadmindServerSettingCompressionLevel setEnabled: YES ];
    [ prefRadmindServerSettingPort setEnabled: YES ];
    
    serverInfo = [ servers objectAtIndex: row ];
    serverName = [ serverInfo objectForKey: @"RadmindServerName" ];
    port = [ serverInfo objectForKey: @"RadmindServerPort" ];
    zlevel = [[ serverInfo objectForKey:
		@"RadmindServerCompressionLevel" ] intValue ];
		
    [ prefRadmindServerSettingBox setTitle: serverName ];
    if ( port != nil ) {
	[ prefRadmindServerSettingPort setObjectValue: port ];
    }

    [ prefRadmindServerSettingCompressionLevel selectItemAtIndex: zlevel ];
    [ prefRadmindServerSettingCompressionLevel setEnabled: YES ];
}

- ( void )dealloc
{
    [ radmindServerList release ];
    [ super dealloc ];
}

@end
