#import "RACertificateManager.h"
#import "RAAuthExec.h"
#import "AppDelegate.h"

#include <Security/Security.h>
#include <Carbon/Carbon.h>

#include <sys/types.h>
#include <sys/param.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>

#include <openssl/pem.h>
#include <openssl/x509.h>

@implementation RACertificateManager

static NSString		*tmpdir = nil;

- ( id )init
{
    _raCertificateArray = nil;
    
    return( [ super init ] );
}

- ( void )awakeFromNib
{
    NSBrowserCell	*cell = [[[ NSBrowserCell alloc ] init ] autorelease ];
    NSTableColumn	*column = [[ certificateTable tableColumns ]
				    objectAtIndex: 0 ];
    
    [ cell setImage: [ NSImage imageNamed: @"certificate.png" ]];
    [ cell setLeaf: YES ];
    [ cell setEditable: NO ];
    [ cell setFont: [ NSFont systemFontOfSize: 12.0 ]];
    [ column setDataCell: cell ];
    [ column setEditable: NO ];
    
    column = [[ certificateTable tableColumns ] objectAtIndex: 1 ];
    [ column setEditable: NO ];
    
    [ certificateTable setUsesAlternatingRowBackgroundColors: YES ];
    [ certificateTable setAllowsColumnSelection: NO ];
    [ certificateTable setTarget: self ];
    [ certificateTable setAction: nil ];
    [ certificateTable setDoubleAction: @selector( viewCertificates: ) ];
    [ certificateTable registerForDraggedTypes:
			[ NSArray arrayWithObject: NSFilenamesPboardType ]];
}

- ( void )setCertificateArray: ( NSArray * )array
{
    if ( _raCertificateArray != nil ) {
	[ _raCertificateArray release ];
	_raCertificateArray = nil;
    }
    
    _raCertificateArray = [ array retain ];
}

- ( NSArray * )certificateArray
{
    return( _raCertificateArray );
}

- ( SecCertificateRef )certificateFromPath: ( NSString * )path
{
    CSSM_DATA		data;
    OSStatus		status;
    SecCertificateRef	cert = NULL;
    X509		*x = NULL;
    FILE		*f;
    const char		*c_path = [ path UTF8String ];
    unsigned char	*buf, *p;
    int			len;
    
    if ( c_path == NULL ) {
	return( NULL );
    }
    
    if (( f = fopen( c_path, "r" )) == NULL ) {
	/* XXX - error alert here */
	NSLog( @"fopen %@: %s", path, strerror( errno ));
	return( NULL );
    }
    
    if (( x = PEM_read_X509( f, NULL, NULL, NULL )) == NULL ) {
	/* XXX - error alert */
	NSLog( @"PEM_read_X509 failed" );
	return( NULL );
    }
    
    if ( fclose( f ) != 0 ) {
	NSLog( @"fclose: %s", strerror( errno ));
	return( NULL );
    }
    
    len = i2d_X509( x, NULL );
    if (( buf = malloc( len )) == NULL ) {
	perror( "malloc" );
	exit( 2 );
    }
    
    /*
     * i2d_X509 increments the buffer to point after
     * the data we just got, so we have to use a temp
     * pointer to get the data we want (d2i_X509(3))
     */
    p = buf;
    
    i2d_X509( x, &p );
    
    ( &data )->Data = (uint8*)buf;
    ( &data )->Length = (uint32)len;
    
    status = SecCertificateCreateFromData( &data, CSSM_CERT_X_509v3,
		CSSM_CERT_ENCODING_UNKNOWN, &cert );
    if ( status != noErr ) {
	NSLog( @"SecCertificateCreateFromData failed (%ld)", status );
	return( NULL );
    }
    
    return( cert );
}

- ( void )installCertificate: ( NSString * )certificatePath
{
    NSArray			*args;
    RAAuthExec			*rae = nil;
    int				rc;
    
    rae = [[ RAAuthExec alloc ] init ];
    
    args = [ NSArray arrayWithObjects: @"-A",
		@"InstallCertificate", @"--",
		certificatePath, nil ];
		
    rc = [ rae executeTool: -1
		withArgs: args
		controller: nil ];
    [ rae release ];
    
    if ( rc != 0 ) {
	NSLog( @"failed to install %@", certificatePath );
	return;
    }
    
    [ self showCertificateManagementPanelForWindow: nil ];
}

- ( IBAction )installCertificates: ( id )sender
{
    NSOpenPanel			*op = [ NSOpenPanel openPanel ];
    NSString			*path = nil;
    int				rc;
    
    [ op setCanChooseDirectories: NO ];
    [ op setAllowsMultipleSelection: NO ];
    [ op setPrompt: NSLocalizedString( @"Install", @"Install" ) ];
    [ op setTitle: NSLocalizedString( @"Install Certificate",
					@"Install Certificate" ) ];
					
    rc = [ op runModalForDirectory: nil file: nil types: nil ];
    switch ( rc ) {
    case NSOKButton:
	break;
	
    default:
	return;
    }
    
    path = [ op filename ];
    
    [ self installCertificate: path ];
}

- ( IBAction )removeCertificates: ( id )sender
{
    NSAlert			*alert;
    NSArray			*args = nil, *items = nil;
    NSString			*certPath, *certTmpPath, *certName;
    RAAuthExec			*rae = nil;
    int				rc, row, optag;

    if (( row = [ certificateTable selectedRow ] ) < 0 ) {
	NSLog( @"No certificate selected" );
	return;
    }
    
    certName = [[[ self certificateArray ] objectAtIndex: row ]
		    objectForKey: @"RACertificateName" ];
    
    alert = [ NSAlert alertWithMessageText:
		NSLocalizedString( @"Delete selected certificate?",
				    @"Delete selected certificate?" )
		defaultButton: NSLocalizedString( @"Delete", @"Delete" )
		alternateButton: NSLocalizedString( @"Cancel", @"Cancel" )
		otherButton: @""
		informativeTextWithFormat:
		    NSLocalizedString( @"%@ will be moved to the trash.",
					@"%@ will be moved to the trash." ), certName ];
					
    rc = [ alert runModal ];
    switch ( rc ) {
    case NSOKButton:
    case NSAlertFirstButtonReturn:
	break;
	
    default:
	return;
    }
    
    certPath = [[[ self certificateArray ] objectAtIndex: row ]
		    objectForKey: @"RACertificatePath" ];
    if ( certPath == nil ) {
	NSLog( @"No certificate path found!" );
	return;
    }
    certTmpPath = [[[ self certificateArray ] objectAtIndex: row ]
		    objectForKey: @"RACertificateTmpPath" ];
    if ( certTmpPath == nil ) {
	NSLog( @"No certificate tmp path found!" );
	return;
    }
    
    rae = [[ RAAuthExec alloc ] init ];
    
    args = [ NSArray arrayWithObjects: @"-A",
		@"RemoveCertificate", @"--", certPath, nil ];
		
    rc = [ rae executeTool: -1 withArgs: args controller: nil ];
    [ rae release ];
    
    if ( rc != 0 ) {
	NSLog( @"failed to remove %@", certPath );
	return;
    }
    
    /*
     * move the temporary certificate copy to the trash
     * so the user can retrieve it later
     */
     items = [ NSArray arrayWithObject: certName ];
     if ( [[ NSWorkspace sharedWorkspace ]
	    performFileOperation: NSWorkspaceRecycleOperation
	    source: tmpdir destination: @"/"
	    files: items tag: &optag ] == NO ) {
	NSLog( @"Failed to move %@ to the trash.", certTmpPath );
	return;
    }
    
    [ self showCertificateManagementPanelForWindow: nil ];
}

- ( IBAction )viewCertificates: ( id )sender
{
    SecCertificateRef	    certificate = NULL;
    NSArray		    *certArray = nil;
    NSString		    *certificatePath = nil;
    int			    row = [ certificateTable selectedRow ];

    if ( row < 0 ) {
	return;
    }
    
    certificatePath = [[[ self certificateArray ] objectAtIndex: row ]
			objectForKey: @"RACertificateTmpPath" ];
    
    if (( certificate = [ self certificateFromPath: certificatePath ] ) == NULL ) {
	NSLog( @"failed to read certificate" );
	return;
    }
    certArray = [ NSArray arrayWithObject: ( id )certificate ];
    [ ( id )certificate release ];

    [[ SFCertificatePanel sharedCertificatePanel ]
	    runModalForCertificates: certArray showGroup: NO ];
}

- ( void )showCertificateManagementPanelForWindow: ( NSWindow * )window
{
    NSArray			*args = nil, *certs = nil;
    NSDictionary		*dict = nil;
    NSDirectoryEnumerator	*de = nil;
    NSString			*file;
    NSString			*dateString = nil;
    RAAuthExec			*rae = nil;
    int				rc;
    
    if (( tmpdir = [[ NSApp delegate ] sessionTemporaryDirectory ] ) == nil ) {
	NSLog( @"failed to get session temporary directory" );
	return;
    }
    
    rae = [[ RAAuthExec alloc ] init ];
    
    args = [ NSArray arrayWithObjects: @"-A",
		@"CopyCertificates", @"--", tmpdir, nil ];
		
    rc = [ rae executeTool: -1
		withArgs: args
		controller: nil ];
    [ rae release ];
    
    if ( rc != 0 ) {
	NSLog( @"failed to copy certificates" );
	return;
    }
    
    de = [[ NSFileManager defaultManager ] enumeratorAtPath: tmpdir ];
    [ de skipDescendents ];
    
    while (( file = [ de nextObject ] ) != nil ) {
	if ( ! [[ file pathExtension ] isEqualToString: @"pem" ] ) {
	    continue;
	}
	
	dict = [[ NSFileManager defaultManager ]
		    fileAttributesAtPath: [ tmpdir stringByAppendingPathComponent: file ]
		    traverseLink: YES ];
	if ( !dict ) {
	    NSLog( @"-fileAttributesAtPath:traverseLink: failed" );
	    continue;
	}
	
	/* format: 10 Feb 1978 12:47 pm */
	dateString = [[ dict objectForKey: NSFileModificationDate ]
		    descriptionWithCalendarFormat: @"%e %b %Y %I:%M %p"
		    timeZone: nil locale: nil ];
	
	dict = [ NSDictionary dictionaryWithObjectsAndKeys:
		    file, @"RACertificateName",
		    [ NSString stringWithFormat: @"/var/radmind/cert/%@", file ],
		    @"RACertificatePath",
		    [ tmpdir stringByAppendingPathComponent: file ],
		    @"RACertificateTmpPath",
		    dateString, @"RACertificateModificationTime", nil ];
		    
	if ( certs == nil ) {
	    certs = [ NSArray arrayWithObject: dict ];
	} else {
	    certs = [ certs arrayByAddingObject: dict ];
	}
    }
    [ self setCertificateArray: certs ];
    
    [ certificateTable reloadData ];
    
    if ( ! [ certificatePanel isVisible ] ) {
	[ NSApp beginSheet: certificatePanel
		modalForWindow: window
		modalDelegate: self
		didEndSelector: @selector( certificateManagementPanelDidEnd:returnCode:contextInfo: )
		contextInfo: window ];
    }
}

- ( IBAction )dismissCertificateManagementPanel: ( id )sender
{
    [ certificatePanel orderOut: self ];
    [ NSApp endSheet: certificatePanel ];
}

- ( void )certificateManagementPanelDidEnd: ( NSPanel * )panel
	    returnCode: ( int )rc contextInfo: ( void * )contextInfo
{
    NSWindow		*window = ( NSWindow * )contextInfo;
    
    if ( window == nil || ! [ window isKindOfClass: [ NSWindow class ]] ) {
	return;
    }
    
    [ window makeKeyAndOrderFront: self ];
}

/* tableview delegate and datasource methods */
- ( int )numberOfRowsInTableView: ( NSTableView * )aTableView
{
    return( [[ self certificateArray ] count ] );
}

- ( id )tableView: ( NSTableView * )aTableView
        objectValueForTableColumn: ( NSTableColumn * )aTableColumn
        row: ( int )rowIndex
{
    NSString		*string = @"";
				    
    if ( [[ aTableColumn identifier ] isEqualToString: @"CertificateName" ] ) {
	string = [[[ self certificateArray ]
		    objectAtIndex: rowIndex ]
		    objectForKey: @"RACertificateName" ];
    } else if ( [[ aTableColumn identifier ] isEqualToString: @"LastModified" ] ) {
	string = [[[ self certificateArray ]
		    objectAtIndex: rowIndex ]
		    objectForKey: @"RACertificateModificationTime" ];
    }
    
    if ( string == nil ) {
	string = @"n/a";
    }
    
    return( string );
}

- ( BOOL )tableView: ( NSTableView * )tableView
        writeRows: ( NSArray * )rows
        toPasteboard: ( NSPasteboard * )pboard
{
    NSArray			*array;
    NSArray			*certArray = [ self certificateArray ];
    int				i;
    
    array = [ NSArray array ];
    for ( i = 0; i < [ rows count ]; i++ ) {
	array = [ array arrayByAddingObject:
		    [[ certArray objectAtIndex:
			[[ rows objectAtIndex: i ] intValue ]]
			objectForKey: @"RACertificateTmpPath" ]];
    }
    
    [ pboard declareTypes: [ NSArray arrayWithObject: NSFilenamesPboardType ]
		owner: self ];
    [ pboard setPropertyList: array forType: NSFilenamesPboardType ];
    
    return( YES );
}

- ( NSDragOperation )tableView: ( NSTableView * )tableView
        validateDrop: ( id <NSDraggingInfo> )info
        proposedRow: ( int )row
        proposedDropOperation: ( NSTableViewDropOperation )operation
{
    if ( operation == NSTableViewDropAbove ) {
	[ tableView setDropRow: -1 dropOperation: NSTableViewDropOn ];
	return( NSDragOperationCopy );
    }

    return( NSDragOperationNone );
}

- ( BOOL )tableView: ( NSTableView * )tableView
        acceptDrop: ( id <NSDraggingInfo> )info
        row: ( int )row
        dropOperation: ( NSTableViewDropOperation )operation
{
    NSPasteboard		*pb;
    id				dragData;
    int				i;

    pb = [ info draggingPasteboard ];
        
    if ( [ pb availableTypeFromArray:
	    [ NSArray arrayWithObject: NSFilenamesPboardType ]] == nil ) {
	return( NO );
    }
    
    dragData = [ pb propertyListForType: [ pb availableTypeFromArray:
		    [ NSArray arrayWithObject: NSFilenamesPboardType ]]];

    if ( dragData == nil ) {
	return( NO );
    }
    
    for ( i = 0; i < [ dragData count ]; i++ ) {
	[ self installCertificate: [ dragData objectAtIndex: i ]];
    }
    
    return( YES );
}

@end
