/* -*-c-*- */

%{
/*
 * scan_dataset.l - scanner for the Qucs dataset
 *
 * Copyright (C) 2003, 2004, 2005, 2006, 2008 Stefan Jahn <stefan@lkcc.org>
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this package; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street - Fifth Floor,
 * Boston, MA 02110-1301, USA.  
 *
 * $Id$
 *
 */

#if HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#ifdef __MINGW32__
#include <io.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "logging.h"
#include "complex.h"
#include "object.h"
#include "vector.h"
#include "dataset.h"
#include "check_dataset.h"
#include "tokens_dataset.h"

%}

WS       [ \t\n\r]
IDENT1   [a-zA-Z_][a-zA-Z0-9_]*
IDENT2   [a-zA-Z_][a-zA-Z0-9_\[\],]*
IDENT3   [a-zA-Z0-9_][a-zA-Z0-9_]*
IDENT    {IDENT1}|{IDENT2}
PIDENT   {IDENT1}|{IDENT2}|{IDENT3}
SIMPLEID {IDENT}
POSTID   "."{PIDENT}
ID       {SIMPLEID}{POSTID}*
DIGIT    [0-9]
EXPONENT [Ee][+-]?{DIGIT}+
RINT     [+-]?{DIGIT}+
IINT     [+-]?[ij]{1}{DIGIT}+
RFLOAT1  [+-]?{DIGIT}+{EXPONENT}
RFLOAT2  [+-]?{DIGIT}*"."{DIGIT}+({EXPONENT})?
IFLOAT1  [+-]?[ij]{1}{DIGIT}+{EXPONENT}
IFLOAT2  [+-]?[ij]{1}{DIGIT}*"."{DIGIT}+({EXPONENT})?
CREAL    ({RFLOAT1}|{RFLOAT2}|{RINT})
CIMAG    ({IFLOAT1}|{IFLOAT2}|{IINT})
COMPLEX  {CREAL}{CIMAG}
SPACE    [ \t]
VERSION  "<Qucs Dataset "{DIGIT}+"."{DIGIT}+"."{DIGIT}+">"
DBEGIN   "dep"
IBEGIN   "indep"
DEND     "/dep"
IEND     "/indep"


%x COMMENT DESCRIPTION
%option yylineno noyywrap nounput prefix="dataset_"

%%

<INITIAL>{VERSION} {
    return Version;
  }

<DESCRIPTION>{DBEGIN} {
    return DepBegin;
  }

<DESCRIPTION>{IBEGIN} {
    return IndepBegin;
  }

<DESCRIPTION>{DEND} {
    return DepEnd;
  }

<DESCRIPTION>{IEND} {
    return IndepEnd;
  }

<INITIAL,DESCRIPTION>{ID} { /* identify identifier */
    dataset_lval.ident = strdup (dataset_text); 
    return Identifier; 
  }

<INITIAL>{CREAL} { /* identify real float */
    dataset_lval.f = strtod (dataset_text, NULL);
    return REAL;
  }

<INITIAL>{CIMAG} { /* identify imaginary float */
    if (dataset_text[0] == 'i' || dataset_text[0] == 'j')
      dataset_text[0] = '0';
    else
      dataset_text[1] = '0';
    dataset_lval.f = strtod (dataset_text, NULL);
    return IMAG;
  }

<INITIAL>{COMPLEX} { /* identify complete complex number */
    int i = 0;
    while (dataset_text[i] != 'i' && dataset_text[i] != 'j') i++;
    dataset_text[i] = dataset_text[i - 1];
    dataset_text[i - 1] = '\0';
    dataset_lval.c.r = strtod (dataset_text, NULL);
    dataset_lval.c.i = strtod (&dataset_text[i], NULL);
    return COMPLEX;
  }

<DESCRIPTION>{RINT} { /* identify integer */
    dataset_lval.n = strtol (dataset_text, NULL, 10);
    return Integer;
  }

<INITIAL>"<" { /* pass the '<' to the parser */ 
    BEGIN(DESCRIPTION);
    return '<';
  }
<DESCRIPTION>">" { /* pass the '>' to the parser */ 
    BEGIN(INITIAL);
    return '>';
  }
<INITIAL>\r?\n { /* detect end of line */ return Eol; }

<*>{SPACE}|\\\r?\n /* skip spaces and the trailing '\' */

<INITIAL>"#" { /* leave these characters */
    BEGIN(COMMENT);
  }
<INITIAL,DESCRIPTION>. { /* any other character in invalid */
    logprint (LOG_ERROR, 
	      "line %d: syntax error, unrecognized character: `%s'\n", 
	      dataset_lineno, dataset_text);
    return InvalidCharacter;
  }

<COMMENT>.     { /* skip any character in here */ }
<COMMENT>\r?\n { BEGIN(INITIAL); /* skipping ends here */ }

%%
