#!/usr/bin/env python
#
# Copyright 2015 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

"""Fingerprints the V8 snapshot blob files.

Constructs a SHA256 fingerprint of the V8 natives and snapshot blob files and
creates a .cc file which includes these fingerprint as variables.
"""

import hashlib
import optparse
import os
import sys

_HEADER = """// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

// This file was generated by fingerprint_v8_snapshot.py.

namespace gin {
"""

_FOOTER = """
}  // namespace gin
"""


def FingerprintFile(file_path):
  input_file = open(file_path, 'rb')
  sha256 = hashlib.sha256()
  while True:
    block = input_file.read(sha256.block_size)
    if not block:
      break
    sha256.update(block)
  return sha256.digest()


def WriteFingerprint(output_file, variable_name, fingerprint):
  output_file.write('\nextern const unsigned char %s[] = { ' % variable_name)
  for byte in fingerprint:
    output_file.write(str(ord(byte)) + ', ')
  output_file.write('};\n')


def WriteOutputFile(natives_fingerprint,
                    snapshot_fingerprint,
                    output_file_path):
  output_dir_path = os.path.dirname(output_file_path)
  if not os.path.exists(output_dir_path):
    os.makedirs(output_dir_path)
  output_file = open(output_file_path, 'w')

  output_file.write(_HEADER)
  WriteFingerprint(output_file, 'g_natives_fingerprint', natives_fingerprint)
  output_file.write('\n')
  WriteFingerprint(output_file, 'g_snapshot_fingerprint', snapshot_fingerprint)
  output_file.write(_FOOTER)


def main():
  parser = optparse.OptionParser()

  parser.add_option('--snapshot_file',
                    help='The input V8 snapshot blob file path.')
  parser.add_option('--natives_file',
                    help='The input V8 natives blob file path.')
  parser.add_option('--output_file',
                    help='The path for the output cc file which will be write.')

  options, _ = parser.parse_args()

  natives_fingerprint = FingerprintFile(options.natives_file)
  snapshot_fingerprint = FingerprintFile(options.snapshot_file)
  WriteOutputFile(
      natives_fingerprint, snapshot_fingerprint, options.output_file)

  return 0


if __name__ == '__main__':
  sys.exit(main())
