//! Qore namespace
namespace Qore {
//! Xml namespace
namespace Xml {
//! The XmlRpcClient class provides easy access to XML-RPC web services
/** This class inherits all public methods of the HTTPClient class. The inherited HTTPClient methods are not listed in this section, see the section on the HTTPClient class for more information on methods provided by the parent class. For a list of low-level XML-RPC functions, see @ref XMLRPC.

    The XmlRpcClient class understands the following protocols in addition to the protocols supported by the HTTPClient class:

    <b>XmlRpcClient Class Protocols</b>
    @htmlonly <style><!-- td.qore { background-color: #5b9409; color: white; } --></style> @endhtmlonly
    <table>
    <tr>
        <td class="qore"><b>Protocol</b></td>
        <td class="qore"><b>Default Port</b></td>
        <td class="qore"><b>SSL?</b></td>
        <td class="qore"><b>Description</b></td>
    </tr>
    <tr>
        <td>\c xmlrpc</td>
        <td>\c 80</td>
        <td>No</td>
        <td>Unencrypted XML-RPC protocol over HTTP</td>
    </tr>
    <tr>
        <td>\c xmlrpcs</td>
        <td>\c 443</td>
        <td>Yes</td>
        <td>XML-RPC protocol over HTTP with SSL/TLS encryption</td>
    </tr>
    </table>

    The XmlRpcClient supplies default values for HTTP headers as follows:

    <b>XmlRpcClient Default, but Overridable Headers</b>
    @htmlonly <style><!-- td.qore { background-color: #5b9409; color: white; } --></style> @endhtmlonly
    <table>
    <tr>
        <td class="qore"><b>Header</b></td>
        <td class="qore"><b>Default Value</b></td>
    </tr>
    <tr>
        <td>\c Accept</td>
        <td>\c text/xml</td>
    </tr>
    <tr>
        <td>\c Content-Type</td>
        <td>\c text/xml</td>
    </tr>
    <tr>
        <td>\c User-Agent</td>
        <td>\c Qore-XML-RPC-Client/1.1</td>
    </tr>
    <tr>
        <td>\c Connection</td>
        <td>\c Keep-Alive</td>
    </tr>
    </table>

    @note
    - if no \c default_path is given in the constructor(), then \c "RPC2" is used
    - This class is not available with the \c PO_NO_NETWORK parse option.
*/
class XmlRpcClient {

public:
//! Calls a remote method taking all arguments after the method name for the method arguments and returns the response as qore data structure
/** @param method The XML-RPC method name to call
    @param ... the arguments to the call

    @return a hash with one of the following keys:
    - \c params: will be present if the call completed normally
    - \c fault: will be present if the call is returning with error information; if this key is present then the value will be a hash with the following two keys:
      - \c faultCode: an integer fault code
      - \c faultString: a string error message

    @throw XMLRPC-SERIALIZATION-ERROR empty member name in hash or cannot serialize type to XML-RPC (ex: object)
    @throw PARSE-XMLRPC-RESPONSE-ERROR missing required element or other syntax error
    @throw PARSE-XMLRPC-ERROR syntax error parsing XML-RPC string
    @throw HTTP-CLIENT-TIMEOUT timeout on response from HTTP server
    @throw HTTP-CLIENT-RECEIVE-ERROR error communicating with HTTP server

    @note other exceptions may be thrown related to communication errors (ex: SSL errors, etc)

    @par Example:
    @code hash result = xrc.call("method.name", arg1, arg2); @endcode
*/
hash call(string method,  ...);

public:
//! Calls a remote method using a single value after the method name for the method arguments and returns the response as a Qore data structure
/** @param method The XML-RPC method name to call
    @param args An optional list of arguments (or single argument) for the method

    @return a hash with one of the following keys:
    - \c params: will be present if the call completed normally
    - \c fault: will be present if the call is returning with error information; if this key is present then the value will be a hash with the following two keys:
      - \c faultCode: an integer fault code
      - \c faultString: a string error message

    @throw XMLRPC-SERIALIZATION-ERROR empty member name in hash or cannot serialize type to XML-RPC (ex: object)
    @throw PARSE-XMLRPC-RESPONSE-ERROR missing required element or other syntax error
    @throw PARSE-XMLRPC-ERROR syntax error parsing XML-RPC string
    @throw HTTP-CLIENT-TIMEOUT timeout on response from HTTP server
    @throw HTTP-CLIENT-RECEIVE-ERROR error communicating with HTTP server

    @note other exceptions may be thrown related to communication errors (ex: SSL errors, etc)

    @par Example:
    @code hash result = xrc.callArgs("method.name", arg_list); @endcode
*/
hash callArgs(string method, any args);

public:
//! Calls a remote method using a single value after the method name for the method arguments and returns the response as qore data structure, accepts a reference to a hash as the first argument to give technical information about the call
/** @param info a reference to a hash that provides the following keys on output giving technical information about the HTTP call:
    - \c request: the literal outgoing request body sent
    - \c request-uri: the first line of the HTTP request
    - \c headers: a hash of HTTP headers in the outgoing request
    - \c response-uri: the first line of the HTTP response
    - \c response: the literal response body received from the server
    - \c response_headers: a hash of headers received in the response
    @param method The XML-RPC method name to call
    @param args An optional list of arguments (or single argument) for the method

    @return a hash with one of the following keys:
    - \c params: will be present if the call completed normally
    - \c fault: will be present if the call is returning with error information; if this key is present then the value will be a hash with the following two keys:
      - \c faultCode: an integer fault code
      - \c faultString: a string error message

    @throw XMLRPC-SERIALIZATION-ERROR empty member name in hash or cannot serialize type to XML-RPC (ex: object)
    @throw PARSE-XMLRPC-RESPONSE-ERROR missing required element or other syntax error
    @throw PARSE-XMLRPC-ERROR syntax error parsing XML-RPC string
    @throw HTTP-CLIENT-TIMEOUT timeout on response from HTTP server
    @throw HTTP-CLIENT-RECEIVE-ERROR error communicating with HTTP server

    @note other exceptions may be thrown related to communication errors (ex: SSL errors, etc)

    @par Example:
    @code
hash info;
hash result = xrc.callArgsWithInfo(\info, "method.name", arg_list);@endcode
*/
hash callArgsWithInfo(reference info, string method, any args);

public:
//! Calls a remote method taking all arguments after the method name for the method arguments and returns the response as qore data structure, accepts a reference to a hash as the first argument to give technical information about the call
/** @param info a reference to a hash that provides the following keys on output giving technical information about the HTTP call:
    - \c request: the literal outgoing request body sent
    - \c request-uri: the first line of the HTTP request
    - \c headers: a hash of HTTP headers in the outgoing request
    - \c response-uri: the first line of the HTTP response
    - \c response: the literal response body received from the server
    - \c response_headers: a hash of headers received in the response
    @param method The XML-RPC method name to call
    @param ... the arguments to the call

    @return a hash with one of the following keys:
    - \c params: will be present if the call completed normally
    - \c fault: will be present if the call is returning with error information; if this key is present then the value will be a hash with the following two keys:
      - \c faultCode: an integer fault code
      - \c faultString: a string error message

    @throw XMLRPC-SERIALIZATION-ERROR empty member name in hash or cannot serialize type to XML-RPC (ex: object)
    @throw PARSE-XMLRPC-RESPONSE-ERROR missing required element or other syntax error
    @throw PARSE-XMLRPC-ERROR syntax error parsing XML-RPC string
    @throw HTTP-CLIENT-TIMEOUT timeout on response from HTTP server
    @throw HTTP-CLIENT-RECEIVE-ERROR error communicating with HTTP server

    @note other exceptions may be thrown related to communication errors (ex: SSL errors, etc)

    @par Example:
    @code
hash info;
hash result = xrc.callWithInfo(\info, "method.name", arg1, arg2);@endcode
*/
hash callWithInfo(reference info, string method,  ...);

public:
//! Creates the XmlRpcClient object with no parameters
/** No connection is made because no connection parameters are set with this call; connection parameters must be set afterwards using the appropriate HTTPClient methods.
    @par Example:
    @code
XmlRpcClient xrc();
xrc.setURL("http://localhost:8080");@endcode
*/
 constructor();

public:
//! Creates the XmlRpcClient object based on the parameters passed
/** By default the object will immediately attempt to establish a connection to the server
    @param opts HTTPClient constructor options:
    - \c url: A string giving the URL to connect to; if no default path is given here or in the \c default_path option, then \c "RPC2" is used
    - \c default_port: The default port number to connect to if none is given in the URL.
    - \c protocols: A hash describing new protocols, the key is the protocol name and the value is either an integer giving the default port number or a hash with \c 'port' and \c 'ssl' keys giving the default port number and a boolean value to indicate that an SSL connection should be established.
    - \c http_version: Either \c '1.0' or \c '1.1' for the claimed HTTP protocol version compliancy in outgoing message headers.
    - \c default_path: The default path to use for new connections if a path is not otherwise specified in the connection URL; if this option is not given and no path appears in the URL, then \c "RPC2" is used instead
    - \c max_redirects: The maximum number of redirects before throwing an exception (the default is 5).
    - \c proxy: The proxy URL for connecting through a proxy.
    - \c timeout: The timeout value in milliseconds (also can be a relative date-time value for clarity, ex: \c 5m)
    - \c connect_timeout: The timeout value in milliseconds for establishing a new socket connection (also can be a relative date-time value for clarity, ex: \c 30s)
    @param no_connect pass a boolean True value argument to suppress the automatic connection and establish a connection on demand with the first request

    @see HTTPClient::constructor() and HTTPClient::connect() for information on possible exceptions

    @par Example:
    @code XmlRpcClient xrc(("url": "http://authuser:authpass@otherhost:8080/XMLRPC"));@endcode
*/
 constructor(hash opts, softbool no_connect = False);

public:
//! Throws an exception; copying XmlRpcClient objects is currently not supported
/** @throw XMLRPCCLIENT-COPY-ERROR copying XmlRpcClient objects is currently not supported
*/
 copy();

public:
//! clears the event queue for the XmlRpcClient object
/** @par Example:
    @code xrc.setEventQueue(); @endcode
 */
nothing setEventQueue();

public:
//! sets the event queue for the XmlRpcClient object
/** @param queue the Queue object to receive network events from the XmlRpcClient object
    @par Example:
    @code
Queue queue();
xrc.setEventQueue(queue); @endcode
 */
nothing setEventQueue(Queue queue);
};
}
}
