/* -*- mode: c++; indent-tabs-mode: nil -*- */
/** @file XmlRpcClient.qpp defines the XmlRpcClient class */
/*
    QC_XmlRpcClient.cpp

    Qore Programming Language

    Copyright (C) 2006 - 2022 Qore Technologies, s.r.o.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "qore-xml-module.h"

#include <qore/QoreHttpClientObject.h>

#include "QC_XmlRpcClient.h"
#include "ql_xml.h"

typedef ReferenceHolder<QoreHttpClientObject> safe_httpclient_t;

class HTTPInfoRefHelper {
protected:
    const ReferenceNode *ref;
    ExceptionSink *xsink;
    ReferenceHolder<QoreHashNode> info;

public:
    DLLLOCAL HTTPInfoRefHelper(const ReferenceNode *n_ref, QoreStringNode *msg, ExceptionSink *n_xsink) : ref(n_ref), xsink(n_xsink), info(new QoreHashNode(autoTypeInfo), xsink) {
        info->setKeyValue("request", msg, xsink);
    }
    DLLLOCAL ~HTTPInfoRefHelper() {
        // we have to create a temporary ExceptionSink if there is
        // an active exception, otherwise writing back the reference will fail
        ExceptionSink *txsink = *xsink ? new ExceptionSink : xsink;

        // write info hash to reference
        AutoVLock vl(txsink);
        QoreTypeSafeReferenceHelper rh(ref, vl, txsink);
        if (!rh)
            return;

        if (rh.assign(info.release()))
            return;

        if (txsink != xsink)
            xsink->assimilate(txsink);
    }
    DLLLOCAL QoreHashNode *operator*() {
        return *info;
    }
};

static void set_xrc_defaults(QoreHttpClientObject& client) {
    // set options for XML-RPC communication
    client.setDefaultPath("RPC2");
    client.setDefaultHeaderValue("Content-Type", "text/xml;charset=utf-8");
    client.setDefaultHeaderValue("Accept", "text/xml");
    client.setDefaultHeaderValue("User-Agent", "Qore-XML-RPC-Client/" PACKAGE_VERSION);

    client.addProtocol("xmlrpc", 80, false);
    client.addProtocol("xmlrpcs", 443, true);
}

static QoreHashNode *make_xmlrpc_call(QoreHttpClientObject* client, QoreStringNode *msg, QoreHashNode *info, ExceptionSink *xsink) {
    ReferenceHolder<QoreHashNode> hdr(xsink);

    if (client->getEncoding() != QCS_UTF8) {
        QoreString str(client->getEncoding()->getCode());
        str.tolwr();
        hdr = new QoreHashNode(autoTypeInfo);
        str.prepend("text/xml;charset=");
        hdr->setKeyValue("Content-Type", new QoreStringNode(str.getBuffer()), xsink);
    }

    ReferenceHolder<QoreHashNode> response(client->send("POST", 0, *hdr, *msg, true, info, xsink), xsink);
    if (!response)
        return 0;

    ValueHolder ans(response->takeKeyValue("body"), xsink);
    if (!ans)
        return 0;

    QoreValue ah = *ans;
    if (info) {
        info->setKeyValue("response", ans.release(), xsink);
        info->setKeyValue("response_headers", response.release(), xsink);
    }

    if (ah.getType() != NT_STRING) {
        xsink->raiseException("XMLRPCCLIENT-RESPONSE-ERROR", "undecoded binary response received from remote server");
        return 0;
    }
    //printd(5, "parse_xmlrpc_response() ah: %s\n", reinterpret_cast<const QoreStringNode *>(ah)->getBuffer());

    // parse XML-RPC response
    return parse_xmlrpc_response(xsink, ah.get<const QoreStringNode>(), QCS_DEFAULT);
}
/* Qore class Qore::Xml::XmlRpcClient */

qore_classid_t CID_XMLRPCCLIENT;
QoreClass* QC_XMLRPCCLIENT;

// hash XmlRpcClient::call(string method, ...){}
static QoreValue XmlRpcClient_call_VsVV(QoreObject* self, QoreHttpClientObject* client, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 248 "QC_XmlRpcClient.qpp"
    // create the outgoing message in XML-RPC call format
   QoreStringNodeHolder msg(make_xmlrpc_call(xsink, client->getEncoding(), 0, args));
   if (!msg)
      return 0;

   // send the message to the server and get the response as an XML string
   return make_xmlrpc_call(client, *msg, 0, xsink);
}

// hash XmlRpcClient::callArgs(string method, any args){}
static QoreValue XmlRpcClient_callArgs_VsVa(QoreObject* self, QoreHttpClientObject* client, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 217 "QC_XmlRpcClient.qpp"
    // create the outgoing message in XML-RPC call format
   QoreStringNodeHolder msg(make_xmlrpc_call_args(xsink, client->getEncoding(), 0, args));
   if (!msg)
      return 0;

   // send the message to the server and get the response as an XML string
   return make_xmlrpc_call(client, *msg, 0, xsink);
}

// hash XmlRpcClient::callArgsWithInfo(reference info, string method, any args){}
static QoreValue XmlRpcClient_callArgsWithInfo_VrVsVa(QoreObject* self, QoreHttpClientObject* client, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const ReferenceNode* info = HARD_QORE_VALUE_REF(args, 0);
# 288 "QC_XmlRpcClient.qpp"
    // get arguments
   ReferenceHolder<QoreListNode> xargs(args->copyListFrom(1), xsink);

   // create the outgoing message in XML-RPC call format
   QoreStringNode *msg = make_xmlrpc_call_args(xsink, client->getEncoding(), 0, *xargs);
   if (!msg)
      return 0;

   HTTPInfoRefHelper irh(info, msg, xsink);

   // send the message to the server and get the response as an XML string
   return make_xmlrpc_call(client, msg, *irh, xsink);
}

// hash XmlRpcClient::callWithInfo(reference info, string method, ...){}
static QoreValue XmlRpcClient_callWithInfo_VrVsVV(QoreObject* self, QoreHttpClientObject* client, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const ReferenceNode* info = HARD_QORE_VALUE_REF(args, 0);
# 333 "QC_XmlRpcClient.qpp"
    // get arguments
   ReferenceHolder<QoreListNode> xargs(args->copyListFrom(1), xsink);

   // create the outgoing message in XML-RPC call format
   QoreStringNode *msg = make_xmlrpc_call(xsink, client->getEncoding(), 0, *xargs);
   if (!msg)
      return 0;

   HTTPInfoRefHelper irh(info, msg, xsink);

   // send the message to the server and get the response as an XML string
   return make_xmlrpc_call(client, msg, *irh, xsink);
}

// XmlRpcClient::constructor() {}
static void XmlRpcClient_constructor(QoreObject* self, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 145 "QC_XmlRpcClient.qpp"
// get HTTPClient object
   safe_httpclient_t client((QoreHttpClientObject*)self->getReferencedPrivateData(CID_HTTPCLIENT, xsink), xsink);
   if (!client)
      return;

   set_xrc_defaults(*(*client));
}

// XmlRpcClient::constructor(hash opts, softbool no_connect = False) {}
static void XmlRpcClient_constructor_Vhvb(QoreObject* self, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreHashNode* opts = HARD_QORE_VALUE_HASH(args, 0);
    bool no_connect = HARD_QORE_VALUE_BOOL(args, 1);
# 173 "QC_XmlRpcClient.qpp"
// get HTTPClient object
   safe_httpclient_t client((QoreHttpClientObject*)self->getReferencedPrivateData(CID_HTTPCLIENT, xsink), xsink);
   if (!client)
      return;

   set_xrc_defaults(*(*client));

   if (client->setOptions(opts, xsink))
      return;

   // do not connect immediately if the second argument is True
   if (!no_connect)
      client->connect(xsink);
}

// XmlRpcClient::copy() {}
static void XmlRpcClient_copy(QoreObject* self, QoreObject* old, QoreHttpClientObject* client, ExceptionSink* xsink) {
# 192 "QC_XmlRpcClient.qpp"
xsink->raiseException("XMLRPCCLIENT-COPY-ERROR", "copying XmlRpcClient objects is not yet supported.");
}

// nothing XmlRpcClient::setEventQueue(){}
static QoreValue XmlRpcClient_setEventQueue(QoreObject* self, QoreHttpClientObject* client, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 352 "QC_XmlRpcClient.qpp"
    client->setEventQueue(0, xsink);
    return QoreValue();
}

// nothing XmlRpcClient::setEventQueue(Queue queue){}
static QoreValue XmlRpcClient_setEventQueue_C5Queue(QoreObject* self, QoreHttpClientObject* client, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    HARD_QORE_VALUE_OBJ_DATA(queue, Queue, args, 0, CID_QUEUE, "XmlRpcClient::setEventQueue()", "Queue", xsink);
    if (*xsink)
        return 0;
# 363 "QC_XmlRpcClient.qpp"
    // pass reference from QoreObject::getReferencedPrivateData() to function
   client->setEventQueue(queue, xsink);
    return QoreValue();
}

DLLLOCAL void preinitXmlRpcClientClass() {
    QC_XMLRPCCLIENT = new QoreClass("XmlRpcClient", "::Qore::Xml::XmlRpcClient", QDOM_DEFAULT);
    CID_XMLRPCCLIENT = QC_XMLRPCCLIENT->getID();
    QC_XMLRPCCLIENT->setSystem();
}

DLLLOCAL QoreClass* initXmlRpcClientClass(QoreNamespace& ns) {
    if (!QC_XMLRPCCLIENT)
        preinitXmlRpcClientClass();

    // set default builtin base class
    assert(QC_HTTPCLIENT);
    QC_XMLRPCCLIENT->addDefaultBuiltinBaseClass(QC_HTTPCLIENT);

    // hash XmlRpcClient::call(string method, ...){}
    QC_XMLRPCCLIENT->addMethod("call", (q_method_n_t)XmlRpcClient_call_VsVV, Public, QCF_USES_EXTRA_ARGS, QDOM_DEFAULT, hashTypeInfo, 1, stringTypeInfo, QORE_PARAM_NO_ARG, "method");

    // hash XmlRpcClient::callArgs(string method, any args){}
    QC_XMLRPCCLIENT->addMethod("callArgs", (q_method_n_t)XmlRpcClient_callArgs_VsVa, Public, QCF_NO_FLAGS, QDOM_DEFAULT, hashTypeInfo, 2, stringTypeInfo, QORE_PARAM_NO_ARG, "method", anyTypeInfo, QORE_PARAM_NO_ARG, "args");

    // hash XmlRpcClient::callArgsWithInfo(reference info, string method, any args){}
    QC_XMLRPCCLIENT->addMethod("callArgsWithInfo", (q_method_n_t)XmlRpcClient_callArgsWithInfo_VrVsVa, Public, QCF_NO_FLAGS, QDOM_DEFAULT, hashTypeInfo, 3, referenceTypeInfo, QORE_PARAM_NO_ARG, "info", stringTypeInfo, QORE_PARAM_NO_ARG, "method", anyTypeInfo, QORE_PARAM_NO_ARG, "args");

    // hash XmlRpcClient::callWithInfo(reference info, string method, ...){}
    QC_XMLRPCCLIENT->addMethod("callWithInfo", (q_method_n_t)XmlRpcClient_callWithInfo_VrVsVV, Public, QCF_USES_EXTRA_ARGS, QDOM_DEFAULT, hashTypeInfo, 2, referenceTypeInfo, QORE_PARAM_NO_ARG, "info", stringTypeInfo, QORE_PARAM_NO_ARG, "method");

    // XmlRpcClient::constructor() {}
    QC_XMLRPCCLIENT->addConstructor(XmlRpcClient_constructor, Public, QCF_NO_FLAGS, QDOM_DEFAULT);

    // XmlRpcClient::constructor(hash opts, softbool no_connect = False) {}
    QC_XMLRPCCLIENT->addConstructor(XmlRpcClient_constructor_Vhvb, Public, QCF_NO_FLAGS, QDOM_DEFAULT, 2, hashTypeInfo, QORE_PARAM_NO_ARG, "opts", softBoolTypeInfo, QoreSimpleValue().assign(false), "no_connect");

    // XmlRpcClient::copy() {}
    QC_XMLRPCCLIENT->setCopy((q_copy_t)XmlRpcClient_copy);

    // nothing XmlRpcClient::setEventQueue(){}
    QC_XMLRPCCLIENT->addMethod("setEventQueue", (q_method_n_t)XmlRpcClient_setEventQueue, Public, QCF_NO_FLAGS, QDOM_DEFAULT, nothingTypeInfo);

    // nothing XmlRpcClient::setEventQueue(Queue queue){}
    QC_XMLRPCCLIENT->addMethod("setEventQueue", (q_method_n_t)XmlRpcClient_setEventQueue_C5Queue, Public, QCF_NO_FLAGS, QDOM_DEFAULT, nothingTypeInfo, 1, QC_QUEUE->getTypeInfo(), QORE_PARAM_NO_ARG, "queue");

    return QC_XMLRPCCLIENT;
}
