// -*- mode: c++; indent-tabs-mode: nil -*-
//! @file WSDL.qm WSDL: Web Services Description Language: http://www.w3.org/TR/wsdl, SOAP 1.1: https://www.w3.org/TR/2000/NOTE-SOAP-20000508/, SOAP 1.2: https://www.w3.org/TR/soap12-part1/

/*  WSDL.qm Copyright (C) 2012 - 2022 Qore Technologies, s.r.o.

    Permission is hereby granted, free of charge, to any person obtaining a
    copy of this software and associated documentation files (the "Software"),
    to deal in the Software without restriction, including without limitation
    the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and/or sell copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.
*/

// make sure we have the required qore version

// requires XML functionality

// need mime definitions


// do not use $ for vars

// allow the use of the := weak reference assignment operator



/*
    WSDL classes
    provides some minimal WSDL and XSD support for SOAP messaging used by the SoapClient class and the SoapHandler

    not complete, needs namespace verification, improved XSD support, element groups, etc
*/

/** @mainpage WSDL Module

    @tableofcontents

    @section wsdlintro Introduction to the WSDL Module

    The %WSDL module provides functionality for <a href="http://en.wikipedia.org/wiki/SOAP">SOAP</a> 1.1 and 1.2.

    The main purpose of this module is to provide the infrastructure for developing SOAP client and server services.

    The main components of this module are as follows:
    - @ref WSDL::WebService "WebService": represents a WSDL file; an easy way to get an object of this type from a URL
      is by calling WSDL::WSDLLib::getWSDL()
    - @ref WSDL::WSDLLib "WSDLLib": a class of static functions providing helper functions for the module
    - @ref WSDL::WSOperation "WSOperation": a class representing a single operation from a SOAP web service
    - @ref WSDL::WSMessageHelper "WSMessageHelper": a class to create sample messages used by SOAP operations

    See the following for modules using the %WSDL module:
    - <a href="../../SoapClient/html/index.html">SoapClient user module</a>
    - <a href="../../SoapHandler/html/index.html">SoapHandler user module</a>

    @section wsdlrelnotes WSDL Module Release Notes

    @subsection wsdl_0_5_4 WSDL v0.5.4
    - fixed a bug returning data provider type info for simpleTypes with union definitions
      (<a href="https://github.com/qorelanguage/qore/issues/4480">issue 4480</a>)

    @subsection wsdl_0_5_3 WSDL v0.5.3
    - fixed a bug handling \c complexType elements with a \c sequence and and \a any element
      (<a href="https://github.com/qorelanguage/qore/issues/4452">issue 4452</a>)
    - fixed a confusing bug in the error message for
      @ref WSDL::WSDLLib::getWebServiceFromUrl() "WSDLLib::getWebServiceFromUrl()" when the file could not be read
      (<a href="https://github.com/qorelanguage/qore/issues/4451">issue 4451</a>)
    - fixed bugs loading URLs with multiple imports of the same external XSD schema
      (<a href="https://github.com/qorelanguage/qore/issues/4449">issue 4449</a>)

    @subsection wsdl_0_5_2 WSDL v0.5.2
    - fixed a bug where the web service's name was not reported correctly by the
      @ref WSDL::WebService "WebService" object
      (<a href="https://github.com/qorelanguage/qore/issues/4437">issue 4437</a>)

    @subsection wsdl_0_5_1 WSDL v0.5.1
    - added support for the FileLocationHandler module

    @subsection wsdl_0_5 WSDL v0.5
    - added support for the data provider API

    @subsection wsdl_0_4 WSDL v0.4
    - added support for serializing and deserilizing WebService objects
      (<a href="https://github.com/qorelanguage/qore/issues/2793">issue 2793</a> and
      <a href="https://github.com/qorelanguage/qore/issues/1598">issue 1598</a>)

    @subsection wsdl_0_3_8 WSDL v0.3.8
    - fixed a bug handling complexTypes as part types
      (<a href="https://github.com/qorelanguage/qore/issues/2900">issue 2900</a>)
    - fixed bugs handling array types
      (<a href="https://github.com/qorelanguage/qore/issues/2899">issue 2899</a>)
    - added support for \c token and \c normalizedString types
      (<a href="https://github.com/qorelanguage/qore/issues/2859">issue 2859</a>)
    - fixed @ref WSDLLib::getWSDL() when passed a WSDL string with no XML preamble
      (<a href="https://github.com/qorelanguage/qore/issues/2857">issue 2857</a>)
    - fixed handling of XSD attribute names with non-word characters
      (<a href="https://github.com/qorelanguage/qore/issues/2856">issue 2856</a>)
    - fixed handling of \c simpleType definitions with \c union members
      (<a href="https://github.com/qorelanguage/qore/issues/2855">issue 2855</a>)

    @subsection wsdl_0_3_7 WSDL v0.3.7
    - implemented supoprt for handling SOAP faults in response messages with SOAP bindings
      (<a href="https://github.com/qorelanguage/qore/issues/2804">issue 2804</a>)
    - fixed an error in namespace resolution with nested namespaces
      (<a href="https://github.com/qorelanguage/qore/issues/2786">issue 2786</a>)
    - fixed a type error in example message generation that resulted in failed example message generation
      (<a href="https://github.com/qorelanguage/qore/issues/2782">issue 2782</a>)
    - fixed a bug in example message generation that resulted in invalid messages
      (<a href="https://github.com/qorelanguage/qore/issues/2752">issue 2752</a>)
    - implemented the @ref WSDL::wsdl_set_global_compat_empty_string_is_nothing()
    "wsdl_set_global_compat_empty_string_is_nothing"
      function and the \c "compat_empty_string_is_nothing" option for the
      @ref WSDL::WebService "WebService" class for backwards compatibility with older versions of the WSDL
      module (<a href="https://github.com/qorelanguage/qore/issues/2754">issue 2754</a>)
    - implemented the @ref WSDL::wsdl_set_global_compat_empty_string_is_nothing()
    "wsdl_set_global_compat_allow_any_header"
      function and the \c "compat_allow_any_header" option for the @ref WSDL::WebService "WebService" class
      for backwards compatibility with older versions of the WSDL module
      (<a href="https://github.com/qorelanguage/qore/issues/2765">issue 2765</a>)

    @subsection wsdl_0_3_6 WSDL v0.3.6
    - reimplmented operation to support multi binding, operation can be assigned to more bindings
    - support for HTTP binding and content-types, "^content-type^" attribute can identify content type to be used
    - extended SOAP binding serialization and deserialization with support to both body and header

    @subsection wsdl_0_3_5_1 WSDL v0.3.5.1
    - supress emitting a SOAPAction header in requests if the binding gives an empty string (<a href="https://github.com/qorelanguage/qore/issues/1226">issue 1226</a>)
    - updated @ref WSDL::WSOperation::serializeRequest() to allow the SOAPAction header to be overridden in each request (<a href="https://github.com/qorelanguage/qore/issues/1226">issue 1226</a>)
    - respect XML generation flags in request generation
    - fixed parsing empty base64Binary and hexBinary elements (<a href="https://github.com/qorelanguage/qore/issues/1227">issue 1227</a>)

    @subsection wsdl_0_3_5 WSDL v0.3.5
    - fixed many message serialization and deserialization issues
    - added @ref WSDL::WebService::getOperation()
    - allow for environment variable substitution in WSDLLib::getWSDL() when retrieving files
    - fixed charset=... header value
    - added @ref WSDL::WSMessageHelper
    - added @ref WSDL::WebService::getReport()

    @subsection wsdl_0_3_4 WSDL v0.3.4
    - updated to a user module

    @subsection wsdl_0_3_3 WSDL v0.3.3
    - added initial support for the anyAttribute element of complexType
    - added initial support for SOAP header processing
    - added initial support for multiple portType and bindings in a WSDL

    @subsection wsdl_0_3_2 WSDL v0.3.2
    - fixed bugs de/serializing negative values for "int" and "short"

    @subsection wsdl_0_3_1 WSDL v0.3.1
    - improved XSD imports and namespace handling

    @subsection wsdl_0_3_0 WSDL v0.3.0
    - implemented WSDLLib class of helper functions
    - implemented support for xsd import statements in WSDLs

    @subsection wsdl_0_2_0 WSDL v0.2.0
    - use parseXMLAsData() instead of parseXML()
    - implemented initial simpleType support
    - fixed xsd:date serialization and deserialization
*/

//! main WSDL namespace
namespace WSDL {
    //! this WSDL implementation version
    const version     = "0.3.6";

    //! SOAP 1.1 envelope URI
    const SOAP_11_ENV  = "http://schemas.xmlsoap.org/soap/envelope/";
    //! SOAP 1.2 envelope URI
    const SOAP_12_ENV  = "http://www.w3.org/2003/05/soap-envelope";

    //! SOAP 1.1 namespace URI
    const SOAP_11_NS   = "http://schemas.xmlsoap.org/wsdl/soap/";

    //! SOAP 1.2 namespace URI
    const SOAP_12_NS   = "http://schemas.xmlsoap.org/wsdl/soap12/";

    //! XSD namespace URI
    const XSD_NS       = "http://www.w3.org/2001/XMLSchema";
    //! XSI namespace URI
    const XSI_NS       = "http://www.w3.org/2001/XMLSchema-instance";

    //! HTTP namespace URI
    const HTTP_NS      = "http://schemas.xmlsoap.org/wsdl/http/";
    //! MIME namespace URI
    const MIME_NS      = "http://schemas.xmlsoap.org/wsdl/mime/";

    //! soap 1.1 envelope namespaces
    const ENVELOPE_11_NS = ...;


    //! soap 1.2 envelope namespaces
    const ENVELOPE_12_NS = ...;


    //! SOAP "use" to "encoded" mappings
    const SoapUseMap = ...;


    //! SOAP "style" to "document" mappings
    const SoapStyleMap = ...;


    //! soap encoding URI
    const SOAP_ENCODING = "http://schemas.xmlsoap.org/soap/encoding/";

    //! mapping from Qore types to xsd types for xsd type "anyType"
    const any_type_map = ...;


    // error codes
    const SOAP_SERIALIZATION_ERROR = "SOAP-SERIALIZATION-ERROR";
    const SOAP_DESERIALIZATION_ERROR = "SOAP-DESERIALIZATION-ERROR";
    const WSDL_ERROR = "WSDL-ERROR";

    //! SOAP HTTP transport URI
    const SOAP_TRANSPORT_HTTP = "http://schemas.xmlsoap.org/soap/http";

    //! known/supported transports
    const SOAP_TRANSPORT = ...;


    //! range of "short" values (16 bits)
    const RANGE_SHORT = (-32768, 32767);

    //! range of "int" values (32 bits)
    const RANGE_INT = (-2147483648, 2147483647);

    // private global variables
    extern bool global_compat_empty_string_is_nothing;
    extern bool global_compat_allow_any_header;

    //! sets the \c global_compat_empty_string_is_nothing variable to the given value to force the WSDL module to return a blank value for a required string field when the associated XML element is present as no value instead of an empty string for backwards compatibility with xml module 1.3
     wsdl_set_global_compat_empty_string_is_nothing(softbool val);


    //! sets the \c global_compat_allow_any_header variable to the given value to force the WSDL module to allow any SOAP header to be sent in SOAP messages for backwards compatibility with xml module 1.3
     wsdl_set_global_compat_allow_any_header(softbool val);

};

//! contains helper methods for retrieving WSDLs from a URL
class WSDL::WSDLLib {

public:
    //! Mime types recognized as SOAP messages
    const SoapMimeTypes = (MimeTypeSoapXml, MimeTypeXml, MimeTypeXmlApp);

      static string getSoapMimeType12(bool soap12);

    //! retrieves a local file and returns the file's contents as a string
/** called by WSDLLib::getFileFromURL() in case the scheme is "file"
    */
      static data getFile(string fn, bool as_string = True);

    //! retrieves a file from a URL with HTTP and returns the file's contents as a string
/** called by WSDLLib::getFileFromURL() in case the scheme is "http"
    */
      static string getHTTP(string url, __7_ string path, __7_ HTTPClient hc, __7_ hash<auto> headers, bool as_string = True);

    //! retrieves a file from a URL with the FTP protocol and returns the file's contents as binary or string data
/** called by WSDLLib::getFileFromURL() in case the scheme is "ftp"
    */
      static data getFTP(string url, string path, bool as_string = True);

    //! retrieves a file from a URL
/** in case the URL is actually a file path, environment variable substitution
        is performed on the path string

        @note uses the \c FileLocationHandler module to retrieve file data for all schemes except \c "file://" and
        \c "http(s)://", which uses @ref WSDL::WSDLLib::getFileFromUrl() "WSDLLib::getFileFromUrl()" and
        @ref WSDL::WSDLLib::getHTTP() "WSDLLib::getHTTP()" respectively for backwards compatibility
     */
  static data getFileFromURL(string url, string def_protocol = 'file', __7_ HTTPClient http_client, *hash<auto> http_headers, bool as_string = True, *string def_path, *reference new_def_path);

    //! retrieves a file from an already-parsed URL
/** in case the URL is actually a file path, environment variable substitution
        is performed on the path string

        @note uses the \c FileLocationHandler module to retrieve file data for all schemes except \c "file://" and
        \c "http(s)://", which uses @ref WSDL::WSDLLib::getFileFromUrl() "WSDLLib::getFileFromUrl()" and
        @ref WSDL::WSDLLib::getHTTP() "WSDLLib::getHTTP()" respectively for backwards compatibility
     */
  static data getFileFromURL(string url, hash<auto> u, string def_protocol = 'file', __7_ HTTPClient http_client, *hash<auto> http_headers, bool as_string = False, *string def_path, *reference new_def_path);

    //! returns the argument
      static WebService getWSDL(WebService wsdl);

    //! returns a WSDL string from a file name, optional HTTPClient object and optional header hash
/** in case the \a wsdl argument is actually a file path, environment variable substitution
        is performed on the path string
     */
      static string getWSDL(string wsdl, __7_ HTTPClient http_client, __7_ hash<auto> http_headers, __7_ reference new_def_path);

    //! returns a WebService object from a URL and other optional arguments
/** @param the URL to use to retrieve the file, the \c "file://", \c "http://", \c "https://", and \c "ftp://" schemes are supported, if no scheme is present, then \c "file://" is assumed
        @param http_client an optional HTTPClient object to use to retrieve external schemas, if no HTTPClient object is passed and one is needed, an HTTPClient object will be created implicitly
        @param http_headers any HTTP headers to send with requests when retrieving external XSD schemas
    */
      static WebService getWebServiceFromUrl(string url, __7_ HTTPClient http_client, __7_ hash<auto> http_headers);

    //! takes a hash representation of a SOAP message and handles multipart messages, checks the content-type, and handles hrefs in the message
      static hash<auto> parseMultiPartSOAPMessage(hash<auto> msg);

    //! returns True is the message has a SOAP mime type
      static bool isSOAPMessage(hash<auto> msg);

/**!
    takes a hash representation returned by parseMultiPartSOAPMessage and parses it to a Qore data structure, checks the content-type, and handles hrefs in the message.
    Operation is not yet known if SoapAction header is not presented
    */
      static __7_ hash<auto> parseSOAPMessage(hash<auto> msg);

protected:
      static processHref(reference xmldata, string hr, hash<auto> parts);
public:


protected:
      static substHref(reference xmldata, hash<auto> parts);
public:

};

// private, nmon-exported class
class WsdlLibPriv {

public:
      static bool isContentType(string ct, list<auto> MimeTypes);

      static checkContentType(string ct, list<auto> MimeTypes);
};

//! base class with helper methods for XSD data processing
class WSDL::XsdBase : public Qore::Serializable {

public:
      static removeNS(reference v);

      static removeNS2(reference v);
};

//! base class for XSD data classes
class WSDL::XsdData : public WSDL::XsdBase {

public:
    auto getValue(__7_ hash<auto> mrh, auto val);

};

//! base class for XSD classes with a "name" attribute
class WSDL::XsdNamedData : public WSDL::XsdData {

public:
        // name of object
        string name;
        //! input namespace prefix (if any given)
        __7_ string ns;
        //! descriptive name flag
        descriptive_name;

    constructor(string n_name, string n_ns);


    constructor(reference e, __7_ string desc_name);


    string getName();


    __7_ string getInputNamespacePrefix();


    bool hasRealName();

};

class WSDL::XsdAbstractType : public WSDL::XsdNamedData {

public:
        //! reference to namespaces
        Namespaces nsc;

        //! namespace output prefix
        string ons;

    constructor(reference e, Namespaces nsc, __7_ string desc_name) ;


    constructor(string name, string ns, Namespaces nsc) ;


protected:
     resolveNamespace();
public:


    checkExtends(XsdAbstractType t, string ename);


    string getNameWithNS();


    bool isNillable();


    bool isRequired();


    bool requiresValue();


    string getOutputNamespacePrefix();


    __7_ list<auto> getAllowedValues();


    abstract auto serialize(Namespaces nsc, auto val, __7_ softbool omitType);
    abstract auto deserialize(string en, hash<string, XsdAbstractType> tmap, __7_ hash<auto> mrh, auto val);
    abstract AbstractDataProviderType getDataProviderType();
};

//! class for XSD base types
class WSDL::XsdBaseType : public WSDL::XsdAbstractType {

public:

    constructor(string t, Namespaces nsc, string ns = 'xsd') ;


    auto serialize(Namespaces nsc, auto val, __7_ softbool omitType);


    auto deserialize(string en, hash<string, XsdAbstractType> tmap, __7_ hash<auto> mrh, auto val);


    AbstractDataProviderType getDataProviderType();


    Type getDataProviderBaseType();

};

//! class for XSD array types; currently only supports single dimensional arrays
class WSDL::XsdArrayType : public WSDL::XsdAbstractType {

public:
protected:
        bool finalized;

public:

        __7_ hash<auto> typeinfo;
        XsdAbstractType elementType;

    constructor(string t, string ns, string arrayType, Namespaces nsc, XsdLateResolverHelper unresolved) ;


    AbstractDataProviderType getDataProviderType();


    finalize(hash<string, XsdAbstractType> tmap, Namespaces nsc);


    auto serialize(Namespaces nsc, auto val, __7_ softbool omitType);


    auto deserialize(string en, hash<string, XsdAbstractType> tmap, __7_ hash<auto> mrh, auto val);

};

//! XSD typed data class
class WSDL::XsdTypedData : public WSDL::XsdNamedData {

public:
        __7_ hash<auto> typeinfo;
        XsdAbstractType type;

    constructor(reference e) ;

};

//! XSD attribute class
class WSDL::XsdAttribute : public WSDL::XsdTypedData {

public:
        string use = "optional";

        const AllowedUseValues = ...;


    constructor(hash<auto> attr, Namespaces nsc, __7_ XsdAbstractType n_type, XsdLateResolverHelper unresolved) ;


    auto getValue(string val);

};

//! XSD element class
class WSDL::XsdElement : public WSDL::XsdTypedData {

public:
        int minOccurs = 1;
        int maxOccurs = 1;
        bool nillable = False;
        // the resolved namespace URI for any reference
        __7_ string ref_ns;
        // the source name for any reference
        __7_ string ref;
        bool usedocns;

constructor(hash<auto> e, Namespaces nsc, __7_ XsdAbstractType n_type, XsdLateResolverHelper unresolved, bool n_usedocns) ;


    AbstractDataProviderType getDataProviderType();


    __7_ list<auto> getAllowedValues();


    assimilate(WSDL::XsdElement other);


    bool isRequired();


    bool isNillable();


    auto serialize(Namespaces nsc, auto h, __7_ softbool omitType, string key, string typename);


protected:
     auto serializeAsIntern(Namespaces nsc, XsdAbstractType type, auto h, __7_ softbool omitType, string key, string typename);
public:


    auto deserialize(hash<string, XsdAbstractType> tmap, __7_ hash<auto> mrh, auto val, bool present);

};

class WSDL::UnionDataType : public AbstractDataProviderType {

public:
        list<AbstractDataProviderType> types;

    constructor(list<XsdAbstractType> unionTypes);


    string getName();


    bool isAssignableFrom(AbstractDataProviderType t);


    __7_ Type getValueType();


    __7_ AbstractDataProviderType getElementType();


    __7_ hash<string, AbstractDataField> getFields();


    hash<string, bool> getAcceptTypeHash();


    hash<string, bool> getReturnTypeHash();


    auto acceptsValue(auto value);

};

//! XSD simple type class
class WSDL::XsdSimpleType : public WSDL::XsdAbstractType {

public:
        hash<auto> enum;
        hash<auto> typeinfo;
        list<hash<auto>> union;
        list<XsdAbstractType> unionTypes;
        XsdAbstractType type;
        AbstractDataProviderType dataType;
        bool usedocns;

    constructor(hash<auto> st, Namespaces nsc, XsdLateResolverHelper unresolved, bool n_usedocns, __7_ string desc_name)
 ;


    finalizeResolved();


    AbstractDataProviderType getDataProviderType();


    __7_ list<auto> getAllowedValues();


    auto serialize(Namespaces nsc, auto val, __7_ softbool omitType);


    auto deserialize(string en, hash<string, XsdAbstractType> tmap, __7_ hash<auto> mrh, auto val);

};

//! complex type choice hash
public struct WSDL::ChoiceInfo {
    hash<string, XsdElement> elementmap;
    bool required;
};

//! XSD complex type class
class WSDL::XsdComplexType : public WSDL::XsdAbstractType {

public:
        __7_ XsdArrayType array;
        __7_ string arrayType;
        __7_ string restriction;
        __7_ string extension;

        bool usedocns;

        hash<string, XsdElement> elementmap();
        bool anyAttribute = False;

        bool nillable = False;
        bool required = False;
        bool requires_value = False;

        // attributes
        hash<string, XsdAttribute> attrs();

        // any annotated documentation
        __7_ string documentation;

        // temporarily used to store type information to resolve simpleType
        __7_ hash<auto> simpleTypeInfo;
        // simpleContent type
        XsdAbstractType simpleType;

        // finalization flag
        bool finalized = False;

        // multiple choice blocks
        list<hash<ChoiceInfo>> choices();

protected:
        //! type of complexType object
        string cx_type;

        const XET_ALL      = "ALL";
        const XET_CHOICE   = "CHOICE";
        const XET_SEQUENCE = "SEQUENCE";
        const XET_SIMPLE   = "SIMPLE";
        const XET_NONE     = "NONE";

public:

    // ct can be NOTHING in case of an empty complex type
    constructor(__7_ hash<auto> ct, Namespaces nsc, XsdLateResolverHelper unresolved, bool n_usedocns, __7_ string desc_name)
 ;


    __7_ list<auto> getAllowedValues();


    finalize(hash<string, XsdAbstractType> tmap, Namespaces nsc);


    //! throws an exception if the types are not compatible
    checkExtends(XsdAbstractType t, string ename);


protected:
     parseData(hash<auto> d, XsdLateResolverHelper unresolved, Namespaces nsc);
public:


protected:
     parseAttributes(reference d, XsdLateResolverHelper unresolved, Namespaces nsc);
public:


    bool isNillable();


    bool isRequired();


    bool requiresValue();


    bool isEmpty();


protected:
 hash<string, XsdElement> parseElements(softlist<auto> el, XsdLateResolverHelper unresolved, Namespaces nsc, bool for_object = True);
public:


    AbstractDataProviderType getDataProviderType();


protected:
     __7_ hash<auto> serializeElement(Namespaces nsc, string key, XsdElement element, auto h, __7_ softbool omitType);
public:


    __7_ hash<auto> serialize(Namespaces nsc, auto h, __7_ softbool omitType);


protected:
 hash<auto> serializeChoice(Namespaces nsc, hash<string, XsdElement> emap, hash<auto> h, *softbool omitType, bool all_members);
public:


    auto deserialize(string en, hash<string, XsdAbstractType> tmap, __7_ hash<auto> mrh, auto oval);


protected:
 __7_ hash<auto> parseChoice(hash<auto> val, hash<string, XsdElement> emap, string en, hash<string, XsdAbstractType> tmap, *hash<auto> mrh, *bool required);
public:

};

//! web service operation class
class WSDL::WSOperation : public WSDL::XsdNamedData {

public:
        //! request message
        __7_ WSMessage input;
        //! response message
        __7_ WSMessage output;
        //! request message name
        __7_ string input_name;
        //! response message name
        __7_ string output_name;

        //! namespace container
        Namespaces nsc;

        //! the SOAPAction header value to send with this operation
        __7_ string soapAction;

        //! a hash of top-level request names for this operation (depends on bindings)
        hash<string, bool> reqh;

        //! params per associated binding; key = binding name; value = OperationalBinding object
        hash<string, OperationalBinding> bindings;

protected:
        //! fault messages keyed by fault name
        hash<string, WSMessage> faults;

public:

    //! creates the WSOperation object from the arguments supplied
/** @param p the raw hash from deserialized XML data for the operation
        @param nsc the namespace container
        @param messages a hash of messages defined in the WSDL; keys are message names; values are WSMessage objects
     */
    constructor(hash<auto> p, Namespaces nsc, __7_ hash<auto> messages) ;


    //! returns a hash of fault messages keyed by fault name
    __7_ hash<string, WSMessage> getFaultMessages();


    //! returns True if the given string is a valid fault message for the operation
    bool hasFault(string fault);


    //! this method is called if this operation supports at least one binding with document style
    setDocStyleBinding(reference idocmap);


    //! this method is called if this operation supports at least one binding with RPC style
    markRpcStyleBinding();


    //! returns all top-level request names for this operation
    list<auto> getTopLevelRequestNames();


    //! returns a hash representing the given binding
/** @param bname the name of the binding, if not provided then uses the first assigned binding

        @return a binding param hash describing the binding with the following keys:
        - \c "httpMethod"
        - \c "soapTransport"

        @throw WSDL-BINDING-ERROR unknown binding
     */
    OperationalBinding getBinding(__7_ string bname);


    //! adds the given binding
/** @param bname binding name, must be unique in operation
        @param opparams a hash with the following keys:
        - \c "httpMethod"
        - \c "soapTransport"

        @throw WSDL-BINDING-ERROR when binding already registered
     */
    addBinding(WSDL::OperationalBinding b);


    //! serializes a fault response to an XML string or HTTP payload for the operation.
/** @param fault an optional fault message for the operation (@see getFaultMessages())
        @param h the fault data to serialize
        @param header optional soap header info to serialize if required (ex: authorization info). In the first step
        headers are matched to binding/input/header WSDL definition, remaining headers are passed as-is
        @param enc the optional encoding to use; if this argument is not present, then the default encoding will be
        used
        @param nsh an optional namespace hash for the output message
        @param xml_opts optional XML generation options
        @param req_soapaction if present will override any SOAPAction value for the request, ignored for HTTP binding
        @param bname SOAP binding name or empty to get the first assigned binding

        @return a hash with keys:
        - \c body: XML string in the SOAP response format or body part of HTTP message (string or binary). Target
          content type is evaluated from WSDL or in case of ambiguity using "^attributes^"."^content-type^". The
          provided content type must match allowed types in the WSDL. Also "application/x-www-form-urlencoded" or any
          general type are supported.
        - \c hdr: hash of HTTP headers
        - \c path: the path part of URL. Used when urlEncoded is defined
        - \c method: the HTTP request method
    */
hash<auto> serializeFault(__7_ string fault, string faultmsg, auto h, __7_ hash<auto> header, __7_ string enc, *hash<auto> nsh, *int xml_opts, *string req_soapaction, *string bname);


    //! serializes a request to an XML string or HTTP payload for the operation.
/** @param h the request to serialize
        @param header optional soap header info to serialize if required (ex: authorization info). In the first step
        headers are matched to binding/input/header WSDL definition, remaining headers are passed as-is
        @param enc the optional encoding to use; if this argument is not present, then the default encoding will be
        used
        @param nsh an optional namespace hash for the output message
        @param xml_opts optional XML generation options
        @param req_soapaction if present will override any SOAPAction value for the request, ignored for HTTP binding
        @param bname SOAP binding name or empty to get the first assigned binding

        @return a hash with keys:
        - \c body: XML string in the SOAP request format or body part of HTTP message (string or binary). Target
          content type is evaluated from WSDL or in case of ambiguity using "^attributes^"."^content-type^".  The
          provided content type must match allowed types in WSDL; also "application/x-www-form-urlencoded" or any
          general type are supported.
        - \c hdr: hash of HTTP headers
        - \c path: the path part of URL. Used when urlEncoded is defined
        - \c method: the HTTP request method

        @throw SOAP-SERIALIZATION-ERROR no input message description
    */
    hash<auto> serializeRequest(auto h, __7_ hash<auto> header, __7_ string enc, __7_ hash<auto> nsh, __7_ int xml_opts, __7_ string req_soapaction, __7_ string bname, __7_ hash<auto> http_headers);


    //! serializes a SOAP response to an XML string for the operation
/** @param h the response to serialize
        @param header SOAP header hash. In the first step headers are matched to binding/input/header WSDL definition,
        remaining headers are passed as-is
        @param enc the optional encoding to use; if this argument is not present, then the default encoding will be
        used
        @param nsh namespace hash
        @param soap12 set to True if the response should use SOAP 1.2 encoding
        @param xml_opts optional XML generation options
        @param bname SOAP binding name, leave empty to get the first assigned binding
        @param http_headers HTTP headers to serialize in the response

        @return a hash with keys:
        - \c body: XML string in the SOAP request format or body part of HTTP message (string or binary). Target
          content type is evaluated from WSDL or in case of ambiguity using "^attributes^"."^content-type^".  The
          provided content type must match allowed types in WSDL; also "application/x-www-form-urlencoded" or any
          general type are supported.
        - \c hdr: hash of HTTP headers

        @throw SOAP-SERIALIZATION-ERROR no output message description
    */
    hash<auto> serializeResponse(auto h, __7_ hash<auto> header, __7_ string enc, __7_ hash<auto> nsh, __7_ bool soap12, __7_ int xml_opts, __7_ string bname, __7_ hash<auto> http_headers);


    //! parses a hash representing a parsed XML request (parsed with @ref parse_xml() "parse_xml(XPF_PRESERVE_ORDER)") for the operation and returns the corresponding Qore data structure
/** @param o the parsed SOAP or HTTP request with @ref WSDLLib::parseSOAPMessage() for the operation
        @param bname SOAP binding name, leave empty to use the first assigned binding
        @return the Qore data structure corresponding to the request data. When SOAP header is deserialized according binding/input WSDL then all data are passed under subhash vales prefixed by message name
        In case of HTTP binding the content type is matched against WSDL binding when wildcards are supported and current content type value is applied to "^attributes^"."^content-type^".
    */
    auto deserializeRequest(hash<auto> o, __7_ string bname);


    //! parses a hash representing a parsed XML response (parsed with @ref parse_xml() "parse_xml(XPF_PRESERVE_ORDER)") for the operation and returns the corresponding Qore data structure
/** @param o the parsed SOAP or HTTP request with @ref WSDLLib::parseSOAPMessage() for the operation
        @param bname SOAP binding name, leave empty to use the first assigned binding
        @return the Qore data structure corresponding to the response data. When SOAP header is deserialized according binding/output WSDL then all data are passed under subhash vales prefixed by message name
        In case of HTTP binding the content type is matched against WSDL binding when wildcards are supported and current content type value is applied to "^attributes^"."^content-type^".
    */
    auto deserializeResponse(hash<auto> o, __7_ string bname);


/** Operation is resolved from method and path so let's look if data are passed in URL.
        When URL replacement is defined for operation then exception is raised.
        @return NOTHING if data are not encoded in URL for operation
    */
    __7_ hash<auto> deserializeRequestPath(string path, __7_ string bname);


    //! processes a hash of raw deserialized XML data and removes any namespace prefix from keys (saved to a \c "ns" key in the hash vale assigned to the key) and returns the processed version of the hash
protected:
      static hash<auto> processNSValue(hash<auto> h);
public:


    //! returns True if the operation is a SOAP 1.2 operation
/** @return True if the operation is a SOAP 1.2 operation
    */
    bool isSoap12();


    //! returns the target namespace for the operation
/** @return the target namespace for the operation
    */
    string getTargetNS();

};

//! message argument hash description
public struct WSDL::ArgInfo {
    //! the arg name
    string name;

    //! unique part name
    string part;

    //! type for the argument
    XsdAbstractType type;

    //! element for the argument
    XsdElement element;
};

//! web service message class
/**
Message definition consists of part definition. Part name is important for simple types to name it at both SOAP and Qore side (hash key).
For elements is not used at the SOAP side, it implies condition the element name must be unique in message. In the other words
it's forbidden definition of two parts with the same element. It's unclear if is legal in Wsdl and how to handle it. Class raises
exception if detects it. The part name is also used for references from binding (header, content).
*/
class WSDL::WSMessage : public WSDL::XsdNamedData {

public:
        //! args keys are part names for types reps. element names for element. Definition must provide unique values.
        hash<string, hash<WSDL::ArgInfo>> args;

        //! maps part names to args key, all parts are in hash
        hash<string, string> pmap;

        //! type map
        hash<string, XsdAbstractType> tmap;

        // keep a weak reference to the namespace map
        Namespaces nsc;

    constructor(hash<auto> m, hash<string, hash<string, XsdElement>> emap, hash<string, XsdAbstractType> tmap, Namespaces nsc) ;


    //! Returns the data provider type for this message
    AbstractDataProviderType getDataProviderType();


/**
        Just serialize value and return namespace
        @param ons is reference to string
    */
    auto serializeRpcValue(string part, bool encoded, reference<hash<auto>> h, reference<string> ons);


    //! serializes data into a hash with SOAP namespaces etc. with RPC-style messages
/**
        @param parts if present then serializes only particular element or part, if NOTHING then serializes all message elements
        @param msginfo a description of the message for the current binding, if available
        @param n_name name of output key
        @param encoded if the encoded use module should be used (or the literal use model)
        @param h data to be serialized, keys are wsdl element names (or optionally the part name) and part names for simple types
        @param fault @ref True if serializing a fault response
     */
    __7_ hash<auto> serializeRpc(__7_ softlist<auto> parts, __7_ WSDL::BindingMessageDescription msginfo, __7_ MultiPartRelatedMessage mpm, string n_name, bool encoded, reference<auto> h, bool fault);


    //! serializes data into a hash with SOAP namespaces etc. with document-style messages
/**
        @param parts if present then serializes only particular element or part, if NOTHING then serializes all message elements
        @param msginfo a description of the message for the current binding, if available
        @param mpm a multipart message object, for future use
        @param encoded if the encoded use module should be used (or the literal use model)
        @param h data to be serialized, keys are wsdl element names (optionally part name is also possible) and part names for simple types
        @param fault @ref True if serializing a fault response
     */
    __7_ hash<auto> serializeDocument(__7_ softlist<auto> parts, __7_ WSDL::BindingMessageDescription msginfo, __7_ MultiPartRelatedMessage mpm, bool encoded, reference<auto> h, bool fault);


    //! deserialize RPC message
/**
    @param val, keys are element names or part names for simple types
    @param part if exists then deserialize only particular part
    @return serialized data, keys are wsdl element names (optionally part name is also possible) and part names for simple types
    */
    __7_ hash<auto> deserializeRpc(__7_ hash<auto> mrh, hash<auto> val, __7_ string part);


    __7_ hash<auto> deserializeDocument(__7_ hash<auto> mrh, auto val, __7_ string parts);


    //! serialize all values as string or binary
/**
        @return hash of all parts in key-value pair
    */
    hash<auto> serializeAllPartData(__7_ hash<auto> val);


    //! serialized value of particular type value as string or binary
/**
        @param val value to be resolved
        @param key member name, it is requires reference to a simple type, not compaund element

        @return as hash with 'value' and optional 'content-type' key passed from v.content-type if exists
     */
    hash<auto> serializeData(Namespaces nsc, string key, __7_ hash<auto> val);


    //! deserialize value in string or binary form
/**
        @return hash<auto> in key-value pair
    */
    hash<auto> deserializeData(string part, hash<auto> val);


    //! deserialize all values in string or binary form
/**
        @return hash of all parts in key-value pair
    */
    hash<auto> deserializeAllPartData(hash<auto> val);


    //! find part in value, remove that value from the hash
/**
        @param ename element/type name
        @param v reference to hash, supposed structure is [msgname.](partname|elemname) = value to support more message data in one hash
        @param removeFound if True found values are removed from the hash

        @return resolved value, if not found then returns NOTHING
    */
private:
     auto getValueFromHash(string ename, reference<auto> v, bool removeFound);
public:


    //! when only one arg then try to get values based on element keys
private:
     auto getValueFromHash(__7_ WSDL::XsdElement element, reference<auto> v, bool removeFound);
public:


    string getSerializedKey(string part);


    //! check if pname is defined as message part
/** @return translated part name to element/type name
     */
    string checkPart(string pname);

};

// private helper class for lazy name resolution
class WSDL::XsdLateResolverHelper {

public:
protected:
        list<object> l();

public:

    constructor();


    add(object v);


    list<auto> getList();


    clearResolved();


      static bool isResolved(XsdTypedData t);

      static bool isResolved(XsdSimpleType t);

      static bool isResolved(XsdComplexType t);

      static bool isResolved(XsdArrayType t);

      static bool isResolved(auto t);
};

//! class for WSDL bindings associated with a SOAP operation
class WSDL::OperationalBinding : public WSDL::XsdNamedData {

public:
        bool docstyle;
        __7_ WSDL::BindingMessageDescription input;
        __7_ WSDL::BindingMessageDescription output;
        __7_ string soapAction;

    constructor(string name, string ns, bool docstyle, __7_ string soapAction, __7_ WSDL::BindingMessageDescription input, __7_ WSDL::BindingMessageDescription output) ;


    hash<auto> serializeMessage(WSOperation op, __7_ WSMessage msg, bool request, bool soap12, auto h, __7_ hash<auto> header, string enc = 'UTF-8', __7_ hash<auto> nsh, __7_ int xml_opts, __7_ string req_soapaction, __7_ string faultmsg, __7_ hash<auto> http_headers);


    auto deserializeMessage(WSOperation op, hash<auto> o, bool request);


    auto deserializeRequestPath(WSOperation op, string path);


    abstract hash<auto> serializeMessageImpl(WSOperation op, __7_ WSMessage msg, bool request, bool soap12, auto h, __7_ hash<auto> header, string enc, __7_ hash<auto> nsh, __7_ int xml_opts, __7_ string req_soapaction, __7_ string faultmsg, __7_ hash<auto> http_headers);

    abstract auto deserializeMessageImpl(WSOperation op, hash<auto> o, bool request);

    abstract auto deserializeRequestPathImpl(WSOperation op, string path);
};

class WSDL::SoapBinding : public WSDL::OperationalBinding {

public:
        string soapTransport;

    constructor(string name, string ns, bool docstyle, string soapTransport, __7_ string soapAction, __7_ WSDL::BindingMessageDescription input, __7_ WSDL::BindingMessageDescription output) ;


    hash<auto> serializeMessageImpl(WSOperation op, __7_ WSMessage msg, bool request, bool soap12, auto h, __7_ hash<auto> header, string enc, __7_ hash<auto> nsh, __7_ int xml_opts, __7_ string req_soapaction, __7_ string faultmsg, __7_ hash<auto> http_headers);


private:
 hash<auto> serializeSoapMessage(WSOperation op, __7_ WSMessage msg, auto val, __7_ hash<auto> header, *hash<auto> nsh, bool request, bool soap12, reference mpm, *string faultmsg);
public:


    auto deserializeMessageImpl(WSOperation op, hash<auto> orig_msg, bool request);


    //! processes multi-part references and returns a hash of multi-part reference info and the updated message body hash
private:
      static list<auto> processMultiRef(hash<auto> body);
public:


    hash<auto> deserializeRequestPathImpl(WSOperation op, string path);

};

class WSDL::HttpBinding : public WSDL::OperationalBinding {

public:
        string httpMethod;
        string location;

constructor(string name, string ns, bool docstyle, string httpMethod, string location, __7_ string soapAction, *WSDL::BindingMessageDescription input, *WSDL::BindingMessageDescription output) ;


hash<auto> serializeMessageImpl(WSOperation op, __7_ WSMessage msg, bool request, bool soap12, auto h, *hash<auto> header, string enc, *hash<auto> nsh, *int xml_opts, *string req_soapaction, *string faultmsg, *hash<auto> http_headers);


    //! serializes path part of URL when urlEncoded WSDL is defined
private:
     string serializeRequestPath(WSOperation op, auto h);
public:


    auto deserializeMessageImpl(WSOperation op, hash<auto> v, bool request);


    hash<auto> deserializeRequestPathImpl(WSOperation op, string path);

};

//! describes the message body for a SOAP operational binding message description
/** @see https://www.w3.org/TR/wsdl#_soap:body
*/
class WSDL::BindingMessageBodyDescription : public Serializable {

public:
        //! required attribute
/** if @ref Qore::True "True", then use = \c "encoded" (each message part references an abstract type using the \c type attribute),
            otherwise use = \c "literal" (each part references a concrete schema definition using either the \c element or \c type attribute)
         */
        bool encoded;

        //! when encoded @ref Qore::True "True", this is used to produce a concrete message by applying the specified encoding
        __7_ string encodingStyle;

        //! namespace; "only applies to content not explicitly defined by the abstract types"
        __7_ string ns;

        //! indicates which parts appear somewhere within the SOAP Body portion of the message
        __7_ list<auto> parts;

    constructor(bool encoded, __7_ string encodingStyle, __7_ string ns, __7_ string parts);


    bool isMultipart();

};

//! describes a SOAP message header for a SOAP operational binding message description
class WSDL::BindingMessageHeaderDescription : public Serializable {

public:
        //! message part name
        string part;

        //! required attribute
/** if @ref Qore::True "True", then use = \c "encoded" (each message part references an abstract type using the \c type attribute),
            otherwise use = \c "literal" (each part references a concrete schema definition using either the \c element or \c type attribute)
            */
        bool encoded;

        //! a weak reference to the actual message
        WSMessage msg;

    //! creates the object
    /*
        @param part message part name
        @param encoded if @ref Qore::True "True", then use = \c "encoded" (each message part references an abstract type using the \c type attribute), otherwise use = \c "literal" (each part references a concrete schema definition using either the \c element or \c type attribute)
        @param msg actual message
    */
    constructor(string part, bool encoded = False, WSMessage msg);

};

//! describes MIME content information for SOAP operational binding message descriptions
/** @see https://www.w3.org/TR/wsdl#_mime:content
 */
class WSDL::BindingContentDescription : public Serializable {

public:
        //! specifies the name of the message part
        string part;

        // list of accepted content types (with "xxx/*")
        list<auto> acceptedContentTypes = ();

        // list of accepted specific content types
        list<auto> acceptedContentTypeSubtypes = ();

        //! are all content-types form URL encoded?
        bool formUrlEncoded = False;

        //! are all content-types accepted?
        bool acceptAllContentTypes = False;

    addContentType(string type);

};

//! describes a mimeXml payload for a SOAP operational binding message description
/** @see https://www.w3.org/TR/wsdl#_mime:mimeXml
 */
class WSDL::MimeXmlMessageDescription : public Serializable {

public:
        //! refers to a message part defining the concrete schema of the root XML element
/** The part references a concrete schema using the element attribute for simple parts or type attribute for composite parts
         */
        string part;

    constructor(string part);

};

//! describes an input or output message in a SOAP operational binding
class WSDL::BindingMessageDescription : public Serializable {

public:
        //! the description of the message body
        __7_ WSDL::BindingMessageBodyDescription body;

        //! a hash of URL replacement values keyed by part name, values are URI paths
/** @note for HTTP bindings only
         */
        __7_ hash<auto> urlReplacement;

        //! indicates that all the message parts are encoded into the HTTP request URI using the standard URI-encoding rules
/** ex: name1=value&name2=value...

            The names of the parameters correspond to the names of the message parts.
            Each value contributed by the part is encoded using a name=value pair.
            This may be used with GET to specify URL encoding, or with POST to specify a FORM-POST.
            For GET, the "?" character is automatically appended as necessary.

            @note for HTTP bindings only

            @see https://www.w3.org/TR/wsdl#_http:urlEncoded
         */
        bool urlEncoded = False;

        //! optional content-type descriptions
        __7_ WSDL::BindingContentDescription content;

        //! optional mimeXml message scription
        __7_ WSDL::MimeXmlMessageDescription mimeXml;

private:
        //! a list of WSDL::BindingMessageHeaderDescription objects
        list<BindingMessageHeaderDescription> headers = cast<list<BindingMessageHeaderDescription>>(());

public:

    addHeader(BindingMessageHeaderDescription hdr);


    //! returns @ref Qore::True "True" if the header descriptions are available
    bool hasHeaders();


    //! returns a list of WSDL::BindingMessageHeaderDescription objects
    list<BindingMessageHeaderDescription> getHeaders();


    setUrlReplacement(string loc);

};

//! class for WSDL bindings
class WSDL::Binding : public WSDL::XsdNamedData {

public:
private:
        string port;

        //! use RPC or document style
/** according to https://www.w3.org/TR/wsdl#_soap:operation
            'If the soap:binding element does not specify a style, it is assumed to be "document".'
         */
        bool docstyle = True;

        __7_ string httpMethod;
        __7_ string soapTransport;

public:

    constructor(hash<auto> data, Namespaces nsc, reference portTypes, reference idocmap, __7_ hash<auto> messages) ;


    //! check if part exists in massage, if part is empty and message contains single part then return it otherwise raise an exception
private:
     string checkMessagePart(WSMessage msg, __7_ string partname, string errs);
public:


    string getPort();


    bool isSoapBinding();


    bool isHttpBinding();

};

// private namespace prefix redefinition class
class WSDL::NamespacePrefixHelper {

public:

protected:
        //! namespaces data
        Namespaces nsc;

        // overriden prefixes
        hash<auto> h;

        // overridden target namespace
        bool targ_ns;

        // overridden default namespace
        bool def_ns;

        // schema description
        string desc;

        //! schema internal ID
        int id = ++count;

        //! schema ID counter
        static int count = 0;

public:

    constructor(Namespaces nsc, __7_ hash<auto> nsattr);


    destructor();


    save(string k, string v);


    string getDesc();

};

//! namespace container class
class WSDL::Namespaces : public Serializable {

public:
        // options are here since this object is passed through all serialization and deserialization
        bool opt_empty_string_is_nothing = False;
        bool opt_allow_any_header = False;

protected:
        //! maps output namespace prefixes to namespace URIs
        hash<string, string> ns = {
            "xsd": XSD_NS,
        };

        //! maps namespace URIs to output namespace prefixes
        hash<string, string> nsr = {
            XSD_NS: "xsd",
        };

        //! hash for valid XSD namespaces, maps input namespace prefixes -> True if it refers to an XSD schema
        hash<string, bool> xsd_schema = {
            "xsd": True,
        };

        //! hash mapping input namespace prefixes to namespaces URIs
        hash<string, string> imap = {
            "xsd": XSD_NS,
        };

        //! hash mapping input namespace URIs to input namespace prefixes
        hash<string, string> imapr = {
            XSD_NS: "xsd",
        };

        //! if True then has SOAP 1.1
        bool hassoap11 = False;

        //! if True then has SOAP 1.2
        bool hassoap12 = False;

        //! current target namespace
        __7_ string target_ns;

        //! target namespace stack;
        list<auto> nss = ();

        //! default namespace for unprefixed definitions
        __7_ string default_ns;

        //! default namespace stack;
        list<auto> dss = ();

        //! base type map
        hash<string, XsdBaseType> base_types;

        //! maps namespaces to default prefixes
        const DefaultPrefixes = ...;


public:

    //! creates the object with the WSDL definitions attribute hash
    constructor(hash<auto> nsh, __7_ Namespaces nsc, __7_ string targetNamespace);


    __7_ string getDefaultNs();


    addNamespaces(hash<auto> nsh, __7_ NamespacePrefixHelper nph);


    restorePrefixes(hash<auto> h);


protected:
     addNamespaceIntern(string ns, string val, __7_ bool override);
public:


    //! merges namespaces when parsing imported schemas
    merge(Namespaces nsc);


    //! returns a unique namespace prefix from a URI path string
    string getUniquePrefix(string uri, __7_ string default_prefix);


    //! returns the namespace URI for the given output namespace prefix
    string getOutputNamespaceUri(string nsp);


    //! returns a hash of namespace attributes for outgoing SOAP messages
    __7_ hash<auto> getOutputNamespaceHash(__7_ hash<auto> nsh);


    //! returns the primary target namespace Uri
    __7_ string getTargetNamespaceUri();


    //! returns the namespace URI for the given prefix or the target namespace Uri
    __7_ string getNamespaceUri(__7_ string nsp);


    //! pushes the current target namespace URI on the stack when parsing schemas and sets the current target namespace URI to the current value
    pushTargetNamespace(string ns);


    //! restores any previous target namespace URI from the stack to the current target namespace URI
    popTargetNamespace();


    //! pushes the current default namespace URI on the stack when parsing schemas and sets the current default namespace URI to the current value
    pushDefaultNamespace(string ns);


    //! restores any previous default namespace URI from the stack to the current default namespace URI
    popDefaultNamespace();


    //! looks up and registers a namespace if necessary, returns the namespace's prefix
    string getOutputNamespacePrefix(string ns);


    //! returns the output namespace prefix for the target namespace, if any
    string getTargetNamespaceOutputPrefix();


    //! returns the input namespace prefix for the target namespace, if any
    string getTargetNamespaceInputPrefix();


    //! registers a namespace internally
protected:
     string registerNamespaceIntern(string uri, __7_ string default_pfx);
public:


    //! returns a hash of namespace prefixes to namespaces URIs actually used
    __7_ hash<auto> getReferencedNamespaceMap();


    //! returns True if using SOAP 1.1, False if not
    bool hasSoap11();


    //! returns True if using SOAP 1.2, False if not
    bool hasSoap12();


    //! returns True if if the input namespace prefix refers to the XSD namespace URI
    __7_ bool isSchema(string ns);


    //! returns the input namespace URI from the input namespace prefix
    string getInputNamespaceUri(string nsa);


    bool doType(string t, reference<hash> typeinfo, reference<XsdAbstractType> rtype, __7_ hash<string, XsdAbstractType> tmap);


    hash<auto> getTypeHash(string t);


    XsdBaseType getBaseType(string t);


    //! returns the output namespace prefix for the given input namespace prefix
    string translateOutputNamespacePrefix(__7_ string nsa);

};

// private functions
XsdAbstractType tmap_get(hash<string, XsdAbstractType> tmap, string name);


*XsdAbstractType tmap_try_get(hash<string, XsdAbstractType> tmap, string name);


// private functions
XsdElement emap_get(hash<string, hash<string, XsdElement>> emap, string ns, string name);


*XsdElement emap_try_get(hash<string, hash<string, XsdElement>> emap, string ns, string name);


//! WSDL port type hash
public struct WSDL::PortTypeInfo {
    hash<string, WSOperation> operations;
};

//! WSDL service info hash
public struct WSDL::ServiceInfo {
    //! service name
    string name;
    //! port info
    hash<string, hash> port;
};

//! WSDL operation info
public struct WSDL::OperationInfo {
    //! the name of the port defining the operation
    string port;

    //! the operation object
    WSDL::WSOperation operation;
};

//! main class representing a parsed WSDL file
/** This is the main class for handling SOAP communication and is based on a WSDL file
*/
class WSDL::WebService : public WSDL::XsdBase,public  Qore::Serializable {

public:
        //! the WSDL string
        string wsdl;

        //! a cryptographic hash for the WSDL source
        string wsdl_hash;

        //! options used in the constructor
        __7_ hash<auto> opts;

        //! flag if the object has a "try_import" closure or call reference
        bool has_try_import = False;

        //! the name of the WebService
/** @since %WSDL 0.5
        */
        string name;

        //! the digest of the WSDL source
/** @since %WSDL 0.5
        */
        string hash_str;

        //! namespace container
        transient Namespaces nsc;

        //! list of services in this WSDL
        transient list<string> wsdl_services();

        //! a hash of element names to XsdElement objects for top-level document-style bindings
        transient hash<string, XsdElement> idocmap();

        //! map of elements
        transient hash<string, hash<string, XsdElement>> emap();

        //! hash of messages names to messages
        transient hash<string, WSMessage> messages();

        //! map of types
        transient hash<string, XsdAbstractType> tmap();

        //! hash of port names to port type information hashes
        transient hash<string, hash<PortTypeInfo>> portType();

        //! optional closure/call reference to try to resolve import declarations
        transient __7_ code try_import;

        //! default path for retrieving XSD references
/** @note there is no need to serialize \c def_path as it will be set with the saved options
        */
        transient __7_ string def_path;

protected:
        // service definitions; name -> service info hash
        hash<string, hash<WSDL::ServiceInfo>> services();

        // service bindings; name -> Binding
        hash<string, Binding> binding();

        //! issue #4449: map of imports used while parsing
        transient hash<string, bool> import_map;

public:

    //! creates the WebService object
/** @param str the XML string representing the WSDL
        @param opts an optional hash of options with the following possible keys:
        - \c "compat_allow_any_header": allow any SOAP header to be sent without checking the WSDL
        - \c "compat_empty_string_is_nothing": deserialize empty strings to @ref nothing instead of an empty string
        - \c "def_path": the default path to use when retrieving referenced XSDs
        - \c "try_import": a call reference or closure to be passed a string name for XSD imports without any scheme to retrieve the data, must take a string argument (the resource name) and return a string (the resource data)
        - \c "http_client": a HTTPClient object for retrieving data from import commands
        - \c "http_headers": a hash of optional HTTP header info to use when retrieving data from import commands

        @since 0.3.7 added the \c "compat_empty_string_is_nothing" and \c "compat_allow_any_header" options for backwards compatibility
    */
    constructor(string str, __7_ hash<auto> opts);


    //! returns the name of the service
/** @return the name of the service

        @since %WSDL 0.5
    */
    string getName();


    //! returns a unique hash for the WSDL that can be used to compare schemas for equality
/** @return a unique hash for the WSDL that can be used to compare schemas for equality

        @since %WSDL 0.5
    */
      synchronized string getHash();

    //! returns the given operation or throws an exception if it cannot be found; operations are searched in portTypes in the order they are declared
/** @throw WSDL-OPERATION-ERROR if the operation is unknown
    */
    WSDL::WSOperation getOperation(string opname);


    //! returns a list of known operation names
    softlist<string> getOperationNames();


    //! returns the given operation for particular porttype or throws an exception if it cannot be found
    WSDL::WSOperation getPortTypeOperation(string ptname, string opname);


    //! returns the given operation for particular binding or throws an exception if it cannot be found
    WSDL::WSOperation getBindingOperation(__7_ string bname, string opname);


    //! returns a list of hashes giving supported operation names for each port in the WSDL
/** @return a list of hashes giving supported operation names for each port in the WSDL; each hash<auto> has the following keys:
        - \c "port": the name of the port defining the operation
        - \c "operation": the @ref WSDL::WSOperation object for the operation
     */
    list<hash<OperationInfo>> listOperations();


    //! return a @ref WSDL::Binding object describing the requested binding
/** @param name the name of the binding

        @return a @ref WSDL::Binding object

        @throw WSDL-BINDING-ERROR unknown binding
     */
    WSDL::Binding getBinding(string name);


    //! returns a list of services defined in the WSDL
/** @return a list of hashes of services defined in the WSDL; each hash<auto> has the following keys:
        - \c "name": the name of the service
        - \c "port": a hash of port information; the keys are port names and the values have the following keys:
          - \c "address": the location of the port
          - \c "binding": the binding of the port
     */
    list<hash<WSDL::ServiceInfo>> listServices();


    //! returns a hash describing the requested service
/** @param name the name of the service
        @return a hash describing the service with the following keys:
        - \c "name": the name of the service
        - \c "port": a hash of port information; where each hash<auto> is keyed by port name and the values have the following keys:
          - \c "address": the location of the port
          - \c "binding": the binding of the port

        @throw WSDL-SERVICE-ERROR the service is not known
     */
    hash<WSDL::ServiceInfo> getService(string name);


    //! return a hash value that can be used to serialize a value as a particular type
    hash<auto> getType(string name, auto v);


protected:
     XsdBaseType getBaseType(string t);
public:


protected:
     resolveType(XsdSimpleType t);
public:


protected:
     resolveType(XsdElement xe);
public:


protected:
     resolveType(XsdAttribute xd);
public:


protected:
     resolveType(XsdComplexType ct);
public:


protected:
     resolveType(XsdArrayType t);
public:


protected:
     XsdAbstractType resolveType(hash<auto> v);
public:


    // parse XSD schema types
protected:
     parseTypes(__7_ hash<auto> data, auto http_client, auto http_headers);
public:


protected:
     parseMessages(__7_ softlist<auto> message);
public:


protected:
     parseService(__7_ softlist<auto> svcs);
public:


protected:
     parsePortType(__7_ softlist<auto> data);
public:


protected:
     parseBinding(__7_ softlist<auto> bindings);
public:


    //! returns True if the WSDL describes a SOAP 1.2 service
/** @return True if the WSDL describes a SOAP 1.2 service
    */
    bool isSoap12();


    //! returns the XML string for the WSDL
/** @return the XML string for the WSDL
    */
    string getWSDL();


    //! returns a cryptographic hash for the WSDL source
/** @return a cryptographic hash for the WSDL source
    */
    string getWSDLHash();


    //! returns the XML string for the WSDL, adjusting the URLs for binding locations depending on the caller
/** @param base_url the url to be prepended before the service name;
                        it must include all the handler prefixes (like \c /SOAP)
                        to produce valid URL
        @return the XML string for the WSDL
     */
    string getWSDL(string base_url);


protected:
     string getOperationParams(WSMessage msg);
public:


    //! prepare a WebService report enumerating the services and operations of the WSDL, outputs to the output stream given as the first argument
/** @param stream the output stream for the report data
        @param wsdl_name name of the WSDL, typically a file name

        @return human readable text report
    */
    getReport(StringOutputStream stream, __7_ string wsdl_name);


    //! prepare a WebService report enumerating the services and operations of the WSDL
/** @param wsdl_name name of the WSDL, typically a file name

        @return human readable text report
    */
    string getReport(__7_ string wsdl_name);


    //! supports the constructor and deserialization
/** @since %WSDL 0.4
    */
private:
     doInit(string str, __7_ hash<auto> opts);
public:


    //! can be used to create a WebService object from a hash created with serialize()
/** @return a WebService object from the argument hash created with serialize()
    */
private:
     deserializeMembers(hash<auto> members);
public:

};

//! helper class implementing sample message generation
/** This is the class for generating sample SOAP messages based on a WSDL file
*/
class WSDL::WSMessageHelper {

public:
protected:
        WebService ws;
        hash<auto> opts;

public:

    const DefaultOpts = ...;


    //! creates the WebService object
/** @param ws the object representing the WSDL
        @param opts an optional hash of options with the following possible keys:
        - \c "comments": (bool) generate comments as values with ^comment^ key, default: False
        - \c "choices": (bool) generate choices as sub hashes ^choices^ key, default: False
        - \c "max_items" (int): max.number of array elements to output, default: 3
    **/
    constructor(WebService ws, __7_ hash<auto> opts);


protected:
     hash<auto> getTypeInfo(XsdBaseType t);
public:


protected:
     hash<auto> getTypeInfo(XsdSimpleType t);
public:


protected:
     hash<auto> getTypeInfo(XsdComplexType t);
public:


    hash<auto> getMessage(XsdElement elem, __7_ softlist<auto> comments);


    //! prepare sample message
/** @param name message name

        @return a hash representing message
    */
    hash<auto> getMessage(string name);


    //! prepare sample message
/** @param msg WSMessage object

        @return a hash representing message
    */
    auto getMessage(WSMessage msg);

};
