//! Qore namespace
namespace Qore {
//! Xml namespace
namespace Xml {
/** @defgroup xml_functions XML Functions
//@{
 */

//! serializes a hash into an XML string with whitespace formatting but without an XML header
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param h a hash of data to serialize: the hash can have any number of keys
    @param encoding an optional string giving the encoding for the output XML string; if this parameter is missing, the output string will have the default encoding

    @return an XML string corresponding to the input data, with whitespace formatting but without an XML header

    @throw MAKE-XML-ERROR An error occurred serializing the %Qore data to an XML string

    @see @ref serialization

    @deprecated use make_xml_fragment(); camel-case function names were deprecated in xml 1.3
 */
string makeFormattedXMLFragment(hash h, __7_ string encoding);

//! This is a variant that is basically a noop, included for backwards-compatibility for functions that ignored type errors in the calling parameters
/**@par Code Flags:
    @ref RUNTIME_NOOP, @ref DEPRECATED

*/
nothing makeFormattedXMLFragment();

//! serializes a hash into an XML string with whitespace formatting and with an XML header
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param key top-level key
    @param h the rest of the data to serialize under the top-level key
    @param encoding an optional string giving the encoding for the output XML string; if this parameter is missing, the output string will have the default encoding

    @return an XML string corresponding to the input data, with whitespace formatting and with an XML header

    @throw MAKE-XML-ERROR An error occurred serializing the %Qore data to an XML string

    @see @ref serialization

    @deprecated use make_xml(); camel-case function names were deprecated in xml 1.3
*/
string makeFormattedXMLString(string key, hash h, __7_ string encoding);

//! serializes a hash into an XML string with whitespace formatting and with an XML header
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param h a hash of data to serialize: the hash must have one top-level key and no more or an exception will be raised
    @param encoding an optional string giving the encoding for the output XML string; if this parameter is missing, the output string will have the default encoding

    @return an XML string corresponding to the input data, with whitespace formatting and with an XML header

    @throw MAKE-FORMATTED-XML-STRING-PARAMETER-EXCEPTION the hash passed not not have a single top-level key (either has no keys or more than one)
    @throw MAKE-XML-ERROR An error occurred serializing the %Qore data to an XML string

    @see @ref serialization

    @deprecated use make_xml(); camel-case function names were deprecated in xml 1.3
 */
string makeFormattedXMLString(hash h, __7_ string encoding);

//! serializes a hash into an XML string without whitespace formatting and without an XML header
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param h a hash of data to serialize: the hash can have any number of keys
    @param encoding an optional string giving the encoding for the output XML string; if this parameter is missing, the output string will have the default encoding

    @return an XML string corresponding to the input data, without whitespace formatting and without an XML header

    @throw MAKE-XML-ERROR An error occurred serializing the %Qore data to an XML string

    @see @ref serialization

    @deprecated use make_xml_fragment(); camel-case function names were deprecated in xml 1.3
 */
string makeXMLFragment(hash h, __7_ string encoding);

//! This is a variant that is basically a noop, included for backwards-compatibility for functions that ignored type errors in the calling parameters
/**@par Code Flags:
    @ref RUNTIME_NOOP, @ref DEPRECATED

*/
nothing makeXMLFragment();

//! serializes a hash into an XML string without whitespace formatting but with an XML header
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param key top-level key
    @param h the rest of the data to serialize under the top-level key
    @param encoding an optional string giving the encoding for the output XML string; if this parameter is missing, the output string will have the default encoding

    @return an XML string corresponding to the input data, without whitespace formatting but with an XML header

    @throw MAKE-XML-ERROR An error occurred serializing the %Qore data to an XML string

    @see @ref serialization

    @deprecated use make_xml(); camel-case function names were deprecated in xml 1.3
 */
string makeXMLString(string key, hash h, __7_ string encoding);

//! serializes a hash into an XML string without whitespace formatting but with an XML header
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param h a hash of data to serialize: the hash must have one top-level key and no more or an exception will be raised
    @param encoding an optional string giving the encoding for the output XML string; if this parameter is missing, the output string will have the default encoding

    @return an XML string corresponding to the input data, without whitespace formatting but with an XML header

    @throw MAKE-XML-STRING-PARAMETER-EXCEPTION the hash passed not not have a single top-level key (either has no keys or more than one)
    @throw MAKE-XML-ERROR An error occurred serializing the %Qore data to an XML string

    @see @ref serialization

    @deprecated use make_xml(); camel-case function names were deprecated in xml 1.3
 */
string makeXMLString(hash h, __7_ string encoding);

//! serializes a hash into an XML string with an XML header
/** @par Code Flags:
    @ref RET_VALUE_ONLY

    @par Example:
    @code string xml = make_xml("key", hash); @endcode

    @param key top-level key
    @param h the rest of the data to serialize under the top-level key
    @param flags optional formatting flags; see @ref xml_generation_constants for more information
    @param encoding an optional string giving the encoding for the output XML string; if this parameter is missing, the output string will have the default encoding

    @return an XML string corresponding to the input data, with an XML header

    @throw MAKE-XML-ERROR An error occurred serializing the %Qore data to an XML string

    @see @ref serialization

    @since xml 1.3 as a replacement for deprecated camel-case makeXMLString() and makeFormattedXMLString()
 */
string make_xml(string key, hash h, __7_ int flags, __7_ string encoding);

//! serializes a hash into an XML string with an XML header
/** @par Code Flags:
    @ref RET_VALUE_ONLY

    @par Example:
    @code string xml = make_xml(hash); @endcode

    @param h a hash of data to serialize: the hash must have one top-level key and no more or an exception will be raised
    @param flags optional formatting flags; see @ref xml_generation_constants for more information
    @param encoding an optional string giving the encoding for the output XML string; if this parameter is missing, the output string will have the default encoding

    @return an XML string corresponding to the input data, with an XML header

    @throw MAKE-XML-STRING-PARAMETER-EXCEPTION the hash passed not not have a single top-level key (either has no keys or more than one)
    @throw MAKE-XML-ERROR An error occurred serializing the %Qore data to an XML string

    @see @ref serialization

    @since xml 1.3 as a replacement for deprecated camel-case makeXMLString() and makeFormattedXMLString()
 */
string make_xml(hash h, __7_ int flags, __7_ string encoding);

//! serializes a hash into an XML string with various options of formatting
/** @par Code Flags:
    @ref RET_VALUE_ONLY

    @par Example:
    @code string xml = make_xml(hash, opts); @endcode

    @param h a hash of data to serialize: the hash must have one top-level key and no more or an exception will be raised
    @param opts formatting and other serialization settings; see @ref xml_generation_opts for more information

    @return an XML string corresponding to the input data

    @throw MAKE-XML-STRING-PARAMETER-EXCEPTION the hash passed does not have a single top-level key (either has no keys or more than one)
    @throw MAKE-XML-OPTS-INVALID the opts hash passed is not valid; see @ref xml_generation_opts for more information
    @throw MAKE-XML-ERROR An error occurred serializing the %Qore data to an XML string

    @see @ref serialization

    @since xml 1.4
 */
string make_xml(hash h, hash opts);

//! serializes a hash into an XML string without whitespace formatting and without an XML header
/** @par Code Flags:
    @ref RET_VALUE_ONLY

    @par Example:
    @code string xml = make_xml_fragment(hash); @endcode

    @param h a hash of data to serialize: the hash can have any number of keys
    @param flags optional formatting flags; see @ref xml_generation_constants for more information
    @param encoding an optional string giving the encoding for the output XML string; if this parameter is missing, the output string will have the default encoding

    @return an XML string corresponding to the input data, without whitespace formatting and without an XML header

    @throw MAKE-XML-ERROR An error occurred serializing the %Qore data to an XML string

    @see @ref serialization

    @since xml 1.3 as a replacement for deprecated camel-case makeXMLFragment()
 */
string make_xml_fragment(hash h, __7_ int flags, __7_ string encoding);

//! Parses an XML string and returns a %Qore hash structure
/** If duplicate, out-of-order XML elements are found in the input string, they are deserialized to %Qore hash elements with the same name as the XML element but including a caret \c '^' and a numeric prefix to maintain the same key order in the %Qore hash as in the input XML string.

    This function should only be used when it is important to maintain the XML element order in the resulting %Qore data structure (for example, when the data must be re-serialized to an XML string and the element order within a subelement must be maintained), for example, when parsing and reserializing an OSX property list in XML format.   In all other cases, parseXMLAsData() should be used instead.

   @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

     @param xml the XML string to parse
    @param encoding an optional string giving the encoding for the output XML string; if this parameter is missing, all strings in the output hash will have the default encoding

    @return a %Qore hash structure corresponding to the XML input string

    @throw PARSE-XML-EXCEPTION Error parsing the XML string

    @see @ref serialization

    @deprecated use parse_xml(); camel-case function names were deprecated in xml 1.3
*/
hash parseXML(string xml, __7_ string encoding);

//! This is a variant that is basically a noop, included for backwards-compatibility for functions that ignored type errors in the calling parameters
/**@par Code Flags:
    @ref RUNTIME_NOOP, @ref DEPRECATED

*/
nothing parseXML();

//! Parses an XML string as data (does not necessarily preserve key order) and returns a %Qore hash structure
/** This function does not preserve hash order with out-of-order duplicate keys; all duplicate keys are collapsed to the same list.

    Note that data deserialized with this function may not be reserialized to an identical XML string to the input due to the fact that duplicate, out-of-order XML elements are collapsed into lists in the resulting %Qore hash, thereby losing the order in the original XML string.

    For a similar function preserving the order of keys in the XML in the resulting %Qore hash by generating %Qore hash element names with numeric suffixes, see parseXML().

   @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

     @param xml the XML string to parse
    @param encoding an optional string giving the encoding for the output XML string; if this parameter is missing, all strings in the output hash will have the default encoding

    @return a %Qore hash structure corresponding to the XML input string

    @throw PARSE-XML-EXCEPTION Error parsing the XML string

    @see
    - parseXML()
    - @ref serialization

    @deprecated use parse_xml(); camel-case function names were deprecated in xml 1.3
*/
hash parseXMLAsData(string xml, __7_ string encoding);

//! This is a variant that is basically a noop, included for backwards-compatibility for functions that ignored type errors in the calling parameters
/**@par Code Flags:
    @ref RUNTIME_NOOP, @ref DEPRECATED

*/
nothing parseXMLAsData();

//! Parses an XML string as data (does not preserve hash order with out-of-order duplicate keys: collapses all to the same list), validates the XML string against a RelaxNG schema string, and returns a %Qore hash structure
/** If any errors occur parsing the RelaxNG schema string, parsing the XML string, or validating the XML against the XSD, exceptions are thrown. If no encoding string argument is passed, then all strings in the resulting hash will be in UTF-8 encoding regardless of the input encoding of the XML string.

    Please note that data deserialized with this function may not be reserialized to an identical XML string to the input due to the fact that duplicate, out-of-order XML elements are collapsed into lists in the resulting %Qore hash, thereby losing the order in the original XML string.

    The availability of this function depends on the presence of libxml2's \c xmlTextReaderRelaxNGSetSchema() function when the xml module was compiled; for maximum portability check the constant @ref optionconstants "HAVE_PARSEXMLWITHRELAXNG" before running this function.

   @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

     @param xml the XML string to parse
    @param relaxng the RelaxNG schema string to use to validate the XML string
    @param encoding an optional string giving the string encoding of any strings output; if this parameter is missing, the any strings output in the output hash will have UTF-8 encoding

    @return a %Qore hash structure corresponding to the input

    @throw PARSE-XML-EXCEPTION error parsing the XML string
    @throw RELAXNG-SYNTAX-ERROR invalid RelaxNG string
    @throw RELAXNG-VALIDATION-ERROR the XML did not pass RelaxNG schema validation
    @throw MISSING-FEATURE-ERROR this exception is thrown when the function is not available; for maximum portability, check the constant @ref optionconstants "HAVE_PARSEXMLWITHRELAXNG" before calling this function

    @deprecated use parse_xml_with_relaxng(); camel-case function names were deprecated in xml 1.3
 */
hash parseXMLAsDataWithRelaxNG(string xml, string relaxng, __7_ string encoding);

//! Parses an XML string as data (does not preserve hash order with out-of-order duplicate keys: collapses all to the same list), validates the XML string against an XSD schema string, and returns a %Qore hash structure
/** If any errors occur parsing the XSD string, parsing the XML string, or validating the XML against the XSD, exceptions are thrown. If no encoding string argument is passed, then all strings in the resulting hash will be in UTF-8 encoding regardless of the input encoding of the XML string.

    Please note that data deserialized with this function may not be reserialized to an identical XML string to the input due to the fact that duplicate, out-of-order XML elements are collapsed into lists in the resulting %Qore hash, thereby losing the order in the original XML string.

    If any errors occur parsing the XSD string, parsing the XML string, or validating the XML against the XSD, exceptions are thrown.  If no encoding string argument is passed, then all strings in the resulting hash will be in UTF-8 encoding regardless of the input encoding of the XML string.

    The availability of this function depends on the presence of libxml2's \c xmlTextReaderSetSchema() function when the xml module was compiled; for maximum portability check the constant @ref optionconstants "HAVE_PARSEXMLWITHSCHEMA" before running this function.

   @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

     @param xml the XML string to parse
    @param xsd the XSD schema string to use to validate the XML string
    @param encoding an optional string giving the string encoding of any strings output; if this parameter is missing, the any strings output in the output hash will have UTF-8 encoding

    @return a %Qore hash structure corresponding to the input

    @throw PARSE-XML-EXCEPTION error parsing the XML string
    @throw XSD-SYNTAX-ERROR invalid XSD string
    @throw XSD-VALIDATION-ERROR the XML did not pass schema validation
    @throw MISSING-FEATURE-ERROR this exception is thrown when the function is not available; for maximum portability, check the constant @ref optionconstants "HAVE_PARSEXMLWITHSCHEMA" before calling this function

    @deprecated use parse_xml_with_schema(); camel-case function names were deprecated in xml 1.3
*/
hash parseXMLAsDataWithSchema(string xml, string xsd, __7_ string encoding);

//! Parses an XML string, validates the XML string against a RelaxNG schema string, and returns a %Qore hash structure
/** If any errors occur parsing the RelaxNG string, parsing the XML string, or validating the XML against the RelaxNG schema, exceptions are thrown. If no encoding string argument is passed, then all strings in the resulting hash will be in UTF-8 encoding regardless of the input encoding of the XML string.

    If duplicate, out-of-order XML elements are found in the input string, they are deserialized to %Qore hash elements with the same name as the XML element but including a caret "^" and a numeric prefix to maintain the same key order in the %Qore hash as in the input XML string.

    The availability of this function depends on the presence of libxml2's \c xmlTextReaderRelaxNGSetSchema() function when the xml module was compiled; for maximum portability check the constant @ref optionconstants "HAVE_PARSEXMLWITHRELAXNG" before running this function.

   @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

     @param xml the XML string to parse
    @param relaxng the RelaxNG schema string to use to validate the XML string
    @param encoding an optional string giving the string encoding of any strings output; if this parameter is missing, the any strings output in the output hash will have UTF-8 encoding

    @return a %Qore hash structure corresponding to the input

    @throw PARSE-XML-EXCEPTION error parsing the XML string
    @throw RELAXNG-SYNTAX-ERROR invalid RelaxNG string
    @throw RELAXNG-VALIDATION-ERROR the XML did not pass RelaxNG schema validation
    @throw MISSING-FEATURE-ERROR this exception is thrown when the function is not available; for maximum portability, check the constant @ref optionconstants "HAVE_PARSEXMLWITHRELAXNG" before calling this function

    @deprecated use parse_xml_with_relaxng(); camel-case function names were deprecated in xml 1.3
*/
hash parseXMLWithRelaxNG(string xml, string relaxng, __7_ string encoding);

//! Parses an XML string, validates the XML string against an XSD schema string, and returns a %Qore hash structure
/** If any errors occur parsing the XSD string, parsing the XML string, or validating the XML against the XSD, exceptions are thrown. If no encoding string argument is passed, then all strings in the resulting hash will be in UTF-8 encoding regardless of the input encoding of the XML string.

    If duplicate, out-of-order XML elements are found in the input string, they are deserialized to %Qore hash elements with the same name as the XML element but including a caret \c '^' and a numeric prefix to maintain the same key order in the %Qore hash as in the input XML string.

    The availability of this function depends on the presence of libxml2's \c xmlTextReaderSetSchema() function when the xml module was compiled; for maximum portability check the constant @ref optionconstants "HAVE_PARSEXMLWITHSCHEMA" before running this function.

   @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

     @param xml the XML string to parse
    @param xsd the XSD schema string to use to validate the XML string
    @param encoding an optional string giving the string encoding of any strings output; if this parameter is missing, the any strings output in the output hash will have UTF-8 encoding

    @return a %Qore hash structure corresponding to the input

    @throw PARSE-XML-EXCEPTION error parsing the XML string
    @throw XSD-SYNTAX-ERROR invalid XSD string
    @throw XSD-VALIDATION-ERROR the XML did not pass schema validation
    @throw MISSING-FEATURE-ERROR this exception is thrown when the function is not available; for maximum portability, check the constant @ref optionconstants "HAVE_PARSEXMLWITHSCHEMA" before calling this function

    @deprecated use parse_xml_with_schema(); camel-case function names were deprecated in xml 1.3
*/
hash parseXMLWithSchema(string xml, string xsd, __7_ string encoding);

//! Parses an XML string and returns a %Qore hash structure
/** @par Code Flags:
    @ref RET_VALUE_ONLY

    @par Example:
    @code hash h = parse_xml(xmlstr); @endcode

    @param xml the XML string to parse
    @param pflags XML parsing flags; see @ref xml_parsing_constants for more information
    @param encoding an optional string giving the encoding for the output XML string; if this parameter is missing, all strings in the output hash will have the default encoding

    @return a %Qore hash structure corresponding to the XML input string

    @throw PARSE-XML-EXCEPTION Error parsing the XML string

    @note the default is to parse XML as serialized data; to preserve key order in XML strings for higher fidelity to the original string when serializing back to XML, use the @ref XPF_PRESERVE_ORDER flag when parsing

    @note use the @ref XPF_ADD_COMMENTS to process XML comments and put them in hash as elements with '^comment^' key

    @see @ref serialization

    @since xml 1.3 as a replacement for deprecated camel-case parseXML() and parseXMLAsData()
*/
hash parse_xml(string xml, __7_ int pflags, __7_ string encoding);

//! Parses an XML string, validates the XML string against a DTD string, and returns a %Qore hash structure
/** @par Code Flags:
    @ref RET_VALUE_ONLY

    @par Example:
    @code hash h = parse_xml_with_dtd(xml, dtd); @endcode

    If any errors occur parsing the DTD string, parsing the XML string, or validating the XML against the DTD, exceptions are thrown. If no encoding string argument is passed, then all strings in the resulting hash will be in UTF-8 encoding regardless of the input encoding of the XML string.

    @param xml the XML string to parse
    @param dtd the DTD string to use to validate the XML string
    @param pflags XML parsing flags; see @ref xml_parsing_constants for more information
    @param encoding an optional string giving the string encoding of any strings output; if this parameter is missing, the any strings output in the output hash will have UTF-8 encoding

    @return a %Qore hash structure corresponding to the input

    @throw PARSE-XML-EXCEPTION error parsing the XML string
    @throw DTD-SYNTAX-ERROR invalid DTD string
    @throw DTD-VALIDATION-ERROR the DTD did not pass validation

    @see
    - parse_xml_with_schema()
    - parse_xml_with_relaxng()
    - @ref serialization

    @since xml 1.4
*/
hash parse_xml_with_dtd(string xml, string dtd, __7_ int pflags, __7_ string encoding);

//! Parses an XML string, validates the XML string against a RelaxNG schema string, and returns a %Qore hash structure
/** @par Code Flags:
    @ref RET_VALUE_ONLY

    @par Example:
    @code hash h = parse_xml_with_relaxng(xml, relaxng); @endcode

    If any errors occur parsing the RelaxNG string, parsing the XML string, or validating the XML against the RelaxNG schema, exceptions are thrown. If no encoding string argument is passed, then all strings in the resulting hash will be in UTF-8 encoding regardless of the input encoding of the XML string.

    The availability of this function depends on the presence of libxml2's \c xmlTextReaderRelaxNGSetSchema() function when the xml module was compiled; for maximum portability check the constant @ref optionconstants "HAVE_PARSEXMLWITHRELAXNG" before running this function.

    @param xml the XML string to parse
    @param relaxng the RelaxNG schema string to use to validate the XML string
    @param pflags XML parsing flags; see @ref xml_parsing_constants for more information
    @param encoding an optional string giving the string encoding of any strings output; if this parameter is missing, the any strings output in the output hash will have UTF-8 encoding

    @return a %Qore hash structure corresponding to the input

    @throw PARSE-XML-EXCEPTION error parsing the XML string
    @throw RELAXNG-SYNTAX-ERROR invalid RelaxNG string
    @throw RELAXNG-VALIDATION-ERROR the XML did not pass RelaxNG schema validation
    @throw MISSING-FEATURE-ERROR this exception is thrown when the function is not available; for maximum portability, check the constant @ref optionconstants "HAVE_PARSEXMLWITHRELAXNG" before calling this function

    @see
    - parse_xml_with_dtd()
    - parse_xml_with_schema()
    - @ref serialization
*/
hash parse_xml_with_relaxng(string xml, string relaxng, __7_ int pflags, __7_ string encoding);

//! Parses an XML string, validates the XML string against an XSD schema string, and returns a %Qore hash structure
/** @par Code Flags:
    @ref RET_VALUE_ONLY

    @par Example:
    @code hash h = parse_xml_with_schema(xml, xsd); @endcode

    If any errors occur parsing the XSD string, parsing the XML string, or validating the XML against the XSD, exceptions are thrown. If no encoding string argument is passed, then all strings in the resulting hash will be in UTF-8 encoding regardless of the input encoding of the XML string.

    The availability of this function depends on the presence of libxml2's \c xmlTextReaderSetSchema() function when the xml module was compiled; for maximum portability check the constant @ref optionconstants "HAVE_PARSEXMLWITHSCHEMA" before running this function.

    @param xml the XML string to parse
    @param xsd the XSD schema string to use to validate the XML string
    @param pflags XML parsing flags; see @ref xml_parsing_constants for more information
    @param encoding an optional string giving the string encoding of any strings output; if this parameter is missing, the any strings output in the output hash will have UTF-8 encoding

    @return a %Qore hash structure corresponding to the input

    @throw PARSE-XML-EXCEPTION error parsing the XML string
    @throw XSD-SYNTAX-ERROR invalid XSD string
    @throw XSD-VALIDATION-ERROR the XML did not pass schema validation
    @throw MISSING-FEATURE-ERROR this exception is thrown when the function is not available; for maximum portability, check the constant @ref optionconstants "HAVE_PARSEXMLWITHSCHEMA" before calling this function

    @see
    - parse_xml_with_dtd()
    - parse_xml_with_relaxng()
    - @ref serialization

    @since xml 1.3 as a replacement for deprecated camel-case parseXMLWithSchema() and parseXMLAsDataWithSchema()
*/
hash parse_xml_with_schema(string xml, string xsd, __7_ int pflags, __7_ string encoding);

//@}
}
}
//! Qore namespace
namespace Qore {
//! Xml namespace
namespace Xml {
/** @defgroup xmlrpc_functions XML-RPC Functions
//@{
 */

//! Helper function returning an XML value
/** This function returns the actual value of a complex data structure as returned by @ref parse_xml().
    When parsing XML attributes or comments (ex: @ref XPF_ADD_COMMENTS), then a hash is returned and the
    actual value of the XML appears under the \c "^value^" key.   This function makes it easier to handle
    this situation by returning the value given if the argument is not a hash or has no \c "^value^" key,
    otherwise, if so, it returns the value assigned to the \c "^value^" key.\n
    Note that only "^value^" is returned even when there are more values as result of @ref XPF_PRESERVE_ORDER flag.

   @par Code Flags:
    @ref CONSTANT

     @param n a simple value or a complex hash as returned by @ref parse_xml(); if the value is a hash with a
    \c "^value^" key, then the value of this key is processed, otherwise the argument itself is processed.
    If the resulting value is a list or a hash, then @ref nothing is returned, otherwise the value is returned.

    @par Example:
    @code
    string xml = "
        <elem attr="a1">
          <!-- comment -->
          My value
          <!-- comment 2 -->
          is split
        </elem>
    ";
    hash h = parse_xml(xml);
    string val = get_xml_value(xml.elem);
    @endcode

    @since xml 1.4
 */
auto get_xml_value(auto n);

//! Serializes the argument into an XML string in XML-RPC call format with whitespace formatting
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param method the method name for the XML-RPC call
    @param ... additional arguments are serialized according to the default XML-RPC serialization rules

    @return an XML string in XML-RPC call format in the default encoding, with whitespace formatting

    @throw XMLRPC-SERIALIZATION-ERROR empty member name in hash or cannot serialize type to XML-RPC (ex: object)

    @see @ref XMLRPC

    @deprecated use make_xmlrpc_call(); camel-case function names were deprecated in xml 1.3
*/
string makeFormattedXMLRPCCallString(string method,  ...);

//! Serializes the argument into an XML string in XML-RPC call format with whitespace formatting
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param method the method name for the XML-RPC call
    @param argv a single argument or a list of arguments to the call

    @return an XML string in XML-RPC call format in the default encoding, with whitespace formatting

    @throw XMLRPC-SERIALIZATION-ERROR empty member name in hash or cannot serialize type to XML-RPC (ex: object)

    @see @ref XMLRPC

    @deprecated use make_xmlrpc_call(); camel-case function names were deprecated in xml 1.3
*/
string makeFormattedXMLRPCCallStringArgs(string method, any argv);

//! Serializes the argument into an XML string in XML-RPC call format with whitespace formatting
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param encoding a string giving the output encoding for the resulting XML string
    @param method the method name for the XML-RPC call
    @param argv a single argument or a list of arguments to the call

    @return a string in XML-RPC call format in the encoding given by the first argument, with whitespace formatting

    @throw XMLRPC-SERIALIZATION-ERROR empty member name in hash or cannot serialize type to XML-RPC (ex: object)

    @see @ref XMLRPC

    @deprecated use make_xmlrpc_call(); camel-case function names were deprecated in xml 1.3
*/
string makeFormattedXMLRPCCallStringArgsWithEncoding(string encoding, string method, any argv);

//! Serializes the argument into an XML string in XML-RPC call format with whitespace formatting with an explicit encoding
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param encoding a string giving the output encoding for the resulting XML string
    @param method the method name for the XML-RPC call

    Additional arguments are serialized according to the default XML-RPC serialization rules

    @return an XML string in XML-RPC call format in the encoding given by the first argument, with whitespace formatting

    @throw XMLRPC-SERIALIZATION-ERROR empty member name in hash or cannot serialize type to XML-RPC (ex: object)

    @see @ref XMLRPC

    @deprecated use make_xmlrpc_call(); camel-case function names were deprecated in xml 1.3
*/
string makeFormattedXMLRPCCallStringWithEncoding(string encoding, string method,  ...);

//! Serializes the argument into an XML string in XML-RPC fault response format with whitespace formatting
/** @par Code Flags:
    @ref CONSTANT, @ref DEPRECATED

    @param code the fault code for the response; will be converted to an integer (stored under key \c 'faultCode' as an int)
    @param msg the fault message string; the encoding of this argument will define the output encoding of the fault string returned (stored under key \c 'faultString' as a string)

    @return a string in XML-RPC fault format in the same encoding as given by the msg argument, with whitespace formatting

    @see @ref XMLRPC

    @deprecated use make_xmlrpc_fault(); camel-case function names were deprecated in xml 1.3
*/
string makeFormattedXMLRPCFaultResponseString(softint code, string msg);

//! Serializes the argument into an XML string in XML-RPC fault response format with whitespace formatting with an explicit output encoding
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param encoding a string giving the output encoding for the resulting XML string
    @param code the fault code for the response; will be converted to an integer (stored under key \c 'faultCode' as an int)
    @param msg the fault message string (stored under key \c 'faultString' as a string)

    @return a string in XML-RPC fault format in the encoding given by the first argument, with whitespace formatting

    @throw ENCODING-CONVERSION-ERROR error converting message string to specified output encoding

    @see @ref XMLRPC

    @deprecated use make_xmlrpc_fault(); camel-case function names were deprecated in xml 1.3
*/
string makeFormattedXMLRPCFaultResponseStringWithEncoding(string encoding, softint code, string msg);

//! Serializes the arguments into an XML string formatted for an XML-RPC response with whitespace formatting
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param ... any top-level arguments to the function will be serialized as the top-level params of the response message

    @return a string in XML-RPC response format with whitespace formatting; the encoding of the resulting string will always be the default encoding
    @throw XMLRPC-SERIALIZATION-ERROR empty member name in hash or cannot serialize type to XML-RPC (ex: object)

    @see @ref XMLRPC

    @deprecated use make_xmlrpc_response(); camel-case function names were deprecated in xml 1.3
*/
string makeFormattedXMLRPCResponseString( ...);

//! Serializes the arguments into an XML string formatted for an XML-RPC response with whitespace formatting and with an explicit output encoding
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param encoding a string giving the output encoding for the resulting XML string
    @param ... any top-level arguments after the first argument will be serialized as the top-level params of the response message

    @return a string in XML-RPC response format with whitespace formatting; the encoding will be that given by the first argument

    @throw XMLRPC-SERIALIZATION-ERROR empty member name in hash or cannot serialize type to XML-RPC (ex: object)

    @see @ref XMLRPC

    @deprecated use make_xmlrpc_response(); camel-case function names were deprecated in xml 1.3
*/
string makeFormattedXMLRPCResponseStringWithEncoding(string encoding,  ...);

//! Serializes the arguments into an XML string in XML-RPC value format with whitespace formatting but without an XML header
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param value the value to serialize to XML-RPC format
    @param encoding an optional string giving the encoding for the output XML string; if this parameter is missing, the output string will have the default encoding

    @return if the \a value argument is NOTHING, then NOTHING is returned, otherwise an XML string in XML-RPC value format with whitespace formatting but without an XML header is returned

    @throw XMLRPC-SERIALIZATION-ERROR empty member name in hash or cannot serialize type to XML-RPC (ex: object)

    @see @ref XMLRPC

    @deprecated use make_xmlrpc_value(); camel-case function names were deprecated in xml 1.3
 */
__7_ string makeFormattedXMLRPCValueString(any value, __7_ string encoding);

//! Serializes the argument into an XML string in XML-RPC call format without whitespace formatting
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param method the method name for the XML-RPC call
    Additional arguments are serialized according to the default XML-RPC serialization rules

    @return an XML string in XML-RPC call format in the default encoding, without whitespace formatting

    @throw XMLRPC-SERIALIZATION-ERROR empty member name in hash or cannot serialize type to XML-RPC (ex: object)

    @see @ref XMLRPC

    @deprecated use make_xmlrpc_call(); camel-case function names were deprecated in xml 1.3
*/
string makeXMLRPCCallString(string method,  ...);

//! Serializes the argument into an XML string in XML-RPC call format without whitespace formatting
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param method the method name for the XML-RPC call
    @param args a single argument or a list of arguments to the call

    @return an XML string in XML-RPC call format in the default encoding, without whitespace formatting

    @throw XMLRPC-SERIALIZATION-ERROR empty member name in hash or cannot serialize type to XML-RPC (ex: object)

    @see @ref XMLRPC

    @deprecated use make_xmlrpc_call(); camel-case function names were deprecated in xml 1.3
*/
string makeXMLRPCCallStringArgs(string method, any args);

//! Serializes the argument into an XML string in XML-RPC call format without whitespace formatting with an explicit encoding
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param encoding a string giving the output encoding for the resulting XML string
    @param method the method name for the XML-RPC call
    @param args a single argument or a list of arguments to the call

    @return a string in XML-RPC call format in the encoding given by the first argument, without whitespace formatting

    @throw XMLRPC-SERIALIZATION-ERROR empty member name in hash or cannot serialize type to XML-RPC (ex: object)

    @see @ref XMLRPC

    @deprecated use make_xmlrpc_call(); camel-case function names were deprecated in xml 1.3
*/
string makeXMLRPCCallStringArgsWithEncoding(string encoding, string method, any args);

//! Serializes the argument into an XML string in XML-RPC call format without whitespace formatting with an explicit encoding
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param encoding a string giving the output encoding for the resulting XML string
    @param method the method name for the XML-RPC call

    Additional arguments are serialized according to the default XML-RPC serialization rules

    @return an XML string in XML-RPC call format in the encoding given by the first argument, without whitespace formatting

    @throw XMLRPC-SERIALIZATION-ERROR empty member name in hash or cannot serialize type to XML-RPC (ex: object)

    @see @ref XMLRPC

    @deprecated use make_xmlrpc_call(); camel-case function names were deprecated in xml 1.3
*/
string makeXMLRPCCallStringWithEncoding(string encoding, string method,  ...);

//! Serializes the argument into an XML string in XML-RPC fault response format without whitespace formatting
/** @par Code Flags:
    @ref CONSTANT, @ref DEPRECATED

    @param code the fault code for the response; will be converted to an integer (stored under key \c 'faultCode' as an int)
    @param msg the fault message string; the encoding of this argument will define the output encoding of the fault string returned (stored under key \c 'faultString' as a string)

    @return a string in XML-RPC fault format in the same encoding as given by the msg argument, without whitespace formatting

    @see @ref XMLRPC

    @deprecated use make_xmlrpc_fault(); camel-case function names were deprecated in xml 1.3
*/
string makeXMLRPCFaultResponseString(softint code, string msg);

//! Serializes the argument into an XML string in XML-RPC fault response format without whitespace formatting with an explicit output encoding
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param encoding a string giving the output encoding for the resulting XML string
    @param code the fault code for the response; will be converted to an integer (stored under key \c 'faultCode' as an int)
    @param msg the fault message string (stored under key \c 'faultString' as a string)

    @return a string in XML-RPC fault format in the encoding given by the first argument, without whitespace formatting

    @throw ENCODING-CONVERSION-ERROR error converting message string to specified output encoding

    @see @ref XMLRPC

    @deprecated use make_xmlrpc_fault(); camel-case function names were deprecated in xml 1.3
*/
string makeXMLRPCFaultResponseStringWithEncoding(string encoding, softint code, string msg);

//! Serializes the arguments into an XML string formatted for an XML-RPC response without whitespace formatting
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

Any top-level arguments to the function will be serialized as the top-level params of the response message

    @return a string in XML-RPC response format; the encoding of the resulting string will always be the default encoding

    @throw XMLRPC-SERIALIZATION-ERROR empty member name in hash or cannot serialize type to XML-RPC (ex: object)

    @see @ref XMLRPC

    @deprecated use make_xmlrpc_response(); camel-case function names were deprecated in xml 1.3
*/
string makeXMLRPCResponseString( ...);

//! Serializes the arguments into an XML string formatted for an XML-RPC response without whitespace formatting and with an explicit output encoding
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param encoding a string giving the output encoding for the resulting XML string
    Any top-level arguments after the first argument will be serialized as the top-level params of the response message
    @return a string in XML-RPC response format; the encoding will be that given by the first argument

    @throw XMLRPC-SERIALIZATION-ERROR empty member name in hash or cannot serialize type to XML-RPC (ex: object)

    @see @ref XMLRPC

    @deprecated use make_xmlrpc_response(); camel-case function names were deprecated in xml 1.3
*/
string makeXMLRPCResponseStringWithEncoding(string encoding,  ...);

//! Serializes the arguments into an XML string in XML-RPC value format without whitespace formatting and without an XML header
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param value the value to serialize to XML-RPC format
    @param encoding an optional string giving the encoding for the output XML string; if this parameter is missing, the output string will have the default encoding

    @return if the \a value argument is NOTHING, then NOTHING is returned, otherwise an XML string in XML-RPC value format without whitespace formatting and without an XML header is returned

    @throw XMLRPC-SERIALIZATION-ERROR empty member name in hash or cannot serialize type to XML-RPC (ex: object)

    @see @ref XMLRPC

    @deprecated use make_xmlrpc_value(); camel-case function names were deprecated in xml 1.3
 */
__7_ string makeXMLRPCValueString(any value, __7_ string encoding);

//! Serializes the argument into an XML string in XML-RPC call format
/** @par Code Flags:
    @ref RET_VALUE_ONLY

    @param method the method name for the XML-RPC call
    @param args a single argument or a list of arguments to the XML-RPC call
    @param flags optional formatting flags; see @ref xml_generation_constants for more information
    @param encoding an optional string giving the encoding for the output XML string; if this parameter is missing, the output string will have the default encoding

    @return an XML string in XML-RPC call format in the default encoding, without whitespace formatting

    @throw XMLRPC-SERIALIZATION-ERROR empty member name in hash or cannot serialize type to XML-RPC (ex: object)

    @par Example:
    @code string xmlcall = make_xmlrpc_call("omq.system.start-workflow", hash); @endcode

    @see @ref XMLRPC

    @since xml 1.3 as a replacement for deprecated camel-case functions:
    - makeFormattedXMLRPCCallString()
    - makeFormattedXMLRPCCallStringArgs()
    - makeFormattedXMLRPCCallStringArgsWithEncoding()
    - makeFormattedXMLRPCCallStringWithEncoding()
    - makeXMLRPCCallString()
    - makeXMLRPCCallStringArgs()
    - makeXMLRPCCallStringArgsWithEncoding()
    - makeXMLRPCCallStringWithEncoding()
*/
string make_xmlrpc_call(string method, any args, __7_ int flags, __7_ string encoding);

//! Serializes the argument into an XML string in XML-RPC fault response format without whitespace formatting
/** @par Code Flags:
    @ref CONSTANT

    @param code the fault code for the response; will be converted to an integer (stored under key \c 'faultCode' as an int)
    @param msg the fault message string; the encoding of this argument will define the output encoding of the fault string returned (stored under key \c 'faultString' as a string)
    @param flags optional formatting flags; see @ref xml_generation_constants for more information
    @param encoding an optional string giving the encoding for the output XML string; if this parameter is missing, the output string will have the default encoding

    @return a string in XML-RPC fault format in the same encoding as given by the msg argument, without whitespace formatting

    @par Example:
    @code string response = make_xmlrpc_fault(500, errmsg); @endcode

    @see @ref XMLRPC

    @since xml 1.3 as a replacement for deprecated camel-case functions:
    - makeXMLRPCFaultResponseString()
    - makeXMLRPCFaultResponseStringWithEncoding()
    - makeFormattedXMLRPCFaultResponseString()
    - makeFormattedXMLRPCFaultResponseStringWithEncoding()
*/
string make_xmlrpc_fault(softint code, string msg, __7_ int flags, __7_ string encoding);

//! Serializes the arguments into an XML string formatted for an XML-RPC response without whitespace formatting
/** Any top-level arguments to the function will be serialized as the top-level params of the response message

   @par Code Flags:
    @ref RET_VALUE_ONLY

     @param call_args a single argument or a list of arguments that represent the XML-RPC response
    @param flags optional formatting flags; see @ref xml_generation_constants for more information
    @param encoding an optional string giving the encoding for the output XML string; if this parameter is missing, the output string will have the default encoding

    @return a string in XML-RPC response format; the encoding of the resulting string will always be the default encoding

    @throw XMLRPC-SERIALIZATION-ERROR empty member name in hash or cannot serialize type to XML-RPC (ex: object)

    @par Example:
    @code string response = make_xmlrpc_response(answer); @endcode

    @see @ref XMLRPC

    @since xml 1.3 as a replacement for deprecated camel-case functions:
    - makeFormattedXMLRPCResponseString()
    - makeFormattedXMLRPCResponseStringWithEncoding()
    - makeXMLRPCResponseString()
    - makeXMLRPCResponseStringWithEncoding()
*/
string make_xmlrpc_response(any call_args, __7_ int flags, __7_ string encoding);

//! Serializes the arguments into an XML string in XML-RPC value format without whitespace formatting and without an XML header
/** @par Code Flags:
    @ref RET_VALUE_ONLY

    @param value the value to serialize to XML-RPC format
    @param encoding an optional string giving the encoding for the output XML string; if this parameter is missing, the output string will have the default encoding
    @param flags optional formatting flags; see @ref xml_generation_constants for more information
    @param encoding an optional string giving the encoding for the output XML string; if this parameter is missing, the output string will have the default encoding

    @return if the \a value argument is NOTHING, then NOTHING is returned, otherwise an XML string in XML-RPC value format without whitespace formatting and without an XML header is returned

    @throw XMLRPC-SERIALIZATION-ERROR empty member name in hash or cannot serialize type to XML-RPC (ex: object)

    @see @ref XMLRPC

    @since xml 1.3 as a replacement for deprecated camel-case makeXMLRPCValueString() and makeFormattedXMLRPCValueString()
 */
__7_ string make_xmlrpc_value(any value, __7_ int flags, __7_ string encoding);

//! Deserializies an XML-RPC call string, returning a %Qore data structure representing the call information
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param xml the XML string in XML-RPC call format to deserialize
    @param encoding an optional string giving the string encoding of any strings output; if this parameter is missing, the any strings output in the output hash will have the default encoding

    @return a hash representing the XML-RPC call with the following keys:
    - \c methodName: the name of the method being called
    - \c params: the arguments to the method

    @throw PARSE-XMLRPC-CALL-ERROR missing 'methodCall' or 'methodName' element or other syntax error
    @throw PARSE-XMLRPC-ERROR syntax error parsing XML-RPC string

    @see @ref XMLRPC

    @deprecated use parse_xmlrpc_call(); camel-case function names were deprecated in xml 1.3
 */
hash parseXMLRPCCall(string xml, __7_ string encoding);

//! This is a variant that is basically a noop, included for backwards-compatibility for functions that ignored type errors in the calling parameters
/**@par Code Flags:
    @ref RUNTIME_NOOP, @ref DEPRECATED

*/
nothing parseXMLRPCCall();

//! Deserializies an XML-RPC response string, returning a %Qore data structure representing the response information
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param xml the XML string in XML-RPC call format to deserialize
    @param encoding an optional string giving the string encoding of any strings output; if this parameter is missing, the any strings output in the output hash will have the default encoding

    @return a hash with one of the following keys:
    - \c fault: a hash describing a fault response
    - \c params: a hash describing a normal, non-fault response

    @throw PARSE-XMLRPC-RESPONSE-ERROR missing required element or other syntax error
    @throw PARSE-XMLRPC-ERROR syntax error parsing XML-RPC string

    @see @ref XMLRPC

    @deprecated use parse_xmlrpc_response(); camel-case function names were deprecated in xml 1.3
 */
hash parseXMLRPCResponse(string xml, __7_ string encoding);

//! This is a variant that is basically a noop, included for backwards-compatibility for functions that ignored type errors in the calling parameters
/**@par Code Flags:
    @ref RUNTIME_NOOP, @ref DEPRECATED

*/
nothing parseXMLRPCResponse();

//! Deserializies an XML-RPC value string and returns a %Qore data structure representing the information
/** @par Code Flags:
    @ref RET_VALUE_ONLY, @ref DEPRECATED

    @param xml the XML string in XML-RPC value format to deserialize
    @param encoding an optional string giving the string encoding of any strings output; if this parameter is missing, the any strings output in the output hash will have the default encoding

    @return the %Qore value corresponding to the XML-RPC value string

    @throw PARSE-XMLRPC-ERROR syntax error parsing XML-RPC string

    @see @ref XMLRPC

    @deprecated use parse_xmlrpc_value(); camel-case function names were deprecated in xml 1.3
 */
any parseXMLRPCValue(string xml, __7_ string encoding);

//! This is a variant that is basically a noop, included for backwards-compatibility for functions that ignored type errors in the calling parameters
/**@par Code Flags:
    @ref RUNTIME_NOOP, @ref DEPRECATED

*/
nothing parseXMLRPCValue();

//! Deserializies an XML-RPC call string, returning a %Qore data structure representing the call information
/** @par Code Flags:
    @ref RET_VALUE_ONLY

    @param xml the XML string in XML-RPC call format to deserialize
    @param encoding an optional string giving the string encoding of any strings output; if this parameter is missing, the any strings output in the output hash will have the default encoding

    @return a hash representing the XML-RPC call with the following keys:
    - \c methodName: the name of the method being called
    - \c params: the arguments to the method

    @throw PARSE-XMLRPC-CALL-ERROR missing 'methodCall' or 'methodName' element or other syntax error
    @throw PARSE-XMLRPC-ERROR syntax error parsing XML-RPC string

    @par Example:
    @code hash h = parse_xmlrpc_call(xml); @endcode

    @see @ref XMLRPC

    @since xml 1.3 as a replacement for deprecated camel-case parseXMLRPCCall()
 */
hash parse_xmlrpc_call(string xml, __7_ string encoding);

//! Deserializies an XML-RPC response string, returning a %Qore data structure representing the response information
/** @par Code Flags:
    @ref RET_VALUE_ONLY

    @param xml the XML string in XML-RPC call format to deserialize
    @param encoding an optional string giving the string encoding of any strings output; if this parameter is missing, the any strings output in the output hash will have the default encoding

    @return a hash with one of the following keys:
    - \c fault: a hash describing a fault response
    - \c params: a hash describing a normal, non-fault response

    @throw PARSE-XMLRPC-RESPONSE-ERROR missing required element or other syntax error
    @throw PARSE-XMLRPC-ERROR syntax error parsing XML-RPC string

    @par Example:
    @code hash h = parse_xmlrpc_response(xml); @endcode

    @see @ref XMLRPC

    @since xml 1.3 as a replacement for deprecated camel-case parseXMLRPCResponse()
 */
hash parse_xmlrpc_response(string xml, __7_ string encoding);

//! Deserializies an XML-RPC value string and returns a %Qore data structure representing the information
/** @par Code Flags:
    @ref RET_VALUE_ONLY

    @param xml the XML string in XML-RPC value format to deserialize
    @param encoding an optional string giving the string encoding of any strings output; if this parameter is missing, the any strings output in the output hash will have the default encoding

    @return the %Qore value corresponding to the XML-RPC value string

    @throw PARSE-XMLRPC-ERROR syntax error parsing XML-RPC string

    @par Example:
    @code any data = parse_xmlrpc_value(xml); @endcode

    @see @ref XMLRPC

    @since xml 1.3 as a replacement for deprecated camel-case parseXMLRPCValue()
 */
any parse_xmlrpc_value(string xml, __7_ string encoding);

//@}
}
}
//! Qore namespace
namespace Qore {
//! Xml namespace
namespace Xml {
/** @defgroup xml_generation_constants XML Generation Constants
    The constants in this group can be combined with @ref bitwise_or_operator "binary or" to affect the output of generated XML
//@{
 */

//! use whitespace formatting including line breaks to make generated XML more readable
    const XGF_ADD_FORMATTING = XGF_ADD_FORMATTING;
//! no flags; standard XML generation without whitespace formatting and characters without unicode character references
    const XGF_NONE = XGF_NONE;
//! use numeric character references instead of native characters for non-ascii character when generating XML
/** note that character references are not applied to CDATA when serializing XML
    (keys that begin with \c "^cdata" are serialized as cdata.
 */
    const XGF_USE_NUMERIC_REFS = XGF_USE_NUMERIC_REFS;
/** @} */
}
}
//! Qore namespace
namespace Qore {
//! Xml namespace
namespace Xml {
/** @defgroup xml_parsing_constants XML Parsing Constants
    The constants in this group can be combined with @ref bitwise_or_operator "binary or" to affect the output of
    parsed XML
//@{
 */

//! decode XML comments
/** If this option is given, then parse_xml() creates hash elements with \c'^comment^' keys corresponding to comments
    found in the XML
 */
    const XPF_ADD_COMMENTS = XPF_ADD_COMMENTS;
//! no flags; standard XML parsing without character reference conversion
    const XPF_NONE = XPF_NONE;
//! decode XML as data; do not try to preserve element order by re-writing hash keys
/** If this option is given and duplicate, out-of-order XML elements are found in the input string, they are
    deserialized to %Qore hash elements with the same name as the XML element but including a suffix with caret \c '^'
    and a prefix to maintain the same key order in the %Qore hash as in the input XML string (ex: \c "key^1").

    In this case, if the data structure is reserialized with make_xml(), an XML string corresponding to the original
    input will be generated.

    This option should only be used when it is important to maintain the XML element order in the resulting %Qore data
    structure (for example, when the data must be re-serialized to an XML string and the element order within a
    subelement must be maintained), for example, when parsing and reserializing an OSX property list in XML format.
 */
    const XPF_PRESERVE_ORDER = XPF_PRESERVE_ORDER;
//! strip namespace prefixes from element names when parsing
/** @since xml 1.4
*/
    const XPF_STRIP_NS_PREFIXES = XPF_STRIP_NS_PREFIXES;
/** @} */
}
}
