/** Java wrapper for the %Qore QUnit::Test class
 *
 */
package org.qore.lang.qunit;

// jni module imports
import org.qore.jni.QoreJavaApi;
import org.qore.jni.QoreObject;
import org.qore.jni.QoreObjectWrapper;

// qore lang imports
import org.qore.lang.qunit.*;

// java imports
import java.math.BigDecimal;

//! Base class (wrapping the %Qore QUnit::Test class) representing a simple test, implements an implicit main() method and all utility methods for testing
/** @note Loads and initializes the Qore library and the jni module in static initialization if necessary

    @deprecated Use @ref jni_dynamic_import_qore_in_java "dynamic imports" instead:
    <tt>import qoremod.QUnit.Test;</tt>
*/
@Deprecated
public class Test extends QoreObjectWrapper {
    // static initialization
    static {
        // loads and initializes the Qore library and the jni module (if necessary) and loads the \c QUnit module
        try {
            QoreJavaApi.initQore();
            QoreJavaApi.callFunction("load_module", "QUnit");
        } catch (Throwable e) {
            throw new ExceptionInInitializerError(e);
        }
    }

    //! creates the object based on the given Qore object
    protected Test(QoreObject obj) throws Throwable {
        super(obj);
    }

    //! creates the object and sets the name of the test
    public Test(String name, String version) throws Throwable {
        super(QoreJavaApi.newObjectSave("QUnit::Test", name, version));
    }

    //! global setup; will be called once before tests are run
    public void globalSetUp() throws Throwable {
        obj.callMethod("globalSetUp");
    }

    //! global tear down; will be called once after all tests are run
    public void globalTearDown() throws Throwable {
        obj.callMethod("globalTearDown");
    }

    //! Prototype method for setting up test environment. It will be called for each test individually.
    public void setUp() throws Throwable {
        obj.callMethod("setUp");
    }

    //! Prototype method for cleaning up test environemnt. It will be called after each test has executed.
    public void tearDown() throws Throwable {
        obj.callMethod("tearDown");
    }

    //! facade for the system environment variables
    public String getEnv(String key, String def) throws Throwable {
        return (String)obj.callMethod("getEnv", key, def);
    }

    //! adds a test case to run
    /** @par Example:
        @code{.java}
addTestCase("MyTest", \myTest());
        @endcode

        @param name the name of the test case
        @param call the code to call that executes the test case
    */
    public void addTestCase(String name, TestCode call) throws Throwable {
        obj.callMethod("addTestCase", name, call);
    }

    //! adds a test case to run
    /** @par Example:
        @code{.java}
addTestCase("MyTest", \myTest());
        @endcode

        @param name the name of the test case
        @param call the code to call that executes the test case
        @param args any optional arguments to the test case call
    */
    public void addTestCase(String name, TestCodeArgs call, Object... args) throws Throwable {
        obj.callMethod("addTestCase", name, call, args);
    }

    //! Tests a value for equality to an expected value
    /** @par Example:
        @code{.java}
testAssertionValue("date > operator", now() > (now() - 1D), True);
        @endcode

        @param name the name or description of the assertion
        @param actual the value generated by the test
        @param expected the expected value

        @return the \a value argument

        @note make sure and use testAssertion() for any calls that could throw an exception
     */
    public Object testAssertionValue(String name, Object actual, Object expected) throws Throwable {
        return obj.callMethod("testAssertionValue", name, actual, expected);
    }

    //! Tests a value for equality to an expected value of number type
    /**

        @param name the name or description of the assertion
        @param actual the value generated by the test
        @param expected the expected value
        @param epsilon accepted difference

        @return the \a actual argument

        @note make sure and use testAssertion() for any calls that could throw an exception
     */
    public Object testAssertionValue(String name, BigDecimal actual, BigDecimal expected, BigDecimal epsilon) throws Throwable {
        return obj.callMethod("testAssertionValue", name, actual, expected, epsilon);
    }

    //! Tests a value for equality to an expected value of number type
    /**

        @param name the name or description of the assertion
        @param actual the value generated by the test
        @param expected the expected value

        @return the \a actual argument

        @note make sure and use testAssertion() for any calls that could throw an exception
     */
    public Object testAssertionValue(String name, BigDecimal actual, BigDecimal expected) throws Throwable {
        return obj.callMethod("testAssertionValue", name, actual, expected);
    }

    //! Tests a value for equality to an expected value of float type
    /**

        @param name the name or description of the assertion
        @param actual the value generated by the test
        @param expected the expected value
        @param epsilon accepted difference

        @return the \a actual argument

        @note make sure and use testAssertion() for any calls that could throw an exception
     */
    public Object testAssertionValue(String name, float actual, float expected, float epsilon) throws Throwable {
        return obj.callMethod("testAssertionValue", name, actual, expected, epsilon);
    }

    //! Tests a value for equality to an expected value of float type
    /**

        @param name the name or description of the assertion
        @param actual the value generated by the test
        @param expected the expected value
        @param epsilon accepted difference

        @return the \a actual argument

        @note make sure and use testAssertion() for any calls that could throw an exception
     */
    public Object testAssertionValue(String name, float actual, float expected) throws Throwable {
        return obj.callMethod("testAssertionValue", name, actual, expected);
    }

    //! Tests that a test value passes a regular expression match
    /** @par Example:
        @code{.java}
assertRegex("regex string", string_method());
        @endcode

        @param regex_pattern the regular expression pattern string that the test value should match
        @param actual the value generated by the test
        @param name the name or description of the assertion
     */
    public void assertRegex(String regex_pattern, String actual, String name) throws Throwable {
        obj.callMethod("assertRegex", regex_pattern, actual, name);
    }

    //! Tests that a test value passes a regular expression match
    /** @par Example:
        @code{.java}
assertRegex("regex string", string_method());
        @endcode

        @param regex_pattern the regular expression pattern string that the test value should match
        @param actual the value generated by the test
        @param name the name or description of the assertion
     */
    public void assertRegex(String regex_pattern, String actual) throws Throwable {
        obj.callMethod("assertRegex", regex_pattern, actual);
    }

    //! Tests that a test value does not pass a regular expression match
    /** @par Example:
        @code{.java}
assertNRegex("regex string", string_method());
        @endcode

        @param regex_pattern the regular expression pattern string that the test value should not match
        @param actual the value generated by the test
        @param name the name or description of the assertion
     */
    public void assertNRegex(String regex_pattern, String actual, String name) throws Throwable {
        obj.callMethod("assertNRegex", regex_pattern, actual, name);
    }

    //! Tests for no value
    /** @par Example:
        @code{.java}
assertNothing(method_that_should_not_return_a value());
        @endcode

        @param actual the value generated by the test
        @param name the name or description of the assertion
     */
    public void assertNothing(Object actual, String name) throws Throwable {
        obj.callMethod("assertNothing", actual, name);
    }

    //! Tests for no value
    /** @par Example:
        @code{.java}
assertNothing(method_that_should_not_return_a value());
        @endcode

        @param actual the value generated by the test
     */
    public void assertNothing(Object actual) throws Throwable {
        obj.callMethod("assertNothing", actual);
    }

    //! Tests a value for equality to an expected value with soft comparisons (types may differ)
    /** @par Example:
        @code{.java}
assertEqSoft("5", method_that_should_return_five());
        @endcode

        @param expected the expected value
        @param actual the value generated by the test
        @param name the name or description of the assertion
     */
    public void assertEqSoft(Object expected, Object actual, String name) throws Throwable {
        obj.callMethod("assertEqSoft", expected, actual, name);
    }

    //! Tests a value for equality to an expected value with soft comparisons (types may differ)
    /** @par Example:
        @code{.java}
assertEqSoft("5", method_that_should_return_five());
        @endcode

        @param expected the expected value
        @param actual the value generated by the test
     */
    public void assertEqSoft(Object expected, Object actual) throws Throwable {
        obj.callMethod("assertEqSoft", expected, actual);
    }

    //! Tests a value for inequality to an expected value with soft comparisons (types may differ)
    /** @par Example:
        @code{.java}
assertNeqSoft("5", method_that_should_not_return_five());
        @endcode

        @param expected the value that should not be equal to the test value with a soft comparison
        @param actual the value generated by the test
        @param name the name or description of the assertion
     */
    public void assertNeqSoft(Object expected, Object actual, String name) throws Throwable {
        obj.callMethod("assertNeqSoft", expected, actual, name);
    }

    //! Tests a value for inequality to an expected value with soft comparisons (types may differ)
    /** @par Example:
        @code{.java}
assertNeqSoft("5", method_that_should_not_return_five());
        @endcode

        @param expected the value that should not be equal to the test value with a soft comparison
        @param actual the value generated by the test
     */
    public void assertNeqSoft(Object expected, Object actual) throws Throwable {
        obj.callMethod("assertNeqSoft", expected, actual);
    }

    //! Tests a value for equality to an expected value with hard comparisons (types and values must be identical)
    /** @par Example:
        @code{.java}
assertEq(5, method_that_should_return_five());
        @endcode

        @param expected the expected value
        @param actual the value generated by the test
        @param name the name or description of the assertion
     */
    public void assertEq(Object expected, Object actual, String name) throws Throwable {
        obj.callMethod("assertEq", expected, actual, name);
    }

    //! Tests a value for equality to an expected value with hard comparisons (types and values must be identical)
    /** @par Example:
        @code{.java}
assertEq(5, method_that_should_return_five());
        @endcode

        @param expected the expected value
        @param actual the value generated by the test
     */
    public void assertEq(Object expected, Object actual) throws Throwable {
        obj.callMethod("assertEq", expected, actual);
    }

    //! Tests a value for inequality to an expected value with hard comparisons (types and values must be identical)
    /** @par Example:
        @code{.java}
assertNeq(5, method_that_should_return_five());
        @endcode

        @param expected the value that should not be equal to the test value with a hard comparison
        @param actual the value generated by the test
        @param name the name or description of the assertion
     */
    public void assertNeq(Object expected, Object actual, String name) throws Throwable {
        obj.callMethod("assertNeq", expected, actual, name);
    }

    //! Tests a value for inequality to an expected value with hard comparisons (types and values must be identical)
    /** @par Example:
        @code{.java}
assertNeq(5, method_that_should_return_five());
        @endcode

        @param expected the value that should not be equal to the test value with a hard comparison
        @param actual the value generated by the test
     */
    public void assertNeq(Object expected, Object actual) throws Throwable {
        obj.callMethod("assertNeq", expected, actual);
    }

    //! Tests a float value for equality to an expected value with an allowed error
    /** @par Example:
        @code{.java}
assertFloatEq(2.5, 2.50001, 0.001);
        @endcode

        @param expected the expected value
        @param actual the value generated by the test
        @param epsilon the allowed error
        @param name the name or description of the assertion
     */
    public void assertFloatEq(float expected, float actual, float epsilon, String name) throws Throwable {
        obj.callMethod("assertFloatEq", expected, actual, epsilon, name);
    }

    //! Tests a float value for equality to an expected value with an allowed error
    /** @par Example:
        @code{.java}
assertFloatEq(2.5, 2.50001, 0.001);
        @endcode

        @param expected the expected value
        @param actual the value generated by the test
        @param epsilon the allowed error
     */
    public void assertFloatEq(float expected, float actual, float epsilon) throws Throwable {
        obj.callMethod("assertFloatEq", expected, actual, epsilon);
    }

    //! Tests a float value for equality to an expected value with an allowed error
    /** @par Example:
        @code{.java}
assertFloatEq(2.5, 2.50001);
        @endcode

        @param expected the expected value
        @param actual the value generated by the test
     */
    public void assertFloatEq(float expected, float actual) throws Throwable {
        obj.callMethod("assertFloatEq", expected, actual);
    }

    //! Tests a number value for equality to an expected value with an allowed error
    /** @par Example:
        @code{.java}
assertNumberEq(2.5n, 2.50001n, 0.001n);
        @endcode

        @param expected the expected value
        @param actual the value generated by the test
        @param epsilon the allowed error
        @param name the name or description of the assertion
     */
    public void assertNumberEq(BigDecimal expected, BigDecimal actual, BigDecimal epsilon, String name) throws Throwable {
        obj.callMethod("assertNumberEq", expected, actual, epsilon, name);
    }

    //! Tests a number value for equality to an expected value with an allowed error
    /** @par Example:
        @code{.java}
assertNumberEq(2.5n, 2.50001n, 0.001n);
        @endcode

        @param expected the expected value
        @param actual the value generated by the test
        @param epsilon the allowed error
        @param name the name or description of the assertion
     */
    public void assertNumberEq(BigDecimal expected, BigDecimal actual, BigDecimal epsilon) throws Throwable {
        obj.callMethod("assertNumberEq", expected, actual, epsilon);
    }

    //! Tests a number value for equality to an expected value with an allowed error
    /** @par Example:
        @code{.java}
assertNumberEq(2.5n, 2.50001n);
        @endcode

        @param expected the expected value
        @param actual the value generated by the test
     */
    public void assertNumberEq(BigDecimal expected, BigDecimal actual) throws Throwable {
        obj.callMethod("assertNumberEq", expected, actual);
    }

    //! Tests that a value is greater than an expected value with soft comparisons (types may differ)
    /** @par Example:
        @code{.java}
assertGtSoft("5", method_that_should_rturn_a_value_greater_than_five());
        @endcode

        @param expected a value that should be less than \a actual
        @param actual the value generated by the test
        @param name the name or description of the assertion
     */
    public void assertGtSoft(Object expected, Object actual, String name) throws Throwable {
        obj.callMethod("assertGtSoft", expected, actual, name);
    }

    //! Tests that a value is greater than an expected value with soft comparisons (types may differ)
    /** @par Example:
        @code{.java}
assertGtSoft("5", method_that_should_rturn_a_value_greater_than_five());
        @endcode

        @param expected a value that should be less than \a actual
        @param actual the value generated by the test
     */
    public void assertGtSoft(Object expected, Object actual) throws Throwable {
        obj.callMethod("assertGtSoft", expected, actual);
    }

    //! Tests that a value is greater than an expected value with hard comparisons (types may not differ)
    /** @par Example:
        @code{.java}
assertGt(5, method_that_should_rturn_a_value_greater_than_five());
        @endcode

        @param expected a value that should be less than \a actual
        @param actual the value generated by the test
        @param name the name or description of the assertion
     */
    public void assertGt(Object expected, Object actual, String name) throws Throwable {
        obj.callMethod("assertGt", expected, actual, name);
    }

    //! Tests that a value is greater than an expected value with hard comparisons (types may not differ)
    /** @par Example:
        @code{.java}
assertGt(5, method_that_should_rturn_a_value_greater_than_five());
        @endcode

        @param expected a value that should be less than \a actual
        @param actual the value generated by the test
     */
    public void assertGt(Object expected, Object actual) throws Throwable {
        obj.callMethod("assertGt", expected, actual);
    }

    //! Tests that a value is greater than or equal to than an expected value with soft comparisons (types may differ)
    /** @par Example:
        @code{.java}
assertGeSoft("5", method_that_should_rturn_a_value_greater_than_or_equal_to_five());
        @endcode

        @param expected a value that should be less than or equal to \a actual
        @param actual the value generated by the test
        @param name the name or description of the assertion
     */
    public void assertGeSoft(Object expected, Object actual, String name) throws Throwable {
        obj.callMethod("assertGeSoft", expected, actual, name);
    }

    //! Tests that a value is greater than or equal to than an expected value with soft comparisons (types may differ)
    /** @par Example:
        @code{.java}
assertGeSoft("5", method_that_should_rturn_a_value_greater_than_or_equal_to_five());
        @endcode

        @param expected a value that should be less than or equal to \a actual
        @param actual the value generated by the test
     */
    public void assertGeSoft(Object expected, Object actual) throws Throwable {
        obj.callMethod("assertGeSoft", expected, actual);
    }

    //! Tests that a value is greater than or equal to an expected value with hard comparisons (types may not differ)
    /** @par Example:
        @code{.java}
assertGe(5, method_that_should_rturn_a_value_greater_than_or_equal_to_five());
        @endcode

        @param expected a value that should be less than or equal to \a actual
        @param actual the value generated by the test
        @param name the name or description of the assertion
     */
    public void assertGe(Object expected, Object actual, String name) throws Throwable {
        obj.callMethod("assertGe", expected, actual, name);
    }

    //! Tests that a value is greater than or equal to an expected value with hard comparisons (types may not differ)
    /** @par Example:
        @code{.java}
assertGe(5, method_that_should_rturn_a_value_greater_than_or_equal_to_five());
        @endcode

        @param expected a value that should be less than or equal to \a actual
        @param actual the value generated by the test
     */
    public void assertGe(Object expected, Object actual) throws Throwable {
        obj.callMethod("assertGe", expected, actual);
    }

    //! Tests that a value is less than an expected value with soft comparisons (types may differ)
    /** @par Example:
        @code{.java}
assertLtSoft("5", method_that_should_rturn_a_value_less_than_five());
        @endcode

        @param expected a value that should be greater than \a actual
        @param actual the value generated by the test
        @param name the name or description of the assertion
     */
    public void assertLtSoft(Object expected, Object actual, String name) throws Throwable {
        obj.callMethod("assertLtSoft", expected, actual, name);
    }

    //! Tests that a value is less than an expected value with soft comparisons (types may differ)
    /** @par Example:
        @code{.java}
assertLtSoft("5", method_that_should_rturn_a_value_less_than_five());
        @endcode

        @param expected a value that should be greater than \a actual
        @param actual the value generated by the test
     */
    public void assertLtSoft(Object expected, Object actual) throws Throwable {
        obj.callMethod("assertLtSoft", expected, actual);
    }

    //! Tests that a value is less than an expected value with hard comparisons (types may not differ)
    /** @par Example:
        @code{.java}
assertLt(5, method_that_should_rturn_a_value_less_than_five());
        @endcode

        @param expected a value that should be greater than \a actual
        @param actual the value generated by the test
        @param name the name or description of the assertion
     */
    public void assertLt(Object expected, Object actual, String name) throws Throwable {
        obj.callMethod("assertLt", expected, actual, name);
    }

    //! Tests that a value is less than an expected value with hard comparisons (types may not differ)
    /** @par Example:
        @code{.java}
assertLt(5, method_that_should_rturn_a_value_less_than_five());
        @endcode

        @param expected a value that should be greater than \a actual
        @param actual the value generated by the test
     */
    public void assertLt(Object expected, Object actual) throws Throwable {
        obj.callMethod("assertLt", expected, actual);
    }

    //! Tests that a value is less than or equal to than an expected value with soft comparisons (types may differ)
    /** @par Example:
        @code{.java}
assertLeSoft("5", method_that_should_rturn_a_value_less_than_or_equal_to_five());
        @endcode

        @param expected a value that should be greater than or equal to \a actual
        @param actual the value generated by the test
        @param name the name or description of the assertion
     */
    public void assertLeSoft(Object expected, Object actual, String name) throws Throwable {
        obj.callMethod("assertLeSoft", expected, actual, name);
    }

    //! Tests that a value is less than or equal to than an expected value with soft comparisons (types may differ)
    /** @par Example:
        @code{.java}
assertLeSoft("5", method_that_should_rturn_a_value_less_than_or_equal_to_five());
        @endcode

        @param expected a value that should be greater than or equal to \a actual
        @param actual the value generated by the test
     */
    public void assertLeSoft(Object expected, Object actual) throws Throwable {
        obj.callMethod("assertLeSoft", expected, actual);
    }

    //! Tests that a value is less than or equal to an expected value with hard comparisons (types may not differ)
    /** @par Example:
        @code{.java}
assertLe(5, method_that_should_rturn_a_value_less_than_or_equal_to_five());
        @endcode

        @param expected a value that should be greater than or equal to \a actual
        @param actual the value generated by the test
        @param name the name or description of the assertion
     */
    public void assertLe(Object expected, Object actual, String name) throws Throwable {
        obj.callMethod("assertLe", expected, actual, name);
    }

    //! Tests that a value is less than or equal to an expected value with hard comparisons (types may not differ)
    /** @par Example:
        @code{.java}
assertLe(5, method_that_should_rturn_a_value_less_than_or_equal_to_five());
        @endcode

        @param expected a value that should be greater than or equal to \a actual
        @param actual the value generated by the test
     */
    public void assertLe(Object expected, Object actual) throws Throwable {
        obj.callMethod("assertLe", expected, actual);
    }

    //! Tests a boolean value
    /** @par Example:
        @code{.java}
assertTrue(methodThatShouldReturnTrue());
        @endcode

        @param actual the value generated by the test
        @param name the name or description of the assertion
     */
    public void assertTrue(Object actual, String name) throws Throwable {
        obj.callMethod("assertTrue", actual, name);
    }

    //! Tests a boolean value
    /** @par Example:
        @code{.java}
assertTrue(methodThatShouldReturnTrue());
        @endcode

        @param actual the value generated by the test
     */
    public void assertTrue(Object actual) throws Throwable {
        obj.callMethod("assertTrue", actual);
    }

    //! Tests a boolean value
    /** @par Example:
        @code{.java}
assertFalse(methodThatShouldReturnFalse());
        @endcode

        @param actual the value generated by the test
        @param name the name or description of the assertion
     */
    public void assertFalse(Object actual, String name) throws Throwable {
        obj.callMethod("assertFalse", actual, name);
    }

    //! Tests a boolean value
    /** @par Example:
        @code{.java}
assertFalse(methodThatShouldReturnFalse());
        @endcode

        @param actual the value generated by the test
     */
    public void assertFalse(Object actual) throws Throwable {
        obj.callMethod("assertFalse", actual);
    }

    //! Tests that a piece of code throws an exception with given description
    /** @par Example:
        @code{.java}
assertThrows("DIVISION-BY-ZERO", "division by zero found in integer expression", sub(int a) {print(5/a);}, 0);
        @endcode

        @param expectedErr the expected exception type
        @param expectedDesc the expected exception detail (desc field), ignored if NOTHING
        @param theCode the code to execute
        @param args optional arguments to the code
        @param name an optional label for the assertion
     */
    public void assertThrows(String expectedErr, String expectedDesc, TestCodeArgs theCode, Object[] args, String name) throws Throwable {
        obj.callMethod("assertThrows", expectedErr, expectedDesc, theCode, args, name);
    }

    //! Tests that a piece of code throws an exception with given description
    /** @par Example:
        @code{.java}
assertThrows("DIVISION-BY-ZERO", "division by zero found in integer expression", sub(int a) {print(5/a);}, 0);
        @endcode

        @param expectedErr the expected exception type
        @param expectedDesc the expected exception detail (desc field), ignored if NOTHING
        @param theCode the code to execute
        @param args optional arguments to the code
     */
    public void assertThrows(String expectedErr, String expectedDesc, TestCodeArgs theCode, Object[] args) throws Throwable {
        obj.callMethod("assertThrows", expectedErr, expectedDesc, theCode, args);
    }

    //! Tests that a piece of code throws an exception with given description
    /** @par Example:
        @code{.java}
assertThrows("DIVISION-BY-ZERO", "division by zero found in integer expression", sub(int a) {print(5/a);}, 0);
        @endcode

        @param expectedErr the expected exception type
        @param expectedDesc the expected exception detail (desc field), ignored if NOTHING
        @param theCode the code to execute
     */
    public void assertThrows(String expectedErr, String expectedDesc, TestCodeArgs theCode) throws Throwable {
        obj.callMethod("assertThrows", expectedErr, expectedDesc, theCode);
    }

    //! Tests that a piece of code throws an exception with given description
    /** @par Example:
        @code{.java}
assertThrows("DIVISION-BY-ZERO", "division by zero found in integer expression", sub(int a) {print(5/a);}, 0);
        @endcode

        @param expectedErr the expected exception type
        @param theCode the code to execute
        @param args optional arguments to the code
        @param name an optional label for the assertion
     */
    public void assertThrows(String expectedErr, TestCodeArgs theCode, Object[] args, String name) throws Throwable {
        obj.callMethod("assertThrows", expectedErr, theCode, args, name);
    }

    //! Tests that a piece of code throws an exception with given description
    /** @par Example:
        @code{.java}
assertThrows("DIVISION-BY-ZERO", "division by zero found in integer expression", sub(int a) {print(5/a);}, 0);
        @endcode

        @param expectedErr the expected exception type
        @param theCode the code to execute
        @param args optional arguments to the code
     */
    public void assertThrows(String expectedErr, TestCodeArgs theCode, Object[] args) throws Throwable {
        obj.callMethod("assertThrows", expectedErr, theCode, args);
    }

    //! Tests that a piece of code throws an exception with given description
    /** @par Example:
        @code{.java}
assertThrows("DIVISION-BY-ZERO", "division by zero found in integer expression", sub(int a) {print(5/a);}, 0);
        @endcode

        @param expectedErr the expected exception type
        @param expectedDesc the expected exception detail (desc field), ignored if NOTHING
        @param theCode the code to execute
        @param name an optional label for the assertion
     */
    public void assertThrows(String expectedErr, String expectedDesc, TestCode theCode, String name) throws Throwable {
        obj.callMethod("assertThrows", expectedErr, expectedDesc, theCode, null, name);
    }

    //! Tests that a piece of code throws an exception with given description
    /** @par Example:
        @code{.java}
assertThrows("DIVISION-BY-ZERO", "division by zero found in integer expression", sub(int a) {print(5/a);}, 0);
        @endcode

        @param expectedErr the expected exception type
        @param expectedDesc the expected exception detail (desc field), ignored if NOTHING
        @param theCode the code to execute
     */
    public void assertThrows(String expectedErr, String expectedDesc, TestCode theCode) throws Throwable {
        obj.callMethod("assertThrows", expectedErr, expectedDesc, theCode);
    }

    //! Tests that a piece of code throws an exception with given description
    /** @par Example:
        @code{.java}
assertThrows("DIVISION-BY-ZERO", "division by zero found in integer expression", sub(int a) {print(5/a);}, 0);
        @endcode

        @param expectedErr the expected exception type
        @param theCode the code to execute
        @param name an optional label for the assertion
     */
    public void assertThrows(String expectedErr, TestCode theCode, String name) throws Throwable {
        obj.callMethod("assertThrows", expectedErr, theCode, null, name);
    }

    //! Tests that a piece of code throws an exception with given description
    /** @par Example:
        @code{.java}
assertThrows("DIVISION-BY-ZERO", "division by zero found in integer expression", sub(int a) {print(5/a);}, 0);
        @endcode

        @param expectedErr the expected exception type
        @param theCode the code to execute
     */
    public void assertThrows(String expectedErr, TestCode theCode) throws Throwable {
        obj.callMethod("assertThrows", expectedErr, theCode);
    }

    //! Skips assertion on purpose
    /**
        @param name the name or description of the assertion

        @par Example:
        @code{.java}
assertSkip("assertion name or reason to skip");
        @endcode
     */
    public void assertSkip(String name) throws Throwable {
        obj.callMethod("assertSkip", name);
    }

    //! Fails the test unconditionally
    /** @par Example:
        @code{.java}
fail("Unexpected code executed");
        @endcode

        @param msg the failure message
     */
    public void fail(String msg) throws Throwable {
        obj.callMethod("fail", msg);
    }

    //! Tests for a single assertion for a call returning no value (for example, to ensure that the call does not throw an exception)
    /**
        @param name the name or description of the assertion
        @param condition A test function whose result we are asserting
        @param args Arguments passed to condition

        @return the result of the \a condition call, if the immediate value has any further use
     */
    public void testNullAssertion(String name, TestConditionArgs condition, Object... args) throws Throwable {
        obj.callMethod("testNullAssertion", name, condition, (Object)args);
    }

    //! Tests for a single assertion for a call returning an integer value and returns the value generated
    /**
        @param name the name or description of the assertion
        @param condition A test function whose result we are asserting
        @param args Arguments passed to condition
        @param expectedResultValue the expected value

        @return the result of the \a condition call, if the immediate value has any further use
     */
    public Object testAssertion(String name, TestConditionArgs condition, Object[] args, Object expectedResultValue) throws Throwable {
        return obj.callMethod("testAssertion", name, condition, (Object)args, expectedResultValue);
    }

    //! Tests for a single assertion for a call returning no value (for example, to ensure that the call does not throw an exception)
    /**
        @param name the name or description of the assertion
        @param condition A test function whose result we are asserting

        @return the result of the \a condition call, if the immediate value has any further use
     */
    public void testNullAssertion(String name, TestCondition condition) throws Throwable {
        obj.callMethod("testNullAssertion", name, condition);
    }

    //! Tests for a single assertion for a call returning an integer value and returns the value generated
    /**
        @param name the name or description of the assertion
        @param condition A test function whose result we are asserting
        @param expectedResultValue the expected value

        @return the result of the \a condition call, if the immediate value has any further use
     */
    public Object testAssertion(String name, TestCondition condition, Object expectedResultValue) throws Throwable {
        return obj.callMethod("testAssertion", name, condition, null, expectedResultValue);
    }

    //! Tests for a single assertion and returns the value generated
    /**
     * @param name the name or description of the assertion
     * @param condition A test function whose result we are asserting
     * @param args Arguments passed to condition
     * @param expectedResult A class describing the expected result of condition; the default is QUnit::TestResultSuccess
     *
     * @return the result of the \a condition call, if the immediate value has any further use
     */
    public Object testAssertion(String name, TestConditionArgs condition, Object[] args, AbstractTestResult expectedResult) throws Throwable {
        return obj.callMethod("testAssertion", name, condition, (Object)args, expectedResult.getQoreObject());
    }

    //! Tests for a single assertion and returns the value generated
    /**
     * @param name the name or description of the assertion
     * @param condition A test function whose result we are asserting
     * @param args Arguments passed to condition
     *
     * @return the result of the \a condition call, if the immediate value has any further use
     */
    public Object testAssertion(String name, TestConditionArgs condition, Object[] args) throws Throwable {
        return obj.callMethod("testAssertion", name, condition, (Object)args);
    }

    //! Tests for a single assertion and returns the value generated
    /**
     * @param name the name or description of the assertion
     * @param condition A test function whose result we are asserting
     * @param expectedResult A class describing the expected result of condition; the default is QUnit::TestResultSuccess
     *
     * @return the result of the \a condition call, if the immediate value has any further use
     */
    public Object testAssertion(String name, TestCondition condition, AbstractTestResult expectedResult) throws Throwable {
        return obj.callMethod("testAssertion", name, condition, null, expectedResult.getQoreObject());
    }

    //! Tests for a single assertion and returns the value generated
    /**
     * @param name the name or description of the assertion
     * @param condition A test function whose result we are asserting
     *
     * @return the result of the \a condition call, if the immediate value has any further use
     */
    public Object testAssertion(String name, TestCondition condition) throws Throwable {
        return obj.callMethod("testAssertion", name, condition);
    }

    //! Skips a given test, eg. because it may be missing some dependencies.
    /**
     * @param reason The reason for the test skip; used as the format argument with @ref Qore::vsprintf() "vsprintf()"
     * with any remaining arguments
     */
    public void testSkip(String reason, Object... args) throws Throwable {
        Object[] new_args = new Object[args.length + 1];
        new_args[0] = reason;
        System.arraycopy(args, 0, new_args, 1, args.length);
        obj.callMethodArgs("testSkip", new_args);
    }

    //! Skips a given test, eg. because it may be missing some dependencies.
    /**
     * @param reason The reason for the test skip
     */
    public void testSkip(String reason) throws Throwable {
        obj.callMethod("testSkip", reason);
    }

    //! Compare two values for equality
    /**
     * @param a Argument 1
     * @param b Argument 2
     *
     * @returns a == b
     */
    public boolean equals(Object a, Object b) throws Throwable {
        return (boolean)obj.callMethod("equals", a, b);
    }

    //! Compare two values for inequality
    /**
     * @param a Argument 1
     * @param b Argument 2
     *
     * @returns a != b
     */
    public boolean notEquals(Object a, Object b) throws Throwable {
        return (boolean)obj.callMethod("notEquals", a, b);
    }

    //! Compare a string for match against a regexp
    /**
     * @param s String to match
     * @param regexp Regular expression to match against
     *
     * @returns a == b
     */
    public boolean regexpMatches(String s, String regexp) throws Throwable {
        return (boolean)obj.callMethod("regexpMatches", s, regexp);
    }

    //! Run the whole suite, report results
    public int main() throws Throwable {
        return ((Long)obj.callMethod("main")).intValue();
    }

    //! returns the assertion name for display purposes
    public static String getAssertionName(String name) throws Throwable {
        return (String)QoreJavaApi.callStaticMethod("QUnit::Test", "getAssertionName", name);
    }

    //! returns the assertion name for display purposes
    public static String getAssertionName() throws Throwable {
        return (String)QoreJavaApi.callStaticMethod("QUnit::Test", "getAssertionName");
    }
}
