/**********************************************************************************************
    Copyright (C) 2006, 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

  Garmin and MapSource are registered trademarks or trademarks of Garmin Ltd.
  or one of its subsidiaries.

  This source is based on John Mechalas documentation "Garmin IMG File Format" found
  at sourceforge. The missing bits and error where rectified by the source code of
  Konstantin Galichsky (kg@geopainting.com), http://www.geopainting.com

**********************************************************************************************/
#ifndef CGARMINIMG_H
#define CGARMINIMG_H

#include <QObject>
#include <QString>
#include <QMap>
#include <QRectF>
#include <QDebug>

#include "GarminStrTbl.h"
#include "CGarminPolygon.h"

class IGarminStrTbl;

/// subfile part (TRE, RGN, ...) location information
struct subfile_part_t
{
    subfile_part_t() : offset(0), size(0){}
    /// file offset of subfile part
    quint32 offset;
    /// size of the subfile part
    quint32 size;
};

class CGarminPoint : public str_info_t
{
    public:
        CGarminPoint() : isLbl6(false), hasSubType(false) {}
        ~CGarminPoint(){}

        quint32 decode(subdiv_desc_t& subdiv, quint8 * pData);

        //quint16 type;
        bool isLbl6;
        bool hasSubType;

        //QString label;
        // double lng;
        // double lat;
        XY point;
};

/// subdivision  information
struct subdiv_desc_t
{
    quint32 n;
    /// section of next level
    quint16 next;
    /// end of section group
    bool terminate;
    /// offset into the subfile's RGN part
    quint32 rgn_start;
    /// end of section in RGN part (last offset = rgn_end - 1)
    quint32 rgn_end;

    /// there are points stored in the RGN subsection
    bool hasPoints;
    /// there are indexd points stored in the RGN subsection
    bool hasIdxPoints;
    /// there are polylines stored in the RGN subsection
    bool hasPolylines;
    /// there are polygons stored in the RGN subsection
    bool hasPolygons;

    /// the center longitude of the area covered by this subdivision
    qint32 iCenterLng;
    /// the center latiude of the area covered by this subdivision
    qint32 iCenterLat;

    /// north boundary of area covered by this subsection
    double north;
    /// east boundary of area covered by this subsection
    double east;
    /// south boundary of area covered by this subsection
    double south;
    /// west boundary of area covered by this subsection
    double west;

    /// area in meter coordinates covered by this subdivision
    QRectF area;

    /// number of left shifts for RGN data
    quint32 shift;
    /// map level this subdivision is shown
    quint32 level;
    /// pointer to string table (LBL section) object
    IGarminStrTbl * strtbl;

    QVector<CGarminPoint>               points;
    QVector<CGarminPoint>               pois;
    QVector<CGarminPolygon>             polylines;
    /// polygons are stored as multimap. See CGarminMap::drawPolygons()
    QMultiMap<quint16,CGarminPolygon>   polygons;
};

/// subfile information
struct subfile_desc_t
{
    subfile_desc_t() : north(0.0), east(0.0), south(0.0), west(0.0), isTransparent(false) {}

    struct maplevel_t
    {
        bool inherited;
        quint8 level;
        quint8 bits;
    };

    /// the name of the subfile (not really needed)
    QString name;
    /// location information of all parts
    QMap<QString,subfile_part_t> parts;

    /// north boundary of area covered by this subfile [rad]
    double north;
    /// east boundary of area covered by this subfile [rad]
    double east;
    /// south boundary of area covered by this subfile [rad]
    double south;
    /// west boundary of area covered by this subfile [rad]
    double west;

    /// area in meter coordinates covered by this subfile
    QRectF area;
    /// pointer collection to definition areas
    QMap<QString,CGarminPolygon> definitionAreas;

    /// list of subdivisions
    QVector<subdiv_desc_t> subdivs;
    /// used maplevels
    QVector<maplevel_t> maplevels;
    /// bit 1 of POI_flags (TRE header @ 0x3F)
    bool isTransparent;
};

enum exce_garmin_e {errOpen,errFormat,errLock};
struct exce_garmin_t
{
    exce_garmin_t(exce_garmin_e err, const QString& msg) : err(err), msg(msg) {}
    exce_garmin_e err;
    QString msg;
};

/// read map data from Garmin *.img file
/**

 */
class CGarminImg : public QObject
{
    Q_OBJECT
        public:
        CGarminImg(QObject * parent);
        virtual ~CGarminImg();
        /// read Garmin *.img file
        void load(const QString& filename, bool peek = false);
        /// get access to map data
        const QMap<QString,subfile_desc_t>& getSubFiles(){return subfiles;}

        const QString& getMapDesc(){return mapdesc;}

        bool isTransparent(){return transparent;}

        static const QString polyline_typestr[];

        struct polygon_typestr_entry_t { quint16 order; QString str; };
        static const polygon_typestr_entry_t polygon_typestr[];
    protected:
        void loadSubFile(subfile_desc_t& subfile, quint8 * const pRawData, bool peek);
        void loadSubDiv(subfile_desc_t& subfile, subdiv_desc_t& subdiv, quint8 * const pRawData);

        /// hold all subfile descriptors
        /**
            In a normal *.img file there is only one subfile. However
            gmapsupp.img files can hold several subfiles each with it's
            own subfile parts.
        */
        QMap<QString,subfile_desc_t> subfiles;

        /// the img headers map descriptor
        QString mapdesc;
        /// relay the transparent flags from the subfiles
        bool transparent;
};

#define DEG(x) ((x) < 0x800000 ? (double)(x) * 360.0 / 16777216.0 : (double)((x) - 0x1000000) * 360.0 / 16777216.0)
#define RAD(x) ((x) < 0x800000 ? (double)(x) * TWOPI / 16777216.0 : (double)((x) - 0x1000000) * TWOPI / 16777216.0)
#endif                           //CGARMINIMG_H
