/***************************************************************************
               qgsdataitem.cpp  - Data items
                             -------------------
    begin                : 2011-04-01
    copyright            : (C) 2011 Radim Blazek
    email                : radim dot blazek at gmail dot com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <QApplication>
#include <QtConcurrentMap>
#include <QtConcurrentRun>
#include <QDateTime>
#include <QElapsedTimer>
#include <QDir>
#include <QFileInfo>
#include <QMenu>
#include <QMouseEvent>
#include <QTreeWidget>
#include <QTreeWidgetItem>
#include <QVector>
#include <QStyle>
#include <QTimer>
#include <mutex>
#include <QRegularExpression>

#include "qgis.h"
#include "qgsdataitem.h"
#include "qgsapplication.h"
#include "qgsdataitemprovider.h"
#include "qgsdataitemproviderregistry.h"
#include "qgsdataprovider.h"
#include "qgslogger.h"
#include "qgsproviderregistry.h"
#include "qgsconfig.h"
#include "qgssettings.h"
#include "qgsanimatedicon.h"
#include "qgsproject.h"
#include "qgsvectorlayer.h"
#include "qgsprovidermetadata.h"

// use GDAL VSI mechanism
#define CPL_SUPRESS_CPLUSPLUS  //#spellok
#include "cpl_vsi.h"
#include "cpl_string.h"

// shared icons

QIcon QgsLayerItem::iconForWkbType( QgsWkbTypes::Type type )
{
  QgsWkbTypes::GeometryType geomType = QgsWkbTypes::geometryType( QgsWkbTypes::Type( type ) );
  switch ( geomType )
  {
    case QgsWkbTypes::NullGeometry:
      return iconTable();
    case QgsWkbTypes::PointGeometry:
      return iconPoint();
    case QgsWkbTypes::LineGeometry:
      return iconLine();
    case QgsWkbTypes::PolygonGeometry:
      return iconPolygon();
    default:
      break;
  }
  return iconDefault();
}

QIcon QgsLayerItem::iconPoint()
{
  return QgsApplication::getThemeIcon( QStringLiteral( "/mIconPointLayer.svg" ) );
}

QIcon QgsLayerItem::iconLine()
{
  return QgsApplication::getThemeIcon( QStringLiteral( "/mIconLineLayer.svg" ) );
}

QIcon QgsLayerItem::iconPolygon()
{
  return QgsApplication::getThemeIcon( QStringLiteral( "/mIconPolygonLayer.svg" ) );
}

QIcon QgsLayerItem::iconTable()
{
  return QgsApplication::getThemeIcon( QStringLiteral( "/mIconTableLayer.svg" ) );
}

QIcon QgsLayerItem::iconRaster()
{
  return QgsApplication::getThemeIcon( QStringLiteral( "/mIconRaster.svg" ) );
}

QIcon QgsLayerItem::iconMesh()
{
  return QgsApplication::getThemeIcon( QStringLiteral( "/mIconMeshLayer.svg" ) );
}

QIcon QgsLayerItem::iconVectorTile()
{
  return QgsApplication::getThemeIcon( QStringLiteral( "/mIconVectorTileLayer.svg" ) );
}

QIcon QgsLayerItem::iconPointCloud()
{
  return QgsApplication::getThemeIcon( QStringLiteral( "/mIconPointCloudLayer.svg" ) );
}

QIcon QgsLayerItem::iconDefault()
{
  return QgsApplication::getThemeIcon( QStringLiteral( "/mIconLayer.png" ) );
}

QIcon QgsDataCollectionItem::iconDataCollection()
{
  return QgsApplication::getThemeIcon( QStringLiteral( "/mIconDbSchema.svg" ) );
}

QIcon QgsDataCollectionItem::openDirIcon()
{
  return QgsApplication::getThemeIcon( QStringLiteral( "/mIconFolderOpen.svg" ) );
}

QIcon QgsDataCollectionItem::homeDirIcon()
{
  return QgsApplication::getThemeIcon( QStringLiteral( "mIconFolderHome.svg" ) );
}

QgsAbstractDatabaseProviderConnection *QgsDataCollectionItem::databaseConnection() const
{
  const QString dataProviderKey { QgsApplication::dataItemProviderRegistry()->dataProviderKey( providerKey() ) };
  QgsProviderMetadata *md { QgsProviderRegistry::instance()->providerMetadata( dataProviderKey ) };

  if ( ! md )
  {
    return nullptr;
  }

  const QString connectionName { name() };

  try
  {
    // First try to retrieve the connection by name if this is a stored connection
    if ( md->findConnection( connectionName ) )
    {
      return static_cast<QgsAbstractDatabaseProviderConnection *>( md->createConnection( connectionName ) );
    }

    // If that fails, try to create a connection from the path, in case this is a
    // filesystem-based DB (gpkg or spatialite)
    // The name is useless, we need to get the file path from the data item path
    const QString databaseFilePath { path().remove( QRegularExpression( R"re([\aZ]{2,}://)re" ) ) };

    if ( QFile::exists( databaseFilePath ) )
    {
      return static_cast<QgsAbstractDatabaseProviderConnection *>( md->createConnection( databaseFilePath, {} ) );
    }
  }
  catch ( QgsProviderConnectionException &ex )
  {
    // This is expected and it is not an error in case the provider does not implement
    // the connections API
  }
  return nullptr;
}

QIcon QgsDataCollectionItem::iconDir()
{
  return QgsApplication::getThemeIcon( QStringLiteral( "/mIconFolder.svg" ) );
}


QgsFieldsItem::QgsFieldsItem( QgsDataItem *parent,
                              const QString &path,
                              const QString &connectionUri,
                              const QString &providerKey,
                              const QString &schema,
                              const QString &tableName )
  : QgsDataItem( QgsDataItem::Fields, parent, tr( "Fields" ), path, providerKey )
  , mSchema( schema )
  , mTableName( tableName )
  , mConnectionUri( connectionUri )
{
  mCapabilities |= ( Fertile | Collapse );
  QgsProviderMetadata *md { QgsProviderRegistry::instance()->providerMetadata( providerKey ) };
  if ( md )
  {
    try
    {
      std::unique_ptr<QgsAbstractDatabaseProviderConnection> conn { static_cast<QgsAbstractDatabaseProviderConnection *>( md->createConnection( mConnectionUri, {} ) ) };
      mTableProperty = qgis::make_unique<QgsAbstractDatabaseProviderConnection::TableProperty>( conn->table( schema, tableName ) );
    }
    catch ( QgsProviderConnectionException &ex )
    {
      QgsDebugMsg( QStringLiteral( "Error creating fields item: %1" ).arg( ex.what() ) );
    }
  }
}

QgsFieldsItem::~QgsFieldsItem()
{

}

QVector<QgsDataItem *> QgsFieldsItem::createChildren()
{
  QVector<QgsDataItem *> children;
  try
  {
    QgsProviderMetadata *md { QgsProviderRegistry::instance()->providerMetadata( providerKey() ) };
    if ( md )
    {
      std::unique_ptr<QgsAbstractDatabaseProviderConnection> conn { static_cast<QgsAbstractDatabaseProviderConnection *>( md->createConnection( mConnectionUri, {} ) ) };
      if ( conn )
      {
        int i = 0;
        const QgsFields constFields { conn->fields( mSchema, mTableName ) };
        for ( const auto &f : constFields )
        {
          QgsFieldItem *fieldItem { new QgsFieldItem( this, f ) };
          fieldItem->setSortKey( i++ );
          children.push_back( fieldItem );
        }
      }
    }
  }
  catch ( const QgsProviderConnectionException &ex )
  {
    children.push_back( new QgsErrorItem( this, ex.what(), path() + QStringLiteral( "/error" ) ) );
  }
  return children;
}

QIcon QgsFieldsItem::icon()
{
  return QgsApplication::getThemeIcon( QStringLiteral( "mSourceFields.svg" ) );
}

QString QgsFieldsItem::connectionUri() const
{
  return mConnectionUri;
}

QgsVectorLayer *QgsFieldsItem::layer()
{
  std::unique_ptr<QgsVectorLayer> vl;
  QgsProviderMetadata *md { QgsProviderRegistry::instance()->providerMetadata( providerKey() ) };
  if ( md )
  {
    try
    {
      std::unique_ptr<QgsAbstractDatabaseProviderConnection> conn { static_cast<QgsAbstractDatabaseProviderConnection *>( md->createConnection( mConnectionUri, {} ) ) };
      if ( conn )
      {
        vl.reset( new QgsVectorLayer( conn->tableUri( mSchema, mTableName ), QStringLiteral( "temp_layer" ), providerKey() ) );
        if ( vl->isValid() )
        {
          return vl.release();
        }
      }
    }
    catch ( const QgsProviderConnectionException & )
    {
      // This should never happen!
      QgsDebugMsg( QStringLiteral( "Error getting connection from %1" ).arg( mConnectionUri ) );
    }
  }
  else
  {
    // This should never happen!
    QgsDebugMsg( QStringLiteral( "Error getting metadata for provider %1" ).arg( providerKey() ) );
  }
  return nullptr;
}

QgsAbstractDatabaseProviderConnection::TableProperty *QgsFieldsItem::tableProperty() const
{
  return mTableProperty.get();
}

QString QgsFieldsItem::tableName() const
{
  return mTableName;
}

QString QgsFieldsItem::schema() const
{
  return mSchema;
}

QgsFieldItem::QgsFieldItem( QgsDataItem *parent, const QgsField &field )
  : QgsDataItem( QgsDataItem::Type::Field, parent, field.name(), parent->path() + '/' + field.name(), parent->providerKey() )
  , mField( field )
{
  // Precondition
  Q_ASSERT( static_cast<QgsFieldsItem *>( parent ) );
  setState( QgsDataItem::State::Populated );
}

QgsFieldItem::~QgsFieldItem()
{
}

QIcon QgsFieldItem::icon()
{
  // Check if this is a geometry column and show the right icon
  QgsFieldsItem *parentFields { static_cast<QgsFieldsItem *>( parent() ) };
  if ( parentFields && parentFields->tableProperty() &&
       parentFields->tableProperty()->geometryColumn() == mName &&
       parentFields->tableProperty()->geometryColumnTypes().count() )
  {
    if ( mField.typeName() == QLatin1String( "raster" ) )
    {
      return QgsLayerItem::iconRaster();
    }
    const QgsWkbTypes::GeometryType geomType { QgsWkbTypes::geometryType( parentFields->tableProperty()->geometryColumnTypes().first().wkbType ) };
    switch ( geomType )
    {
      case QgsWkbTypes::GeometryType::LineGeometry:
        return QgsLayerItem::iconLine();
      case QgsWkbTypes::GeometryType::PointGeometry:
        return QgsLayerItem::iconPoint();
      case QgsWkbTypes::GeometryType::PolygonGeometry:
        return QgsLayerItem::iconPolygon();
      case QgsWkbTypes::GeometryType::UnknownGeometry:
      case QgsWkbTypes::GeometryType::NullGeometry:
        return QgsLayerItem::iconDefault();
    }
  }
  const QIcon icon { QgsFields::iconForFieldType( mField.type() ) };
  // Try subtype if icon is null
  if ( icon.isNull() )
  {
    return QgsFields::iconForFieldType( mField.subType() );
  }
  return icon;
}

QIcon QgsFavoritesItem::iconFavorites()
{
  return QgsApplication::getThemeIcon( QStringLiteral( "/mIconFavorites.svg" ) );
}

QVariant QgsFavoritesItem::sortKey() const
{
  return QStringLiteral( " 0" );
}

QIcon QgsZipItem::iconZip()
{
  return QgsApplication::getThemeIcon( QStringLiteral( "/mIconZip.svg" ) );
}

QgsAnimatedIcon *QgsDataItem::sPopulatingIcon = nullptr;

QgsDataItem::QgsDataItem( QgsDataItem::Type type, QgsDataItem *parent, const QString &name, const QString &path, const QString &providerKey )
// Do not pass parent to QObject, Qt would delete this when parent is deleted
  : mType( type )
  , mCapabilities( NoCapabilities )
  , mParent( parent )
  , mState( NotPopulated )
  , mName( name )
  , mProviderKey( providerKey )
  , mPath( path )
  , mDeferredDelete( false )
  , mFutureWatcher( nullptr )
{
}

QgsDataItem::~QgsDataItem()
{
  QgsDebugMsgLevel( QStringLiteral( "mName = %1 mPath = %2 mChildren.size() = %3" ).arg( mName, mPath ).arg( mChildren.size() ), 2 );
  const auto constMChildren = mChildren;
  for ( QgsDataItem *child : constMChildren )
  {
    if ( !child ) // should not happen
      continue;
    child->deleteLater();
  }
  mChildren.clear();

  if ( mFutureWatcher && !mFutureWatcher->isFinished() )
  {
    // this should not usually happen (until the item was deleted directly when createChildren was running)
    QgsDebugMsg( QStringLiteral( "mFutureWatcher not finished (should not happen) -> waitForFinished()" ) );
    mDeferredDelete = true;
    mFutureWatcher->waitForFinished();
  }

  delete mFutureWatcher;
}

QString QgsDataItem::pathComponent( const QString &string )
{
  return QString( string ).replace( QRegExp( "[\\\\/]" ), QStringLiteral( "|" ) );
}

QVariant QgsDataItem::sortKey() const
{
  return mSortKey.isValid() ? mSortKey : name();
}

void QgsDataItem::setSortKey( const QVariant &key )
{
  mSortKey = key;
}

void QgsDataItem::deleteLater()
{
  QgsDebugMsgLevel( "path = " + path(), 3 );
  setParent( nullptr ); // also disconnects parent
  const auto constMChildren = mChildren;
  for ( QgsDataItem *child : constMChildren )
  {
    if ( !child ) // should not happen
      continue;
    child->deleteLater();
  }
  mChildren.clear();

  if ( mFutureWatcher && !mFutureWatcher->isFinished() )
  {
    QgsDebugMsg( QStringLiteral( "mFutureWatcher not finished -> schedule to delete later" ) );
    mDeferredDelete = true;
  }
  else
  {
    QObject::deleteLater();
  }
}

void QgsDataItem::deleteLater( QVector<QgsDataItem *> &items )
{
  const auto constItems = items;
  for ( QgsDataItem *item : constItems )
  {
    if ( !item ) // should not happen
      continue;
    item->deleteLater();
  }
  items.clear();
}

void QgsDataItem::moveToThread( QThread *targetThread )
{
  // QObject::moveToThread() cannot move objects with parent, but QgsDataItem is not using paren/children from QObject
  const auto constMChildren = mChildren;
  for ( QgsDataItem *child : constMChildren )
  {
    if ( !child ) // should not happen
      continue;
    QgsDebugMsgLevel( "moveToThread child " + child->path(), 3 );
    child->QObject::setParent( nullptr ); // to be sure
    child->moveToThread( targetThread );
  }
  QObject::moveToThread( targetThread );
}

QgsAbstractDatabaseProviderConnection *QgsDataItem::databaseConnection() const
{
  return nullptr;
}

QIcon QgsDataItem::icon()
{
  if ( state() == Populating && sPopulatingIcon )
    return sPopulatingIcon->icon();

  if ( !mIcon.isNull() )
    return mIcon;

  if ( !mIconMap.contains( mIconName ) )
  {
    mIconMap.insert( mIconName, mIconName.startsWith( ':' ) ? QIcon( mIconName ) : QgsApplication::getThemeIcon( mIconName ) );
  }

  return mIconMap.value( mIconName );
}

void QgsDataItem::setName( const QString &name )
{
  mName = name;
  emit dataChanged( this );
}

QVector<QgsDataItem *> QgsDataItem::createChildren()
{
  return QVector<QgsDataItem *>();
}

void QgsDataItem::populate( bool foreground )
{
  if ( state() == Populated || state() == Populating )
    return;

  QgsDebugMsgLevel( "mPath = " + mPath, 2 );

  if ( capabilities2() & QgsDataItem::Fast || foreground )
  {
    populate( createChildren() );
  }
  else
  {
    setState( Populating );
    // The watcher must not be created with item (in constructor) because the item may be created in thread and the watcher created in thread does not work correctly.
    if ( !mFutureWatcher )
    {
      mFutureWatcher = new QFutureWatcher< QVector <QgsDataItem *> >( this );
    }

    connect( mFutureWatcher, &QFutureWatcherBase::finished, this, &QgsDataItem::childrenCreated );
    mFutureWatcher->setFuture( QtConcurrent::run( runCreateChildren, this ) );
  }
}

// This is expected to be run in a separate thread
QVector<QgsDataItem *> QgsDataItem::runCreateChildren( QgsDataItem *item )
{
  QgsDebugMsgLevel( "path = " + item->path(), 2 );
  QElapsedTimer time;
  time.start();
  QVector <QgsDataItem *> children = item->createChildren();
  QgsDebugMsgLevel( QStringLiteral( "%1 children created in %2 ms" ).arg( children.size() ).arg( time.elapsed() ), 3 );
  // Children objects must be pushed to main thread.
  const auto constChildren = children;
  for ( QgsDataItem *child : constChildren )
  {
    if ( !child ) // should not happen
      continue;
    QgsDebugMsgLevel( "moveToThread child " + child->path(), 2 );
    if ( qApp )
      child->moveToThread( qApp->thread() ); // moves also children
  }
  QgsDebugMsgLevel( QStringLiteral( "finished path %1: %2 children" ).arg( item->path() ).arg( children.size() ), 3 );
  return children;
}

void QgsDataItem::childrenCreated()
{
  QgsDebugMsgLevel( QStringLiteral( "path = %1 children.size() = %2" ).arg( path() ).arg( mFutureWatcher->result().size() ), 3 );

  if ( deferredDelete() )
  {
    QgsDebugMsg( QStringLiteral( "Item was scheduled to be deleted later" ) );
    QObject::deleteLater();
    return;
  }

  if ( mChildren.isEmpty() ) // usually populating but may also be refresh if originally there were no children
  {
    populate( mFutureWatcher->result() );
  }
  else // refreshing
  {
    refresh( mFutureWatcher->result() );
  }
  disconnect( mFutureWatcher, &QFutureWatcherBase::finished, this, &QgsDataItem::childrenCreated );
  emit dataChanged( this ); // to replace loading icon by normal icon
}

void QgsDataItem::updateIcon()
{
  emit dataChanged( this );
}

void QgsDataItem::populate( const QVector<QgsDataItem *> &children )
{
  QgsDebugMsgLevel( "mPath = " + mPath, 3 );

  const auto constChildren = children;
  for ( QgsDataItem *child : constChildren )
  {
    if ( !child ) // should not happen
      continue;
    // update after thread finished -> refresh
    addChildItem( child, true );
  }
  setState( Populated );
}

void QgsDataItem::depopulate()
{
  QgsDebugMsgLevel( "mPath = " + mPath, 3 );

  const auto constMChildren = mChildren;
  for ( QgsDataItem *child : constMChildren )
  {
    QgsDebugMsgLevel( "remove " + child->path(), 3 );
    child->depopulate(); // recursive
    deleteChildItem( child );
  }
  setState( NotPopulated );
}

void QgsDataItem::refresh()
{
  if ( state() == Populating )
    return;

  QgsDebugMsgLevel( "mPath = " + mPath, 3 );

  if ( capabilities2() & QgsDataItem::Fast )
  {
    refresh( createChildren() );
  }
  else
  {
    setState( Populating );
    if ( !mFutureWatcher )
    {
      mFutureWatcher = new QFutureWatcher< QVector <QgsDataItem *> >( this );
    }
    connect( mFutureWatcher, &QFutureWatcherBase::finished, this, &QgsDataItem::childrenCreated );
    mFutureWatcher->setFuture( QtConcurrent::run( runCreateChildren, this ) );
  }
}

void QgsDataItem::refreshConnections( const QString &key )
{
  // Walk up until the root node is reached
  if ( mParent )
  {
    mParent->refreshConnections( key );
  }
  else
  {
    // if a specific key was specified then we use that -- otherwise we assume the connections
    // changed belong to the same provider as this item
    emit connectionsChanged( key.isEmpty() ? providerKey() : key );
  }
}

void QgsDataItem::refresh( const QVector<QgsDataItem *> &children )
{
  QgsDebugMsgLevel( "mPath = " + mPath, 2 );

  // Remove no more present children
  QVector<QgsDataItem *> remove;
  const auto constMChildren = mChildren;
  for ( QgsDataItem *child : constMChildren )
  {
    if ( !child ) // should not happen
      continue;
    if ( findItem( children, child ) >= 0 )
      continue;
    remove.append( child );
  }
  const auto constRemove = remove;
  for ( QgsDataItem *child : constRemove )
  {
    QgsDebugMsgLevel( "remove " + child->path(), 3 );
    deleteChildItem( child );
  }

  // Add new children
  const auto constChildren = children;
  for ( QgsDataItem *child : constChildren )
  {
    if ( !child ) // should not happen
      continue;

    int index = findItem( mChildren, child );
    if ( index >= 0 )
    {
      // Refresh recursively (some providers may create more generations of descendants)
      if ( !( child->capabilities2() & QgsDataItem::Fertile ) )
      {
        // The child cannot createChildren() itself
        mChildren.value( index )->refresh( child->children() );
      }

      child->deleteLater();
      continue;
    }
    addChildItem( child, true );
  }
  setState( Populated );
}

QString QgsDataItem::providerKey() const
{
  return mProviderKey;
}

void QgsDataItem::setProviderKey( const QString &value )
{
  mProviderKey = value;
}

int QgsDataItem::rowCount()
{
  return mChildren.size();
}
bool QgsDataItem::hasChildren()
{
  return ( state() == Populated ? !mChildren.isEmpty() : true );
}

bool QgsDataItem::layerCollection() const
{
  return false;
}

void QgsDataItem::setParent( QgsDataItem *parent )
{
  if ( mParent )
  {
    disconnect( this, nullptr, mParent, nullptr );
  }
  if ( parent )
  {
    connect( this, &QgsDataItem::beginInsertItems, parent, &QgsDataItem::beginInsertItems );
    connect( this, &QgsDataItem::endInsertItems, parent, &QgsDataItem::endInsertItems );
    connect( this, &QgsDataItem::beginRemoveItems, parent, &QgsDataItem::beginRemoveItems );
    connect( this, &QgsDataItem::endRemoveItems, parent, &QgsDataItem::endRemoveItems );
    connect( this, &QgsDataItem::dataChanged, parent, &QgsDataItem::dataChanged );
    connect( this, &QgsDataItem::stateChanged, parent, &QgsDataItem::stateChanged );
  }
  mParent = parent;
}

void QgsDataItem::addChildItem( QgsDataItem *child, bool refresh )
{
  Q_ASSERT( child );
  QgsDebugMsgLevel( QStringLiteral( "path = %1 add child #%2 - %3 - %4" ).arg( mPath ).arg( mChildren.size() ).arg( child->mName ).arg( child->mType ), 3 );

  //calculate position to insert child
  int i;
  if ( type() == Directory )
  {
    for ( i = 0; i < mChildren.size(); i++ )
    {
      // sort items by type, so directories are before data items
      if ( mChildren.at( i )->mType == child->mType &&
           mChildren.at( i )->mName.localeAwareCompare( child->mName ) > 0 )
        break;
    }
  }
  else
  {
    for ( i = 0; i < mChildren.size(); i++ )
    {
      if ( mChildren.at( i )->mName.localeAwareCompare( child->mName ) >= 0 )
        break;
    }
  }

  if ( refresh )
    emit beginInsertItems( this, i, i );

  mChildren.insert( i, child );
  child->setParent( this );

  if ( refresh )
    emit endInsertItems();
}

void QgsDataItem::deleteChildItem( QgsDataItem *child )
{
  QgsDebugMsgLevel( "mName = " + child->mName, 2 );
  int i = mChildren.indexOf( child );
  Q_ASSERT( i >= 0 );
  emit beginRemoveItems( this, i, i );
  mChildren.remove( i );
  child->deleteLater();
  emit endRemoveItems();
}

QgsDataItem *QgsDataItem::removeChildItem( QgsDataItem *child )
{
  QgsDebugMsgLevel( "mName = " + child->mName, 2 );
  int i = mChildren.indexOf( child );
  Q_ASSERT( i >= 0 );
  if ( i < 0 )
  {
    child->setParent( nullptr );
    return nullptr;
  }

  emit beginRemoveItems( this, i, i );
  mChildren.remove( i );
  emit endRemoveItems();
  return child;
}

int QgsDataItem::findItem( QVector<QgsDataItem *> items, QgsDataItem *item )
{
  for ( int i = 0; i < items.size(); i++ )
  {
    Q_ASSERT_X( items[i], "findItem", QStringLiteral( "item %1 is nullptr" ).arg( i ).toLatin1() );
    QgsDebugMsgLevel( QString::number( i ) + " : " + items[i]->mPath + " x " + item->mPath, 2 );
    if ( items[i]->equal( item ) )
      return i;
  }
  return -1;
}

bool QgsDataItem::equal( const QgsDataItem *other )
{
  return ( metaObject()->className() == other->metaObject()->className() &&
           mPath == other->path() );
}

QList<QAction *> QgsDataItem::actions( QWidget *parent )
{
  Q_UNUSED( parent )
  return QList<QAction *>();
}

bool QgsDataItem::handleDoubleClick()
{
  return false;
}

QgsMimeDataUtils::Uri QgsDataItem::mimeUri() const
{
  return mimeUris().isEmpty() ? QgsMimeDataUtils::Uri() : mimeUris().first();
}

bool QgsDataItem::rename( const QString & )
{
  return false;
}

QgsDataItem::State QgsDataItem::state() const
{
  return mState;
}

void QgsDataItem::setState( State state )
{
  QgsDebugMsgLevel( QStringLiteral( "item %1 set state %2 -> %3" ).arg( path() ).arg( this->state() ).arg( state ), 3 );
  if ( state == mState )
    return;

  State oldState = mState;

  if ( state == Populating ) // start loading
  {
    if ( !sPopulatingIcon )
    {
      // TODO: ensure that QgsAnimatedIcon is created on UI thread only
      sPopulatingIcon = new QgsAnimatedIcon( QgsApplication::iconPath( QStringLiteral( "/mIconLoading.gif" ) ), QgsApplication::instance() );
    }

    sPopulatingIcon->connectFrameChanged( this, &QgsDataItem::updateIcon );
  }
  else if ( mState == Populating && sPopulatingIcon ) // stop loading
  {
    sPopulatingIcon->disconnectFrameChanged( this, &QgsDataItem::updateIcon );
  }


  mState = state;

  emit stateChanged( this, oldState );
  if ( state == Populated )
    updateIcon();
}

QList<QMenu *> QgsDataItem::menus( QWidget *parent )
{
  Q_UNUSED( parent )
  return QList<QMenu *>();
}

// ---------------------------------------------------------------------

QgsLayerItem::QgsLayerItem( QgsDataItem *parent, const QString &name, const QString &path,
                            const QString &uri, LayerType layerType, const QString &providerKey )
  : QgsDataItem( Layer, parent, name, path, providerKey )
  , mUri( uri )
  , mLayerType( layerType )
{
  mIconName = iconName( layerType );
}

QgsMapLayerType QgsLayerItem::mapLayerType() const
{
  switch ( mLayerType )
  {
    case QgsLayerItem::Raster:
      return QgsMapLayerType::RasterLayer;

    case QgsLayerItem::Mesh:
      return QgsMapLayerType::MeshLayer;

    case QgsLayerItem::VectorTile:
      return QgsMapLayerType::VectorTileLayer;

    case QgsLayerItem::Plugin:
      return QgsMapLayerType::PluginLayer;

    case QgsLayerItem::PointCloud:
      return QgsMapLayerType::PointCloudLayer;

    case QgsLayerItem::NoType:
    case QgsLayerItem::Vector:
    case QgsLayerItem::Point:
    case QgsLayerItem::Polygon:
    case QgsLayerItem::Line:
    case QgsLayerItem::TableLayer:
    case QgsLayerItem::Table:
    case QgsLayerItem::Database:
      return QgsMapLayerType::VectorLayer;
  }

  return QgsMapLayerType::VectorLayer; // no warnings
}

QgsLayerItem::LayerType QgsLayerItem::typeFromMapLayer( QgsMapLayer *layer )
{
  switch ( layer->type() )
  {
    case QgsMapLayerType::VectorLayer:
    {
      switch ( qobject_cast< QgsVectorLayer * >( layer )->geometryType() )
      {
        case QgsWkbTypes::PointGeometry:
          return Point;

        case QgsWkbTypes::LineGeometry:
          return Line;

        case QgsWkbTypes::PolygonGeometry:
          return Polygon;

        case QgsWkbTypes::NullGeometry:
          return TableLayer;

        case QgsWkbTypes::UnknownGeometry:
          return Vector;
      }

      return Vector; // no warnings
    }

    case QgsMapLayerType::RasterLayer:
      return Raster;
    case QgsMapLayerType::PluginLayer:
      return Plugin;
    case QgsMapLayerType::MeshLayer:
      return Mesh;
    case QgsMapLayerType::PointCloudLayer:
      return PointCloud;
    case QgsMapLayerType::VectorTileLayer:
      return VectorTile;
    case QgsMapLayerType::AnnotationLayer:
      return Vector; // will never happen!
  }
  return Vector; // no warnings
}

QString QgsLayerItem::layerTypeAsString( QgsLayerItem::LayerType layerType )
{
  static int enumIdx = staticMetaObject.indexOfEnumerator( "LayerType" );
  return staticMetaObject.enumerator( enumIdx ).valueToKey( layerType );
}

QString QgsLayerItem::iconName( QgsLayerItem::LayerType layerType )
{
  switch ( layerType )
  {
    case Point:
      return QStringLiteral( "/mIconPointLayer.svg" );
    case Line:
      return QStringLiteral( "/mIconLineLayer.svg" );
    case Polygon:
      return QStringLiteral( "/mIconPolygonLayer.svg" );
    // TODO add a new icon for generic Vector layers
    case Vector :
      return QStringLiteral( "/mIconVector.svg" );
    case TableLayer:
    case Table:
      return QStringLiteral( "/mIconTableLayer.svg" );
    case Raster:
      return QStringLiteral( "/mIconRaster.svg" );
    case Mesh:
      return QStringLiteral( "/mIconMeshLayer.svg" );
    case PointCloud:
      return QStringLiteral( "/mIconPointCloudLayer.svg" );
    default:
      return QStringLiteral( "/mIconLayer.png" );
  }
}

bool QgsLayerItem::deleteLayer()
{
  return false;
}

bool QgsLayerItem::equal( const QgsDataItem *other )
{
  //QgsDebugMsg ( mPath + " x " + other->mPath );
  if ( type() != other->type() )
  {
    return false;
  }
  //const QgsLayerItem *o = qobject_cast<const QgsLayerItem *> ( other );
  const QgsLayerItem *o = qobject_cast<const QgsLayerItem *>( other );
  if ( !o )
    return false;

  return ( mPath == o->mPath && mName == o->mName && mUri == o->mUri && mProviderKey == o->mProviderKey );
}

QgsMimeDataUtils::UriList QgsLayerItem::mimeUris() const
{
  QgsMimeDataUtils::Uri u;

  switch ( mapLayerType() )
  {
    case QgsMapLayerType::VectorLayer:
      u.layerType = QStringLiteral( "vector" );
      switch ( mLayerType )
      {
        case Point:
          u.wkbType = QgsWkbTypes::Point;
          break;
        case Line:
          u.wkbType = QgsWkbTypes::LineString;
          break;
        case Polygon:
          u.wkbType = QgsWkbTypes::Polygon;
          break;
        case TableLayer:
          u.wkbType = QgsWkbTypes::NoGeometry;
          break;

        case Database:
        case Table:
        case NoType:
        case Vector:
        case Raster:
        case Plugin:
        case Mesh:
        case PointCloud:
        case VectorTile:
          break;
      }
      break;
    case QgsMapLayerType::RasterLayer:
      u.layerType = QStringLiteral( "raster" );
      break;
    case QgsMapLayerType::MeshLayer:
      u.layerType = QStringLiteral( "mesh" );
      break;
    case QgsMapLayerType::VectorTileLayer:
      u.layerType = QStringLiteral( "vector-tile" );
      break;
    case QgsMapLayerType::PointCloudLayer:
      u.layerType = QStringLiteral( "pointcloud" );
      break;
    case QgsMapLayerType::PluginLayer:
      u.layerType = QStringLiteral( "plugin" );
      break;
    case QgsMapLayerType::AnnotationLayer:
      u.layerType = QStringLiteral( "annotation" );
      break;
  }

  u.providerKey = providerKey();
  u.name = layerName();
  u.uri = uri();
  u.supportedCrs = supportedCrs();
  u.supportedFormats = supportedFormats();
  return { u };
}

// ---------------------------------------------------------------------
QgsDataCollectionItem::QgsDataCollectionItem( QgsDataItem *parent,
    const QString &name,
    const QString &path,
    const QString &providerKey )
  : QgsDataItem( Collection, parent, name, path, providerKey )
{
  mCapabilities = Fertile;
  mIconName = QStringLiteral( "/mIconDbSchema.svg" );
}

QgsDataCollectionItem::~QgsDataCollectionItem()
{
  QgsDebugMsgLevel( "mName = " + mName + " mPath = " + mPath, 2 );

// Do not delete children, children are deleted by QObject parent
#if 0
  const auto constMChildren = mChildren;
  for ( QgsDataItem *i : constMChildren )
  {
    QgsDebugMsgLevel( QStringLiteral( "delete child = 0x%0" ).arg( static_cast<qlonglong>( i ), 8, 16, QLatin1Char( '0' ) ), 2 );
    delete i;
  }
#endif
}

//-----------------------------------------------------------------------

QgsDirectoryItem::QgsDirectoryItem( QgsDataItem *parent, const QString &name, const QString &path )
  : QgsDataCollectionItem( parent, QDir::toNativeSeparators( name ), path )
  , mDirPath( path )
  , mRefreshLater( false )
{
  mType = Directory;
  init();
}

QgsDirectoryItem::QgsDirectoryItem( QgsDataItem *parent, const QString &name,
                                    const QString &dirPath, const QString &path,
                                    const QString &providerKey )
  : QgsDataCollectionItem( parent, QDir::toNativeSeparators( name ), path, providerKey )
  , mDirPath( dirPath )
  , mRefreshLater( false )
{
  mType = Directory;
  init();
}

void QgsDirectoryItem::init()
{
  setToolTip( QDir::toNativeSeparators( mDirPath ) );
}

QIcon QgsDirectoryItem::icon()
{
  if ( mDirPath == QDir::homePath() )
    return homeDirIcon();

  // still loading? show the spinner
  if ( state() == Populating )
    return QgsDataItem::icon();

  // symbolic link? use link icon
  QFileInfo fi( mDirPath );
  if ( fi.isDir() && fi.isSymLink() )
  {
    return QgsApplication::getThemeIcon( QStringLiteral( "mIconFolderLink.svg" ) );
  }

  // loaded? show the open dir icon
  if ( state() == Populated )
    return openDirIcon();

  // show the closed dir icon
  return iconDir();
}


QVector<QgsDataItem *> QgsDirectoryItem::createChildren()
{
  QVector<QgsDataItem *> children;
  QDir dir( mDirPath );

  const QList<QgsDataItemProvider *> providers = QgsApplication::dataItemProviderRegistry()->providers();

  QStringList entries = dir.entryList( QDir::AllDirs | QDir::NoDotAndDotDot, QDir::Name | QDir::IgnoreCase );
  const auto constEntries = entries;
  for ( const QString &subdir : constEntries )
  {
    if ( mRefreshLater )
    {
      deleteLater( children );
      return children;
    }

    QString subdirPath = dir.absoluteFilePath( subdir );

    QgsDebugMsgLevel( QStringLiteral( "creating subdir: %1" ).arg( subdirPath ), 2 );

    QString path = mPath + '/' + subdir; // may differ from subdirPath
    if ( QgsDirectoryItem::hiddenPath( path ) )
      continue;

    bool handledByProvider = false;
    for ( QgsDataItemProvider *provider : providers )
    {
      if ( provider->handlesDirectoryPath( path ) )
      {
        handledByProvider = true;
        break;
      }
    }
    if ( handledByProvider )
      continue;

    QgsDirectoryItem *item = new QgsDirectoryItem( this, subdir, subdirPath, path );

    // we want directories shown before files
    item->setSortKey( QStringLiteral( "  %1" ).arg( subdir ) );

    // propagate signals up to top

    children.append( item );
  }

  QStringList fileEntries = dir.entryList( QDir::Dirs | QDir::NoDotAndDotDot | QDir::Files, QDir::Name );
  const auto constFileEntries = fileEntries;
  for ( const QString &name : constFileEntries )
  {
    if ( mRefreshLater )
    {
      deleteLater( children );
      return children;
    }

    QString path = dir.absoluteFilePath( name );
    QFileInfo fileInfo( path );

    if ( fileInfo.suffix().compare( QLatin1String( "zip" ), Qt::CaseInsensitive ) == 0 ||
         fileInfo.suffix().compare( QLatin1String( "tar" ), Qt::CaseInsensitive ) == 0 )
    {
      QgsDataItem *item = QgsZipItem::itemFromPath( this, path, name, mPath + '/' + name );
      if ( item )
      {
        children.append( item );
        continue;
      }
    }

    bool createdItem = false;
    for ( QgsDataItemProvider *provider : providers )
    {
      int capabilities = provider->capabilities();

      if ( !( ( fileInfo.isFile() && ( capabilities & QgsDataProvider::File ) ) ||
              ( fileInfo.isDir() && ( capabilities & QgsDataProvider::Dir ) ) ) )
      {
        continue;
      }

      QgsDataItem *item = provider->createDataItem( path, this );
      if ( item )
      {
        children.append( item );
        createdItem = true;
      }
    }

    if ( !createdItem )
    {
      // if item is a QGIS project, and no specific item provider has overridden handling of
      // project items, then use the default project item behavior
      if ( fileInfo.suffix().compare( QLatin1String( "qgs" ), Qt::CaseInsensitive ) == 0 ||
           fileInfo.suffix().compare( QLatin1String( "qgz" ), Qt::CaseInsensitive ) == 0 )
      {
        QgsDataItem *item = new QgsProjectItem( this, fileInfo.completeBaseName(), path );
        children.append( item );
        continue;
      }
    }

  }
  return children;
}

void QgsDirectoryItem::setState( State state )
{
  QgsDataCollectionItem::setState( state );

  if ( state == Populated )
  {
    if ( !mFileSystemWatcher )
    {
      mFileSystemWatcher = new QFileSystemWatcher( this );
      mFileSystemWatcher->addPath( mDirPath );
      connect( mFileSystemWatcher, &QFileSystemWatcher::directoryChanged, this, &QgsDirectoryItem::directoryChanged );
    }
    mLastScan = QDateTime::currentDateTime();
  }
  else if ( state == NotPopulated )
  {
    if ( mFileSystemWatcher )
    {
      delete mFileSystemWatcher;
      mFileSystemWatcher = nullptr;
    }
  }
}

void QgsDirectoryItem::directoryChanged()
{
  // If the last scan was less than 10 seconds ago, skip this
  if ( mLastScan.msecsTo( QDateTime::currentDateTime() ) < QgsSettings().value( QStringLiteral( "browser/minscaninterval" ), 10000 ).toInt() )
  {
    return;
  }
  if ( state() == Populating )
  {
    // schedule to refresh later, because refresh() simply returns if Populating
    mRefreshLater = true;
  }
  else
  {
    // We definintely don't want the temporary files created by sqlite
    // to re-trigger a refresh in an infinite loop.
    disconnect( mFileSystemWatcher, &QFileSystemWatcher::directoryChanged, this, &QgsDirectoryItem::directoryChanged );
    // QFileSystemWhatcher::directoryChanged is emitted when a
    // file is created and not when it is closed/flushed.
    //
    // Delay to give to OS the time to complete writing the file
    // this happens when a new file appears in the directory and
    // the item's children thread will try to open the file with
    // GDAL or OGR even if it is still being written.
    QTimer::singleShot( 100, this, [ = ] { refresh(); } );
  }
}

bool QgsDirectoryItem::hiddenPath( const QString &path )
{
  QgsSettings settings;
  QStringList hiddenItems = settings.value( QStringLiteral( "browser/hiddenPaths" ),
                            QStringList() ).toStringList();
  int idx = hiddenItems.indexOf( path );
  return ( idx > -1 );
}

void QgsDirectoryItem::childrenCreated()
{
  QgsDebugMsgLevel( QStringLiteral( "mRefreshLater = %1" ).arg( mRefreshLater ), 3 );

  if ( mRefreshLater )
  {
    QgsDebugMsgLevel( QStringLiteral( "directory changed during createChidren() -> refresh() again" ), 3 );
    mRefreshLater = false;
    setState( Populated );
    refresh();
  }
  else
  {
    QgsDataCollectionItem::childrenCreated();
  }
  // Re-connect the file watcher after all children have been created
  connect( mFileSystemWatcher, &QFileSystemWatcher::directoryChanged, this, &QgsDirectoryItem::directoryChanged );
}

bool QgsDirectoryItem::equal( const QgsDataItem *other )
{
  //QgsDebugMsg ( mPath + " x " + other->mPath );
  if ( type() != other->type() )
  {
    return false;
  }
  return ( path() == other->path() );
}

QWidget *QgsDirectoryItem::paramWidget()
{
  return new QgsDirectoryParamWidget( mPath );
}

QgsMimeDataUtils::UriList QgsDirectoryItem::mimeUris() const
{
  QgsMimeDataUtils::Uri u;
  u.layerType = QStringLiteral( "directory" );
  u.name = mName;
  u.uri = mDirPath;
  return { u };
}

QgsDirectoryParamWidget::QgsDirectoryParamWidget( const QString &path, QWidget *parent )
  : QTreeWidget( parent )
{
  setRootIsDecorated( false );

  // name, size, date, permissions, owner, group, type
  setColumnCount( 7 );
  QStringList labels;
  labels << tr( "Name" ) << tr( "Size" ) << tr( "Date" ) << tr( "Permissions" ) << tr( "Owner" ) << tr( "Group" ) << tr( "Type" );
  setHeaderLabels( labels );

  QIcon iconDirectory = QgsApplication::getThemeIcon( QStringLiteral( "mIconFolder.svg" ) );
  QIcon iconFile = QgsApplication::getThemeIcon( QStringLiteral( "mIconFile.svg" ) );
  QIcon iconDirLink = QgsApplication::getThemeIcon( QStringLiteral( "mIconFolderLink.svg" ) );
  QIcon iconFileLink = QgsApplication::getThemeIcon( QStringLiteral( "mIconFileLink.svg" ) );

  QList<QTreeWidgetItem *> items;

  QDir dir( path );
  QStringList entries = dir.entryList( QDir::AllEntries | QDir::NoDotAndDotDot, QDir::Name | QDir::IgnoreCase );
  const auto constEntries = entries;
  for ( const QString &name : constEntries )
  {
    QFileInfo fi( dir.absoluteFilePath( name ) );
    QStringList texts;
    texts << name;
    QString size;
    if ( fi.size() > 1024 )
    {
      size = QStringLiteral( "%1 KiB" ).arg( QString::number( fi.size() / 1024.0, 'f', 1 ) );
    }
    else if ( fi.size() > 1.048576e6 )
    {
      size = QStringLiteral( "%1 MiB" ).arg( QString::number( fi.size() / 1.048576e6, 'f', 1 ) );
    }
    else
    {
      size = QStringLiteral( "%1 B" ).arg( fi.size() );
    }
    texts << size;
    texts << QLocale().toString( fi.lastModified(), QLocale::ShortFormat );
    QString perm;
    perm += fi.permission( QFile::ReadOwner ) ? 'r' : '-';
    perm += fi.permission( QFile::WriteOwner ) ? 'w' : '-';
    perm += fi.permission( QFile::ExeOwner ) ? 'x' : '-';
    // QFile::ReadUser, QFile::WriteUser, QFile::ExeUser
    perm += fi.permission( QFile::ReadGroup ) ? 'r' : '-';
    perm += fi.permission( QFile::WriteGroup ) ? 'w' : '-';
    perm += fi.permission( QFile::ExeGroup ) ? 'x' : '-';
    perm += fi.permission( QFile::ReadOther ) ? 'r' : '-';
    perm += fi.permission( QFile::WriteOther ) ? 'w' : '-';
    perm += fi.permission( QFile::ExeOther ) ? 'x' : '-';
    texts << perm;

    texts << fi.owner();
    texts << fi.group();

    QString type;
    QIcon icon;
    if ( fi.isDir() && fi.isSymLink() )
    {
      type = tr( "folder" );
      icon = iconDirLink;
    }
    else if ( fi.isDir() )
    {
      type = tr( "folder" );
      icon = iconDirectory;
    }
    else if ( fi.isFile() && fi.isSymLink() )
    {
      type = tr( "file" );
      icon = iconFileLink;
    }
    else if ( fi.isFile() )
    {
      type = tr( "file" );
      icon = iconFile;
    }

    texts << type;

    QTreeWidgetItem *item = new QTreeWidgetItem( texts );
    item->setIcon( 0, icon );
    items << item;
  }

  addTopLevelItems( items );

  // hide columns that are not requested
  QgsSettings settings;
  QList<QVariant> lst = settings.value( QStringLiteral( "dataitem/directoryHiddenColumns" ) ).toList();
  const auto constLst = lst;
  for ( const QVariant &colVariant : constLst )
  {
    setColumnHidden( colVariant.toInt(), true );
  }
}

void QgsDirectoryParamWidget::mousePressEvent( QMouseEvent *event )
{
  if ( event->button() == Qt::RightButton )
  {
    // show the popup menu
    QMenu popupMenu;

    QStringList labels;
    labels << tr( "Name" ) << tr( "Size" ) << tr( "Date" ) << tr( "Permissions" ) << tr( "Owner" ) << tr( "Group" ) << tr( "Type" );
    for ( int i = 0; i < labels.count(); i++ )
    {
      QAction *action = popupMenu.addAction( labels[i], this, &QgsDirectoryParamWidget::showHideColumn );
      action->setObjectName( QString::number( i ) );
      action->setCheckable( true );
      action->setChecked( !isColumnHidden( i ) );
    }

    popupMenu.exec( event->globalPos() );
  }
}

void QgsDirectoryParamWidget::showHideColumn()
{
  QAction *action = qobject_cast<QAction *>( sender() );
  if ( !action )
    return; // something is wrong

  int columnIndex = action->objectName().toInt();
  setColumnHidden( columnIndex, !isColumnHidden( columnIndex ) );

  // save in settings
  QgsSettings settings;
  QList<QVariant> lst;
  for ( int i = 0; i < columnCount(); i++ )
  {
    if ( isColumnHidden( i ) )
      lst.append( QVariant( i ) );
  }
  settings.setValue( QStringLiteral( "dataitem/directoryHiddenColumns" ), lst );
}

QgsProjectItem::QgsProjectItem( QgsDataItem *parent, const QString &name,
                                const QString &path, const QString &providerKey )
  : QgsDataItem( QgsDataItem::Project, parent, name, path, providerKey )
{
  mIconName = QStringLiteral( ":/images/icons/qgis_icon.svg" );
  setToolTip( QDir::toNativeSeparators( path ) );
  setState( Populated ); // no more children
}

QgsMimeDataUtils::UriList QgsProjectItem::mimeUris() const
{
  QgsMimeDataUtils::Uri u;
  u.layerType = QStringLiteral( "project" );
  u.name = mName;
  u.uri = mPath;
  return { u };
}

QgsErrorItem::QgsErrorItem( QgsDataItem *parent, const QString &error, const QString &path )
  : QgsDataItem( QgsDataItem::Error, parent, error, path )
{
  mIconName = QStringLiteral( "/mIconDelete.svg" );

  setState( Populated ); // no more children
}

QgsFavoritesItem::QgsFavoritesItem( QgsDataItem *parent, const QString &name, const QString &path )
  : QgsDataCollectionItem( parent, name, QStringLiteral( "favorites:" ), QStringLiteral( "special:Favorites" ) )
{
  Q_UNUSED( path )
  mCapabilities |= Fast;
  mType = Favorites;
  mIconName = QStringLiteral( "/mIconFavorites.svg" );
  populate();
}

QVector<QgsDataItem *> QgsFavoritesItem::createChildren()
{
  QVector<QgsDataItem *> children;

  QgsSettings settings;

  const QStringList favDirs = settings.value( QStringLiteral( "browser/favourites" ), QVariant() ).toStringList();

  for ( const QString &favDir : favDirs )
  {
    QStringList parts = favDir.split( QStringLiteral( "|||" ) );
    if ( parts.empty() )
      continue;

    QString dir = parts.at( 0 );
    QString name = dir;
    if ( parts.count() > 1 )
      name = parts.at( 1 );

    children << createChildren( dir, name );
  }

  return children;
}

void QgsFavoritesItem::addDirectory( const QString &favDir, const QString &n )
{
  QString name = n.isEmpty() ? favDir : n;

  QgsSettings settings;
  QStringList favDirs = settings.value( QStringLiteral( "browser/favourites" ) ).toStringList();
  favDirs.append( QStringLiteral( "%1|||%2" ).arg( favDir, name ) );
  settings.setValue( QStringLiteral( "browser/favourites" ), favDirs );

  if ( state() == Populated )
  {
    QVector<QgsDataItem *> items = createChildren( favDir, name );
    const auto constItems = items;
    for ( QgsDataItem *item : constItems )
    {
      addChildItem( item, true );
    }
  }
}

void QgsFavoritesItem::removeDirectory( QgsDirectoryItem *item )
{
  if ( !item )
    return;

  QgsSettings settings;
  QStringList favDirs = settings.value( QStringLiteral( "browser/favourites" ) ).toStringList();
  for ( int i = favDirs.count() - 1; i >= 0; --i )
  {
    QStringList parts = favDirs.at( i ).split( QStringLiteral( "|||" ) );
    if ( parts.empty() )
      continue;

    QString dir = parts.at( 0 );
    if ( dir == item->dirPath() )
      favDirs.removeAt( i );
  }
  settings.setValue( QStringLiteral( "browser/favourites" ), favDirs );

  int idx = findItem( mChildren, item );
  if ( idx < 0 )
  {
    QgsDebugMsg( QStringLiteral( "favorites item %1 not found" ).arg( item->path() ) );
    return;
  }

  if ( state() == Populated )
    deleteChildItem( mChildren.at( idx ) );
}

void QgsFavoritesItem::renameFavorite( const QString &path, const QString &name )
{
  // update stored name
  QgsSettings settings;
  QStringList favDirs = settings.value( QStringLiteral( "browser/favourites" ) ).toStringList();
  for ( int i = 0; i < favDirs.count(); ++i )
  {
    QStringList parts = favDirs.at( i ).split( QStringLiteral( "|||" ) );
    if ( parts.empty() )
      continue;

    QString dir = parts.at( 0 );
    if ( dir == path )
    {
      favDirs[i] = QStringLiteral( "%1|||%2" ).arg( path, name );
      break;
    }
  }
  settings.setValue( QStringLiteral( "browser/favourites" ), favDirs );

  // also update existing data item
  const QVector<QgsDataItem *> ch = children();
  for ( QgsDataItem *child : ch )
  {
    if ( QgsFavoriteItem *favorite = qobject_cast< QgsFavoriteItem * >( child ) )
    {
      if ( favorite->dirPath() == path )
      {
        favorite->setName( name );
        break;
      }
    }
  }
}

QVector<QgsDataItem *> QgsFavoritesItem::createChildren( const QString &favDir, const QString &name )
{
  QVector<QgsDataItem *> children;
  QString pathName = pathComponent( favDir );
  const auto constProviders = QgsApplication::dataItemProviderRegistry()->providers();
  for ( QgsDataItemProvider *provider : constProviders )
  {
    int capabilities = provider->capabilities();

    if ( capabilities & QgsDataProvider::Dir )
    {
      QgsDataItem *item = provider->createDataItem( favDir, this );
      if ( item )
      {
        item->setName( name );
        children.append( item );
      }
    }
  }
  if ( children.isEmpty() )
  {
    QgsFavoriteItem *item = new QgsFavoriteItem( this, name, favDir, mPath + '/' + pathName );
    if ( item )
    {
      children.append( item );
    }
  }
  return children;
}

//-----------------------------------------------------------------------
QStringList QgsZipItem::sProviderNames = QStringList();


QgsZipItem::QgsZipItem( QgsDataItem *parent, const QString &name, const QString &path )
  : QgsDataCollectionItem( parent, name, path )
{
  mFilePath = path;
  init();
}

QgsZipItem::QgsZipItem( QgsDataItem *parent, const QString &name,
                        const QString &filePath, const QString &path,
                        const QString &providerKey )
  : QgsDataCollectionItem( parent, name, path, providerKey )
  , mFilePath( filePath )
{
  init();
}

void QgsZipItem::init()
{
  mType = Collection; //Zip??
  mIconName = QStringLiteral( "/mIconZip.svg" );
  mVsiPrefix = vsiPrefix( mFilePath );

  static std::once_flag initialized;
  std::call_once( initialized, [ = ]
  {
    sProviderNames << QStringLiteral( "OGR" ) << QStringLiteral( "GDAL" );
  } );
}

QVector<QgsDataItem *> QgsZipItem::createChildren()
{
  QVector<QgsDataItem *> children;
  QString tmpPath;
  QgsSettings settings;
  QString scanZipSetting = settings.value( QStringLiteral( "qgis/scanZipInBrowser2" ), "basic" ).toString();

  mZipFileList.clear();

  QgsDebugMsgLevel( QStringLiteral( "mFilePath = %1 path = %2 name= %3 scanZipSetting= %4 vsiPrefix= %5" ).arg( mFilePath, path(), name(), scanZipSetting, mVsiPrefix ), 3 );

  // if scanZipBrowser == no: skip to the next file
  if ( scanZipSetting == QLatin1String( "no" ) )
  {
    return children;
  }

  // first get list of files
  getZipFileList();

  const QList<QgsDataItemProvider *> providers = QgsApplication::dataItemProviderRegistry()->providers();

  // loop over files inside zip
  const auto constMZipFileList = mZipFileList;
  for ( const QString &fileName : constMZipFileList )
  {
    QFileInfo info( fileName );
    tmpPath = mVsiPrefix + mFilePath + '/' + fileName;
    QgsDebugMsgLevel( "tmpPath = " + tmpPath, 3 );

    for ( QgsDataItemProvider *provider : providers )
    {
      if ( !sProviderNames.contains( provider->name() ) )
        continue;

      // ugly hack to remove .dbf file if there is a .shp file
      if ( provider->name() == QLatin1String( "OGR" ) )
      {
        if ( info.suffix().compare( QLatin1String( "dbf" ), Qt::CaseInsensitive ) == 0 )
        {
          if ( mZipFileList.indexOf( fileName.left( fileName.count() - 4 ) + ".shp" ) != -1 )
            continue;
        }
        if ( info.completeSuffix().compare( QLatin1String( "shp.xml" ), Qt::CaseInsensitive ) == 0 )
        {
          continue;
        }
      }

      QgsDebugMsgLevel( QStringLiteral( "trying to load item %1 with %2" ).arg( tmpPath, provider->name() ), 3 );
      QgsDataItem *item = provider->createDataItem( tmpPath, this );
      if ( item )
      {
        // the item comes with zipped file name, set the name to relative path within zip file
        item->setName( fileName );
        children.append( item );
      }
      else
      {
        QgsDebugMsgLevel( QStringLiteral( "not loaded item" ), 3 );
      }
    }
  }

  return children;
}

QgsDataItem *QgsZipItem::itemFromPath( QgsDataItem *parent, const QString &path, const QString &name )
{
  return itemFromPath( parent, path, name, path );
}

QgsDataItem *QgsZipItem::itemFromPath( QgsDataItem *parent, const QString &filePath, const QString &name, const QString &path )
{
  QgsSettings settings;
  QString scanZipSetting = settings.value( QStringLiteral( "qgis/scanZipInBrowser2" ), "basic" ).toString();
  QStringList zipFileList;
  QString vsiPrefix = QgsZipItem::vsiPrefix( filePath );
  QgsZipItem *zipItem = nullptr;
  bool populated = false;

  QgsDebugMsgLevel( QStringLiteral( "path = %1 name= %2 scanZipSetting= %3 vsiPrefix= %4" ).arg( path, name, scanZipSetting, vsiPrefix ), 3 );

  // don't scan if scanZipBrowser == no
  if ( scanZipSetting == QLatin1String( "no" ) )
    return nullptr;

  // don't scan if this file is not a /vsizip/ or /vsitar/ item
  if ( ( vsiPrefix != QLatin1String( "/vsizip/" ) && vsiPrefix != QLatin1String( "/vsitar/" ) ) )
    return nullptr;

  zipItem = new QgsZipItem( parent, name, filePath, path );

  if ( zipItem )
  {
    // force populate zipItem if it has less than 10 items and is not a .tgz or .tar.gz file (slow loading)
    // for other items populating will be delayed until item is opened
    // this might be polluting the tree with empty items but is necessary for performance reasons
    // could also accept all files smaller than a certain size and add options for file count and/or size

    // first get list of files inside .zip or .tar files
    if ( path.endsWith( QLatin1String( ".zip" ), Qt::CaseInsensitive ) ||
         path.endsWith( QLatin1String( ".tar" ), Qt::CaseInsensitive ) )
    {
      zipFileList = zipItem->getZipFileList();
    }
    // force populate if less than 10 items
    if ( !zipFileList.isEmpty() && zipFileList.count() <= 10 )
    {
      zipItem->populate( zipItem->createChildren() );
      populated = true; // there is no QgsDataItem::isPopulated() function
      QgsDebugMsgLevel( QStringLiteral( "Got zipItem with %1 children, path=%2, name=%3" ).arg( zipItem->rowCount() ).arg( zipItem->path(), zipItem->name() ), 3 );
    }
    else
    {
      QgsDebugMsgLevel( QStringLiteral( "Delaying populating zipItem with path=%1, name=%2" ).arg( zipItem->path(), zipItem->name() ), 3 );
    }
  }

  // only display if has children or if is not populated
  if ( zipItem && ( !populated || zipItem->rowCount() > 0 ) )
  {
    QgsDebugMsgLevel( QStringLiteral( "returning zipItem" ), 3 );
    return zipItem;
  }

  return nullptr;
}

QStringList QgsZipItem::getZipFileList()
{
  if ( ! mZipFileList.isEmpty() )
    return mZipFileList;

  QString tmpPath;
  QgsSettings settings;
  QString scanZipSetting = settings.value( QStringLiteral( "qgis/scanZipInBrowser2" ), "basic" ).toString();

  QgsDebugMsgLevel( QStringLiteral( "mFilePath = %1 name= %2 scanZipSetting= %3 vsiPrefix= %4" ).arg( mFilePath, name(), scanZipSetting, mVsiPrefix ), 3 );

  // if scanZipBrowser == no: skip to the next file
  if ( scanZipSetting == QLatin1String( "no" ) )
  {
    return mZipFileList;
  }

  // get list of files inside zip file
  QgsDebugMsgLevel( QStringLiteral( "Open file %1 with gdal vsi" ).arg( mVsiPrefix + mFilePath ), 3 );
  char **papszSiblingFiles = VSIReadDirRecursive( QString( mVsiPrefix + mFilePath ).toLocal8Bit().constData() );
  if ( papszSiblingFiles )
  {
    for ( int i = 0; papszSiblingFiles[i]; i++ )
    {
      tmpPath = papszSiblingFiles[i];
      QgsDebugMsgLevel( QStringLiteral( "Read file %1" ).arg( tmpPath ), 3 );
      // skip directories (files ending with /)
      if ( tmpPath.right( 1 ) != QLatin1String( "/" ) )
        mZipFileList << tmpPath;
    }
    CSLDestroy( papszSiblingFiles );
  }
  else
  {
    QgsDebugMsg( QStringLiteral( "Error reading %1" ).arg( mFilePath ) );
  }

  return mZipFileList;
}


QgsDatabaseSchemaItem::QgsDatabaseSchemaItem( QgsDataItem *parent, const QString &name, const QString &path, const QString &providerKey )
  : QgsDataCollectionItem( parent, name, path, providerKey )
{

}

QgsDatabaseSchemaItem::~QgsDatabaseSchemaItem()
{

}

QIcon QgsDatabaseSchemaItem::iconDataCollection()
{
  return QgsApplication::getThemeIcon( QStringLiteral( "/mIconDbSchema.svg" ) );
}

QgsAbstractDatabaseProviderConnection *QgsDatabaseSchemaItem::databaseConnection() const
{
  const QString dataProviderKey { QgsApplication::dataItemProviderRegistry()->dataProviderKey( providerKey() ) };
  QgsProviderMetadata *md { QgsProviderRegistry::instance()->providerMetadata( dataProviderKey ) };
  if ( ! md )
  {
    return nullptr;
  }
  const QString connectionName { parent()->name() };
  try
  {
    return static_cast<QgsAbstractDatabaseProviderConnection *>( md->createConnection( connectionName ) );
  }
  catch ( QgsProviderConnectionException &ex )
  {
    // This is expected and it is not an error in case the provider does not implement
    // the connections API
  }
  return nullptr;
}


QgsConnectionsRootItem::QgsConnectionsRootItem( QgsDataItem *parent, const QString &name, const QString &path, const QString &providerKey )
  : QgsDataCollectionItem( parent, name, path, providerKey )
{
}


///@cond PRIVATE

QgsProjectHomeItem::QgsProjectHomeItem( QgsDataItem *parent, const QString &name, const QString &dirPath, const QString &path )
  : QgsDirectoryItem( parent, name, dirPath, path, QStringLiteral( "special:ProjectHome" ) )
{
}

QIcon QgsProjectHomeItem::icon()
{
  if ( state() == Populating )
    return QgsDirectoryItem::icon();
  return QgsApplication::getThemeIcon( QStringLiteral( "mIconFolderProject.svg" ) );
}

QVariant QgsProjectHomeItem::sortKey() const
{
  return QStringLiteral( " 1" );
}


QgsFavoriteItem::QgsFavoriteItem( QgsFavoritesItem *parent, const QString &name, const QString &dirPath, const QString &path )
  : QgsDirectoryItem( parent, name, dirPath, path, QStringLiteral( "special:Favorites" ) )
  , mFavorites( parent )
{
  mCapabilities |= Rename;
}

bool QgsFavoriteItem::rename( const QString &name )
{
  mFavorites->renameFavorite( dirPath(), name );
  return true;
}


///@endcond
