
/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "qcdefs.h"

char           *code = "q.out";
int		modno = NONE;
int		prio = 0;
FILE           *codefp;

int		tcodespsz, tmpspsz, exprtbsz;
OPREC		tcodesp[TCODESPSZ];
EXPR            exprtb[EXPRTBSZ];
VARREC   	vartb[VARTBSZ];
char     	tmpsp[TMPSPSZ];

struct {
  int sz;
  int *tb;
  byte *ib;
} _imports[MAXFILES];

/* current size allocated to symbol table, string space and init table: */

static unsigned		asymtbsz = SYMTBSZ, astrspsz = STRSPSZ,
                        alimbspsz = STRSPSZ, ainittbsz = INITTBSZ;

/* keep track of labels for backpatching: */

static int label = -1;

#define builtinfun(s,fno,c) newfun(s, fno), symtb[fno].flags |= DCL, \
			    symtb[fno].argc = c
#define builtinvar(s,vno) newfvar(s, vno), symtb[vno].flags |= DCL
#define builtintype(s,type) newtype(s, type), symtb[type].flags |= DCL

void
inittables(void)
{
  int k;

  /* allocate tables: */

  if (!(hashtb = (int *)aalloc(hashtbsz, sizeof(int))) ||
      !(symtb = (SYMREC *)aalloc(asymtbsz, sizeof(SYMREC))) ||
      !(strsp = (char *)aalloc(astrspsz, sizeof(char))) ||
      !(limbsp = (mp_limb_t *)aalloc(alimbspsz, sizeof(mp_limb_t))) ||
      !(inittb = (int *)aalloc(ainittbsz, sizeof(int))))
    fatal(qcmsg[MEM_OVF]);

  /* initialize hash table: */

  for (k = 0; k < hashtbsz; k++)
    hashtb[k] = NONE;

  /* enter predefined symbols into table: */

  builtinfun("def", DEFOP, 2);		symtb[DEFOP].flags |= SPEC;
					symtb[DEFOP].argv = 1;
  builtinfun("undef", UNDEFOP, 1);	symtb[UNDEFOP].flags |= SPEC;
					symtb[UNDEFOP].argv = 1;

  builtinfun("[|]", CONSOP, 0);		symtb[CONSOP].flags |= CST;
  builtinfun("(|)", PAIROP, 0);		symtb[PAIROP].flags |= CST;
  builtinfun("@", APPOP, 0);
  
  builtinfun("false", FALSEOP, 0);	symtb[FALSEOP].flags |= CST;
                                        symtb[FALSEOP].type = BOOLTYPE;
  builtinfun("true", TRUEOP, 0);	symtb[TRUEOP].flags |= CST;
					symtb[TRUEOP].type = BOOLTYPE;
  builtinfun("[]", NILOP, 0);		symtb[NILOP].flags |= CST;
					symtb[NILOP].type = LISTTYPE;
  builtinfun("nil_stream", SNILOP, 0);	symtb[SNILOP].flags |= CST;
					symtb[SNILOP].type = STREAMTYPE;
  builtinfun("cons_stream", SCONSOP, 2);symtb[SCONSOP].flags |= CST|SPEC;
					symtb[SCONSOP].argv = 3;
					symtb[SCONSOP].type = STREAMTYPE;
  builtinfun("()", VOIDOP, 0);		symtb[VOIDOP].flags |= CST;
					symtb[VOIDOP].type = TUPLETYPE;

  builtinfun("++", CATOP, 2);		symtb[CATOP].prec = 3;
  builtinfun("+", ADDOP, 2);		symtb[ADDOP].prec = 3;
  builtinfun("-", MINOP, 2);		symtb[MINOP].prec = 3;
  builtinfun("*", MULOP, 2);		symtb[MULOP].prec = 4;
  builtinfun("/", FDIVOP, 2);		symtb[FDIVOP].prec = 4;
  builtinfun("div", DIVOP, 2);		symtb[DIVOP].prec = 4;
  builtinfun("mod", MODOP, 2);		symtb[MODOP].prec = 4;
  builtinfun("^", POWOP, 2);		symtb[POWOP].prec = 6;
  builtinfun("!", IDXOP, 2);		symtb[IDXOP].prec = 6;
  builtinfun(".", COMPOP, 2);		symtb[COMPOP].prec = 7;
  builtinfun("minus", UMINOP, 1);	/*symtb[UMINOP].prec = 5;*/
  builtinfun("#", HASHOP, 1);		symtb[HASHOP].prec = 5;

  builtinfun("'", QUOTEOP, 1);		symtb[QUOTEOP].flags |= CST|SPEC;
					symtb[QUOTEOP].argv = 1;
					symtb[QUOTEOP].prec = 9;
  builtinfun("`", UNQUOTEOP, 1);	symtb[UNQUOTEOP].prec = 9;
  builtinfun("~", FORCEOP, 1);		symtb[FORCEOP].prec = 9;
  builtinfun("&", MEMOP, 1);		symtb[MEMOP].prec = 9;

  builtinfun("or", OROP, 2);		symtb[OROP].prec = 3;
  builtinfun("and", ANDOP, 2);		symtb[ANDOP].prec = 4;
  builtinfun("or else", ORELSEOP, 2);	symtb[ORELSEOP].flags |= SPEC;
					symtb[ORELSEOP].argv = 2;
					symtb[ORELSEOP].prec = 3;
  builtinfun("and then", ANDTHENOP, 2);	symtb[ANDTHENOP].flags |= SPEC;
					symtb[ANDTHENOP].argv = 2;
					symtb[ANDTHENOP].prec = 4;
  builtinfun("not", NOTOP, 1);		symtb[NOTOP].prec = 5;

  builtinfun("<", LEOP, 2);		symtb[LEOP].prec = 2;
  builtinfun(">", GROP, 2);		symtb[GROP].prec = 2;
  builtinfun("=", EQOP, 2);		symtb[EQOP].prec = 2;
  builtinfun("<=", LEQOP, 2);		symtb[LEQOP].prec = 2;
  builtinfun(">=", GEQOP, 2);		symtb[GEQOP].prec = 2;
  builtinfun("<>", NEQOP, 2);		symtb[NEQOP].prec = 2;
  builtinfun("==", IDOP, 2);		symtb[IDOP].flags |= SPEC;
					symtb[IDOP].argv = 3;
					symtb[IDOP].prec = 2;

  builtinfun("$", RAPPOP, 2);		symtb[RAPPOP].prec = 1;
  builtinfun("||", SEQOP, 2);		symtb[SEQOP].prec = 0;

  builtinfun("shl", SHLOP, 2);
  builtinfun("shr", SHROP, 2);
  builtinfun("pred", PREDOP, 1);
  builtinfun("succ", SUCCOP, 1);
  builtinfun("enum", ENUMOP, 2);
  builtinfun("enum_from", ENUM1OP, 1);
  builtinfun("tupleenum", TENUMOP, 2);
  builtinfun("tupleenum_from", TENUM1OP, 1);
  builtinfun("streamenum", SENUMOP, 2);
  builtinfun("streamenum_from", SENUM1OP, 1);

  builtinfun("exp", EXPOP, 1);
  builtinfun("ln", LNOP, 1);
  builtinfun("sqrt", SQRTOP, 1);
  builtinfun("sin", SINOP, 1);
  builtinfun("cos", COSOP, 1);
  builtinfun("atan", ATANOP, 1);
  builtinfun("atan2", ATAN2OP, 2);
  builtinfun("random", RANDOMOP, 0);
  builtinfun("seed", SEEDOP, 1);

  builtinfun("sub", SUBOP, 3);
  builtinfun("substr", SUBSTROP, 3);
  builtinfun("pos", POSOP, 2);

  builtinfun("int", INTOP, 1);
  builtinfun("frac", FRACOP, 1);
  builtinfun("trunc", TRUNCOP, 1);
  builtinfun("round", ROUNDOP, 1);
  builtinfun("float", FLOATOP, 1);
  builtinfun("hash", HASHNUMOP, 1);
  builtinfun("ord", ORDOP, 1);
  builtinfun("chr", CHROP, 1);
  builtinfun("list", LISTOP, 1);
  builtinfun("tuple", TUPLEOP, 1);
  builtinfun("str", STROP, 1);
  builtinfun("val", VALOP, 1);
  builtinfun("strq", STRQOP, 1);
  builtinfun("valq", VALQOP, 1);

  builtinfun("isspecial", ISSPECIALOP, 1);symtb[ISSPECIALOP].flags |= SPEC;
					symtb[ISSPECIALOP].argv = 1;
  builtinfun("isconst", ISCONSTOP, 1);	symtb[ISCONSTOP].flags |= SPEC;
					symtb[ISCONSTOP].argv = 1;
  builtinfun("isfun", ISFUNOP, 1);	symtb[ISFUNOP].flags |= SPEC;
					symtb[ISFUNOP].argv = 1;
  builtinfun("isvar", ISVAROP, 1);	symtb[ISVAROP].flags |= SPEC;
					symtb[ISVAROP].argv = 1;
  builtinfun("isdef", ISDEFOP, 1);	symtb[ISDEFOP].flags |= SPEC;
					symtb[ISDEFOP].argv = 1;
  builtinfun("flip", FLIPOP, 3);

  builtinfun("read", READOP, 0);
  builtinfun("readq", READQOP, 0);
  builtinfun("readc", READCOP, 0);
  builtinfun("reads", READSOP, 0);
  builtinfun("write", WRITEOP, 1);
  builtinfun("writeq", WRITEQOP, 1);
  builtinfun("writec", WRITECOP, 1);
  builtinfun("writes", WRITESOP, 1);

  builtinfun("fread", FREADOP, 1);
  builtinfun("freadq", FREADQOP, 1);
  builtinfun("freadc", FREADCOP, 1);
  builtinfun("freads", FREADSOP, 1);
  builtinfun("fwrite", FWRITEOP, 2);
  builtinfun("fwriteq", FWRITEQOP, 2);
  builtinfun("fwritec", FWRITECOP, 2);
  builtinfun("fwrites", FWRITESOP, 2);

  builtinfun("fopen", FOPENOP, 2);
  builtinfun("popen", POPENOP, 2);
  builtinfun("fclose", FCLOSEOP, 1);
  builtinfun("eof", EOFOP, 0);
  builtinfun("feof", FEOFOP, 1);
  builtinfun("flush", FLUSHOP, 0);
  builtinfun("fflush", FFLUSHOP, 1);

  builtinfun("version", VERSIONOP, 0);
  builtinfun("sysinfo", SYSINFOOP, 0);
  builtinfun("which", WHICHOP, 1);
  builtinfun("halt", HALTOP, 0);
  builtinfun("quit", QUITOP, 0);
  builtinfun("break", BREAKOP, 0);
  builtinfun("catch", CATCHOP, 2);	symtb[CATCHOP].flags |= SPEC;
					symtb[CATCHOP].argv = 3;
  builtinfun("throw", THROWOP, 1);
  builtinfun("trap", TRAPOP, 2);
  builtinfun("fail", FAILOP, 0);
  builtinfun("_FAIL_", FAIL2OP, 0);
  builtinfun("syserr", SYSERROP, 1);	symtb[SYSERROP].flags |= CST;
                                        symtb[SYSERROP].type = SYSEXCEPTTYPE;
  builtinfun("lambda", LAMBDAOP, 2);	symtb[LAMBDAOP].flags |= SPEC|VIRT;
					symtb[LAMBDAOP].argv = 3;
					symtb[LAMBDAOP].type = FUNCTIONTYPE;
  builtinfun("lambdax", LAMBDAXOP, 1);	symtb[LAMBDAXOP].flags |= SPEC;
					symtb[LAMBDAXOP].argv = 1;

  builtinfun("time", TIMEOP, 0);
  builtinfun("sleep", SLEEPOP, 1);

  builtinfun("view", UNPARSEOP, 1);	symtb[UNPARSEOP].flags |= SPEC;
					symtb[UNPARSEOP].argv = 1;
	
  builtinvar("INPUT", INPUTOP);		symtb[INPUTOP].flags |= CST;
  builtinvar("OUTPUT", OUTPUTOP);	symtb[OUTPUTOP].flags |= CST;
  builtinvar("ERROR", ERROROP);		symtb[ERROROP].flags |= CST;
  builtinvar("ARGS", ARGSOP);		symtb[ARGSOP].flags |= CST;
  builtinvar("_", DEFVAROP);
   
  builtintype("Num", NUMTYPE);
  builtintype("Real", REALTYPE);	symtb[REALTYPE].type = NUMTYPE;
  builtintype("Int", INTTYPE);		symtb[INTTYPE].type = REALTYPE;
  builtintype("Float", FLOATTYPE);	symtb[FLOATTYPE].type = REALTYPE;
  builtintype("String", STRTYPE);
  builtintype("Char", CHARTYPE);	symtb[CHARTYPE].type = STRTYPE;
  builtintype("File", FILETYPE);
  builtintype("List", LISTTYPE);
  builtintype("Stream", STREAMTYPE);
  builtintype("Tuple", TUPLETYPE);
  builtintype("Bool", BOOLTYPE);	symtb[BOOLTYPE].fno_min = FALSEOP;
					symtb[BOOLTYPE].fno_max = TRUEOP;
  builtintype("Lambda", LAMBDATYPE);
  builtintype("<LambdaVar>", LAMBDAVARTYPE);
  builtintype("Function", FUNCTIONTYPE);
					symtb[FUNCTIONTYPE].flags |= VIRT;
  builtintype("Exception", EXCEPTTYPE);
  builtintype("SysException", SYSEXCEPTTYPE);
					symtb[SYSEXCEPTTYPE].type = EXCEPTTYPE;

  /* assert: symtbsz == BUILTIN */

}

void
clear(void)
{
  int i;
  if (tmpspsz)
    for (i = 0; i < VARTBSZ; i++)
      vartb[i].dflag = 0;
  for (i = 0; i < exprtbsz; i++)
    if (exprtb[i].fno == INTVALOP)
      mpz_clear(exprtb[i].tag.iv);
  tcodespsz = tmpspsz = exprtbsz = 0;
  label = -1;
}

EXPR *
intexpr(mpz_t iv)
{
  if (exprtbsz >= EXPRTBSZ)
    fatal(qcmsg[EXPRTB_OVF]);
  else {
    EXPR           *x = exprtb+(exprtbsz++);
    x->fno = INTVALOP;
    x->type = INTTYPE;
    x->argc = x->expc = 0;
    memcpy(x->tag.iv, iv, sizeof(mpz_t));
    return x;
  }
}

EXPR *
floatexpr(double fv)
{
  if (exprtbsz >= EXPRTBSZ)
    fatal(qcmsg[EXPRTB_OVF]);
  else {
    EXPR           *x = exprtb+(exprtbsz++);
    x->fno = FLOATVALOP;
    x->type = FLOATTYPE;
    x->argc = x->expc = 0;
    x->tag.fv = fv;
    return x;
  }
}

EXPR *
strexpr(int sv)
{
  if (exprtbsz >= EXPRTBSZ)
    fatal(qcmsg[EXPRTB_OVF]);
  else {
    EXPR           *x = exprtb+(exprtbsz++);
    x->fno = STRVALOP;
    x->type = STRTYPE;
    x->argc = x->expc = 0;
    x->tag.sv = sv;
    return x;
  }
}

EXPR *
varexpr(int vno)
{
  if (exprtbsz >= EXPRTBSZ)
    fatal(qcmsg[EXPRTB_OVF]);
  else {
    EXPR           *x = exprtb+(exprtbsz++);
    x->fno = VAROP;
    x->type = vartb[vno].type;
    x->argc = x->expc = 0;
    x->tag.vno = vno;
    return x;
  }
}

EXPR *
consexpr(EXPR *x1, EXPR *x2)
{
  if (exprtbsz >= EXPRTBSZ)
    fatal(qcmsg[EXPRTB_OVF]);
  else {
    EXPR           *x = exprtb+(exprtbsz++);
    x->fno = CONSOP;
    x->type = LISTTYPE;
    x->argc = x->expc = 0;
    x->x1 = x1;
    x->x2 = x2;
    return x;
  }
}

EXPR *
pairexpr(EXPR *x1, EXPR *x2)
{
  if (exprtbsz >= EXPRTBSZ)
    fatal(qcmsg[EXPRTB_OVF]);
  else {
    EXPR           *x = exprtb+(exprtbsz++);
    x->fno = PAIROP;
    x->type = TUPLETYPE;
    x->argc = x->expc = 0;
    x->x1 = x1;
    x->x2 = x2;
    return x;
  }
}

EXPR *
appexpr(EXPR *x1, EXPR *x2)
{
  if (exprtbsz >= EXPRTBSZ)
    fatal(qcmsg[EXPRTB_OVF]);
  else {
    EXPR           *x = exprtb+(exprtbsz++);
    x->fno = APPOP;
    x->argc = x1->argc+1;
    x->expc = x1->expc;
    x->type = (x1->fno==VAROP||x1->type==0)?NONE:x1->type;
    x->x1 = x1;
    x->x2 = x2;
    return x;
  }
}

EXPR *
funexpr(int fno)
{
  if (exprtbsz >= EXPRTBSZ)
    fatal(qcmsg[EXPRTB_OVF]);
  else if (fno == NONE) {
    /* provide a dummy expression */
    EXPR           *x = exprtb+(exprtbsz++);
    x->fno = VOIDOP;
    x->type = 0;
    x->argc = 0;
    x->expc = 0;
    return x;
  }
  else {
    EXPR           *x = exprtb+(exprtbsz++);
    x->fno = fno;
    x->type = symtb[fno].type;
    x->argc = 0;
    x->expc = symtb[fno].argc;
    return x;
  }
}

EXPR *
ternexpr(int fno, EXPR *x1, EXPR *x2, EXPR *x3)
{
  return (appexpr(appexpr(appexpr(funexpr(fno), x1), x2), x3));
}

EXPR *
binexpr(int fno, EXPR *x1, EXPR *x2)
{
  return (appexpr(appexpr(funexpr(fno), x1), x2));
}

EXPR *
unexpr(int fno, EXPR *x1)
{
  return (appexpr(funexpr(fno), x1));
}

int
putlimbs(mpz_t z)
{
  int             k = limbspsz, l = mpz_size(z);

  if (l > 0) {
    while (l >= alimbspsz - k)
      if (!(limbsp = (mp_limb_t*)arealloc(limbsp, alimbspsz, LIMBSPSZ/10,
					  sizeof(mp_limb_t))))
	fatal(qcmsg[LIMBSP_OVF]);
      else
	alimbspsz += LIMBSPSZ/10;
    memcpy(limbsp+k, z->_mp_d, l*sizeof(mp_limb_t));
    limbspsz += l;
  }
  return (k);
}

int
putstr(char *s)
{
  int             k = strspsz, l = strlen(s);

  while (l >= astrspsz - k)
    if (!(strsp = (char*)arealloc(strsp, astrspsz, STRSPSZ/10,
				  sizeof(char))))
      fatal(qcmsg[STRSP_OVF]);
    else
      astrspsz += STRSPSZ/10;
  strcpy(strsp+k, s);
  strspsz += l + 1;
  return (k);
}

int
puttmp(char *s)
{
  int             k = tmpspsz, l = strlen(s);

  if (l >= TMPSPSZ - tmpspsz)
    fatal(qcmsg[STRSP_OVF]);
  else {
    strcpy(&tmpsp[tmpspsz], s);
    tmpspsz += l + 1;
    return (k);
  }
}

extern char *_realname[];
extern int _modtb[], _fnametb[];
extern int _modtbsz;

static byte imps[MAXFILES];

static void
putimp1(int mno, int b)
{
  if (!imps[mno])
    imps[mno] |= 4 | (b?2:0);
  else if (!b)
    imps[mno] &= ~2;
}

void
putimp(int mno, int b)
{
  int i;
  putimp1(mno, b);
  for (i = 0; i < _imports[mno].sz; i++)
    if (_imports[mno].ib[i] & 1)
      putimp1(_imports[mno].tb[i], b);
}

void
putinc(int mno, int b)
{
  int i;
  putimp1(mno, b);
  imps[mno] |= 1;
  for (i = 0; i < _imports[mno].sz; i++)
    if (_imports[mno].ib[i] & 1) {
      putimp1(_imports[mno].tb[i], b);
      imps[_imports[mno].tb[i]] |= 1;
    }
}

void saveimps(void)
{
  int i, sz;
  for (i = sz = 0; i < _modtbsz; i++)
    if (imps[i]) sz++;
  if (_imports[modno].tb) {
    free(_imports[modno].tb);
    free(_imports[modno].ib);
  }
  _imports[modno].tb = NULL; _imports[modno].ib = NULL;
  if (sz > 0)
    if (!(_imports[modno].tb = aalloc(sz, sizeof(int))) ||
	!(_imports[modno].ib = aalloc(sz, sizeof(byte))))
      fatal(qcmsg[MEM_OVF]);
  for (i = sz = 0; i < _modtbsz; i++)
    if (imps[i]) {
      _imports[modno].tb[sz] = i;
      _imports[modno].ib[sz] = imps[i] & 3;
      sz++;
    }
  _imports[modno].sz = sz;
  for (i = 0; i < _modtbsz; i++)
    imps[i] = 0;
}

void restoreimps(void)
{
  int i;
  for (i = 0; i < _imports[modno].sz; i++)
    imps[_imports[modno].tb[i]] = _imports[modno].ib[i] | 4;
}

static int
strhash(char *s, int sz)
{
  unsigned h = 0, g;
  while (*s) {
    h = (h<<4)+*(s++);
    if ((g = (h & 0xf0000000)))	{
      h = h^(g>>24);
      h = h^g;
    }
  }
  return h % sz;
}

int xxxsym(int sym)
{
  if (sym != NONE)
    while (symtb[sym].ref)
      sym = symtb[sym].ref;
  return sym;
}

int
getsym(int mno, char *s)
{
  int k = strhash(s, hashtbsz);
  int sym, sym1 = NONE;

  /* look for qualified symbol in given module */
  for (sym = hashtb[k]; sym != NONE; sym = symtb[sym].next)
    if (strcmp(s, strsp+symtb[sym].pname) == 0 && symtb[sym].modno == mno &&
	!(symtb[sym].flags & PRIV)) {
      sym1 = sym;
      break;
    }
  sym = sym1;
  if (sym != NONE)
    while (symtb[sym].ref)
      sym = symtb[sym].ref;
  return sym;
}

static int
checksym(int mno, int sym, int sym0)
{
  char *s;
  int sym1 = NONE;

  if (sym == NONE || sym0 == NONE) return 0;
  while (symtb[sym0].ref) sym0 = symtb[sym0].ref;
  s = strsp+symtb[sym].pname;
  for (sym = symtb[sym].next; sym != NONE; sym = symtb[sym].next)
    if (strcmp(s, strsp+symtb[sym].pname) == 0 && symtb[sym].modno == mno) {
      int sym2 = sym;
      while (symtb[sym2].ref) sym2 = symtb[sym2].ref;
      if (sym2 != sym0) {
	sym1 = sym;
	break;
      }
    }
  return sym1 != NONE;
}

#define streq(s1,s2) (!strcmp(s1,s2))

#define NIL (-2)

static int
getmodno(char *s)
{
  int i;
  if (modno >= 0 && strcmp(s, _realname[modno]) == 0)
    /* this is a "self" qualifier */
    return modno;
  for (i = 0; i < _modtbsz; i++)
    if (strcmp(s, strsp+_modtb[i]) == 0)
      return i;
  return NIL;
}

#define ANY (-3)

static int
splitid(char *s, char *mnm)
{
  char *p;
  int mno = ANY;
  *mnm = 0;
  if ((p = strstr(s, "::"))) {
    char t[MAXSTRLEN];
    strcpy(t, p+2);
    *p = 0;
    strcpy(mnm, s);
    if (!*s)
      mno = NONE;
    else {
      mno = getmodno(s);
      if (mno != NIL && mno != modno && !imps[mno])
	mno = NIL;
    }
    strcpy(s, t);
  }
  return mno;
}

static
symprio(int sym)
{
  if (symtb[sym].modno == NONE)
    return -1;
  else if (imps[symtb[sym].modno] & 2)
    return 0;
  else
    return 1;
}

#define matchfun(fno,s) (!(symtb[fno].flags & TSYM)&&\
			 streq((s), strsp+symtb[fno].pname))

int
getfun(char *s)
{
  char msg[MAXSTRLEN], mnm[MAXSTRLEN];
  int mno = splitid(s, mnm);
  int k = strhash(s, hashtbsz);
  int fno, fno1 = NONE;

  if (mno == NIL) {
    /* bad module name */
    return NONE;
  } else if (mno == ANY) {
    /* look for symbol in all imported modules */
    for (fno = hashtb[k]; fno != NONE; fno = symtb[fno].next)
      if (matchfun(fno, s) &&
	  (symtb[fno].modno == NONE || symtb[fno].modno == modno ||
	   imps[symtb[fno].modno]))
	if (symtb[fno].modno == modno) {
	  /* found symbol in current module, done */
	  fno1 = fno;
	  break;
	} else if (symtb[fno].flags & PRIV)
	  /* private symbol in other module, skip */
	  ;
	else if (fno1 != NONE) {
	  int r1 = fno1, r = fno;
	  while (symtb[r1].ref) r1 = symtb[r1].ref;
	  while (symtb[r].ref) r = symtb[r].ref;
	  if (r1 == r)
	    /* aliases for same symbol, skip */
	    ;
	  else if (symprio(fno) == symprio(fno1)) {
	    /* symbol imported from multiple modules */
	    return NONE;
	  } else
	    break;
	} else
	  fno1 = fno;
  } else {
    /* look for qualified symbol in given module */
    for (fno = hashtb[k]; fno != NONE; fno = symtb[fno].next)
      if (matchfun(fno, s) && symtb[fno].modno == mno) {
	if (!(symtb[fno].flags & PRIV) || mno == modno)
	  fno1 = fno;
	break;
      }
  }
  fno = fno1;
  if (fno != NONE)
    while (symtb[fno].ref)
      fno = symtb[fno].ref;
  return fno;
}

static int dcontext = 0;

extern int yylineno;

int
mkfun(char *s)
{
  char msg[MAXSTRLEN], mnm[MAXSTRLEN];
  int mno = splitid(s, mnm);
  int k = strhash(s, hashtbsz);
  int fno, fno1 = NONE;

  if (mno == NIL) {
    /* error: bad module name */
    sprintf(msg, qcmsg[UNKNOWN_REF], utf8_to_sys(mnm));
    yyerror(msg);
    return NONE;
  } else if (mno == ANY) {
    /* look for symbol in all imported modules */
    for (fno = hashtb[k]; fno != NONE; fno = symtb[fno].next)
      if (matchfun(fno, s) &&
	  (symtb[fno].modno == NONE || symtb[fno].modno == modno ||
	   imps[symtb[fno].modno]))
	if (symtb[fno].modno == modno) {
	  /* found symbol in current module, done */
	  fno1 = fno;
	  break;
	} else if (symtb[fno].flags & PRIV)
	  /* private symbol in other module, skip */
	  ;
	else if (fno1 != NONE) {
	  int r1 = fno1, r = fno;
	  while (symtb[r1].ref) r1 = symtb[r1].ref;
	  while (symtb[r].ref) r = symtb[r].ref;
	  if (r1 == r)
	    /* aliases for same symbol, skip */
	    ;
	  else if (!dcontext && symprio(fno) == symprio(fno1)) {
	    /* symbol imported from multiple modules, error */
	    sprintf(msg, qcmsg[AMBIG_IMP], utf8_to_sys(s));
	    yyerror(msg);
	    return NONE;
	  } else
	    break;
	} else
	  fno1 = fno;
  } else {
    /* look for qualified symbol in given module */
    for (fno = hashtb[k]; fno != NONE; fno = symtb[fno].next)
      if (matchfun(fno, s) && symtb[fno].modno == mno) {
	if (!(symtb[fno].flags & PRIV) || mno == modno)
	  fno1 = fno;
	break;
      }
  }
  fno = fno1;
  if (fno == NONE) {
    if (mno != ANY && mno != modno) {
      /* error: undeclared symbol */
      sprintf(msg, qcmsg[MISS_DCL], utf8_to_sys(s), utf8_to_sys(mnm));
      yyerror(msg);
      return NONE;
    }
    /* create a new private symbol in the current module */
    if (symtbsz > SHRT_MAX) fatal(qcmsg[SYMTB_OVF]);
    if (symtbsz >= asymtbsz)
      if (!(symtb = (SYMREC *)arealloc(symtb, asymtbsz,
				       SYMTBSZ/10, sizeof(SYMREC))))
	fatal(qcmsg[SYMTB_OVF]);
      else
	asymtbsz += SYMTBSZ/10;
    fno = symtbsz++;
    symtb[fno].ref = 0;
    symtb[fno].prec = NONE;
    symtb[fno].flags = PRIV;
    symtb[fno].type = 0;
    symtb[fno].fno_min = symtb[fno].fno_max = 0;
    symtb[fno].argc = 0;
    symtb[fno].argv = 0;
    symtb[fno].modno = modno;
    symtb[fno].lineno = yylineno;
    symtb[fno].xfno = fno;
    symtb[fno].pname = putstr(s);
    symtb[fno].x = symtb[fno].xp = NULL;
    symtb[fno].f = NULL;
    symtb[fno].next = hashtb[k];
    hashtb[k] = fno;
  } else {
    int sym_modno = symtb[fno].modno;
    while (symtb[fno].ref)
      fno = symtb[fno].ref;
    if (!dcontext && wflag >= 2 && mno == ANY && mainno >= 0 &&
	(wflag >= 3 && symtb[fno].modno >= 0 ||
	 symtb[fno].modno >= mainno) && sym_modno != modno) {
      sprintf(msg, qcmsg[UNQUALIFIED_REF], utf8_to_sys(s),
	      utf8_to_sys(strsp+_modtb[sym_modno]));
      yywarn(msg);
    }
  }
  return fno;
}

/* Same as above, but return an unnormalized symbol (i.e., plain symbol "as
   is", without resolving aliases). We need this to properly handle alias
   declarations. */

int
mkxxxfun(char *s)
{
  char msg[MAXSTRLEN], mnm[MAXSTRLEN];
  int mno = splitid(s, mnm);
  int k = strhash(s, hashtbsz);
  int fno, fno1 = NONE;

  if (mno == NIL) {
    /* error: bad module name */
    sprintf(msg, qcmsg[UNKNOWN_REF], utf8_to_sys(mnm));
    yyerror(msg);
    return NONE;
  } else if (mno == ANY) {
    /* look for symbol in all imported modules */
    for (fno = hashtb[k]; fno != NONE; fno = symtb[fno].next)
      if (matchfun(fno, s) &&
	  (symtb[fno].modno == NONE || symtb[fno].modno == modno ||
	   imps[symtb[fno].modno]))
	if (symtb[fno].modno == modno) {
	  /* found symbol in current module, done */
	  fno1 = fno;
	  break;
	} else if (symtb[fno].flags & PRIV)
	  /* private symbol in other module, skip */
	  ;
	else if (fno1 != NONE) {
	  int r1 = fno1, r = fno;
	  while (symtb[r1].ref) r1 = symtb[r1].ref;
	  while (symtb[r].ref) r = symtb[r].ref;
	  if (r1 == r)
	    /* aliases for same symbol, skip */
	    ;
	  else if (!dcontext && symprio(fno) == symprio(fno1)) {
	    /* symbol imported from multiple modules, error */
	    sprintf(msg, qcmsg[AMBIG_IMP], utf8_to_sys(s));
	    yyerror(msg);
	    return NONE;
	  } else
	    break;
	} else
	  fno1 = fno;
  } else {
    /* look for qualified symbol in given module */
    for (fno = hashtb[k]; fno != NONE; fno = symtb[fno].next)
      if (matchfun(fno, s) && symtb[fno].modno == mno) {
	if (!(symtb[fno].flags & PRIV) || mno == modno)
	  fno1 = fno;
	break;
      }
  }
  fno = fno1;
  if (fno == NONE) {
    if (mno != ANY && mno != modno) {
      /* error: undeclared symbol */
      sprintf(msg, qcmsg[MISS_DCL], utf8_to_sys(s), utf8_to_sys(mnm));
      yyerror(msg);
      return NONE;
    }
    /* create a new private symbol in the current module */
    if (symtbsz > SHRT_MAX) fatal(qcmsg[SYMTB_OVF]);
    if (symtbsz >= asymtbsz)
      if (!(symtb = (SYMREC *)arealloc(symtb, asymtbsz,
				       SYMTBSZ/10, sizeof(SYMREC))))
	fatal(qcmsg[SYMTB_OVF]);
      else
	asymtbsz += SYMTBSZ/10;
    fno = symtbsz++;
    symtb[fno].ref = 0;
    symtb[fno].prec = NONE;
    symtb[fno].flags = PRIV;
    symtb[fno].type = 0;
    symtb[fno].fno_min = symtb[fno].fno_max = 0;
    symtb[fno].argc = 0;
    symtb[fno].argv = 0;
    symtb[fno].modno = modno;
    symtb[fno].lineno = yylineno;
    symtb[fno].xfno = fno;
    symtb[fno].pname = putstr(s);
    symtb[fno].x = symtb[fno].xp = NULL;
    symtb[fno].f = NULL;
    symtb[fno].next = hashtb[k];
    hashtb[k] = fno;
  } else {
    int sym = fno, sym_modno = symtb[fno].modno;
    while (symtb[sym].ref)
      sym = symtb[sym].ref;
    if (!dcontext && wflag >= 2 && mno == ANY && mainno >= 0 &&
	(wflag >= 3 && symtb[fno].modno >= 0 ||
	 symtb[sym].modno >= mainno) && sym_modno != modno) {
      sprintf(msg, qcmsg[UNQUALIFIED_REF], utf8_to_sys(s),
	      utf8_to_sys(strsp+_modtb[sym_modno]));
      yywarn(msg);
    }
  }
  return fno;
}

int
newfun(char *s, int fno)
{
  int             k = strhash(s, hashtbsz);

  /* create a new public symbol in the current module */
  if (fno > SHRT_MAX) fatal(qcmsg[SYMTB_OVF]);
  while (fno >= asymtbsz)
    if (!(symtb = (SYMREC *)arealloc(symtb, asymtbsz,
				     SYMTBSZ/10, sizeof(SYMREC))))
      fatal(qcmsg[SYMTB_OVF]);
    else
      asymtbsz += SYMTBSZ/10;
  while (fno > symtbsz) {
    symtb[symtbsz].ref = 0;
    symtb[symtbsz].prec = NONE;
    symtb[symtbsz].flags = 0;
    symtb[symtbsz].type = 0;
    symtb[symtbsz].fno_min = symtb[symtbsz].fno_max = 0;
    symtb[symtbsz].argc = 0;
    symtb[symtbsz].argv = 0;
    symtb[symtbsz].modno = NONE;
    symtb[symtbsz].lineno = 0;
    symtb[symtbsz].xfno = symtbsz;
    symtb[symtbsz].pname = 0;
    symtb[symtbsz].x = symtb[symtbsz].xp = NULL;
    symtb[symtbsz].f = NULL;
    symtb[symtbsz].next = NONE;
    symtbsz++;
  }
  symtb[fno].ref = 0;
  symtb[fno].prec = NONE;
  symtb[fno].flags = 0;
  symtb[fno].type = 0;
  symtb[fno].fno_min = symtb[fno].fno_max = 0;
  symtb[fno].argc = 0;
  symtb[fno].argv = 0;
  symtb[fno].modno = modno;
  symtb[fno].lineno = yylineno;
  symtb[fno].xfno = fno;
  symtb[fno].pname = putstr(s);
  symtb[fno].x = symtb[fno].xp = NULL;
  symtb[fno].f = NULL;
  symtb[fno].next = hashtb[k];
  hashtb[k] = fno;
  symtbsz++;
  return fno;
}

void
asfun(int fno, int fno2, byte argc, unsigned long argv,
      short flags, int prec)
{
  char msg[MAXSTRLEN], *s;
  if (fno == NONE || fno2 == NONE) return;
  s = strsp+symtb[fno].pname;
  fno = xxxsym(fno);
  flags |= DCL; symtb[fno].flags &= ~FWD;
  if (!argv) flags &= ~SPEC;
  if (!fno2) {
    if (symtb[fno].modno != modno)
      fno2 = newfun(s, symtbsz);
  } else if (symtb[fno2].modno != modno)
    fno2 = newfun(strsp+symtb[fno2].pname, symtbsz);
  if ((symtb[fno].flags & ~(EXT|PRIV)) != (flags & ~(EXT|PRIV)) || 
      symtb[fno].prec != prec ||
      (symtb[fno].argc != argc ||
       symtb[fno].argv != argv)) {
    sprintf(msg, qcmsg[MISM_DCL], utf8_to_sys(strsp+symtb[fno].pname));
    yyerror(msg);
  } else if ((flags & EXT) &&
	     (symtb[fno].flags & EXT)) {
    sprintf(msg, qcmsg[MISM_DCL],
	    utf8_to_sys(strsp+symtb[fno].pname));
    yyerror(msg);
  } else if (checksym(modno, fno2, fno)) {
    sprintf(msg, qcmsg[MISM_DCL],
	    utf8_to_sys(strsp+symtb[fno2].pname));
    yyerror(msg);
  } else if (flags & EXT) {
    symtb[fno].flags |= EXT;
    symtb[fno].xfno = fno2;
  }
  if (!fno2) return;
  symtb[fno2].flags = flags & ~(EXT|FWD);
  symtb[fno2].type = symtb[fno].type;
  symtb[fno2].argc = argc;
  symtb[fno2].argv = argv;
  symtb[fno2].prec = prec;
  symtb[fno2].ref = fno;
}

int
dclfun(int fno, int type, byte argc, unsigned long argv,
       short flags, int prec)
{
  char msg[MAXSTRLEN];
  if (fno == NONE) return NONE;
  flags |= DCL; symtb[fno].flags &= ~FWD;
  if (!argv) flags &= ~SPEC;
  if (symtb[fno].modno != modno)
    fno = newfun(strsp+symtb[fno].pname, symtbsz);
  if (symtb[fno].flags & DCL) {
    if ((symtb[fno].flags & ~EXT) != (flags & ~EXT) || 
	symtb[fno].prec != prec ||
	symtb[fno].type != type && !(flags & EXT) ||
	(symtb[fno].argc != argc ||
	 symtb[fno].argv != argv)) {
      sprintf(msg, qcmsg[MISM_DCL], utf8_to_sys(strsp+symtb[fno].pname));
      yyerror(msg);
    } else if ((flags & EXT) &&
	       !(symtb[fno].flags & EXT) &&
	       symtb[fno].modno != modno) {
      sprintf(msg, qcmsg[MISM_DCL],
	      utf8_to_sys(strsp+symtb[fno].pname));
      yyerror(msg);
    } else if (type) {
      sprintf(msg, qcmsg[DOUBLE_DCL],
	      utf8_to_sys(strsp+symtb[fno].pname));
      yyerror(msg);
    } else if (checksym(modno, fno, fno)) {
      sprintf(msg, qcmsg[MISM_DCL],
	      utf8_to_sys(strsp+symtb[fno].pname));
      yyerror(msg);
    } else if ((flags & EXT) && !(symtb[fno].flags & EXT)) {
      symtb[fno].flags |= EXT;
    }
  } else {
    if (checksym(modno, fno, fno)) {
      sprintf(msg, qcmsg[MISM_DCL],
	      utf8_to_sys(strsp+symtb[fno].pname));
      yyerror(msg);
    }
    symtb[fno].flags = flags;
    symtb[fno].prec = prec;
    symtb[fno].type = type;
    if (!symtb[fno].argc)
      symtb[fno].argc = argc;
    symtb[fno].argv = argv;
    for (; type && argc>symtb[type].argc; type = symtb[type].type)
      symtb[type].argc = argc;
  }
  return fno;
}

void
unresolved_forwards(void)
{
  int fno, save_lineno = yylineno;
  for (fno = 0; fno < symtbsz && symtb[fno].modno != modno; fno++) ;
  while (fno < symtbsz) {
    if (symtb[fno].modno == modno && (symtb[fno].flags & FWD)) {
      char msg[MAXSTRLEN];
      sprintf(msg, qcmsg[UNDECLARED_FUN],
	      utf8_to_sys(strsp + symtb[fno].pname));
      yylineno = symtb[fno].lineno;
      yywarn(msg);
      symtb[fno].flags &= ~FWD;
    }
    fno++;
  }
  yylineno = save_lineno;
}

int
mkfvar(char *s)
{
  char msg[MAXSTRLEN], mnm[MAXSTRLEN];
  int mno = splitid(s, mnm);
  int k = strhash(s, hashtbsz);
  int vno, vno1 = NONE;

  if (mno == NIL) {
    /* error: bad module name */
    sprintf(msg, qcmsg[UNKNOWN_REF], utf8_to_sys(mnm));
    yyerror(msg);
    return NONE;
  } else if (mno == ANY) {
    /* look for symbol in all imported modules */
    for (vno = hashtb[k]; vno != NONE; vno = symtb[vno].next)
      if (matchfun(vno, s) &&
	  (symtb[vno].modno == NONE || symtb[vno].modno == modno ||
	   imps[symtb[vno].modno]))
	if (symtb[vno].modno == modno) {
	  /* found symbol in current module, done */
	  vno1 = vno;
	  break;
	} else if (symtb[vno].flags & PRIV)
	  /* private symbol in other module, skip */
	  ;
	else if (vno1 != NONE) {
	  int r1 = vno1, r = vno;
	  while (symtb[r1].ref) r1 = symtb[r1].ref;
	  while (symtb[r].ref) r = symtb[r].ref;
	  if (r1 == r)
	    /* aliases for same symbol, skip */
	    ;
	  else if (!dcontext && symprio(vno) == symprio(vno1)) {
	    /* symbol imported from multiple modules, error */
	    sprintf(msg, qcmsg[AMBIG_IMP], utf8_to_sys(s));
	    yyerror(msg);
	    return NONE;
	  } else
	    break;
	} else
	  vno1 = vno;
  } else {
    /* look for qualified symbol in given module */
    for (vno = hashtb[k]; vno != NONE; vno = symtb[vno].next)
      if (matchfun(vno, s) && symtb[vno].modno == mno) {
	if (!(symtb[vno].flags & PRIV) || mno == modno)
	  vno1 = vno;
	break;
      }
  }
  vno = vno1;
  if (vno == NONE) {
    if (mno != ANY && mno != modno) {
      /* error: undeclared symbol */
      sprintf(msg, qcmsg[MISS_DCL], utf8_to_sys(s), utf8_to_sys(mnm));
      yyerror(msg);
      return NONE;
    }
    /* create a new private symbol in the current module */
    if (symtbsz > SHRT_MAX) fatal(qcmsg[SYMTB_OVF]);
    if (symtbsz >= asymtbsz)
      if (!(symtb = (SYMREC *)arealloc(symtb, asymtbsz,
				       SYMTBSZ/10, sizeof(SYMREC))))
	fatal(qcmsg[SYMTB_OVF]);
      else
	asymtbsz += SYMTBSZ/10;
    vno = symtbsz++;
    symtb[vno].ref = 0;
    symtb[vno].prec = NONE;
    symtb[vno].flags = PRIV|VSYM;
    symtb[vno].type = 0;
    symtb[vno].fno_min = symtb[vno].fno_max = 0;
    symtb[vno].argc = 0;
    symtb[vno].argv = 0;
    symtb[vno].modno = modno;
    symtb[vno].lineno = yylineno;
    symtb[vno].xfno = vno;
    symtb[vno].pname = putstr(s);
    symtb[vno].x = symtb[vno].xp = NULL;
    symtb[vno].f = NULL;
    symtb[vno].next = hashtb[k];
    hashtb[k] = vno;
  } else {
    int sym_modno = symtb[vno].modno;
    while (symtb[vno].ref)
      vno = symtb[vno].ref;
    if (!dcontext && wflag >= 2 && mno == ANY && mainno >= 0 &&
	(wflag >= 3 && symtb[vno].modno >= 0 ||
	 symtb[vno].modno >= mainno) && sym_modno != modno) {
      sprintf(msg, qcmsg[UNQUALIFIED_REF], utf8_to_sys(s),
	      utf8_to_sys(strsp+_modtb[sym_modno]));
      yywarn(msg);
    }
  }
  return vno;
}

int
mkxxxfvar(char *s)
{
  char msg[MAXSTRLEN], mnm[MAXSTRLEN];
  int mno = splitid(s, mnm);
  int k = strhash(s, hashtbsz);
  int vno, vno1 = NONE;

  if (mno == NIL) {
    /* error: bad module name */
    sprintf(msg, qcmsg[UNKNOWN_REF], utf8_to_sys(mnm));
    yyerror(msg);
    return NONE;
  } else if (mno == ANY) {
    /* look for symbol in all imported modules */
    for (vno = hashtb[k]; vno != NONE; vno = symtb[vno].next)
      if (matchfun(vno, s) &&
	  (symtb[vno].modno == NONE || symtb[vno].modno == modno ||
	   imps[symtb[vno].modno]))
	if (symtb[vno].modno == modno) {
	  /* found symbol in current module, done */
	  vno1 = vno;
	  break;
	} else if (symtb[vno].flags & PRIV)
	  /* private symbol in other module, skip */
	  ;
	else if (vno1 != NONE) {
	  int r1 = vno1, r = vno;
	  while (symtb[r1].ref) r1 = symtb[r1].ref;
	  while (symtb[r].ref) r = symtb[r].ref;
	  if (r1 == r)
	    /* aliases for same symbol, skip */
	    ;
	  else if (!dcontext && symprio(vno) == symprio(vno1)) {
	    /* symbol imported from multiple modules, error */
	    sprintf(msg, qcmsg[AMBIG_IMP], utf8_to_sys(s));
	    yyerror(msg);
	    return NONE;
	  } else
	    break;
	} else
	  vno1 = vno;
  } else {
    /* look for qualified symbol in given module */
    for (vno = hashtb[k]; vno != NONE; vno = symtb[vno].next)
      if (matchfun(vno, s) && symtb[vno].modno == mno) {
	if (!(symtb[vno].flags & PRIV) || mno == modno)
	  vno1 = vno;
	break;
      }
  }
  vno = vno1;
  if (vno == NONE) {
    if (mno != ANY && mno != modno) {
      /* error: undeclared symbol */
      sprintf(msg, qcmsg[MISS_DCL], utf8_to_sys(s), utf8_to_sys(mnm));
      yyerror(msg);
      return NONE;
    }
    /* create a new private symbol in the current module */
    if (symtbsz > SHRT_MAX) fatal(qcmsg[SYMTB_OVF]);
    if (symtbsz >= asymtbsz)
      if (!(symtb = (SYMREC *)arealloc(symtb, asymtbsz,
				       SYMTBSZ/10, sizeof(SYMREC))))
	fatal(qcmsg[SYMTB_OVF]);
      else
	asymtbsz += SYMTBSZ/10;
    vno = symtbsz++;
    symtb[vno].ref = 0;
    symtb[vno].prec = NONE;
    symtb[vno].flags = PRIV|VSYM;
    symtb[vno].type = 0;
    symtb[vno].fno_min = symtb[vno].fno_max = 0;
    symtb[vno].argc = 0;
    symtb[vno].argv = 0;
    symtb[vno].modno = modno;
    symtb[vno].lineno = yylineno;
    symtb[vno].xfno = vno;
    symtb[vno].pname = putstr(s);
    symtb[vno].x = symtb[vno].xp = NULL;
    symtb[vno].f = NULL;
    symtb[vno].next = hashtb[k];
    hashtb[k] = vno;
  } else {
    int sym = vno, sym_modno = symtb[vno].modno;
    while (symtb[sym].ref)
      sym = symtb[sym].ref;
    if (!dcontext && wflag >= 2 && mno == ANY && mainno >= 0 &&
	(wflag >= 3 && symtb[vno].modno >= 0 ||
	 symtb[sym].modno >= mainno) && sym_modno != modno) {
      sprintf(msg, qcmsg[UNQUALIFIED_REF], utf8_to_sys(s),
	      utf8_to_sys(strsp+_modtb[sym_modno]));
      yywarn(msg);
    }
  }
  return vno;
}

int
newfvar(char *s, int vno)
{
  int             k = strhash(s, hashtbsz);

  if (vno > SHRT_MAX) fatal(qcmsg[SYMTB_OVF]);
  while (vno >= asymtbsz)
    if (!(symtb = (SYMREC *)arealloc(symtb, asymtbsz,
				     SYMTBSZ/10, sizeof(SYMREC))))
      fatal(qcmsg[SYMTB_OVF]);
    else
      asymtbsz += SYMTBSZ/10;
  while (vno > symtbsz) {
    symtb[symtbsz].ref = 0;
    symtb[symtbsz].prec = NONE;
    symtb[symtbsz].flags = 0;
    symtb[symtbsz].type = 0;
    symtb[symtbsz].fno_min = symtb[symtbsz].fno_max = 0;
    symtb[symtbsz].argc = 0;
    symtb[symtbsz].argv = 0;
    symtb[symtbsz].modno = NONE;
    symtb[symtbsz].lineno = 0;
    symtb[symtbsz].xfno = symtbsz;
    symtb[symtbsz].pname = 0;
    symtb[symtbsz].x = symtb[symtbsz].xp = NULL;
    symtb[symtbsz].f = NULL;
    symtb[symtbsz].next = NONE;
    symtbsz++;
  }
  symtb[vno].ref = 0;
  symtb[vno].prec = NONE;
  symtb[vno].flags = VSYM;
  symtb[vno].type = 0;
  symtb[vno].fno_min = symtb[vno].fno_max = 0;
  symtb[vno].argc = 0;
  symtb[vno].argv = 0;
  symtb[vno].modno = modno;
  symtb[vno].lineno = yylineno;
  symtb[vno].xfno = vno;
  symtb[vno].pname = putstr(s);
  symtb[vno].x = symtb[vno].xp = NULL;
  symtb[vno].f = NULL;
  symtb[vno].next = hashtb[k];
  hashtb[k] = vno;
  symtbsz++;
  return vno;
}

void
asfvar(int vno, int vno2, short flags)
{
  char msg[MAXSTRLEN], *s;
  if (vno == NONE || vno2 == NONE) return;
  s = strsp+symtb[vno].pname;
  vno = xxxsym(vno);
  flags |= DCL|VSYM;
  if (!vno2)
    vno2 = newfvar(s, symtbsz);
  else if (symtb[vno2].modno != modno)
    vno2 = newfvar(strsp+symtb[vno2].pname, symtbsz);
  if ((symtb[vno].flags & ~PRIV) != (flags & ~PRIV)) {
    sprintf(msg, qcmsg[MISM_DCL],
	    utf8_to_sys(strsp+symtb[vno].pname));
    yyerror(msg);
  } else if (checksym(modno, vno2, vno)) {
    sprintf(msg, qcmsg[MISM_DCL],
	    utf8_to_sys(strsp+symtb[vno2].pname));
    yyerror(msg);
  }
  symtb[vno2].flags = flags;
  symtb[vno2].type = 0;
  symtb[vno2].argc = 0;
  symtb[vno2].ref = vno;
}

int
dclfvar(int vno, short flags)
{
  char msg[MAXSTRLEN];
  if (vno == NONE) return NONE;
  flags |= DCL|VSYM;
  if (symtb[vno].modno != modno)
    vno = newfvar(strsp+symtb[vno].pname, symtbsz);
  if (symtb[vno].flags & DCL) {
    if (symtb[vno].flags != flags) {
      sprintf(msg, qcmsg[MISM_DCL],
	      utf8_to_sys(strsp+symtb[vno].pname));
      yyerror(msg);
    } else if (checksym(modno, vno, vno)) {
      sprintf(msg, qcmsg[MISM_DCL],
	      utf8_to_sys(strsp+symtb[vno].pname));
      yyerror(msg);
    }
  } else {
    if (checksym(modno, vno, vno)) {
      sprintf(msg, qcmsg[MISM_DCL],
	      utf8_to_sys(strsp+symtb[vno].pname));
      yyerror(msg);
    }
    symtb[vno].flags = flags;
    symtb[vno].type = 0;
    symtb[vno].argc = 0;
  }
  return vno;
}

#define matchtyp(type,s) ((symtb[type].flags & TSYM)&&\
			  streq((s), strsp+symtb[type].pname))

int
mktype(char *s)
{
  char msg[MAXSTRLEN], mnm[MAXSTRLEN];
  int mno = splitid(s, mnm);
  int k = strhash(s, hashtbsz);
  int type, type1 = NONE;

  if (mno == NIL) {
    /* error: bad module name */
    sprintf(msg, qcmsg[UNKNOWN_REF], mnm);
    yyerror(msg);
    return NONE;
  } else if (mno == ANY) {
    /* look for symbol in all imported modules */
    for (type = hashtb[k]; type != NONE; type = symtb[type].next)
      if (matchtyp(type, s) &&
	  (symtb[type].modno == NONE || symtb[type].modno == modno ||
	   imps[symtb[type].modno]))
	if (symtb[type].modno == modno) {
	  /* found symbol in current module, done */
	  type1 = type;
	  break;
	} else if (symtb[type].flags & PRIV)
	  /* private symbol in other module, skip */
	  ;
	else if (type1 != NONE) {
	  int r1 = type1, r = type;
	  while (symtb[r1].ref) r1 = symtb[r1].ref;
	  while (symtb[r].ref) r = symtb[r].ref;
	  if (r1 == r)
	    /* aliases for same symbol, skip */
	    ;
	  else if (!dcontext && symprio(type) == symprio(type1)) {
	    /* symbol imported from multiple modules, error */
	    sprintf(msg, qcmsg[AMBIG_IMP], utf8_to_sys(s));
	    yyerror(msg);
	    return NONE;
	  } else
	    break;
	} else
	  type1 = type;
  } else {
    /* look for qualified symbol in given module */
    for (type = hashtb[k]; type != NONE; type = symtb[type].next)
      if (matchtyp(type, s) && symtb[type].modno == mno) {
	if (!(symtb[type].flags & PRIV) || mno == modno)
	  type1 = type;
	break;
      }
  }
  type = type1;
  if (type == NONE) {
    if (mno != ANY && mno != modno) {
      /* error: undeclared symbol */
      sprintf(msg, qcmsg[MISS_DCL], utf8_to_sys(s), utf8_to_sys(mnm));
      yyerror(msg);
      return NONE;
    }
    /* create a new private symbol in the current module */
    if (symtbsz > SHRT_MAX) fatal(qcmsg[SYMTB_OVF]);
    if (symtbsz >= asymtbsz)
      if (!(symtb = (SYMREC *)arealloc(symtb, asymtbsz,
				       SYMTBSZ/10, sizeof(SYMREC))))
	fatal(qcmsg[SYMTB_OVF]);
      else
	asymtbsz += SYMTBSZ/10;
    type = symtbsz++;
    symtb[type].ref = 0;
    symtb[type].prec = NONE;
    symtb[type].flags = PRIV|TSYM;
    symtb[type].type = 0;
    symtb[type].fno_min = symtb[type].fno_max = 0;
    symtb[type].argc = 0;
    symtb[type].argv = 0;
    symtb[type].modno = modno;
    symtb[type].lineno = yylineno;
    symtb[type].xfno = type;
    symtb[type].pname = putstr(s);
    symtb[type].x = symtb[type].xp = NULL;
    symtb[type].f = NULL;
    symtb[type].next = hashtb[k];
    hashtb[k] = type;
  } else {
    int sym_modno = symtb[type].modno;
    while (symtb[type].ref)
      type = symtb[type].ref;
    if (!dcontext && wflag >= 2 && mno == ANY && mainno >= 0 &&
	(wflag >= 3 && symtb[type].modno >= 0 ||
	 symtb[type].modno >= mainno) && sym_modno != modno) {
      sprintf(msg, qcmsg[UNQUALIFIED_REF], utf8_to_sys(s),
	      utf8_to_sys(strsp+_modtb[sym_modno]));
      yywarn(msg);
    }
  }
  return type;
}

int
mkxxxtype(char *s)
{
  char msg[MAXSTRLEN], mnm[MAXSTRLEN];
  int mno = splitid(s, mnm);
  int k = strhash(s, hashtbsz);
  int type, type1 = NONE;

  if (mno == NIL) {
    /* error: bad module name */
    sprintf(msg, qcmsg[UNKNOWN_REF], mnm);
    yyerror(msg);
    return NONE;
  } else if (mno == ANY) {
    /* look for symbol in all imported modules */
    for (type = hashtb[k]; type != NONE; type = symtb[type].next)
      if (matchtyp(type, s) &&
	  (symtb[type].modno == NONE || symtb[type].modno == modno ||
	   imps[symtb[type].modno]))
	if (symtb[type].modno == modno) {
	  /* found symbol in current module, done */
	  type1 = type;
	  break;
	} else if (symtb[type].flags & PRIV)
	  /* private symbol in other module, skip */
	  ;
	else if (type1 != NONE) {
	  int r1 = type1, r = type;
	  while (symtb[r1].ref) r1 = symtb[r1].ref;
	  while (symtb[r].ref) r = symtb[r].ref;
	  if (r1 == r)
	    /* aliases for same symbol, skip */
	    ;
	  else if (!dcontext && symprio(type) == symprio(type1)) {
	    /* symbol imported from multiple modules, error */
	    sprintf(msg, qcmsg[AMBIG_IMP], utf8_to_sys(s));
	    yyerror(msg);
	    return NONE;
	  } else
	    break;
	} else
	  type1 = type;
  } else {
    /* look for qualified symbol in given module */
    for (type = hashtb[k]; type != NONE; type = symtb[type].next)
      if (matchtyp(type, s) && symtb[type].modno == mno) {
	if (!(symtb[type].flags & PRIV) || mno == modno)
	  type1 = type;
	break;
      }
  }
  type = type1;
  if (type == NONE) {
    if (mno != ANY && mno != modno) {
      /* error: undeclared symbol */
      sprintf(msg, qcmsg[MISS_DCL], utf8_to_sys(s), utf8_to_sys(mnm));
      yyerror(msg);
      return NONE;
    }
    /* create a new private symbol in the current module */
    if (symtbsz > SHRT_MAX) fatal(qcmsg[SYMTB_OVF]);
    if (symtbsz >= asymtbsz)
      if (!(symtb = (SYMREC *)arealloc(symtb, asymtbsz,
				       SYMTBSZ/10, sizeof(SYMREC))))
	fatal(qcmsg[SYMTB_OVF]);
      else
	asymtbsz += SYMTBSZ/10;
    type = symtbsz++;
    symtb[type].ref = 0;
    symtb[type].prec = NONE;
    symtb[type].flags = PRIV|TSYM;
    symtb[type].type = 0;
    symtb[type].fno_min = symtb[type].fno_max = 0;
    symtb[type].argc = 0;
    symtb[type].argv = 0;
    symtb[type].modno = modno;
    symtb[type].lineno = yylineno;
    symtb[type].xfno = type;
    symtb[type].pname = putstr(s);
    symtb[type].x = symtb[type].xp = NULL;
    symtb[type].f = NULL;
    symtb[type].next = hashtb[k];
    hashtb[k] = type;
  } else {
    int sym = type, sym_modno = symtb[type].modno;
    while (symtb[sym].ref)
      sym = symtb[sym].ref;
    if (!dcontext && wflag >= 2 && mno == ANY && mainno >= 0 &&
	(wflag >= 3 && symtb[type].modno >= 0 ||
	 symtb[sym].modno >= mainno) && sym_modno != modno) {
      sprintf(msg, qcmsg[UNQUALIFIED_REF], utf8_to_sys(s),
	      utf8_to_sys(strsp+_modtb[sym_modno]));
      yywarn(msg);
    }
  }
  return type;
}

int
newtype(char *s, int type)
{
  int             k = strhash(s, hashtbsz);
  
  if (type > SHRT_MAX) fatal(qcmsg[SYMTB_OVF]);
  while (type >= asymtbsz)
    if (!(symtb = (SYMREC *)arealloc(symtb, asymtbsz,
				     SYMTBSZ/10, sizeof(SYMREC))))
      fatal(qcmsg[SYMTB_OVF]);
    else
      asymtbsz += SYMTBSZ/10;
  while (type > symtbsz) {
    symtb[symtbsz].ref = 0;
    symtb[symtbsz].prec = NONE;
    symtb[symtbsz].flags = 0;
    symtb[symtbsz].type = 0;
    symtb[symtbsz].fno_min = symtb[symtbsz].fno_max = 0;
    symtb[symtbsz].argc = 0;
    symtb[symtbsz].argv = 0;
    symtb[symtbsz].modno = NONE;
    symtb[symtbsz].lineno = 0;
    symtb[symtbsz].xfno = symtbsz;
    symtb[symtbsz].pname = 0;
    symtb[symtbsz].x = symtb[symtbsz].xp = NULL;
    symtb[symtbsz].f = NULL;
    symtb[symtbsz].next = NONE;
    symtbsz++;
  }
  symtb[type].ref = 0;
  symtb[type].prec = NONE;
  symtb[type].flags = TSYM;
  symtb[type].type = 0;
  symtb[type].fno_min = symtb[type].fno_max = 0;
  symtb[type].argc = 0;
  symtb[type].argv = 0;
  symtb[type].modno = modno;
  symtb[type].lineno = yylineno;
  symtb[type].xfno = type;
  symtb[type].pname = putstr(s);
  symtb[type].x = symtb[type].xp = NULL;
  symtb[type].f = NULL;
  symtb[type].next = hashtb[k];
  hashtb[k] = type;
  symtbsz++;
  return type;
}

void
astype(int type, int type2, short flags)
{
  char msg[MAXSTRLEN], *s;
  if (type == NONE || type2 == NONE) return;
  s = strsp+symtb[type].pname;
  type = xxxsym(type);
  flags |= DCL|TSYM;
  if (!type2)
    type2 = newtype(s, symtbsz);
  else if (symtb[type2].modno != modno)
    type2 = newtype(strsp+symtb[type2].pname, symtbsz);
  if ((symtb[type].flags & ~(PRIV|EXT)) != (flags & ~PRIV)) {
    sprintf(msg, qcmsg[MISM_DCL],
	    utf8_to_sys(strsp+symtb[type].pname));
    yyerror(msg);
  } else if (checksym(modno, type2, type)) {
    sprintf(msg, qcmsg[MISM_DCL],
	    utf8_to_sys(strsp+symtb[type2].pname));
    yyerror(msg);
  }
  symtb[type2].flags = flags;
  symtb[type2].type = symtb[type].type;
  symtb[type2].ref = type;
}

int
dcltype(int type, int supertype, short flags)
{
  char msg[MAXSTRLEN];
  if (type == NONE) return 0;
  if (symtb[type].modno != modno)
    type = newtype(strsp+symtb[type].pname, symtbsz);
  if (symtb[type].flags & DCL) {
    sprintf(msg, qcmsg[DOUBLE_TYPE_DCL],
	    utf8_to_sys(strsp+symtb[type].pname));
    yyerror(msg);
    return type;
  } else if (checksym(modno, type, type)) {
    sprintf(msg, qcmsg[MISM_DCL],
	    utf8_to_sys(strsp+symtb[type].pname));
    yyerror(msg);
  }
  symtb[type].flags = flags|DCL|TSYM;
  symtb[type].type = supertype;
  return type;
}

int
checktype(int type)
{
  if (type != NONE && !(symtb[type].flags & DCL)) {
    char msg[MAXSTRLEN];
    sprintf(msg, qcmsg[INVALID_TYPE],
	    utf8_to_sys(strsp+symtb[type].pname));
    yyerror(msg);
  }
  return type;
}

int mkxfun(char *s)
{
  int i;
  char t[MAXSTRLEN];
  strcpy(t, s);
  dcontext = 1;
  i = mkfun(t);
  dcontext = 0;
  if (i != NONE && symtb[i].modno == modno)
    return i;
  else
    return newfun(s, symtbsz);
}

int mkxfvar(char *s)
{
  int i;
  char t[MAXSTRLEN];
  strcpy(t, s);
  dcontext = 1;
  i = mkfvar(t);
  dcontext = 0;
  if (i != NONE && symtb[i].modno == modno)
    return i;
  else
    return newfvar(s, symtbsz);
}

int mkxtype(char *s)
{
  int i;
  char t[MAXSTRLEN];
  strcpy(t, s);
  dcontext = 1;
  i = mktype(t);
  dcontext = 0;
  if (i != NONE && symtb[i].modno == modno)
    return i;
  else
    return newtype(s, symtbsz);
}

/* fix UNIQ/VIS flags in symbol table */

#define matchsym(sym,s,t) (((symtb[sym].flags&TSYM)==t)&&\
			   streq((s), strsp+symtb[sym].pname))

static int
ref(int sym)
{
  while (symtb[sym].ref) sym = symtb[sym].ref;
  return sym;
}

static int
uniq(int sym)
{
  if (_modtbsz > 0) {
    int sym0 = sym, sym1 = NONE;
    char *s = strsp+symtb[sym].pname;
    int t = symtb[sym].flags&TSYM, k = strhash(s, hashtbsz);
      
    for (sym = hashtb[k]; sym != NONE; sym = symtb[sym].next)
      if (matchsym(sym, s, t) &&
	  (symtb[sym].modno == NONE || symtb[sym].modno == modno ||
	   imps[symtb[sym].modno]))
	if (symtb[sym].modno == modno)
	  return ref(sym)==ref(sym0);
	else if (symtb[sym].flags & PRIV)
	  ;
	else if (sym1 != NONE) {
	  if (ref(sym1) == ref(sym))
	    ;
	  else
	    return (symprio(sym) != symprio(sym1)) &&
	      ref(sym1)==ref(sym0);
	} else
	  sym1 = sym;
    return sym1 == NONE || ref(sym1)==ref(sym0);
  } else
    return 1;
}

static void
fix_symtb(void)
{
  int i;
  modno = (mainno!=NONE)?mainno:0;
  if (_modtbsz > 0) {
    for (i = 0; i < _modtbsz; i++)
      imps[i] = 0;
    restoreimps();
  }
  for (i = 0; i < symtbsz; i++) {
    if (symtb[i].modno == NONE || symtb[i].modno == 0 ||
	symtb[i].modno == modno || imps[symtb[i].modno])
      if (!(symtb[i].flags & PRIV) || symtb[i].modno == modno)
	symtb[i].flags |= VIS;
      if (uniq(i))
	symtb[i].flags |= UNIQ;
  }
  /* These need special treatment: */
  symtb[ANDTHENOP].flags &= ~VIS;
  symtb[ANDTHENOP].flags |= symtb[ANDOP].flags&VIS;
  symtb[ANDTHENOP].flags &= ~UNIQ;
  symtb[ANDTHENOP].flags |= symtb[ANDOP].flags&UNIQ;
  symtb[ORELSEOP].flags &= ~VIS;
  symtb[ORELSEOP].flags |= symtb[OROP].flags&VIS;
  symtb[ORELSEOP].flags &= ~UNIQ;
  symtb[ORELSEOP].flags |= symtb[OROP].flags&UNIQ;
}

/* generic quadratic hashing algorithm: */

static int
qhash(data, size, hash, peek, cmp, add)
     char           *data, *(*peek) ();
     int             size, (*hash) (), (*cmp) (), (*add) ();
{
  int             pos, incr, count;
  char           *data2;

  pos = (*hash) (data, size);
  for (count = 0, incr = 1; count <= size; count++, pos += incr,
	 pos %= size, incr += 2) {
    if ((data2 = (*peek) (pos)) == NULL)
      if (add == NULL || !(*add) (data, pos))
	return (NONE);
      else
	return (pos);
    else if ((*cmp) (data, data2))
      return (pos);
  }
  return (NONE);
}

static char    *
vartbpeek(i)
     int             i;
{
  if (vartb[i].dflag)
    return (&tmpsp[vartb[i].pname]);
  else
    return (NULL);
}

static int
vartbadd(s, i)
     char           *s;
     int             i;
{
  vartb[i].pname = puttmp(s);
  vartb[i].pname_s = -1;
  vartb[i].dflag = 1;
  vartb[i].offs = 0;
  vartb[i].plen = 0;
  vartb[i].type = NONE;
  return (1);
}

static
vareq(s1, s2)
     char           *s1, *s2;
{
  return strcmp(s1, "_") && strcmp(s2, "_") && strcmp(s1, s2) == 0;
}

int
mkvar(char *s)
{
  int             k;

  if ((k = qhash(s, VARTBSZ, strhash, vartbpeek, vareq, vartbadd))
      == NONE)
    fatal(qcmsg[SYMTB_OVF]);
  else
    return k;
}

int
mkvarsym(int vno)
{
  if (vartb[vno].pname_s >= 0)
    return vartb[vno].pname_s;
  else
    return (vartb[vno].pname_s = putstr(tmpsp+vartb[vno].pname));
}

void
backpatch(int addr, int offs)
{
  if (addr >= 0) addr += codespsz;
  while (label >= 0) {
    OPREC *op = tcodesp+label;
    if (op->opcode == QUERYOP || op->opcode == MATCHOP) {
      label = op->opargs.qual.addr;
      op->opargs.qual.addr = addr;
      op->opargs.qual.offs = offs;
    } else if (op->opcode == INFOP) {
      label = op->opargs.info.addr;
      op->opargs.info.addr = addr;
      op->opargs.info.offs = offs;
    } else
      label = -1;
  }
}

void
genop(int opcode)
{
  OPREC		*op = tcodesp+tcodespsz;
  if (tcodespsz >= TCODESPSZ)
    fatal(qcmsg[CODETB_OVF]);
  op->opcode = opcode;
  op->mode = 0;
  if (opcode == QUERYOP) {
    op->opargs.qual.m = -1;
    op->opargs.qual.addr = label;
    op->opargs.qual.offs = 0;
    label = tcodespsz;
  }
  tcodespsz++;
}

void
genlval(byte offs, byte plen, PATH p, int vsym)
{
  OPREC		*op = tcodesp+tcodespsz;
  if (tcodespsz >= TCODESPSZ)
    fatal(qcmsg[CODETB_OVF]);
  op->opcode = LVALOP;
  op->mode = 0;
  op->opargs.lval.offs = offs;
  op->opargs.lval.plen = plen;
  op->opargs.lval.p = p;
  op->opargs.lval.vsym = vsym;
  tcodespsz++;
}

void
genmatch(int m)
{
  OPREC		*op = tcodesp+tcodespsz;
  if (tcodespsz >= TCODESPSZ)
    fatal(qcmsg[CODETB_OVF]);
  op->opcode = MATCHOP;
  op->mode = 0;
  op->opargs.qual.m = m;
  op->opargs.qual.addr = label;
  op->opargs.qual.offs = 0;
  label = tcodespsz;
  tcodespsz++;
}

void
geninfop(int modno, int lineno)
{
  OPREC		*op = tcodesp+tcodespsz;
  if (tcodespsz >= TCODESPSZ)
    fatal(qcmsg[CODETB_OVF]);
  op->opcode = INFOP;
  op->opargs.info.modno = modno;
  op->opargs.info.lineno = lineno;
  op->opargs.info.addr = -1;
  op->opargs.info.offs = 0;
  op->mode = 0;
  label = tcodespsz;
  tcodespsz++;
}

void
fixinfop(int modno, int lineno)
{
  OPREC		*op = tcodesp;
  if (tcodespsz <= 0 || op->opcode != INFOP)
    return;
  op->opargs.info.modno = modno;
  op->opargs.info.lineno = lineno;
  op->mode = 0;
  op->opargs.info.addr = -1;
  label = 0;
}

void
genintval(mpz_t iv)
{
  OPREC		*op = tcodesp+tcodespsz;
  if (tcodespsz >= TCODESPSZ)
    fatal(qcmsg[CODETB_OVF]);
  op->opcode = INTVALOP;
  op->mode = 0;
  op->opargs.iv.len = iv->_mp_size;
  op->opargs.iv.l = putlimbs(iv);
  tcodespsz++;
}

void
genfloatval(double fv)
{
  OPREC		*op = tcodesp+tcodespsz;
  if (tcodespsz >= TCODESPSZ)
    fatal(qcmsg[CODETB_OVF]);
  op->opcode = FLOATVALOP;
  op->mode = 0;
  op->opargs.fv = fv;
  tcodespsz++;
}

void
genstrval(int sv)
{
  OPREC		*op = tcodesp+tcodespsz;
  if (tcodespsz >= TCODESPSZ)
    fatal(qcmsg[CODETB_OVF]);
  op->opcode = STRVALOP;
  op->mode = 0;
  op->opargs.sv = sv;
  tcodespsz++;
}

void
set_mode(void)
{
  OPREC		*op = tcodesp+tcodespsz-1;
  op->mode = 1;
}

void
add_init(int offs)
{
  if (inittbsz >= ainittbsz)
    if (!(inittb = (int *)arealloc(inittb, ainittbsz,
				   INITTBSZ/4, sizeof(int))))
      fatal(qcmsg[CODETB_OVF]);
    else
      ainittbsz += INITTBSZ/4;
  inittb[inittbsz++] = offs;
}

void
write_header(void)
{
  char outid1[MAXSTRLEN];
  strcpy(outid1, outid);
  *outid1 = '?';
  fwrite(outid1, sizeof(char), strlen(outid)+1, codefp);
  fwrite(&mainno, sizeof mainno, 1, codefp);
  fwrite(&codespsz, sizeof codespsz, 1, codefp);
  fwrite(&strspsz, sizeof strspsz, 1, codefp);
  fwrite(&limbspsz, sizeof limbspsz, 1, codefp);
  fwrite(&hashtbsz, sizeof hashtbsz, 1, codefp);
  fwrite(&symtbsz, sizeof symtbsz, 1, codefp);
  fwrite(&statetbsz, sizeof statetbsz, 1, codefp);
  fwrite(&transtbsz, sizeof transtbsz, 1, codefp);
  fwrite(&roffstbsz, sizeof roffstbsz, 1, codefp);
  fwrite(&matchtbsz, sizeof matchtbsz, 1, codefp);
  fwrite(&inittbsz, sizeof inittbsz, 1, codefp);
  fwrite(&modtbsz, sizeof modtbsz, 1, codefp);
  fwrite(&imptbsz, sizeof imptbsz, 1, codefp);
  if (ferror(codefp))
    fatal(qcmsg[CODE_FILE_ERR]);
}

void
write_code(void)
{
  if (codespsz > INT_MAX - tcodespsz ||
      codespsz + tcodespsz > LONG_MAX / sizeof(OPREC))
    fatal(qcmsg[CODETB_OVF]);
  fwrite(tcodesp, sizeof(OPREC), tcodespsz, codefp);
  codespsz += tcodespsz;
  if (ferror(codefp))
    fatal(qcmsg[CODE_FILE_ERR]);
}

void
write_strsp(void)
{
  fwrite(strsp, sizeof(char), strspsz, codefp);
  if (ferror(codefp))
    fatal(qcmsg[CODE_FILE_ERR]);
}

void
write_limbsp(void)
{
  fwrite(limbsp, sizeof(mp_limb_t), limbspsz, codefp);
  if (ferror(codefp))
    fatal(qcmsg[CODE_FILE_ERR]);
}

void
write_hashtb(void)
{
  fwrite(hashtb, sizeof(int), hashtbsz, codefp);
  if (ferror(codefp))
    fatal(qcmsg[CODE_FILE_ERR]);
}

void
write_symtb(void)
{
  fix_symtb();
  fwrite(symtb, sizeof(SYMREC), symtbsz, codefp);
  if (ferror(codefp))
    fatal(qcmsg[CODE_FILE_ERR]);
}

void
write_state(int ntrans, int trans, int nrules, int roffs)
{
  STATEREC s;
  s.ntrans = ntrans;
  s.trans = trans;
  s.nrules = nrules;
  s.roffs = roffs;
  fwrite(&s, sizeof(STATEREC), 1, codefp);
  if (ferror(codefp))
    fatal(qcmsg[CODE_FILE_ERR]);
}

void
write_trans(int type, int fno, int next)
{
  TRANSREC t;
  t.type = type;
  t.fno = fno;
  t.next = next;
  fwrite(&t, sizeof(TRANSREC), 1, codefp);
  if (ferror(codefp))
    fatal(qcmsg[CODE_FILE_ERR]);
}

void
write_roffs(int roffs)
{
  fwrite(&roffs, sizeof roffs, 1, codefp);
  if (ferror(codefp))
    fatal(qcmsg[CODE_FILE_ERR]);
}

void
write_rmno(int rmno)
{
  fwrite(&rmno, sizeof rmno, 1, codefp);
  if (ferror(codefp))
    fatal(qcmsg[CODE_FILE_ERR]);
}

void
write_matchtb(void)
{
  fwrite(matchtb, sizeof(int), matchtbsz, codefp);
  if (ferror(codefp))
    fatal(qcmsg[CODE_FILE_ERR]);
}

void
write_inittb(void)
{
  fwrite(inittb, sizeof(int), inittbsz, codefp);
  if (ferror(codefp))
    fatal(qcmsg[CODE_FILE_ERR]);
}

static int intcmp(int *ip, int *jp)
{
  return *ip-*jp;
}

void
write_modtb(void)
{
  int i;

  modtbsz = _modtbsz;
  fwrite(_modtb, sizeof(int*), modtbsz, codefp);
  fwrite(_fnametb, sizeof(int*), modtbsz, codefp);
  imptbsz = 0;
  for (i = 0; i < modtbsz; i++) {
    fwrite(&imptbsz, sizeof(imptbsz), 1, codefp);
    imptbsz += _imports[i].sz;
  }
#ifdef DEBUG
  printf("\nimports:\n");
  for (i = 0; i < modtbsz; i++) {
    int j;
    printf("%s (%s):", utf8_to_sys(strsp+_modtb[i]), strsp+_fnametb[i]);
    for (j = 0; j < _imports[i].sz; j++)
      if (_imports[i].ib[j]&2)
	printf(" (%s%s)", utf8_to_sys(strsp+_modtb[_imports[i].tb[j]]),
	       (_imports[i].ib[j]&1)?"*":"");
      else
	printf(" %s%s", utf8_to_sys(strsp+_modtb[_imports[i].tb[j]]),
	       (_imports[i].ib[j]&1)?"*":"");
    printf("\n");
  }
#endif  
  for (i = 0; i < modtbsz; i++)
    fwrite(_imports[i].tb, sizeof(int), _imports[i].sz, codefp);
  for (i = 0; i < modtbsz; i++)
    fwrite(_imports[i].ib, sizeof(byte), _imports[i].sz, codefp);
  if (ferror(codefp))
    fatal(qcmsg[CODE_FILE_ERR]);
}

void
fix_header(void)
{
  rewind(codefp);
  fwrite(outid, sizeof(char), strlen(outid)+1, codefp);
  fwrite(&mainno, sizeof mainno, 1, codefp);
  fwrite(&codespsz, sizeof codespsz, 1, codefp);
  fwrite(&strspsz, sizeof strspsz, 1, codefp);
  fwrite(&limbspsz, sizeof limbspsz, 1, codefp);
  fwrite(&hashtbsz, sizeof hashtbsz, 1, codefp);
  fwrite(&symtbsz, sizeof symtbsz, 1, codefp);
  fwrite(&statetbsz, sizeof statetbsz, 1, codefp);
  fwrite(&transtbsz, sizeof transtbsz, 1, codefp);
  fwrite(&roffstbsz, sizeof roffstbsz, 1, codefp);
  fwrite(&matchtbsz, sizeof matchtbsz, 1, codefp);
  fwrite(&inittbsz, sizeof inittbsz, 1, codefp);
  fwrite(&modtbsz, sizeof modtbsz, 1, codefp);
  fwrite(&imptbsz, sizeof imptbsz, 1, codefp);
  if (ferror(codefp))
    fatal(qcmsg[CODE_FILE_ERR]);
}
