
/* qbase.h: shared definitions of the Q programs */

/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "sys.h"
#include "opcodes.h"

/* sign-on messages */

#define TERMS \
"This software is distributed under the terms of the GNU General Public\n\
License version 2 or later; type `copying' for details.\n"

#define COPYING \
"\n\
This program is free software; you can redistribute it and/or modify\n\
it under the terms of the GNU General Public License as published by\n\
the Free Software Foundation; either version 2 of the License, or\n\
(at your option) any later version.\n\
\n\
This program is distributed in the hope that it will be useful,\n\
but WITHOUT ANY WARRANTY; without even the implied warranty of\n\
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n\
GNU General Public License for more details.\n\
\n\
You should have received a copy of the GNU General Public License\n\
along with this program; if not, write to the Free Software\n\
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n\
\n"

#define HELPMSG \
"Type `%s --help' for help.\n"

#define QC_SIGNON \
"Q compiler version %s (%s)\nCopyright (c) 1991-%s by Albert Graef\n"

#define QC_USAGE "Usage: %s [options] [file | -] ...\n"

#define QC_OPTS1 "dnhl:o:p:t:vw::V"

#define QC_PRELUDE 1
#define QC_NO_PRELUDE 2
#define QC_ENCODING 3
#define QC_PEDANTIC 4
#define QC_PARANOID 5

#define QC_OPTS { \
{ "debug", 0, NULL, 'd' }, \
{ "encoding", 1, NULL, QC_ENCODING }, \
{ "help", 0, NULL, 'h' }, \
{ "list", 1, NULL, 'l' }, \
{ "no-code", 0, NULL, 'n' }, \
{ "no-prelude", 0, NULL, QC_NO_PRELUDE }, \
{ "output", 1, NULL, 'o' }, \
{ "paranoid", 0, NULL, QC_PARANOID }, \
{ "path", 1, NULL, 'p' }, \
{ "pedantic", 0, NULL, QC_PEDANTIC }, \
{ "prelude", 1, NULL, QC_PRELUDE }, \
{ "tablesize", 1, NULL, 't' }, \
{ "verbose", 0, NULL, 'v' }, \
{ "version", 0, NULL, 'V' }, \
{ "warnings", 2, NULL, 'w' }, \
{ NULL, 0, NULL, 0 } \
}

#define QC_OPTMSG \
"Options:\n\
  --debug, -d		Print bytecode listing on standard output.\n\
  --encoding=ENC        Specify the default encoding of scripts.\n\
  --help, -h		Print this list.\n\
  --list=FILE, -l FILE	Specify list file name (default: standard error).\n\
  --no-code, -n		Suppress code generation.\n\
  --no-prelude		Suppress inclusion of the prelude.\n\
  --output=FILE, -o FILE\n\
			Specify output file name (default: `q.out').\n\
  --paranoid		Synonym for level 3 warnings (-w3).\n\
  --path=PATH, -p PATH	Specify the search path (default: `%s').\n\
  --pedantic		Synonym for level 2 warnings (-w2).\n\
  --prelude=FILE	Specify the prelude file name (default: `prelude.q').\n\
  --tablesize=SIZE, -t SIZE\n\
			Specify the size of the hash table (default: %d).\n\
  --verbose, -v		Display processed source files and statistics.\n\
  --version, -V		Display version information and copyright notices.\n\
  --warnings[=LEVEL], -w[LEVEL]\n\
			Enable warning messages (default level is 1).\n\n\
  file			Main script (`-' reads from standard input).\n"

#if 0

/* old style sign-on */
#define Q_SIGNON \
"Q interpreter version %s (%s)\nCopyright (c) 1991-%s by Albert Graef\n"

#else

/* new style sign-on */
#define Q_SIGNON \
"  ____\n\
 / __ \\    Q interpreter version %s (%s)\n\
/ /_/ /	   Copyright (c) 1991-%s by Albert Graef\n\
\\___\\_\\    http://q-lang.sourceforge.net\n"

#endif

#define Q_USAGE "Usage: %s [options] [file | -] [argument ...]\n"

#define Q_OPTS1 "hVnl:o:p:s:t:vw::deiqc:"

#define Q_PROMPT 11
#define Q_HISTFILE 12
#define Q_HISTSIZE 13
#define Q_INITRC 14
#define Q_EXITRC 15
#define Q_NO_INITRC 16
#define Q_NO_EXITRC 17
#define Q_NO_EDITING 18
#define Q_DEBUG_OPTIONS 19
#define Q_MEMSIZE 20
#define Q_STACKSIZE 21
#define Q_CSTACKSIZE 22
#define Q_GNUCLIENT 23
#define Q_DEC 24
#define Q_HEX 25
#define Q_OCT 26
#define Q_STD 27
#define Q_SCI 28
#define Q_FIX 29
#define Q_BREAK 30

#define Q_OPTS { \
{ "break", 0, NULL, Q_BREAK }, \
{ "command", 1, NULL, 'c' }, \
{ "debug", 0, NULL, 'd' }, \
{ "debug-options", 1, NULL, Q_DEBUG_OPTIONS }, \
{ "dec", 0, NULL, Q_DEC }, \
{ "echo", 0, NULL, 'e' }, \
{ "exitrc", 1, NULL, Q_EXITRC }, \
{ "fix", 2, NULL, Q_FIX }, \
{ "gnuclient", 0, NULL, Q_GNUCLIENT }, \
{ "help", 0, NULL, 'h' }, \
{ "hex", 0, NULL, Q_HEX }, \
{ "histfile", 1, NULL, Q_HISTFILE }, \
{ "histsize", 1, NULL, Q_HISTSIZE }, \
{ "initrc", 1, NULL, Q_INITRC }, \
{ "interactive", 0, NULL, 'i' }, \
{ "memsize", 1, NULL, Q_MEMSIZE }, \
{ "no-editing", 0, NULL, Q_NO_EDITING }, \
{ "no-exitrc", 0, NULL, Q_NO_EXITRC }, \
{ "no-initrc", 0, NULL, Q_NO_INITRC }, \
{ "oct", 0, NULL, Q_OCT }, \
{ "path", 1, NULL, 'p' }, \
{ "prompt", 1, NULL, Q_PROMPT }, \
{ "quiet", 0, NULL, 'q' }, \
{ "sci", 2, NULL, Q_SCI }, \
{ "source", 1, NULL, 's' }, \
{ "stacksize", 1, NULL, Q_STACKSIZE }, \
{ "cstacksize", 1, NULL, Q_CSTACKSIZE }, \
{ "std", 2, NULL, Q_STD }, \
{ "version", 0, NULL, 'V' }, \
/* compiler options: */ \
{ "encoding", 1, NULL, QC_ENCODING }, \
{ "list", 1, NULL, 'l' }, \
{ "no-code", 0, NULL, 'n' }, \
{ "no-prelude", 0, NULL, QC_NO_PRELUDE }, \
{ "output", 1, NULL, 'o' }, \
{ "paranoid", 0, NULL, QC_PARANOID }, \
{ "pedantic", 0, NULL, QC_PEDANTIC }, \
{ "prelude", 1, NULL, QC_PRELUDE }, \
{ "tablesize", 1, NULL, 't' }, \
{ "verbose", 0, NULL, 'v' }, \
{ "warnings", 2, NULL, 'w' }, \
{ NULL, 0, NULL, 0 } \
}

#define Q_OPTMSG \
"Options:\n\
  --break		Enable debugging after break.\n\
  --command=CMD, -c CMD\tExecute the given interpreter command (batch mode).\n\
  --cstacksize=SIZE	Specify the maximum C stack size (default: %dKB).\n\
  --debug, -d		Trace reductions with the debugger.\n\
  --debug-options=OPTIONS\n\
			Set debugger options.\n\
  --dec, --hex, --oct	Specify integer format (default: `%s').\n\
  --echo, -e		Echo commands in batch mode.\n\
  --exitrc=FILE		Specify the exit rc file (default: `%s').\n\
  --fix[=PREC], --sci[=PREC], --std[=PREC]\n\
			Specify floating point format (default: `%s')\n\
			with given precision (default: sci/std:15, fix:2).\n\
  --gnuclient		Run as a gnuserv client.\n\
  --help, -h		Print this list.\n\
  --histfile=FILE	Specify the history file (default: `%s').\n\
  --histsize=SIZE	Specify the history size (default: %d).\n\
  --initrc=FILE		Specify the init rc file (default: `%s').\n\
  --interactive, -i	Run interactively.\n\
  --memsize=SIZE	Specify the maximum memory size (default: %d).\n\
  --no-editing		Disable command line editing using readline.\n\
  --no-exitrc		Suppress execution of the exit rc file.\n\
  --no-initrc		Suppress execution of the init rc file.\n\
  --path=PATH, -p PATH	Specify the search path (default: `%s').\n\
  --prompt=STR		Specify the prompt string (default: `%s').\n\
  --quiet, -q		Quiet startup (suppress sign-on).\n\
  --source=FILE, -s FILE\n\
			Source file with interpreter commands (batch mode).\n\
  --stacksize=SIZE	Specify the maximum stack size (default: %d).\n\
  --version, -V		Display version information and copyright notices.\n\
Compiler Options:\n\
  --encoding=ENC        Specify the default encoding of scripts.\n\
  --list=FILE, -l FILE	Specify list file name (default: standard error).\n\
  --no-prelude		Suppress inclusion of the prelude.\n\
  --output=FILE, -o FILE\n\
			Specify output file name (default: `q.out').\n\
  --paranoid		Synonym for level 3 warnings (-w3).\n\
  --pedantic		Synonym for level 2 warnings (-w2).\n\
  --prelude=FILE	Specify the prelude file name (default: `prelude.q').\n\
  --tablesize=SIZE, -t SIZE\n\
			Specify the size of the hash table (default: %d).\n\
  --verbose, -v		Display processed source files and statistics.\n\
  --warnings[=LEVEL], -w[LEVEL]\n\
			Enable warning messages (default level is 1).\n\n\
  file		Script or code file (`-' reads from standard input).\n\
  argument ...	Script parameters (use `--' to prevent option parsing).\n"

/* symbolic constants */

#define OUTID "-*- q.out %s-%s -*-"
			/* code file tag (1st %s is replaced by VERSION
			   2nd %s by sysinfo) */

#define NONE -1			/* NONE value */
#define HASHTBSZ 5711		/* default hash table size (prime number) */

#define MAXDEPTH (8*sizeof(PATH))	/* maximum pattern depth */
#define MAXSTRLEN 1024			/* internal string length */

#ifndef STDFORMAT
#define STDFORMAT "%%.%dg"	/* standard double format */
#endif

#ifndef SCIFORMAT
#define SCIFORMAT "%%.%de"	/* scientific double format */
#endif

#ifndef FIXFORMAT
#define FIXFORMAT "%%#.%df"	/* fixed double format */
#endif

#ifndef PROMPT
#define PROMPT "\n==> "		/* default prompt */
#endif

#ifndef QPATH
#define QPATH ""		/* default search path */
#endif

#ifndef HISTFILE
#define HISTFILE "~/.q_history"	/* default history file */
#endif

#ifndef HISTMAX
#define HISTMAX 500		/* default history size */
#endif

#ifndef INITRC
#define INITRC "~/.qinitrc"	/* default initrc file */
#endif

#ifndef EXITRC
#define EXITRC "~/.qexitrc"	/* default exitrc file */
#endif

#ifndef MEMMAX
#define MEMMAX 4096000		/* default maximum memory size */
#endif

#ifndef MEMMIN
#define MEMMIN 4096		/* minimum memory size */
#endif

#ifndef STACKMAX
#define STACKMAX 1024000	/* default maximum stack size */
#endif

#ifndef STACKMIN
#define STACKMIN 100		/* minimum stack size */
#endif

#ifndef CSTACKMAX
#define CSTACKMAX (256*1024)	/* default maximum C stack size (bytes) */
#endif

#ifndef CSTACKMIN
#define CSTACKMIN (64*1024)	/* minimum C stack size */
#endif

extern char version[], year[], sysinfo[], fformat[],
  *prompt, *prompt2, *prompt3, *default_codeset,
  *qpath, *histfile, *initrc, *exitrc, *prelude, dirstr[], volstr[];
extern int histmax, memmax, stackmax, cstackmax, imode, fmode, fprec;

/* types used by the Q machine */

typedef unsigned long		PATH;	/* subterm paths */

#define getpath(p,i) ((int) ((p & ((PATH) 1 << i)) >> i))

/* dynamic array allocation */

/*

aalloc(nelems,size)
- allocate an array with nelems elements of size size
arealloc(ptr,nelems,newelems,size)
- reallocate an array ptr consisting of nelems elements of the given size to
  make room for newelems additional elements

*/

extern void *arealloc(void *ptr, int nelems, int newelems, int size);

#define aalloc(nelems,size) arealloc(NULL,0,nelems,size)

typedef struct oprec {
	short		opcode;	/* operation/function number */
	bool		mode;	/* mode flag */
	union {
	  /* QUERYOP, MATCHOP: */
	  struct {	/* qualifier */
	    int		m;	/* index into match table (MATCHOP only) */
	    int		addr;	/* branch address (-1 if none) */
	    byte	offs;	/* stack offset */
	  }		qual;
	  /* LVALOP: */
	  struct {	/* lhs value */
	    byte	offs;	/* offset from base pointer */
	    byte	plen;	/* path length */
	    PATH        p;	/* path */
	    int         vsym;	/* variable symbol (-1 if none) */
	  }		lval;
	  /* INFOP: */
	  struct {	/* debugging info */
	    int		modno, lineno, addr;
	    byte	offs;
	  }		info;
	  /* INTVALOP: */
	  struct {	/* Q machine integer */
	    int		len;	/* size */
	    int		l;	/* limbsp index */
	  }		iv;
	  /* FLOATVALOP: */
	  double	fv;	/* Q machine floating point number */
	  /* STRVALOP: */
	  int		sv;	/* string (index into string space) */
	}               opargs;
}               OPREC;

/* Symbol table enries: */

/* Always make sure that any type gets a LARGER symbol than its supertype!
   The pattern matching code depends on it. */

#define DCL	0x0001		/* declared? */
#define PRIV	0x0002		/* private symbol? */
#define CST	0x0004		/* const symbol? */
#define SPEC	0x0008		/* special symbol? */
#define TSYM	0x0010		/* type symbol? */
#define VSYM	0x0020		/* variable symbol? */
#define EXT	0x0040		/* external symbol? */
#define UNIQ	0x0080		/* unique symbol in global context? */
#define VIS	0x0100		/* visible in global context? */
#define MODIF	0x0200		/* modified? (def) */
#define BRK	0x0400		/* break? (debugger) */
#define TBRK	0x0800		/* temporary break? (debugger) */
#define PROF	0x1000		/* profiling enabled? */
#define FWD	0x2000		/* forward declaration? */
#define VIRT	0x4000		/* virtual constructor? */

typedef struct symrec {
	short		flags;		/* symbol attributes */
	short		argc;		/* declared arg count */
	unsigned long	argv;		/* bit vector of special args */
        unsigned long   nredns;		/* profiling info */
	int		prec;		/* operator precedence */
	int		ref;		/* aliased symbol */
	int		type;		/* type (0 if none); also supertype
					   of type symbol */
	int		fno_min, fno_max;	/* range of enumeration type */
	int		modno;	/* number of module in which symbol is
				   defined (NONE for built-in symbols) */
	int		lineno; /* line number of declaration/1st occurrence */
        int		xfno;	/* symbol with extern definition */
	int		pname;	/* print name: index into string space */
	void	       *x;	/* used by interpreter to implement global
				   environment; init'ed to NULL */
        void           *xp;     /* temporary used for variable defs */
	void	       *f;	/* external function (dll interface) */
	int		next;	/* next symbol in bucket (NONE if none) */
}               SYMREC;

/* The TA table stores the pattern matching automaton for the lhs's of
   equations. The basic technique is discussed in Graef: "Left-to-right tree
   pattern matching", Proc. RTA 91, Springer, 1991 (LNCS 488), it is modified
   here to cope with type guards. The automaton consists of three parts:

   - the state table which stores for each state the corresponding
     offsets into transition and rule offset table (the start state of the
     automaton is always state 0).

   - the transition table which stores for each state the corresponding
     triples of types, fno's and next states. The entries for each state
     are ordered lexicographically s.t. transitions may be found quickly
     using binary search (see qmmatch.c).

   - the rule offset table which stores for each final state of the
     automaton the offsets (entry point addresses) in the code space
     for the matched rules (sorted in ascending order). */

typedef struct staterec {
	int		ntrans;	/* number of transitions */
	int		trans;	/* offset into transition table */
	int             nrules;	/* number of rule offsets (0 if none) */
	int             roffs;	/* index into rule offs table for final state */
}               STATEREC;

typedef struct transrec {
	short		type;	/* type tag (0 for default type) */
	short           fno;	/* function number (0 if variable) */
	int             next;	/* next state */
}               TRANSREC;

/* Q code format: ****************************************************/

/* header: */

extern char outid[MAXSTRLEN];
extern int mainno;
extern int codespsz;
extern int strspsz;
extern int limbspsz;
extern int hashtbsz;
extern int symtbsz;
extern int statetbsz;
extern int transtbsz;
extern int roffstbsz;
extern int matchtbsz;
extern int inittbsz;
extern int modtbsz;
extern int imptbsz;

/* The following tables are allocated dynamically to the actual sizes. */

extern OPREC *codesp; 		/* code space */
extern char *strsp;		/* string space */
extern mp_limb_t *limbsp;	/* limb space */
extern int *hashtb;		/* hash table */
extern SYMREC *symtb;		/* symbol table */
extern STATEREC *statetb;	/* state table */
extern TRANSREC *transtb;	/* transition table */
extern int *roffstb;		/* rule offset table */
extern int *matchtb;		/* match states (where clauses) */
extern int *inittb;		/* initialization rule offsets */
extern int *modtb; 		/* module name table */
extern int *fnametb; 		/* file name table (same size as modtb) */
extern int *imports; 		/* imports table (offsets into imptb, same
				   size as modtb) */
extern int *imptb;		/* imports for each module */
extern byte *impib;		/* include flags (same size as imptb) */

extern byte *globs;		/* global imports (same size as modtb) */

#define visible(fno) (symtb[fno].flags&VIS)
#define unique(fno) (symtb[fno].flags&UNIQ)

/* routines */

extern char *charstr(char *s, char c);
extern char *substr(char *s, char *t, int n);
extern char *pstr(char *s1, char *s2);
extern char *pmpz(char *s, int imode, mpz_t z);
extern char *pfloat(char *s, char *fformat, double f);
extern char *pname(char *s, int fno);
extern char *scanstr(char *s1, char *s2);
extern char *dirname(char *t, char *s);
extern char *basename(char *t, char *s, char c);
extern char *absname(char *t, char *s);
extern char *expand(char *s1, char *s2);
extern char *searchlib(char *s1, char *s2);

extern int dirprefix(char *s, char *prefix);
extern int absolute(char *s);
extern int chkfile(char *s);

extern void setpath(PATH *p, int i, int v);
extern int getint(mpz_t z, int len, int l);

extern void init_qpath(char *s);
extern void change_qpath(char *s);

/* unicode helpers */

extern char *default_encoding(void);
extern char *utf8_to_sys(char *s);
extern char *sys_to_utf8(char *s);
extern char *utf8_to_sys_dup(char *s);
extern char *sys_to_utf8_dup(char *s);

/* helper functions to convert between doubles and strings in a thread-safe,
   locale-independent way (from GNOME glib) */

#define DTOSTR_BUF_SIZE (29 + 10)

extern double my_strtod(const char *nptr, char **endptr);
extern char *my_formatd(char *buffer, const char *format, double d);

/* unicode character classification routines from the ICU library
   (cf. uchar.c) */

extern bool u_islower(int c);
extern bool u_isupper(int c);
extern bool u_isdigit(int c);
extern bool u_isalpha(int c);
extern bool u_isalnum(int c);
extern bool u_ispunct(int c);

/* random number generator (cf. random.c) */

extern void seedMT(unsigned long seed);
extern unsigned long randomMT(void);
