
/* $Id: q.c,v 1.70 2007/10/27 18:59:05 agraef Exp $
   q.c: the Q interpreter */

/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <setjmp.h>
#include "qdefs.h"

static char		signon[] = Q_SIGNON;
static char   		usage[] = Q_USAGE;
static char		opts[4096];
static char		terms[] = TERMS;
static char		copying[] = COPYING;
static char		helpmsg[] = HELPMSG;

static char           *self = "q";
static char            qcprog[MAXSTRLEN] = "qc";

static bool		batch = 0, docompile = 1, donecompile = 0,
  doexitrc = 0, oset = 0;

bool		eflag = 0, qflag = 0, iflag = 0, hflag = 0, Vflag = 0,
  gflag = 0, norl = 0;

char *source = NULL;
char *which = NULL;

/* The following is set to compile for the libqint target. */
#ifdef QINT_LIBRARY
#undef USE_READLINE
#endif

static unload_dlls(void);
#ifdef USE_READLINE
static void fini_readline(void);
#endif

void exitproc(void)
{
  FILE *fp;
  char fname[MAXSTRLEN];
  int i;

  if (donecompile) remove(code);
  if (doexitrc && exitrc && chkfile(expand(fname, exitrc)) &&
      (fp = fopen(fname, "r"))) {
    fclose(fp);
    quitflag = 0;
    parsesrc(fname, 1);
  }
#ifndef _WIN32
  /* Looks like this causes segfaults on Windows under certain
     circumstances. Maybe a bug in the dynamic loader? Disabled for now. */
  unload_dlls();
  lt_dlexit();
#endif
#if defined (USE_READLINE)
  fini_readline();
  rl_deprep_terminal();
#endif
}

#ifdef QINT_LIBRARY

static jmp_buf fatal_target;
static int fatal_set = 0, fatal_status = OK;

/* catch fatal error conditions for libqint target */

void fatal(char *s)
{
  if (fatal_set) {
    int i;
    fatal_status = -1; /* unknown error */
    if (!s) goto skip;
    for (i = 0; i <= THIS_CANT_HAPPEN; i++)
      if (!strcmp(s, qmmsg[i])) {
	fatal_status = i;
	goto skip;
      }
    if (!strcmp(s, "memory overflow"))
      fatal_status = MEM_OVF;
    else if (!strcmp(s, "stack overflow"))
      fatal_status = XST_OVF;
  skip:
    longjmp(fatal_target, 1);
  } else {
    flush_shift();
    if (s) fprintf(stderr, "%s: %s\n", self, s);
    doexitrc = 0;
    exit(1);
  }
}

#else

void fatal(char *s)
{
  flush_shift();
  if (s) fprintf(stderr, "%s: %s\n", self, s);
  doexitrc = 0;
  exit(1);
}

#endif

void echo(char *s)
{
  char *t = s;
  while (*s && isspace(*s)) ++s;
  if (*s != '@') {
    flush_shift();
    printf("%s\n", t);
  }
}

#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)

#define CHUNKSZ 128

static inline char *
icfromutf8(iconv_t ic[2], char *s)
{
  if (ic[1] == (iconv_t)-2) {
    char *codeset = default_encoding();
    if (codeset && strcmp(codeset, "UTF-8"))
      ic[1] = iconv_open(codeset, "UTF-8");
    else
      ic[1] = (iconv_t)-1;
  }
  if (ic[1] == (iconv_t)-1)
    return NULL;
  else {
    /* Here the input buffer may be NULL, to emit a terminating shift
       sequence. In this case we initialize an output buffer of size
       CHUNKSZ. */
    size_t l = s?strlen(s):0, m = s?l:CHUNKSZ;
    char *t = malloc(m+1), *t1;
    char *inbuf = s, *outbuf = t;
    size_t inbytes = l, outbytes = m;

    while (iconv(ic[1], &inbuf, &inbytes, &outbuf, &outbytes) ==
	   (size_t)-1)
      if (errno == E2BIG) {
	/* try to enlarge the output buffer */
	size_t k = outbuf-t;
	if ((t1 = realloc(t, m+CHUNKSZ+1))) {
	  t = t1;
	  outbuf = t+k;
	  m += CHUNKSZ;
	  outbytes += CHUNKSZ;
	} else {
	  /* memory overflow */
	  free(t);
	  return NULL;
	}
      } else {
	/* conversion error */
	free(t);
	return NULL;
      }
    /* terminate the output string */
    *outbuf = 0;
    if (!(t1 = realloc(t, strlen(t)+1)))
      /* this shouldn't happen */
      return t;
    else
      return t1;
  }
}

static inline char *
ictoutf8(iconv_t ic[2], char *s)
{
  if (ic[0] == (iconv_t)-2) {
    char *codeset = default_encoding();
    if (codeset && strcmp(codeset, "UTF-8"))
      ic[0] = iconv_open("UTF-8", codeset);
    else
      ic[0] = (iconv_t)-1;
  }
  if (ic[0] == (iconv_t)-1)
    return NULL;
  else {
    size_t l = strlen(s);
    char *t = malloc(l+1), *t1;
    char *inbuf = s, *outbuf = t;
    size_t inbytes = l, outbytes = l;

    while (iconv(ic[0], &inbuf, &inbytes, &outbuf, &outbytes) ==
	   (size_t)-1)
      if (errno == E2BIG) {
	/* try to enlarge the output buffer */
	size_t k = outbuf-t;
	if ((t1 = realloc(t, l+CHUNKSZ+1))) {
	  t = t1;
	  outbuf = t+k;
	  l += CHUNKSZ;
	  outbytes += CHUNKSZ;
	} else {
	  /* memory overflow */
	  free(t);
	  return NULL;
	}
      } else {
	/* conversion error */
	free(t);
	return NULL;
      }
    /* terminate the output string */
    *outbuf = 0;
    if (!(t1 = realloc(t, strlen(t)+1)))
      /* this shouldn't happen */
      return t;
    else
      return t1;
  }
}

#endif

/* flush out any pending shift sequences on the standard output streams, so
   that we have a clear state for subsequent ASCII output */

void flush_shift(void)
{
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
  if (symtb) {
    EXPR *x = (EXPR*)symtb[ERROROP].x;
    if (x && x->fno == FILEVALOP &&
	x->data.fargs.ic[1] != (iconv_t)-2 &&
	x->data.fargs.ic[1] != (iconv_t)-1) {
      char *s = icfromutf8(x->data.fargs.ic, NULL), *t = s;
      if (t) {
	while (*s) putc(*s++, x->data.fp);
	free(t);
	fflush(x->data.fp);
      }
    }
    x = (EXPR*)symtb[OUTPUTOP].x;
    if (x && x->fno == FILEVALOP &&
	x->data.fargs.ic[1] != (iconv_t)-2 &&
	x->data.fargs.ic[1] != (iconv_t)-1) {
      char *s = icfromutf8(x->data.fargs.ic, NULL), *t = s;
      if (t) {
	while (*s) putc(*s++, x->data.fp);
	free(t);
	fflush(x->data.fp);
      }
    }
  }
#endif
}

#ifdef HAVE_BACKTRACE
/* Obtain a backtrace and print it to `stdout'. */
void print_trace(void)
{
  void *array[50];
  size_t size;

  size = backtrace (array, 50);

  fprintf(stderr, "%d stack frames\n", (int)size);
  backtrace_symbols_fd(array, size, fileno(stderr));
}
#endif

#if defined(USE_READLINE) && defined(HAVE_POSIX_SIGNALS)
/* used to implement nonlocal exit while reading input using readline */
jmp_buf longjmp_target;
volatile int longjump_set = 0;
#endif

RETSIGTYPE
break_handler(int sig)
/* handle SIGINT */
{
  SIGHANDLER_RESTORE(sig, break_handler);
  brkflag = 1;
#if defined(USE_READLINE) && defined(HAVE_POSIX_SIGNALS)
  if (longjump_set) longjmp(longjmp_target, 1);
#endif
  SIGHANDLER_RETURN(0);
}

RETSIGTYPE
term_handler(int sig)
/* handle SIGTERM */
{
  SIGHANDLER_RESTORE(sig, term_handler);
  quitflag = 1;
#if defined(USE_READLINE) && defined(HAVE_POSIX_SIGNALS)
  if (longjump_set) longjmp(longjmp_target, 1);
#endif
  SIGHANDLER_RETURN(0);
}

RETSIGTYPE
tty_handler(int sig)
/* handle SIGTTIN/SIGTTOU */
{
  SIGHANDLER_RESTORE(sig, tty_handler);
#if defined (USE_READLINE)
  rl_deprep_terminal();
#endif
#ifdef SIGTTIN
  fprintf(stderr, "%s[pid %d]: stopped (tty %s)\n", self, getpid(),
	  (sig==SIGTTIN)?"input":"output");
#endif
#ifndef _WIN32
  raise(SIGSTOP);
#endif
  SIGHANDLER_RETURN(0);
}

RETSIGTYPE
segv_handler(int sig)
/* handle fatal program errors */
{
  static volatile bool recursive = 0;
  /* when we come here, many things can already be broken; we proceed (with
     fingers crossed) anyway */
  if (recursive)
    SIGHANDLER_RETURN(0);
  else
    recursive = 1;
#if defined (USE_READLINE)
  rl_deprep_terminal();
#endif
#ifdef HAVE_BACKTRACE
#ifdef HAVE_STRSIGNAL
  fprintf(stderr, "%s[pid %d]: caught signal %d (%s), printing backtrace... ",
	  self, getpid(), sig, strsignal(sig));
#else
  fprintf(stderr, "%s[pid %d]: caught signal %d, printing backtrace... ",
	  self, getpid(), sig);
#endif
  fflush(stderr);
  print_trace();
#else
#ifdef HAVE_STRSIGNAL
  fprintf(stderr, "%s[pid %d]: caught signal %d (%s), exiting\n",
	  self, getpid(), sig, strsignal(sig));
#else
  fprintf(stderr, "%s[pid %d]: caught signal %d, exiting\n",
	  self, getpid(), sig);
#endif
#endif
  fflush(NULL);
  syssignal(sig, SIG_DFL);
  raise(sig);
  SIGHANDLER_RETURN(0);
}

RETSIGTYPE (*old_handler)() = NULL;

void push_sigint(RETSIGTYPE (*new_handler)())
{
  if (!old_handler)
    old_handler = sigint(new_handler);
}

void pop_sigint(void)
{
  if (old_handler) {
    sigint(old_handler);
    old_handler = NULL;
  }
}

/* gmp memory handlers */

static void *
gmp_allocate (size)
     size_t size;
{
  void *ret;

  ret = malloc (size);
  return ret;
}

static void *
gmp_reallocate (oldptr, old_size, new_size)
     void *oldptr;
     size_t old_size;
     size_t new_size;
{
  void *ret;

  ret = realloc (oldptr, new_size);
  return ret;
}

static void
gmp_free (blk_ptr, blk_size)
     void *blk_ptr;
     size_t blk_size;
{
  free (blk_ptr);
}

/* safe (non-leaking) version of _mpz_realloc */

void *my_mpz_realloc(mpz_ptr m, mp_size_t new_size)
{
  mpz_t m1;
  memcpy(m1, m, sizeof(mpz_t));
  if (_mpz_realloc(m, new_size))
    return m->_mp_d;
  else {
    if (m1->_mp_d) mpz_clear(m1);
    return NULL;
  }
}

#ifdef _WIN32

/* win32 tmpnam is broken, provide a reasonable replacement */

#define tmpnam mytmpnam

static char *tmpnam(char *s)
{
  static char *t = NULL;
  char *p;
  if (t) free(t);
  t = _tempnam(NULL, "t");
  if (!t) return NULL;
  /* make sure to convert all \'s to /'s */
  while ((p = strchr(t, '\\'))) *p = '/';
  if (s) {
    strcpy(s, t);
    return s;
  } else
    return t;
}

#endif

/* readline support */

#ifdef USE_READLINE

#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
static char *rl_codeset = NULL;
#endif
static char **fsyms = NULL, **vsyms = NULL;

static char *csyms[] = {
  "break", "cd", "chdir", "clear", "completion_matches", "copying",
  "cstacksize", "debug", "dec", "def", "echo", "edit", "fix", "format",
  "help", "hex", "histfile", "histsize", "import", "imports", "load",
  "ls", "memsize", "modules", "oct", "off", "on", "path", "profile",
  "prompt", "pwd","run", "save", "sci", "source", "stacksize", "stats",
  "std", "tbreak", "undef", "which", "who", "whos",
  NULL
};

static comp(x, y)
     char **x, **y;
{
  return strcmp(*x, *y);
}

static build_fsym_table()
{
  int i, fno;
  fsyms = (char**)calloc(2*(symtbsz-BINARY)+1, sizeof(char*));
  if (!fsyms) return;
  for (i = 0, fno = BINARY; fno < symtbsz; fno++) {
    char *base = strsp+symtb[fno].pname, name[MAXSTRLEN];
    if (('a' <= *base && *base <= 'z' ||
	 'A' <= *base && *base <= 'Z' ||
	 *base == '_') &&
	!(symtb[fno].flags & TSYM)
	/*&& (!(symtb[fno].flags & VSYM)||symtb[fno].x)*/) {
      pname(name, fno);
      fsyms[i++] = utf8_to_sys_dup(name);
      if (visible(fno) && unique(fno) && (fno >= BUILTIN || symtb[fno].prec == NONE)) {
	if (symtb[fno].modno == NONE)
	  sprintf(name, "::%s", strsp + symtb[fno].pname);
	else
	  sprintf(name, "%s::%s", strsp + modtb[symtb[fno].modno],
		  strsp + symtb[fno].pname);
	fsyms[i++] = utf8_to_sys_dup(name);
      }
    }
  }
  qsort(fsyms, i, sizeof(char*), comp);
}

static build_vsym_table()
{
  int i, fno;
  vsyms = (char**)calloc(2*tmptbsz+1, sizeof(char*));
  if (!vsyms) return;
  for (i = 0, fno = symtbsz; fno < symtbsz+tmptbsz; fno++) {
    char *base = strsp+symtb[fno].pname, name[MAXSTRLEN];
    if (('a' <= *base && *base <= 'z' ||
	 'A' <= *base && *base <= 'Z' ||
	 *base == '_') &&
	!(symtb[fno].flags & TSYM) &&
	(!(symtb[fno].flags & VSYM)||symtb[fno].x)) {
      pname(name, fno);
      vsyms[i++] = utf8_to_sys_dup(name);
      if (visible(fno) && unique(fno)) {
	if (symtb[fno].modno == NONE)
	  sprintf(name, "::%s", strsp + symtb[fno].pname);
	else
	  sprintf(name, "%s::%s", strsp + modtb[symtb[fno].modno],
		  strsp + symtb[fno].pname);
	vsyms[i++] = utf8_to_sys_dup(name);
      }
    }
  }
  qsort(vsyms, i, sizeof(char*), comp);
}

static search(table, text, len)
     char **table, *text;
     int len;
{
  /* hmm, this should be binary search, maybe next time ... ;-) */
  int i = 0;
  while (table[i] && strncmp(text, table[i], len) > 0) i++;
  return i;
}

static clear_fsym_table()
{
  int i;
  if (!fsyms) return;
  for (i = 0; fsyms[i]; i++)
    free(fsyms[i]);
  free(fsyms);
  fsyms = NULL;
}

static clear_vsym_table()
{
  int i;
  if (!vsyms) return;
  for (i = 0; vsyms[i]; i++)
    free(vsyms[i]);
  free(vsyms);
  vsyms = NULL;
}

static char *
sym_generator (text, state)
     char *text;
     int state;
{
  static int i_csym, i_fsym, i_vsym, len;
  int cmp;
  char *name;
     
  if (!state)
    {
      len = strlen (text);
      if (!fsyms) {
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
	char *codeset = default_encoding();
	if (rl_codeset) free(rl_codeset);
	if (codeset)
	  rl_codeset = strdup(codeset);
	else
	  rl_codeset = NULL;
#endif
	build_fsym_table();
      }
      if (!vsyms)
	build_vsym_table();
      if (!fsyms || !vsyms)
	fatal("memory overflow");
      i_csym = search(csyms, text, len);
      i_fsym = search(fsyms, text, len);
      i_vsym = search(vsyms, text, len);
    }

  /* command keywords */
  while (name = csyms[i_csym])
    {
      i_csym++;

      if ((cmp = strncmp (name, text, len)) == 0)
	return (strdup(name));
      else if (cmp > 0)
	break;
    }

  /* function or predefined var symbol */
  while (name = fsyms[i_fsym])
    {
      i_fsym++;

      if ((cmp = strncmp (name, text, len)) == 0)
	return (strdup(name));
      else if (cmp > 0)
	break;
    }

  /* temporary variable symbols */
  while (name = vsyms[i_vsym])
    {
      i_vsym++;
      
      if ((cmp = strncmp (name, text, len)) == 0)
	return (strdup(name));
      else if (cmp > 0)
	break;
    }
  
  return ((char *)NULL);
}

#ifndef HAVE_RL_COMPLETION_MATCHES
#define rl_completion_matches completion_matches
#endif

char **sym_completion (char *text, int start, int end)
{
  return rl_completion_matches (text, sym_generator);
}

int use_readline = 0;
static bool history_read = 0;
static HISTORY_STATE *save_hist = NULL;

static void init_readline(void)
{
  rl_readline_name = "Q";
  rl_basic_word_break_characters = " \t\n\"\\'`@$><=,;|%&~{[(";
  rl_attempted_completion_function = (CPPFunction *)sym_completion;
  save_hist = history_get_history_state();
}

static void init_history(void)
{
  stifle_history(histmax);
  if (!history_read && !gflag && use_readline) {
    char h[MAXSTRLEN];
    if (histmax > 0)
      read_history(expand(h, histfile));
    stifle_history(histmax);
    history_read = 1;
  }
}

static void fini_readline(void)
{
  if (history_read && histmax > 0) {
    char h[MAXSTRLEN];
    write_history(expand(h, histfile));
  }
}

#endif

void switch_history(void)
{
#ifdef USE_READLINE
  HISTORY_STATE *hist;
  if (!save_hist) return;
  hist = history_get_history_state();
  history_set_history_state(save_hist);
  free(save_hist);
  save_hist = hist;
  stifle_history(histmax);
#endif
}

void list_completions(char *s)
{
#ifdef USE_READLINE
  char **matches = sym_completion(s, 0, strlen(s));
  if (matches) {
    if (matches[0])
      if (!matches[1]) {
	printf("%s\n", matches[0]);
	free(matches[0]);
      } else {
	int i;
	free(matches[0]);
	for (i = 1; matches[i]; i++) {
	  printf("%s\n", matches[i]);
	  free(matches[i]);
	}
      }
    free(matches);
  }
#endif
}

static char *mygetline1(fp, prompt)
     FILE *fp;
     char *prompt;
{
  char *buf, *bufp;
#ifdef USE_READLINE
  if (fp == stdin && use_readline && isatty(fileno(stdin))) {
    static char *last = NULL;
#ifdef HAVE_POSIX_SIGNALS
    save_sigmask();
    if (setjmp(longjmp_target) == 0) {
      longjump_set = 1;
      buf = readline(prompt);
      longjump_set = 0;
    } else {
      /* clean up */
      longjump_set = 0;
      rl_free_line_state();
      rl_cleanup_after_signal();
/*       printf("\n"); */
      buf = malloc(sizeof(char));
      if (!buf) fatal("memory overflow");
      *buf = 0;
      restore_sigmask();
    }
#else
    buf = readline(prompt);
#endif
    if (buf && *buf &&
	(history_length <= 0 ||
	 strcmp(buf, history_get(history_length-1+history_base)->line) != 0))
      add_history(buf);
    return buf;
  } else {
#endif
    int chunksz = 10000;
    int actsz, l;
    buf = malloc(chunksz*sizeof(char));
    actsz = chunksz;
    bufp = buf;
    if (bufp) {
      *bufp = 0;
      if (iflag && fp == stdin) {
	printf("%s", prompt);
	fflush(stdout);
      }
    }
    while (buf && !ferror(fp) && !feof(fp)) {
      int k;
      if (!fgets(bufp, chunksz, fp) || ferror(fp) || feof(fp) ||
	  (l = strlen(bufp)) > 0 && bufp[l-1] == '\n')
	break;
      /* enlarge the buffer */
      k = bufp-buf+l;
      actsz += chunksz;
      buf = realloc(buf, actsz*sizeof(char));
      bufp = buf+k;
    }
    if (buf)
      if (*buf) {
	l = strlen(buf);
	if (buf[l-1] == '\n')
	  buf[--l] = 0;
	buf = realloc(buf, (strlen(buf)+1)*sizeof(char));
      } else {
	free(buf);
	return NULL;
      }
    if (buf)
      if (ferror(fp) || feof(fp) && !*buf) {
	free(buf);
	return NULL;
      } else
	return buf;
    else
      fatal("memory overflow");
#ifdef USE_READLINE
  }
#endif
}

static contd();

int actlineno = 0;

static char *ps = NULL, *psx = NULL, *psdef = PROMPT;

void new_xprompt(void)
{
  if (psx && psx != psdef) free(psx);
  ps = psx = NULL;
}

static char *xprompt(char *prompt)
{
  if (ps != prompt) {
    /* expand placeholders in prompt string */
    int l, sl = strlen(sysinfo), vl = strlen(version), wl, WL, ml, ML;
    char *s, *buf, wd[MAXSTRLEN], WD[MAXSTRLEN];
    char *m = (mainno==-1)?"":strsp+fnametb[mainno];
    char *M = (mainno==-1)?"":strsp+modtb[mainno];
    if (!getcwd(wd, MAXSTRLEN)) strcpy(wd, "");
    basename(WD, wd, 0);
    wl = strlen(wd); WL = strlen(WD);
    ml = strlen(m); ML = strlen(M);
    s = ps = prompt; l = strlen(ps);
    if (psx && psx != psdef) free(psx);
    while ((s = strstr(s, "\\v"))) {
      l += vl-2; s++;
    }
    s = ps;
    while ((s = strstr(s, "\\s"))) {
      l += sl-2; s++;
    }
    s = ps;
    while ((s = strstr(s, "\\w"))) {
      l += wl-2; s++;
    }
    s = ps;
    while ((s = strstr(s, "\\W"))) {
      l += WL-2; s++;
    }
    s = ps;
    while ((s = strstr(s, "\\m"))) {
      l += ml-2; s++;
    }
    s = ps;
    while ((s = strstr(s, "\\M"))) {
      l += ML-2; s++;
    }
    if ((psx = (char*)malloc((l+1)*sizeof(char))) &&
	(buf = (char*)malloc((l+1)*sizeof(char)))) {
      strcpy(psx, ps);
      while ((s = strstr(psx, "\\v"))) {
	strcpy(buf, s+2); strcpy(s, version);
	strcpy(s+vl, buf);
      }
      while ((s = strstr(psx, "\\s"))) {
	strcpy(buf, s+2); strcpy(s, sysinfo);
	strcpy(s+sl, buf);
      }
      while ((s = strstr(psx, "\\w"))) {
	strcpy(buf, s+2); strcpy(s, wd);
	strcpy(s+wl, buf);
      }
      while ((s = strstr(psx, "\\W"))) {
	strcpy(buf, s+2); strcpy(s, WD);
	strcpy(s+WL, buf);
      }
      while ((s = strstr(psx, "\\m"))) {
	strcpy(buf, s+2); strcpy(s, m);
	strcpy(s+ml, buf);
      }
      while ((s = strstr(psx, "\\M"))) {
	strcpy(buf, s+2); strcpy(s, M);
	strcpy(s+ML, buf);
      }
      free(buf);
    } else {
      if (psx) free(psx);
      psx = psdef;
    }
  }
  return psx;
}

char *mygetline(FILE *fp, char *prompt, int expand)
{
  char *buf;
  int l;
#ifdef USE_READLINE
  if (fp == stdin) {
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
    char *codeset = default_encoding();
    if ((rl_codeset != NULL) != (codeset != NULL) ||
	codeset != NULL && strcmp(codeset, rl_codeset) != 0)
      clear_fsym_table();
#endif
    clear_vsym_table();
  }
#endif
  if (!(buf = mygetline1(fp, expand?xprompt(prompt):prompt))) return NULL;
  l = strlen(buf);
  actlineno++;
  while (contd(buf)) {
    char *buf2;
    if (!(buf2 = mygetline1(fp, prompt2))) break;
    actlineno++;
    buf[l] = '\n'; 
    if ((buf = realloc(buf, (l+strlen(buf2)+2)*sizeof(char)))) {
      strcpy(buf+l+1, buf2);
      l += strlen(buf2)+1;
      free(buf2);
    } else
      fatal("memory overflow");
  }
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
  {
    char *buf1 = ictoutf8(((EXPR*)symtb[INPUTOP].x)->data.fargs.ic, buf);
    if (buf1) { free(buf); buf = buf1; }
  }
#endif
  return buf;
}

static int contd(char *p)
{
  int strmode = 0, l;
  if (!p) return 0;
  l = strlen(p);
  for (; *p; p++) {
    if (strmode) {
      if (*p == '\\') {
	if (!*++p)
	  return 1;
      } else if (*p == '"')
	strmode = 0;
    } else if (*p == '%' || strncmp(p, "//", 2) == 0)
      return 0;
    else if (*p == '"')
      strmode = 1;
  }
  return l >= 1 && p[-1] == '\\';
}

/* command line options */

/* this should be plenty */
#define MAXARGC 1200

static char	       *qcargv[MAXARGC+1];
static int		qcargc;

static wsarg(char *arg)
{
  while (*arg)
    if (isspace(*arg))
      return 1;
    else
      arg++;
  return 0;
}

static char *quotearg(char *arg)
{
  if (!arg) return NULL;
#ifdef _WIN32
  /* MS C lib kludge: empty command args or args containing whitespace need
     quoting */
  if (!*arg || wsarg(arg)) {
    char *newarg = malloc(2*strlen(arg)+3), *s;
    if (!newarg)
      return 0;
    s = newarg;
    *s++ = '"';
    while (*arg) {
      if (*arg == '"') *s++ = '\\';
      *s++ = *arg++;
    }
    *s++ = '"';
    *s = 0;
    return newarg;
  } else
#endif
    return strdup(arg);
}

static qcarg(char *arg)
{
  if (qcargc >= MAXARGC)
    fatal("too many command line parameters");
  else if (arg && !(arg = quotearg(arg)))
    fatal("memory overflow");
  else {
    if (qcargv[qcargc]) free(qcargv[qcargc]);
    qcargv[qcargc++] = arg;
  }
}

static set_qcarg(int i, char *arg)
{
  if (i >= MAXARGC)
    fatal("too many command line parameters");
  else if (arg && !(arg = quotearg(arg)))
    fatal("memory overflow");
  else {
    if (qcargv[i]) free(qcargv[i]);
    qcargv[i] = arg;
  }
}

static char *list = "", *hsz = "";
static struct option longopts[] = Q_OPTS;

static int
getintarg(char *s, int *i)
{
  char *t = s;
  while (isspace(*t)) t++;
  s = t;
  while (isdigit(*t)) t++;
  if (t == s) return 0;
  while (isspace(*t)) t++;
  if (*t) return 0;
  *i = atoi(s);
  return 1;
}

static void
parse_opts(int argc, char **argv, int pass)
     /* pass = 0 denotes source, 1 command line pass */
{
  int c, longind;
  optind = 0;
  while ((c = getopt_long(argc, argv, Q_OPTS1, longopts,
			  &longind)) != EOF)
    switch (c) {
    case Q_GNUCLIENT:
      gflag = 1;
      break;
    case Q_DEBUG_OPTIONS: {
      char opts[MAXSTRLEN];
      strcpy(opts, optarg?optarg:"");
      if (!debug_parse_opts(opts)) {
	char msg[MAXSTRLEN];
	sprintf(msg, "invalid option string `%s'", optarg?optarg:"");
	fatal(msg);
      }
      break;
    } case Q_BREAK:
      brkdbg = 1;
      break;
    case Q_PROMPT:
      prompt = optarg;
      break;
    case Q_DEC:
      imode = 0;
      break;
    case Q_HEX:
      imode = 1;
      break;
    case Q_OCT:
      imode = 2;
      break;
    case Q_STD: {
      int prec;
      if (!optarg)
	prec = 15;
      else if (!getintarg(optarg, &prec))
	prec = -1;
      if (prec >= 0) {
	fmode = 0; fprec = prec; sprintf(fformat, STDFORMAT, prec);
      } else {
	char msg[MAXSTRLEN];
	sprintf(msg, "invalid precision `%s'", optarg?optarg:"");
	fatal(msg);
      }
      break;
    }
    case Q_SCI: {
      int prec;
      if (!optarg)
	prec = 15;
      else if (!getintarg(optarg, &prec))
	prec = -1;
      if (prec >= 0) {
	fmode = 1; fprec = prec; sprintf(fformat, SCIFORMAT, prec);
      } else {
	char msg[MAXSTRLEN];
	sprintf(msg, "invalid precision `%s'", optarg?optarg:"");
	fatal(msg);
      }
      break;
    }
    case Q_FIX: {
      int prec;
      if (!optarg)
	prec = 2;
      else if (!getintarg(optarg, &prec))
	prec = -1;
      if (prec >= 0) {
	fmode = 2; fprec = prec; sprintf(fformat, FIXFORMAT, prec);
      } else {
	char msg[MAXSTRLEN];
	sprintf(msg, "invalid precision `%s'", optarg?optarg:"");
	fatal(msg);
      }
      break;
    }
    case Q_HISTFILE:
      histfile = optarg;
      break;
    case Q_HISTSIZE: {
      int sz = optarg?atoi(optarg):0;
      if (optarg && getintarg(optarg, &sz) && sz >= 0)
	histmax = sz;
      else {
	char msg[MAXSTRLEN];
	sprintf(msg, "invalid size `%s'", optarg?optarg:"");
	fatal(msg);
      }
      break;
    }
    case Q_INITRC:
      initrc = optarg;
      break;
    case Q_NO_INITRC:
      initrc = NULL;
      break;
    case Q_EXITRC:
      exitrc = optarg;
      break;
    case Q_NO_EXITRC:
      exitrc = NULL;
      break;
    case Q_NO_EDITING:
      norl = 1;
      break;
    case Q_CSTACKSIZE: {
      int sz;
      if (optarg && getintarg(optarg, &sz) && sz >= 0) {
	sz *= 1024;
	if (sz == 0 || sz >= CSTACKMIN)
	  cstackmax = sz;
	else {
	  fprintf(stderr, "%s: bad C stack size `%d', using default\n", self,
		  sz/1024);
	  cstackmax = CSTACKMAX;
	}
      } else {
	char msg[MAXSTRLEN];
	sprintf(msg, "invalid size `%s'", optarg?optarg:"");
	fatal(msg);
      }
      break;
    }
    case Q_STACKSIZE: {
      int sz;
      if (optarg && getintarg(optarg, &sz) && sz >= 0)
	if (sz == 0 || sz >= STACKMIN)
	  stackmax = sz;
	else {
	  fprintf(stderr, "%s: bad stack size `%d', using default\n", self,
		  sz);
	  stackmax = STACKMAX;
	}
      else {
	char msg[MAXSTRLEN];
	sprintf(msg, "invalid size `%s'", optarg?optarg:"");
	fatal(msg);
      }
      break;
    }
    case Q_MEMSIZE: {
      int sz;
      if (optarg && getintarg(optarg, &sz) && sz >= 0) {
	if (sz == 0 || sz >= MEMMIN)
	  memmax = sz;
	else {
	  fprintf(stderr, "%s: bad memory size `%d', using default\n", self,
		  sz);
	  memmax = MEMMAX;
	}
	lastblksz = memmax % XBLKSZ;
	maxnblks = memmax/XBLKSZ+((memmax <= 0||lastblksz==0)?0:1);
	if (lastblksz == 0) lastblksz = XBLKSZ;
      } else {
	char msg[MAXSTRLEN];
	sprintf(msg, "invalid size `%s'", optarg?optarg:"");
	fatal(msg);
      }
      break;
    }
    case 'd':
      debug = 1;
      break;
    case 'e':
      eflag = 1;
      break;
    case 'h':
      hflag = 1;
      break;
    case 'i':
      iflag = 1;
      break;
    case 'q':
      qflag = 1;
      break;
    case 'c':
    case 's':
      /* these will be taken care of later ... */
      batch = 1;
      break;
    case 'V':
      Vflag = 1;
      break;
    /* qc options: */
    case QC_PEDANTIC:
      if (pass) qcarg("--pedantic");
      break;
    case QC_PARANOID:
      if (pass) qcarg("--paranoid");
      break;
    case QC_NO_PRELUDE:
      prelude = NULL;
      if (pass) qcarg("--no-prelude");
      break;
    case QC_PRELUDE:
      prelude = optarg?optarg:prelude;
      if (pass) {
	qcarg("--prelude");
	qcarg(prelude);
      }
      break;
    case QC_ENCODING:
      default_codeset = optarg?optarg:default_codeset;
      if (pass) {
	qcarg("--encoding");
	qcarg(default_codeset);
      }
      break;
    case 'n':
      /* ignored; we need the code file ;-) */
      break;
    case 'l':
      if (pass) {
	list = optarg?optarg:list;
	qcarg("-l");
	qcarg(list);
      }
      break;
    case 'o':
      code = optarg?optarg:code;
      oset = 1;
      if (pass) {
	qcarg("-o");
	qcarg(code);
      }
      break;
    case 'p':
      if (optarg && !pass) {
	change_qpath(optarg);
	if (!qpath) fatal("memory overflow");
      }
      break;
    case 't':
      if (pass) {
	hsz = optarg?optarg:hsz;
	qcarg("-t");
	qcarg(hsz);
      }
      break;
    case 'v':
      if (pass) qcarg("-v");
      break;
    case 'w':
      if (pass) {
	static char optstr[13];
	char *level = optarg?optarg:"";
	strcpy(optstr, "-w");
	strncpy(optstr+2, level, 10);
	qcarg(optstr);
      }
      break;
    default:
      fatal(NULL);
    }
}

static int sargc = 0;
static char **sargv = NULL;

static void
get_source_opts(FILE *fp)
{
  char s[MAXSTRLEN];
  int i;

  sargc = 1;
  sargv = aalloc(1, sizeof(char*));
  *sargv = strdup(self);
  while (!feof(fp) && !ferror(fp) &&
	 fgets(s, MAXSTRLEN, fp)) {
    int l = strlen(s);
    if (l > 0 && s[l-1] == '\n') s[l-1] = '\0', l--;
    if (l == 0)
      continue;
    else if (strncmp(s, "#!", 2) == 0)
      if (isspace(s[2])) {
	char *p = s+3;
	while (isspace(*p)) p++;
	sargv = arealloc(sargv, sargc, 1, sizeof(char*));
	sargv[sargc++] = strdup(p);
      } else
	continue;
    else
      break;
  }
  sargv = arealloc(sargv, sargc, 1, sizeof(char*));
  sargv[sargc] = NULL;
}

/* spawn child processes */

#ifndef _WIN32
static volatile int pid = 0;
static RETSIGTYPE
spawn_term_handler(int sig)
/* pass termination signals to child process */
{
  if (pid) kill(pid, sig);
  SIGHANDLER_RETURN(0);
}
#endif

static
spawn(char *prog, char *argv[])
{
  RETSIGTYPE (*oldint)(), (*oldterm)(), (*oldhup)();
  int status;
#ifdef _WIN32
  oldint = sigint(SIG_IGN);
  status = spawnvp(P_WAIT, prog, (const char* const*)argv);
  if (status < 0)
    fatal("exec failed -- check installation");
  else {
    sigint(oldint);
    return status;
  }
#else
  oldterm = sigterm(SIG_IGN);
  switch ((pid = fork())) {
  case 0:
    execvp(prog, argv);
  case -1:
    fatal("exec failed -- check installation");
  }
  oldint = sigint(SIG_IGN);
  sigterm(spawn_term_handler);
  oldhup = sighup(spawn_term_handler);
  waitpid(pid, &status, 0);
  sigint(oldint);
  sigterm(oldterm);
  sighup(oldhup);
  return status;
#endif
}

/* rerun the interpreter */

static resolve(), init_dlls();
static void evaldefs(void);
extern int ximpsz;
extern char **ximp;

static int dorun(const char *path, int argc, char *const *argv)
{
  char *_source = path?((char*)path):source;
  char fname[MAXSTRLEN], fname2[MAXSTRLEN], msg[MAXSTRLEN];
  FILE *fp = NULL;
  int i, nargs;
  EXPR *args, *in, *out, *err;

  /* check the new source script */
  if (!_source || !*_source || strcmp(_source, "-") == 0 ||
      chkfile(searchlib(fname, _source)) &&
      (fp = fopen(fname, "rb")) != NULL ||
      chkfile(searchlib(fname, strcat(strcpy(fname2, _source), ".q"))) &&
      (fp = fopen(fname, "rb")) != NULL) {
    int res;
    if (!fp || !(res = iscode(fp))) {
      /* check whether the file compiles ok */
      if (fp) fclose(fp);
      set_qcarg(qcargc-2, qpath);
      set_qcarg(qcargc-1, (_source||!ximpsz)?_source:"");
      if (!oset) {
	code = tmpnam(NULL);
	set_qcarg(qcargc-4, code);
      }
      for (i = 0; i < ximpsz; i++)
	qcarg(ximp[i]);
      remove(code);
      if (spawn(qcprog, qcargv)) {
	for (i = 0; i < ximpsz; i++)
	  free(qcargv[--qcargc]), qcargv[qcargc] = NULL;
	set_qcarg(qcargc-1, source);
	remove(code);
	return COMPILE_ERR;
      } else
	for (i = 0; i < ximpsz; i++)
	  free(qcargv[--qcargc]), qcargv[qcargc] = NULL;
      if (_source != source) {
	if (source) free(source);
	if (_source) {
	  _source = strdup(_source);
	  if (!_source) fatal("memory overflow");
	}
	source = _source;
      }
      donecompile = 1;
    } else {
      fclose(fp);
      if (res == -1)
	return FILE_FORMAT_ERR;
      if (_source != source) {
	if (source) free(source);
	if (_source) {
	  _source = strdup(_source);
	  if (!_source) fatal("memory overflow");
	}
	source = _source;
      }
      code = source;
      if (ximpsz) {
	int i;
	for (i = 0; i < ximpsz; i++)
	  free(ximp[i]);
	ximpsz = 0;
      }
    }
  } else {
    errno = 0;
    return FILE_NOT_FOUND;
  }
  /* source exitrc file if necessary */
  if (doexitrc && exitrc && chkfile(expand(fname, exitrc)) &&
      (fp = fopen(fname, "r"))) {
    fclose(fp);
    parsesrc(fname, 1);
    if (quitflag || thr0->qmstat == QUIT) exit(0);
  }
  /* kill all threads and wait for them to finish */
  kill_threads(); wait_threads();
  /* purge all variables (except builtins) */
  qmfree(thr0, symtb[DEFVAROP].x);
  symtb[DEFVAROP].x = NULL;
  symtb[DEFVAROP].flags &= ~MODIF;
  for (i = BUILTIN; i<symtbsz+tmptbsz; i++)
    if (symtb[i].x) {
      qmfree(thr0, symtb[i].x);
      symtb[i].x = NULL;
      symtb[i].flags &= ~MODIF;
    }
  /* we only keep the ARGS variable if we have been invoked without args */
  if (argc > 0 && symtb[ARGSOP].x) {
    qmfree(thr0, symtb[ARGSOP].x);
    symtb[ARGSOP].x = NULL;
  } else {
    /* record the number of arguments (needed below) */
    EXPR *x = symtb[ARGSOP].x;
    nargs = 0;
    while (x->fno == CONSOP) {
      nargs++;
      x = x->data.args.x2;
    }
  }
  /* clean up the main stack and the heap */
  clear(1);
  /* save the builtin variables s.t. we can recover them after the symbol
     table has been reinitialized */
  in = symtb[INPUTOP].x;
  out = symtb[OUTPUTOP].x;
  err = symtb[ERROROP].x;
  if (argc <= 0) args = symtb[ARGSOP].x;
  /* reinitialize signal handlers, in case some modules installed their own */
  sigint(break_handler); sigterm(term_handler); sighup(term_handler);
#ifdef SIGTTIN
  syssignal(SIGTTIN, tty_handler);
#endif
#ifdef SIGTTOU
  syssignal(SIGTTOU, tty_handler);
#endif
#ifdef SIGFPE
  syssignal(SIGFPE, segv_handler);
#endif
#ifdef SIGILL
  syssignal(SIGILL, segv_handler);
#endif
#ifdef SIGSEGV
  syssignal(SIGSEGV, segv_handler);
#endif
#ifdef SIGBUS
  syssignal(SIGBUS, segv_handler);
#endif
#ifdef SIGTRAP
  syssignal(SIGTRAP, segv_handler);
#endif
  /* unload dlls */
  unload_dlls();
  /* purge the code table */
  if (codesp) free(codesp); codesp = NULL; codespsz = 0;
  if (strsp) free(strsp); strsp = NULL; strspsz = tmpspsz = 0;
  atmpspsz = TMPSPSZ;
  if (limbsp) free(limbsp); limbsp = NULL; limbspsz = 0;
  if (hashtb) free(hashtb); hashtb = NULL; hashtbsz = 0;
  if (symtb) free(symtb); symtb = NULL; symtbsz = tmptbsz = 0;
  atmptbsz = TMPTBSZ;
  if (statetb) free(statetb); statetb = NULL; statetbsz = 0;
  if (transtb) free(transtb); transtb = NULL; transtbsz = 0;
  if (roffstb) free(roffstb); roffstb = NULL; roffstbsz = 0;
  if (matchtb) free(matchtb); matchtb = NULL; matchtbsz = 0;
  if (inittb) free(inittb); inittb = NULL; inittbsz = 0;
  if (modtb) free(modtb); modtb = NULL;
  if (fnametb) free(fnametb); fnametb = NULL; modtbsz = 0;
  /* reinitialize the module search path, in case some modules modified it */
  lt_dlsetsearchpath(qpath);
  /* reinitialize */
  readtables();
  resolve();
  reinit();
  if (donecompile) remove(code);
  donecompile = 0;
  /* standard I/O streams */
  symtb[INPUTOP].x = in;
  symtb[OUTPUTOP].x = out;
  symtb[ERROROP].x = err;
  /* ARGS variable */
  if (argc > 0) {
    int count = 0; char *const *argv0 = argv;
    while (argc-- > 0) {
      char *s;
      if (*argv)
	s = sys_to_utf8_dup(*argv++);
      else
	s = strdup("");
      if (!s)
	fatal("memory overflow");
      else if (!pushstr(thr0, s))
	fatal("stack overflow");
      else
	count++;
    }
    if (!pushfun(thr0, NILOP))
      fatal("stack overflow");
    while (count-- > 0)
      if (!pushfun(thr0, CONSOP))
	fatal("memory overflow");
    symtb[ARGSOP].x = (void*) thr0->xsp[-1];
    thr0->xsp = thr0->xst;
  } else {
    /* KLUDGE ALERT: we have to fix the nil cell in the last component, since
       this is a preallocated function symbol and the symbol array may have
       moved after reinit() */
    if (nargs == 0)
      symtb[ARGSOP].x = qmnew(funexpr(thr0, NILOP));
    else {
      EXPR *x = args;
      while (nargs-- > 1)
	x = x->data.args.x2;
      x->data.args.x2 = qmnew(funexpr(thr0, NILOP));
      symtb[ARGSOP].x = args;
    }
  }
  /* `which' */
  if (which) free(which);
  if (mainno == -1)
    which = "";
  else
    which = strsp+fnametb[mainno];
  which = strdup(which);
  if (!which) fatal("memory overflow");
  /* module initializations */
  init_dlls();
  /* script initialization code */
  errno = 0;
  if (inittbsz) {
    evaldefs();
    clear(0);
    clearerr(stdin);
  }
  /* source initrc file */
  doexitrc = 1;
  if (iflag && initrc && chkfile(expand(fname, initrc)) &&
      (fp = fopen(fname, "r"))) {
    fclose(fp);
    parsesrc(fname, 1);
    if (quitflag || thr0->qmstat == QUIT) exit(0);
  }
  /* notify gnu server */
  if (gflag)
    if (source)
      gcmd_s("q-run-cmd", source);
    else
      gcmd_b("q-run-cmd", 0);
#if defined (USE_READLINE)
  clear_fsym_table();
  clear_vsym_table();
#endif
  new_xprompt();
  errno = 0;
  return OK;
}

int rerun(const char *path, int argc, char *const *argv)
{
  int status = dorun(path, argc, argv);
  if (status) {
    char *_source = path?((char*)path):source, msg[MAXSTRLEN];
    if (!_source || !*_source) _source = "script";
    sprintf(msg, qmmsg[status], _source);
    error(msg);
    return 0;
  } else
    return 1;
}

/* additional interface functions required by libq */

EXPR *_qinter_intexpr(long i)
{
  return intexpr(get_thr(), i);
}

EXPR *_qinter_uintexpr(unsigned long i)
{
  return uintexpr(get_thr(), i);
}

EXPR *_qinter_mpzexpr(mpz_t z)
{
  return mpzexpr(get_thr(), z);
}

EXPR *_qinter_floatexpr(double f)
{
  return floatexpr(get_thr(), f);
}

EXPR *_qinter_strexpr(char *s)
{
  return strexpr(get_thr(), s);
}

EXPR *_qinter_fileexpr(FILE *fp)
{
  return fileexpr(get_thr(), fp);
}

EXPR *_qinter_pipeexpr(FILE *fp)
{
  return pipeexpr(get_thr(), fp);
}

EXPR *_qinter_funexpr(int fno)
{
  THREAD *thr = get_thr();
  bool actmode = thr->mode;
  EXPR *ret;
  if (fno >= RESERVED && !(symtb[fno].flags&CST)) {
    /* force quote mode on */
    thr->mode = 1;
    ret = funexpr(thr, fno);
    thr->mode = actmode;
  } else
    ret = funexpr(thr, fno);
  return ret;
}

EXPR *_qinter_usrexpr(int type, void *vp)
{
  return usrexpr(get_thr(), type, vp);
}

EXPR *_qinter_vectexpr(int n, EXPR **xv)
{
  return vectexpr(get_thr(), n, xv);
}

EXPR *_qinter_mpz_floatexpr(double f)
{
  mpz_t z;
  double ip, fp;
  int sz;
  fp = modf(f, &ip);
  if (ip < 0) ip = -ip;
  sz = log(ip)/log(2)+1;
  if (sz < 0) return 0;
  sz = sz/(CHAR_BIT*sizeof(mp_limb_t)) + 2;
  mpz_init(z);
  if (z->_mp_d && my_mpz_realloc(z, sz)) {
    EXPR *x;
    int __sz;
    mpz_set_d(z, f);
    __sz = mpz_size(z);
    if (__sz < sz && !my_mpz_realloc(z, __sz)) {
      get_thr()->qmstat = MEM_OVF;
      return NULL;
    }
    x = mpzexpr(get_thr(), z);
    return x;
  } else {
    get_thr()->qmstat = MEM_OVF;
    return NULL;
  }
}

EXPR *_qinter_consexpr(int fno, EXPR *x1, EXPR *x2)
{
  if (fno == APPOP) {
    THREAD *thr = get_thr();
    bool actmode = thr->mode;
    EXPR *ret;
    /* force quote mode on */
    thr->mode = 1;
    ret = consexpr(thr, fno, x1, x2);
    thr->mode = actmode;
    return ret;
  } else
    return consexpr(get_thr(), fno, x1, x2);
}

int getintexpr(EXPR *x, long *i)
{
  if (x->fno == INTVALOP && my_mpz_fits_slong_p(x->data.z)) {
    *i = mpz_get_si(x->data.z);
    return 1;
  } else
    return 0;
}

int getuintexpr(EXPR *x, unsigned long *i)
{
  if (x->fno == INTVALOP && mpz_sgn(x->data.z) >= 0 &&
      my_mpz_fits_ulong_p(x->data.z)) {
    *i = mpz_get_ui(x->data.z);
    return 1;
  } else
    return 0;
}

int getmpzexpr(EXPR *x, mpz_t z)
{
  if (x->fno == INTVALOP) {
    memcpy(z, x->data.z, sizeof(mpz_t));
    return 1;
  } else
    return 0;
}

int getmpz_floatexpr(EXPR *x, double *f)
{
  mpz_t z;
  if (x->fno == INTVALOP) {
    *f = mpz_get_d(x->data.z);
    return 1;
  } else
    return 0;
}

EXPR *_qinter_eval(EXPR *x)
{
  THREAD *thr = get_thr();
  if (x && eval(thr, x)) {
    thr->xsp[-1]->refc--;
    return *--thr->xsp;
  } else
    return NULL;
}

void _qinter_free(EXPR *x)
{
  qmfree(get_thr(), x);
}

void _qinter_sentinel(EXPR *x)
{
  qmsentinel(get_thr(), x);
}

int issym(int sym)
{
  return sym >= BINARY && sym < symtbsz+tmptbsz &&
    !(symtb[sym].flags & TSYM);
}

int istype(int type)
{
  return type >= BINARY && type < symtbsz &&
    (symtb[type].flags & TSYM);
}

int isusrtype(int type)
{
  return istype(type) && (symtb[type].flags & EXT);
}

int _qinter_getsym(const char *name, int modno)
{
  char *s = strdup(name);
  int ret;
  if (s) {
    ret = getsym(s, modno);
    free(s);
  } else
    ret = -1;
  return ret;
}

int _qinter_gettype(const char *name, int modno)
{
  char *s = strdup(name);
  int ret;
  if (s) {
    ret = gettype(s, modno);
    free(s);
  } else
    ret = -1;
  return ret;
}

void _qinter_error(void)
{
  THREAD *thr = get_thr();
  if (thr->qmstat == OK) thr->qmstat = EXT_ERR;
}

/* dl interface */

char **dll_name = NULL;
lt_dlhandle *dll_handle = NULL;
void (**dll_init)() = NULL, (**dll_fini)() = NULL;
thread_atfork_t *dll_atfork = NULL;

void thread_atfork(void (*prepare)(void), void (*parent)(void),
		   void (*child)(void), int modno)
{
  dll_atfork[modno].prepare = prepare;
  dll_atfork[modno].parent = parent;
  dll_atfork[modno].child = child;
}

extern __libq_init();

static resolve()
{
  int fno, count;

  /* initialize */

  if (modtbsz <= 0) return 1;

  dll_name = (char**)calloc(modtbsz, sizeof(char*));
  dll_handle = (lt_dlhandle*)calloc(modtbsz, sizeof(void*));
  dll_init = (void(**)())calloc(modtbsz, sizeof(void(**)()));
  dll_fini = (void(**)())calloc(modtbsz, sizeof(void(**)()));
  dll_atfork = (thread_atfork_t*)calloc(modtbsz, sizeof(thread_atfork_t));

  if (!dll_name || !dll_handle || !dll_init || !dll_fini || !dll_atfork)
    fatal("memory overflow");

  /* set up libq interface (back links into the interpreter) */

  __libq_init(_qinter_intexpr,
	      _qinter_uintexpr,
	      _qinter_mpzexpr,
	      _qinter_mpz_floatexpr,
	      _qinter_floatexpr,
	      _qinter_strexpr,
	      _qinter_fileexpr,
	      _qinter_pipeexpr,
	      _qinter_funexpr,
	      _qinter_usrexpr,
	      _qinter_consexpr,
	      _qinter_vectexpr,

	      getintexpr,
	      getuintexpr,
	      getmpzexpr,
	      getmpz_floatexpr,
	    
	      _qinter_eval,
	    
	      _qinter_free,
	      _qinter_sentinel,

	      issym,
	      istype,
	      isusrtype,
	      _qinter_getsym,
	      _qinter_gettype,

	      init_thread,
	      exit_thread,
	      fini_thread,
	      this_thread,
	      have_lock,
	      release_lock,
	      acquire_lock,
	      acquire_tty,
	      release_tty,
	      thread_atfork,

	      _qinter_error);

  /* resolve external symbols */

  for (fno = BINARY; fno < symtbsz; fno++)
    if (symtb[fno].flags & EXT) {
      int xfno = symtb[fno].xfno, modno = symtb[xfno].modno;
      char *fname = strsp+fnametb[modno];
      char sym[MAXSTRLEN];
      if (!dll_name[modno]) {
	char modname[MAXSTRLEN];
	basename(modname, fname, '.');
	dll_name[modno] = strdup(modname);
	if (!dll_name[modno])
	  fatal("memory overflow");
	else if (!(dll_handle[modno] = lt_dlopenext(dll_name[modno])))
	  fprintf(stderr, "%s: error loading module\n", fname);
	else {
	  char initmod[MAXSTRLEN], init[MAXSTRLEN], fini[MAXSTRLEN];
	  void (*__initmod)();
	  sprintf(initmod, "__%s__initmod", dll_name[modno]);
	  sprintf(init, "__%s__init", dll_name[modno]);
	  sprintf(fini, "__%s__fini", dll_name[modno]);
	  if (!(__initmod = lt_dlsym(dll_handle[modno], initmod))) {
	    fprintf(stderr, "%s: invalid module header\n", fname);
	    lt_dlclose(dll_handle[modno]);
	    dll_handle[modno] = NULL;
	  } else {
	    void (*__init)() = lt_dlsym(dll_handle[modno], init);
	    void (*__fini)() = lt_dlsym(dll_handle[modno], fini);
	    (*__initmod)(modno, gmp_allocate, gmp_reallocate, gmp_free);
	    dll_init[modno] = __init;
	    dll_fini[modno] = __fini;
	  }
	}
      }
      if (symtb[fno].flags & TSYM)
	strcpy(sym, "__D__");
      else
	strcpy(sym, "__F__");
      strcat(strcat(sym, dll_name[modno]), "_");
      strcat(sym, strsp+symtb[xfno].pname);
      if (dll_handle[modno] &&
	  !(symtb[fno].f = lt_dlsym(dll_handle[modno], sym)) &&
	  !(symtb[fno].flags & TSYM)) {
	char pn[MAXSTRLEN];
	fprintf(stderr, "%s: unresolved symbol `%s'\n", fname,
		pname(pn, fno));
      } else if (symtb[fno].f && symtb[fno].argc > maxargs)
	maxargs = symtb[fno].argc;
    }

  /* check for errors */

  for (count = 0, fno = BINARY; fno < symtbsz; fno++)
    if ((symtb[fno].flags & EXT) && !(symtb[fno].flags & TSYM) &&
	!symtb[fno].f)
      count++;
  if (count) {
    fprintf(stderr, "Warning: %d unresolved external symbol%s\n",
	    count, count>1?"s":"");
    fflush(stderr);
    return 0;
  } else
    return 1;
}

static init_dlls()
{
  if (dll_name) {
    int i;
    for (i = 0; i < modtbsz; i++)
      if (dll_name[i] && dll_init[i])
	(*dll_init[i])();
  }
}

static unload_dlls()
{
  if (dll_name) {
    int i;
    for (i = 0; i < modtbsz; i++)
      if (dll_name[i] && dll_fini[i])
	(*dll_fini[i])();
    for (i = 0; i < modtbsz; i++)
      if (dll_name[i] && dll_handle[i])
	lt_dlclose(dll_handle[i]);
    for (i = 0; i < modtbsz; i++)
      if (dll_name[i])
	free(dll_name[i]);
    free(dll_name); dll_name = NULL;
    free(dll_handle); dll_handle = NULL;
    free(dll_init); dll_init = NULL;
    free(dll_fini); dll_fini = NULL;
    free(dll_atfork); dll_atfork = NULL;
  }
}

/* interface to gnuserv */

static gnuclient(char *sexp)
{
  static char *gnuclient_prog = NULL;
  static char *argv[] = { NULL, "-q", "-eval", NULL, NULL };
  if (!gnuclient_prog && !(gnuclient_prog = getenv("GNUCLIENT_PROGRAM")))
    argv[0] = gnuclient_prog = "gnuclient";
  argv[3] = sexp;
  spawn(gnuclient_prog, argv);
}

void gcmd(char *name)
{
  char *sexp = malloc(strlen(name)+3);
  if (sexp) {
    sprintf(sexp, "(%s)", name);
    gnuclient(sexp);
    free(sexp);
  }
}

void gcmd_i(char *name, int i)
{
  char *sexp = malloc(strlen(name)+100);
  if (sexp) {
    sprintf(sexp, "(%s %d)", name, i);
    gnuclient(sexp);
    free(sexp);
  }
}

void gcmd_b(char *name, int b)
{
  char *sexp = malloc(strlen(name)+100);
  if (sexp) {
    sprintf(sexp, "(%s %s)", name, b?"t":"nil");
    gnuclient(sexp);
    free(sexp);
  }
}

void gcmd_s(char *name, char *s)
{
  char *sexp = malloc(strlen(name)+4*strlen(s)+100);
  if (sexp) {
    char *buf = malloc(4*strlen(s)+1);
    if (buf) {
      sprintf(sexp, "(%s \"%s\")", name, pstr(buf, s));
      gnuclient(sexp);
      free(buf);
    }
    free(sexp);
  }
}

static void evaldefs(void)
{
  int i = 0, done = 1, nerrs = 0;
  int modno = NONE, lineno = NONE;
  start_init();
  while (done && i < inittbsz) {
    if (!evaldef(inittb[i], &modno, &lineno))
      /* recover on all errors except quit */
      done = !(quitflag || thr0->qmstat == QUIT);
#if 0
      /* only recover after break and halt */
      done = done && (checkbrk || thr0->qmstat == HALT);
#endif
    if (thr0->qmstat != OK && !(quitflag || thr0->qmstat == QUIT)) {
      if (thr0->qmstat != HALT) {
	if (modno == NONE || lineno == NONE)
	  error(qmmsg[thr0->qmstat]);
	else {
	  char fname[MAXSTRLEN], msg[MAXSTRLEN];
	  if (debug_long)
	    strcpy(fname, strsp+fnametb[modno]);
	  else
	    basename(fname, strsp+fnametb[modno], 0);
	  sprintf(msg, "File %s, line %d: %s", fname, lineno,
		  qmmsg[thr0->qmstat]);
	  error(msg);
	}
	if (thr0->qmstat == XCEPT && thr0->xsp > thr0->xst) {
	  printx(thr0->xsp[-1]); flush_shift(); printf("\n");
	}
	nerrs++;
      }
      clear(0);
    }
    i++;
  }
  end_init();
  if (batch && nerrs > 0)
    exit(2);
  if (quitflag || thr0->qmstat == QUIT)
    exit(0);
}

static int parsexs(char *s)
{
  int ret;
  s = sys_to_utf8_dup(s);
  if (s)
    ret = parsex(s);
  else
    ret = 0;
  free(s);
  return ret;
}

#ifndef QINT_LIBRARY

/* main program */

main(int argc, char **argv)
{
  int c, bargc = argc, longind;
  char **bargv = argv, *s;
  char fname[MAXSTRLEN], fname2[MAXSTRLEN], msg[MAXSTRLEN];
  char prefix[MAXSTRLEN];
  FILE *fp;

#if defined(HAVE_UNICODE) && defined(HAVE_LOCALE_H)
  setlocale(LC_ALL, "");
  /* Force C locale for numbers on Windows. This works around some bugs with
     third-party software (in particular: Tcl <= 8.4) related to the MSVC
     implementation of strtod. Note that this may still break if the Q script
     later calls setlocale itself, but at least the patch should make
     locale-agnostic applications work as expected. */
#ifdef _WIN32
  setlocale(LC_NUMERIC, "C");
#endif
#endif

#ifdef _WIN32
  InstallSignalHandler();
#endif

  atexit(exitproc);

  sprintf(fformat, STDFORMAT, fprec);

  LTDL_SET_PRELOADED_SYMBOLS();

  /* get program name: */
  self = argv[0];

  /* check to see whether we were invoked with a pathname; in this case we
     also invoke the compiler from there if possible */
  if (*(dirname(prefix, self))) {
    FILE *fp;
    sprintf(qcprog, "%s%s", prefix, "qc");
    if ((fp = fopen(qcprog, "rb"))) {
      fclose(fp);
    } else
      strcpy(qcprog, "qc");
  }

  /* get environment strings: */
  if ((s = getenv("QPATH")) != NULL)
    init_qpath(s);
  else
    init_qpath(QPATH);
  if (!qpath) fatal("memory overflow");
  
  /* set code file id: */
  sprintf(outid, OUTID, version, sysinfo);
  
  /* initialize the qc command line */
  qcarg(qcprog);

  /* scan the command line to obtain the source/code file name: */
  opterr = 0;
  while ((c = getopt_long(argc, argv, Q_OPTS1,
			  longopts, &longind)) != EOF)
    if (c == 'p' && optarg) {
      /* set the path so that we can find the source file even before the
	 remaining command line options have been parsed */
      change_qpath(optarg);
      if (!qpath) fatal("memory overflow");
    }
  opterr = 1;

  if (argc-optind >= 1) {
    source = strdup(argv[optind]);
    if (!source) fatal("memory overflow");
    if (!*source || strcmp(source, "-") ==  0) goto opts;
  } else
    goto opts;

  /* check to see whether the source actually is a precompiled bytecode file,
     then we can skip the compilation step */
    
  if (chkfile(searchlib(fname, source)) &&
      (fp = fopen(fname, "rb")) != NULL ||
      chkfile(searchlib(fname, strcat(strcpy(fname2, source), ".q"))) &&
      (fp = fopen(fname, "rb")) != NULL) {
    if (iscode(fp)) {
      fclose(fp);
      code = source;
      docompile = 0;
    } else {
      /* get options from the source file */
      if (!(fp = freopen(fname, "r", fp)))
	/* this shouldn't happen, but Murphy knows ... */
	goto err;
      get_source_opts(fp);
      fclose(fp);
      parse_opts(sargc, sargv, 0);
    }
  } else {
  err:
    sprintf(msg, qmmsg[FILE_NOT_FOUND], source);
    fatal(msg);
  }

 opts:

  /* process command line options: */
  parse_opts(argc, argv, 1);
  argc -= optind, argv += optind;

  if (Vflag) {
    printf(signon, version, sysinfo, year);
    printf(copying);
    printf(helpmsg, self);
    exit(0);
  }
  if (hflag) {
    char p[100];
    printf(usage, self);
    sprintf(opts, Q_OPTMSG, CSTACKMAX/1024, "dec", EXITRC, "std",
	    HISTFILE, HISTMAX, INITRC, MEMMAX, QPATH, pstr(p, PROMPT),
	    STACKMAX, HASHTBSZ);
    fputs(opts, stdout);
    exit(0);
  }

  /* if no code file name is set, make a temporary file name */
  if (!oset) {
    char *_code = tmpnam(NULL);
    qcarg("-o");
    qcarg(_code);
    if (docompile) code = _code;
  }

  /* make sure that -p is the last option on the qc command line before
     the source file, rerun() depends on it */
  qcarg("-p");
  qcarg(qpath);
  qcarg(source);

  if (!docompile) goto run;

  /* if we come here, we have a source file which we compile first */

  remove(code);
  if (spawn(qcprog, qcargv))
    exit(1);
  else
    donecompile = 1;

 run:
  /* now we should have a bytecode file which we can finally run */

  /* make dynamic copies of the setup variables */
  prompt = strdup(prompt);
  histfile = strdup(histfile);
  if (!(prompt && histfile))
    fatal("memory overflow");

  /* initialize: */
  if (lt_dlinit() || lt_dlsetsearchpath(qpath)) 
    fatal("error initializing module interface");
  readtables();
  resolve();
  init();
  /* install gmp memory handlers */
  mp_set_memory_functions(gmp_allocate, gmp_reallocate, gmp_free);

  /* if all went well, we can now safely remove the code file if we created
     it */
  if (donecompile) remove(code);
  donecompile = 0;

  /* get the file name of the main module */
  if (mainno == -1)
    which = "";
  else
    which = strsp+fnametb[mainno];
  which = strdup(which);
  if (!which) fatal("memory overflow");

  /* set up standard devices: */
  symtb[INPUTOP].x = (void*) fileexpr(thr0, stdin);
  ((EXPR*)symtb[INPUTOP].x)->refc = 1;
  symtb[OUTPUTOP].x = (void*) fileexpr(thr0, stdout);
  ((EXPR*)symtb[OUTPUTOP].x)->refc = 1;
  symtb[ERROROP].x = (void*) fileexpr(thr0, stderr);
  ((EXPR*)symtb[ERROROP].x)->refc = 1;

  /* process remaining command line arguments: */
  {	int count = 0;
  while (argc-- > 0) {
    char *s;
    if (*argv)
      s = sys_to_utf8_dup(*argv++);
    else
      s = strdup("");
    if (!s)
      fatal("memory overflow");
    else if (!pushstr(thr0, s))
      fatal("stack overflow");
    else
      count++;
  }
  if (!pushfun(thr0, NILOP))
    fatal("stack overflow");
  while (count-- > 0)
    if (!pushfun(thr0, CONSOP))
      fatal("memory overflow");
  symtb[ARGSOP].x = (void*) thr0->xsp[-1];
  thr0->xsp = thr0->xst;
  }

  /* suppress batch options if in forced interactive mode */
  if (iflag) batch = 0;
  /* run interactively when connected to a terminal */
  iflag = iflag || (isatty(fileno(stdin)) && isatty(fileno(stdout)));
#ifdef USE_READLINE
  use_readline = (iflag && !norl && isatty(fileno(stdin)));
  init_readline();
#endif
  if (!batch && !qflag && iflag) {
    /* sign-on: */
    printf(signon, version, sysinfo, year);
    printf("\n");
    printf(terms);
  }

  /* install signal handlers: */
  sigint(break_handler); sigterm(term_handler); sighup(term_handler);
#ifdef SIGTTIN
  syssignal(SIGTTIN, tty_handler);
#endif
#ifdef SIGTTOU
  syssignal(SIGTTOU, tty_handler);
#endif

  /* handle fatal program errors */
#ifdef SIGFPE
  syssignal(SIGFPE, segv_handler);
#endif
#ifdef SIGILL
  syssignal(SIGILL, segv_handler);
#endif
#ifdef SIGSEGV
  syssignal(SIGSEGV, segv_handler);
#endif
#ifdef SIGBUS
  syssignal(SIGBUS, segv_handler);
#endif
#ifdef SIGTRAP
  syssignal(SIGTRAP, segv_handler);
#endif
    
  /* initialize random seed: */
  seedMT(((unsigned long)time(NULL)) << 1 | 1);

  /* initialize external modules: */
  init_dlls();

  /* execute the script's initialization code (def and undef): */
  errno = 0;
  if (inittbsz) {
    evaldefs();
    clear(0);
    clearerr(stdin);
  }

  /* execute -c and -s options, if any */
  if (batch) {
    if (sargc && sargv) {
      optind = 1;
      while ((c = getopt_long(sargc, sargv, Q_OPTS1, longopts,
			      &longind)) != EOF) {
	switch (c) {
	case 'c':
	  if (eflag) echo(optarg);
	  parsexs(optarg);
	  break;
	case 's':
	  parsesrc(optarg, 1);
	  break;
	default:
	  break;
	}
	if (quitflag || thr0->qmstat == QUIT) exit(0);
      }
    }
    optind = 1;
    while ((c = getopt_long(bargc, bargv, Q_OPTS1, longopts,
			    &longind)) != EOF) {
      switch (c) {
      case 'c':
	if (eflag) echo(optarg);
	parsexs(optarg);
	break;
      case 's':
	parsesrc(optarg, 1);
	break;
      default:
	break;
      }
      if (quitflag || thr0->qmstat == QUIT) exit(0);
    }
    exit(0);
  }

  /* if we come here, we are running interactively; source the initrc file
     and enter the evaluation loop */

  quitflag = 0;
  thr0->qmstat = OK;
  doexitrc = 1;
  if (initrc && chkfile(expand(fname, initrc)) && (fp = fopen(fname, "r"))) {
    fclose(fp);
    parsesrc(fname, 1);
    if (quitflag || thr0->qmstat == QUIT) exit(0);
  }

  quitflag = 0;
  thr0->qmstat = OK;
#ifdef USE_READLINE
  init_history();
#endif

  /* the read/eval/print loop */
  errno = 0;
  while (1) {
    char *buf;
    brkflag = 0;
    /* release the lock on the interpreter while we're waiting for input */
    release_lock();
    /* get hold of the input line */
    acquire_input();
    clearerr(stdin);
    flush_shift();
    if (feof(stdin) || !(buf = mygetline(stdin, prompt, 1))) {
      if (iflag) putchar('\n');
      release_input();
      acquire_lock();
      break;
    }
    release_input();
    acquire_lock();
    if (!iflag && eflag) echo(utf8_to_sys(buf));
    parsex(buf);
    if (thr0->sentinels) process_sentinels(thr0);
    if (quitflag || thr0->qmstat == QUIT) break;
    free(buf);
    fflush(stderr);
    fflush(stdout);
  }

  fflush(stderr);
  fflush(stdout);

  thr0->qmstat = OK;
  exit(0);
}

#else

/* libqint interface */

static libmain(const char *path, int argc, char **argv)
{
  int c, bargc = argc, longind, ret;
  char **bargv = argv, *s;
  char fname[MAXSTRLEN], fname2[MAXSTRLEN], msg[MAXSTRLEN];
  char prefix[MAXSTRLEN];
  FILE *fp;

#ifdef _WIN32
  InstallSignalHandler();
#endif

  atexit(exitproc);

  sprintf(fformat, STDFORMAT, fprec);

#if 0
  /* FIXME: This is needed to make the static build work, but causes severe
     breakage if client programs are not built with libtool. Disabled for
     now. */
  LTDL_SET_PRELOADED_SYMBOLS();
#endif

  /* get program name: */
  self = argv[0];

  /* check to see whether we were invoked with a pathname; in this case we
     also invoke the compiler from there if possible */
  if (*(dirname(prefix, self))) {
    FILE *fp;
    sprintf(qcprog, "%s%s", prefix, "qc");
    if ((fp = fopen(qcprog, "rb"))) {
      fclose(fp);
    } else
      strcpy(qcprog, "qc");
  }

  /* get environment strings: */
  if ((s = getenv("QPATH")) != NULL)
    init_qpath(s);
  else
    init_qpath(QPATH);
  if (!qpath) fatal("memory overflow");
  
  /* set code file id: */
  sprintf(outid, OUTID, version, sysinfo);
  
  /* initialize the qc command line */
  qcarg(qcprog);

  /* scan the command line to obtain the source/code file name: */
  opterr = 0;
  optind = 0;
  while ((c = getopt_long(argc, argv, Q_OPTS1,
			  longopts, &longind)) != EOF)
    if (c == 'p' && optarg) {
      /* set the path so that we can find the source file even before the
	 remaining command line options have been parsed */
      change_qpath(optarg);
      if (!qpath) fatal("memory overflow");
    }
  opterr = 1;

  source = path?strdup(path):strdup("");
  if (!source) fatal("memory overflow");
  if (!*source) goto opts;

  /* check to see whether the source actually is a precompiled bytecode file,
     then we can skip the compilation step */
    
  if (chkfile(searchlib(fname, source)) &&
      (fp = fopen(fname, "rb")) != NULL ||
      chkfile(searchlib(fname, strcat(strcpy(fname2, source), ".q"))) &&
      (fp = fopen(fname, "rb")) != NULL) {
    if (iscode(fp)) {
      fclose(fp);
      code = source;
      docompile = 0;
    } else
      fclose(fp);
  } else {
  err:
    ret = FILE_NOT_FOUND;
    goto errexit;
  }

 opts:

  /* process command line options: */
  parse_opts(argc, argv, 1);
  argc -= optind, argv += optind;

  /* if no code file name is set, make a temporary file name */
  if (!oset) {
    char *_code = tmpnam(NULL);
    qcarg("-o");
    qcarg(_code);
    if (docompile) code = _code;
  }

  /* make sure that -p is the last option on the qc command line before
     the source file, rerun() depends on it */
  qcarg("-p");
  qcarg(qpath);
  qcarg(source);

  if (!docompile) goto run;

  /* if we come here, we have a source file which we compile first */

  remove(code);
  if (spawn(qcprog, qcargv)) {
    ret = COMPILE_ERR;
    goto errexit;
  } else
    donecompile = 1;

 run:
  /* now we should have a bytecode file which we can finally run */

  /* make dynamic copies of the setup variables */
  prompt = strdup(prompt);
  histfile = strdup(histfile);
  if (!(prompt && histfile))
    fatal("memory overflow");

  /* initialize: */
  if (lt_dlinit() || lt_dlsetsearchpath(qpath)) 
    fatal("error initializing module interface");
  readtables();
  resolve();
  init();
  /* install gmp memory handlers */
  mp_set_memory_functions(gmp_allocate, gmp_reallocate, gmp_free);

  /* if all went well, we can now safely remove the code file if we created
     it */
  if (donecompile) remove(code);
  donecompile = 0;

  /* get the file name of the main module */
  if (mainno == -1)
    which = "";
  else
    which = strsp+fnametb[mainno];
  which = strdup(which);
  if (!which) fatal("memory overflow");

  /* set up standard devices: */
  symtb[INPUTOP].x = (void*) fileexpr(thr0, stdin);
  ((EXPR*)symtb[INPUTOP].x)->refc = 1;
  symtb[OUTPUTOP].x = (void*) fileexpr(thr0, stdout);
  ((EXPR*)symtb[OUTPUTOP].x)->refc = 1;
  symtb[ERROROP].x = (void*) fileexpr(thr0, stderr);
  ((EXPR*)symtb[ERROROP].x)->refc = 1;

  /* process remaining command line arguments: */
  {	int count = 0;
  while (argc-- > 0) {
    char *s;
    if (*argv)
      s = sys_to_utf8_dup(*argv++);
    else
      s = strdup("");
    if (!s)
      fatal("memory overflow");
    else if (!pushstr(thr0, s))
      fatal("stack overflow");
    else
      count++;
  }
  if (!pushfun(thr0, NILOP))
    fatal("stack overflow");
  while (count-- > 0)
    if (!pushfun(thr0, CONSOP))
      fatal("memory overflow");
  symtb[ARGSOP].x = (void*) thr0->xsp[-1];
  thr0->xsp = thr0->xst;
  }

  /* force batch processing */
  batch = 1; iflag = 0; qflag = 1;

  /* install signal handlers: */
  sigint(break_handler); sigterm(term_handler); sighup(term_handler);
#ifdef SIGTTIN
  syssignal(SIGTTIN, tty_handler);
#endif
#ifdef SIGTTOU
  syssignal(SIGTTOU, tty_handler);
#endif

#if 0 /* the hosting application is supposed to do this */
  /* handle fatal program errors */
#ifdef SIGFPE
  syssignal(SIGFPE, segv_handler);
#endif
#ifdef SIGILL
  syssignal(SIGILL, segv_handler);
#endif
#ifdef SIGSEGV
  syssignal(SIGSEGV, segv_handler);
#endif
#ifdef SIGBUS
  syssignal(SIGBUS, segv_handler);
#endif
#ifdef SIGTRAP
  syssignal(SIGTRAP, segv_handler);
#endif
#endif
    
  /* initialize random seed: */
  seedMT(((unsigned long)time(NULL)) << 1 | 1);

  /* initialize external modules: */
  init_dlls();

  /* execute the script's initialization code (def and undef): */
  errno = 0;
  if (inittbsz) {
    evaldefs();
    clear(0);
    clearerr(stdin);
  }

  /* execute -c and -s options, if any */
  if (sargc && sargv) {
    optind = 1;
    while ((c = getopt_long(sargc, sargv, Q_OPTS1, longopts,
			    &longind)) != EOF) {
      switch (c) {
      case 'c':
	if (eflag) echo(optarg);
	parsexs(optarg);
	break;
      case 's':
	parsesrc(optarg, 1);
	break;
      default:
	break;
      }
      if (quitflag || thr0->qmstat == QUIT) exit(0);
    }
  }
  optind = 1;
  while ((c = getopt_long(bargc, bargv, Q_OPTS1, longopts,
			  &longind)) != EOF) {
    switch (c) {
    case 'c':
      if (eflag) echo(optarg);
      parsexs(optarg);
      break;
    case 's':
      parsesrc(optarg, 1);
      break;
    default:
      break;
    }
    if (quitflag || thr0->qmstat == QUIT) exit(0);
  }
  errno = 0;
  quitflag = 0;
  thr0->qmstat = OK;
  return OK;

 errexit:
  while (qcargc > 0)
    if (qcargv[--qcargc]) {
      free(qcargv[qcargc]);
      qcargv[qcargc] = NULL;
    }
  self = "q";
  strcpy(qcprog, "qc");
  batch = 0; docompile = 1; donecompile = 0;
  doexitrc = 0; oset = 0;
  eflag = 0; qflag = 0; iflag = 0; hflag = 0; Vflag = 0;
  gflag = 0; norl = 0;
  if (source) free(source);
  if (which) free(which);
  source = which = NULL;
  code = "q.out";
  return ret;
}

#define __DLL_BUILD 1
#if defined _WIN32
#define __DLLIMPORT __declspec(dllexport)
#define __DLLEXPORT __declspec(dllimport)
#else
#define __DLLIMPORT
#define __DLLEXPORT extern
#endif

#include <stdarg.h>
#include "qint.h"

__DLLIMPORT char *qstrerror(int status)
{
  if (status >= 0 && status <= ARGS_ERR)
    return qmmsg[status];
  else
    return "Unknown error";
}

static void free_argv(int argc, char **const argv)
{
  if (argv) {
    int i;
    for (i = 0; i < argc; i++)
      free(argv[i]);
    free(argv);
  }
}

static char **copy_argv(int argc, char *const argv[])
{
  char **_argv = malloc((argc+1)*sizeof(char*));
  int i;
  if (!_argv) return NULL;
  for (i = 0; i < argc; i++) {
    _argv[i] = argv[i]?strdup(argv[i]):NULL;
    if (!_argv[i]) {
      free_argv(i, _argv);
      return NULL;
    }
  }
  _argv[argc] = NULL;
  return _argv;
}

__DLLIMPORT int qexecv(const char *_path, int _argc, char *const _argv[])
{
  static int init = 0, ret, argc = 0;
  static char **argv = NULL, *path = NULL;
  /* get rid of previous arg vector, if any */
  if (path) {
    free(path);
    path = NULL;
  }
  if (argv) {
    free_argv(argc, argv);
    argv = NULL; argc = 0;
  }
  /* make a copy of the new arg vector */
  if (_path) {
    path = strdup(_path);
    if (!path) return MEM_OVF;
  }
  if (_argv && _argc > 0) {
    argv = copy_argv(_argc, _argv);
    if (!argv) return MEM_OVF;
    argc = _argc;
  }
  if (setjmp(fatal_target) == 0) {
    fatal_set = 1;
    if (!init) {
      static char **myargv;
      int i, myargc = (argc?argc:path?1:0)+2;
      /* rearrange arguments */
      if (myargv) free(myargv);
      myargv = malloc((myargc+1)*sizeof(char*));
      if (!myargv) return MEM_OVF;
      myargv[0] = "q";
      myargv[1] = "--";
      myargv[2] = path;
      for (i = 0; i < argc; i++)
	myargv[i+2] = argv[i];
      myargv[myargc] = NULL;
      ret = libmain(path, myargc, myargv);
      if (ret) {
	path = NULL;
	argv = NULL; argc = 0;
      } else
	init = 1;
    } else
      ret = dorun(path, argc, argv);
    fatal_set = 0;
    fatal_status = OK;
  } else {
    fatal_set = 0;
    ret = fatal_status;
    fatal_status = OK;
  }
  return ret;
}

__DLLIMPORT int qexecl(const char *path, int argc, ...)
{
  if (argc == 0)
    return qexecv(path, 0, NULL);
  else {
    char **argv = malloc(argc*sizeof(char*));
    if (argv) {
      int i, ret;
      va_list ap;
      va_start(ap, argc);
      for (i = 0; i < argc; i++)
	argv[i] = va_arg(ap, char*);
      va_end(ap);
      ret = qexecv(path, argc, argv);
      free(argv);
      return ret;
    } else
      return MEM_OVF;
  }
}

__DLLIMPORT int qexecvx(const void *buf, size_t size, int argc,
			char *const argv[])
{
  int ret = OK;
  char *path = tmpnam(NULL);
  FILE *fp = fopen(path, "wb");
  if (fp) {
    if (write(fileno(fp), buf, size) != size) ret = FILE_ERR;
    fclose(fp);
  } else
    ret = FILE_ERR;
  if (!ret)
    ret = qexecv(path, argc, argv);
  if (fp)
    remove(path);
  return ret;
}

__DLLIMPORT int qexeclx(const void *buf, size_t size, int argc, ...)
{
  if (argc == 0)
    return qexecvx(buf, size, 0, NULL);
  else {
    char **argv = malloc(argc*sizeof(char*));
    if (argv) {
      int i, ret;
      va_list ap;
      va_start(ap, argc);
      for (i = 0; i < argc; i++)
	argv[i] = va_arg(ap, char*);
      va_end(ap);
      ret = qexecvx(buf, size, argc, argv);
      free(argv);
      return ret;
    } else
      return MEM_OVF;
  }
}

__DLLIMPORT char *qeval(const char *s, int *status)
{
  qexpr x;
  if (!(x = qparse(s, status)))
    return NULL;
  else if (!(x = qevalx(x, status)))
    return NULL;
  else {
    int status2;
    char *t = qprint(x, &status2);
    if (status2) *status = status2;
    return t;
  }
}

static inline void collect(THREAD *thr, qexpr x)
{
  if (x && ((EXPR*)x)->refc == 0) {
    ((EXPR*)x)->refc = 1;
    qmfree(thr, (EXPR*)x);
  }
}

static void clear_stack(THREAD *thr)
{
  /* this stuff adopted from qm.c::clear() */
  int defxstsz = XSTSZ, defastsz = ASTSZ; 

  while (thr->xsp > thr->xst)
    qmfree(thr, *--thr->xsp);

  while (thr->asp > thr->ast)
    free(*--thr->asp);

  thr->maxxstsz = thr->maxastsz = stackmax;

  if (thr->maxxstsz > 0 && thr->maxxstsz < defxstsz)
    defxstsz = thr->maxxstsz;
  if (thr->maxastsz > 0 && thr->maxastsz < defastsz)
    defastsz = thr->maxastsz;

  if (thr->xstsz > defxstsz) {
    thr->xst = (EXPR**)realloc(thr->xst, defxstsz*sizeof(EXPR*));
    thr->xstsz = defxstsz;
  }
  if (thr->astsz > defastsz) {
    thr->ast = (AREC**)realloc(thr->ast, defastsz*sizeof(AREC*));
    thr->astsz = defastsz;
  }
  if (!thr->xst || !thr->ast)
    fatal(qmmsg[THIS_CANT_HAPPEN]);

  if (thr->mark) free(thr->mark);
  thr->mark = thr->markp = NULL;
  thr->marksz = 0;
  
  thr->xsp = thr->xst;
  thr->asp = thr->ast;
  thr->qmstat = thr->qmstat_save = OK;
  thr->mode = 0;
  thr->debug = debug;
  thr->brkdbg = brkdbg;
  thr->brkflag = 0;
  thr->nsig = 0;
  thr->sigpend = thr->sigblk = 0;

  if (thr->sentinels) process_sentinels(thr);
}

__DLLIMPORT qexpr qevalx(qexpr x, int *status)
{
  THREAD *thr = get_thr();
  EXPR *_x = (EXPR*)x;
  thr->qmstat = OK;
  if (setjmp(fatal_target) == 0) {
    int ret;
    fatal_set = 1;
    _x->refc++;
    ret = eval(thr, _x);
    if (quitflag || thr->qmstat == QUIT) exit(0);
    if (ret || thr->qmstat == XCEPT && thr->xsp > thr->xst) {
      EXPR *_y = *--thr->xsp;
      *status = thr->qmstat;
      clear_stack(thr);
      _x->refc--;
      _y->refc--;
      if (_x != _y) collect(thr, _x);
      fatal_set = 0;
      return _y;
    } else {
      *status = thr->qmstat;
      clear_stack(thr);
      qmfree(thr, _x);
      fatal_set = 0;
      return NULL;
    }
  } else {
    fatal_set = 0;
    *status = fatal_status;
    fatal_status = OK;
    clear_stack(thr);
    collect(thr, _x);
    return NULL;
  }
}

__DLLIMPORT qexpr qparse(const char *s, int *status)
{
  THREAD *thr = get_thr();
  char *t = qto_utf8(s, NULL);
  if (!t) {
    *status = MEM_OVF;
    return NULL;
  }
  if (sparsex(t)) {
    EXPR *x = *--thr->xsp;
    x->refc--;
    free(t);
    return x;
  } else {
    *status = thr->qmstat;
    free(t);
    return NULL;
  }
}

__DLLIMPORT char *qprint(qexpr x, int *status)
{
  char *s = sprintx(qnewref(x));
  qfreeref(x);
  if (s) {
    char *t = qfrom_utf8(s, NULL);
    free(s);
    if (t)
      *status = OK;
    else
      *status = MEM_OVF;
    return t;
  } else {
    *status = MEM_OVF;
    return NULL;
  }
}

__DLLIMPORT int qdef(const char *name, qexpr x)
{
  THREAD *thr = get_thr();
  char *_name = strdup(name);
  EXPR *_x = (EXPR*)x;
  int vno;
  if (!_name) return MEM_OVF;
  thr->qmstat = OK;
  if ((vno = getsym(_name, mainno)) != NONE) {
    free(_name);
    setvar(vno, _x);
    return thr->qmstat;
  } else if (x) {
    strcpy(_name, name);
    if ((vno = mksym(_name)) != NONE) {
      free(_name);
      setvar(vno, _x);
      return thr->qmstat;
    } else {
      free(_name);
      return thr->qmstat;
    }
  } else {
    free(_name);
    return OK;
  }
}

__DLLIMPORT void qdispose(qexpr x)
{
  collect(get_thr(), x);
}

__DLLIMPORT qexpr qnewref(qexpr x)
{
  EXPR *_x = (EXPR*)x;
  if (_x)
    _x->refc++;
  return x;
}

__DLLIMPORT void qfreeref(qexpr x)
{
  EXPR *_x = (EXPR*)x;
  if (_x) {
    if (_x->refc == 0) _x->refc = 1;
    qmfree(get_thr(), _x);
  }
}

/* Functions to access the symbol table. */

__DLLIMPORT int qgetsym(const char *name)
{
  char *s = strdup(name);
  int ret;
  if (s) {
    ret = getsym(s, mainno);
    free(s);
  } else
    ret = -1;
  return ret;
}

__DLLIMPORT int qgettype(const char *name)
{
  char *s = strdup(name);
  int ret;
  if (s) {
    ret = gettype(s, mainno);
    free(s);
  } else
    ret = -1;
  return ret;
}

__DLLIMPORT int qsymprec(int sym)
{
  switch (sym) {
  case SEQOP:
    return 0;	/* || */
  case RAPPOP:
    return 1;	/* $ */
  case LEOP:
  case GROP:
  case EQOP:
  case LEQOP:
  case GEQOP:
  case NEQOP:
  case IDOP:
    return 2;	/* relational operators */
  case CATOP:
  case ADDOP:
  case MINOP:
  case OROP:
  case ORELSEOP:
    return 3;	/* addition operators */
  case MULOP:
  case FDIVOP:
  case DIVOP:
  case MODOP:
  case ANDOP:
  case ANDTHENOP:
    return 4;	/* multiplication operators */
  case UMINOP:
  case NOTOP:
  case HASHOP:
    return 5;	/* prefix operators */
  case POWOP:
  case IDXOP:
    return 6;	/* super/subscript */
  case COMPOP:
    return 7;	/* composition */
  case APPOP:
    return 8;	/* @ */
  case QUOTEOP:
  case UNQUOTEOP:
  case FORCEOP:
  case MEMOP:
    return 9;	/* quotation */
  default:
    /* ordinary function symbol or user-defined operator */
    if (symtb[sym].prec < 0)
      return -1;
    else
      return symtb[sym].prec;
  }
}

/* Predefined function and type symbols. */

__DLLIMPORT const int qtruesym = TRUEOP, qfalsesym = FALSEOP, qnilsym = NILOP,
  qvoidsym = VOIDOP;
__DLLIMPORT const int qinttype = INTTYPE, qfloattype = FLOATTYPE,
  qbooltype =BOOLTYPE, qstrtype = STRTYPE, qfiletype = FILETYPE,
  qlisttype = LISTTYPE, qtupletype = TUPLETYPE;

/* Expression construction. */

__DLLIMPORT qexpr qmkint(long i)
{
  return _qinter_intexpr(i);
}

__DLLIMPORT qexpr qmkuint(unsigned long i)
{
  return _qinter_uintexpr(i);
}

__DLLIMPORT qexpr qmkmpz(void *z)
{
  return _qinter_mpzexpr(z);
}

__DLLIMPORT qexpr qmkmpz_float(double f)
{
  return _qinter_mpz_floatexpr(f);
}

__DLLIMPORT qexpr qmkfloat(double f)
{
  return _qinter_floatexpr(f);
}

__DLLIMPORT qexpr qmkstr(char *s)
{
  return s?_qinter_strexpr(s):NULL;
}

__DLLIMPORT qexpr qmkfile(FILE *fp)
{
  return fp?_qinter_fileexpr(fp):NULL;
}

__DLLIMPORT qexpr qmkpipe(FILE *fp)
{
  return fp?_qinter_pipeexpr(fp):NULL;
}


__DLLIMPORT qexpr qmksym(int sym)
{
  if (issym(sym))
    return _qinter_funexpr(sym);
  else
    return NULL;
}

__DLLIMPORT qexpr qmkbool(int flag)
{
  if (flag)
    return qmktrue;
  else
    return qmkfalse;
}

__DLLIMPORT qexpr qmkobj(int type, void *ptr)
{
  if (isusrtype(type))
    return _qinter_usrexpr(type, ptr);
  else
    return NULL;
}


__DLLIMPORT qexpr qmkapp(qexpr fun, qexpr arg)
{
  qexpr x = (fun&&arg)?_qinter_consexpr(APPOP, fun, arg):NULL;
  if (!x) {
    THREAD *thr = get_thr();
    collect(thr, fun); collect(thr, arg);
  }
  return x;
}

__DLLIMPORT qexpr qmkcons(qexpr hd, qexpr tl)
{
  qexpr x = (hd&&tl)?_qinter_consexpr(CONSOP, hd, tl):NULL;
  if (!x) {
    THREAD *thr = get_thr();
    collect(thr, hd); collect(thr, tl);
  }
  return x;
}

__DLLIMPORT qexpr qmkcont(qexpr hd, qexpr tl)
{
  qexpr x = (hd&&tl)?_qinter_consexpr(PAIROP, hd, tl):NULL;
  if (!x) {
    THREAD *thr = get_thr();
    collect(thr, hd); collect(thr, tl);
  }
  return x;
}


__DLLIMPORT qexpr qmklistl(int nelems, ...)
{
  if (nelems <= 0)
    return qmknil;
  else {
    qexpr *elems = malloc(nelems*sizeof(qexpr));
    if (elems) {
      int i;
      va_list ap;
      va_start(ap, nelems);
      for (i = 0; i < nelems; i++)
	elems[i] = va_arg(ap, qexpr);
      va_end(ap);
      return qmklistv(nelems, elems);
    } else
      return NULL;
  }
}

__DLLIMPORT qexpr qmklistv(int nelems, qexpr *elems)
{
  if (nelems < 0) nelems = 0;
  if (nelems > 0 && !elems)
    return NULL;
  else {
    int i;
    qexpr x = qmknil;
    for (i = nelems-1; x && i >= 0; i--) {
      qexpr y = qmkcons(elems[i], x);
      x = y;
      if (!x) {
	THREAD *thr = get_thr();
	int j;
	for (j = 0; j < i; j++) collect(thr, elems[j]);
	break;
      }
    }
    if (elems) free(elems);
    return x;
  }
}


__DLLIMPORT qexpr qmktuplel(int nelems, ...)
{
  if (nelems <= 0)
    return qmkvoid;
  else {
    qexpr *elems = malloc(nelems*sizeof(qexpr));
    if (elems) {
      int i;
      va_list ap;
      va_start(ap, nelems);
      for (i = 0; i < nelems; i++)
	elems[i] = va_arg(ap, qexpr);
      va_end(ap);
      return qmktuplev(nelems, elems);
    } else
      return NULL;
  }
}

__DLLIMPORT qexpr qmktuplev(int nelems, qexpr *elems)
{
  if (nelems < 0) nelems = 0;
  if (nelems > 0 && !elems)
    return NULL;
  else if (nelems > 0) {
    int i, chk = 1;
    for (i = 0; i < nelems; i++)
      if (!elems[i]) {
	chk = 0;
	break;
      }
    if (!chk) {
      THREAD *thr = get_thr();
      for (i = 0; i < nelems; i++)
	collect(thr, elems[i]);
      free(elems);
      return NULL;
    } else {
      for (i = 0; i < nelems; i++)
	((EXPR*)elems[i])->refc++;
      return _qinter_vectexpr(nelems, (EXPR**)elems);
    }
  } else
    return qmkvoid;
}

/* Type checking and unboxing. */

__DLLIMPORT int qexprsym(const qexpr x)
{
  if (((EXPR*)x)->fno >= BINARY)
    return ((EXPR*)x)->fno;
  else
    return 0;
}

__DLLIMPORT int qexprtype(const qexpr x)
{
  return (((EXPR*)x)->argc||((EXPR*)x)->virt)?0:((EXPR*)x)->type;
}


__DLLIMPORT int qisint(const qexpr x, long *i)
{
  return getintexpr((EXPR*)x, i);
}

__DLLIMPORT int qisuint(const qexpr x, unsigned long *i)
{
  return getuintexpr((EXPR*)x, i);
}

__DLLIMPORT int qismpz(const qexpr x, void *z)
{
  return getmpzexpr((EXPR*)x, z);
}

__DLLIMPORT int qismpz_float(const qexpr x, double *f)
{
  return getmpz_floatexpr((EXPR*)x, f);
}

__DLLIMPORT int qisfloat(const qexpr x, double *f)
{
  if (((EXPR*)x)->fno == FLOATVALOP) {
    *f = ((EXPR*)x)->data.f;
    return 1;
  } else
    return 0;
}

__DLLIMPORT int qisstr(const qexpr x, char **s)
{
  if (((EXPR*)x)->fno == STRVALOP) {
    *s = ((EXPR*)x)->data.s;
    return 1;
  } else
    return 0;
}

__DLLIMPORT int qisfile(const qexpr x, FILE **fp)
{
  if (((EXPR*)x)->fno == FILEVALOP) {
    *fp = ((EXPR*)x)->data.fp;
    return 1;
  } else
    return 0;
}


__DLLIMPORT int qissym(const qexpr x, int sym)
{
   return sym >= BINARY && ((EXPR*)x)->fno == sym;
}

__DLLIMPORT int qisbool(const qexpr x, int *flag)
{
  if (qistrue(x)) {
    *flag = 1;
    return 1;
  } else if (qisfalse(x)) {
    *flag = 0;
    return 1;
  } else
    return 0;
}


__DLLIMPORT int qisobj(const qexpr x, int type, void **ptr)
{
  if (((EXPR*)x)->fno == USRVALOP && ((EXPR*)x)->type == type) {
    *ptr = ((EXPR*)x)->data.vp;
    return 1;
  } else
    return 0;
}

__DLLIMPORT int qisapp(const qexpr x, qexpr *fun, qexpr *arg)
{
  if (((EXPR*)x)->fno == APPOP) {
    *fun = ((EXPR*)x)->data.args.x1;
    *arg = ((EXPR*)x)->data.args.x2;
    return 1;
  } else
    return 0;
}

__DLLIMPORT int qiscons(const qexpr x, qexpr *hd, qexpr *tl)
{
  if (((EXPR*)x)->fno == CONSOP) {
    *hd = ((EXPR*)x)->data.args.x1;
    *tl = ((EXPR*)x)->data.args.x2;
    return 1;
  } else
    return 0;
}

__DLLIMPORT int qiscont(const qexpr x, qexpr *hd, qexpr *tl)
{
  if (((EXPR*)x)->fno == PAIROP) {
    *hd = ((EXPR*)x)->data.args.x1;
    *tl = ((EXPR*)x)->data.args.x2;
    return 1;
  } else
    return 0;
}


__DLLIMPORT int qistuple(const qexpr x, int *nelems, qexpr **elems)
{
  if (((EXPR*)x)->fno == VECTOP) {
    *nelems = ((EXPR*)x)->data.vect.n;
    *elems = (qexpr*)((EXPR*)x)->data.vect.xv;
    return 1;
  } else
    return 0;
}


/* Unicode support. */

#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)

#define CHUNKSZ 128

__DLLIMPORT char *qfrom_utf8(const char *s, const char *encoding)
{
  iconv_t ic;

  if (!s) return NULL;

  if (!encoding || !*encoding)
    encoding = default_encoding();
  if (encoding && strcmp(encoding, "UTF-8"))
    ic = iconv_open(encoding, "UTF-8");
  else
    ic = (iconv_t)-1;

  if (ic == (iconv_t)-1)
    return strdup(s);

  else {
    size_t l = strlen(s);
    char *t = malloc(l+1), *t1;
    char *inbuf = (char*)s, *outbuf = t;
    size_t inbytes = l, outbytes = l;

    while (iconv(ic, &inbuf, &inbytes, &outbuf, &outbytes) ==
	   (size_t)-1)
      if (errno == E2BIG) {
	/* try to enlarge the output buffer */
	size_t k = outbuf-t;
	if ((t1 = realloc(t, l+CHUNKSZ+1))) {
	  t = t1;
	  outbuf = t+k;
	  l += CHUNKSZ;
	  outbytes += CHUNKSZ;
	} else {
	  /* memory overflow */
	  free(t);
	  return NULL;
	}
      } else {
	/* conversion error */
	free(t);
	return strdup(s);
      }

    /* here we might have to deal with a stateful encoding, so make sure that
       we emit the closing shift sequence */

    while (iconv(ic, NULL, NULL, &outbuf, &outbytes) ==
	   (size_t)-1)
      if (errno == E2BIG) {
	/* try to enlarge the output buffer */
	size_t k = outbuf-t;
	if ((t1 = realloc(t, l+CHUNKSZ+1))) {
	  t = t1;
	  outbuf = t+k;
	  l += CHUNKSZ;
	  outbytes += CHUNKSZ;
	} else {
	  /* memory overflow */
	  free(t);
	  return NULL;
	}
      } else {
	/* conversion error */
	free(t);
	return strdup(s);
      }

    /* terminate the output string */
    *outbuf = 0;
    iconv_close(ic);

    if (!(t1 = realloc(t, strlen(t)+1)))
      /* this shouldn't happen */
      return t;
    else
      return t1;
  }
}

__DLLIMPORT char *qto_utf8(const char *s, const char *encoding)
{
  iconv_t ic;

  if (!s) return NULL;

  if (!encoding || !*encoding)
    encoding = default_encoding();
  if (encoding && strcmp(encoding, "UTF-8"))
    ic = iconv_open("UTF-8", encoding);
  else
    ic = (iconv_t)-1;

  if (ic == (iconv_t)-1)
    return strdup(s);

  else {
    size_t l = strlen(s);
    char *t = malloc(l+1), *t1;
    char *inbuf = (char*)s, *outbuf = t;
    size_t inbytes = l, outbytes = l;

    while (iconv(ic, &inbuf, &inbytes, &outbuf, &outbytes) ==
	   (size_t)-1)
      if (errno == E2BIG) {
	/* try to enlarge the output buffer */
	size_t k = outbuf-t;
	if ((t1 = realloc(t, l+CHUNKSZ+1))) {
	  t = t1;
	  outbuf = t+k;
	  l += CHUNKSZ;
	  outbytes += CHUNKSZ;
	} else {
	  /* memory overflow */
	  free(t);
	  return NULL;
	}
      } else {
	/* conversion error */
	free(t);
	return strdup(s);
      }

    /* terminate the output string */
    *outbuf = 0;
    iconv_close(ic);

    if (!(t1 = realloc(t, strlen(t)+1)))
      /* this shouldn't happen */
      return t;
    else
      return t1;
  }
}

__DLLIMPORT char *qfile_from_utf8(const char *s, qexpr file)
{
  iconv_t *ic = ((EXPR*)file)->data.fargs.ic;

  if (((EXPR*)file)->fno != FILEVALOP)
    return NULL;

  if (ic[1] == (iconv_t)-2) {
    char *codeset = default_encoding();
    if (codeset && strcmp(codeset, "UTF-8"))
      ic[1] = iconv_open(codeset, "UTF-8");
    else
      ic[1] = (iconv_t)-1;
  }
  if (ic[1] == (iconv_t)-1)
    return s?strdup(s):NULL;
  else {
    /* Here the input buffer may be NULL, to emit a terminating shift
       sequence. In this case we initialize an output buffer of size
       CHUNKSZ. */
    size_t l = s?strlen(s):0, m = s?l:CHUNKSZ;
    char *t = malloc(m+1), *t1;
    char *inbuf = (char*)s, *outbuf = t;
    size_t inbytes = l, outbytes = m;

    while (iconv(ic[1], &inbuf, &inbytes, &outbuf, &outbytes) ==
	   (size_t)-1)
      if (errno == E2BIG) {
	/* try to enlarge the output buffer */
	size_t k = outbuf-t;
	if ((t1 = realloc(t, m+CHUNKSZ+1))) {
	  t = t1;
	  outbuf = t+k;
	  m += CHUNKSZ;
	  outbytes += CHUNKSZ;
	} else {
	  /* memory overflow */
	  free(t);
	  return NULL;
	}
      } else {
	/* conversion error */
	free(t);
	return s?strdup(s):NULL;
      }
    /* terminate the output string */
    *outbuf = 0;
    if (!(t1 = realloc(t, strlen(t)+1)))
      /* this shouldn't happen */
      return t;
    else
      return t1;
  }
}

__DLLIMPORT char *qfile_to_utf8(const char *s, qexpr file)
{
  iconv_t *ic = ((EXPR*)file)->data.fargs.ic;

  if (!s || ((EXPR*)file)->fno != FILEVALOP)
    return NULL;

  if (ic[0] == (iconv_t)-2) {
    char *codeset = default_encoding();
    if (codeset && strcmp(codeset, "UTF-8"))
      ic[0] = iconv_open("UTF-8", codeset);
    else
      ic[0] = (iconv_t)-1;
  }
  if (ic[0] == (iconv_t)-1)
    return strdup(s);
  else {
    size_t l = strlen(s);
    char *t = malloc(l+1), *t1;
    char *inbuf = (char*)s, *outbuf = t;
    size_t inbytes = l, outbytes = l;

    while (iconv(ic[0], &inbuf, &inbytes, &outbuf, &outbytes) ==
	   (size_t)-1)
      if (errno == E2BIG) {
	/* try to enlarge the output buffer */
	size_t k = outbuf-t;
	if ((t1 = realloc(t, l+CHUNKSZ+1))) {
	  t = t1;
	  outbuf = t+k;
	  l += CHUNKSZ;
	  outbytes += CHUNKSZ;
	} else {
	  /* memory overflow */
	  free(t);
	  return NULL;
	}
      } else {
	/* conversion error */
	free(t);
	return strdup(s);
      }
    /* terminate the output string */
    *outbuf = 0;
    if (!(t1 = realloc(t, strlen(t)+1)))
      /* this shouldn't happen */
      return t;
    else
      return t1;
  }
}

__DLLIMPORT int qfile_encoding(qexpr file, const char *encoding)
{
  iconv_t *ic = ((EXPR*)file)->data.fargs.ic, ic1[2];

  if (((EXPR*)file)->fno != FILEVALOP)
    return 0;

  if (!encoding || !*encoding)
    encoding = default_encoding();
  if (encoding && strcmp(encoding, "UTF-8")) {
    ic1[0] = iconv_open("UTF-8", encoding);
    ic1[1] = iconv_open(encoding, "UTF-8");
    if (ic1[0] == (iconv_t)-1 || ic1[1] == (iconv_t)-1) {
      if (ic1[0] != (iconv_t)-1)
	iconv_close(ic1[0]);
      if (ic1[1] != (iconv_t)-1)
	iconv_close(ic1[1]);
      return 0;
    }
  } else
    ic1[0] = ic1[1] = (iconv_t)-1;

  /* close existing descriptors */
  if (ic[0] != (iconv_t)-2 &&
      ic[0] != (iconv_t)-1) {
    iconv_close(ic[0]);
    ic[0] = (iconv_t)-1;
  }
  if (ic[1] != (iconv_t)-2 &&
      ic[1] != (iconv_t)-1) {
    /* In a stateful encoding we might have to emit a terminating shift
       sequence. */
    FILE *fp = ((EXPR*)file)->data.fp;
    char *s = qfile_from_utf8(NULL, file), *t = s;
    if (t) {
      while (*s) putc(*s++, fp);
      free(t);
    }
    iconv_close(ic[1]);
    ic[1] = (iconv_t)-1;
  }

  ic[0] = ic1[0];
  ic[1] = ic1[1];
  return 1;
}

#else

__DLLIMPORT char *qfrom_utf8(const char *s, const char *encoding)
{
  return strdup(s);
}

__DLLIMPORT char *qto_utf8(const char *s, const char *encoding)
{
  return strdup(s);
}

__DLLIMPORT char *qfile_from_utf8(const char *s, qexpr file)
{
  return strdup(s);
}

__DLLIMPORT char *qfile_to_utf8(const char *s, qexpr file)
{
  return strdup(s);
}

__DLLIMPORT int qfile_encoding(qexpr file, const char *encoding)
{
  return 0;
}

#endif


/* Multithreading. */

__DLLIMPORT void qacquire(void)
{
  acquire_lock();
}

__DLLIMPORT void qrelease(void)
{
  release_lock();
}

__DLLIMPORT int qinit_thread(void)
{
  return init_thread();
}

__DLLIMPORT void qfini_thread(int id)
{
  if (id >= 0) {
    exit_thread(id);
    fini_thread(id);
  }
}


/* Memory allocation routines provided for Mingw/MSVC compatibility under
   Windows (see qint.h). */

__DLLIMPORT void *qmalloc(size_t size)
{
  return malloc(size);
}

__DLLIMPORT void *qrealloc(void *p, size_t size)
{
  return realloc(p, size);
}

__DLLIMPORT void *qcalloc(size_t num, size_t size)
{
  return calloc(num, size);
}

__DLLIMPORT void qfree(void *p)
{
  free(p);
}

#endif
