
/* This file is part of the Q programming system.

   The Q programming system is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   The Q programming system is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

#if defined (HAVE_CONFIG_H)
#  include "config.h"
#endif

/* system headers */

/* get the SUSV2 stuff (PTHREAD_MUTEX_RECURSIVE etc.) */
#ifndef _XOPEN_SOURCE
#define _XOPEN_SOURCE 500
#endif

/* get some BSDish stuff from the GNU library (Linux etc.) */
#define _BSD_SOURCE

/* get IPv6 stuff from Solaris */
#define __EXTENSIONS__

#include <stdio.h>
#include <ctype.h>
#include <signal.h>
#include <math.h>

#ifdef HAVE_ERRNO_H
#include <errno.h>
#else
# ifndef errno
extern int errno;
# endif
#endif

#include <sys/stat.h>

/* check for standard C headers */
#if STDC_HEADERS
# include <stdlib.h>
# include <string.h>
#else
# ifndef HAVE_STRCHR
#  define strchr index
#  define strrchr rindex
# endif
char *strchr (), *strrchr ();
#endif

#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif

#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif

#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#include <sys/types.h>
#if HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif
#ifndef WEXITSTATUS
# define WEXITSTATUS(stat_val) ((unsigned)(stat_val) >> 8)
#endif
#ifndef WIFEXITED
# define WIFEXITED(stat_val) (((stat_val) & 255) == 0)
#endif

#if HAVE_DIRENT_H
# include <dirent.h>
# define NAMLEN(dirent) strlen((dirent)->d_name)
#else
# define dirent direct
# define NAMLEN(dirent) (dirent)->d_namlen
# if HAVE_SYS_NDIR_H
#  include <sys/ndir.h>
# endif
# if HAVE_SYS_DIR_H
#  include <sys/dir.h>
# endif
# if HAVE_NDIR_H
#  include <ndir.h>
# endif
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_PWD_H
#include <pwd.h>
#endif

#ifdef HAVE_GRP_H
#include <grp.h>
#endif

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif

#ifdef HAVE_TERMIOS_H
#include <termios.h>
#endif

#ifdef HAVE_PTY_H
#include <pty.h>
#else
#ifdef HAVE_UTIL_H
#include <util.h>
#endif
#endif

#ifdef HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>
#endif

#ifdef HAVE_STROPTS_H
#include <stropts.h>
#endif

#ifdef USE_READLINE
#include <readline/readline.h>
#include <readline/history.h>
#endif

#ifdef HAVE_SYS_SOCKET_H
#ifdef __APPLE__
/* Apparently on OSX we need to undef _POSIX_C_SOURCE to get AF_LOCAL
   defined. Patch by Andrew Berg. */
#undef _POSIX_C_SOURCE
#endif
#include <sys/socket.h>
#ifdef HAVE_SYS_UN_H
#include <sys/un.h>
#endif
#include <netinet/in.h>
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#endif

#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif

#ifdef HAVE_UTIME_H
#include <utime.h>
#endif

#ifdef HAVE_SYS_UTSNAME_H
#include <sys/utsname.h>
#endif

#ifdef HAVE_SYS_TIMES_H
#include <sys/times.h>
#endif

#ifdef USE_THREADS
#ifdef HAVE_SCHED_H
#include <sched.h>
#endif
#include <pthread.h>
#include <semaphore.h>
#endif

#ifdef USE_UNICODE
#ifdef HAVE_WCHAR_H
#include <wchar.h>
#include <wctype.h>
#define HAVE_UNICODE 1
#endif
#endif

#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif
#ifdef HAVE_LANGINFO_CODESET
#include <langinfo.h>
#endif
#ifdef HAVE_ICONV
#include <iconv.h>
#endif
#ifdef ENABLE_NLS
#include <libintl.h>
#endif

#if defined(HAVE_SYS_SOCKET_H) || defined(WIN32)
#define HAVE_BSD_SOCKETS 1
#endif

#ifdef WIN32

#include <windows.h>
#include <process.h>
#include <io.h>
#include <fcntl.h>
#include <sys/utime.h>
#include <wchar.h>
#include <wctype.h>
#include <locale.h>
#include <iconv.h>
#include <libintl.h>
#include <readline.h>
#include <history.h>

#define USE_READLINE 1
#define HAVE_RL_COMPLETION_MATCHES 1

#define HAVE_STRDUP 1
#define HAVE_MEMCPY 1
#define HAVE_MEMCMP 1
#define HAVE_MEMMOVE 1
#define HAVE_MEMSET 1

#define HAVE_DUP 1
#define HAVE_DUP2 1
#define HAVE_PIPE 1
#define HAVE_FSTAT 1
#define HAVE_ACCESS 1
#define HAVE_CHMOD 1
#define HAVE_UTIME_H 1
#define HAVE_READDIR 1
#define HAVE_REWINDDIR 1

/* this one only works on sockets in Windows */
#define HAVE_SELECT 1

#define HAVE_SOCKET 1
#define HAVE_SHUTDOWN 1
#define HAVE_CLOSESOCKET 1
#define HAVE_BIND 1
#define HAVE_LISTEN 1
#define HAVE_ACCEPT 1
#define HAVE_CONNECT 1
#define HAVE_GETSOCKNAME 1
#define HAVE_GETPEERNAME 1
#define HAVE_GETSOCKOPT 1
#define HAVE_SETSOCKOPT 1
#define HAVE_RECV 1
#define HAVE_SEND 1
#define HAVE_RECVFROM 1
#define HAVE_SENDTO 1
#define HAVE_GETHOSTNAME 1
#define HAVE_GETHOSTBYNAME 1
#define HAVE_GETHOSTBYADDR 1
#define HAVE_GETPROTOBYNAME 1
#define HAVE_GETPROTOBYNUMBER 1
#define HAVE_GETSERVBYNAME 1
#define HAVE_GETSERVBYPORT 1

#define MSG_EOR (-1)
#define MSG_WAITALL (-1)

#define HAVE_IN_PORT_T 1
#define HAVE_IN_ADDR_T 1
#define HAVE_UINT16_T 1
#define uint16_t u_short

#define HAVE_DECL_TZNAME 1
#define HAVE_DECL_DAYLIGHT 1

#define HAVE_UNICODE 1
#define HAVE_LOCALE_H 1
#define HAVE_ICONV 1
#ifdef __MINGW32__
#define ENABLE_NLS 1
#define HAVE_WCSXFRM 1
#define HAVE_WCSCOLL 1
#define HAVE_TOWUPPER 1
#define HAVE_TOWLOWER 1
#endif

#endif

/* make sure that gmp.h is included prior to libq.h */

#include <gmp.h>
#include <libq.h>

/* work around a bug in some gmp versions */

#define my_mpz_fits_slong_p(z) (mpz_size(z) == 0 || mpz_fits_slong_p(z))
#define my_mpz_fits_ulong_p(z) (mpz_size(z) == 0 || mpz_fits_ulong_p(z))
#define my_mpz_fits_uint_p(z) (mpz_size(z) == 0 || mpz_fits_uint_p(z))

#ifdef DMALLOC
#include <dmalloc.h>
#endif

/* make sure that this matches up with the declaration in clib.c */

typedef unsigned char bool;
typedef struct {
  bool			active, canceled, used;	/* status */
  expr			arg;			/* thread argument */
  expr			result;			/* thread result, if any */
  expr			thread;			/* thread object */
#ifdef USE_THREADS
  pthread_t		id;			/* thread id */
  pthread_mutex_t	exit_mutex;		/* exit mutex and condition */
  pthread_cond_t	exit_cond;
#ifdef WIN32
  HANDLE		handle;			/* thread handle */
  int			pol;			/* scheduling policy */
#endif
#endif
} THREAD;

MODULE(system)

#ifndef HAVE_STRDUP

static char *strdup(char *s)
{
  char *t;
  return ((t=malloc(strlen(s)+1))?strcpy(t, s):NULL);
}

#endif

#ifndef HAVE_MEMCPY

#if __GNUC__ > 1
#define memcpy(TO,FROM,COUNT)	__builtin_memcpy(TO,FROM,COUNT)
#else
static void *memcpy (to, from, count)
     char *to;
     char *from;
     int count;
{
  register char *f = from;
  register char *t = to;
  register int i = count;

  while (i-- > 0)
    *t++ = *f++;
  return (void*)to;
}
#endif

#endif

#ifndef HAVE_MEMMOVE

#if __GNUC__ > 1
#define memmove(TO,FROM,COUNT)	__builtin_memmove(TO,FROM,COUNT)
#else
static void *memmove (char *to, char *from, int count)
{
  register char *f = from;
  register char *t = to;
  register int i = count;

  if (from > to) {
    while (i-- > 0)
      *t++ = *f++;
  } else if (from < to) {
    from += count; to += count;
    while (i-- > 0)
      *--t = *--f;
  }
  return (void*)to;
}
#endif

#endif

#ifndef HAVE_MEMSET

#if __GNUC__ > 1
#define memset(TO,C,COUNT)	__builtin_memset(TO,C,COUNT)
#else
static void *memset (char *to, int c, int count)
{
  register char f = (char)c;
  register char *t = to;
  register int i = count;

  while (i-- > 0)
    *t++ = f;
  return (void*)to;
}
#endif

#endif

#ifndef HAVE_MEMCMP

#if __GNUC__ > 1
#define memcmp(S1,S2,COUNT)	__builtin_memcmp(S1,S2,COUNT)
#else
static int memcmp (char *s1, char *s2, int count)
{
  register char *t1 = s1;
  register char *t2 = s2;
  register int i = count;
  register int c;

  while (i-- > 0)
    if ((c = *t1++ - *t2++) != 0)
      return c;
  return 0;
}
#endif

#endif

#define BUFSZ 1024

/* GMP convenience stuff */

#define BYTES_PER_LIMB sizeof(mp_limb_t)

static void*
my_mpz_realloc(m, new_size)
     mpz_ptr m;
     mp_size_t new_size;
{
  mpz_t m1;
  memcpy(m1, m, sizeof(mpz_t));
  if (_mpz_realloc(m, new_size))
    return m->_mp_d;
  else {
    if (m1->_mp_d) mpz_clear(m1);
    return NULL;
  }
}

static void *mpz_new(mpz_t z, mp_size_t sz)
{
  mpz_init(z);
  if (z->_mp_d && my_mpz_realloc(z, sz))
    return z->_mp_d;
  else
    return NULL;
}

static void *mpz_copy(mpz_t z, mpz_t u)
{
  if (mpz_new(z, mpz_size(u))) {
    mpz_set(z, u);
    return z->_mp_d;
  } else
    return NULL;
}

static void *mpz_resize(mpz_t z, mp_size_t sz)
{
  if (sz < mpz_size(z)) sz = mpz_size(z);
  if (sz == 0) sz = 1;
  if (sz != z->_mp_alloc && !my_mpz_realloc(z, sz))
    return NULL;
  else
    return z->_mp_d;
}

#define mpz_actsize(z) mpz_resize(z, mpz_size(z))

#define abs(x) (((x)<0.0)?-x:x)

static void *mpz_from_double(mpz_t z, double x)
{
  double ip, fp, dsz;
  int sz;
  fp = modf(x, &ip);
  /* estimate the number of limbs required */
  dsz = log(abs(ip))/log(2)/((double)CHAR_BIT*sizeof(mp_limb_t))+1.0;
  if (dsz < 1.0) dsz = 1.0; /* this can't happen?? */
  /* add an extra limb to be safe */
  sz = ((int)dsz)+1;
  /* this shouldn't happen but ... ;-) */
  if (((double)INT_MAX) <= dsz || sz < 0) return NULL;
  if (mpz_new(z, sz)) {
    mpz_set_d(z, x);
    return mpz_actsize(z);
  } else
    return NULL;
}

static long long_max(long x, long y)
{
  return (x > y)?x:y;
}

static long long_min(long x, long y)
{
  return (x < y)?x:y;
}

static void *mpz_addop2(void (*f)(), mpz_t z, mpz_t u, mpz_t v)
{
  int sz = long_max(mpz_size(u),mpz_size(v))+1;
  if (sz < 0) return NULL;
  if (mpz_new(z, sz)) {
    f(z, u, v);
    return mpz_actsize(z);
  } else
    return NULL;
}

static void *mpz_addop1(void (*f)(), mpz_t z, mpz_t u, mp_limb_t v)
{
  int sz = mpz_size(u)+1;
  if (sz < 0) return NULL;
  if (mpz_new(z, sz)) {
    f(z, u, v);
    return mpz_actsize(z);
  } else
    return NULL;
}

/* reverse the limbs of an mpz_t value to convert between little- and
   big-endian limb order */

static void reverse_limbs(mp_limb_t *p, unsigned n)
{
  unsigned i;
  for (i = 0; i < n/2; i++) {
    mp_limb_t x = p[i];
    p[i] = p[n-i-1];
    p[n-i-1] = x;
  }
}

/* utf-8/unicode helpers */

#define sys_to_utf8(s) to_utf8(s, NULL)
#define utf8_to_sys(s) from_utf8(s, NULL)

#ifdef HAVE_UNICODE

static inline long
u8decode(char *s)
{
  size_t n = 0;
  unsigned p = 0, q = 0;
  unsigned long c = 0;
  if (s[0] == 0)
    return -1;
  else if (s[1] == 0)
    return (unsigned char)s[0];
  for (; n == 0 && *s; s++) {
    unsigned char uc = (unsigned char)*s;
    if (q == 0) {
      if (((signed char)uc) < 0) {
	switch (uc & 0xf0) {
	case 0xc0: case 0xd0:
	  q = 1;
	  c = uc & 0x1f;
	  break;
	case 0xe0:
	  q = 2;
	  c = uc & 0xf;
	  break;
	case 0xf0:
	  if ((uc & 0x8) == 0) {
	    q = 3;
	    c = uc & 0x7;
	  } else
	    c = uc;
	  break;
	default:
	  c = uc;
	  break;
	}
      } else
	c = uc;
      p = 0; if (q == 0) n++;
    } else if ((uc & 0xc0) == 0x80) {
      /* continuation byte */
      c = c << 6 | (uc & 0x3f);
      if (--q == 0)
	n++;
      else
	p++;
    } else {
      /* malformed char */
      return -1;
    }
  }
  if (n == 1 && *s == 0)
    return c;
  else
    return -1;
}

static inline long
u8decodes(char *s, char **t)
{
  size_t n = 0;
  unsigned p = 0, q = 0;
  unsigned long c = 0;
  if (s[0] == 0)
    return -1;
  else if (s[1] == 0) {
    *t = s+1;
    return (unsigned char)s[0];
  }
  for (; n == 0 && *s; s++) {
    unsigned char uc = (unsigned char)*s;
    if (q == 0) {
      if (((signed char)uc) < 0) {
	switch (uc & 0xf0) {
	case 0xc0: case 0xd0:
	  q = 1;
	  c = uc & 0x1f;
	  break;
	case 0xe0:
	  q = 2;
	  c = uc & 0xf;
	  break;
	case 0xf0:
	  if ((uc & 0x8) == 0) {
	    q = 3;
	    c = uc & 0x7;
	  } else
	    c = uc;
	  break;
	default:
	  c = uc;
	  break;
	}
      } else
	c = uc;
      p = 0; if (q == 0) n++;
    } else if ((uc & 0xc0) == 0x80) {
      /* continuation byte */
      c = c << 6 | (uc & 0x3f);
      if (--q == 0)
	n++;
      else
	p++;
    } else {
      /* malformed char */
      return -1;
    }
  }
  if (n == 1) {
    *t = s;
    return c;
  } else
    return -1;
}

static inline char *
u8encode(char *t, unsigned long c)
{
  unsigned char *uc = (unsigned char*)t;
  if (c < 0x80) {
    uc[1] = 0;
    uc[0] = c;
  } else if (c < 0x800) {
    uc[2] = 0;
    uc[1] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[0] = 0xc0 | c;
  } else if (c < 0x10000) {
    uc[3] = 0;
    uc[2] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[1] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[0] = 0xe0 | c;
  } else {
    uc[4] = 0;
    uc[3] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[2] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[1] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[0] = 0xf0 | c;
  }
  return t;
}

static inline size_t
u8strlen(char *s)
{
  size_t n = 0;
  unsigned p = 0, q = 0;
 start:
  for (; *s; s++) {
    unsigned char uc = (unsigned char)*s;
    if (q == 0) {
      if (((signed char)uc) < 0) {
	switch (uc & 0xf0) {
	case 0xc0: case 0xd0:
	  q = 1;
	  break;
	case 0xe0:
	  q = 2;
	  break;
	case 0xf0:
	  if ((uc & 0x8) == 0)
	    q = 3;
	  break;
	}
      }
      p = 0; n++;
    } else if ((uc & 0xc0) == 0x80) {
      /* continuation byte */
      if (--q == 0)
	p = 0;
      else
	p++;
    } else {
      /* malformed char */
      s -= p+1; p = q = 0;
    }
  }
  if (q > 0) {
    /* unterminated char */
    s -= p; p = q = 0;
    goto start;
  }
  return n;
}

#ifdef HAVE_ICONV

/* conversion between UTF-8 and wchar_t */

static iconv_t myic[2] = { (iconv_t)-1, (iconv_t)-1 };

static inline wchar_t *
ictowcs(wchar_t *t, char *s)
{
  if (myic[1] == (iconv_t)-1)
    myic[1] = iconv_open("WCHAR_T", "UTF-8");
  if (myic[1] == (iconv_t)-1)
    return NULL;
  else {
    size_t l = strlen(s);
    char *inbuf = s; wchar_t *outbuf = t;
    size_t inbytes = l, outbytes = l*sizeof(wchar_t);

    if (iconv(myic[1], &inbuf, &inbytes, (char**)&outbuf, &outbytes) ==
	(size_t)-1)
      return NULL;
    /* terminate the output string */
    *outbuf = 0;
    return t;
  }
}

static inline char *
icfromwcs(char *t, wchar_t *s)
{
  if (myic[0] == (iconv_t)-1)
    myic[0] = iconv_open("UTF-8", "WCHAR_T");
  if (myic[0] == (iconv_t)-1)
    return NULL;
  else {
    size_t l = wcslen(s);
    wchar_t *inbuf = s; char *outbuf = t;
    size_t inbytes = l*sizeof(wchar_t), outbytes = l*4;

    if (iconv(myic[0], (char**)&inbuf, &inbytes, &outbuf, &outbytes) ==
	(size_t)-1)
      return NULL;
    /* terminate the output string */
    *outbuf = 0;
    return t;
  }
}

#endif

#ifdef __STDC_ISO_10646__
#define MY_ISO_10646
#else
#ifndef HAVE_ICONV
#warning "wchar_t encoding unknown and iconv not available, assuming ISO 10646"
#define MY_ISO_10646
#endif
#endif

#ifdef MY_ISO_10646
#define towchar(c) ((wchar_t)(c))
#else
static wchar_t towchar(unsigned long c)
{
  char s[5]; /* single utf-8 char can have at most 4 bytes, plus terminal 0 */
  wchar_t t[5]; /* just to be safe; 2 should actually be enough */
  u8encode(s, c);
  if (ictowcs(t, s))
    return t[0];
  else
    /* Your system is so utterly broken that we can't even convert UTF-8 to
       wchar_t. You should probably configure with --without-unicode. But
       let's just pretend we have an ISO 10646 compatible encoding anyway. */
    return (wchar_t)c;
}
#endif

static wchar_t *u8towcs(wchar_t *t, char *s)
{
#ifdef MY_ISO_10646
  wchar_t *t0 = t;
  while (*s) {
    char *s1;
    long c = u8decodes(s, &s1);
    if (c < 0)
      *t++ = (wchar_t)(*s++);
    else {
      *t++ = (wchar_t)c;
      s = s1;
    }
  }
  *t = 0;
  return t0;
#else
  return ictowcs(t, s);
#endif
}

static char *wcstou8(char *t, wchar_t *s)
{
#ifdef MY_ISO_10646
  char *t0 = t;
  while (*s) {
    u8encode(t, *s++);
    t += strlen(t);
  }
  return t0;
#else
  return icfromwcs(t, s);
#endif
}

#endif

#ifdef USE_THREADS
static pthread_mutex_t format_mutex;
#endif

static void lock_format(void)
{
#ifdef USE_THREADS
  pthread_mutex_lock(&format_mutex);
#endif
}

static void unlock_format(void)
{
#ifdef USE_THREADS
  pthread_mutex_unlock(&format_mutex);
#endif
}

/* manifest constants */

/* O_NDELAY and O_NONBLOCK are synonymous */

#ifndef O_NDELAY
#ifdef O_NONBLOCK
#define O_NDELAY O_NONBLOCK
#else
#define O_NDELAY 0
#endif
#endif

#ifndef O_NONBLOCK
#define O_NONBLOCK O_NDELAY
#endif

#ifndef O_NOCTTY
#define O_NOCTTY 0
#endif

#ifndef O_BINARY
#define O_BINARY 0
#endif

#ifndef FD_CLOEXEC
#define FD_CLOEXEC 0
#endif

/* POSIX requires this one, but I am not sure whether it is defined on all
   systems */

#ifndef O_ACCMODE
#define O_ACCMODE (O_RDONLY|O_WRONLY|O_RDWR)
#endif

/* File type bits. */

#ifndef S_IFBLK
#define S_IFBLK 0
#endif
#ifndef S_IFCHR
#define S_IFCHR 0
#endif
#ifndef S_IFIFO
#define S_IFIFO 0
#endif
#ifndef S_IFREG
#define S_IFREG 0
#endif
#ifndef S_IFDIR
#define S_IFDIR 0
#endif
#ifndef S_IFLNK
#define S_IFLNK 0
#endif
#ifndef S_IFSOCK
#define S_IFSOCK 0
#endif

#ifndef S_IFMT
#define S_IFMT (S_IFMT|S_IFBLK|S_IFCHR|S_IFIFO|S_IFREG|S_IFDIR|S_IFLNK|S_IFSOCK)
#endif

/* Fcntl command codes. */

#ifndef F_DUPFD
#define F_DUPFD (-1)
#endif
#ifndef F_GETFD
#define F_GETFD (-1)
#endif
#ifndef F_SETFD
#define F_SETFD (-1)
#endif
#ifndef F_GETFL
#define F_GETFL (-1)
#endif

#ifndef F_SETFL
#define F_SETFL (-1)
#endif
#ifndef F_GETLK
#define F_GETLK (-1)
#endif
#ifndef F_SETLK
#define F_SETLK (-1)
#endif
#ifndef F_SETLKW
#define F_SETLKW (-1)
#endif

/* Lock types. */

#ifndef F_RDLCK
#define F_RDLCK (-1)
#endif
#ifndef F_WRLCK
#define F_WRLCK (-1)
#endif
#ifndef F_UNLCK
#define F_UNLCK (-1)
#endif

/* Access options. */

#ifndef F_OK
#define F_OK 0
#endif
#ifndef R_OK
#define R_OK 0
#endif
#ifndef W_OK
#define W_OK 0
#endif
#ifndef X_OK
#define X_OK 0
#endif

/* Wait options. */

#ifndef WNOHANG
#define WNOHANG 0
#endif
#ifndef WUNTRACED
#define WUNTRACED 0
#endif

/* Time units. */

#ifndef CLK_TCK
#ifdef HAVE_SYSCONF
#define CLK_TCK sysconf(_SC_CLK_TCK)
#else
#define CLK_TCK 0
#endif
#endif

/* Termios constants. */

#ifndef TCSANOW
#define TCSANOW (-1)
#endif
#ifndef TCSADRAIN
#define TCSADRAIN (-1)
#endif
#ifndef TCSAFLUSH
#define TCSAFLUSH (-1)
#endif

#ifndef TCIFLUSH
#define TCIFLUSH (-1)
#endif
#ifndef TCIOFLUSH
#define TCIOFLUSH (-1)
#endif
#ifndef TCOFLUSH
#define TCOFLUSH (-1)
#endif

#ifndef TCIOFF
#define TCIOFF (-1)
#endif
#ifndef TCION
#define TCION (-1)
#endif
#ifndef TCOOFF
#define TCOOFF (-1)
#endif
#ifndef TCOON
#define TCOON (-1)
#endif

#ifndef BRKINT
#define BRKINT 0
#endif
#ifndef ICRNL
#define ICRNL 0
#endif
#ifndef IGNBRK
#define IGNBRK 0
#endif
#ifndef IGNCR
#define IGNCR 0
#endif
#ifndef IGNPAR
#define IGNPAR 0
#endif
#ifndef INLCR
#define INLCR 0
#endif
#ifndef INPCK
#define INPCK 0
#endif
#ifndef ISTRIP
#define ISTRIP 0
#endif
#ifndef IUCLC
#define IUCLC 0
#endif
#ifndef IXANY
#define IXANY 0
#endif
#ifndef IXOFF
#define IXOFF 0
#endif
#ifndef IXON
#define IXON 0
#endif
#ifndef PARMRK
#define PARMRK 0
#endif

#ifndef OPOST
#define OPOST 0
#endif
#ifndef OLCUC
#define OLCUC 0
#endif
#ifndef ONLCR
#define ONLCR 0
#endif
#ifndef OCRNL
#define OCRNL 0
#endif
#ifndef ONOCR
#define ONOCR 0
#endif
#ifndef ONLRET
#define ONLRET 0
#endif
#ifndef OFILL
#define OFILL 0
#endif
#ifndef NLDLY
#define NLDLY 0
#endif
#ifndef NL0
#define NL0 0
#endif
#ifndef NL1
#define NL1 0
#endif
#ifndef CRDLY
#define CRDLY 0
#endif
#ifndef CR0
#define CR0 0
#endif
#ifndef CR1
#define CR1 0
#endif
#ifndef CR2
#define CR2 0
#endif
#ifndef CR3
#define CR3 0
#endif
#ifndef TABDLY
#define TABDLY 0
#endif
#ifndef TAB0
#define TAB0 0
#endif
#ifndef TAB1
#define TAB1 0
#endif
#ifndef TAB2
#define TAB2 0
#endif
#ifndef TAB3
#define TAB3 0
#endif
#ifndef BSDLY
#define BSDLY 0
#endif
#ifndef BS0
#define BS0 0
#endif
#ifndef BS1
#define BS1 0
#endif
#ifndef VTDLY
#define VTDLY 0
#endif
#ifndef VT0
#define VT0 0
#endif
#ifndef VT1
#define VT1 0
#endif
#ifndef FFDLY
#define FFDLY 0
#endif
#ifndef FF0
#define FF0 0
#endif
#ifndef FF1
#define FF1 0
#endif

#ifndef CSIZE
#define CSIZE 0
#endif
#ifndef CS5
#define CS5 0
#endif
#ifndef CS6
#define CS6 0
#endif
#ifndef CS7
#define CS7 0
#endif
#ifndef CS8
#define CS8 0
#endif
#ifndef CSTOPB
#define CSTOPB 0
#endif
#ifndef CREAD
#define CREAD 0
#endif
#ifndef PARENB
#define PARENB 0
#endif
#ifndef PARODD
#define PARODD 0
#endif
#ifndef HUPCL
#define HUPCL 0
#endif
#ifndef CLOCAL
#define CLOCAL 0
#endif

#ifndef ECHO
#define ECHO 0
#endif
#ifndef ECHOE
#define ECHOE 0
#endif
#ifndef ECHOK
#define ECHOK 0
#endif
#ifndef ECHONL
#define ECHONL 0
#endif
#ifndef ICANON
#define ICANON 0
#endif
#ifndef IEXTEN
#define IEXTEN 0
#endif
#ifndef ISIG
#define ISIG 0
#endif
#ifndef NOFLSH
#define NOFLSH 0
#endif
#ifndef TOSTOP
#define TOSTOP 0
#endif
#ifndef XCASE
#define XCASE 0
#endif

#ifndef B0
#define B0 (-1)
#endif
#ifndef B50
#define B50 (-1)
#endif
#ifndef B75
#define B75 (-1)
#endif
#ifndef B110
#define B110 (-1)
#endif
#ifndef B134
#define B134 (-1)
#endif
#ifndef B150
#define B150 (-1)
#endif
#ifndef B200
#define B200 (-1)
#endif
#ifndef B300
#define B300 (-1)
#endif
#ifndef B600
#define B600 (-1)
#endif
#ifndef B1200
#define B1200 (-1)
#endif
#ifndef B1800
#define B1800 (-1)
#endif
#ifndef B2400
#define B2400 (-1)
#endif
#ifndef B4800
#define B4800 (-1)
#endif
#ifndef B9600
#define B9600 (-1)
#endif
#ifndef B19200
#define B19200 (-1)
#endif
#ifndef B38400
#define B38400 (-1)
#endif

#ifndef VEOF
#define VEOF (-1)
#endif
#ifndef VEOL
#define VEOL (-1)
#endif
#ifndef VERASE
#define VERASE (-1)
#endif
#ifndef VINTR
#define VINTR (-1)
#endif
#ifndef VKILL
#define VKILL (-1)
#endif
#ifndef VMIN
#define VMIN (-1)
#endif
#ifndef VQUIT
#define VQUIT (-1)
#endif
#ifndef VSTART
#define VSTART (-1)
#endif
#ifndef VSTOP
#define VSTOP (-1)
#endif
#ifndef VSUSP
#define VSUSP (-1)
#endif
#ifndef VTIME
#define VTIME (-1)
#endif

/* legacy stuff */

#ifndef IUCLC
#define IUCLC 0
#endif
#ifndef OLCUC
#define OLCUC 0
#endif
#ifndef XCASE
#define XCASE 0
#endif

/* Socket-related stuff. */

#ifndef HAVE_BSD_SOCKETS

#define AF_LOCAL (-1)
#define AF_INET (-1)
#define AF_INET6 (-1)
#define SOCK_STREAM (-1)
#define SOCK_DGRAM (-1)
#define SOCK_SEQPACKET (-1)
#define SOCK_RAW (-1)
#define SOCK_RDM (-1)
#define SHUT_RD (-1)
#define SHUT_WR (-1)
#define SHUT_RDWR (-1)
#define MSG_EOR (-1)
#define MSG_OOB (-1)
#define MSG_PEEK (-1)
#define MSG_WAITALL (-1)
#define SOL_SOCKET (-1)
#define SO_ACCEPTCONN (-1)
#define SO_BROADCAST (-1)
#define SO_DEBUG (-1)
#define SO_DONTROUTE (-1)
#define SO_ERROR (-1)
#define SO_KEEPALIVE (-1)
#define SO_LINGER (-1)
#define SO_OOBINLINE (-1)
#define SO_RCVBUF (-1)
#define SO_RCVLOWAT (-1)
#define SO_RCVTIMEO (-1)
#define SO_REUSEADDR (-1)
#define SO_SNDBUF (-1)
#define SO_SNDLOWAT (-1)
#define SO_SNDTIMEO (-1)
#define SO_TYPE (-1)
#define IPPROTO_IP (-1)
#define IPPROTO_ICMP (-1)
#define IPPROTO_TCP (-1)
#define IPPROTO_UDP (-1)

#else

#if !defined(AF_LOCAL) && defined(PF_LOCAL)
#define AF_LOCAL PF_LOCAL
#endif
#if !defined(AF_INET) && defined(PF_INET)
#define AF_INET PF_INET
#endif
#if !defined(AF_INET6) && defined(PF_INET6)
#define AF_INET6 PF_INET6
#endif

#ifndef SOCK_RAW
#define SOCK_RAW (-1)
#endif
#ifndef SOCK_RDM
#define SOCK_RDM (-1)
#endif
#ifndef SOCK_SEQPACKET
#define SOCK_SEQPACKET (-1)
#endif
#ifndef SHUT_RD
#define SHUT_RD 0
#endif
#ifndef SHUT_WR
#define SHUT_WR 1
#endif
#ifndef SHUT_RDWR
#define SHUT_RDWR 2
#endif

#ifndef MSG_EOR
#define MSG_EOR 0
#endif
#ifndef MSG_PEEK
#define MSG_PEEK 0
#endif
#ifndef MSG_WAITALL
#define MSG_WAITALL 0
#endif

#ifndef SO_ACCEPTCONN
#define SO_ACCEPTCONN (-1)
#endif
#ifndef SO_BROADCAST
#define SO_BROADCAST (-1)
#endif
#ifndef SO_DONTROUTE
#define SO_DONTROUTE (-1)
#endif
#ifndef SO_ERROR
#define SO_ERROR (-1)
#endif
#ifndef SO_KEEPALIVE
#define SO_KEEPALIVE (-1)
#endif
#ifndef SO_LINGER
#define SO_LINGER (-1)
#endif
#ifndef SO_OOBINLINE
#define SO_OOBINLINE (-1)
#endif
#ifndef SO_RCVBUF
#define SO_RCVBUF (-1)
#endif
#ifndef SO_RCVLOWAT
#define SO_RCVLOWAT (-1)
#endif
#ifndef SO_RCVTIMEO
#define SO_RCVTIMEO (-1)
#endif
#ifndef SO_SNDBUF
#define SO_SNDBUF (-1)
#endif
#ifndef SO_SNDLOWAT
#define SO_SNDLOWAT (-1)
#endif
#ifndef SO_SNDTIMEO
#define SO_SNDTIMEO (-1)
#endif
#ifndef SO_TYPE
#define SO_TYPE (-1)
#endif

#ifndef IPPROTO_IP
#define IPPROTO_IP 0
#endif

#ifndef INADDR_NONE
#define INADDR_NONE ((in_addr_t)(-1))
#endif

#ifndef HAVE_SOCKLEN_T
#define socklen_t int
#endif
#ifndef HAVE_IN_PORT_T
#define in_port_t short
#endif
#ifndef HAVE_IN_ADDR_T
#define in_addr_t int
#endif
#ifndef HAVE_UINT16_T
#define uint16_t unsigned short
#endif

#endif

#ifndef P_WAIT
#define P_WAIT 0
#define P_NOWAIT 1
#define P_OVERLAY 2
#define P_DETACH 4
#endif

FUNCTION(system,sys_vars,argc,argv)
{
  if (argc != 0) return __FAIL;
  return mktuplel
    (246,

     mkint(P_WAIT), mkint(P_NOWAIT), mkint(P_OVERLAY), mkint(P_DETACH),

     mkint(O_RDONLY), mkint(O_WRONLY), mkint(O_RDWR), mkint(O_ACCMODE),

     mkint(FD_CLOEXEC),

     mkint(O_CREAT), mkint(O_EXCL), mkint(O_TRUNC), mkint(O_APPEND),
     mkint(O_NONBLOCK), mkint(O_NDELAY), mkint(O_NOCTTY),
     mkint(O_BINARY),

     mkint(S_IFMT), mkint(S_IFBLK), mkint(S_IFCHR), mkint(S_IFIFO),
     mkint(S_IFREG), mkint(S_IFDIR), mkint(S_IFLNK), mkint(S_IFSOCK),

     mkint(F_DUPFD), mkint(F_GETFD), mkint(F_SETFD), mkint(F_GETFL),
     mkint(F_SETFL), mkint(F_GETLK), mkint(F_SETLK), mkint(F_SETLKW),

     mkint(F_RDLCK), mkint(F_WRLCK), mkint(F_UNLCK),

     mkint(F_OK), mkint(R_OK), mkint(W_OK), mkint(X_OK),

     mkint(WNOHANG), mkint(WUNTRACED),

     mkint((long)CLOCKS_PER_SEC), mkint((long)CLK_TCK),

     mkint(TCSANOW),
     mkint(TCSADRAIN),
     mkint(TCSAFLUSH),

     mkint(TCIFLUSH),
     mkint(TCIOFLUSH),
     mkint(TCOFLUSH),

     mkint(TCIOFF),
     mkint(TCION),
     mkint(TCOOFF),
     mkint(TCOON),

     mkint(BRKINT),
     mkint(ICRNL),
     mkint(IGNBRK),
     mkint(IGNCR),
     mkint(IGNPAR),
     mkint(INLCR),
     mkint(INPCK),
     mkint(ISTRIP),
     mkint(IUCLC),
     mkint(IXANY),
     mkint(IXOFF),
     mkint(IXON),
     mkint(PARMRK),

     mkint(OPOST),
     mkint(OLCUC),
     mkint(ONLCR),
     mkint(OCRNL),
     mkint(ONOCR),
     mkint(ONLRET),
     mkint(OFILL),
     mkint(NLDLY),
     mkint(NL0),
     mkint(NL1),
     mkint(CRDLY),
     mkint(CR0),
     mkint(CR1),
     mkint(CR2),
     mkint(CR3),
     mkint(TABDLY),
     mkint(TAB0),
     mkint(TAB1),
     mkint(TAB2),
     mkint(TAB3),
     mkint(BSDLY),
     mkint(BS0),
     mkint(BS1),
     mkint(VTDLY),
     mkint(VT0),
     mkint(VT1),
     mkint(FFDLY),
     mkint(FF0),
     mkint(FF1),

     mkint(CSIZE),
     mkint(CS5),
     mkint(CS6),
     mkint(CS7),
     mkint(CS8),
     mkint(CSTOPB),
     mkint(CREAD),
     mkint(PARENB),
     mkint(PARODD),
     mkint(HUPCL),
     mkint(CLOCAL),

     mkint(ECHO),
     mkint(ECHOE),
     mkint(ECHOK),
     mkint(ECHONL),
     mkint(ICANON),
     mkint(IEXTEN),
     mkint(ISIG),
     mkint(NOFLSH),
     mkint(TOSTOP),
     mkint(XCASE),

     mkint(B0),
     mkint(B50),
     mkint(B75),
     mkint(B110),
     mkint(B134),
     mkint(B150),
     mkint(B200),
     mkint(B300),
     mkint(B600),
     mkint(B1200),
     mkint(B1800),
     mkint(B2400),
     mkint(B4800),
     mkint(B9600),
     mkint(B19200),
     mkint(B38400),

     mkint(VEOF),
     mkint(VEOL),
     mkint(VERASE),
     mkint(VINTR),
     mkint(VKILL),
     mkint(VMIN),
     mkint(VQUIT),
     mkint(VSTART),
     mkint(VSTOP),
     mkint(VSUSP),
     mkint(VTIME),

#ifdef AF_LOCAL
     mkint(AF_LOCAL),
#else
     /* not supported on Windows */
     mkint(-1),
#endif
#ifdef AF_INET
     mkint(AF_INET),
#else
     mkint(-1),
#endif
#ifdef AF_INET6
     mkint(AF_INET6),
#else
     mkint(-1),
#endif
     mkint(SOCK_STREAM),
     mkint(SOCK_DGRAM),
     mkint(SOCK_RAW),
     mkint(SOCK_RDM),
     mkint(SOCK_SEQPACKET),
     mkint(SHUT_RD),
     mkint(SHUT_WR),
     mkint(SHUT_RDWR),
     mkint(MSG_EOR),
     mkint(MSG_OOB),
     mkint(MSG_PEEK),
     mkint(MSG_WAITALL),
     mkint(SOL_SOCKET),
     mkint(SO_ACCEPTCONN),
     mkint(SO_BROADCAST),
     mkint(SO_DEBUG),
     mkint(SO_DONTROUTE),
     mkint(SO_ERROR),
     mkint(SO_KEEPALIVE),
     mkint(SO_LINGER),
     mkint(SO_OOBINLINE),
     mkint(SO_RCVBUF),
     mkint(SO_RCVLOWAT),
     mkint(SO_RCVTIMEO),
     mkint(SO_REUSEADDR),
     mkint(SO_SNDBUF),
     mkint(SO_SNDLOWAT),
     mkint(SO_SNDTIMEO),
     mkint(SO_TYPE),
     mkint(IPPROTO_IP),
     mkint(IPPROTO_ICMP),
     mkint(IPPROTO_TCP),
     mkint(IPPROTO_UDP),

#ifdef _POSIX_TIMERS
     mkint(CLOCK_REALTIME),
#ifdef _POSIX_MONOTONIC_CLOCK
     mkint(CLOCK_MONOTONIC),
#else
     mkvoid,
#endif
#ifdef _POSIX_CPUTIME
     mkint(CLOCK_PROCESS_CPUTIME_ID),
#else
     mkvoid,
#endif
#ifdef _POSIX_THREAD_CPUTIME
     mkint(CLOCK_THREAD_CPUTIME_ID),
#else
     mkvoid,
#endif
#else
     mkvoid,
     mkvoid,
     mkvoid,
     mkvoid,
#endif

#ifdef HAVE_LOCALE_H
     mkint(LC_ALL),
     mkint(LC_COLLATE),
     mkint(LC_CTYPE),
     mkint(LC_MESSAGES),
     mkint(LC_MONETARY),
     mkint(LC_NUMERIC),
     mkint(LC_TIME),
#else
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
#endif

#ifdef HAVE_LANGINFO_CODESET     
     mkint(CODESET),
     
     mkint(D_T_FMT),
     mkint(D_FMT),
     mkint(T_FMT),
     mkint(T_FMT_AMPM),
     
     mkint(AM_STR),
     mkint(PM_STR),
     
     mkint(DAY_1),
     mkint(DAY_2),
     mkint(DAY_3),
     mkint(DAY_4),
     mkint(DAY_5),
     mkint(DAY_6),
     mkint(DAY_7),
     
     mkint(ABDAY_1),
     mkint(ABDAY_2),
     mkint(ABDAY_3),
     mkint(ABDAY_4),
     mkint(ABDAY_5),
     mkint(ABDAY_6),
     mkint(ABDAY_7),
     
     mkint(MON_1),
     mkint(MON_2),
     mkint(MON_3),
     mkint(MON_4),
     mkint(MON_5),
     mkint(MON_6),
     mkint(MON_7),
     mkint(MON_8),
     mkint(MON_9),
     mkint(MON_10),
     mkint(MON_11),
     mkint(MON_12),
     
     mkint(ABMON_1),
     mkint(ABMON_2),
     mkint(ABMON_3),
     mkint(ABMON_4),
     mkint(ABMON_5),
     mkint(ABMON_6),
     mkint(ABMON_7),
     mkint(ABMON_8),
     mkint(ABMON_9),
     mkint(ABMON_10),
     mkint(ABMON_11),
     mkint(ABMON_12),
     
     mkint(ERA),
     mkint(ERA_D_T_FMT),
     mkint(ERA_D_FMT),
     mkint(ERA_T_FMT),
     
     mkint(ALT_DIGITS),
     mkint(RADIXCHAR),
     mkint(THOUSEP),
     
     mkint(YESEXPR),
     mkint(NOEXPR),
     
     mkint(CRNCYSTR)
#else
     mkint(-1),
     
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     
     mkint(-1),
     mkint(-1),
     
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     
     mkint(-1),
     mkint(-1),
     mkint(-1),
     mkint(-1),
     
     mkint(-1),
     mkint(-1),
     mkint(-1),
     
     mkint(-1),
     mkint(-1),
     
     mkint(-1)
#endif

     );
}

/* byte strings: ************************************************************/

/* Note that this type is actually implemented in clib, but we need to
   duplicate the C type definition and the creation function here. */

typedef struct bstr {
  long size;
  unsigned char *v;
} bstr_t;

static expr mkbstr(long size, void *v)
{
  bstr_t *m;
  if ((m = malloc(sizeof(bstr_t)))) {
    m->size = size;
    m->v = (unsigned char*)v;
    return mkobj(type(ByteStr), m);
  } else {
    if (v) free(v);
    return __ERROR;
  }
}

/* low-level I/O ************************************************************/

FUNCTION(system,open,argc,argv)
{
  char *name;
  long flags, mode, fd;
  if (argc != 3 || !isstr(argv[0], &name) || !isint(argv[1], &flags) ||
      !isint(argv[2], &mode))
    return __FAIL;
  if (!(name = utf8_to_sys(name)))
    return __ERROR;
  release_lock();
  fd = open(name, flags, mode);
  acquire_lock();
  free(name);
  if (fd < 0)
    return __FAIL;
  else
    return mkint(fd);
}

FUNCTION(system,close,argc,argv)
{
  long fd;
  if (argc != 1 || !isint(argv[0], &fd))
    return __FAIL;
  if (close(fd))
    return __FAIL;
  else
    return mkvoid;
}

FUNCTION(system,dup,argc,argv)
{
#ifdef HAVE_DUP
  long fd;
  if (argc != 1 || !isint(argv[0], &fd))
    return __FAIL;
  fd = dup(fd);
  if (fd < 0)
    return __FAIL;
  else
    return mkint(fd);
#else
  return __FAIL;
#endif
}

FUNCTION(system,dup2,argc,argv)
{
#ifdef HAVE_DUP2
  long fd, fd2;
  if (argc != 2 || !isint(argv[0], &fd) || !isint(argv[1], &fd2))
    return __FAIL;
  fd = dup2(fd, fd2);
  if (fd < 0)
    return __FAIL;
  else
    return mkint(fd);
#else
  return __FAIL;
#endif
}

FUNCTION(system,pipe,argc,argv)
{
#ifdef HAVE_PIPE
  int fd[2];
  if (argc != 0)
    return __FAIL;
#ifdef WIN32
  if (_pipe(fd, 256, O_BINARY))
#else
  if (pipe(fd))
#endif
    return __FAIL;
  else
    return mktuplel(2, mkint(fd[0]), mkint(fd[1]));
#else
  return __FAIL;
#endif
}

static expr statres(struct stat *buf)
{
  expr *st, x;
  if (!(st = malloc(11*sizeof(expr)))) return __ERROR;
  /* FIXME: all types are assumed to fit into a 4 byte value; this may
     not be true for some (dev_t, maybe others?) */
  st[0] = mkuint(buf->st_dev);
  st[1] = mkuint(buf->st_ino);
  st[2] = mkuint(buf->st_mode);
  st[3] = mkuint(buf->st_nlink);
  st[4] = mkuint(buf->st_uid);
  st[5] = mkuint(buf->st_gid);
#ifdef HAVE_ST_RDEV
  st[6] = mkuint(buf->st_rdev);
#else
  st[6] = mkuint(0);
#endif
  st[7] = mkint(buf->st_size);
  st[8] = mkint(buf->st_atime);
  st[9] = mkint(buf->st_mtime);
  st[10] = mkint(buf->st_ctime);
  x = mktuplev(11, st);
  if (x)
    return x;
  else
    return __ERROR;
}

FUNCTION(system,fstat,argc,argv)
{
#ifdef HAVE_FSTAT
  long fd;
  struct stat buf;
  if (argc != 1 || !isint(argv[0], &fd))
    return __FAIL;
  if (fstat(fd, &buf))
    return __FAIL;
  else
    return statres(&buf);
#else
  return __FAIL;
#endif
}

FUNCTION(system,fchdir,argc,argv)
{
#ifdef HAVE_FCHDIR
  long fd;
  if (argc != 1 || !isint(argv[0], &fd))
    return __FAIL;
  if (fchdir(fd))
    return __FAIL;
  else
    return mkvoid;
#else
  return __FAIL;
#endif
}

FUNCTION(system,fchmod,argc,argv)
{
#ifdef HAVE_FCHMOD
  long fd, mode;
  if (argc != 2 || !isint(argv[0], &fd) || !isint(argv[1], &mode))
    return __FAIL;
  if (fchmod(fd, mode))
    return __FAIL;
  else
    return mkvoid;
#else
  return __FAIL;
#endif
}

FUNCTION(system,fchown,argc,argv)
{
#ifdef HAVE_FCHOWN
  long fd, uid, gid;
  if (argc != 3 || !isint(argv[0], &fd) || !isint(argv[1], &uid) ||
      !isint(argv[2], &gid))
    return __FAIL;
  if (fchown(fd, uid, gid))
    return __FAIL;
  else
    return mkvoid;
#else
  return __FAIL;
#endif
}

FUNCTION(system,ftruncate,argc,argv)
{
#ifdef HAVE_FTRUNCATE
  long fd, len;
  if (argc != 2 || !isint(argv[0], &fd) || !isint(argv[1], &len))
    return __FAIL;
  if (ftruncate(fd, len))
    return __FAIL;
  else
    return mkvoid;
#else
  return __FAIL;
#endif
}

FUNCTION(system,fsync,argc,argv)
{
#ifdef HAVE_FSYNC
  long fd;
  int res;
  if (argc != 1 || !isint(argv[0], &fd))
    return __FAIL;
  release_lock();
  res = fsync(fd);
  acquire_lock();
  if (res)
    return __FAIL;
  else
    return mkvoid;
#else
  return __FAIL;
#endif
}

FUNCTION(system,fdatasync,argc,argv)
{
#ifdef HAVE_FDATASYNC
  long fd;
  int res;
  if (argc != 1 || !isint(argv[0], &fd))
    return __FAIL;
  release_lock();
  res = fdatasync(fd);
  acquire_lock();
  if (res)
    return __FAIL;
  else
    return mkvoid;
#else
  return __FAIL;
#endif
}

FUNCTION(system,bread,argc,argv)
{
  FILE *fp;
  long fd, count, res;
  void *v = NULL, *v1;
  if (argc != 2 || !isint(argv[1], &count) || count < 0)
    return __FAIL;
  if (isint(argv[0], &fd)) {
    if (count && !(v = malloc(count)))
      return __ERROR;
    release_lock();
    res = read(fd, v, count);
    acquire_lock();
  } else if (isfile(argv[0], &fp)) {
    if (count && !(v = malloc(count)))
      return __ERROR;
    release_lock();
    res = fread(v, 1, count, fp);
    acquire_lock();
    if (res == 0 && ferror(fp)) res = -1;
  } else
    return __FAIL;
  if (res < 0) {
    free(v);
    return __FAIL;
  } else {
    if (res != count)
      if (res == 0) {
	free(v);
	v = NULL;
      } else if ((v1 = realloc(v, res)))
	v = v1;
    return mkbstr(res, v);
  }
}

FUNCTION(system,bwrite,argc,argv)
{
  FILE *fp;
  long fd, res;
  bstr_t *m;
  if (argc != 2 || !isobj(argv[1], type(ByteStr), (void**)&m))
    return __FAIL;
  if (isint(argv[0], &fd)) {
    if (!m->v)
      return mkint(0);
    else {
      release_lock();
      res = write(fd, m->v, m->size);
      acquire_lock();
    }
  } else if (isfile(argv[0], &fp)) {
    if (!m->v)
      return mkint(0);
    else {
      release_lock();
      res = fwrite(m->v, 1, m->size, fp);
      acquire_lock();
    }
    if (res == 0 && ferror(fp)) res = -1;
  } else
    return __FAIL;
  if (res < 0)
    return __FAIL;
  else
    return mkint(res);
}

FUNCTION(system,lseek,argc,argv)
{
  long fd, pos, whence;
  if (argc == 3 && isint(argv[0], &fd) &&
      isint(argv[1], &pos) && isint(argv[2], &whence)) {
    off_t res = lseek(fd, pos, whence);
    if (res == (off_t)-1)
      return __FAIL;
    else
      return mkint((long)res);
  } else
    return __FAIL;
}

FUNCTION(system,fcntl,argc,argv)
{
#ifdef HAVE_FCNTL_H
  long fd, cmd, arg, flags, type, pos, len, whence = SEEK_SET;
  struct flock lock;
  int n;
  expr *xs;
  if (argc != 3 || !isint(argv[0], &fd) || !isint(argv[1], &cmd))
    return __FAIL;
  switch (cmd) {
  case F_DUPFD:
    if (!isint(argv[2], &arg)) return __FAIL;
    fd = fcntl(fd, cmd, arg);
    if (fd >= 0)
      return mkint(fd);
    else
      return __FAIL;
  case F_GETFD:
  case F_GETFL:
    flags = fcntl(fd, cmd);
    if (flags >= 0)
      return mkint(flags);
    else
      return __FAIL;
  case F_SETFD:
  case F_SETFL:
    if (!isint(argv[2], &flags)) return __FAIL;
    if (!fcntl(fd, cmd, flags))
      return mkvoid;
    else
      return __FAIL;
  case F_GETLK:
  case F_SETLK:
  case F_SETLKW:
    if (!istuple(argv[2], &n, &xs) || n < 3 || n > 4 ||
	!isint(xs[0], &type) || !isint(xs[1], &pos) || !isint(xs[2], &len) ||
	n == 4 && !isint(xs[3], &whence))
      return __FAIL;
    lock.l_type = type;
    lock.l_whence = whence;
    lock.l_start = pos;
    lock.l_len = len;
    lock.l_pid = getpid();
    if (!fcntl(fd, cmd, &lock))
      if (cmd==F_GETLK)
	if (n == 3)
	  return mktuplel(4, mkint(lock.l_type),
			  mkint(lock.l_start),
			  mkint(lock.l_len),
			  mkint(lock.l_pid));
	else
	  return mktuplel(5, mkint(lock.l_type),
			  mkint(lock.l_start),
			  mkint(lock.l_len),
			  mkint(lock.l_whence),
			  mkint(lock.l_pid));
      else
	return mkvoid;
    else
      return __FAIL;
  default:
    return __FAIL;
  }
#else
  return __FAIL;
#endif
}

#ifdef HAVE_SELECT
static int getfds(expr x, fd_set *set)
{
  expr hd, tl;
  FILE *fp;
  long i;
  int n = -1;
  FD_ZERO(set);
  while (iscons(x, &hd, &tl)) {
    int fd;
    if (isfile(hd, &fp))
      fd = fileno(fp);
    else if (isint(hd, &i))
      fd = (int)i;
    else
      break;
    if (fd > n) n = fd;
    FD_SET(fd, set);
    x = tl;
  }
  if (isnil(x))
    return n+1;
  else
    return -1;
}

static expr listfds(expr x, fd_set *set)
{
  expr *ys, y, hd, tl;
  FILE *fp;
  long i;
  int n;
  for (n = 0, y = x; iscons(y, &hd, &tl); y = tl)
    if (isfile(hd, &fp) && FD_ISSET(fileno(fp), set) ||
	isint(hd, &i) && FD_ISSET((int)i, set)) n++;
  if (!(ys = malloc(n*sizeof(expr))))
    return NULL;
  for (n = 0, y = x; iscons(y, &hd, &tl); y = tl)
    if (isfile(hd, &fp) && FD_ISSET(fileno(fp), set) ||
	isint(hd, &i) && FD_ISSET((int)i, set))
      ys[n++] = hd;
  return mklistv(n, ys);
}
#endif

FUNCTION(system,select,argc,argv)
{
#ifdef HAVE_SELECT
  long ti;
  double tf;
  struct timeval tv, *timeout = NULL;
  fd_set in, out, err;
  int n, n_in, n_out, n_err;
  expr *xs, x;
  if (argc != 1 || !istuple(argv[0], &n, &xs) || (n != 3 && n != 4) ||
      (n_in = getfds(xs[0], &in)) < 0 || (n_out = getfds(xs[1], &out)) < 0 ||
      (n_err = getfds(xs[2], &err)) < 0)
    return __FAIL;
  if (n == 4)
    /* decode timeout argument */
    if (isint(xs[3], &ti))
      if (ti >= 0) {
	tv.tv_sec = ti;
	tv.tv_usec = 0;
	timeout = &tv;
      } else
	return __FAIL;
    else if (isfloat(xs[3], &tf))
      if (tf >= 0.0) {
	double ip, fp;
	if (tf > LONG_MAX) tf = LONG_MAX;
	fp = modf(tf, &ip);
	tv.tv_sec = (unsigned long)ip;
	tv.tv_usec = (unsigned long)(fp*1e6);
	timeout = &tv;
      } else
	return __FAIL;
    else
      return __FAIL;
  n = n_in;
  if (n_out > n) n = n_out;
  if (n_err > n) n = n_err;
  release_lock();
  n = select(n, &in, &out, &err, timeout);
  acquire_lock();
  if (n >= 0) {
    expr x = mktuplel(3, listfds(xs[0], &in), listfds(xs[1], &out),
		      listfds(xs[2], &err));
    return x?x:__ERROR;
  } else
    return __FAIL;
#else
  return __FAIL;
#endif
}

FUNCTION(system,isatty,argc,argv)
{
  long fd;
  if (argc != 1 || !isint(argv[0], &fd))
    return __FAIL;
  if (isatty(fd))
    return mktrue;
  else
    return mkfalse;
}

FUNCTION(system,ttyname,argc,argv)
{
#ifdef HAVE_TTYNAME
  long fd;
  char *name;
  if (argc != 1 || !isint(argv[0], &fd))
    return __FAIL;
  name = ttyname(fd);
  if (name)
    return mkstr(sys_to_utf8(name));
  else
    return __FAIL;
#else
  return __FAIL;
#endif
}

FUNCTION(system,ctermid,argc,argv)
{
#ifdef HAVE_CTERMID
  char *name;
  if (argc != 0)
    return __FAIL;
  name = ctermid(NULL);
  if (name)
    return mkstr(sys_to_utf8(name));
  else
    return __FAIL;
#else
  return __FAIL;
#endif
}

#if (!defined(HAVE_OPENPTY) || !defined(HAVE_FORKPTY)) && defined(HAVE_GRANTPT) && !defined(HAVE_GETPT)
static int getpt(void)
{
  return open("/dev/ptmx", O_RDWR);
}
#endif

#if !defined(HAVE_OPENPTY) && defined(HAVE_GRANTPT)
static int open_pty(int *amaster, int *aslave)
{
  int master, slave;
  char *name;
     
  master = getpt();
  if (master < 0)
    return -1;
     
  if (grantpt(master) < 0 || unlockpt(master) < 0)
    goto close_master;
  name = ptsname (master);
  if (name == NULL)
    goto close_master;
     
  slave = open(name, O_RDWR);
  if (slave == -1)
    goto close_master;
     
  if (isastream(slave)) {
    if (ioctl(slave, I_PUSH, "ptem") < 0 ||
	ioctl(slave, I_PUSH, "ldterm") < 0)
      goto close_slave;
  }
     
  *amaster = master;
  *aslave = slave;
  return 0;
     
 close_slave:
  close(slave);
     
 close_master:
  close(master);
  return -1;
}
#endif

#if !defined(HAVE_FORKPTY) && defined(HAVE_GRANTPT)
static int fork_pty(int *amaster)
{
  int master, slave, pid;
  char *name;

  if ((master = getpt()) < 0)
    return -1;

  if (grantpt(master) < 0 || unlockpt(master) < 0)
    goto close_master;
  name = ptsname (master);
  if (name == NULL)
    goto close_master;
     
  if ((pid = fork()) < 0)
    goto close_master;

  if (pid == 0) {
    if (setsid() < 0)
      exit(1);
    /* under SVR3 the following open() will acquire the controlling terminal */
    if ((slave = open(name, O_RDWR)) == -1)
      exit(1);
    if (isastream(slave)) {
      if (ioctl(slave, I_PUSH, "ptem") < 0 ||
	  ioctl(slave, I_PUSH, "ldterm") < 0)
	exit(1);
    }
    close(master);
    if (dup2(slave, STDIN_FILENO) != STDIN_FILENO ||
	dup2(slave, STDOUT_FILENO) != STDOUT_FILENO ||
	dup2(slave, STDERR_FILENO) != STDERR_FILENO)
      exit(1);
    if (slave > STDERR_FILENO)
      close(slave);
    *amaster = master;
    return 0;
  } else {
    *amaster = master;
    return pid;
  }
     
 close_slave:
  close(slave);
     
 close_master:
  close(master);
  return -1;
}
#endif

FUNCTION(system,openpty,argc,argv)
{
#if defined(HAVE_OPENPTY) || defined(HAVE_GRANTPT)
  int master, slave;
  if (argc != 0 ||
#ifdef HAVE_OPENPTY
      openpty(&master, &slave, NULL, NULL, NULL)
#else
      open_pty(&master, &slave)
#endif
      )
    return __FAIL;
  else
    return mktuplel(2, mkint(master), mkint(slave));
#else
  return __FAIL;
#endif
}

FUNCTION(system,forkpty,argc,argv)
{
#if defined(HAVE_FORKPTY) || defined(HAVE_GRANTPT)
  int pid, master;
  if (argc != 0 || (pid =
#ifdef HAVE_FORKPTY
		    forkpty(&master, NULL, NULL, NULL)
#else
		    fork_pty(&master)
#endif
		    ) < 0)
    return __FAIL;
  else
    return mktuplel(2, mkint(pid), mkint(master));
#else
  return __FAIL;
#endif
}

/* termios interface ********************************************************/

#ifdef HAVE_TERMIOS_H
static expr mkcharlist(cc_t *cc)
{
  int i;
  expr *xv = malloc(NCCS*sizeof(expr));
  if (!xv) return NULL;
  for (i = 0; i < NCCS; i++) {
    if (!(xv[i] = mkint((long)cc[i]))) {
      while (i > 0) dispose(xv[--i]);
      free(xv);
      return NULL;
    }
  }
  return mklistv(NCCS, xv);
}

static expr decode_termios_val(struct termios *attr)
{
  speed_t ispeed = cfgetispeed(attr), ospeed = cfgetospeed(attr);
  expr x = mktuplel(7, mkint(attr->c_iflag), mkint(attr->c_oflag),
		    mkint(attr->c_cflag), mkint(attr->c_lflag),
		    mkint(ispeed), mkint(ospeed),
		    mkcharlist(attr->c_cc));
  if (x)
    return x;
  else
    return __ERROR;
}

static int ischarlist(expr v, cc_t *cc)
{
  int n;
  expr x, hd, tl;
  long c;
  for (n = 0, x = v; iscons(x, &hd, &tl) && isint(hd, &c); x = tl)
    n++;
  if (!isnil(x) || n != NCCS) return 0;
  for (n = 0, x = v; iscons(x, &hd, &tl) && isint(hd, &c); x = tl)
    cc[n++] = (cc_t)c;
  return 1;
}

static struct termios *encode_termios_val(int fd, expr x)
{
  static struct termios attr;
  int n;
  expr *xv;
  long iflag, oflag, cflag, lflag, ispeed, ospeed;
  /* to be safe, we first fill the structure with the current values, since an
     implementation might have more fields than those required by POSIX */
  tcgetattr(fd, &attr);
  if (istuple(x, &n, &xv) && n == 7 &&
      isint(xv[0], &iflag) && isint(xv[1], &oflag) &&
      isint(xv[2], &cflag) && isint(xv[3], &lflag) &&
      isint(xv[4], &ispeed) && isint(xv[5], &ospeed) &&
      ischarlist(xv[6], attr.c_cc)) {
    attr.c_iflag = iflag; attr.c_oflag = oflag;
    attr.c_cflag = cflag; attr.c_lflag = lflag;
    cfsetispeed(&attr, ispeed);
    cfsetospeed(&attr, ospeed);
    return &attr;
  } else
    return NULL;
}
#endif

FUNCTION(system,tcgetattr,argc,argv)
{
#ifdef HAVE_TERMIOS_H
  struct termios attr;
  long fd;
  if (argc == 1 && isint(argv[0], &fd) &&
      !tcgetattr(fd, &attr))
    return decode_termios_val(&attr);
  else
#endif
    return __FAIL;
}

FUNCTION(system,tcsetattr,argc,argv)
{
#ifdef HAVE_TERMIOS_H
  struct termios *attr;
  long fd, when;
  if (argc == 3 && isint(argv[0], &fd) && isint(argv[1], &when) &&
      (attr = encode_termios_val(fd, argv[2])) &&
      !tcsetattr(fd, when, attr))
    return mkvoid;
  else
#endif
    return __FAIL;
}

FUNCTION(system,tcsendbreak,argc,argv)
{
#ifdef HAVE_TERMIOS_H
  long fd, duration;
  if (argc == 2 && isint(argv[0], &fd) && isint(argv[1], &duration) &&
      !tcsendbreak(fd, duration))
    return mkvoid;
  else
#endif
    return __FAIL;
}

FUNCTION(system,tcdrain,argc,argv)
{
#ifdef HAVE_TERMIOS_H
  long fd;
  if (argc == 1 && isint(argv[0], &fd) && !tcdrain(fd))
    return mkvoid;
  else
#endif
    return __FAIL;
}

FUNCTION(system,tcflush,argc,argv)
{
#ifdef HAVE_TERMIOS_H
  long fd, queue;
  if (argc == 2 && isint(argv[0], &fd) && isint(argv[1], &queue) &&
      !tcflush(fd, queue))
    return mkvoid;
  else
#endif
    return __FAIL;
}

FUNCTION(system,tcflow,argc,argv)
{
#ifdef HAVE_TERMIOS_H
  long fd, action;
  if (argc == 2 && isint(argv[0], &fd) && isint(argv[1], &action) &&
      !tcflow(fd, action))
    return mkvoid;
  else
#endif
    return __FAIL;
}

FUNCTION(system,tcgetpgrp,argc,argv)
{
#ifdef HAVE_TCGETPGRP
  long fd, pgrp;
  if (argc != 1 || !isint(argv[0], &fd))
    return __FAIL;
  pgrp = tcgetpgrp(fd);
  if (pgrp < 0)
    return __FAIL;
  else
    return mkint(pgrp);
#else
  return __FAIL;
#endif
}

FUNCTION(system,tcsetpgrp,argc,argv)
{
#ifdef HAVE_TCSETPGRP
  long fd, pgrp;
  if (argc != 2 || !isint(argv[0], &fd) || !isint(argv[1], &pgrp))
    return __FAIL;
  if (tcsetpgrp(fd, pgrp))
    return __FAIL;
  else
    return mkvoid;
#else
  return __FAIL;
#endif
}

/* socket interface: ******************************************************/

FUNCTION(system,socket,argc,argv)
{
#ifdef HAVE_SOCKET
  long domain, type, proto;
  int fd;
  if (argc != 3 || !isint(argv[0], &domain) || !isint(argv[1], &type) ||
      !isint(argv[2], &proto))
    return __FAIL;
  if ((fd = socket(domain, type, proto)) < 0)
    return __FAIL;
  else
    return mkint(fd);
#else
  return __FAIL;
#endif
}

FUNCTION(system,socketpair,argc,argv)
{
#ifdef HAVE_SOCKETPAIR
  long domain, type, proto;
  int fd[2];
  if (argc != 3 || !isint(argv[0], &domain) || !isint(argv[1], &type) ||
      !isint(argv[2], &proto))
    return __FAIL;
  if (socketpair(domain, type, proto, fd))
    return __FAIL;
  else
    return mktuplel(2, mkint(fd[0]), mkint(fd[1]));
#else
  return __FAIL;
#endif
}

FUNCTION(system,shutdown,argc,argv)
{
#ifdef HAVE_SHUTDOWN
  long fd, how;
  if (argc != 2 || !isint(argv[0], &fd) || !isint(argv[1], &how))
    return __FAIL;
  if (shutdown(fd, how))
    return __FAIL;
  else
    return mkvoid;
#else
  return __FAIL;
#endif
}

FUNCTION(system,closesocket,argc,argv)
{
#ifdef HAVE_BSD_SOCKETS
  long fd;
  if (argc != 1 || !isint(argv[0], &fd))
    return __FAIL;
#ifdef HAVE_CLOSESOCKET
  if (closesocket(fd))
#else
  if (close(fd))
#endif
    return __FAIL;
  else
    return mkvoid;
#else
  return __FAIL;
#endif
}

#ifdef HAVE_BSD_SOCKETS

/* FIXME: is this large enough for any type of address? */
#define SOCKADDR_SIZE 1024

static int isaddr(expr x, long *domain, char **name, long *port,
		  unsigned long *flowinfo, unsigned long *scopeid)
{
  int n;
  expr *xv;
#ifdef AF_LOCAL
 if (isstr(x, name)) {
    *domain = AF_LOCAL;
    return 1;
  } else
#endif
  if (!istuple(x, &n, &xv))
    return 0;
  else if (n == 2)
    if (isstr(xv[0], name) && isint(xv[1], port)) {
      *domain = AF_INET;
      return 1;
    } else
      return 0;
#ifdef HAVE_IPV6
  else if (n == 4)
    if (isstr(xv[0], name) && isint(xv[1], port) &&
	isuint(xv[2], flowinfo) && isuint(xv[3], scopeid)) {
      *domain = AF_INET6;
      return 1;
    } else
      return 0;
#endif
  else
    return 0;
}

static struct sockaddr*
encode_addr(long domain, char *name,
	    long port,
	    unsigned long flowinfo, unsigned long scopeid,
#ifdef AF_LOCAL
	    struct sockaddr_un *sa_un,
#endif
	    struct sockaddr_in *sa_in,
#ifdef HAVE_IPV6
	    struct sockaddr_in6 *sa_in6,
#endif
	    int *len)
{
  int l;
  name = utf8_to_sys(name);
  if (!name) return NULL;
  l = strlen(name);
#ifdef AF_LOCAL
  if (domain == AF_LOCAL) {
    memset(sa_un, 0, sizeof(struct sockaddr_un));
    sa_un->sun_family = AF_LOCAL;
    strncpy(sa_un->sun_path, name, sizeof(sa_un->sun_path));
    if (l >= sizeof(sa_un->sun_path))
      sa_un->sun_path[sizeof(sa_un->sun_path)-1] = 0;
    *len = sizeof(sa_un->sun_family) + strlen(sa_un->sun_path) + 1;
    free(name);
    return (struct sockaddr*)sa_un;
  } else
#endif
  if (domain == AF_INET) {
    if (port < 0 || port > 0xffff) {
      free(name);
      return NULL;
    }
    memset(sa_in, 0, sizeof(struct sockaddr_in));
    /* first check for numeric addresses in standard dot notation */
#ifdef HAVE_INET_ATON
    /* inet_aton is the recommended interface, but it only seems to be
       available on BSDish systems */
    if (!inet_aton(name, &sa_in->sin_addr)) {
#else
    /* otherwise we just employ inet_addr which should be available anyway */
    if ((sa_in->sin_addr.s_addr = inet_addr(name)) == INADDR_NONE) {
#endif
      /* if the host wasn't found that way, it's probably a symbolic name;
	 query the host database for it */
      struct hostent *h = gethostbyname(name);
      if (!h || h->h_addrtype != AF_INET) {
	free(name);
	return NULL;
      }
      memcpy(&sa_in->sin_addr, h->h_addr, sizeof(struct in_addr));
    }
    sa_in->sin_family = AF_INET;
    sa_in->sin_port = htons((uint16_t)port);
    *len = sizeof(struct sockaddr_in);
    free(name);
    return (struct sockaddr*)sa_in;
#ifdef HAVE_IPV6
  } else if (domain == AF_INET6) {
    if (port < 0 || port > 0xffff) {
      free(name);
      return NULL;
    }
    memset(sa_in6, 0, sizeof(struct sockaddr_in6));
    if (!inet_pton(AF_INET6, name, &sa_in6->sin6_addr)) {
      struct hostent *h = gethostbyname(name);
      if (!h || h->h_addrtype != AF_INET6) {
	free(name);
	return NULL;
      }
      memcpy(&sa_in6->sin6_addr, h->h_addr, sizeof(struct in6_addr));
    }
    sa_in6->sin6_family = AF_INET6;
    sa_in6->sin6_port = htons((uint16_t)port);
    sa_in6->sin6_flowinfo = htonl((uint32_t)flowinfo);
    sa_in6->sin6_scope_id = htonl((uint32_t)scopeid);
    *len = sizeof(struct sockaddr_in6);
    free(name);
    return (struct sockaddr*)sa_in6;
#endif
  } else {
    free(name);
    return NULL;
  }
}

static expr decode_addr(struct sockaddr *sa, int len)
{
#ifdef AF_LOCAL
  if (sa->sa_family == AF_LOCAL) {
    struct sockaddr_un *sa_un = (struct sockaddr_un*)sa;
    return mkstr(sys_to_utf8(sa_un->sun_path));
  } else
#endif
  if (sa->sa_family == AF_INET) {
    struct sockaddr_in *sa_in = (struct sockaddr_in*)sa;
    char *name = inet_ntoa(sa_in->sin_addr);
    return mktuplel(2, mkstr(sys_to_utf8(name)),
		    mkuint((unsigned long)ntohs(sa_in->sin_port)));
#ifdef HAVE_IPV6
  } else if (sa->sa_family == AF_INET6) {
    struct sockaddr_in6 *sa_in6 = (struct sockaddr_in6*)sa;
    char buf[BUFSZ];
    const char *name;
    name = inet_ntop(AF_INET6, &sa_in6->sin6_addr, buf, BUFSZ);
    if (name)
      return mktuplel(4, mkstr(sys_to_utf8(name)),
		      mkuint((unsigned long)ntohs(sa_in6->sin6_port)),
		      mkuint((unsigned long)ntohl(sa_in6->sin6_flowinfo)),
		      mkuint((unsigned long)ntohl(sa_in6->sin6_scope_id)));
    else
      return NULL;
#endif
  } else
    return NULL;
}
#endif

FUNCTION(system,bind,argc,argv)
{
#ifdef HAVE_BIND
  long fd, domain, port;
  unsigned long flowinfo, scopeid;
  char *name;
  struct sockaddr *sa;
#ifdef AF_LOCAL
  struct sockaddr_un sa_un;
#endif
  struct sockaddr_in sa_in;
#ifdef HAVE_IPV6
  struct sockaddr_in6 sa_in6;
#endif
  int len;
  if (argc != 2 || !isint(argv[0], &fd) ||
      !isaddr(argv[1], &domain, &name, &port, &flowinfo, &scopeid))
    return __FAIL;
  release_lock();
  sa = encode_addr(domain, name, port, flowinfo, scopeid,
#ifdef AF_LOCAL
		   &sa_un,
#endif
		   &sa_in,
#ifdef HAVE_IPV6
		   &sa_in6,
#endif
		   &len);
  acquire_lock();
  if (!sa)
    return __FAIL;
  if (bind(fd, sa, len))
    return __FAIL;
  else
    return mkvoid;
#else
  return __FAIL;
#endif
}

FUNCTION(system,listen,argc,argv)
{
#ifdef HAVE_LISTEN
  long fd, n;
  if (argc != 2 || !isint(argv[0], &fd) || !isint(argv[1], &n))
    return __FAIL;
  if (listen(fd, n))
    return __FAIL;
  else
    return mkvoid;
#else
  return __FAIL;
#endif
}

FUNCTION(system,accept,argc,argv)
{
#ifdef HAVE_ACCEPT
  long fd;
  struct sockaddr *sa;
  socklen_t len;
  expr x;
  if (argc != 1 || !isint(argv[0], &fd))
    return __FAIL;
  if (!(sa = malloc(SOCKADDR_SIZE)))
    return __ERROR;
  release_lock();
  fd = accept(fd, sa, &len);
  acquire_lock();
  if (fd < 0)
    return __FAIL;
  else if (!(x = decode_addr(sa, len))) {
    close(fd);
    return NULL;
  } else
    return mktuplel(2, mkint(fd), x);
#else
  return __FAIL;
#endif
}

FUNCTION(system,connect,argc,argv)
{
#ifdef HAVE_CONNECT
  long fd, domain, port;
  unsigned long flowinfo, scopeid;
  char *name;
  struct sockaddr *sa;
#ifdef AF_LOCAL
  struct sockaddr_un sa_un;
#endif
  struct sockaddr_in sa_in;
#ifdef HAVE_IPV6
  struct sockaddr_in6 sa_in6;
#endif
  int len, res;
  if (argc != 2 || !isint(argv[0], &fd) ||
      !isaddr(argv[1], &domain, &name, &port, &flowinfo, &scopeid))
    return __FAIL;
  release_lock();
  sa = encode_addr(domain, name, port, flowinfo, scopeid,
#ifdef AF_LOCAL
		   &sa_un,
#endif
		   &sa_in,
#ifdef HAVE_IPV6
		   &sa_in6,
#endif
		   &len);
  if (!sa) {
    acquire_lock();
    return __FAIL;
  }
  res = connect(fd, sa, len);
  acquire_lock();
  if (res)
    return __FAIL;
  else
    return mkvoid;
#else
  return __FAIL;
#endif
}

FUNCTION(system,getsockname,argc,argv)
{
#ifdef HAVE_GETSOCKNAME
  long fd;
  int res;
  struct sockaddr *sa;
  socklen_t len = SOCKADDR_SIZE;
  expr x;
  if (argc != 1 || !isint(argv[0], &fd))
    return __FAIL;
  if (!(sa = malloc(SOCKADDR_SIZE)))
    return __ERROR;
  release_lock();
  res = getsockname(fd, sa, &len);
  acquire_lock();
  x = decode_addr(sa, len);
  free(sa);
  if (res)
    return __FAIL;
  else
    return x;
#else
  return __FAIL;
#endif
}

FUNCTION(system,getpeername,argc,argv)
{
#ifdef HAVE_GETPEERNAME
  long fd;
  int res;
  struct sockaddr *sa;
  socklen_t len = SOCKADDR_SIZE;
  expr x;
  if (argc != 1 || !isint(argv[0], &fd))
    return __FAIL;
  if (!(sa = malloc(SOCKADDR_SIZE)))
    return __ERROR;
  release_lock();
  res = getpeername(fd, sa, &len);
  acquire_lock();
  x = decode_addr(sa, len);
  free(sa);
  if (res)
    return __FAIL;
  else
    return x;
#else
  return __FAIL;
#endif
}

FUNCTION(system,getsockopt,argc,argv)
{
#ifdef HAVE_GETSOCKOPT
  long fd, level, opt;
  socklen_t len = BUFSZ;
  void *v;
  if (argc != 3 || !isint(argv[0], &fd) || !isint(argv[1], &level) ||
      !isint(argv[2], &opt))
    return __FAIL;
  if (!(v = malloc(BUFSZ)))
    return __ERROR;
  if (getsockopt(fd, level, opt, v, &len))
    return __FAIL;
  else if (len == 0) {
    free(v);
    return mkbstr(0, NULL);
  } else {
    void *v1 = realloc(v, len);
    if (v1) v = v1;
    return mkbstr(len, v);
  }
#else
  return __FAIL;
#endif
}

FUNCTION(system,setsockopt,argc,argv)
{
#ifdef HAVE_SETSOCKOPT
  long fd, level, opt;
  bstr_t *val;
  if (argc != 4 || !isint(argv[0], &fd) || !isint(argv[1], &level) ||
      !isint(argv[2], &opt) || !isobj(argv[3], type(ByteStr), (void**)&val))
    return __FAIL;
  if (setsockopt(fd, level, opt, val->v, val->size))
    return __FAIL;
  else
    return mkvoid;
#else
  return __FAIL;
#endif
}

FUNCTION(system,recv,argc,argv)
{
#ifdef HAVE_RECV
  long fd, flags, size;
  void *v;
  int res;
  if (argc != 3 || !isint(argv[0], &fd) || !isint(argv[1], &flags) ||
      !isint(argv[2], &size) || size < 0)
    return __FAIL;
  if (!(v = malloc(size)))
    return __ERROR;
  release_lock();
  res = recv(fd, v, size, flags);
  acquire_lock();
  if (res < 0) {
    free(v);
    return __FAIL;
  } else if (res == 0) {
    free(v);
    return mkbstr(0, NULL);
  } else {
    void *v1 = realloc(v, res);
    if (v1) v = v1;
    return mkbstr(res, v);
  }
#else
  return __FAIL;
#endif
}

FUNCTION(system,send,argc,argv)
{
#ifdef HAVE_SEND
  long fd, flags;
  int res;
  bstr_t *m;
  if (argc != 3 || !isint(argv[0], &fd) || !isint(argv[1], &flags) ||
      !isobj(argv[2], type(ByteStr), (void**)&m))
    return __FAIL;
  release_lock();
  res = send(fd, m->v, m->size, flags);
  acquire_lock();
  if (res < 0)
    return __FAIL;
  else
    return mkint(res);
#else
  return __FAIL;
#endif
}

FUNCTION(system,recvfrom,argc,argv)
{
#ifdef HAVE_RECVFROM
  long fd, flags, size;
  void *v;
  int res;
  struct sockaddr *sa;
  socklen_t len = SOCKADDR_SIZE;
  expr x;
  if (argc != 3 || !isint(argv[0], &fd) || !isint(argv[1], &flags) ||
      !isint(argv[2], &size) || size < 0)
    return __FAIL;
  if (!(v = malloc(size)))
    return __ERROR;
  if (!(sa = malloc(SOCKADDR_SIZE)))
    return __ERROR;
  release_lock();
  res = recvfrom(fd, v, size, flags, sa, &len);
  acquire_lock();
  x = decode_addr(sa, len);
  free(sa);
  if (res < 0) {
    free(v);
    return __FAIL;
  } else if (res == 0) {
    free(v);
    v = NULL;
  } else {
    void *v1 = realloc(v, res);
    if (v1) v = v1;
  }
  if (x)
    return mktuplel(2, x, mkbstr(res, v));
  else
    return mkbstr(res, v);
#else
  return __FAIL;
#endif
}

FUNCTION(system,sendto,argc,argv)
{
#ifdef HAVE_SENDTO
  long fd, flags, domain, port;
  unsigned long flowinfo, scopeid;
  char *name;
  struct sockaddr *sa;
#ifdef AF_LOCAL
  struct sockaddr_un sa_un;
#endif
  struct sockaddr_in sa_in;
#ifdef HAVE_IPV6
  struct sockaddr_in6 sa_in6;
#endif
  int n, len, res;
  bstr_t *m;
  expr *xv;
  if (argc != 3 || !isint(argv[0], &fd) || !isint(argv[1], &flags) ||
      !istuple(argv[2], &n, &xv) || n != 2 ||
      !isaddr(xv[0], &domain, &name, &port, &flowinfo, &scopeid) ||
      !isobj(xv[1], type(ByteStr), (void**)&m))
    return __FAIL;
  release_lock();
  sa = encode_addr(domain, name, port, flowinfo, scopeid,
#ifdef AF_LOCAL
		   &sa_un,
#endif
		   &sa_in,
#ifdef HAVE_IPV6
		   &sa_in6,
#endif
		   &len);
  if (!sa) {
    acquire_lock();
    return __FAIL;
  }
  res = sendto(fd, m->v, m->size, flags, sa, len);
  acquire_lock();
  if (res < 0)
    return __FAIL;
  else
    return mkint(res);
#else
  return __FAIL;
#endif
}

static expr mkstrlist(char **l)
{
  int i, n;
  expr *xv;
  for (i = 0; l[i]; i++) ;
  n = i;
  if (!(xv = malloc(n*sizeof(expr))))
    return __ERROR;
  for (i = 0; i < n; i++)
    xv[i] = mkstr(sys_to_utf8(l[i]));
  return mklistv(n, xv);
}

/* net database functions ***************************************************/

#if defined(HAVE_GETHOSTBYNAME) || defined(HAVE_GETHOSTENT)
static expr mkaddrlist(int addrtype, char **l)
{
  int i, n;
  expr *xv;
    if (addrtype != AF_INET
#ifdef HAVE_IPV6
	&& addrtype != AF_INET6
#endif
	)
      return NULL;
  for (i = 0; l[i]; i++) ;
  n = i;
  if (!(xv = malloc(n*sizeof(expr))))
    return __ERROR;
  for (i = 0; i < n; i++) {
    const char *s;
    if (addrtype == AF_INET) {
      struct in_addr in;
      memcpy(&in, l[i], sizeof(struct in_addr));
      s = inet_ntoa(in);
#ifdef HAVE_IPV6
    } else if (addrtype == AF_INET6) {
      char buf[BUFSZ];
      s = inet_ntop(addrtype, l[i], buf, BUFSZ);
#endif
    } else
      s = NULL;
    xv[i] = mkstr(sys_to_utf8(s));
  }
  return mklistv(n, xv);
}
#endif

FUNCTION(system,gethostbyname,argc,argv)
{
#ifdef HAVE_GETHOSTBYNAME
  char *name;
  struct hostent *h;
  if (argc == 1 && isstr(argv[0], &name)) {
    if (!(name = utf8_to_sys(name)))
      return __ERROR;
    h = gethostbyname(name);
    free(name);
    if (h)
      return mktuplel(4, mkstr(sys_to_utf8(h->h_name)),
		      mkstrlist(h->h_aliases),
		      mkint(h->h_addrtype),
		      mkaddrlist(h->h_addrtype, h->h_addr_list));
    else
      return __FAIL;
 } else
#endif
    return __FAIL;
}

FUNCTION(system,gethostbyaddr,argc,argv)
{
#ifdef HAVE_GETHOSTBYADDR
  char *cp, *addr = NULL;
  int len, type;
  struct hostent *h;
#ifdef AF_INET
  struct in_addr sin_addr;
#endif
#ifdef HAVE_IPV6
  struct in6_addr sin6_addr;
#endif
  if (argc != 1 || !isstr(argv[0], &cp))
    return __FAIL;
  if (!(cp = utf8_to_sys(cp)))
    return __ERROR;
#ifdef AF_INET
#ifdef HAVE_INET_ATON
  if (!addr && inet_aton(cp, &sin_addr)) {
#else
  memset(&sin_addr, 0, sizeof(sin_addr));
  if (!addr && (sin_addr.s_addr = inet_addr(cp)) != INADDR_NONE) {
#endif
    addr = (char*)&sin_addr;
    len = sizeof(sin_addr);
    type = AF_INET;
  }
#endif
#ifdef HAVE_IPV6
  if (!addr && inet_pton(AF_INET6, cp, &sin6_addr)) {
    addr = (char*)&sin6_addr;
    len = sizeof(sin6_addr);
    type = AF_INET6;
  }
#endif
  free(cp);
  if (addr && (h = gethostbyaddr(addr, len, type)))
    return mktuplel(4, mkstr(sys_to_utf8(h->h_name)),
		    mkstrlist(h->h_aliases),
		    mkint(h->h_addrtype),
		    mkaddrlist(h->h_addrtype, h->h_addr_list));
  else
#endif
    return __FAIL;
}

FUNCTION(system,gethostent,argc,argv)
{
#ifdef HAVE_GETHOSTENT
  if (argc == 0) {
    struct hostent *h;
    int i, n;
    expr *xv;
    sethostent(1);
    /* count entries */
    for (n = 0, h = gethostent(); h; h = gethostent()) n++;
    /* rewind */
    endhostent();
    sethostent(1);
    /* allocate vector */
    if (!(xv = malloc(n*sizeof(expr))))
      return __ERROR;
    /* list entries */
    for (i = 0, h = gethostent(); h; h = gethostent())
      if (i < n &&
	  (xv[i] =
	   mktuplel(4, mkstr(sys_to_utf8(h->h_name)),
		    mkstrlist(h->h_aliases),
		    mkint(h->h_addrtype),
		    mkaddrlist(h->h_addrtype, h->h_addr_list))))
	i++;
      else {
	while (i > 0) dispose(xv[--i]);
	if (n > 0) free(xv);
	return (i>=n)?__FAIL:__ERROR;
      }
    /* close */
    endhostent();
    /* check number of items read */
    if (i < n) {
      /* read error */
      while (i > 0) dispose(xv[--i]);
      if (n > 0) free(xv);
      return __FAIL;
    }
    /* return the list */
    return mklistv(n, xv);
  } else
#endif
    return __FAIL;
}

FUNCTION(system,getprotobyname,argc,argv)
{
#ifdef HAVE_GETPROTOBYNAME
  char *name;
  struct protoent *p;
  if (argc == 1 && isstr(argv[0], &name)) {
    if (!(name = utf8_to_sys(name)))
      return __ERROR;
    p = getprotobyname(name);
    free(name);
    if (p)
      return mktuplel(3, mkstr(sys_to_utf8(p->p_name)),
		      mkstrlist(p->p_aliases),
		      mkint(p->p_proto));
    else
      return __FAIL;
  } else
#endif
    return __FAIL;
}

FUNCTION(system,getprotobynumber,argc,argv)
{
#ifdef HAVE_GETPROTOBYNUMBER
  long proto;
  struct protoent *p;
  if (argc == 1 && isint(argv[0], &proto) &&
      (p = getprotobynumber(proto)))
    return mktuplel(3, mkstr(sys_to_utf8(p->p_name)),
		    mkstrlist(p->p_aliases),
		    mkint(p->p_proto));
  else
#endif
    return __FAIL;
}

FUNCTION(system,getprotoent,argc,argv)
{
#ifdef HAVE_GETPROTOENT
  if (argc == 0) {
    struct protoent *p;
    int i, n;
    expr *xv;
    setprotoent(1);
    /* count entries */
    for (n = 0, p = getprotoent(); p; p = getprotoent()) n++;
    /* rewind */
    endprotoent();
    setprotoent(1);
    /* allocate vector */
    if (!(xv = malloc(n*sizeof(expr))))
      return __ERROR;
    /* list entries */
    for (i = 0, p = getprotoent(); p; p = getprotoent())
      if (i < n &&
	  (xv[i] =
	   mktuplel(3, mkstr(sys_to_utf8(p->p_name)),
		    mkstrlist(p->p_aliases),
		    mkint(p->p_proto))))
	i++;
      else {
	while (i > 0) dispose(xv[--i]);
	if (n > 0) free(xv);
	return (i>=n)?__FAIL:__ERROR;
      }
    /* close */
    endprotoent();
    /* check number of items read */
    if (i < n) {
      /* read error */
      while (i > 0) dispose(xv[--i]);
      if (n > 0) free(xv);
      return __FAIL;
    }
    /* return the list */
    return mklistv(n, xv);
  } else
#endif
    return __FAIL;
}

FUNCTION(system,getservbyname,argc,argv)
{
#ifdef HAVE_GETSERVBYNAME
  char *name, *proto;
  expr *xv;
  int n;
  struct servent *s;
  if (argc == 1) {
    if (isstr(argv[0], &name)) {
      if (!(name = utf8_to_sys(name)))
	return __ERROR;
      s = getservbyname(name, NULL);
      free(name);
    } else if (istuple(argv[0], &n, &xv) && n == 2 &&
	       isstr(xv[0], &name) &&
	       isstr(xv[1], &proto)) {
      if (!(name = utf8_to_sys(name)))
	return __ERROR;
      else if (!(proto = utf8_to_sys(proto))) {
	free(name);
	return __ERROR;
      }
      s = getservbyname(name, proto);
      free(name); free(proto);
    } else
      s = NULL;
    if (s)
      return mktuplel(4, mkstr(sys_to_utf8(s->s_name)),
		      mkstrlist(s->s_aliases),
		      mkint(s->s_port),
		      mkstr(sys_to_utf8(s->s_proto)));
    else
      return __FAIL;
  } else
#endif
    return __FAIL;
}

FUNCTION(system,getservbyport,argc,argv)
{
#ifdef HAVE_GETSERVBYPORT
  long port;
  char *proto;
  expr *xv;
  int n;
  struct servent *s;
  if (argc == 1) {
    if (isint(argv[0], &port))
      s = getservbyport(port, NULL);
    else if (istuple(argv[0], &n, &xv) && n == 2 &&
	     isint(xv[0], &port) &&
	     isstr(xv[1], &proto)) {
      if (!(proto = utf8_to_sys(proto)))
	return __ERROR;
      s = getservbyport(port, proto);
      free(proto);
    } else
      s = NULL;
    if (s)
      return mktuplel(4, mkstr(sys_to_utf8(s->s_name)),
		      mkstrlist(s->s_aliases),
		      mkint(s->s_port),
		      mkstr(sys_to_utf8(s->s_proto)));
    else
      return __FAIL;
  } else
#endif
    return __FAIL;
}

FUNCTION(system,getservent,argc,argv)
{
#ifdef HAVE_GETSERVENT
  if (argc == 0) {
    struct servent *s;
    int i, n;
    expr *xv;
    setservent(1);
    /* count entries */
    for (n = 0, s = getservent(); s; s = getservent()) n++;
    /* rewind */
    endservent();
    setservent(1);
    /* allocate vector */
    if (!(xv = malloc(n*sizeof(expr))))
      return __ERROR;
    /* list entries */
    for (i = 0, s = getservent(); s; s = getservent())
      if (i < n &&
	  (xv[i] =
	   mktuplel(4, mkstr(sys_to_utf8(s->s_name)),
		    mkstrlist(s->s_aliases),
		    mkint(s->s_port),
		    mkstr(sys_to_utf8(s->s_proto)))))
	i++;
      else {
	while (i > 0) dispose(xv[--i]);
	if (n > 0) free(xv);
	return (i>=n)?__FAIL:__ERROR;
      }
    /* close */
    endservent();
    /* check number of items read */
    if (i < n) {
      /* read error */
      while (i > 0) dispose(xv[--i]);
      if (n > 0) free(xv);
      return __FAIL;
    }
    /* return the list */
    return mklistv(n, xv);
  } else
#endif
    return __FAIL;
}

/* file and directory functions: ********************************************/

FUNCTION(system,rename,argc,argv)
{
  char *old, *new;
  if (argc == 2 && isstr(argv[0], &old) && isstr(argv[1], &new)) {
    int res;
    old = utf8_to_sys(old); new = utf8_to_sys(new);
    if (!old || !new) {
      if (old) free(old); if (new) free(new);
      return __ERROR;
    }
    res = rename(old, new);
    free(old); free(new);
    if (res)
      return __FAIL;
    else
      return mkvoid;
  } else
    return __FAIL;
}

FUNCTION(system,unlink,argc,argv)
{
  char *name;
  if (argc == 1 && isstr(argv[0], &name)) {
    int res;
    name = utf8_to_sys(name);
    if (!name) return __ERROR;
    res = unlink(name);
    free(name);
    if (res)
      return __FAIL;
    else
      return mkvoid;
  } else
    return __FAIL;
}

FUNCTION(system,truncate,argc,argv)
{
#ifdef HAVE_TRUNCATE
  char *name;
  long len;
  int res;
  if (argc != 2 || !isstr(argv[0], &name) || !isint(argv[1], &len))
    return __FAIL;
  name = utf8_to_sys(name);
  if (!name) return __ERROR;
  res = truncate(name, len);
  free(name);
  if (res)
    return __FAIL;
  else
    return mkvoid;
#else
  return __FAIL;
#endif
}

FUNCTION(system,getcwd,argc,argv)
{
  char pwd[BUFSZ];
  if (argc == 0 && getcwd(pwd, BUFSZ)) {
    char *s = sys_to_utf8(pwd);
    if (s)
      return mkstr(s);
    else
      return __ERROR;
  } else
    return __FAIL;
}

FUNCTION(system,chdir,argc,argv)
{
  char *name;
  if (argc == 1 && isstr(argv[0], &name)) {
    int res;
    name = utf8_to_sys(name);
    if (!name) return __ERROR;
    res = chdir(name);
    free(name);
    if (res)
      return __FAIL;
    else
      return mkvoid;
  } else
    return __FAIL;
}

FUNCTION(system,mkdir,argc,argv)
{
  char *name;
  long mode;
  if (argc == 2 && isstr(argv[0], &name) && isint(argv[1], &mode)) {
    int res;
    name = utf8_to_sys(name);
    if (!name) return __ERROR;
#ifdef WIN32
    /* mode argument is ignored */
    res = mkdir(name);
#else
    res = mkdir(name, mode);
#endif
    free(name);
    if (res)
      return __FAIL;
    else
      return mkvoid;
  } else
    return __FAIL;
}

FUNCTION(system,rmdir,argc,argv)
{
  char *name;
  if (argc == 1 && isstr(argv[0], &name)) {
    int res;
    name = utf8_to_sys(name);
    if (!name) return __ERROR;
    res = rmdir(name);
    free(name);
    if (res)
      return __FAIL;
    else
      return mkvoid;
  } else
    return __FAIL;
}

FUNCTION(system,readdir,argc,argv)
{
#ifdef HAVE_READDIR
  char *name;
  if (argc == 1 && isstr(argv[0], &name)) {
    DIR *dir;
    struct dirent *d;
    int i, n;
    expr *xv;
    name = utf8_to_sys(name);
    if (!name) return __ERROR;
    /* open directory */
    if (!(dir = opendir(name))) {
      free(name);
      return __FAIL;
    }
    /* count entries */
    for (n = 0, d = readdir(dir); d; d = readdir(dir)) n++;
    /* rewind */
#ifdef HAVE_REWINDDIR
    rewinddir(dir);
#else
    closedir(dir);
    if (!(dir = opendir(name))) {
      free(name);
      return __FAIL;
    }
#endif
    free(name);
    /* allocate vector */
    if (!(xv = malloc(n*sizeof(expr))))
      return __ERROR;
    /* list entries */
    for (i = 0, d = readdir(dir); d; d = readdir(dir))
      if (i < n && (xv[i] = mkstr(sys_to_utf8(d->d_name))))
	i++;
      else {
	while (i > 0) dispose(xv[--i]);
	if (n > 0) free(xv);
	return (i>=n)?__FAIL:__ERROR;
      }
    /* close directory */
    closedir(dir);
    /* check number of items read */
    if (i < n) {
      /* read error */
      while (i > 0) dispose(xv[--i]);
      if (n > 0) free(xv);
      return __FAIL;
    }
    /* return the list */
    return mklistv(n, xv);
  } else
#endif
    return __FAIL;
}

FUNCTION(system,link,argc,argv)
{
#ifdef HAVE_LINK
  char *old, *new;
  if (argc == 2 && isstr(argv[0], &old) && isstr(argv[1], &new)) {
    int res;
    old = utf8_to_sys(old); new = utf8_to_sys(new);
    if (!old || !new) {
      if (old) free(old); if (new) free(new);
      return __ERROR;
    }
    res = link(old, new);
    free(old); free(new);
    if (res)
      return __FAIL;
    else
      return mkvoid;
  } else
#endif
    return __FAIL;
}

FUNCTION(system,symlink,argc,argv)
{
#ifdef HAVE_SYMLINK
  char *old, *new;
  if (argc == 2 && isstr(argv[0], &old) && isstr(argv[1], &new)) {
    int res;
    old = utf8_to_sys(old); new = utf8_to_sys(new);
    if (!old || !new) {
      if (old) free(old); if (new) free(new);
      return __ERROR;
    }
    res = symlink(old, new);
    free(old); free(new);
    if (res)
      return __FAIL;
    else
      return mkvoid;
  } else
#endif
    return __FAIL;
}

FUNCTION(system,readlink,argc,argv)
{
#ifdef HAVE_READLINK
  char *name, buf[BUFSZ+1];
  int n;
  if (argc == 1 && isstr(argv[0], &name)) {
    name = utf8_to_sys(name);
    if (!name) return __ERROR;
    n = readlink(name, buf, BUFSZ);
    free(name);
    if (n < 0)
      return __FAIL;
    else {
      buf[n] = 0;
      return mkstr(sys_to_utf8(buf));
    }
  } else
#endif
    return __FAIL;
}

FUNCTION(system,mkfifo,argc,argv)
{
#ifdef HAVE_MKFIFO
  char *name;
  long mode;
  if (argc == 2 && isstr(argv[0], &name) && isint(argv[1], &mode)) {
    int res;
    name = utf8_to_sys(name);
    if (!name) return __ERROR;
    res = mkfifo(name, mode);
    free(name);
    if (res)
      return __FAIL;
    else
      return mkvoid;
  } else
    return __FAIL;
#else
  return __FAIL;
#endif
}

FUNCTION(system,access,argc,argv)
{
#ifdef HAVE_ACCESS
  char *name;
  long mode;
  if (argc == 2 && isstr(argv[0], &name) && isint(argv[1], &mode)) {
    int res;
    name = utf8_to_sys(name);
    if (!name) return __ERROR;
    res = access(name, mode);
    free(name);
    if (res)
      return mkfalse;
    else
      return mktrue;
  } else
    return __FAIL;
#else
  return __FAIL;
#endif
}

FUNCTION(system,chmod,argc,argv)
{
#ifdef HAVE_CHMOD
  char *name;
  long mode;
  int res;
  if (argc != 2 || !isstr(argv[0], &name) || !isint(argv[1], &mode))
    return __FAIL;
  name = utf8_to_sys(name);
  if (!name) return __ERROR;
  res = chmod(name, mode);
  free(name);
  if (res)
    return __FAIL;
  else
    return mkvoid;
#else
  return __FAIL;
#endif
}

FUNCTION(system,chown,argc,argv)
{
#ifdef HAVE_CHOWN
  char *name;
  long uid, gid;
  int res;
  if (argc != 3 || !isstr(argv[0], &name) || !isint(argv[1], &uid) ||
      !isint(argv[2], &gid))
    return __FAIL;
  name = utf8_to_sys(name);
  if (!name) return __ERROR;
  res = chown(name, uid, gid);
  free(name);
  if (res)
    return __FAIL;
  else
    return mkvoid;
#else
  return __FAIL;
#endif
}

FUNCTION(system,lchown,argc,argv)
{
#ifdef HAVE_LCHOWN
  char *name;
  long uid, gid;
  int res;
  if (argc != 3 || !isstr(argv[0], &name) || !isint(argv[1], &uid) ||
      !isint(argv[2], &gid))
    return __FAIL;
  name = utf8_to_sys(name);
  if (!name) return __ERROR;
  res = lchown(name, uid, gid);
  free(name);
  if (res)
    return __FAIL;
  else
    return mkvoid;
#else
  return __FAIL;
#endif
}

static int get_timeval(expr x, long *t)
{
  double ft;
  if (isfloat(x, &ft))
    if (ft < (double)INT_MIN || ft > (double)INT_MAX)
      return 0;
    else
      *t = (int)ft;
  else if (!isint(x, t))
    return 0;
  return 1;
}

FUNCTION(system,utime,argc,argv)
{
  char *name;
  expr *xs;
  int n;
  long t1, t2;
#ifdef HAVE_UTIME_H
  struct utimbuf ut;
#endif
  int res;
  if (argc != 2 || !isstr(argv[0], &name))
    return __FAIL;
  name = utf8_to_sys(name);
  if (!name) return __ERROR;
  if (istuple(argv[1], &n, &xs)) {
    if (n != 2 ||
	!get_timeval(xs[0], &t1) ||
	!get_timeval(xs[1], &t2)) {
      free(name);
      return __FAIL;
    }
  } else if (get_timeval(argv[1], &t1))
    t2 = t1;
  else {
    free(name);
    return __FAIL;
  }
#ifdef HAVE_UTIME_H
  ut.actime = t1;
  ut.modtime = t2;
  res = utime(name, &ut);
  free(name);
  if (res)
    return __FAIL;
  else
    return mkvoid;
#else
  return __FAIL;
#endif
}

FUNCTION(system,umask,argc,argv)
{
  long mask;
  if (argc == 1 && isint(argv[0], &mask) && mask >= 0 && mask <= 0777)
    return mkint(umask(mask));
  else
    return __FAIL;
}

FUNCTION(system,stat,argc,argv)
{
  char *name;
  if (argc == 1 && isstr(argv[0], &name)) {
    struct stat buf;
    int res;
    name = utf8_to_sys(name);
    if (!name) return __ERROR;
    res = stat(name, &buf);
    free(name);
    if (res)
      return __FAIL;
    else
      return statres(&buf);
  } else
    return __FAIL;
}

#ifdef WIN32
#define lstat stat
#endif

FUNCTION(system,lstat,argc,argv)
{
  char *name;
  if (argc == 1 && isstr(argv[0], &name)) {
    struct stat buf;
    int res;
    name = utf8_to_sys(name);
    if (!name) return __ERROR;
    res = lstat(name, &buf);
    free(name);
    if (res)
      return __FAIL;
    else
      return statres(&buf);
  } else
    return __FAIL;
}

/* process control **********************************************************/

FUNCTION(system,system,argc,argv)
{
  char *cmd;
  if (argc == 1 && isstr(argv[0], &cmd)) {
    int ret;
    cmd = utf8_to_sys(cmd);
    if (!cmd) return __ERROR;
    errno = 0;
    release_lock();
    ret = system(cmd);
    acquire_lock();
    free(cmd);
    if (ret)
      if (errno)
	return __FAIL;
      else
	return mkint(ret);
    else
      return mkint(ret);
  } else
    return __FAIL;
}

FUNCTION(system,fork,argc,argv)
{
#ifdef __MINGW32__
  return __FAIL;
#else
  if (argc == 0/* && this_thread() == 0*/) {
    int pid = fork();
    if (pid < 0)
      return __FAIL;
    else
      return mkint(pid);
  } else
    return __FAIL;
#endif
}

FUNCTION(system,exec,argc,argv)
{
  char *prog, **args;
  if (argc == 2 && isstr(argv[0], &prog)) {
    int i, n;
    expr x, hd, tl;
    char *s;
    for (n = 0, x = argv[1]; iscons(x, &hd, &tl); n++)
      if (n >= INT_MAX)
	return __ERROR;
      else if (!isstr(hd, &s))
	return __FAIL;
      else
	x = tl;
    if (!isnil(x)) return __FAIL;
    if (!(args = malloc((n+1)*sizeof(char*))))
      return __ERROR;
    prog = utf8_to_sys(prog);
    if (!prog) {
      free(args);
      return __ERROR;
    }
    for (n = 0, x = argv[1]; iscons(x, &hd, &tl); n++) {
      isstr(hd, args+n);
      args[n] = utf8_to_sys(args[n]);
      if (!args[n]) {
	free(prog);
	for (i = 0; i < n; i++)
	  free(args[i]);
	free(args);
	return __ERROR;
      }
      x = tl;
    }
    args[n] = NULL;
#ifdef __MINGW32__
    execvp(prog, (const char* const*)args);
#else
    execvp(prog, args);
#endif
    /* if we come here, something has gone wrong */
    free(prog);
    for (i = 0; i < n; i++)
      free(args[i]);
    free(args);
    return __FAIL;
  } else
    return __FAIL;
}

static int spawn(int mode, char *prog, char *argv[], int *status)
{
#ifdef __MINGW32__
  *status = spawnvp(mode, prog, (const char* const*)argv);
  return *status >= 0;
#else
  int pid;
  if (mode == P_OVERLAY) {
    execvp(prog, argv);
    return 0;
  }
  switch ((pid = fork())) {
  case 0:
    execvp(prog, argv);
  case -1:
    return 0;
  }
  if (mode == P_WAIT)
    waitpid(pid, status, 0);
  else
    *status = pid;
  return 1;
#endif
}

FUNCTION(system,spawn,argc,argv)
{
  char *prog, **args;
  if (argc == 2 && isstr(argv[0], &prog)) {
    int i, n, status, res;
    expr x, hd, tl;
    char *s;
    for (n = 0, x = argv[1]; iscons(x, &hd, &tl); n++)
      if (n >= INT_MAX)
	return __ERROR;
      else if (!isstr(hd, &s))
	return __FAIL;
      else
	x = tl;
    if (!isnil(x)) return __FAIL;
    if (!(args = malloc((n+1)*sizeof(char*))))
      return __ERROR;
    prog = utf8_to_sys(prog);
    if (!prog) {
      free(args);
      return __ERROR;
    }
    for (n = 0, x = argv[1]; iscons(x, &hd, &tl); n++) {
      isstr(hd, args+n);
      args[n] = utf8_to_sys(args[n]);
      if (!args[n]) {
	free(prog);
	for (i = 0; i < n; i++)
	  free(args[i]);
	free(args);
	return __ERROR;
      }
      x = tl;
    }
    args[n] = NULL;
    res = spawn(P_NOWAIT, prog, args, &status);
    free(prog);
    for (i = 0; i < n; i++)
      free(args[i]);
    free(args);
    if (res)
      return mkint(status);
    else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(system,_spawn,argc,argv)
{
  char *prog, **args;
  long mode;
  if (argc == 3 && isint(argv[0], &mode) && isstr(argv[1], &prog)) {
    int i, n, status, res;
    expr x, hd, tl;
    char *s;
    for (n = 0, x = argv[2]; iscons(x, &hd, &tl); n++)
      if (n >= INT_MAX)
	return __ERROR;
      else if (!isstr(hd, &s))
	return __FAIL;
      else
	x = tl;
    if (!isnil(x)) return __FAIL;
    if (!(args = malloc((n+1)*sizeof(char*))))
      return __ERROR;
    prog = utf8_to_sys(prog);
    if (!prog) {
      free(args);
      return __ERROR;
    }
    for (n = 0, x = argv[2]; iscons(x, &hd, &tl); n++) {
      isstr(hd, args+n);
      args[n] = utf8_to_sys(args[n]);
      if (!args[n]) {
	free(prog);
	for (i = 0; i < n; i++)
	  free(args[i]);
	free(args);
	return __ERROR;
      }
      x = tl;
    }
    args[n] = NULL;
    res = spawn(mode, prog, args, &status);
    free(prog);
    for (i = 0; i < n; i++)
      free(args[i]);
    free(args);
    if (res)
      return mkint(status);
    else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(system,nice,argc,argv)
{
#ifdef HAVE_NICE
  long inc, res;
  if (argc != 1 || !isint(argv[0], &inc))
    return __FAIL;
  errno = 0;
  res = nice(inc);
  if (res == -1 && errno)
    return __FAIL;
  else
    return mkint(res);
#else
  return __FAIL;
#endif
}

FUNCTION(system,pause,argc,argv)
{
#ifdef HAVE_PAUSE
  long sig;
  if (argc == 0) {
    release_lock();
    pause();
    acquire_lock();
    return mkvoid;
  } else
#endif
    return __FAIL;
}

FUNCTION(system,raise,argc,argv)
{
  long sig;
  if (argc == 1 && isint(argv[0], &sig)) {
    if (raise(sig))
      return __FAIL;
    else
      return mkvoid;
  } else
    return __FAIL;
}

FUNCTION(system,kill,argc,argv)
{
#ifdef __MINGW32__
  return __FAIL;
#else
  long sig, pid;
  if (argc == 2 && isint(argv[0], &sig) && isint(argv[1], &pid)) {
    if (kill(pid, sig) < 0)
      return __FAIL;
    else
      return mkvoid;
  } else
    return __FAIL;
#endif
}

FUNCTION(system,getpid,argc,argv)
{
  if (argc == 0)
    return mkint(getpid());
  else
    return __FAIL;
}

FUNCTION(system,getppid,argc,argv)
{
#ifdef HAVE_GETPPID
  if (argc == 0)
    return mkint(getppid());
  else
#endif
    return __FAIL;
}

#ifndef __MINGW32__
static expr waitres(int pid, int status)
{
  if (pid < 0)
    return __FAIL;
  else if (pid == 0)
    return mkvoid;
  else
    return mktuplel(2, mkint(pid), mkint(status));
}
#endif

FUNCTION(system,wait,argc,argv)
{
#ifdef __MINGW32__
  return __FAIL;
#else
  if (argc == 0) {
    int status, pid;
    release_lock();
    pid = wait(&status);
    acquire_lock();
    return waitres(pid, status);
  } else
    return __FAIL;
#endif
}

FUNCTION(system,waitpid,argc,argv)
{
#ifdef __MINGW32__
  return __FAIL;
#else
  long pid, options;
  if (argc == 2 && isint(argv[0], &pid) && isint(argv[1], &options)) {
    int status;
    release_lock();
    pid = waitpid(pid, &status, options);
    acquire_lock();
    return waitres(pid, status);
  } else
    return __FAIL;
#endif
}

FUNCTION(system,isactive,argc,argv)
{
#ifdef __MINGW32__
  return __FAIL;
#else
  long s;
  if (argc == 1 && isint(argv[0], &s))
    return mkfalse;
  else if (argc == 1 && isvoid(argv[0]))
    return mktrue;
  else
    return __FAIL;
#endif
}

FUNCTION(system,isexited,argc,argv)
{
#ifdef __MINGW32__
  return __FAIL;
#else
  long s;
  if (argc == 1 && isint(argv[0], &s)) {
    int status = s;
    if (WIFEXITED(status))
      return mktrue;
    else
      return mkfalse;
  } else if (argc == 1 && isvoid(argv[0]))
    return mkfalse;
  else
    return __FAIL;
#endif
}

FUNCTION(system,exitstatus,argc,argv)
{
#ifdef __MINGW32__
  return __FAIL;
#else
  long s;
  if (argc == 1 && isint(argv[0], &s)) {
    int status = s;
    if (WIFEXITED(status))
      return mkint(WEXITSTATUS(status));
    else
      return __FAIL;
  } else
    return __FAIL;
#endif
}

FUNCTION(system,issignaled,argc,argv)
{
#ifdef __MINGW32__
  return __FAIL;
#else
  long s;
  if (argc == 1 && isint(argv[0], &s)) {
    int status = s;
    if (WIFSIGNALED(status))
      return mktrue;
    else
      return mkfalse;
  } else if (argc == 1 && isvoid(argv[0]))
    return mkfalse;
  else
    return __FAIL;
#endif
}

FUNCTION(system,termsig,argc,argv)
{
#ifdef __MINGW32__
  return __FAIL;
#else
  long s;
  if (argc == 1 && isint(argv[0], &s)) {
    int status = s;
    if (WIFSIGNALED(status))
      return mkint(WTERMSIG(status));
    else
      return __FAIL;
  } else
    return __FAIL;
#endif
}

FUNCTION(system,isstopped,argc,argv)
{
#ifdef __MINGW32__
  return __FAIL;
#else
  long s;
  if (argc == 1 && isint(argv[0], &s)) {
    int status = s;
    if (WIFSTOPPED(status))
      return mktrue;
    else
      return mkfalse;
  } else if (argc == 1 && isvoid(argv[0]))
    return mkfalse;
  else
    return __FAIL;
#endif
}

FUNCTION(system,stopsig,argc,argv)
{
#ifdef __MINGW32__
  return __FAIL;
#else
  long s;
  if (argc == 1 && isint(argv[0], &s)) {
    int status = s;
    if (WIFSTOPPED(status))
      return mkint(WSTOPSIG(status));
    else
      return __FAIL;
  } else
    return __FAIL;
#endif
}

FUNCTION(system,getenv,argc,argv)
{
  char *name, *val;
  if (argc == 1 && isstr(argv[0], &name)) {
    name = utf8_to_sys(name);
    if (!name) return __ERROR;
    val = getenv(name);
    free(name);
    if (val)
      return mkstr(sys_to_utf8(val));
    else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(system,setenv,argc,argv)
{
  char *name, *val;
  if (argc == 2 && isstr(argv[0], &name) && isstr(argv[1], &val)) {
    /* According to POSIX, the string passed to putenv itself becomes
       part of the environment, so it has to be allocated dynamically. */
    char *s, *envstr = malloc(strlen(name)+strlen(val)+2);
    if (envstr)
      sprintf(envstr, "%s=%s", name, val);
    else
      return __ERROR;
    s = utf8_to_sys(envstr);
    free(envstr);
    if (!s)
      return __ERROR;
    else if (putenv(s)) {
      free(s);
      return __FAIL;
    } else
      return mkvoid;
  } else
    return __FAIL;
}

FUNCTION(system,uname,argc,argv)
{
#ifdef HAVE_SYS_UTSNAME_H
  struct utsname u;
  if (argc == 0 && !uname(&u))
    return mktuplel(5, mkstr(sys_to_utf8(u.sysname)), mkstr(sys_to_utf8(u.nodename)),
		    mkstr(sys_to_utf8(u.release)), mkstr(sys_to_utf8(u.version)),
		    mkstr(sys_to_utf8(u.machine)));
  else
#endif
    return __FAIL;
}

FUNCTION(system,gethostname,argc,argv)
{
#ifdef HAVE_GETHOSTNAME
  char buf[BUFSZ+1];
  if (argc == 0 && !gethostname(buf, BUFSZ)) {
    buf[BUFSZ] = 0;
    return mkstr(sys_to_utf8(buf));
  } else
#endif
    return __FAIL;
}

#ifdef HAVE_GETGROUPS
static expr mkgidlist(int n, GETGROUPS_T *l)
{
  int i;
  expr *xv;
  if (!(xv = malloc(n*sizeof(expr))))
    return __ERROR;
  for (i = 0; i < n; i++)
    xv[i] = mkint(l[i]);
  return mklistv(n, xv);
}
#endif

FUNCTION(system,getgroups,argc,argv)
{
#ifdef HAVE_GETGROUPS
  if (argc == 0) {
    GETGROUPS_T l[BUFSZ+1];
    int n = getgroups(BUFSZ, l);
    if (n >= 0)
      return mkgidlist(n, l);
    else
      return __FAIL;
  } else
#endif
    return __FAIL;
}

FUNCTION(system,setgroups,argc,argv)
{
#ifdef HAVE_SETGROUPS
  if (argc == 1) {
    int n, res;
    long gid;
    expr x, hd, tl;
    gid_t *l;
    for (n = 0, x = argv[0]; iscons(x, &hd, &tl) && isint(hd, &gid); x = tl)
      n++;
    if (!isnil(x)) return __FAIL;
    if (!(l = malloc(n*sizeof(gid_t)))) return __ERROR;
    for (n = 0, x = argv[0]; iscons(x, &hd, &tl) && isint(hd, &gid); x = tl)
      l[n++] = (gid_t)gid;
    res = setgroups(n, l);
    free(l);
    return res?__FAIL:mkvoid;
  } else
#endif
    return __FAIL;
}

FUNCTION(system,setuid,argc,argv)
{
#ifdef HAVE_SETUID
  long uid;
  if (argc == 1 && isint(argv[0], &uid) && !setuid(uid))
    return mkvoid;
  else
#endif
    return __FAIL;
}

FUNCTION(system,setgid,argc,argv)
{
#ifdef HAVE_SETGID
  long gid;
  if (argc == 1 && isint(argv[0], &gid) && !setgid(gid))
    return mkvoid;
  else
#endif
    return __FAIL;
}

FUNCTION(system,seteuid,argc,argv)
{
#ifdef HAVE_SETEUID
  long uid;
  if (argc == 1 && isint(argv[0], &uid) && !seteuid(uid))
    return mkvoid;
  else
#endif
    return __FAIL;
}

FUNCTION(system,setegid,argc,argv)
{
#ifdef HAVE_SETEGID
  long gid;
  if (argc == 1 && isint(argv[0], &gid) && !setegid(gid))
    return mkvoid;
  else
#endif
    return __FAIL;
}

FUNCTION(system,setreuid,argc,argv)
{
#ifdef HAVE_SETREUID
  long ruid, euid;
  if (argc == 2 && isint(argv[0], &ruid) && isint(argv[1], &euid) && 
      !setreuid(ruid, euid))
    return mkvoid;
  else
#endif
    return __FAIL;
}

FUNCTION(system,setregid,argc,argv)
{
#ifdef HAVE_SETREGID
  long rgid, egid;
  if (argc == 2 && isint(argv[0], &rgid) && isint(argv[1], &egid) &&
      !setregid(rgid, egid))
    return mkvoid;
  else
#endif
    return __FAIL;
}

FUNCTION(system,getuid,argc,argv)
{
#ifdef HAVE_GETUID
  if (argc == 0)
    return mkint(getuid());
  else
#endif
    return __FAIL;
}

FUNCTION(system,geteuid,argc,argv)
{
#ifdef HAVE_GETEUID
  if (argc == 0)
    return mkint(geteuid());
  else
#endif
    return __FAIL;
}

FUNCTION(system,getgid,argc,argv)
{
#ifdef HAVE_GETGID
  if (argc == 0)
    return mkint(getgid());
  else
#endif
    return __FAIL;
}

FUNCTION(system,getegid,argc,argv)
{
#ifdef HAVE_GETEGID
  if (argc == 0)
    return mkint(getegid());
  else
#endif
    return __FAIL;
}

FUNCTION(system,getlogin,argc,argv)
{
#ifdef HAVE_GETLOGIN
  char *s;
  if (argc == 0 && (s = getlogin()))
    return mkstr(sys_to_utf8(s));
  else
#endif
    return __FAIL;
}

FUNCTION(system,getpgid,argc,argv)
{
#ifdef HAVE_GETPGID
  long pid;
  if (argc == 1 && isint(argv[0], &pid) && (pid = getpgid(pid)) >= 0)
    return mkint(pid);
  else
#endif
    return __FAIL;
}

FUNCTION(system,setpgid,argc,argv)
{
#ifdef HAVE_SETPGID
  long pid, pgid;
  if (argc == 2 && isint(argv[0], &pid) && isint(argv[1], &pgid) &&
      !setpgid(pid, pgid))
    return mkvoid;
  else
#endif
    return __FAIL;
}

FUNCTION(system,getpgrp,argc,argv)
{
#ifdef HAVE_GETPGRP
  long pid;
#ifdef GETPGRP_VOID
  if (argc == 0 && (pid = getpgrp()) >= 0)
#else
  if (argc == 0 && (pid = getpgrp(0)) >= 0)
#endif
    return mkint(pid);
  else
#endif
    return __FAIL;
}

/* configure might detect wrong interface on these systems */
#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
#undef SETPGRP_VOID
#endif
#if defined(__APPLE__)
#define SETPGRP_VOID
#endif

FUNCTION(system,setpgrp,argc,argv)
{
#ifdef HAVE_SETPGRP
#ifdef SETPGRP_VOID
  if (argc == 0 && !setpgrp())
#else
  if (argc == 0 && !setpgrp(0, 0))
#endif
    return mkvoid;
  else
#endif
    return __FAIL;
}

FUNCTION(system,setsid,argc,argv)
{
#ifdef HAVE_SETSID
  long pid;
  if (argc == 0 && (pid = setsid()) >= 0)
    return mkint(pid);
  else
#endif
    return __FAIL;
}

FUNCTION(system,getsid,argc,argv)
{
#ifdef HAVE_GETSID
  long pid;
  if (argc == 1 && isint(argv[0], &pid) && (pid = getsid(pid)) >= 0)
    return mkint(pid);
  else
#endif
    return __FAIL;
}

/* passwd/group database *****************************************************/

FUNCTION(system,getpwuid,argc,argv)
{
#ifdef HAVE_GETPWUID
  long uid;
  struct passwd *pw;
  if (argc == 1 && isint(argv[0], &uid) &&
      (pw = getpwuid(uid)))
    return mktuplel(7, mkstr(sys_to_utf8(pw->pw_name)),
		    mkstr(strdup(pw->pw_passwd)),
		    mkint(pw->pw_uid), mkint(pw->pw_gid),
		    mkstr(sys_to_utf8(pw->pw_gecos)),
		    mkstr(sys_to_utf8(pw->pw_dir)),
		    mkstr(sys_to_utf8(pw->pw_shell)));
  else
#endif
    return __FAIL;
}

FUNCTION(system,getpwnam,argc,argv)
{
#ifdef HAVE_GETPWNAM
  char *name;
  struct passwd *pw;
  if (argc == 1 && isstr(argv[0], &name)) {
    name = utf8_to_sys(name);
    if (!name) return __ERROR;
    pw = getpwnam(name);
    free(name);
    if (pw)
      return mktuplel(7, mkstr(sys_to_utf8(pw->pw_name)),
		      mkstr(strdup(pw->pw_passwd)),
		      mkint(pw->pw_uid), mkint(pw->pw_gid),
		      mkstr(sys_to_utf8(pw->pw_gecos)),
		      mkstr(sys_to_utf8(pw->pw_dir)),
		      mkstr(sys_to_utf8(pw->pw_shell)));
    else
      return __FAIL;
  } else
#endif
    return __FAIL;
}

FUNCTION(system,getpwent,argc,argv)
{
#ifdef HAVE_GETPWENT
  if (argc == 0) {
    struct passwd *pw;
    int i, n;
    expr *xv;
    setpwent();
    /* count entries */
    for (n = 0, pw = getpwent(); pw; pw = getpwent()) n++;
    /* rewind */
    endpwent();
    setpwent();
    /* allocate vector */
    if (!(xv = malloc(n*sizeof(expr))))
      return __ERROR;
    /* list entries */
    for (i = 0, pw = getpwent(); pw; pw = getpwent())
      if (i < n &&
	  (xv[i] =
	   mktuplel(7, mkstr(sys_to_utf8(pw->pw_name)),
		    mkstr(strdup(pw->pw_passwd)),
		    mkint(pw->pw_uid), mkint(pw->pw_gid),
		    mkstr(sys_to_utf8(pw->pw_gecos)),
		    mkstr(sys_to_utf8(pw->pw_dir)),
		    mkstr(sys_to_utf8(pw->pw_shell)))))
	i++;
      else {
	while (i > 0) dispose(xv[--i]);
	if (n > 0) free(xv);
	return (i>=n)?__FAIL:__ERROR;
      }
    /* close */
    endpwent();
    /* check number of items read */
    if (i < n) {
      /* read error */
      while (i > 0) dispose(xv[--i]);
      if (n > 0) free(xv);
      return __FAIL;
    }
    /* return the list */
    return mklistv(n, xv);
  } else
#endif
    return __FAIL;
}

FUNCTION(system,getgrgid,argc,argv)
{
#ifdef HAVE_GETGRGID
  long gid;
  struct group *gr;
  if (argc == 1 && isint(argv[0], &gid) &&
      (gr = getgrgid(gid)))
    return mktuplel(4, mkstr(sys_to_utf8(gr->gr_name)),
		    mkstr(strdup(gr->gr_passwd)),
		    mkint(gr->gr_gid),
		    mkstrlist(gr->gr_mem));
  else
#endif
    return __FAIL;
}

FUNCTION(system,getgrnam,argc,argv)
{
#ifdef HAVE_GETGRNAM
  char *name;
  struct group *gr;
  if (argc == 1 && isstr(argv[0], &name)) {
    name = utf8_to_sys(name);
    if (!name) return __ERROR;
    gr = getgrnam(name);
    free(name);
    if (gr)
      return mktuplel(4, mkstr(sys_to_utf8(gr->gr_name)),
		      mkstr(strdup(gr->gr_passwd)),
		      mkint(gr->gr_gid),
		      mkstrlist(gr->gr_mem));
    else
      return __FAIL;
  } else
#endif
    return __FAIL;
}

FUNCTION(system,getgrent,argc,argv)
{
#ifdef HAVE_GETGRENT
  if (argc == 0) {
    struct group *gr;
    int i, n;
    expr *xv;
    setgrent();
    /* count entries */
    for (n = 0, gr = getgrent(); gr; gr = getgrent()) n++;
    /* rewind */
    endgrent();
    setgrent();
    /* allocate vector */
    if (!(xv = malloc(n*sizeof(expr))))
      return __ERROR;
    /* list entries */
    for (i = 0, gr = getgrent(); gr; gr = getgrent())
      if (i < n &&
	  (xv[i] =
	   mktuplel(4, mkstr(sys_to_utf8(gr->gr_name)),
		    mkstr(strdup(gr->gr_passwd)),
		    mkint(gr->gr_gid),
		    mkstrlist(gr->gr_mem))))
	i++;
      else {
	while (i > 0) dispose(xv[--i]);
	if (n > 0) free(xv);
	return (i>=n)?__FAIL:__ERROR;
      }
    /* close */
    endgrent();
    /* check number of items read */
    if (i < n) {
      /* read error */
      while (i > 0) dispose(xv[--i]);
      if (n > 0) free(xv);
      return __FAIL;
    }
    /* return the list */
    return mklistv(n, xv);
  } else
#endif
    return __FAIL;
}

FUNCTION(system,crypt,argc,argv)
{
#ifdef HAVE_CRYPT
  char *key, *salt;
  if (argc == 2 && isstr(argv[0], &key) && isstr(argv[1], &salt)) {
    char *s;
    key = utf8_to_sys(key); salt = utf8_to_sys(salt);
    if (!key || !salt) {
      if (key) free(key); if (salt) free(salt);
      return __ERROR;
    }
    s = crypt(key, salt);
    free(key); free(salt);
    return mkstr(sys_to_utf8(s));
  } else
#endif
    return __FAIL;
}

/* basic readline support ***************************************************/

#ifdef USE_READLINE

static expr completion_function = NULL, word_break_chars = NULL;
static char *word_break_chars_s = NULL;

static int rl_ind = 0, rl_histmax = -1;

static char *my_sym_generator(text, state)
     char *text;
     int state;
{
  static expr completions = NULL, current;
  expr hd, tl;
  char *s;
     
  if (!state) {
    if (completions) freeref(completions);
    completions =
      newref(eval(mkapp(mkapp(completion_function, mkstr(sys_to_utf8(text))),
			mkint(rl_ind))));
    current = completions;
    rl_attempted_completion_over = 0;
  }

  if (completions && iscons(current, &hd, &tl))
    if (isstr(hd, &s)) {
      current = tl;
      return utf8_to_sys(s);
    } else if (isnil(tl) && isvoid(hd))
      rl_attempted_completion_over = 1;

  if (completions) {
    freeref(completions);
    completions = NULL;
  }
  return NULL;
}

#ifndef HAVE_RL_COMPLETION_MATCHES
#define rl_completion_matches completion_matches
#endif

static char **my_sym_completion(char *text, int start, int end)
{
  if (start == 0)
    rl_ind = 0;
  else {
    /* determine the start index in the translated string: */
    char *s = malloc(start+1), *t = NULL;
    if (s) {
      strncpy(s, rl_line_buffer, start); s[start] = 0;
      t = sys_to_utf8(s);
    }
    if (s) free(s);
    if (t) {
      rl_ind = u8strlen(t);
      free(t);
    } else
      rl_ind = 0;
  }
  return rl_completion_matches(text, my_sym_generator);
}

static CPPFunction *my_completion_function(void)
{
  expr x = mksym(sym(RL_COMPLETION_FUNCTION)), f = eval(x);
  if (f != x && f != completion_function) {
    if (completion_function) freeref(completion_function);
    completion_function = newref(f);
  } else
    dispose(f);
  return (CPPFunction *) (completion_function?my_sym_completion:NULL);
}

static char *my_word_break_characters(void)
{
  expr x = mksym(sym(RL_WORD_BREAK_CHARS)), y = eval(x);
  char *s;
  if (y != x && y != word_break_chars && isstr(y, &s)) {
    if (word_break_chars) freeref(word_break_chars);
    if (word_break_chars_s) free(word_break_chars_s);
    word_break_chars = newref(y);
    word_break_chars_s = utf8_to_sys(s);
  } else
    dispose(y);
  return word_break_chars_s?word_break_chars_s:" \t\n\"\\'`@$><=;|&{(";
}

static HISTORY_STATE *my_hist = NULL;

static char *my_readline(const char* prompt)
{
  char *buf;
  /* These are used by the interpreter, so we reset and later restore them. */
  const char *save_rl_readline_name = rl_readline_name;
  const char *save_rl_basic_word_break_characters =
    rl_basic_word_break_characters;
  char *save_rl_completer_word_break_characters =
    rl_completer_word_break_characters;
  CPPFunction *save_rl_attempted_completion_function =
    rl_attempted_completion_function;
  HISTORY_STATE *save_hist = history_get_history_state();
  int histmax = unstifle_history();
  rl_readline_name = NULL;
  rl_completer_word_break_characters = my_word_break_characters();
  rl_basic_word_break_characters = rl_completer_word_break_characters;
  rl_attempted_completion_function = my_completion_function();
  history_set_history_state(my_hist);
  if (rl_histmax >= 0) stifle_history(rl_histmax);
  buf = readline(prompt);
  free(my_hist);
  my_hist = history_get_history_state();
  rl_readline_name = save_rl_readline_name;
  rl_basic_word_break_characters = save_rl_basic_word_break_characters;
  rl_completer_word_break_characters = save_rl_completer_word_break_characters;
  rl_attempted_completion_function = save_rl_attempted_completion_function;
  history_set_history_state(save_hist);
  free(save_hist);
  unstifle_history(); if (histmax>=0) stifle_history(histmax);
  return buf;
}

static void my_add_history(const char* line)
{
  HISTORY_STATE *save_hist = history_get_history_state();
  int histmax = unstifle_history();
  history_set_history_state(my_hist);
  add_history(line);
  free(my_hist);
  my_hist = history_get_history_state();
  history_set_history_state(save_hist);
  free(save_hist);
  if (histmax>=0) stifle_history(histmax);
}

static void my_stifle_history(int max)
{
  HISTORY_STATE *save_hist = history_get_history_state();
  int histmax = unstifle_history();
  history_set_history_state(my_hist);
  if (max >= 0) stifle_history(max);
  free(my_hist);
  my_hist = history_get_history_state();
  history_set_history_state(save_hist);
  free(save_hist);
  if (histmax>0) stifle_history(histmax);
}

static int my_read_history(const char* fname)
{
  int res;
  HISTORY_STATE *save_hist = history_get_history_state();
  int histmax = unstifle_history();
  history_set_history_state(my_hist);
  if (rl_histmax >= 0) stifle_history(rl_histmax);
  res = read_history(fname);
  free(my_hist);
  my_hist = history_get_history_state();
  history_set_history_state(save_hist);
  free(save_hist);
  if (histmax>0) stifle_history(histmax);
  return res;
}

static int my_write_history(const char* fname)
{
  int res;
  HISTORY_STATE *save_hist = history_get_history_state();
  int histmax = unstifle_history();
  history_set_history_state(my_hist);
  res = write_history(fname);
  history_set_history_state(save_hist);
  free(save_hist);
  if (histmax>0) stifle_history(histmax);
  return res;
}

#endif

FUNCTION(system,rl_line_buffer,argc,argv)
{
#ifdef USE_READLINE
  if (argc == 0 && rl_line_buffer)
    return mkstr(sys_to_utf8(rl_line_buffer));
  else
#endif
    return __FAIL;
}

FUNCTION(system,readline,argc,argv)
{
#ifdef USE_READLINE
  char *prompt, *buf;
  if (argc == 1 && isstr(argv[0], &prompt) && (buf = my_readline(prompt)))
    return mkstr(buf);
  else
#endif
    return __FAIL;
}

FUNCTION(system,add_history,argc,argv)
{
#ifdef USE_READLINE
  char *line;
  if (argc == 1 && isstr(argv[0], &line)) {
    my_add_history(line);
    return mkvoid;
  } else
#endif
    return __FAIL;
}

FUNCTION(system,stifle_history,argc,argv)
{
#ifdef USE_READLINE
  long histmax;
  if (argc == 1 && isint(argv[0], &histmax)) {
    int old_histmax = rl_histmax;
    rl_histmax = histmax;
    my_stifle_history(rl_histmax);
    return mkint(old_histmax);
  } else
#endif
    return __FAIL;
}

FUNCTION(system,read_history,argc,argv)
{
#ifdef USE_READLINE
  char *fname;
  if (argc == 1 && isstr(argv[0], &fname) && my_read_history(fname) == 0)
    return mkvoid;
  else
#endif
    return __FAIL;
}

FUNCTION(system,write_history,argc,argv)
{
#ifdef USE_READLINE
  char *fname;
  if (argc == 1 && isstr(argv[0], &fname) && my_write_history(fname) == 0)
    return mkvoid;
  else
#endif
    return __FAIL;
}

/* errno and friends ********************************************************/

FUNCTION(system,errno,argc,argv)
{
  if (argc == 0)
    return mkint(errno);
  else
    return __FAIL;
}

FUNCTION(system,seterrno,argc,argv)
{
  long n;
  if (argc == 1 && isint(argv[0], &n)) {
    errno = n;
    return mkvoid;
  } else
    return __FAIL;
}

FUNCTION(system,perror,argc,argv)
{
  char *s;
  if (argc == 1 && isstr(argv[0], &s)) {
    s = utf8_to_sys(s);
    if (!s) return __ERROR;
    perror(s);
    free(s);
    return mkvoid;
  } else
    return __FAIL;
}

FUNCTION(system,strerror,argc,argv)
{
  long n;
  char *s;
  if (argc == 1 && isint(argv[0], &n) && (s = strerror(n)))
    return mkstr(sys_to_utf8(s));
  else
    return __FAIL;
}

/* time functions: ********************************************************/

static struct tm *encode_tmval(expr x)
{
  static struct tm tm;
  expr *xs;
  int n;
  long k;
  if (!istuple(x, &n, &xs) || n != 9)
    return NULL;
  if (!isint(xs[0], &k))
    return NULL;
  else
    tm.tm_year = k;
  if (!isint(xs[1], &k))
    return NULL;
  else
    tm.tm_mon = k;
  if (!isint(xs[2], &k))
    return NULL;
  else
    tm.tm_mday = k;
  if (!isint(xs[3], &k))
    return NULL;
  else
    tm.tm_hour = k;
  if (!isint(xs[4], &k))
    return NULL;
  else
    tm.tm_min = k;
  if (!isint(xs[5], &k))
    return NULL;
  else
    tm.tm_sec = k;
  if (!isint(xs[6], &k))
    return NULL;
  else
    tm.tm_wday = k;
  if (!isint(xs[7], &k))
    return NULL;
  else
    tm.tm_yday = k;
  if (!isint(xs[8], &k))
    return NULL;
  else
    tm.tm_isdst = k;
  return &tm;
}

static expr decode_tmval(struct tm *tm)
{
  if (!tm) return __FAIL;
  return
    mktuplel(9, mkint(tm->tm_year), mkint(tm->tm_mon), mkint(tm->tm_mday),
	     mkint(tm->tm_hour), mkint(tm->tm_min), mkint(tm->tm_sec),
	     mkint(tm->tm_wday), mkint(tm->tm_yday), mkint(tm->tm_isdst));
}

FUNCTION(system,tzname,argc,argv)
{
#if HAVE_DECL_TZNAME
  if (argc == 0)
    return mktuplel(2, mkstr(sys_to_utf8(tzname[0])), mkstr(sys_to_utf8(tzname[1])));
  else
#endif
    return __FAIL;
}

FUNCTION(system,timezone,argc,argv)
{
#if HAVE_DECL_TZNAME && HAVE_DECL_DAYLIGHT
  if (argc == 0)
    return mkint(timezone);
  else
#endif
    return __FAIL;
}

FUNCTION(system,daylight,argc,argv)
{
#if HAVE_DECL_TZNAME && HAVE_DECL_DAYLIGHT
  if (argc == 0)
    return mkint(daylight);
  else
#endif
    return __FAIL;
}

FUNCTION(system,ctime,argc,argv)
{
  if (argc == 1) {
    long it;
    time_t t;
    char *s;
    if (!get_timeval(argv[0], &it))
      return __FAIL;
    t = it;
    if ((s = asctime(localtime(&t))))
      return mkstr(sys_to_utf8(s));
    else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(system,gmtime,argc,argv)
{
  if (argc == 1) {
    long it;
    time_t t;
    if (!get_timeval(argv[0], &it))
      return __FAIL;
    t = it;
    return decode_tmval(gmtime(&t));
  } else
    return __FAIL;
}

FUNCTION(system,localtime,argc,argv)
{
  if (argc == 1) {
    long it;
    time_t t;
    if (!get_timeval(argv[0], &it))
      return __FAIL;
    t = it;
    return decode_tmval(localtime(&t));
  } else
    return __FAIL;
}

FUNCTION(system,mktime,argc,argv)
{
  if (argc == 1) {
    struct tm *tm;
    if (!(tm = encode_tmval(argv[0])))
      return __FAIL;
    else
      return mkint((long)mktime(tm));
  } else
    return __FAIL;
}

FUNCTION(system,asctime,argc,argv)
{
  if (argc == 1) {
    struct tm *tm;
    char *s;
    if (!(tm = encode_tmval(argv[0])) || !(s = asctime(tm)))
      return __FAIL;
    else
      return mkstr(sys_to_utf8(s));
  } else
    return __FAIL;
}

FUNCTION(system,strftime,argc,argv)
{
  char *format;
  if (argc == 2 && isstr(argv[0], &format)) {
    struct tm *tm;
    char s[BUFSZ+1];
    if (!(tm = encode_tmval(argv[1])))
      return __FAIL;
    else {
      /* The interface to strftime is rather brain-damaged since it returns
	 zero both in case of a buffer overflow and when the resulting string
	 is empty. We just pretend that there cannot be any errors and return
	 an empty string in both cases. */
      format = utf8_to_sys(format);
      if (!format) return __ERROR;
      if (!strftime(s, BUFSZ, format, tm))
	*s = 0;
      free(format);
      return mkstr(sys_to_utf8(s));
    }
  } else
    return __FAIL;
}

FUNCTION(system,clock,argc,argv)
{
  if (argc == 0) {
    clock_t t = clock();
    if (t == (clock_t)-1)
      return __FAIL;
    else
      return mkint((long)t);
  } else
    return __FAIL;
}

FUNCTION(system,times,argc,argv)
{
#ifdef HAVE_TIMES
  if (argc == 0) {
    struct tms ts;
    clock_t t = times(&ts);
    if (t == (clock_t)-1)
      return __FAIL;
    else
      return mktuplel(5, mkint(t),
		      mkint((long)ts.tms_utime), mkint((long)ts.tms_stime),
		      mkint((long)ts.tms_cutime), mkint((long)ts.tms_cstime));
  } else
#endif
    return __FAIL;
}

FUNCTION(system,nanotime,argc,argv)
{
#ifdef HAVE_CLOCK_GETTIME
  long id;
  if (argc == 1 && isint(argv[0], &id)) {
    struct timespec tv;
    int res = clock_gettime(id, &tv);
    if (res)
      return __FAIL;
    else {
      mpz_t z;
      /* two limbs should always be enough */
      if (!mpz_new(z, 2)) return __ERROR;
      mpz_set_ui(z, tv.tv_sec);
      mpz_mul_ui(z, z, 1000000000UL);
      mpz_add_ui(z, z, tv.tv_nsec);
      if (!mpz_actsize(z)) return __ERROR;
      return mkmpz(z);
    }
  } else
#endif
    return __FAIL;
}

FUNCTION(system,nanores,argc,argv)
{
#ifdef HAVE_CLOCK_GETRES
  long id;
  if (argc == 1 && isint(argv[0], &id)) {
    struct timespec tv;
    int res = clock_getres(id, &tv);
    if (res)
      return __FAIL;
    else {
      mpz_t z;
      /* two limbs should always be enough */
      if (!mpz_new(z, 2)) return __ERROR;
      mpz_set_ui(z, tv.tv_sec);
      mpz_mul_ui(z, z, 1000000000UL);
      mpz_add_ui(z, z, tv.tv_nsec);
      if (!mpz_actsize(z)) return __ERROR;
      return mkmpz(z);
    }
  } else
#endif
    return __FAIL;
}

FUNCTION(system,nanosleep,argc,argv)
{
#ifdef HAVE_CLOCK_NANOSLEEP
  long id;
  mpz_t z;
  if (argc == 2 && isint(argv[0], &id) && ismpz(argv[1], z)) {
    int res;
    struct timespec tv, rv;
    mpz_t s, n;
    int l = mpz_size(z);
    if (l <= 0) l = 1;
    if (!mpz_new(s, l)) return __ERROR;
    if (!mpz_new(n, 1)) return __ERROR;
    mpz_fdiv_qr_ui(s, n, z, 1000000000UL);
    tv.tv_sec = s->_mp_d[0];
    tv.tv_nsec = n->_mp_d[0];
    mpz_clear(s); mpz_clear(n);
    res = clock_nanosleep(id, 0, &tv, &rv);
    if (res == EINTR) {
      /* interrupt, return remaining time */
      mpz_t z;
      if (!mpz_new(z, 2)) return __ERROR;
      mpz_set_ui(z, rv.tv_sec);
      mpz_mul_ui(z, z, 1000000000UL);
      mpz_add_ui(z, z, rv.tv_nsec);
      if (!mpz_actsize(z)) return __ERROR;
      errno = res;
      return mkmpz(z);
    } else if (res) {
      /* other error, set errno appropriately */
      errno = res;
      return __FAIL;
    } else
      /* sleep timed out, return 0 to indicate success */
      return mkint(0);
  } else
#endif
    return __FAIL;
}

FUNCTION(system,nanosleep_until,argc,argv)
{
#ifdef HAVE_CLOCK_NANOSLEEP
  long id;
  mpz_t z;
  if (argc == 2 && isint(argv[0], &id) && ismpz(argv[1], z)) {
    int res;
    struct timespec tv;
    mpz_t s, n;
    int l = mpz_size(z);
    if (l <= 0) l = 1;
    if (!mpz_new(s, l)) return __ERROR;
    if (!mpz_new(n, 1)) return __ERROR;
    mpz_fdiv_qr_ui(s, n, z, 1000000000UL);
    tv.tv_sec = s->_mp_d[0];
    tv.tv_nsec = n->_mp_d[0];
    mpz_clear(s); mpz_clear(n);
    res = clock_nanosleep(id, TIMER_ABSTIME, &tv, NULL);
    if (res) {
      errno = res;
      return __FAIL;
    } else
      return mkvoid;
  } else
#endif
    return __FAIL;
}

FUNCTION(system,process_cpu_clockid,argc,argv)
{
#ifdef _POSIX_CPUTIME
  long pid;
  if (argc == 1 && isint(argv[0], &pid)) {
    clockid_t id;
    int res = clock_getcpuclockid(pid, &id);
    if (res) {
      errno = res;
      return __FAIL;
    } else
      return mkint(id);
  } else
#endif
    return __FAIL;
}

FUNCTION(system,thread_cpu_clockid,argc,argv)
{
#ifdef _POSIX_THREAD_CPUTIME
  THREAD *thr;
  if (argc == 1 && isobj(argv[0], type(Thread), (void**)&thr)) {
    clockid_t id;
    int res = pthread_getcpuclockid(thr->id, &id);
    if (res) {
      errno = res;
      return __FAIL;
    } else
      return mkint(id);
  } else
#endif
    return __FAIL;
}

/* i18n functions: ********************************************************/

FUNCTION(system,setlocale,argc,argv)
{
#ifdef HAVE_LOCALE_H
  long category;
  char *locale = NULL;
  if (argc == 2 && isint(argv[0], &category) &&
      (isvoid(argv[1]) || isstr(argv[1], &locale)) &&
      (locale = setlocale(category, locale)))
    return mkstr(strdup(locale));
  else
#endif
    return __FAIL;
}

#define lcval(c) ((c==CHAR_MAX)?(long)(-1):(long)c)

FUNCTION(system,localeconv,argc,argv)
{
#ifdef HAVE_LOCALE_H
  struct lconv *lc;
  if (argc == 0 && (lc = localeconv()))
    return mktuplel(18,
		    mkstr(sys_to_utf8(lc->decimal_point)),
		    mkstr(sys_to_utf8(lc->thousands_sep)),
		    mkstr(sys_to_utf8(lc->grouping)),
		    mkstr(sys_to_utf8(lc->int_curr_symbol)),
		    mkstr(sys_to_utf8(lc->currency_symbol)),
		    mkstr(sys_to_utf8(lc->mon_decimal_point)),
		    mkstr(sys_to_utf8(lc->mon_thousands_sep)),
		    mkstr(sys_to_utf8(lc->mon_grouping)),
		    mkstr(sys_to_utf8(lc->positive_sign)),
		    mkstr(sys_to_utf8(lc->negative_sign)),
		    mkint(lcval(lc->int_frac_digits)),
		    mkint(lcval(lc->frac_digits)),
		    mkint(lcval(lc->p_cs_precedes)),
		    mkint(lcval(lc->n_cs_precedes)),
		    mkint(lcval(lc->p_sep_by_space)),
		    mkint(lcval(lc->n_sep_by_space)),
		    mkint(lcval(lc->p_sign_posn)),
		    mkint(lcval(lc->n_sign_posn)));
  else
#endif
    return __FAIL;
}

FUNCTION(system,nl_langinfo,argc,argv)
{
#ifdef HAVE_LANGINFO_CODESET
  long item;
  char *res;
  if (argc == 1 && isint(argv[0], &item) &&
      (res = nl_langinfo(item)))
    return mkstr(sys_to_utf8(res));
  else
#endif
    return __FAIL;
}

FUNCTION(system,strcoll,argc,argv)
{
  char *s1, *s2;
  if (argc == 2 && isstr(argv[0], &s1) && isstr(argv[1], &s2)) {
#ifdef HAVE_UNICODE
    int l1 = strlen(s1), l2 = strlen(s2), res;
    wchar_t *wcs1 = malloc((l1+1)*sizeof(wchar_t)),
      *wcs2 = malloc((l2+1)*sizeof(wchar_t));
    if (!wcs1 || !wcs2) {
      if (wcs1) free(wcs1);
      if (wcs2) free(wcs2);
      return __ERROR;
    }
    if (!u8towcs(wcs1, s1) || !u8towcs(wcs2, s2)) {
      free(wcs1);
      free(wcs2);
      return __FAIL;
    }
#ifdef HAVE_WCSCOLL
    res = wcscoll(wcs1, wcs2);
#else
    res = wcscmp(wcs1, wcs2);
#endif
    free(wcs1);
    free(wcs2);
    return mkint(res);
#else
    return mkint(strcoll(s1, s2));
#endif
  } else
    return __FAIL;
}

FUNCTION(system,strxfrm,argc,argv)
{
  char *s;
  if (argc == 1 && isstr(argv[0], &s)) {
#ifdef HAVE_UNICODE
#ifdef HAVE_WCSXFRM
    int l = strlen(s), k;
    char *t;
    wchar_t *wcs = malloc((l+1)*sizeof(wchar_t)), *wct;
    if (!wcs) return __ERROR;
    if (!u8towcs(wcs, s)) { free(wcs); return __FAIL; }
    k = wcsxfrm(NULL, wcs, 0);
    wct = malloc((k+1)*sizeof(wchar_t));
    if (!wct) { free(wcs); return __ERROR; }
    wcsxfrm(wct, wcs, k+1);
    free(wcs);
    t = malloc(4*k+1);
    if (!t) { free(wct); return __ERROR; }
    if (!wcstou8(t, wct)) {
      free(t); free(wct);
      return __FAIL;
    }
    free(wct);
    if (!(s = realloc(t, strlen(t)+1))) {
      free(t); return __ERROR;
    }
    return mkstr(s);
#else
    return argv[0];
#endif
#else
    int k = strxfrm(NULL, s, 0);
    char *t = malloc(k+1);
    if (!t) return __ERROR;
    strxfrm(t, s, k+1);
    return mkstr(t);
#endif
  } else
    return __FAIL;
}

FUNCTION(system,wcswidth,argc,argv)
{
#if defined(HAVE_UNICODE) && defined(HAVE_WCSWIDTH)
  char *s;
  if (argc == 1 && isstr(argv[0], &s)) {
    int l = strlen(s), res;
    wchar_t *wcs = malloc((l+1)*sizeof(wchar_t));
    if (!wcs) return __ERROR;
    if (!u8towcs(wcs, s)) { free(wcs); return __FAIL; }
    res = wcswidth(wcs, l);
    free(wcs);
    return mkint(res);
  } else
#endif
    return __FAIL;
}

#ifdef HAVE_MONETARY_H

/* this is duplicated from the printf/scanf implementation in clib.c */

static char *buf = NULL, *bufptr = NULL;
static long alloc = 0, leng = 0;

static int addbuf(long newleng)
{
  if (!buf)
    alloc = leng = 0;
  else
    leng = strlen(buf);
  if (leng+newleng+1 <= 0) {
    if (!buf) bufptr = NULL;
    return 0;
  }
  while (leng+newleng >= alloc) {
    if (buf) {
      char *newbuf;
      if (alloc+BUFSZ <= 0 || !(newbuf = realloc(buf, alloc+BUFSZ))) {
	bufptr = buf+leng;
	return 0;
      } else {
	buf = newbuf;
	alloc += BUFSZ;
      }
    } else {
      if (!(buf = malloc(BUFSZ))) {
	bufptr = NULL;
	return 0;
      } else {
	alloc = BUFSZ;
	*buf = 0;
      }
    }
  }
  bufptr = buf+leng;
  return 1;
}

#define F_SZ 1024

static int f_err, f_var_wd, f_var_prec;
static int f_wd, f_prec, f_sz;

static char *f_ptr, *f_str, f_format[F_SZ], f_flags[F_SZ], f_mod[F_SZ];
static char f_wd_str[F_SZ], f_prec_str[F_SZ];

static void f_init(char *format)
{
  f_ptr = format;
  f_err = 0;
  if (!format && f_sz > 4*F_SZ) {
    size_t m = F_SZ+1;
    char *f_str1 = realloc(f_str, m);
    if (f_str1) {
      f_str = f_str1;
      f_sz = m;
    }
  }
}

static int set_f_str(char *s, size_t n)
{
  size_t m = (n>F_SZ)?n:F_SZ;
  if (m+1 < m) return 0;
  m++;
  if (!f_str) {
    f_str = malloc(m);
    if (f_str) f_sz = m;
  } else if (f_sz < m) {
    char *f_str1 = realloc(f_str, m);
    if (f_str1) {
      f_str = f_str1;
      f_sz = m;
    }
  }
  if (n >= m) return 0;
  strncpy(f_str, s, n);
  f_str[n] = 0;
  return 1;
}

#define F_SZ_CHK(len) if (len>=F_SZ) { f_err = 1; return 0; } else
#define SET_F_STR(s,len) if (!set_f_str(s,len)) { f_err = 1; return 0; }

#include <monetary.h>

/* scan strfmon format string */

static char f_parse_mf(void)
{
  char c, *p, *q, *r;
  f_var_wd = f_var_prec = f_wd = f_prec = 0;
  for (p = strchr(f_ptr, '%'); p && p[1] == '%'; p = strchr(p+2, '%'))
    ;
  if (!p) {
    /* no more conversions, return the rest of the string */
    SET_F_STR(f_ptr, strlen(f_ptr));
    *f_format = *f_flags = *f_mod = 0;
    f_ptr = f_ptr + strlen(f_ptr);
    return 0;
  }
  /* conversion starts at p, parse it */
  SET_F_STR(f_ptr, p-f_ptr);
  q = r = p+1;
  /* flags */
  while (strchr("-+(^!", *r) || *r == '=' && *++r != 0) r++;
  F_SZ_CHK(r-q) strncpy(f_flags, q, r-q);
  f_flags[r-q] = 0;
  q = r;
  /* field width */
  while (isdigit(*r)) r++;
  F_SZ_CHK(r-q) strncpy(f_wd_str, q, r-q);
  f_wd_str[r-q] = 0;
  if (*f_wd_str) f_wd = atoi(f_wd_str);
  q = r;
  /* precision */
  if (*r == '#') {
    r++;
    while (isdigit(*r)) r++;
    F_SZ_CHK(r-q) strncpy(f_prec_str, q, r-q);
    f_prec_str[r-q] = 0;
    if (*f_prec_str) f_prec = atoi(f_prec_str);
  }
  if (*r == '.') {
    r++;
    while (isdigit(*r)) r++;
    F_SZ_CHK(r-q) strncpy(f_prec_str, q, r-q);
    f_prec_str[r-q] = 0;
    if (*f_prec_str) f_prec += atoi(f_prec_str);
  }
  q = r;
  /* conversion specifier */
  c = *(r++);
  F_SZ_CHK(r-p) strncpy(f_format, p, r-p);
  f_format[r-p] = 0;
  f_ptr = r;
  return c;
}

static int coerce_float(expr x, double *d)
{
  if (isfloat(x, d) || ismpz_float(x, d))
    return 1;
  else
    return 0;
}

FUNCTION(system,strfmon,argc,argv)
{
  expr x, *xs;
  int i, n, ret;
  long sz, l;
  char f, *format, *s;

  double dblval;

  if (argc != 2 || !isstr(argv[0], &format))
    return __FAIL;
  format = utf8_to_sys(format);
  if (!format) return __ERROR;
  lock_format();
  if (!istuple(argv[1], &n, &xs)) {
    x = argv[1];
    xs = &x;
    n = 1;
  }
  /* parse the format string and check arguments, guestimate needed
     buffer size */
  f_init(format);
  i = 0; sz = BUFSZ;
  while ((f = f_parse_mf())) {
    if (f_wd < 0) f_wd = -f_wd;
    if (f_prec < 0) f_prec = 0;
    l = f_wd+f_prec+100;
    switch (f) {
    case 'i': case 'n':
      if (i >= n || !coerce_float(xs[i], &dblval)) {
	unlock_format();
	free(format);
	return __FAIL;
      } else
	i++;
      if (BUFSZ+l > sz) sz = BUFSZ+l;
      break;
    default:
      unlock_format();
      free(format);
      return __FAIL;
    }
  }
  if (f_err) {
    unlock_format();
    free(format);
    return __FAIL;
  }
  /* allocate buffer */
  if (sz <= 0 || !(s = malloc(sz+1))) {
    unlock_format();
    free(format);
    return __ERROR;
  }
  buf = NULL;
  /* print */
  f_init(format);
  i = 0;
  while ((f = f_parse_mf())) {
    if (!addbuf(strlen(f_str))) goto errexit2;
    if (sprintf(bufptr, f_str) < 0) goto errexit;
    bufptr += strlen(bufptr);
    *s = 0;
    switch (f) {
    case 'i': case 'n':
      coerce_float(xs[i++], &dblval);
      ret = strfmon(s, sz+1, f_format, dblval);
      break;
    }
    if (ret < 0) goto errexit;
    if (!addbuf(strlen(s))) goto errexit2;
    strcpy(bufptr, s);
    bufptr += strlen(bufptr);
  }
  if (!addbuf(strlen(f_str))) goto errexit2;
  if (sprintf(bufptr, f_str) < 0) goto errexit;
  bufptr += strlen(bufptr);
  f_init(NULL);
  unlock_format();
  free(format);
  free(s);
  s = sys_to_utf8(buf);
  free(buf);
  return mkstr(s);
 errexit:
  f_init(NULL);
  unlock_format();
  free(format);
  if (buf) free(buf); free(s);
  return __FAIL;
 errexit2:
  f_init(NULL);
  unlock_format();
  free(format);
  if (buf) free(buf); free(s);
  return __ERROR;
}

#else

FUNCTION(system,strfmon,argc,argv)
{
  return __FAIL;
}

#endif

#ifdef HAVE_ICONV

DESTRUCTOR(system,IConv,ptr)
{
  iconv_t **ic = (iconv_t**)ptr;
  if (ic) {
    if (*ic) iconv_close(*ic);
    free(ic);
  }
}

FUNCTION(system,iconv_open,argc,argv)
{
  char *to, *from;
  if (argc == 2 && isstr(argv[0], &to) && isstr(argv[1], &from)) {
    iconv_t **ic = malloc(sizeof(iconv_t*));
    if (!ic) return __ERROR;
    *ic = iconv_open(to, from);
    if (*ic)
      return mkobj(type(IConv), ic);
    else {
      free(ic);
      return __FAIL;
    }
  } else
    return __FAIL;
}

FUNCTION(system,iconv_close,argc,argv)
{
  iconv_t **ic;
  if (argc == 1 && isobj(argv[0], type(IConv), (void**)&ic) && *ic) {
    if (iconv_close(*ic))
      return __FAIL;
    else {
      *ic = NULL;
      return mkvoid;
    }
  } else
    return __FAIL;
}

FUNCTION(system,iconv,argc,argv)
{
  iconv_t **ic;
  if (argc == 2 && isobj(argv[0], type(IConv), (void**)&ic) && *ic) {
    int save_errno = errno;
    expr x = NULL;
    bstr_t *m;
    char *inbuf = NULL, *outbuf = malloc(BUFSZ), *buf = outbuf;
    size_t nin = 0, nout = BUFSZ, aout = BUFSZ;
    if (!buf)
      return __ERROR;
    else if (isvoid(argv[1]))
      ;
    else if (isobj(argv[1], type(ByteStr), (void**)&m)) {
      inbuf = (char*)m->v;
      nin = (size_t)m->size;
    } else
      return __FAIL;
  restart:
    if (iconv(*ic, &inbuf, &nin, &outbuf, &nout) == (size_t)-1) {
      if (errno == E2BIG) {
	/* Need more space in the output buffer. */
	char *buf0 = buf, *outbuf0 = outbuf;
	if (!(buf = realloc(buf, aout+BUFSZ))) {
	  free(buf0);
	  return __ERROR;
	}
	outbuf = buf + (outbuf0-buf0);
	nout += BUFSZ;
	aout += BUFSZ;
	errno = save_errno;
	goto restart;
      }
      /* If we come here then there was an error during the conversion; copy
	 the remaining bytes in the input to a new byte string to be returned
	 together with the successfully converted part of the sequence. The
	 application will have to check errno to determine the actual cause of
	 the error. */
      if (!(m = malloc(sizeof(bstr_t)))) {
	free(buf);
	return __ERROR;
      } else if (nin > 0) {
	if (!(m->v = malloc(nin))) {
	  free(m);
	  free(buf);
	  return __ERROR;
	}
	memcpy(m->v, inbuf, nin);
	m->size = nin;
      } else {
	m->v = NULL;
	m->size = 0;
      }
      x = mkobj(type(ByteStr), m);
      if (!x) {
	free(buf);
	return __ERROR;
      }
    }
    if (aout == nout) {
      free(buf);
      buf = NULL;
    } else if (!(outbuf = realloc(buf, aout-nout))) {
      if (x) dispose(x);
      free(buf);
      return __ERROR;
    } else
      buf = outbuf;
    if (!(m = malloc(sizeof(bstr_t)))) {
      if (x) dispose(x);
      free(buf);
      return __ERROR;
    }
    m->size = aout-nout;
    m->v = (unsigned char*)buf;
    if (x)
      return mktuplel(2, mkobj(type(ByteStr), m), x);
    else
      return mkobj(type(ByteStr), m);
  } else
    return __FAIL;
}

#else

FUNCTION(system,iconv_open,argc,argv)
{
  return __FAIL;
}

FUNCTION(system,iconv_close,argc,argv)
{
  return __FAIL;
}

FUNCTION(system,iconv,argc,argv)
{
  return __FAIL;
}

#endif

#ifdef ENABLE_NLS

FUNCTION(system,textdomain,argc,argv)
{
  char *domain = NULL;
  if (argc == 1 && (isvoid(argv[0]) || isstr(argv[0], &domain))) {
    if (domain) {
      char *s;
      domain = utf8_to_sys(domain);
      if (!domain) return __ERROR;
      s = textdomain(domain);
      free(domain);
      domain = s;
    } else
      domain = textdomain(domain);
    if (domain)
      return mkstr(sys_to_utf8(domain));
    else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(system,bindtextdomain,argc,argv)
{
  char *domain, *dir = NULL;
  if (argc == 2 && isstr(argv[0], &domain) &&
      (isvoid(argv[1]) || isstr(argv[1], &dir))) {
    domain = utf8_to_sys(domain);
    if (!domain) return __ERROR;
    if (dir) {
      char *s;
      dir = utf8_to_sys(dir);
      if (!dir) {
	free(domain);
	return __ERROR;
      }
      s = bindtextdomain(domain, dir);
      free(dir);
      dir = s;
    } else
      dir = bindtextdomain(domain, dir);
    free(domain);
    if (dir)
      return mkstr(sys_to_utf8(dir));
    else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(system,gettext,argc,argv)
{
  char *msg;
  if (argc == 1 && isstr(argv[0], &msg)) {
    char *s;
    msg = utf8_to_sys(msg);
    if (!msg) return __ERROR;
    s = gettext(msg);
    free(msg);
    if (s)
      return mkstr(sys_to_utf8(s));
    else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(system,dgettext,argc,argv)
{
  char *domain = NULL, *msg;
  if (argc == 2 && (isvoid(argv[0]) || isstr(argv[0], &domain)) &&
      isstr(argv[1], &msg)) {
    char *s;
    msg = utf8_to_sys(msg);
    if (!msg) return __ERROR;
    if (domain) {
      domain = utf8_to_sys(domain);
      if (!domain) {
	free(msg);
	return __ERROR;
      }
      s = dgettext(domain, msg);
      free(domain);
    } else
      s = dgettext(domain, msg);
    free(msg);
    if (s)
      return mkstr(sys_to_utf8(s));
    else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(system,dcgettext,argc,argv)
{
  char *domain = NULL, *msg;
  long category;
  if (argc == 3 && (isvoid(argv[0]) || isstr(argv[0], &domain)) &&
      isstr(argv[1], &msg) && isint(argv[2], &category)) {
    char *s;
    msg = utf8_to_sys(msg);
    if (!msg) return __ERROR;
    if (domain) {
      domain = utf8_to_sys(domain);
      if (!domain) {
	free(msg);
	return __ERROR;
      }
      s = dcgettext(domain, msg, (int)category);
      free(domain);
    } else
      s = dcgettext(domain, msg, (int)category);
    free(msg);
    if (s)
      return mkstr(sys_to_utf8(s));
    else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(system,ngettext,argc,argv)
{
  char *msg, *msg2;
  unsigned long n;
  if (argc == 3 && isstr(argv[0], &msg) && isstr(argv[1], &msg2) &&
      isuint(argv[2], &n)) {
    char *s;
    msg = utf8_to_sys(msg);
    msg2 = utf8_to_sys(msg2);
    if (!msg || !msg2) {
      if (msg) free(msg);
      if (msg2) free(msg2);
      return __ERROR;
    }
    s = ngettext(msg, msg2, n);
    free(msg); free(msg2);
    if (s)
      return mkstr(sys_to_utf8(s));
    else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(system,dngettext,argc,argv)
{
  char *domain = NULL, *msg, *msg2;
  unsigned long n;
  if (argc == 4 && (isvoid(argv[0]) || isstr(argv[0], &domain)) &&
      isstr(argv[1], &msg) && isstr(argv[2], &msg2) &&
      isuint(argv[3], &n)) {
    char *s;
    msg = utf8_to_sys(msg);
    msg2 = utf8_to_sys(msg2);
    if (!msg || !msg2) {
      if (msg) free(msg);
      if (msg2) free(msg2);
      return __ERROR;
    }
    if (domain) {
      domain = utf8_to_sys(domain);
      if (!domain) {
	free(msg); free(msg2);
	return __ERROR;
      }
      s = dngettext(domain, msg, msg2, n);
      free(domain);
    } else
      s = dngettext(domain, msg, msg2, n);
    free(msg); free(msg2);
    if (s)
      return mkstr(sys_to_utf8(s));
    else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(system,dcngettext,argc,argv)
{
  char *domain = NULL, *msg, *msg2;
  unsigned long n;
  long category;
  if (argc == 5 && (isvoid(argv[0]) || isstr(argv[0], &domain)) &&
      isstr(argv[1], &msg) && isstr(argv[2], &msg2) &&
      isuint(argv[3], &n) && isint(argv[4], &category)) {
    char *s;
    msg = utf8_to_sys(msg);
    msg2 = utf8_to_sys(msg2);
    if (!msg || !msg2) {
      if (msg) free(msg);
      if (msg2) free(msg2);
      return __ERROR;
    }
    if (domain) {
      domain = utf8_to_sys(domain);
      if (!domain) {
	free(msg); free(msg2);
	return __ERROR;
      }
      s = dcngettext(domain, msg, msg2, n, (int)category);
      free(domain);
    } else
      s = dcngettext(domain, msg, msg2, n, (int)category);
    free(msg); free(msg2);
    if (s)
      return mkstr(sys_to_utf8(s));
    else
      return __FAIL;
  } else
    return __FAIL;
}

#else

FUNCTION(system,textdomain,argc,argv)
{
  return __FAIL;
}

FUNCTION(system,bindtextdomain,argc,argv)
{
  return __FAIL;
}

FUNCTION(system,gettext,argc,argv)
{
  return __FAIL;
}

FUNCTION(system,dgettext,argc,argv)
{
  return __FAIL;
}

FUNCTION(system,dcgettext,argc,argv)
{
  return __FAIL;
}

FUNCTION(system,ngettext,argc,argv)
{
  return __FAIL;
}

FUNCTION(system,dngettext,argc,argv)
{
  return __FAIL;
}

FUNCTION(system,dcngettext,argc,argv)
{
  return __FAIL;
}

#endif

INIT(system)
{
#ifdef WIN32
  WORD wVersionRequested = MAKEWORD(2, 0);
  WSADATA wsaData;
  WSAStartup(wVersionRequested, &wsaData);
#endif
  tzset();
#ifdef USE_READLINE
  my_hist = history_get_history_state();
  if (my_hist->entries) {
    /* KLUDGE: If we're being rerun, history_get_history_state() will return
       the most recent interpreter history instead of an empty one, so null it
       out. */
    my_hist->entries = NULL;
    my_hist->offset = 0;
    my_hist->length = 0;
    my_hist->size = 0;
    my_hist->flags = 0;
  }
#endif
}

FINI(system)
{
#ifdef WIN32
  WSACleanup();
#endif
#ifndef __MINGW32__
  /* clean up zombies */
  int pid, status, serrno;
  serrno = errno;
  while (1) {
    pid = waitpid (-1, &status, WNOHANG);
    if (pid <= 0) break;
  }
  errno = serrno;
#endif
#ifdef USE_READLINE
  free(my_hist);
#endif
}
