
/* This file is part of the Q programming system.

   The Q programming system is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   The Q programming system is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

#if defined (HAVE_CONFIG_H)
#  include "config.h"
#endif

/* system headers */

/* get the SUSV2 stuff (PTHREAD_MUTEX_RECURSIVE etc.) */
#ifndef _XOPEN_SOURCE
#define _XOPEN_SOURCE 500
#endif

/* get some BSDish stuff from the GNU library (Linux etc.) */
#define _BSD_SOURCE

/* get IPv6 stuff from Solaris */
#define __EXTENSIONS__

#include <stdio.h>
#include <ctype.h>
#include <signal.h>
#include <math.h>

#ifdef HAVE_ERRNO_H
#include <errno.h>
#else
# ifndef errno
extern int errno;
# endif
#endif

#include <sys/stat.h>

/* check for standard C headers */
#if STDC_HEADERS
# include <stdlib.h>
# include <string.h>
#else
# ifndef HAVE_STRCHR
#  define strchr index
#  define strrchr rindex
# endif
char *strchr (), *strrchr ();
#endif

#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif

#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_REGEX_H
#include <regex.h>
#endif

#ifdef HAVE_GLOB_H
#include <glob.h>
#endif

#ifdef HAVE_FNMATCH_H
#include <fnmatch.h>
#endif

#ifdef USE_THREADS
#ifdef HAVE_SCHED_H
#include <sched.h>
#endif
#include <pthread.h>
#include <semaphore.h>
#endif

#ifdef USE_UNICODE
#ifdef HAVE_WCHAR_H
#include <wchar.h>
#include <wctype.h>
#define HAVE_UNICODE 1
#endif
#endif

#ifdef HAVE_ICONV
#include <iconv.h>
#endif

#if defined(HAVE_SYS_SOCKET_H) || defined(WIN32)
#define HAVE_BSD_SOCKETS 1
#endif

#ifdef WIN32

#include <windows.h>
#include <wchar.h>
#include <wctype.h>
#include <iconv.h>
#include <fnmatch.h>
#include <glob.h>
#include <regex.h>

#define HAVE_STRDUP 1
#define HAVE_MEMCPY 1
#define HAVE_MEMCMP 1
#define HAVE_MEMMOVE 1
#define HAVE_MEMSET 1

#define HAVE_FNMATCH 1
#define HAVE_GLOB 1
#define HAVE_REGCOMP 1

#define HAVE_UNICODE 1
#define HAVE_LOCALE_H 1
#define HAVE_ICONV 1
#ifdef __MINGW32__
#define HAVE_TOWUPPER 1
#define HAVE_TOWLOWER 1
#endif

#define HAVE_PTHREAD_MUTEX_TIMEDLOCK 1
#define HAVE_SEM_TIMEDWAIT 1

#endif

/* make sure that gmp.h is included prior to libq.h */

#include <gmp.h>
#include <libq.h>

/* work around a bug in some gmp versions */

#define my_mpz_fits_slong_p(z) (mpz_size(z) == 0 || mpz_fits_slong_p(z))
#define my_mpz_fits_ulong_p(z) (mpz_size(z) == 0 || mpz_fits_ulong_p(z))
#define my_mpz_fits_uint_p(z) (mpz_size(z) == 0 || mpz_fits_uint_p(z))

#ifdef DMALLOC
#include <dmalloc.h>
#endif

MODULE(clib)

#ifndef HAVE_STRDUP

static char *strdup(char *s)
{
  char *t;
  return ((t=malloc(strlen(s)+1))?strcpy(t, s):NULL);
}

#endif

#ifndef HAVE_MEMCPY

#if __GNUC__ > 1
#define memcpy(TO,FROM,COUNT)	__builtin_memcpy(TO,FROM,COUNT)
#else
static void *memcpy (to, from, count)
     char *to;
     char *from;
     int count;
{
  register char *f = from;
  register char *t = to;
  register int i = count;

  while (i-- > 0)
    *t++ = *f++;
  return (void*)to;
}
#endif

#endif

#ifndef HAVE_MEMMOVE

#if __GNUC__ > 1
#define memmove(TO,FROM,COUNT)	__builtin_memmove(TO,FROM,COUNT)
#else
static void *memmove (char *to, char *from, int count)
{
  register char *f = from;
  register char *t = to;
  register int i = count;

  if (from > to) {
    while (i-- > 0)
      *t++ = *f++;
  } else if (from < to) {
    from += count; to += count;
    while (i-- > 0)
      *--t = *--f;
  }
  return (void*)to;
}
#endif

#endif

#ifndef HAVE_MEMSET

#if __GNUC__ > 1
#define memset(TO,C,COUNT)	__builtin_memset(TO,C,COUNT)
#else
static void *memset (char *to, int c, int count)
{
  register char f = (char)c;
  register char *t = to;
  register int i = count;

  while (i-- > 0)
    *t++ = f;
  return (void*)to;
}
#endif

#endif

#ifndef HAVE_MEMCMP

#if __GNUC__ > 1
#define memcmp(S1,S2,COUNT)	__builtin_memcmp(S1,S2,COUNT)
#else
static int memcmp (char *s1, char *s2, int count)
{
  register char *t1 = s1;
  register char *t2 = s2;
  register int i = count;
  register int c;

  while (i-- > 0)
    if ((c = *t1++ - *t2++) != 0)
      return c;
  return 0;
}
#endif

#endif

#define BUFSZ 1024

/* GMP convenience stuff */

#define BYTES_PER_LIMB sizeof(mp_limb_t)

static void*
my_mpz_realloc(m, new_size)
     mpz_ptr m;
     mp_size_t new_size;
{
  mpz_t m1;
  memcpy(m1, m, sizeof(mpz_t));
  if (_mpz_realloc(m, new_size))
    return m->_mp_d;
  else {
    if (m1->_mp_d) mpz_clear(m1);
    return NULL;
  }
}

static void *mpz_new(mpz_t z, mp_size_t sz)
{
  mpz_init(z);
  if (z->_mp_d && my_mpz_realloc(z, sz))
    return z->_mp_d;
  else
    return NULL;
}

static void *mpz_copy(mpz_t z, mpz_t u)
{
  if (mpz_new(z, mpz_size(u))) {
    mpz_set(z, u);
    return z->_mp_d;
  } else
    return NULL;
}

static void *mpz_resize(mpz_t z, mp_size_t sz)
{
  if (sz < mpz_size(z)) sz = mpz_size(z);
  if (sz == 0) sz = 1;
  if (sz != z->_mp_alloc && !my_mpz_realloc(z, sz))
    return NULL;
  else
    return z->_mp_d;
}

#define mpz_actsize(z) mpz_resize(z, mpz_size(z))

#define abs(x) (((x)<0.0)?-x:x)

static void *mpz_from_double(mpz_t z, double x)
{
  double ip, fp, dsz;
  int sz;
  fp = modf(x, &ip);
  /* estimate the number of limbs required */
  dsz = log(abs(ip))/log(2)/((double)CHAR_BIT*sizeof(mp_limb_t))+1.0;
  if (dsz < 1.0) dsz = 1.0; /* this can't happen?? */
  /* add an extra limb to be safe */
  sz = ((int)dsz)+1;
  /* this shouldn't happen but ... ;-) */
  if (((double)INT_MAX) <= dsz || sz < 0) return NULL;
  if (mpz_new(z, sz)) {
    mpz_set_d(z, x);
    return mpz_actsize(z);
  } else
    return NULL;
}

static long long_max(long x, long y)
{
  return (x > y)?x:y;
}

static long long_min(long x, long y)
{
  return (x < y)?x:y;
}

static void *mpz_addop2(void (*f)(), mpz_t z, mpz_t u, mpz_t v)
{
  int sz = long_max(mpz_size(u),mpz_size(v))+1;
  if (sz < 0) return NULL;
  if (mpz_new(z, sz)) {
    f(z, u, v);
    return mpz_actsize(z);
  } else
    return NULL;
}

static void *mpz_addop1(void (*f)(), mpz_t z, mpz_t u, mp_limb_t v)
{
  int sz = mpz_size(u)+1;
  if (sz < 0) return NULL;
  if (mpz_new(z, sz)) {
    f(z, u, v);
    return mpz_actsize(z);
  } else
    return NULL;
}

/* reverse the limbs of an mpz_t value to convert between little- and
   big-endian limb order */

static void reverse_limbs(mp_limb_t *p, unsigned n)
{
  unsigned i;
  for (i = 0; i < n/2; i++) {
    mp_limb_t x = p[i];
    p[i] = p[n-i-1];
    p[n-i-1] = x;
  }
}

/* utf-8/unicode helpers */

#define sys_to_utf8(s) to_utf8(s, NULL)
#define utf8_to_sys(s) from_utf8(s, NULL)

#ifdef HAVE_UNICODE

static inline long
u8decode(char *s)
{
  size_t n = 0;
  unsigned p = 0, q = 0;
  unsigned long c = 0;
  if (s[0] == 0)
    return -1;
  else if (s[1] == 0)
    return (unsigned char)s[0];
  for (; n == 0 && *s; s++) {
    unsigned char uc = (unsigned char)*s;
    if (q == 0) {
      if (((signed char)uc) < 0) {
	switch (uc & 0xf0) {
	case 0xc0: case 0xd0:
	  q = 1;
	  c = uc & 0x1f;
	  break;
	case 0xe0:
	  q = 2;
	  c = uc & 0xf;
	  break;
	case 0xf0:
	  if ((uc & 0x8) == 0) {
	    q = 3;
	    c = uc & 0x7;
	  } else
	    c = uc;
	  break;
	default:
	  c = uc;
	  break;
	}
      } else
	c = uc;
      p = 0; if (q == 0) n++;
    } else if ((uc & 0xc0) == 0x80) {
      /* continuation byte */
      c = c << 6 | (uc & 0x3f);
      if (--q == 0)
	n++;
      else
	p++;
    } else {
      /* malformed char */
      return -1;
    }
  }
  if (n == 1 && *s == 0)
    return c;
  else
    return -1;
}

static inline long
u8decodes(char *s, char **t)
{
  size_t n = 0;
  unsigned p = 0, q = 0;
  unsigned long c = 0;
  if (s[0] == 0)
    return -1;
  else if (s[1] == 0) {
    *t = s+1;
    return (unsigned char)s[0];
  }
  for (; n == 0 && *s; s++) {
    unsigned char uc = (unsigned char)*s;
    if (q == 0) {
      if (((signed char)uc) < 0) {
	switch (uc & 0xf0) {
	case 0xc0: case 0xd0:
	  q = 1;
	  c = uc & 0x1f;
	  break;
	case 0xe0:
	  q = 2;
	  c = uc & 0xf;
	  break;
	case 0xf0:
	  if ((uc & 0x8) == 0) {
	    q = 3;
	    c = uc & 0x7;
	  } else
	    c = uc;
	  break;
	default:
	  c = uc;
	  break;
	}
      } else
	c = uc;
      p = 0; if (q == 0) n++;
    } else if ((uc & 0xc0) == 0x80) {
      /* continuation byte */
      c = c << 6 | (uc & 0x3f);
      if (--q == 0)
	n++;
      else
	p++;
    } else {
      /* malformed char */
      return -1;
    }
  }
  if (n == 1) {
    *t = s;
    return c;
  } else
    return -1;
}

static inline char *
u8encode(char *t, unsigned long c)
{
  unsigned char *uc = (unsigned char*)t;
  if (c < 0x80) {
    uc[1] = 0;
    uc[0] = c;
  } else if (c < 0x800) {
    uc[2] = 0;
    uc[1] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[0] = 0xc0 | c;
  } else if (c < 0x10000) {
    uc[3] = 0;
    uc[2] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[1] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[0] = 0xe0 | c;
  } else {
    uc[4] = 0;
    uc[3] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[2] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[1] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[0] = 0xf0 | c;
  }
  return t;
}

static inline size_t
u8strlen(char *s)
{
  size_t n = 0;
  unsigned p = 0, q = 0;
 start:
  for (; *s; s++) {
    unsigned char uc = (unsigned char)*s;
    if (q == 0) {
      if (((signed char)uc) < 0) {
	switch (uc & 0xf0) {
	case 0xc0: case 0xd0:
	  q = 1;
	  break;
	case 0xe0:
	  q = 2;
	  break;
	case 0xf0:
	  if ((uc & 0x8) == 0)
	    q = 3;
	  break;
	}
      }
      p = 0; n++;
    } else if ((uc & 0xc0) == 0x80) {
      /* continuation byte */
      if (--q == 0)
	p = 0;
      else
	p++;
    } else {
      /* malformed char */
      s -= p+1; p = q = 0;
    }
  }
  if (q > 0) {
    /* unterminated char */
    s -= p; p = q = 0;
    goto start;
  }
  return n;
}

#ifdef HAVE_ICONV

/* conversion between UTF-8 and wchar_t */

static iconv_t myic[2] = { (iconv_t)-1, (iconv_t)-1 };

static inline wchar_t *
ictowcs(wchar_t *t, char *s)
{
  if (myic[1] == (iconv_t)-1)
    myic[1] = iconv_open("WCHAR_T", "UTF-8");
  if (myic[1] == (iconv_t)-1)
    return NULL;
  else {
    size_t l = strlen(s);
    char *inbuf = s; wchar_t *outbuf = t;
    size_t inbytes = l, outbytes = l*sizeof(wchar_t);

    if (iconv(myic[1], &inbuf, &inbytes, (char**)&outbuf, &outbytes) ==
	(size_t)-1)
      return NULL;
    /* terminate the output string */
    *outbuf = 0;
    return t;
  }
}

static inline char *
icfromwcs(char *t, wchar_t *s)
{
  if (myic[0] == (iconv_t)-1)
    myic[0] = iconv_open("UTF-8", "WCHAR_T");
  if (myic[0] == (iconv_t)-1)
    return NULL;
  else {
    size_t l = wcslen(s);
    wchar_t *inbuf = s; char *outbuf = t;
    size_t inbytes = l*sizeof(wchar_t), outbytes = l*4;

    if (iconv(myic[0], (char**)&inbuf, &inbytes, &outbuf, &outbytes) ==
	(size_t)-1)
      return NULL;
    /* terminate the output string */
    *outbuf = 0;
    return t;
  }
}

#endif

#ifdef __STDC_ISO_10646__
#define MY_ISO_10646
#else
#ifndef HAVE_ICONV
#warning "wchar_t encoding unknown and iconv not available, assuming ISO 10646"
#define MY_ISO_10646
#endif
#endif

#ifdef MY_ISO_10646
#define towchar(c) ((wchar_t)(c))
#else
static wchar_t towchar(unsigned long c)
{
  char s[5]; /* single utf-8 char can have at most 4 bytes, plus terminal 0 */
  wchar_t t[5]; /* just to be safe; 2 should actually be enough */
  u8encode(s, c);
  if (ictowcs(t, s))
    return t[0];
  else
    /* Your system is so utterly broken that we can't even convert UTF-8 to
       wchar_t. You should probably configure with --without-unicode. But
       let's just pretend we have an ISO 10646 compatible encoding anyway. */
    return (wchar_t)c;
}
#endif

static wchar_t *u8towcs(wchar_t *t, char *s)
{
#ifdef MY_ISO_10646
  wchar_t *t0 = t;
  while (*s) {
    char *s1;
    long c = u8decodes(s, &s1);
    if (c < 0)
      *t++ = (wchar_t)(*s++);
    else {
      *t++ = (wchar_t)c;
      s = s1;
    }
  }
  *t = 0;
  return t0;
#else
  return ictowcs(t, s);
#endif
}

static char *wcstou8(char *t, wchar_t *s)
{
#ifdef MY_ISO_10646
  char *t0 = t;
  while (*s) {
    u8encode(t, *s++);
    t += strlen(t);
  }
  return t0;
#else
  return icfromwcs(t, s);
#endif
}

#endif

#ifdef USE_THREADS
static pthread_mutex_t format_mutex;
#endif

static void lock_format(void)
{
#ifdef USE_THREADS
  pthread_mutex_lock(&format_mutex);
#endif
}

static void unlock_format(void)
{
#ifdef USE_THREADS
  pthread_mutex_unlock(&format_mutex);
#endif
}

/* manifest constants */

/* data sizes */

#ifndef SIZEOF_CHAR
#define SIZEOF_CHAR sizeof(char)
#endif

#ifndef SIZEOF_SHORT
#define SIZEOF_SHORT sizeof(short)
#endif

#ifndef SIZEOF_INT
#define SIZEOF_INT sizeof(int)
#endif

#ifndef SIZEOF_LONG
#define SIZEOF_LONG sizeof(long)
#endif

#ifndef SIZEOF_LONG_LONG
#if __GNUC__ > 1
#define SIZEOF_LONG_LONG sizeof(long long)
#else
/* can't be sure that this one exists */
#define SIZEOF_LONG_LONG 0
#endif
#endif

#ifndef SIZEOF_FLOAT
#define SIZEOF_FLOAT sizeof(float)
#endif

#ifndef SIZEOF_DOUBLE
#define SIZEOF_DOUBLE sizeof(double)
#endif

/* Signal numbers. */

#ifdef SIGABRT
#define _SIGABRT SIGABRT
#else
#define _SIGABRT (-1)
#endif
#ifdef SIGALRM
#define _SIGALRM SIGALRM
#else
#define _SIGALRM (-1)
#endif
#ifdef SIGFPE
#define _SIGFPE SIGFPE
#else
#define _SIGFPE (-1)
#endif
#ifdef SIGHUP
#define _SIGHUP SIGHUP
#else
#define _SIGHUP (-1)
#endif
#ifdef SIGILL
#define _SIGILL SIGILL
#else
#define _SIGILL (-1)
#endif
#ifdef SIGINT
#define _SIGINT SIGINT
#else
#define _SIGINT (-1)
#endif
#ifdef SIGKILL
#define _SIGKILL SIGKILL
#else
#define _SIGKILL (-1)
#endif
#ifdef SIGPIPE
#define _SIGPIPE SIGPIPE
#else
#define _SIGPIPE (-1)
#endif
#ifdef SIGQUIT
#define _SIGQUIT SIGQUIT
#else
#define _SIGQUIT (-1)
#endif
#ifdef SIGSEGV
#define _SIGSEGV SIGSEGV
#else
#define _SIGSEGV (-1)
#endif
#ifdef SIGTERM
#define _SIGTERM SIGTERM
#else
#define _SIGTERM (-1)
#endif
#ifdef SIGUSR1
#define _SIGUSR1 SIGUSR1
#else
#define _SIGUSR1 (-1)
#endif
#ifdef SIGUSR2
#define _SIGUSR2 SIGUSR2
#else
#define _SIGUSR2 (-1)
#endif
#ifdef SIGCHLD
#define _SIGCHLD SIGCHLD
#else
#define _SIGCHLD (-1)
#endif
#ifdef SIGCONT
#define _SIGCONT SIGCONT
#else
#define _SIGCONT (-1)
#endif
#ifdef SIGSTOP
#define _SIGSTOP SIGSTOP
#else
#define _SIGSTOP (-1)
#endif
#ifdef SIGTSTP
#define _SIGTSTP SIGTSTP
#else
#define _SIGTSTP (-1)
#endif
#ifdef SIGTTIN
#define _SIGTTIN SIGTTIN
#else
#define _SIGTTIN (-1)
#endif
#ifdef SIGTTOU
#define _SIGTTOU SIGTTOU
#else
#define _SIGTTOU (-1)
#endif
#ifdef SIGBUS
#define _SIGBUS SIGBUS
#else
#define _SIGBUS (-1)
#endif
#ifdef SIGPOLL
#define _SIGPOLL SIGPOLL
#else
#define _SIGPOLL (-1)
#endif
#ifdef SIGPROF
#define _SIGPROF SIGPROF
#else
#define _SIGPROF (-1)
#endif
#ifdef SIGSYS
#define _SIGSYS SIGSYS
#else
#define _SIGSYS (-1)
#endif
#ifdef SIGTRAP
#define _SIGTRAP SIGTRAP
#else
#define _SIGTRAP (-1)
#endif
#ifdef SIGURG
#define _SIGURG SIGURG
#else
#define _SIGURG (-1)
#endif
#ifdef SIGVTALRM
#define _SIGVTALRM SIGVTALRM
#else
#define _SIGVTALRM (-1)
#endif
#ifdef SIGXCPU
#define _SIGXCPU SIGXCPU
#else
#define _SIGXCPU (-1)
#endif
#ifdef SIGXFSZ
#define _SIGXFSZ SIGXFSZ
#else
#define _SIGXFSZ (-1)
#endif

#ifndef SEEK_SET
#define SEEK_SET 0
#endif
#ifndef SEEK_CUR
#define SEEK_CUR 1
#endif
#ifndef SEEK_END
#define SEEK_END 2
#endif

FUNCTION(clib,sys_vars,argc,argv)
{
  if (argc != 0) return __FAIL;
  return mktuplel
    (41,

     mkint(SIZEOF_CHAR), mkint(SIZEOF_SHORT), mkint(SIZEOF_INT),
     mkint(SIZEOF_LONG), mkint(SIZEOF_LONG_LONG),
     mkint(SIZEOF_FLOAT), mkint(SIZEOF_DOUBLE),

     mkint(_SIGABRT),
     mkint(_SIGALRM),
     mkint(_SIGFPE),
     mkint(_SIGHUP),
     mkint(_SIGILL),
     mkint(_SIGINT),
     mkint(_SIGKILL),
     mkint(_SIGPIPE),
     mkint(_SIGQUIT),
     mkint(_SIGSEGV),
     mkint(_SIGTERM),
     mkint(_SIGUSR1),
     mkint(_SIGUSR2),
     mkint(_SIGCHLD),
     mkint(_SIGCONT),
     mkint(_SIGSTOP),
     mkint(_SIGTSTP),
     mkint(_SIGTTIN),
     mkint(_SIGTTOU),
     mkint(_SIGBUS),
     mkint(_SIGPOLL),
     mkint(_SIGPROF),
     mkint(_SIGSYS),
     mkint(_SIGTRAP),
     mkint(_SIGURG),
     mkint(_SIGVTALRM),
     mkint(_SIGXCPU),
     mkint(_SIGXFSZ),

     mkint(_IONBF), mkint(_IOLBF), mkint(_IOFBF),

     mkint(SEEK_SET), mkint(SEEK_CUR), mkint(SEEK_END));
}

/* exit function: *********************************************************/

FUNCTION(clib,exit,argc,argv)
{
  long code;
  if (argc == 1 && isint(argv[0], &code)) {
    exit(code);
    /* we shouldn't arrive here */
    return __FAIL;
  } else
    return __FAIL;
}

/* C replacements of common stdlib functions: *****************************/

/* Note: In difference to our usual philosophy to make an operation __FAIL if
   certain implementation limits are exceeded, the following operations return
   an __ERROR in such cases. This is done whenever the operation, if __FAILed,
   would certainly cause some stack or memory overflow later, so we might as
   well abort it right here. */

FUNCTION(clib,append,argc,argv)
{
  if (argc == 2) {
    int i, n = 0;
    expr x, *xs, *ys, hd, tl;
    if (isvoid(argv[0]) || istuple(argv[0], &n, &xs)) {
      if (!(ys = malloc((n+1)*sizeof(expr))))
	return __ERROR;
      for (i = 0; i < n; i++)
	ys[i] = xs[i];
      ys[i++] = argv[1];
      return mktuplev(n+1, ys);
    } else {
      for (n = 0, x = argv[0]; iscons(x, &hd, &tl); n++)
	if (n >= INT_MAX/sizeof(expr)-1)
	  return __ERROR;
	else
	  x = tl;
      if (!isnil(x))
	return __FAIL;
      else if (!(ys = malloc((n+1)*sizeof(expr))))
	return __ERROR;
      for (n = 0, x = argv[0]; iscons(x, &hd, &tl); n++) {
	ys[n] = hd;
	x = tl;
      }
      ys[n++] = argv[1];
      return mklistv(n, ys);
    }
  } else
    return __FAIL;
}

FUNCTION(clib,cat,argc,argv)
{
  expr *xs, x, y, hd, tl, hd1, tl1;
  int n;
  if (argc != 1) return __FAIL;
  for (n = 0, x = argv[0]; iscons(x, &hd, &tl); x = tl) {
    for (y = hd; iscons(y, &hd1, &tl1); y = tl1)
      if (n >= INT_MAX/sizeof(expr))
	return __ERROR;
      else
	n++;
    if (!isnil(y)) return __FAIL;
  }
  if (!isnil(x)) return __FAIL;
  if (!(xs = malloc(n*sizeof(expr))))
    return __ERROR;
  for (n = 0, x = argv[0]; iscons(x, &hd, &tl); x = tl)
    for (y = hd; iscons(y, &hd1, &tl1); y = tl1)
      xs[n++] = hd1;
  return mklistv(n, xs);
}

FUNCTION(clib,mklist,argc,argv)
{
  long n;
  if (argc == 2 && isint(argv[1], &n)) {
    expr x = argv[0], y = mknil;
    while (y && n-- > 0)
      y = mkcons(x, y);
    if (!y)
      return __ERROR;
    else
      return y;
  } else
    return __FAIL;
}

FUNCTION(clib,nums,argc,argv)
{
  mpz_t n, m, u;
  double f, g;
  long len, i, j;
  expr *xs;
  unsigned char n_is_mpz, m_is_mpz;

  if (argc != 2 ||
      !((n_is_mpz = ismpz(argv[0], n)) || isfloat(argv[0], &f)) ||
      !((m_is_mpz = ismpz(argv[1], m)) || isfloat(argv[1], &g)))
    return __FAIL;
  if (n_is_mpz) {
    if (!m_is_mpz && !mpz_from_double(m, g))
      return __ERROR;
    if (mpz_cmp(n, m) > 0)
      return mknil;
    if (!mpz_addop2(mpz_sub, u, m, n))
      return __ERROR;
    if (!my_mpz_fits_slong_p(u)) {
      mpz_clear(u);
      return __ERROR;
    }
    len = mpz_get_si(u)+1;
    mpz_clear(u);
    if (len <= 0 || len > INT_MAX/sizeof(expr)) return __ERROR;
    if (!(xs = malloc(len*sizeof(expr))))
      return __ERROR;
    if (!mpz_copy(u, n)) return __ERROR;
    i = 0;
    while (mpz_cmp(u, m) <= 0) {
      mpz_t v;
      if (!(xs[i++] = mkmpz(u))) {
	for (j = 0; j < i-1; j++) dispose(xs[j]);
	free(xs);
	return __ERROR;
      }
      if (!mpz_addop1(mpz_add_ui, v, u, 1U)) {
	for (j = 0; j < i; j++) dispose(xs[j]);
	free(xs);
	mpz_clear(u);
	return __ERROR;
      }
      memcpy(u, v, sizeof(mpz_t));
    }
  } else {
    double ip;
    if (m_is_mpz) g = mpz_get_d(m);
    if (g < f) return mknil;
    modf(g-f, &ip);
    if (ip >= (double)INT_MAX) return __ERROR;
    /* add 1 to be safe (will be corrected later) */
    len = ((int)ip)+2;
    if (len <= 0 || len > INT_MAX/sizeof(expr)) return __ERROR;
    if (!(xs = malloc(len*sizeof(expr))))
      return __ERROR;
    i = 0;
    while (f <= g) {
      if (!(xs[i++] = mkfloat(f))) {
	for (j = 0; j < i-1; j++) dispose(xs[j]);
	free(xs);
	return __ERROR;
      }
      f += 1.0;
    }
    if (i < len) len = i;
  }
  return mklistv(len, xs);
}

FUNCTION(clib,numsby,argc,argv)
{
  mpz_t n, m, k, u;
  double f, g, h;
  long len, i, j;
  expr *xs;
  unsigned char k_is_mpz, n_is_mpz, m_is_mpz;

  if (argc != 3 ||
      !((k_is_mpz = ismpz(argv[0], k)) || isfloat(argv[0], &h)) ||
      !((n_is_mpz = ismpz(argv[1], n)) || isfloat(argv[1], &f)) ||
      !((m_is_mpz = ismpz(argv[2], m)) || isfloat(argv[2], &g)))
    return __FAIL;
  if (n_is_mpz && k_is_mpz) {
    int sgn = mpz_sgn(k);
    if (sgn == 0) return __FAIL;
    if (!m_is_mpz && !mpz_from_double(m, g))
      return __ERROR;
    if (mpz_cmp(n, m)*sgn > 0)
      return mknil;
    if (!mpz_addop2(mpz_sub, u, m, n))
      return __ERROR;
    /* this one shouldn't fail */
    mpz_tdiv_q(u, u, k);
    if (!my_mpz_fits_slong_p(u)) {
      mpz_clear(u);
      return __ERROR;
    }
    len = mpz_get_si(u)+1;
    mpz_clear(u);
    if (len <= 0 || len > INT_MAX/sizeof(expr)) return __ERROR;
    if (!(xs = malloc(len*sizeof(expr))))
      return __ERROR;
    if (!mpz_copy(u, n)) return __ERROR;
    i = 0;
    while (sgn*mpz_cmp(u, m) <= 0) {
      mpz_t v;
      if (!(xs[i++] = mkmpz(u))) {
	for (j = 0; j < i-1; j++) dispose(xs[j]);
	free(xs);
	return __ERROR;
      }
      if (!mpz_addop2(mpz_add, v, u, k)) {
	for (j = 0; j < i; j++) dispose(xs[j]);
	free(xs);
	mpz_clear(u);
	return __ERROR;
      }
      memcpy(u, v, sizeof(mpz_t));
    }
  } else {
    double ip, sgn;
    if (k_is_mpz) h = mpz_get_d(k);
    if (n_is_mpz) f = mpz_get_d(n);
    if (m_is_mpz) g = mpz_get_d(m);
    sgn = (h<0.0)?-1.0:(h>0.0)?1.0:0.0;
    if (sgn == 0.0) return __FAIL;
    if ((f-g)*sgn > 0.0) return mknil;
    modf((g-f)/h, &ip);
    if (ip >= (double)INT_MAX) return __ERROR;
    len = ((int)ip)+2;
    if (len <= 0 || len > INT_MAX/sizeof(expr)) return __ERROR;
    if (!(xs = malloc(len*sizeof(expr))))
      return __ERROR;
    i = 0;
    while (sgn*(f-g) <= 0) {
      if (!(xs[i++] = mkfloat(f))) {
	for (j = 0; j < i-1; j++) dispose(xs[j]);
	free(xs);
	return __ERROR;
      }
      f += h;
    }
    if (i < len) len = i;
  }
  return mklistv(len, xs);
}

FUNCTION(clib,reverse,argc,argv)
{
  if (argc == 1) {
    expr x = argv[0], y = mknil, hd, tl, *xv = NULL;
    int n = 0;
    if (isvoid(x) || istuple(x, &n, &xv)) {
      if (n > 0) {
	expr *yv = malloc(n*sizeof(expr));
	int i;
	if (!yv) return __ERROR;
	for (i = 0; i < n; i++)
	  yv[i] = xv[n-i-1];
	return mktuplev(n, yv);
      } else
	return mkvoid;
    } else {
      while (y && iscons(x, &hd, &tl)) {
	expr z = mkcons(hd, y);
	y = z; x = tl;
      }
      if (!y)
	return __ERROR;
      else if (isnil(x))
	return y;
      else {
	dispose(y);
	return __FAIL;
      }
    }
  } else
    return __FAIL;
}

FUNCTION(clib,tuplecat,argc,argv)
{
  expr *xs, *ys, x, hd, tl;
  int n, l;
  if (argc != 1) return __FAIL;
  for (n = 0, x = argv[0]; iscons(x, &hd, &tl); x = tl) {
    if (istuple(hd, &l, &ys))
      if (l < 0 || n > INT_MAX/sizeof(expr)-l)
	return __ERROR;
      else
	n += l;
    else
      return __FAIL;
  }
  if (!isnil(x)) return __FAIL;
  if (!(xs = malloc(n*sizeof(expr))))
    return __ERROR;
  for (n = 0, x = argv[0]; iscons(x, &hd, &tl); x = tl)
    if (istuple(hd, &l, &ys) & l > 0) {
      memcpy(xs+n, ys, l*sizeof(expr));
      n += l;
    }
  return mktuplev(n, xs);
}

FUNCTION(clib,chars,argc,argv)
{
  char *s;
  if (argc == 1 && isstr(argv[0], &s)) {
#ifdef HAVE_UNICODE
    int l = strlen(s);
    if (l < 0 || l > INT_MAX/sizeof(expr))
      return __ERROR;
    else if (l == 0)
      return mknil;
    else {
      expr *xs = malloc(l*sizeof(expr));
      char t[5];
      int i = 0, j;
      unsigned p = 0, q = 0;
      if (!xs) return __ERROR;
    start:
      for (; *s; s++) {
	unsigned char uc = (unsigned char)*s;
	if (q == 0) {
	  if (((signed char)uc) < 0) {
	    switch (uc & 0xf0) {
	    case 0xc0: case 0xd0:
	      q = 1;
	      break;
	    case 0xe0:
	      q = 2;
	      break;
	    case 0xf0:
	      if ((uc & 0x8) == 0)
		q = 3;
	      break;
	    }
	  }
	  p = 0;
	  if (q == 0) {
	    t[0] = *s; t[1] = 0;
	    if (!(xs[i] = mkstr(strdup(t))))
	      goto errexit;
	    else
	      i++;
	  }
	} else if ((uc & 0xc0) == 0x80) {
	  /* continuation byte */
	  if (--q == 0) {
	    strncpy(t, s-p-1, p+2);
	    t[p+2] = 0;
	    if (!(xs[i] = mkstr(strdup(t))))
	      goto errexit;
	    else
	      i++;
	    p = 0;
	  } else
	    p++;
	} else {
	  /* malformed char */
	  s -= p+1;
	  t[0] = *s; t[1] = 0;
	  if (!(xs[i] = mkstr(strdup(t))))
	    goto errexit;
	  else
	    i++;
	  p = q = 0;
	}
      }
      if (q > 0) {
	/* unterminated char */
	s -= p;
	t[0] = *s; t[1] = 0;
	if (!(xs[i] = mkstr(strdup(t))))
	  goto errexit;
	else
	  i++;
	p = q = 0;
	goto start;
      }
      return mklistv(i, xs);
    errexit:
      for (j = 0; j < i; j++) dispose(xs[j]);
      free(xs);
      return __ERROR;
    }
#else
    int l = strlen(s);
    if (l < 0 || l > INT_MAX/sizeof(expr))
      return __ERROR;
    else if (l == 0)
      return mknil;
    else {
      expr *xs = malloc(l*sizeof(expr));
      if (xs) {
	int i = 0, j;
	while(*s) {
	  char *t = malloc(2*sizeof(char));
	  if (t) {
	    t[0] = *(s++);
	    t[1] = 0;
	    if (!(xs[i] = mkstr(t)))
	      goto errexit;
	    else
	      i++;
	  } else
	    goto errexit;
	}
	return mklistv(l, xs);
      errexit:
	for (j = 0; j < i; j++) dispose(xs[j]);
	free(xs);
	return __ERROR;
      } else
	return __ERROR;
    }
#endif
  } else
    return __FAIL;
}

FUNCTION(clib,join,argc,argv)
{
  expr x, hd, tl;
  char *s, *t, *delim;
  int n, l, k, init;
  if (argc != 2 || !isstr(argv[0], &delim)) return __FAIL;
  k = strlen(delim);
  for (init = n = 0, x = argv[1]; iscons(x, &hd, &tl); x = tl) {
    if (isstr(hd, &s)) {
      l = strlen(s);
      if (init) {
	if (l < 0 || l >= INT_MAX-k)
	  return __ERROR;
	l += k;
      }
      if (l < 0 || n >= INT_MAX-l)
	return __ERROR;
      else
	n += l;
      init = 1;
    } else
      return __FAIL;
  }
  if (!isnil(x)) return __FAIL;
  if (!(t = malloc(n+1)))
    return __ERROR;
  *t = 0;
  for (init = n = 0, x = argv[1]; iscons(x, &hd, &tl); x = tl)
    if (isstr(hd, &s)) {
      if (init) {
	strcpy(t+n, delim);
	n += k;
      }
      strcpy(t+n, s);
      n += strlen(s);
      init = 1;
    }
  return mkstr(t);
}

/* This is essentially the same algorithm as in string.q, but implemented
   in C for greater efficiency. */

FUNCTION(clib,split,argc,argv)
{
  char *s, *delim;
  if (argc == 2 && isstr(argv[0], &delim) && isstr(argv[1], &s)) {
#ifdef HAVE_UNICODE
    if (*s == 0)
      return mknil;
    else {
      int k = 100;
      expr *xs = malloc(k*sizeof(expr));
      char *r = s;
      int i = 0, j;
      unsigned p = 0, q = 0;
      if (!xs) return __ERROR;
    start:
      for (; *s; s++) {
	unsigned char uc = (unsigned char)*s;
	if (q == 0) {
	  if (((signed char)uc) < 0) {
	    switch (uc & 0xf0) {
	    case 0xc0: case 0xd0:
	      q = 1;
	      break;
	    case 0xe0:
	      q = 2;
	      break;
	    case 0xf0:
	      if ((uc & 0x8) == 0)
		q = 3;
	      break;
	    }
	  }
	  p = 0;
	  if (q == 0) {
	    if (strchr(delim, *s)) {
	      char *u;
	      if (i >= k) {
		expr *xs1 = realloc(xs, (k+100)*sizeof(expr));
		if (xs1) {
		  xs = xs1;
		  k += 100;
		} else
		  goto errexit;
	      }
	      u = malloc(s-r+1);
	      if (!u)
		goto errexit;
	      strncpy(u, r, s-r);
	      u[s-r] = 0;
	      if (!(xs[i] = mkstr(u)))
		goto errexit;
	      else
		i++;
	      r = s+1;
	    }
	  }
	} else if ((uc & 0xc0) == 0x80) {
	  /* continuation byte */
	  if (--q == 0) {
	    char t[5];
	    strncpy(t, s-p-1, p+2);
	    t[p+2] = 0;
	    if (strstr(delim, t)) {
	      int l = s-p-1-r;
	      char *u;
	      if (i >= k) {
		expr *xs1 = realloc(xs, (k+100)*sizeof(expr));
		if (xs1) {
		  xs = xs1;
		  k += 100;
		} else
		  goto errexit;
	      }
	      u = malloc(l+1);
	      if (!u)
		goto errexit;
	      strncpy(u, r, l);
	      u[l] = 0;
	      if (!(xs[i] = mkstr(u)))
		goto errexit;
	      else
		i++;
	      r = s+1;
	    }
	    p = 0;
	  } else
	    p++;
	} else {
	  /* malformed char */
	  s -= p+1;
	  if (strchr(delim, *s)) {
	    char *u;
	    if (i >= k) {
	      expr *xs1 = realloc(xs, (k+100)*sizeof(expr));
	      if (xs1) {
		xs = xs1;
		k += 100;
	      } else
		goto errexit;
	    }
	    u = malloc(s-r+1);
	    if (!u)
	      goto errexit;
	    strncpy(u, r, s-r);
	    u[s-r] = 0;
	    if (!(xs[i] = mkstr(u)))
	      goto errexit;
	    else
	      i++;
	    r = s+1;
	  }
	  p = q = 0;
	}
      }
      if (q > 0) {
	/* unterminated char */
	s -= p;
	if (strchr(delim, *s)) {
	  char *u;
	  if (i >= k) {
	    expr *xs1 = realloc(xs, (k+3)*sizeof(expr));
	    if (xs1) {
	      xs = xs1;
	      k += 3;
	    } else
	      goto errexit;
	  }
	  u = malloc(s-r+1);
	  if (!u)
	    goto errexit;
	  strncpy(u, r, s-r);
	  u[s-r] = 0;
	  if (!(xs[i] = mkstr(u)))
	    goto errexit;
	  else
	    i++;
	  r = s+1;
	}
	p = q = 0;
	goto start;
      }
      if (s > r || i > 0) {
	char *u;
	if (i >= k) {
	  expr *xs1 = realloc(xs, (k+1)*sizeof(expr));
	  if (xs1) {
	    xs = xs1;
	    k += 1;
	  } else
	    goto errexit;
	}
	u = malloc(s-r+1);
	if (!u)
	  goto errexit;
	strcpy(u, r);
	if (!(xs[i] = mkstr(u)))
	  goto errexit;
	else
	  i++;
      }
      return mklistv(i, xs);
    errexit:
      for (j = 0; j < i; j++) dispose(xs[j]);
      free(xs);
      return __ERROR;
    }
#else
    char *t;
    expr *xs;
    int n, l, i, j;
    l = strlen(s);
    if (l < 0)
      return __FAIL;
    else if (l == 0)
      return mknil;
    i = j = n = 0;
    while (i < l) {
      if (j < l && !strchr(delim, s[j]))
	j++;
      else {
	n++;
	if (j == l-1) {
	  /* trailing delimiter */
	  n++;
	  break;
	} else
	  i = ++j;
      }
    }
    if (n > INT_MAX/sizeof(expr)) return __ERROR;
    if (!(xs = malloc(n*sizeof(expr)))) return __ERROR;
    i = j = n = 0;
    while (i < l) {
      if (j < l && !strchr(delim, s[j]))
	j++;
      else {
	if (!(t = malloc(j-i+1))) goto errexit;
	strncpy(t, s+i, j-i);
	t[j-i] = 0;
	if (!(xs[n] = mkstr(t))) goto errexit;
	n++;
	if (j == l-1) {
	  /* trailing delimiter */
	  if (!(t = malloc(1))) goto errexit;
	  *t = 0;
	  if (!(xs[n] = mkstr(t))) goto errexit;
	  n++;
	  break;
	} else
	  i = ++j;
      }
    }
    return mklistv(n, xs);
  errexit:
    for (i = 0; i < n; i++) dispose(xs[i]);
    free(xs);
    return __ERROR;
#endif
  } else
    return __FAIL;
}

FUNCTION(clib,strcat,argc,argv)
{
  expr x, hd, tl;
  char *s, *t;
  int n, l;
  if (argc != 1) return __FAIL;
  for (n = 0, x = argv[0]; iscons(x, &hd, &tl); x = tl) {
    if (isstr(hd, &s)) {
      l = strlen(s);
      if (l < 0 || n >= INT_MAX-l)
	return __ERROR;
      else
	n += l;
    } else
      return __FAIL;
  }
  if (!isnil(x)) return __FAIL;
  if (!(t = malloc(n+1)))
    return __ERROR;
  *t = 0;
  for (n = 0, x = argv[0]; iscons(x, &hd, &tl); x = tl)
    if (isstr(hd, &s)) {
      strcpy(t+n, s);
      n += strlen(s);
    }
  return mkstr(t);
}

static expr pred;
static int err;

static int eval_pred(const void *x, const void *y)
{
  expr t = mkapp(mkapp(pred, *((expr*)x)), *((expr*)y)), p;
  if (t && (p = eval(t))) {
    int ret;
    if (istrue(p))
      ret = 1;
    else if (isfalse(p))
      ret = 0;
    else {
      /* values are incomparable */
      err = 1;
      ret = 0;
    }
    dispose(p);
    return ret;
  } else {
    /* fatal error */
    err = -1;
    return 0;
  }
}

static int cmp_p(const void *x, const void *y)
{
  if (err)
    return 0;
  else if (eval_pred(x, y))
    return -1;
  else if (err)
    return 0;
  else if (eval_pred(y, x))
    return 1;
  else
    return 0;
}

FUNCTION(clib,sort,argc,argv)
{
  if (argc == 2) {
    expr *xs, p = argv[0], x, hd, tl;
    int n;
    for (n = 0, x = argv[1]; iscons(x, &hd, &tl); x = tl) {
      if (n >= INT_MAX/sizeof(expr))
	return __ERROR;
      else
	n++;
    }
    if (!isnil(x)) return __FAIL;
    if (!(xs = malloc(n*sizeof(expr))))
      return __ERROR;
    for (n = 0, x = argv[1]; iscons(x, &hd, &tl); x = tl)
      xs[n++] = hd;
    err = 0;
    pred = p;
    qsort(xs, n, sizeof(expr), cmp_p);
    if (err) {
      free(xs);
      return (err==-1)?__ERROR:__FAIL;
    } else
      return mklistv(n, xs);
  } else
    return __FAIL;
}

/* additional string functions: *******************************************/

FUNCTION(clib,islower,argc,argv)
{
  char *s;
#ifdef HAVE_UNICODE
  long c;
  if (argc == 1 && isstr(argv[0], &s) && (c = u8decode(s)) >= 0)
    if (iswlower(towchar(c)))
#else
  if (argc == 1 && isstr(argv[0], &s) && s[0] && !s[1])
    if (islower(s[0]))
#endif
      return mktrue;
    else
      return mkfalse;
  else
    return __FAIL;
}

FUNCTION(clib,isupper,argc,argv)
{
  char *s;
#ifdef HAVE_UNICODE
  long c;
  if (argc == 1 && isstr(argv[0], &s) && (c = u8decode(s)) >= 0)
    if (iswupper(towchar(c)))
#else
  if (argc == 1 && isstr(argv[0], &s) && s[0] && !s[1])
    if (isupper(s[0]))
#endif
      return mktrue;
    else
      return mkfalse;
  else
    return __FAIL;
}

FUNCTION(clib,isalpha,argc,argv)
{
  char *s;
#ifdef HAVE_UNICODE
  long c;
  if (argc == 1 && isstr(argv[0], &s) && (c = u8decode(s)) >= 0)
    if (iswalpha(towchar(c)))
#else
  if (argc == 1 && isstr(argv[0], &s) && s[0] && !s[1])
    if (isalpha(s[0]))
#endif
      return mktrue;
    else
      return mkfalse;
  else
    return __FAIL;
}

FUNCTION(clib,isdigit,argc,argv)
{
  char *s;
#ifdef HAVE_UNICODE
  long c;
  if (argc == 1 && isstr(argv[0], &s) && (c = u8decode(s)) >= 0)
    if (iswdigit(towchar(c)))
#else
  if (argc == 1 && isstr(argv[0], &s) && s[0] && !s[1])
    if (isdigit(s[0]))
#endif
      return mktrue;
    else
      return mkfalse;
  else
    return __FAIL;
}

FUNCTION(clib,isxdigit,argc,argv)
{
  char *s;
#ifdef HAVE_UNICODE
  long c;
  if (argc == 1 && isstr(argv[0], &s) && (c = u8decode(s)) >= 0)
    if (iswxdigit(towchar(c)))
#else
  if (argc == 1 && isstr(argv[0], &s) && s[0] && !s[1])
    if (isxdigit(s[0]))
#endif
      return mktrue;
    else
      return mkfalse;
  else
    return __FAIL;
}

FUNCTION(clib,isalnum,argc,argv)
{
  char *s;
#ifdef HAVE_UNICODE
  long c;
  if (argc == 1 && isstr(argv[0], &s) && (c = u8decode(s)) >= 0)
    if (iswalnum(towchar(c)))
#else
  if (argc == 1 && isstr(argv[0], &s) && s[0] && !s[1])
    if (isalnum(s[0]))
#endif
      return mktrue;
    else
      return mkfalse;
  else
    return __FAIL;
}

FUNCTION(clib,ispunct,argc,argv)
{
  char *s;
#ifdef HAVE_UNICODE
  long c;
  if (argc == 1 && isstr(argv[0], &s) && (c = u8decode(s)) >= 0)
    if (iswpunct(towchar(c)))
#else
  if (argc == 1 && isstr(argv[0], &s) && s[0] && !s[1])
    if (ispunct(s[0]))
#endif
      return mktrue;
    else
      return mkfalse;
  else
    return __FAIL;
}

FUNCTION(clib,isspace,argc,argv)
{
  char *s;
#ifdef HAVE_UNICODE
  long c;
  if (argc == 1 && isstr(argv[0], &s) && (c = u8decode(s)) >= 0)
    if (iswspace(towchar(c)))
#else
  if (argc == 1 && isstr(argv[0], &s) && s[0] && !s[1])
    if (isspace(s[0]))
#endif
      return mktrue;
    else
      return mkfalse;
  else
    return __FAIL;
}

FUNCTION(clib,isgraph,argc,argv)
{
  char *s;
#ifdef HAVE_UNICODE
  long c;
  if (argc == 1 && isstr(argv[0], &s) && (c = u8decode(s)) >= 0)
    if (iswgraph(towchar(c)))
#else
  if (argc == 1 && isstr(argv[0], &s) && s[0] && !s[1])
    if (isgraph(s[0]))
#endif
      return mktrue;
    else
      return mkfalse;
  else
    return __FAIL;
}

FUNCTION(clib,isprint,argc,argv)
{
  char *s;
#ifdef HAVE_UNICODE
  long c;
  if (argc == 1 && isstr(argv[0], &s) && (c = u8decode(s)) >= 0)
    if (iswprint(towchar(c)))
#else
  if (argc == 1 && isstr(argv[0], &s) && s[0] && !s[1])
    if (isprint(s[0]))
#endif
      return mktrue;
    else
      return mkfalse;
  else
    return __FAIL;
}

FUNCTION(clib,iscntrl,argc,argv)
{
  char *s;
#ifdef HAVE_UNICODE
  long c;
  if (argc == 1 && isstr(argv[0], &s) && (c = u8decode(s)) >= 0)
    if (iswcntrl(towchar(c)))
#else
  if (argc == 1 && isstr(argv[0], &s) && s[0] && !s[1])
    if (iscntrl(s[0]))
#endif
      return mktrue;
    else
      return mkfalse;
  else
    return __FAIL;
}

FUNCTION(clib,isascii,argc,argv)
{
  char *s;
#ifdef HAVE_UNICODE
  long c;
  if (argc == 1 && isstr(argv[0], &s) && (c = u8decode(s)) >= 0)
    if (c <= 0xff && isascii((char)c))
#else
  if (argc == 1 && isstr(argv[0], &s) && s[0] && !s[1])
    if (isascii(s[0]))
#endif
      return mktrue;
    else
      return mkfalse;
  else
    return __FAIL;
}

FUNCTION(clib,tolower,argc,argv)
{
  char *s;
  if (argc == 1 && isstr(argv[0], &s)) {
    int l = strlen(s);
#ifdef HAVE_UNICODE
    char *r, *t, *t0;
    if (l > (INT_MAX-1)/5 || !(t = malloc(5*l+1)))
      return __ERROR;
    t0 = t;
    while (*s) {
      long c = u8decodes(s, &r);
      if (c < 0) {
	free(t0);
	return __FAIL;
      } else {
#ifdef HAVE_TOWLOWER
	c = towlower(towchar(c));
#else
	if (c < 256)
	  c = tolower((int)c);
#endif
	u8encode(t, c);
	t += strlen(t);
	s = r;
      }
    }
    *t = 0;
    t = realloc(t0, strlen(t0)+1);
    if (!t) {
      free(t0);
      return __ERROR;
    } else
      return mkstr(t);
#else
    char *t = strdup(s);
    if (t) {
      int i;
      for (i = 0; i < l; i++)
	t[i] = tolower(t[i]);
      return mkstr(t);
    } else
      return __ERROR;
#endif
  } else
    return __FAIL;
}

FUNCTION(clib,toupper,argc,argv)
{
  char *s;
  if (argc == 1 && isstr(argv[0], &s)) {
    int l = strlen(s);
#ifdef HAVE_UNICODE
    char *r, *t, *t0;
    if (l > (INT_MAX-1)/5 || !(t = malloc(5*l+1)))
      return __ERROR;
    t0 = t;
    while (*s) {
      long c = u8decodes(s, &r);
      if (c < 0) {
	free(t0);
	return __FAIL;
      } else {
#ifdef HAVE_TOWUPPER
	c = towupper(towchar(c));
#else
	if (c < 256)
	  c = toupper((int)c);
#endif
	u8encode(t, c);
	t += strlen(t);
	s = r;
      }
    }
    *t = 0;
    t = realloc(t0, strlen(t0)+1);
    if (!t) {
      free(t0);
      return __ERROR;
    } else
      return mkstr(t);
#else
    char *t = strdup(s);
    if (t) {
      int i;
      for (i = 0; i < l; i++)
	t[i] = toupper(t[i]);
      return mkstr(t);
    } else
      return __ERROR;
#endif
  } else
    return __FAIL;
}

/* additional integer functions: ******************************************/

FUNCTION(clib,pow,argc,argv)
{
  mpz_t m, u;
  unsigned long n;
  if (argc != 2 || !ismpz(argv[0], m) || !isuint(argv[1], &n) ||
      n > 0 && mpz_size(m) > INT_MAX/n)
    return __FAIL;
  if (!mpz_new(u, n*mpz_size(m)))
    return __ERROR;
  mpz_pow_ui(u, m, n);
  if (!mpz_actsize(u))
    return __ERROR;
  return mkmpz(u);
}

FUNCTION(clib,root,argc,argv)
{
  mpz_t m, u;
  unsigned long n;
  if (argc != 2 || !ismpz(argv[0], m) || !isuint(argv[1], &n) || n == 0 ||
      mpz_sgn(m) == -1 && (n&1) == 0)
    return __FAIL;
  if (!mpz_new(u, mpz_size(m)/n+1))
    return __ERROR;
  mpz_root(u, m, n);
  if (!mpz_actsize(u))
    return __ERROR;
  return mkmpz(u);
}

FUNCTION(clib,intsqrt,argc,argv)
{
  mpz_t m, u;
  if (argc != 1 || !ismpz(argv[0], m) || mpz_sgn(m) < 0)
    return __FAIL;
  if (!mpz_new(u, (mpz_size(m)>>2)+1))
    return __ERROR;
  mpz_sqrt(u, m);
  if (!mpz_actsize(u))
    return __ERROR;
  return mkmpz(u);
}

FUNCTION(clib,powmod,argc,argv)
{
  mpz_t k, k2, m, n, u;
  int sgn;
  if (argc != 3 || !ismpz(argv[0], k) || !ismpz(argv[1], m) ||
      !ismpz(argv[2], n) || mpz_sgn(n) < 0 ||
      mpz_sgn(k) == 0)
    return __FAIL;
  if (!mpz_new(u, mpz_size(k)))
    return __ERROR;
  sgn = mpz_sgn(k);
  if (!mpz_copy(k2, k)) { mpz_clear(u); return __ERROR; }
  if (sgn < 0) k2->_mp_size = -k2->_mp_size;
  mpz_powm(u, m, n, k2);
  mpz_clear(k2);
  if (!mpz_actsize(u))
    return __ERROR;
  return mkmpz(u);
}

FUNCTION(clib,invmod,argc,argv)
{
  mpz_t k, m, u;
  if (argc != 2 || !ismpz(argv[0], k) || !ismpz(argv[1], m) ||
      mpz_sgn(k) == 0)
    return __FAIL;
  if (!mpz_new(u, mpz_size(k)))
    return __ERROR;
  if (!mpz_invert(u, m, k)) {
    mpz_clear(u);
    return __FAIL;
  }
  if (!mpz_actsize(u))
    return __ERROR;
  return mkmpz(u);
}

FUNCTION(clib,isprime,argc,argv)
{
  static long rep = 0;
  mpz_t n;
  int ret;
  if (argc != 1 || !ismpz(argv[0], n))
    return __FAIL;
  if (!rep) {
    /* get this value only once when we're first invoked, to avoid the
       overhead in subsequent calls */
    expr x = eval(mksym(sym(ISPRIME_REP)));
    if (x) {
      if (!isint(x, &rep) || rep <= 0) rep = 5;
      dispose(x);
    } else 
      rep = 5;
  }
  ret = mpz_probab_prime_p(n, rep);
  if (ret == 2)
    return mktrue;
  else if (ret == 0)
    return mkfalse;
  else
    return __FAIL;
}

FUNCTION(clib,gcd,argc,argv)
{
  mpz_t m, n, u;
  if (argc != 2 || !ismpz(argv[0], m) || !ismpz(argv[1], n) ||
      mpz_sgn(m) == 0 && mpz_sgn(n) == 0)
    return __FAIL;
  if (!mpz_new(u, long_min(mpz_size(m),mpz_size(n))))
    return __ERROR;
  mpz_gcd(u, m, n);
  if (!mpz_actsize(u))
    return __ERROR;
  return mkmpz(u);
}

FUNCTION(clib,lcm,argc,argv)
{
  mpz_t m, n, u;
  if (argc != 2 || !ismpz(argv[0], m) || !ismpz(argv[1], n) ||
      /*mpz_sgn(m) == 0 || mpz_sgn(n) == 0 ||*/
      mpz_size(m)+mpz_size(n) < 0)
    return __FAIL;
  if (!mpz_new(u, mpz_size(m)+mpz_size(n)))
    return __ERROR;
  mpz_lcm(u, m, n);
  if (!mpz_actsize(u))
    return __ERROR;
  return mkmpz(u);
}

FUNCTION(clib,remove_factor,argc,argv)
{
  mpz_t m, n, u;
  unsigned long ret;
  long k;
  if (argc != 2 || !ismpz(argv[0], m) || !ismpz(argv[1], n) ||
      mpz_sgn(m) == 0 || mpz_sgn(n) <= 0 ||
      isint(argv[1], &k) && k == 1)
    return __FAIL;
  if (!mpz_new(u, mpz_size(m)))
    return __ERROR;
  ret = mpz_remove(u, m, n);
  if (!mpz_actsize(u))
    return __ERROR;
  return mktuplel(2, mkuint(ret), mkmpz(u));
}

FUNCTION(clib,jacobi,argc,argv)
{
  mpz_t m, n;
  int ret;
  if (argc != 2 || !ismpz(argv[0], m) || !ismpz(argv[1], n) ||
      mpz_sgn(n) <= 0)
    return __FAIL;
  ret = mpz_jacobi(m, n);
  return mkint(ret);
}

/* enhanced file functions: ***********************************************/

FUNCTION(clib,fopen,argc,argv)
{
  char *name, *mode;
  FILE *fp;
  if (argc != 2 || !isstr(argv[0], &name) || !isstr(argv[1], &mode))
    return __FAIL;
  if (!strchr("rwa", mode[0]) ||
      mode[1] && !strchr("b+", mode[1]) ||
      mode[1] && mode[2] && (mode[1] == mode[2] || !strchr("b+", mode[2])))
    return __FAIL;
  if (!(name = utf8_to_sys(name)))
    return __ERROR;
  release_lock();
  fp = fopen(name, mode);
  acquire_lock();
  free(name);
  if (fp)
    return mkfile(fp);
  else
    return __FAIL;
}

FUNCTION(clib,fdopen,argc,argv)
{
  long fd;
  char *mode;
  FILE *fp;
  if (argc != 2 || !isint(argv[0], &fd) || !isstr(argv[1], &mode))
    return __FAIL;
  if (!strchr("rwa", mode[0]) ||
      mode[1] && !strchr("b+", mode[1]) ||
      mode[1] && mode[2] && (mode[1] == mode[2] || !strchr("b+", mode[2])))
    return __FAIL;
  fp = fdopen(fd, mode);
  if (fp)
    return mkfile(fp);
  else
    return __FAIL;
}

FUNCTION(clib,freopen,argc,argv)
{
  char *name, *mode;
  FILE *fp;
  if (argc != 3 || !isstr(argv[0], &name) || !isstr(argv[1], &mode) ||
      !isfile(argv[2], &fp))
    return __FAIL;
  if (!strchr("rwa", mode[0]) ||
      mode[1] && !strchr("b+", mode[1]) ||
      mode[1] && mode[2] && (mode[1] == mode[2] || !strchr("b+", mode[2])))
    return __FAIL;
  if (!(name = utf8_to_sys(name)))
    return __ERROR;
  release_lock();
  fp = freopen(name, mode, fp);
  acquire_lock();
  free(name);
  if (fp)
    return argv[2];
  else
    return __FAIL;
}

FUNCTION(clib,fileno,argc,argv)
{
  FILE *fp;
  if (argc != 1 || !isfile(argv[0], &fp))
    return __FAIL;
  else
    return mkint(fileno(fp));
}

FUNCTION(clib,setvbuf,argc,argv)
{
  FILE *fp;
  long mode;
  if (argc != 2 || !isfile(argv[0], &fp) || !isint(argv[1], &mode) ||
      setvbuf(fp, NULL, mode, 0))
    return __FAIL;
  else
    return mkvoid;
}

FUNCTION(clib,fconv,argc,argv)
{
  FILE *fp;
  char *codeset;
  if (argc != 2 || !isfile(argv[0], &fp) || !isstr(argv[1], &codeset) ||
      !file_encoding(argv[0], codeset))
    return __FAIL;
  else
    return mkvoid;
}

FUNCTION(clib,tmpnam,argc,argv)
{
  if (argc == 0) {
#ifdef __MINGW32__
    /* win32 tmpnam is broken, provide a reasonable replacement */
    char *s = _tempnam("\\tmp", "t"), *t, *p;
    if (!s || !(t = sys_to_utf8(s))) {
      if (s) free(s);
      return __ERROR;
    }
    free(s);
    /* make sure to convert all \'s to /'s */
    while ((p = strchr(t, '\\'))) *p = '/';
    return mkstr(t);
#else
    char s[L_tmpnam];
    tmpnam(s);
    return mkstr(sys_to_utf8(s));
#endif
  } else
    return __FAIL;
}

FUNCTION(clib,tmpfile,argc,argv)
{
  if (argc == 0) {
    FILE * fp = tmpfile();
    if (fp)
      return mkfile(fp);
    else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(clib,ftell,argc,argv)
{
  FILE *fp;
  long res;
  if (argc != 1 || !isfile(argv[0], &fp))
    return __FAIL;
  res = ftell(fp);
  if (res >= 0)
    return mkint(res);
  else
    return __FAIL;
}

FUNCTION(clib,fseek,argc,argv)
{
  FILE *fp;
  long pos, whence;
  int res;
  if (argc != 3 || !isfile(argv[0], &fp) || !isint(argv[1], &pos) ||
      !isint(argv[2], &whence))
    return __FAIL;
  res = fseek(fp, pos, whence);
  if (res)
    return __FAIL;
  else
    return mkvoid;
}

FUNCTION(clib,fgets,argc,argv)
{
  FILE *fp;
  char *s, *t, *r;
  int a, l;

  if (argc != 1 || !isfile(argv[0], &fp))
    return __FAIL;
  s = malloc(BUFSZ); t = s;
  a = BUFSZ;
  if (!s) return __ERROR;
  *s = 0;
  release_lock();
  if (fp == stdin) acquire_tty();
  while ((r = fgets(t, BUFSZ, fp)) && *t &&
	 t[(l = strlen(t))-1] != '\n') {
    /* try to enlarge the buffer: */
    int k = t-s+l;
    char *s1;
    if (s1 = (char*) realloc(s, a+BUFSZ)) {
      s = s1;
      t = s+k;
      a += BUFSZ;
    } else {
      free(s);
      if (fp == stdin) release_tty();
      acquire_lock();
      return __ERROR;
    }
  }
  if (ferror(fp)) {
    clearerr(fp);
    free(s);
    if (fp == stdin) release_tty();
    acquire_lock();
    return __FAIL;
  }
  if ((t = file_to_utf8(s, argv[0]))) {
    free(s);
    s = t;
  } else {
    free(s);
    if (fp == stdin) release_tty();
    acquire_lock();
    return __ERROR;
  }
  if (!r && !*s) {
    free(s);
    if (fp == stdin) release_tty();
    acquire_lock();
    return __FAIL;
  }
  if (fp == stdin) release_tty();
  acquire_lock();
  return mkstr(s);
}

FUNCTION(clib,gets,argc,argv)
{
  if (argc == 0) {
    expr input = eval(mksym(sym(INPUT)));
    if (input) {
      expr argv1[1] = { input };
      expr ret = FUNCALL(clib, fgets, 1, argv1);
      dispose(input);
      return ret;
    } else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(clib,fget,argc,argv)
{
  FILE *fp;
  char *s, *t;
  int a;

  if (argc != 1 || !isfile(argv[0], &fp))
    return __FAIL;
  s = malloc(BUFSZ); t = s;
  a = BUFSZ;
  if (!s) return __ERROR;
  *s = 0;
  release_lock();
  if (fp == stdin) acquire_tty();
  while (fgets(t, BUFSZ, fp)) {
    /* try to enlarge the buffer: */
    int l = strlen(t), k = t-s+l;
    char *s1;
    if (s1 = (char*) realloc(s, a+BUFSZ)) {
      s = s1;
      t = s+k;
      a += BUFSZ;
    } else {
      free(s);
      if (fp == stdin) release_tty();
      acquire_lock();
      return __ERROR;
    }
  }
  if (ferror(fp)) {
    clearerr(fp);
    free(s);
    if (fp == stdin) release_tty();
    acquire_lock();
    return __FAIL;
  }
  if ((t = file_to_utf8(s, argv[0]))) {
    free(s);
    s = t;
  } else {
    free(s);
    if (fp == stdin) release_tty();
    acquire_lock();
    return __ERROR;
  }
  if (fp == stdin) release_tty();
  acquire_lock();
  return mkstr(s);
}

FUNCTION(clib,ungetc,argc,argv)
{
  FILE *fp;
  char *s;
#ifdef HAVE_UNICODE
  if (argc == 1 && isstr(argv[0], &s) && u8decode(s) >= 0) {
#else
  if (argc == 1 && isstr(argv[0], &s) && s[0] && !s[1]) {
#endif
    expr input = eval(mksym(sym(INPUT)));
    int i;
    if (!isfile(input, &fp))
      return __FAIL;
    if (!(s = utf8_to_sys(s)))
      return __ERROR;
    for (i = strlen(s)-1; i>=0; --i)
      if (ungetc(s[i], fp) == EOF) {
	free(s);
	return __FAIL;
      }
    free(s);
    return mkvoid;
  } else
    return __FAIL;
}

FUNCTION(clib,fungetc,argc,argv)
{
  FILE *fp;
  char *s;
#ifdef HAVE_UNICODE
  if (argc == 2 && isfile(argv[0], &fp) && isstr(argv[1], &s) &&
      u8decode(s) >= 0) {
#else
  if (argc == 2 && isfile(argv[0], &fp) && isstr(argv[1], &s) &&
      s[0] && !s[1]) {
#endif
    int i;
    if (!(s = utf8_to_sys(s)))
      return __ERROR;
    for (i = strlen(s)-1; i>=0; --i)
      if (ungetc(s[i], fp) == EOF) {
	free(s);
	return __FAIL;
      }
    free(s);
    return mkvoid;
  } else
    return __FAIL;
}

/* C-style formatted I/O: *************************************************/

#define F_SZ 1024

static int f_err, f_var_wd, f_var_prec;
static int f_wd, f_prec, f_sz;

static char *f_ptr, *f_str, f_format[F_SZ], f_flags[F_SZ], f_mod[F_SZ];
static char f_wd_str[F_SZ], f_prec_str[F_SZ];

static void f_init(char *format)
{
  f_ptr = format;
  f_err = 0;
  if (!format && f_sz > 4*F_SZ) {
    size_t m = F_SZ+1;
    char *f_str1 = realloc(f_str, m);
    if (f_str1) {
      f_str = f_str1;
      f_sz = m;
    }
  }
}

static int set_f_str(char *s, size_t n)
{
  size_t m = (n>F_SZ)?n:F_SZ;
  if (m+1 < m) return 0;
  m++;
  if (!f_str) {
    f_str = malloc(m);
    if (f_str) f_sz = m;
  } else if (f_sz < m) {
    char *f_str1 = realloc(f_str, m);
    if (f_str1) {
      f_str = f_str1;
      f_sz = m;
    }
  }
  if (n >= m) return 0;
  strncpy(f_str, s, n);
  f_str[n] = 0;
  return 1;
}

#define F_SZ_CHK(len) if (len>=F_SZ) { f_err = 1; return 0; } else
#define SET_F_STR(s,len) if (!set_f_str(s,len)) { f_err = 1; return 0; }

/* scan printf format string */

static char f_parse_pf(void)
{
  char c, *p, *q, *r;
  f_var_wd = f_var_prec = f_wd = f_prec = 0;
  for (p = strchr(f_ptr, '%'); p && p[1] == '%'; p = strchr(p+2, '%'))
    ;
  if (!p) {
    /* no more conversions, return the rest of the string */
    SET_F_STR(f_ptr, strlen(f_ptr));
    *f_format = *f_flags = *f_mod = 0;
    f_ptr = f_ptr + strlen(f_ptr);
    return 0;
  }
  /* conversion starts at p, parse it */
  SET_F_STR(f_ptr, p-f_ptr);
  q = r = p+1;
  /* flags */
  while (strchr("#0- +", *r)) r++;
  F_SZ_CHK(r-q) strncpy(f_flags, q, r-q);
  f_flags[r-q] = 0;
  q = r;
  /* field width */
  if (*r == '*') {
    r++;
    f_var_wd = 1;
    *f_wd_str = 0;
  } else {
    while (isdigit(*r)) r++;
    F_SZ_CHK(r-q) strncpy(f_wd_str, q, r-q);
    f_wd_str[r-q] = 0;
  }
  if (*f_wd_str) f_wd = atoi(f_wd_str);
  q = r;
  /* precision */
  if (*r == '.') {
    r++;
    if (*r == '*') {
      r++;
      f_var_prec = 1;
      *f_prec_str = 0;
    } else {
      while (isdigit(*r)) r++;
      F_SZ_CHK(r-q) strncpy(f_prec_str, q, r-q);
      f_prec_str[r-q] = 0;
    }
    if (*f_prec_str) f_prec = atoi(f_prec_str);
  }
  q = r;
  /* length modifier */
  while (strchr("hl", *r)) r++;
  F_SZ_CHK(r-q) strncpy(f_mod, q, r-q);
  f_mod[r-q] = 0;
  q = r;
  /* conversion specifier */
  c = *(r++);
  F_SZ_CHK(r-p) strncpy(f_format, p, r-p);
  f_format[r-p] = 0;
  f_ptr = r;
  return c;
}

/* scan scanf format string */

static char f_parse_sf(void)
{
  char c, *p, *q, *r;
  f_wd = -1;
  for (p = strchr(f_ptr, '%'); p && p[1] == '%'; p = strchr(p+2, '%'))
    ;
  if (!p) {
    /* no more conversions, return the rest of the string */
    SET_F_STR(f_ptr, strlen(f_ptr));
    *f_format = *f_flags = *f_mod = 0;
    f_ptr = f_ptr + strlen(f_ptr);
    return 0;
  }
  /* conversion starts at p, parse it */
  SET_F_STR(f_ptr, p-f_ptr);
  q = r = p+1;
  /* flags */
  while (strchr("*", *r)) r++;
  F_SZ_CHK(r-q) strncpy(f_flags, q, r-q);
  f_flags[r-q] = 0;
  q = r;
  /* field width */
  while (isdigit(*r)) r++;
  F_SZ_CHK(r-q) strncpy(f_wd_str, q, r-q);
  f_wd_str[r-q] = 0;
  if (*f_wd_str) f_wd = atoi(f_wd_str);
  q = r;
  /* length modifier */
  while (strchr("hl", *r)) r++;
  F_SZ_CHK(r-q) strncpy(f_mod, q, r-q);
  f_mod[r-q] = 0;
  q = r;
  /* conversion specifier */
  c = *(r++);
  if (c == '[') {
    /* character class */
    if (*r == ']')
      r++;
    else if (r[0] == '^' && r[1] == ']')
      r += 2;
    while (*r && *r != ']') r++;
    if (*r == ']')
      r++;
    else {
      f_err = 1;
      c = 0;
    }
  }
  F_SZ_CHK(r-p) strncpy(f_format, p, r-p);
  f_format[r-p] = 0;
  f_ptr = r;
  return c;
}

static int coerce_uint(expr x, unsigned long *u)
{
  mpz_t z;
  double d;
  if (ismpz(x, z)) {
    if (mpz_sgn(z) != 0)
      *u = z->_mp_d[0];
    else
      *u = 0;
    if (mpz_sgn(z) < 0)
      *u = ~(*u)+1U;
    return 1;
  } else if (isfloat(x, &d)) {
    *u = (unsigned long)((d>=0.0)?d:(-d));
    if (d < 0)
      *u = ~(*u)+1U;
    return 1;
  } else
    return 0;
}

static inline int coerce_int(expr x, long *i)
{
  return coerce_uint(x, (unsigned long*)i);
}

static int coerce_float(expr x, double *d)
{
  if (isfloat(x, d) || ismpz_float(x, d))
    return 1;
  else
    return 0;
}

#define CALL_FPRINTF(fp,val) \
((f_var_wd && f_var_prec)? \
fprintf(fp, f_format, (int)wdval, (int)precval, val): \
(f_var_wd)? \
fprintf(fp, f_format, (int)wdval, val): \
(f_var_prec)? \
fprintf(fp, f_format, (int)precval, val): \
fprintf(fp, f_format, val))

FUNCTION(clib,fprintf,argc,argv)
{
  expr x, *xs;
  int i, n, ret;
  char f, *format;
  FILE *fp;

  long intval, wdval, precval;
  unsigned long uintval;
  double dblval;
  char *strval;

  if (argc != 3 || !isfile(argv[0], &fp) || !isstr(argv[1], &format))
    return __FAIL;
  format = utf8_to_sys(format);
  if (!format) return __ERROR;
  lock_format();
  if (!istuple(argv[2], &n, &xs)) {
    x = argv[2];
    xs = &x;
    n = 1;
  }
  /* parse the format string and check arguments */
  f_init(format);
  i = 0;
  while ((f = f_parse_pf())) {
    if (*f_mod)
      if (strcmp(f_mod, "l") && strcmp(f_mod, "h") ||
	  !strchr("diouxX", f)) {
	unlock_format();
	free(format);
	return __FAIL;
      }
    if (f_var_wd)
      if ((i >= n || !coerce_int(xs[i], &wdval))) {
	unlock_format();
	free(format);
	return __FAIL;
      } else
	i++;
    if (f_var_prec)
      if ((i >= n || !coerce_int(xs[i], &precval))) {
	unlock_format();
	free(format);
	return __FAIL;
      } else
	i++;
    switch (f) {
    case 'd': case 'i':
      if (i >= n || !coerce_int(xs[i], &intval)) {
	unlock_format();
	free(format);
	return __FAIL;
      } else
	i++;
      break;
    case 'o': case 'u': case 'x': case 'X':
      if (i >= n || !coerce_uint(xs[i], &uintval)) {
	unlock_format();
	free(format);
	return __FAIL;
      } else
	i++;
      break;
    case 'e': case 'E':
    case 'f':
    case 'g': case 'G':
      if (i >= n || !coerce_float(xs[i], &dblval)) {
	unlock_format();
	free(format);
	return __FAIL;
      } else
	i++;
      break;
    case 'c':
      if (i >= n || !isstr(xs[i], &strval)) {
	unlock_format();
	free(format);
	return __FAIL;
      } else if (!(strval = utf8_to_sys(strval))) {
	unlock_format();
	free(format);
	return __ERROR;
      } else if (!strval[0] || strval[1]) {
	free(strval);
	unlock_format();
	free(format);
	return __FAIL;
      } else
	i++;
      free(strval);
      break;
    case 's':
      if (i >= n || !isstr(xs[i], &strval)) {
	unlock_format();
	free(format);
	return __FAIL;
      } else
	i++;
      break;
    default:
      unlock_format();
      free(format);
      return __FAIL;
    }
  }
  if (f_err) {
    unlock_format();
    free(format);
    return __FAIL;
  }
  /* print */
  f_init(format);
  i = 0;
  while ((f = f_parse_pf())) {
    if (fprintf(fp, f_str) < 0) goto errexit;
    if (f_var_wd)
      coerce_int(xs[i++], &wdval);
    if (f_var_prec)
      coerce_int(xs[i++], &precval);
    ret = 0;
    switch (f) {
    case 'd': case 'i':
      coerce_int(xs[i++], &intval);
      if (*f_mod == 'l')
	ret = CALL_FPRINTF(fp, intval);
      else if (*f_mod == 'h')
	ret = CALL_FPRINTF(fp, (short)intval);
      else
	ret = CALL_FPRINTF(fp, (int)intval);
      break;
    case 'o': case 'u': case 'x': case 'X':
      coerce_uint(xs[i++], &uintval);
      if (*f_mod == 'l')
	ret = CALL_FPRINTF(fp, uintval);
      else if (*f_mod == 'h')
	ret = CALL_FPRINTF(fp, (unsigned short)uintval);
      else
	ret = CALL_FPRINTF(fp, (unsigned)uintval);
      break;
    case 'e': case 'E':
    case 'f':
    case 'g': case 'G':
      coerce_float(xs[i++], &dblval);
      ret = CALL_FPRINTF(fp, dblval);
      break;
    case 'c':
      isstr(xs[i++], &strval);
      if (!(strval = utf8_to_sys(strval))) goto errexit2;
      ret = CALL_FPRINTF(fp, (int)*strval);
      free(strval);
      break;
    case 's':
      isstr(xs[i++], &strval);
      if (!(strval = utf8_to_sys(strval))) goto errexit2;
      ret = CALL_FPRINTF(fp, strval);
      free(strval);
      break;
    }
    if (ret < 0) goto errexit;
  }
  if (fprintf(fp, f_str) < 0) goto errexit;
  f_init(NULL);
  unlock_format();
  free(format);
  return mkvoid;
 errexit:
  f_init(NULL);
  unlock_format();
  free(format);
  return __FAIL;
 errexit2:
  f_init(NULL);
  unlock_format();
  free(format);
  return __ERROR;
}

FUNCTION(clib,printf,argc,argv)
{
  char *format;
  if (argc == 2 && isstr(argv[0], &format)) {
    expr output = eval(mksym(sym(OUTPUT)));
    if (output) {
      expr argv1[3] = { output, argv[0], argv[1] };
      expr ret = FUNCALL(clib, fprintf, 3, argv1);
      dispose(output);
      return ret;
    } else
      return __FAIL;
  } else
    return __FAIL;
}

static char *buf = NULL, *bufptr = NULL;
static long alloc = 0, leng = 0;

static int addbuf(long newleng)
{
  if (!buf)
    alloc = leng = 0;
  else
    leng = strlen(buf);
  if (leng+newleng+1 <= 0) {
    if (!buf) bufptr = NULL;
    return 0;
  }
  while (leng+newleng >= alloc) {
    if (buf) {
      char *newbuf;
      if (alloc+BUFSZ <= 0 || !(newbuf = realloc(buf, alloc+BUFSZ))) {
	bufptr = buf+leng;
	return 0;
      } else {
	buf = newbuf;
	alloc += BUFSZ;
      }
    } else {
      if (!(buf = malloc(BUFSZ))) {
	bufptr = NULL;
	return 0;
      } else {
	alloc = BUFSZ;
	*buf = 0;
      }
    }
  }
  bufptr = buf+leng;
  return 1;
}

#define CALL_SPRINTF(s,val) \
((f_var_wd && f_var_prec)? \
sprintf(s, f_format, (int)wdval, (int)precval, val): \
(f_var_wd)? \
sprintf(s, f_format, (int)wdval, val): \
(f_var_prec)? \
sprintf(s, f_format, (int)precval, val): \
sprintf(s, f_format, val))

FUNCTION(clib,sprintf,argc,argv)
{
  expr x, *xs;
  int i, n, ret;
  long sz, k, l;
  char f, *format, *s;

  long intval, wdval, precval;
  unsigned long uintval;
  double dblval;
  char *strval;

  if (argc != 2 || !isstr(argv[0], &format))
    return __FAIL;
  format = utf8_to_sys(format);
  if (!format) return __ERROR;
  lock_format();
  if (!istuple(argv[1], &n, &xs)) {
    x = argv[1];
    xs = &x;
    n = 1;
  }
  /* parse the format string and check arguments, guestimate needed
     buffer size */
  f_init(format);
  i = 0; sz = BUFSZ;
  while ((f = f_parse_pf())) {
    if (*f_mod)
      if (strcmp(f_mod, "l") && strcmp(f_mod, "h") ||
	  !strchr("diouxX", f)) {
	unlock_format();
	free(format);
	return __FAIL;
      }
    if (f_var_wd)
      if ((i >= n || !coerce_int(xs[i], &wdval))) {
	unlock_format();
	free(format);
	return __FAIL;
      } else {
	f_wd = wdval;
	i++;
      }
    if (f_var_prec)
      if ((i >= n || !coerce_int(xs[i], &precval))) {
	unlock_format();
	free(format);
	return __FAIL;
      } else {
	f_prec = precval;
	i++;
      }
    if (f_wd < 0) f_wd = -f_wd;
    if (f_prec < 0) f_prec = 0;
    l = f_wd+f_prec;
    switch (f) {
    case 'd': case 'i':
      if (i >= n || !coerce_int(xs[i], &intval)) {
	unlock_format();
	free(format);
	return __FAIL;
      } else
	i++;
      if (BUFSZ+l > sz) sz = BUFSZ+l;
      break;
    case 'o': case 'u': case 'x': case 'X':
      if (i >= n || !coerce_uint(xs[i], &uintval)) {
	unlock_format();
	free(format);
	return __FAIL;
      } else
	i++;
      if (BUFSZ+l > sz) sz = BUFSZ+l;
      break;
    case 'e': case 'E':
    case 'f':
    case 'g': case 'G':
      if (i >= n || !coerce_float(xs[i], &dblval)) {
	unlock_format();
	free(format);
	return __FAIL;
      } else
	i++;
      if (f == 'f') {
	k = log10(abs(dblval)+1)+2;
	if (BUFSZ+l+k > sz) sz = BUFSZ+l+k;
      } else {
	if (BUFSZ+l > sz) sz = BUFSZ+l;
      }
      break;
    case 'c':
      if (i >= n || !isstr(xs[i], &strval)) {
	unlock_format();
	free(format);
	return __FAIL;
      } else if (!(strval = utf8_to_sys(strval))) {
	unlock_format();
	free(format);
	return __ERROR;
      } else if (!strval[0] || strval[1]) {
	free(strval);
	unlock_format();
	free(format);
	return __FAIL;
      } else
	i++;
      free(strval);
      if (l+1 > sz) sz = l+1;
      break;
    case 's':
      if (i >= n || !isstr(xs[i], &strval)) {
	unlock_format();
	free(format);
	return __FAIL;
      } else if (!(strval = utf8_to_sys(strval))) {
	unlock_format();
	free(format);
	return __ERROR;
      } else
	i++;
      k = strlen(strval);
      free(strval);
      if (k+l > sz) sz = k+l;
      break;
    default:
      unlock_format();
      free(format);
      return __FAIL;
    }
  }
  if (f_err) {
    unlock_format();
    free(format);
    return __FAIL;
  }
  /* allocate buffer */
  if (sz <= 0 || !(s = malloc(sz+1))) {
    unlock_format();
    free(format);
    return __ERROR;
  }
  buf = NULL;
  /* print */
  f_init(format);
  i = 0;
  while ((f = f_parse_pf())) {
    if (!addbuf(strlen(f_str))) goto errexit2;
    if (sprintf(bufptr, f_str) < 0) goto errexit;
    bufptr += strlen(bufptr);
    if (f_var_wd)
      coerce_int(xs[i++], &wdval);
    if (f_var_prec)
      coerce_int(xs[i++], &precval);
    *s = 0;
    switch (f) {
    case 'd': case 'i':
      coerce_int(xs[i++], &intval);
      if (*f_mod == 'l')
	ret = CALL_SPRINTF(s, intval);
      else if (*f_mod == 'h')
	ret = CALL_SPRINTF(s, (short)intval);
      else
	ret = CALL_SPRINTF(s, (int)intval);
      break;
    case 'o': case 'u': case 'x': case 'X':
      coerce_uint(xs[i++], &uintval);
      if (*f_mod == 'l')
	ret = CALL_SPRINTF(s, uintval);
      else if (*f_mod == 'h')
	ret = CALL_SPRINTF(s, (unsigned short)uintval);
      else
	ret = CALL_SPRINTF(s, (unsigned)uintval);
      break;
    case 'e': case 'E':
    case 'f':
    case 'g': case 'G':
      coerce_float(xs[i++], &dblval);
      ret = CALL_SPRINTF(s, dblval);
      break;
    case 'c':
      isstr(xs[i++], &strval);
      if (!(strval = utf8_to_sys(strval))) goto errexit2;
      ret = CALL_SPRINTF(s, (int)*strval);
      free(strval);
      break;
    case 's':
      isstr(xs[i++], &strval);
      if (!(strval = utf8_to_sys(strval))) goto errexit2;
      ret = CALL_SPRINTF(s, strval);
      free(strval);
      break;
    }
    if (ret < 0) goto errexit;
    if (!addbuf(strlen(s))) goto errexit2;
    strcpy(bufptr, s);
    bufptr += strlen(bufptr);
  }
  if (!addbuf(strlen(f_str))) goto errexit2;
  if (sprintf(bufptr, f_str) < 0) goto errexit;
  bufptr += strlen(bufptr);
  f_init(NULL);
  unlock_format();
  free(format);
  free(s);
  s = sys_to_utf8(buf);
  free(buf);
  return mkstr(s);
 errexit:
  f_init(NULL);
  unlock_format();
  free(format);
  if (buf) free(buf); free(s);
  return __FAIL;
 errexit2:
  f_init(NULL);
  unlock_format();
  free(format);
  if (buf) free(buf); free(s);
  return __ERROR;
}

#define DFLT_STRLEN 10240

/* scanning literals is a bit tricky, since there is no obvious sign
   indicating that the match has failed -- we have to check for non-ws chars
   in the template */
static int fscan_literal(FILE *fp, long *ncount, char *template)
{
  long count = -1;
  short empty = 1;
  char *p;

  for (p = template; *p; p++)
    if (!isspace(*p)) {
      empty = 0;
      break;
    }
  if (strlen(template) >= F_SZ-3) return 0;
#ifndef WIN32
  strcat(template, "%ln");
#else /* WIN32 */
  strcat(template, "%n");
#endif /* WIN32 */
  if (fscanf(fp, template, &count) < 0) return 0;
  if (count < 0) return 0;
  if (count == 0 && !empty) return 0;
  *ncount += count;
  return 1;
}

FUNCTION(clib,fscanf,argc,argv)
{
  expr *xs, x;
  int i, n, ret;
  long count, ncount = 0;
  char f, *format;
  FILE *fp;

  short shortval;
  unsigned short ushortval;
  int intval;
  unsigned uintval;
  long longval;
  unsigned long ulongval;
  float fltval;
  double dblval;
  char *strval;

  if (argc != 2 || !isfile(argv[0], &fp) || !isstr(argv[1], &format))
    return __FAIL;
  format = utf8_to_sys(format);
  if (!format) return __ERROR;
  release_lock();
  if (fp == stdin) acquire_tty();
  lock_format();
  f_init(format);
  i = 0;
  while ((f = f_parse_sf())) {
    if (*f_flags)
      if (strcmp(f_flags, "*")) {
	unlock_format();
	free(format);
	if (fp == stdin) release_tty();
	acquire_lock();
	return __FAIL;
      }
    if (*f_mod)
      if (strcmp(f_mod, "l") && strcmp(f_mod, "h") ||
	  *f_mod == 'l' && !strchr("ndiouxXeEfgG", f) ||
	  *f_mod == 'h' && !strchr("ndiouxX", f)) {
	unlock_format();
	free(format);
	if (fp == stdin) release_tty();
	acquire_lock();
	return __FAIL;
      }
    switch (f) {
    case 'n':
    case 'd': case 'i':
    case 'o': case 'u': case 'x': case 'X':
    case 'e': case 'E':
    case 'f':
    case 'g': case 'G':
    case 'c':
    case 's': case '[':
      if (!*f_flags) i++;
      break;
    default:
      unlock_format();
      free(format);
      if (fp == stdin) release_tty();
      acquire_lock();
      return __FAIL;
    }
  }
  if (f_err) {
    unlock_format();
    free(format);
    if (fp == stdin) release_tty();
    acquire_lock();
    return __FAIL;
  }
  n = i;
  if (n == 0)
    xs = NULL;
  else if (!(xs = malloc(n*sizeof(expr)))) {
    unlock_format();
    free(format);
    if (fp == stdin) release_tty();
    acquire_lock();
    return __ERROR;
  }
  f_init(format);
  i = 0;
  while ((f = f_parse_sf())) {
    if (!fscan_literal(fp, &ncount, f_str)) goto errexit;
    if (strlen(f_format) >= F_SZ-3) goto errexit;
#ifndef WIN32
    strcat(f_format, "%ln");
#else /* WIN32 */
    strcat(f_format, "%n");
#endif /* WIN32 */
    if (*f_flags) {
      count = -1; fscanf(fp, f_format, &count);
      if (count < 0) goto errexit;
      ncount += count;
      continue;
    }
    switch (f) {
    case 'n':
      if (*f_mod == 'h') {
	shortval = ncount;
	ret = (xs[i] = mkint(shortval)) != NULL;
      } else if (*f_mod == 'h') {
	longval = ncount;
	ret = (xs[i] = mkint(longval)) != NULL;
      } else {
	intval = ncount;
	ret = (xs[i] = mkint(intval)) != NULL;
      }
      if (!ret)
	goto errexit2;
      else
	i++;
      count = 0;
      break;
    case 'd': case 'i':
      count = -1;
      if (*f_mod == 'h')
	ret = !(fscanf(fp, f_format, &shortval, &count) > 0 && count >= 0)?1:
	  !(xs[i] = mkint(shortval))?-1:0;
      else if (*f_mod == 'l')
	ret = !(fscanf(fp, f_format, &longval, &count) > 0 && count >= 0)?1:
	  !(xs[i] = mkint(longval))?-1:0;
      else
	ret = !(fscanf(fp, f_format, &intval, &count) > 0 && count >= 0)?1:
	  !(xs[i] = mkint(intval))?-1:0;
      if (ret == -1)
	goto errexit2;
      else if (ret == 1)
	goto errexit;
      else
	i++;
      break;
    case 'o': case 'u': case 'x': case 'X':
      count = -1;
      if (*f_mod == 'h')
	ret = !(fscanf(fp, f_format, &ushortval, &count) > 0 && count >= 0)?1:
	  !(xs[i] = mkuint(ushortval))?-1:0;
      else if (*f_mod == 'l')
	ret = !(fscanf(fp, f_format, &ulongval, &count) > 0 && count >= 0)?1:
	  !(xs[i] = mkuint(ulongval))?-1:0;
      else
	ret = !(fscanf(fp, f_format, &uintval, &count) > 0 && count >= 0)?1:
	  !(xs[i] = mkuint(uintval))?-1:0;
      if (ret == -1)
	goto errexit2;
      else if (ret == 1)
	goto errexit;
      else
	i++;
      break;
    case 'e': case 'E':
    case 'f':
    case 'g': case 'G':
      count = -1;
      if (*f_mod == 'l')
	ret = !(fscanf(fp, f_format, &dblval, &count) > 0 && count >= 0)?1:
	  !(xs[i] = mkfloat(dblval))?-1:0;
      else
	ret = !(fscanf(fp, f_format, &fltval, &count) > 0 && count >= 0)?1:
	  !(xs[i] = mkfloat(fltval))?-1:0;
      if (ret == -1)
	goto errexit2;
      else if (ret == 1)
	goto errexit;
      else
	i++;
      break;
    case 'c':
      count = -1;
      if (f_wd <= 0) f_wd = 1;
      if (!(strval = malloc(f_wd+1))) goto errexit2;
      *strval = 0;
      if (fscanf(fp, f_format, strval, &count) <= 0 || count < 0) {
	free(strval);
	goto errexit;
      } else {
	char *s;
	strval[f_wd] = 0;
	s = sys_to_utf8(strval);
	free(strval);
	if (!(xs[i] = mkstr(s)))
	  goto errexit2;
	else
	  i++;
      }
      break;
    case 's': case '[':
      count = -1;
      if (f_wd <= 0) {
	char new_format[F_SZ];
	f_wd = DFLT_STRLEN;
	if (strlen(f_format)+10 > F_SZ) goto errexit;
	sprintf(new_format, "%%%d%s", f_wd, f_format+1);
	strcpy(f_format, new_format);
      }
      if (!(strval = malloc(f_wd+1))) goto errexit2;
      *strval = 0;
      if (fscanf(fp, f_format, strval, &count) <= 0 || count < 0) {
	free(strval);
	goto errexit;
      } else {
	char *s = sys_to_utf8(strval);
	free(strval);
	if (!(xs[i] = mkstr(s)))
	  goto errexit2;
	else
	  i++;
      }
      break;
    }
    ncount += count;
  }
  if (!fscan_literal(fp, &ncount, f_str)) goto errexit;
  unlock_format();
  free(format);
  if (fp == stdin) release_tty();
  acquire_lock();
  if (n == 0)
    return mkvoid;
  else if (n == 1) {
    x = xs[0]; free(xs);
    return x;
  } else
    return mktuplev(n, xs);
 errexit:
  unlock_format();
  free(format);
  if (fp == stdin) release_tty();
  acquire_lock();
  while (i > 0) dispose(xs[--i]);
  if (n > 0) free(xs);
  return __FAIL;
 errexit2:
  unlock_format();
  free(format);
  if (fp == stdin) release_tty();
  acquire_lock();
  while (i > 0) dispose(xs[--i]);
  if (n > 0) free(xs);
  return __ERROR;
}

FUNCTION(clib,scanf,argc,argv)
{
  char *format;
  if (argc == 1 && isstr(argv[0], &format)) {
    expr input = eval(mksym(sym(INPUT)));
    if (input) {
      expr argv1[2] = { input, argv[0] };
      expr ret = FUNCALL(clib, fscanf, 2, argv1);
      dispose(input);
      return ret;
    } else
      return __FAIL;
  } else
    return __FAIL;
}

static int sscan_literal(char **s, long *ncount, char *template)
{
  long count = -1;
  int empty = 1;
  char *p;

  for (p = template; *p; p++)
    if (!isspace(*p)) {
      empty = 0;
      break;
    }
  if (strlen(template) >= F_SZ-3) return 0;
#ifndef WIN32
  strcat(template, "%ln");
#else /* WIN32 */
  /* work around a bug in the msvc sscanf function -- it doesn't recognize
     %ln */
  strcat(template, "%n");
#endif /* WIN32 */
  if (sscanf(*s, template, &count) < 0) return 0;
  if (count < 0) return 0;
  if (count == 0 && !empty) return 0;
  *s += count;
  *ncount += count;
  return 1;
}

FUNCTION(clib,sscanf,argc,argv)
{
  expr *xs, x;
  int i, n, ret;
  long count, ncount = 0;
  char f, *format;
  char *s, *s0;

  short shortval;
  unsigned short ushortval;
  int intval;
  unsigned uintval;
  long longval;
  unsigned long ulongval;
  float fltval;
  double dblval;
  char *strval;

  if (argc != 2 || !isstr(argv[0], &s) || !isstr(argv[1], &format))
    return __FAIL;
  format = utf8_to_sys(format);
  if (!format) return __ERROR;
  s0 = s = utf8_to_sys(s);
  if (!s) {
    free(format);
    return __ERROR;
  }
  release_lock();
  lock_format();
  f_init(format);
  i = 0;
  while ((f = f_parse_sf())) {
    if (*f_flags)
      if (strcmp(f_flags, "*")) {
	unlock_format();
	free(format); free(s0);
	acquire_lock();
	return __FAIL;
      }
    if (*f_mod)
      if (strcmp(f_mod, "l") && strcmp(f_mod, "h") ||
	  *f_mod == 'l' && !strchr("ndiouxXeEfgG", f) ||
	  *f_mod == 'h' && !strchr("ndiouxX", f)) {
	unlock_format();
	free(format); free(s0);
	acquire_lock();
	return __FAIL;
      }
    switch (f) {
    case 'n':
    case 'd': case 'i':
    case 'o': case 'u': case 'x': case 'X':
    case 'e': case 'E':
    case 'f':
    case 'g': case 'G':
    case 'c':
    case 's': case '[':
      if (!*f_flags) i++;
      break;
    default:
      unlock_format();
      free(format); free(s0);
      acquire_lock();
      return __FAIL;
    }
  }
  if (f_err) {
    unlock_format();
    free(format); free(s0);
    acquire_lock();
    return __FAIL;
  }
  n = i;
  if (n == 0)
    xs = NULL;
  else if (!(xs = malloc(n*sizeof(expr)))) {
    unlock_format();
    free(format); free(s0);
    acquire_lock();
    return __ERROR;
  }
  f_init(format);
  i = 0;
  while ((f = f_parse_sf())) {
    if (!sscan_literal(&s, &ncount, f_str)) goto errexit;
    if (strlen(f_format) >= F_SZ-3) goto errexit;
#ifndef WIN32
    strcat(f_format, "%ln");
#else /* WIN32 */
    strcat(f_format, "%n");
#endif /* WIN32 */
    if (*f_flags) {
      count = -1; sscanf(s, f_format, &count);
      if (count < 0) goto errexit;
      s += count; ncount += count;
      continue;
    }
    switch (f) {
    case 'n':
      if (*f_mod == 'h') {
	shortval = ncount;
	ret = (xs[i] = mkint(shortval)) != NULL;
      } else if (*f_mod == 'h') {
	longval = ncount;
	ret = (xs[i] = mkint(longval)) != NULL;
      } else {
	intval = ncount;
	ret = (xs[i] = mkint(intval)) != NULL;
      }
      if (!ret)
	goto errexit2;
      else
	i++;
      count = 0;
      break;
    case 'd': case 'i':
      count = -1;
      if (*f_mod == 'h')
	ret = !(sscanf(s, f_format, &shortval, &count) > 0 && count >= 0)?1:
	  !(xs[i] = mkint(shortval))?-1:0;
      else if (*f_mod == 'l')
	ret = !(sscanf(s, f_format, &longval, &count) > 0 && count >= 0)?1:
	  !(xs[i] = mkint(longval))?-1:0;
      else
	ret = !(sscanf(s, f_format, &intval, &count) > 0 && count >= 0)?1:
	  !(xs[i] = mkint(intval))?-1:0;
      if (ret == -1)
	goto errexit2;
      else if (ret == 1)
	goto errexit;
      else
	i++;
      break;
    case 'o': case 'u': case 'x': case 'X':
      count = -1;
      if (*f_mod == 'h')
	ret = !(sscanf(s, f_format, &ushortval, &count) > 0 && count >= 0)?1:
	  !(xs[i] = mkuint(ushortval))?-1:0;
      else if (*f_mod == 'l')
	ret = !(sscanf(s, f_format, &ulongval, &count) > 0 && count >= 0)?1:
	  !(xs[i] = mkuint(ulongval))?-1:0;
      else
	ret = !(sscanf(s, f_format, &uintval, &count) > 0 && count >= 0)?1:
	  !(xs[i] = mkuint(uintval))?-1:0;
      if (ret == -1)
	goto errexit2;
      else if (ret == 1)
	goto errexit;
      else
	i++;
      break;
    case 'e': case 'E':
    case 'f':
    case 'g': case 'G':
      count = -1;
      if (*f_mod == 'l')
	ret = !(sscanf(s, f_format, &dblval, &count) > 0 && count >= 0)?1:
	  !(xs[i] = mkfloat(dblval))?-1:0;
      else
	ret = !(sscanf(s, f_format, &fltval, &count) > 0 && count >= 0)?1:
	  !(xs[i] = mkfloat(fltval))?-1:0;
      if (ret == -1)
	goto errexit2;
      else if (ret == 1)
	goto errexit;
      else
	i++;
      break;
    case 'c':
      count = -1;
      if (f_wd == -1) f_wd = 1;
      if (f_wd > strlen(s)) goto errexit;
      if (!(strval = malloc(f_wd+1))) goto errexit2;
      *strval = 0;
      if (sscanf(s, f_format, strval, &count) <= 0 || count < 0) {
	free(strval);
	goto errexit;
      } else {
	char *s;
	strval[f_wd] = 0;
	s = sys_to_utf8(strval);
	free(strval);
	if (!(xs[i] = mkstr(s)))
	  goto errexit2;
	else
	  i++;
      }
      break;
    case 's': case '[':
      count = -1;
      if (f_wd == -1) {
	char new_format[F_SZ];
	f_wd = DFLT_STRLEN;
	if (strlen(f_format)+10 > F_SZ) goto errexit;
	sprintf(new_format, "%%%d%s", f_wd, f_format+1);
	strcpy(f_format, new_format);
      }
      if (!(strval = malloc(f_wd+1))) goto errexit2;
      *strval = 0;
      if (sscanf(s, f_format, strval, &count) <= 0 || count < 0) {
	free(strval);
	goto errexit;
      } else {
	char *s = sys_to_utf8(strval);
	free(strval);
	if (!(xs[i] = mkstr(s)))
	  goto errexit2;
	else
	  i++;
      }
      break;
    default:
      goto errexit;
    }
    s += count; ncount += count;
  }
  if (!sscan_literal(&s, &ncount, f_str)) goto errexit;
  unlock_format();
  free(format); free(s0);
  acquire_lock();
  if (n == 0)
    return mkvoid;
  else if (n == 1) {
    x = xs[0]; free(xs);
    return x;
  } else
    return mktuplev(n, xs);
 errexit:
  unlock_format();
  free(format); free(s0);
  acquire_lock();
  while (i > 0) dispose(xs[--i]);
  if (n > 0) free(xs);
  return __FAIL;
 errexit2:
  unlock_format();
  free(format); free(s0);
  acquire_lock();
  while (i > 0) dispose(xs[--i]);
  if (n > 0) free(xs);
  return __ERROR;
}

/* byte strings: ************************************************************/

typedef struct bstr {
  long size;
  unsigned char *v;
} bstr_t;

DESTRUCTOR(clib,ByteStr,ptr)
{
  bstr_t *m = (bstr_t*)ptr;
  if (m) {
    if (m->v) free(m->v);
    free(m);
  }
}

static expr mkbstr(long size, void *v)
{
  bstr_t *m;
  if ((m = malloc(sizeof(bstr_t)))) {
    m->size = size;
    m->v = (unsigned char*)v;
    return mkobj(type(ByteStr), m);
  } else {
    if (v) free(v);
    return __ERROR;
  }
}

FUNCTION(clib,bytestr,argc,argv)
{
  expr x, hd, tl, *xv;
  int k;
  long count = -1;
  mpz_t z;
  double d;
  float f;
  char *s, *codeset = NULL;
  unsigned char *v = NULL;
  if (argc != 1) return __FAIL;
  if (iscons(argv[0], &hd, &tl) || isnil(argv[0])) {
    long n, b;
    for (n = 0, x = argv[0];
	 iscons(x, &hd, &tl) && isint(hd, &b) && b>=0 && b<=0xff;
	 x = tl)
      if (n > 0 && n+1 <= 0)
	return __ERROR;
      else
	n++;
    if (!isnil(x)) return __FAIL;
    count = n;
    if (count && !(v = malloc(count)))
      return __ERROR;
    if (v) memset(v, 0, count);
    for (n = 0, x = argv[0];
	 n < count && iscons(x, &hd, &tl) && isint(hd, &b);
	 x = tl)
      v[n++] = (unsigned char)b;
    return mkbstr(count, v);
  }
  if (istuple(argv[0], &k, &xv)) {
    if (k == 2) {
      if (isint(xv[1], &count)) {
	if (count < 0) return __FAIL;
      } else if (!isstr(xv[0], &s) || !isstr(xv[1], &codeset))
	return __FAIL;
    } else if (k == 3 && isstr(xv[0], &s) && isstr(xv[1], &codeset) &&
	       isint(xv[2], &count)) {
      if (count < 0) return __FAIL;
    } else
      return __FAIL;
    x = xv[0];
  } else
    x = argv[0];
  if (ismpz(x, z)) {
    unsigned long n;
    long c;
    unsigned char pad_byte = 0;
    mpz_t u;
    n = mpz_size(z);
    if (count < 0) count = ((n>0)?n:1)*sizeof(mp_limb_t);
    c = count;
    if (c % sizeof(mp_limb_t) > 0)
      c = (c/sizeof(mp_limb_t)+1)*sizeof(mp_limb_t);
    if (c < 0) return __ERROR;
    if (c && !(v = malloc(c)))
      return __ERROR;
    if (!mpz_new(u, n)) {
      if (v) free(v);
      return __ERROR;
    }
    mpz_set(u, z);
    if (mpz_sgn(z) < 0) {
      unsigned long j;
      pad_byte = 0xff;
      /* 2's complement */
      for (j = 0; j < n; j++) u->_mp_d[j] = ~u->_mp_d[j];
      mpn_add_1(u->_mp_d, u->_mp_d, n, 1U);
    }
    if (c/sizeof(mp_limb_t) < n) n = c/sizeof(mp_limb_t);
    if (v) {
      memset(v, pad_byte, count);
      memcpy(v, u->_mp_d, n*sizeof(mp_limb_t));
#ifdef WORDS_BIGENDIAN
      /* correct last limb on big-endian systems */
      if (count < c && c/sizeof(mp_limb_t) == n)
	memmove(v+c-sizeof(mp_limb_t), v+c-sizeof(mp_limb_t)+(c-count),
		sizeof(mp_limb_t)-(c-count));
#endif
      if (count != c) {
	unsigned char *v1 = realloc(v, count);
	if (v1) v = v1;
      }
    }
    mpz_clear(u);
  } else if (isfloat(x, &d)) {
    f = (float)d;
    if (count < 0) count = sizeof(double);
    if (count && !(v = malloc(count)))
      return __ERROR;
    if (v) memset(v, 0, count);
    if (count >= sizeof(double))
      memcpy(v, &d, sizeof(double));
    else if (count >= sizeof(float))
      memcpy(v, &f, sizeof(float));
    else
      memcpy(v, &f, count);
  } else if (isstr(x, &s)) {
    if (!(s = from_utf8(s, codeset)))
      return __ERROR;
    if (count < 0) count = strlen(s);
    if (count && !(v = malloc(count))) {
      free(s);
      return __ERROR;
    }
    if (v) memset(v, 0, count);
    if (v) strncpy((char*)v, s, count);
    free(s);
  } else
    return __FAIL;
  return mkbstr(count, v);
}

FUNCTION(clib,bcat,argc,argv)
{
  bstr_t *m;
  unsigned char *v = NULL;
  expr x, hd, tl;
  long n;
  if (argc != 1) return __FAIL;
  for (n = 0, x = argv[0];
       iscons(x, &hd, &tl) && isobj(hd, type(ByteStr), (void**)&m);
       x = tl)
    if (n > 0 && n+m->size <= 0)
      return __ERROR;
    else
      n += m->size;
  if (!isnil(x)) return __FAIL;
  if (n && !(v = malloc(n)))
    return __ERROR;
  for (n = 0, x = argv[0];
       iscons(x, &hd, &tl) && isobj(hd, type(ByteStr), (void**)&m);
       x = tl, n += m->size)
    memcpy(v+n, m->v, m->size);
  return mkbstr(n, v);
}

FUNCTION(clib,bsize,argc,argv)
{
  bstr_t *m;
  if (argc == 1 && isobj(argv[0], type(ByteStr), (void**)&m))
    return mkint(m->size);
  else
    return __FAIL;
}

FUNCTION(clib,byte,argc,argv)
{
  bstr_t *m;
  long i;
  if (argc == 2 && isint(argv[0], &i) &&
      isobj(argv[1], type(ByteStr), (void**)&m) &&
      i >= 0 && i < m->size)
    return mkint(m->v[i]);
  else
    return __FAIL;
}

FUNCTION(clib,bsub,argc,argv)
{
  bstr_t *m;
  void *v = NULL;
  long i, j, c, l;
  if (argc == 3 && isobj(argv[0], type(ByteStr), (void**)&m) &&
      isint(argv[1], &i) && isint(argv[2], &j)) {
    if (i < 0) i = 0; c = j-i+1;
    l = m->size;
    if (i >= l || j < i)
      l = 0;
    else if ((l -= i) > c)
      l = c;
    if (l < 0)
      l = 0;
    if (l && !(v = malloc(l)))
      return __ERROR;
    if (l) memcpy(v, m->v+i, l);
    return mkbstr(l, v);
  } else
    return __FAIL;
}

FUNCTION(clib,bcmp,argc,argv)
{
  bstr_t *m1, *m2;
  if (argc == 2 && isobj(argv[0], type(ByteStr), (void**)&m1) &&
      isobj(argv[1], type(ByteStr), (void**)&m2))
    if (!m1->v)
      if (!m2->v)
	return mkint(0);
      else
	return mkint(-1);
    else if (!m2->v)
      return mkint(1);
    else {
      long i, n = m1->size, res;
      if (n > m2->size) n = m2->size;
      res = memcmp(m1->v, m2->v, n);
      if (res == 0)
	if (m1->size < m2->size)
	  res = -1;
      else if (m1->size > m2->size)
	  res = 1;
      return mkint(res);
    }
  else
    return __FAIL;
}

FUNCTION(clib,bint,argc,argv)
{
  bstr_t *m;
  if (argc == 1 && isobj(argv[0], type(ByteStr), (void**)&m)) {
    long n = m->size/sizeof(mp_limb_t), k = m->size%sizeof(mp_limb_t), c;
    mpz_t z;
    if (k > 0) n++;
    if (mpz_new(z, n)) {
      memset(z->_mp_d, 0, n*sizeof(mp_limb_t));
#ifdef WORDS_BIGENDIAN
      c = n*sizeof(mp_limb_t)-m->size;
      if (c > 0) {
	memcpy(z->_mp_d, m->v, m->size-k);
	memcpy(((unsigned char*)z->_mp_d)+(n-1)*sizeof(mp_limb_t)+c,
	       m->v+(n-1)*sizeof(mp_limb_t),
	       k);
      } else
#endif
	memcpy(z->_mp_d, m->v, m->size);
      while (n > 0 && !z->_mp_d[n-1]) n--;
      z->_mp_size = n;
      if (!mpz_actsize(z))
	return __FAIL;
      else
	return mkmpz(z);
    } else
      return __ERROR;
  } else
    return __FAIL;
}

FUNCTION(clib,bfloat,argc,argv)
{
  bstr_t *m;
  if (argc == 1 && isobj(argv[0], type(ByteStr), (void**)&m)) {
    if (m->size >= sizeof(double)) {
      double d;
      memcpy(&d, m->v, sizeof(double));
      return mkfloat(d);
    } else if (m->size >= sizeof(float)) {
      float f;
      memcpy(&f, m->v, sizeof(float));
      return mkfloat((double)f);
    } else {
      float f;
      memset(&f, 0, sizeof(float));
      memcpy(&f, m->v, m->size);
      return mkfloat((double)f);
    }
  } else
    return __FAIL;
}

FUNCTION(clib,bstr,argc,argv)
{
  expr *xv;
  int k;
  bstr_t *m;
  char *codeset = NULL;
  if (argc == 1 &&
      (isobj(argv[0], type(ByteStr), (void**)&m) ||
       istuple(argv[0], &k, &xv) && k == 2 &&
       isobj(xv[0], type(ByteStr), (void**)&m) &&
       isstr(xv[1], &codeset))) {
    char *s = malloc(m->size+1), *s1;
    if (!s) return __ERROR;
    if (m->size) memcpy(s, m->v, m->size);
    s[m->size] = 0;
    s1 = to_utf8(s, codeset);
    free(s);
    return mkstr(s1);
  } else
    return __FAIL;
}

static inline long get_clamp(long *i, long *j, long m)
{
  if (*i < 0) *i = 0;
  if (*j < *i) *j = *i-1;
  if (*j >= m) {
    *j = m-1;
    if (*i > *j) *i = *j+1;
  }
  return *j-*i+1;
}

static inline long put_clamp(long *i, long *j, long k, long m)
{
  long l = k;
  if (k < 0) l = k = 0;
  if (*i < 0) { k += *i; *j -= *i; *i = 0; }
  if (*i > m) *i = m;
  if (k > m-*i) k = m-*i;
  if (k < 0) k = 0;
  if (*j < 0) *j = 0;
  if (*j > l) *j = l;
  return k;
}

FUNCTION(clib,get_int8,argc,argv)
{
  bstr_t *m;
  expr *xv;
  int n;
  long i, j;
  if (argc == 2 && isobj(argv[0], type(ByteStr), (void**)&m)) {
    signed char *u = m->v;
    if (isint(argv[1], &i) && i >= 0 && i < m->size)
      return mkint(u[i]);
    else if (istuple(argv[1], &n, &xv) && n==2 &&
	     isint(xv[0], &i) && isint(xv[1], &j)) {
      long k = get_clamp(&i, &j, m->size);
      if (k > INT_MAX)
	return __ERROR;
      else if (k > 0) {
	signed char *v = (signed char*)malloc(k);
	if (!v) return __ERROR;
	memcpy(v, u+i, k);
	return mkbstr(k, v);
      } else
	return mkbstr(0, NULL);
    } else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(clib,put_int8,argc,argv)
{
  bstr_t *m, *m1;
  long i, x;
  if (argc == 3 && isobj(argv[0], type(ByteStr), (void**)&m) &&
      isint(argv[1], &i)) {
    signed char *u = m->v;
    if (isint(argv[2], &x) && i >= 0 && i < m->size) {
      u[i] = (signed char)x;
      return mkvoid;
    } else if (isobj(argv[2], type(ByteStr), (void**)&m1)) {
      signed char *v = m1->v;
      long j = 0, k = put_clamp(&i, &j, m1->size, m->size);
      if (k > 0) memcpy(u+i, v+j, k);
      return mkvoid;
    } else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(clib,get_int16,argc,argv)
{
  bstr_t *m;
  expr *xv;
  int n;
  long i, j;
  if (argc == 2 && isobj(argv[0], type(ByteStr), (void**)&m)) {
    long N = m->size/sizeof(short);
    short *u = (short*)m->v;
    if (isint(argv[1], &i) && i >= 0 && i < N)
      return mkint(u[i]);
    else if (istuple(argv[1], &n, &xv) && n==2 &&
	     isint(xv[0], &i) && isint(xv[1], &j)) {
      long k = get_clamp(&i, &j, N);
      if (k > INT_MAX)
	return __ERROR;
      else if (k > 0) {
	short *v = (short*)malloc(k*sizeof(short));
	if (!v) return __ERROR;
	memcpy(v, u+i, k*sizeof(short));
	return mkbstr(k*sizeof(short), v);
      } else
	return mkbstr(0, NULL);
    } else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(clib,put_int16,argc,argv)
{
  bstr_t *m, *m1;
  long i, x;
  if (argc == 3 && isobj(argv[0], type(ByteStr), (void**)&m) &&
      isint(argv[1], &i)) {
    long N = m->size/sizeof(short);
    short *u = (short*)m->v;
    if (isint(argv[2], &x) && i >= 0 && i < N) {
      u[i] = (short)x;
      return mkvoid;
    } else if (isobj(argv[2], type(ByteStr), (void**)&m1)) {
      long M = m1->size/sizeof(short);
      short *v = (short*)m1->v;
      long j = 0, k = put_clamp(&i, &j, M, N);
      if (k > 0) memcpy(u+i, v+j, k*sizeof(short));
      return mkvoid;
    } else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(clib,get_int32,argc,argv)
{
  bstr_t *m;
  expr *xv;
  int n;
  long i, j;
  if (argc == 2 && isobj(argv[0], type(ByteStr), (void**)&m)) {
    long N = m->size/sizeof(long);
    long *u = (long*)m->v;
    if (isint(argv[1], &i) && i >= 0 && i < N)
      return mkint(u[i]);
    else if (istuple(argv[1], &n, &xv) && n==2 &&
	     isint(xv[0], &i) && isint(xv[1], &j)) {
      long k = get_clamp(&i, &j, N);
      if (k > INT_MAX)
	return __ERROR;
      else if (k > 0) {
	long *v = (long*)malloc(k*sizeof(long));
	if (!v) return __ERROR;
	memcpy(v, u+i, k*sizeof(long));
	return mkbstr(k*sizeof(long), v);
      } else
	return mkbstr(0, NULL);
    } else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(clib,put_int32,argc,argv)
{
  bstr_t *m, *m1;
  long i, x;
  if (argc == 3 && isobj(argv[0], type(ByteStr), (void**)&m) &&
      isint(argv[1], &i)) {
    long N = m->size/sizeof(long);
    long *u = (long*)m->v;
    if (isint(argv[2], &x) && i >= 0 && i < N) {
      u[i] = (long)x;
      return mkvoid;
    } else if (isobj(argv[2], type(ByteStr), (void**)&m1)) {
      long M = m1->size/sizeof(long);
      long *v = (long*)m1->v;
      long j = 0, k = put_clamp(&i, &j, M, N);
      if (k > 0) memcpy(u+i, v+j, k*sizeof(long));
      return mkvoid;
    } else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(clib,get_uint8,argc,argv)
{
  bstr_t *m;
  expr *xv;
  int n;
  long i, j;
  if (argc == 2 && isobj(argv[0], type(ByteStr), (void**)&m)) {
    unsigned char *u = m->v;
    if (isint(argv[1], &i) && i >= 0 && i < m->size)
      return mkuint(u[i]);
    else if (istuple(argv[1], &n, &xv) && n==2 &&
	     isint(xv[0], &i) && isint(xv[1], &j)) {
      long k = get_clamp(&i, &j, m->size);
      if (k > INT_MAX)
	return __ERROR;
      else if (k > 0) {
	unsigned char *v = (unsigned char*)malloc(k);
	if (!v) return __ERROR;
	memcpy(v, u+i, k);
	return mkbstr(k, v);
      } else
	return mkbstr(0, NULL);
    } else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(clib,put_uint8,argc,argv)
{
  bstr_t *m, *m1;
  long i;
  unsigned long x;
  if (argc == 3 && isobj(argv[0], type(ByteStr), (void**)&m) &&
      isint(argv[1], &i)) {
    unsigned char *u = m->v;
    if (isuint(argv[2], &x) && i >= 0 && i < m->size) {
      u[i] = (unsigned char)x;
      return mkvoid;
    } else if (isobj(argv[2], type(ByteStr), (void**)&m1)) {
      unsigned char *v = m1->v;
      long j = 0, k = put_clamp(&i, &j, m1->size, m->size);
      if (k > 0) memcpy(u+i, v+j, k);
      return mkvoid;
    } else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(clib,get_uint16,argc,argv)
{
  bstr_t *m;
  expr *xv;
  int n;
  long i, j;
  if (argc == 2 && isobj(argv[0], type(ByteStr), (void**)&m)) {
    long N = m->size/sizeof(unsigned short);
    unsigned short *u = (unsigned short*)m->v;
    if (isint(argv[1], &i) && i >= 0 && i < N)
      return mkuint(u[i]);
    else if (istuple(argv[1], &n, &xv) && n==2 &&
	     isint(xv[0], &i) && isint(xv[1], &j)) {
      long k = get_clamp(&i, &j, N);
      if (k > INT_MAX)
	return __ERROR;
      else if (k > 0) {
	unsigned short *v = (unsigned short*)malloc(k*sizeof(unsigned short));
	if (!v) return __ERROR;
	memcpy(v, u+i, k*sizeof(unsigned short));
	return mkbstr(k*sizeof(unsigned short), v);
      } else
	return mkbstr(0, NULL);
    } else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(clib,put_uint16,argc,argv)
{
  bstr_t *m, *m1;
  long i;
  unsigned long x;
  if (argc == 3 && isobj(argv[0], type(ByteStr), (void**)&m) &&
      isint(argv[1], &i)) {
    long N = m->size/sizeof(unsigned short);
    unsigned short *u = (unsigned short*)m->v;
    if (isuint(argv[2], &x) && i >= 0 && i < N) {
      u[i] = (unsigned short)x;
      return mkvoid;
    } else if (isobj(argv[2], type(ByteStr), (void**)&m1)) {
      long M = m1->size/sizeof(unsigned short);
      unsigned short *v = (unsigned short*)m1->v;
      long j = 0, k = put_clamp(&i, &j, M, N);
      if (k > 0) memcpy(u+i, v+j, k*sizeof(unsigned short));
      return mkvoid;
    } else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(clib,get_uint32,argc,argv)
{
  bstr_t *m;
  expr *xv;
  int n;
  long i, j;
  if (argc == 2 && isobj(argv[0], type(ByteStr), (void**)&m)) {
    long N = m->size/sizeof(unsigned long);
    unsigned long *u = (unsigned long*)m->v;
    if (isint(argv[1], &i) && i >= 0 && i < N)
      return mkuint(u[i]);
    else if (istuple(argv[1], &n, &xv) && n==2 &&
	     isint(xv[0], &i) && isint(xv[1], &j)) {
      long k = get_clamp(&i, &j, N);
      if (k > INT_MAX)
	return __ERROR;
      else if (k > 0) {
	unsigned long *v = (unsigned long*)malloc(k*sizeof(unsigned long));
	if (!v) return __ERROR;
	memcpy(v, u+i, k*sizeof(unsigned long));
	return mkbstr(k*sizeof(unsigned long), v);
      } else
	return mkbstr(0, NULL);
    } else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(clib,put_uint32,argc,argv)
{
  bstr_t *m, *m1;
  long i;
  unsigned long x;
  if (argc == 3 && isobj(argv[0], type(ByteStr), (void**)&m) &&
      isint(argv[1], &i)) {
    long N = m->size/sizeof(unsigned long);
    unsigned long *u = (unsigned long*)m->v;
    if (isuint(argv[2], &x) && i >= 0 && i < N) {
      u[i] = (unsigned long)x;
      return mkvoid;
    } else if (isobj(argv[2], type(ByteStr), (void**)&m1)) {
      long M = m1->size/sizeof(unsigned long);
      unsigned long *v = (unsigned long*)m1->v;
      long j = 0, k = put_clamp(&i, &j, M, N);
      if (k > 0) memcpy(u+i, v+j, k*sizeof(unsigned long));
      return mkvoid;
    } else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(clib,get_float,argc,argv)
{
  bstr_t *m;
  expr *xv;
  int n;
  long i, j;
  if (argc == 2 && isobj(argv[0], type(ByteStr), (void**)&m)) {
    long N = m->size/sizeof(float);
    float *u = (float*)m->v;
    if (isint(argv[1], &i) && i >= 0 && i < N)
      return mkfloat(u[i]);
    else if (istuple(argv[1], &n, &xv) && n==2 &&
	     isint(xv[0], &i) && isint(xv[1], &j)) {
      long k = get_clamp(&i, &j, N);
      if (k > INT_MAX)
	return __ERROR;
      else if (k > 0) {
	float *v = (float*)malloc(k*sizeof(float));
	if (!v) return __ERROR;
	memcpy(v, u+i, k*sizeof(float));
	return mkbstr(k*sizeof(float), v);
      } else
	return mkbstr(0, NULL);
    } else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(clib,put_float,argc,argv)
{
  bstr_t *m, *m1;
  long i;
  double x;
  if (argc == 3 && isobj(argv[0], type(ByteStr), (void**)&m) &&
      isint(argv[1], &i)) {
    long N = m->size/sizeof(float);
    float *u = (float*)m->v;
    if ((isfloat(argv[2], &x) || ismpz_float(argv[2], &x)) &&
	i >= 0 && i < N) {
      u[i] = (float)x;
      return mkvoid;
    } else if (isobj(argv[2], type(ByteStr), (void**)&m1)) {
      long M = m1->size/sizeof(float);
      float *v = (float*)m1->v;
      long j = 0, k = put_clamp(&i, &j, M, N);
      if (k > 0) memcpy(u+i, v+j, k*sizeof(float));
      return mkvoid;
    } else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(clib,get_double,argc,argv)
{
  bstr_t *m;
  expr *xv;
  int n;
  long i, j;
  if (argc == 2 && isobj(argv[0], type(ByteStr), (void**)&m)) {
    long N = m->size/sizeof(double);
    double *u = (double*)m->v;
    if (isint(argv[1], &i) && i >= 0 && i < N)
      return mkfloat(u[i]);
    else if (istuple(argv[1], &n, &xv) && n==2 &&
	     isint(xv[0], &i) && isint(xv[1], &j)) {
      long k = get_clamp(&i, &j, N);
      if (k > INT_MAX)
	return __ERROR;
      else if (k > 0) {
	double *v = (double*)malloc(k*sizeof(double));
	if (!v) return __ERROR;
	memcpy(v, u+i, k*sizeof(double));
	return mkbstr(k*sizeof(double), v);
      } else
	return mkbstr(0, NULL);
    } else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(clib,put_double,argc,argv)
{
  bstr_t *m, *m1;
  long i;
  double x;
  if (argc == 3 && isobj(argv[0], type(ByteStr), (void**)&m) &&
      isint(argv[1], &i)) {
    long N = m->size/sizeof(double);
    double *u = (double*)m->v;
    if ((isfloat(argv[2], &x) || ismpz_float(argv[2], &x)) &&
	i >= 0 && i < N) {
      u[i] = (double)x;
      return mkvoid;
    } else if (isobj(argv[2], type(ByteStr), (void**)&m1)) {
      long M = m1->size/sizeof(double);
      double *v = (double*)m1->v;
      long j = 0, k = put_clamp(&i, &j, M, N);
      if (k > 0) memcpy(u+i, v+j, k*sizeof(double));
      return mkvoid;
    } else
      return __FAIL;
  } else
    return __FAIL;
}

FUNCTION(clib,int8_list,argc,argv)
{
  bstr_t *m;
  if (argc == 1 && isobj(argv[0], type(ByteStr), (void**)&m)) {
    int i, n = m->size;
    signed char *v = (signed char*)m->v;
    expr *xv;
    if (n <= 0) return mknil;
    xv = malloc(n*sizeof(expr));
    if (!xv) return __ERROR;
    for (i = 0; i < n; i++)
      xv[i] = mkint(v[i]);
    return mklistv(n, xv);
  } else
    return __FAIL;
}

FUNCTION(clib,uint8_list,argc,argv)
{
  bstr_t *m;
  if (argc == 1 && isobj(argv[0], type(ByteStr), (void**)&m)) {
    int i, n = m->size;
    unsigned char *v = (unsigned char*)m->v;
    expr *xv;
    if (n <= 0) return mknil;
    xv = malloc(n*sizeof(expr));
    if (!xv) return __ERROR;
    for (i = 0; i < n; i++)
      xv[i] = mkuint(v[i]);
    return mklistv(n, xv);
  } else
    return __FAIL;
}

FUNCTION(clib,int16_list,argc,argv)
{
  bstr_t *m;
  if (argc == 1 && isobj(argv[0], type(ByteStr), (void**)&m)) {
    int i, n = m->size >> 1;
    short *v = (short*)m->v;
    expr *xv;
    if (n <= 0) return mknil;
    xv = malloc(n*sizeof(expr));
    if (!xv) return __ERROR;
    for (i = 0; i < n; i++)
      xv[i] = mkint(v[i]);
    return mklistv(n, xv);
  } else
    return __FAIL;
}

FUNCTION(clib,uint16_list,argc,argv)
{
  bstr_t *m;
  if (argc == 1 && isobj(argv[0], type(ByteStr), (void**)&m)) {
    int i, n = m->size >> 1;
    unsigned short *v = (unsigned short*)m->v;
    expr *xv;
    if (n <= 0) return mknil;
    xv = malloc(n*sizeof(expr));
    if (!xv) return __ERROR;
    for (i = 0; i < n; i++)
      xv[i] = mkuint(v[i]);
    return mklistv(n, xv);
  } else
    return __FAIL;
}

FUNCTION(clib,int32_list,argc,argv)
{
  bstr_t *m;
  if (argc == 1 && isobj(argv[0], type(ByteStr), (void**)&m)) {
    int i, n = m->size >> 2;
    long *v = (long*)m->v;
    expr *xv;
    if (n <= 0) return mknil;
    xv = malloc(n*sizeof(expr));
    if (!xv) return __ERROR;
    for (i = 0; i < n; i++)
      xv[i] = mkint(v[i]);
    return mklistv(n, xv);
  } else
    return __FAIL;
}

FUNCTION(clib,uint32_list,argc,argv)
{
  bstr_t *m;
  if (argc == 1 && isobj(argv[0], type(ByteStr), (void**)&m)) {
    int i, n = m->size >> 2;
    unsigned long *v = (unsigned long*)m->v;
    expr *xv;
    if (n <= 0) return mknil;
    xv = malloc(n*sizeof(expr));
    if (!xv) return __ERROR;
    for (i = 0; i < n; i++)
      xv[i] = mkuint(v[i]);
    return mklistv(n, xv);
  } else
    return __FAIL;
}

FUNCTION(clib,float_list,argc,argv)
{
  bstr_t *m;
  if (argc == 1 && isobj(argv[0], type(ByteStr), (void**)&m)) {
    int i, n = m->size / sizeof(float);
    float *v = (float*)m->v;
    expr *xv;
    if (n <= 0) return mknil;
    xv = malloc(n*sizeof(expr));
    if (!xv) return __ERROR;
    for (i = 0; i < n; i++)
      xv[i] = mkfloat(v[i]);
    return mklistv(n, xv);
  } else
    return __FAIL;
}

FUNCTION(clib,double_list,argc,argv)
{
  bstr_t *m;
  if (argc == 1 && isobj(argv[0], type(ByteStr), (void**)&m)) {
    int i, n = m->size / sizeof(double);
    double *v = (double*)m->v;
    expr *xv;
    if (n <= 0) return mknil;
    xv = malloc(n*sizeof(expr));
    if (!xv) return __ERROR;
    for (i = 0; i < n; i++)
      xv[i] = mkfloat(v[i]);
    return mklistv(n, xv);
  } else
    return __FAIL;
}

FUNCTION(clib,int8_vect,argc,argv)
{
  if (argc == 1) {
    expr xs = argv[0], hd, tl;
    int i, n = 0;
    long x;
    signed char *v;
    while (iscons(xs, &hd, &tl) && isint(hd, &x)) {
      xs = tl; n++;
    }
    if (!isnil(xs)) return __FAIL;
    if (n <= 0) return mkbstr(0, NULL);
    if (!(v = malloc(n))) return __ERROR;
    xs = argv[0]; n = 0;
    while (iscons(xs, &hd, &tl) && isint(hd, &x)) {
      xs = tl; v[n++] = (signed char)x;
    }
    return mkbstr(n, v);
  } else
    return __FAIL;
}

FUNCTION(clib,int16_vect,argc,argv)
{
  if (argc == 1) {
    expr xs = argv[0], hd, tl;
    int i, n = 0;
    long x;
    short *v;
    while (iscons(xs, &hd, &tl) && isint(hd, &x)) {
      xs = tl; n++;
    }
    if (!isnil(xs)) return __FAIL;
    if (n <= 0) return mkbstr(0, NULL);
    if (!(v = malloc(n*sizeof(short)))) return __ERROR;
    xs = argv[0]; n = 0;
    while (iscons(xs, &hd, &tl) && isint(hd, &x)) {
      xs = tl; v[n++] = (short)x;
    }
    return mkbstr(n*sizeof(short), v);
  } else
    return __FAIL;
}

FUNCTION(clib,int32_vect,argc,argv)
{
  if (argc == 1) {
    expr xs = argv[0], hd, tl;
    int i, n = 0;
    long x;
    long *v;
    while (iscons(xs, &hd, &tl) && isint(hd, &x)) {
      xs = tl; n++;
    }
    if (!isnil(xs)) return __FAIL;
    if (n <= 0) return mkbstr(0, NULL);
    if (!(v = malloc(n*sizeof(long)))) return __ERROR;
    xs = argv[0]; n = 0;
    while (iscons(xs, &hd, &tl) && isint(hd, &x)) {
      xs = tl; v[n++] = x;
    }
    return mkbstr(n*sizeof(long), v);
  } else
    return __FAIL;
}

FUNCTION(clib,uint8_vect,argc,argv)
{
  if (argc == 1) {
    expr xs = argv[0], hd, tl;
    int i, n = 0;
    unsigned long x;
    unsigned char *v;
    while (iscons(xs, &hd, &tl) && isuint(hd, &x)) {
      xs = tl; n++;
    }
    if (!isnil(xs)) return __FAIL;
    if (n <= 0) return mkbstr(0, NULL);
    if (!(v = malloc(n))) return __ERROR;
    xs = argv[0]; n = 0;
    while (iscons(xs, &hd, &tl) && isuint(hd, &x)) {
      xs = tl; v[n++] = (unsigned char)x;
    }
    return mkbstr(n, v);
  } else
    return __FAIL;
}

FUNCTION(clib,uint16_vect,argc,argv)
{
  if (argc == 1) {
    expr xs = argv[0], hd, tl;
    int i, n = 0;
    unsigned long x;
    unsigned short *v;
    while (iscons(xs, &hd, &tl) && isuint(hd, &x)) {
      xs = tl; n++;
    }
    if (!isnil(xs)) return __FAIL;
    if (n <= 0) return mkbstr(0, NULL);
    if (!(v = malloc(n*sizeof(unsigned short)))) return __ERROR;
    xs = argv[0]; n = 0;
    while (iscons(xs, &hd, &tl) && isuint(hd, &x)) {
      xs = tl; v[n++] = (unsigned short)x;
    }
    return mkbstr(n*sizeof(unsigned short), v);
  } else
    return __FAIL;
}

FUNCTION(clib,uint32_vect,argc,argv)
{
  if (argc == 1) {
    expr xs = argv[0], hd, tl;
    int i, n = 0;
    unsigned long x;
    unsigned long *v;
    while (iscons(xs, &hd, &tl) && isuint(hd, &x)) {
      xs = tl; n++;
    }
    if (!isnil(xs)) return __FAIL;
    if (n <= 0) return mkbstr(0, NULL);
    if (!(v = malloc(n*sizeof(unsigned long)))) return __ERROR;
    xs = argv[0]; n = 0;
    while (iscons(xs, &hd, &tl) && isuint(hd, &x)) {
      xs = tl; v[n++] = x;
    }
    return mkbstr(n*sizeof(unsigned long), v);
  } else
    return __FAIL;
}

FUNCTION(clib,float_vect,argc,argv)
{
  if (argc == 1) {
    expr xs = argv[0], hd, tl;
    int i, n = 0;
    double x;
    float *v;
    while (iscons(xs, &hd, &tl) && (isfloat(hd, &x) || ismpz_float(hd, &x))) {
      xs = tl; n++;
    }
    if (!isnil(xs)) return __FAIL;
    if (n <= 0) return mkbstr(0, NULL);
    if (!(v = malloc(n*sizeof(float)))) return __ERROR;
    xs = argv[0]; n = 0;
    while (iscons(xs, &hd, &tl) && (isfloat(hd, &x) || ismpz_float(hd, &x))) {
      xs = tl; v[n++] = (float)x;
    }
    return mkbstr(n*sizeof(float), v);
  } else
    return __FAIL;
}

FUNCTION(clib,double_vect,argc,argv)
{
  if (argc == 1) {
    expr xs = argv[0], hd, tl;
    int i, n = 0;
    double x;
    double *v;
    while (iscons(xs, &hd, &tl) && (isfloat(hd, &x) || ismpz_float(hd, &x))) {
      xs = tl; n++;
    }
    if (!isnil(xs)) return __FAIL;
    if (n <= 0) return mkbstr(0, NULL);
    if (!(v = malloc(n*sizeof(double)))) return __ERROR;
    xs = argv[0]; n = 0;
    while (iscons(xs, &hd, &tl) && (isfloat(hd, &x) || ismpz_float(hd, &x))) {
      xs = tl; v[n++] = x;
    }
    return mkbstr(n*sizeof(double), v);
  } else
    return __FAIL;
}

/* references: ************************************************************/

DESTRUCTOR(clib,Ref,ptr)
{
  freeref(*((expr*)ptr));
  free(ptr);
}

FUNCTION(clib,ref,argc,argv)
{
  if (argc == 1) {
    expr *x = (expr*)malloc(sizeof(expr));
    if (x) {
      *x = newref(argv[0]);
      return mkobj(type(Ref), x);
    } else
      return __ERROR;
  } else
    return __FAIL;
}

FUNCTION(clib,put,argc,argv)
{
  expr *x;
  if (argc == 2 && isobj(argv[0], type(Ref), (void**)&x)) {
    freeref(*x);
    *x = newref(argv[1]);
    return mkvoid;
  } else
    return __FAIL;
}

/* see the semaphore operations below for a definition of get */

DESTRUCTOR(clib,Sentinel,ptr)
{
  sentinel(*((expr*)ptr));
  free(ptr);
}

FUNCTION(clib,sentinel,argc,argv)
{
  if (argc == 1) {
    expr *x = (expr*)malloc(sizeof(expr));
    if (x) {
      *x = newref(argv[0]);
      return mkobj(type(Sentinel), x);
    } else
      return __ERROR;
  } else
    return __FAIL;
}

/* multithreading: ********************************************************/

/* the thread table (make sure that the size matches with qdefs.h) */

#ifdef USE_THREADS
#define MAXTHREAD 1024
#else
#define MAXTHREAD 1
#endif

typedef unsigned char bool;
typedef struct {
  bool			active, canceled, used;	/* status */
  expr			arg;			/* thread argument */
  expr			result;			/* thread result, if any */
  expr			thread;			/* thread object */
#ifdef USE_THREADS
  pthread_t		id;			/* thread id */
  pthread_mutex_t	exit_mutex;		/* exit mutex and condition */
  pthread_cond_t	exit_cond;
#ifdef WIN32
  HANDLE		handle;			/* thread handle */
  int			pol;			/* scheduling policy */
#endif
#endif
} THREAD;

static THREAD threads[MAXTHREAD], *thr0 = threads;
static int maxused, stamp;

#ifdef WIN32
/* handle thread priorities under windows */

static int prio_class[] = {0,1,0,0};

static int max_prio_class(void)
{
  int i;
  for (i = 3; i >= 0; i--)
    if (prio_class[i] > 0) break;
  i--;
  switch (i) {
  case -1: return IDLE_PRIORITY_CLASS;
  case 1: return HIGH_PRIORITY_CLASS;
  case 2: return REALTIME_PRIORITY_CLASS;
  default: return NORMAL_PRIORITY_CLASS;
  }
}
#endif

DESTRUCTOR(clib,Thread,ptr)
{
#ifdef USE_THREADS
  THREAD *thr = (THREAD*)ptr;
  pthread_t id = thr->id;
  if (id == pthread_self()) {
    thr->thread = NULL;
    return;
  /* XXX FIXME: experimental stuff here. Second attempt to prevent
     waiting for dead child threads in thread destructor (see also
     ChangeLog entry from 2002-08-29). */
#if 0
  } else if (!thr->active)
#else
  } else if (!thr->used || !thr->thread)
#endif
    return;
#ifdef DEBUG
  printf("destroying thread %d (id %d)\n", thr-thr0, (int)id);
#endif
  pthread_cancel(id);
  /* make sure that the thread object is not garbage-collected in the
     cancelled thread while we're already destroying it */
  newref(thr->thread);
  /* wait for the canceled thread to finish */
  release_lock();
  pthread_join(id, NULL);
  acquire_lock();
  unref(thr->thread);
  thr->active = thr->canceled = thr->used = 0;
  if (thr->arg) {
    freeref(thr->arg);
    thr->arg = NULL;
  }
  if (thr->result) {
    freeref(thr->result);
    thr->result = NULL;
  }
  if (thr->thread) thr->thread = NULL;
  pthread_mutex_destroy(&thr->exit_mutex);
  pthread_cond_destroy(&thr->exit_cond);
#ifdef WIN32
  CloseHandle(thr->handle);
  thr->handle = INVALID_HANDLE_VALUE;
  prio_class[thr->pol+1]--;
#ifdef DEBUG
  printf("set priority class %d\n", max_prio_class());
#endif
  SetPriorityClass(GetCurrentProcess(), max_prio_class());
#endif
  /* deregister the thread with the interpreter */
  fini_thread(thr-thr0);
#endif
}

FUNCTION(clib,thread_no,argc,argv)
{
  THREAD *thr;
  if (argc == 1 && isobj(argv[0], type(Thread), (void**)&thr)) {
    return mkint(thr-thr0);
  } else
    return __FAIL;
}

FUNCTION(clib,this_thread,argc,argv)
{
  if (argc == 0) {
    THREAD *thr = thr0+this_thread();
    if (thr->thread)
      return thr->thread;
    else if (thr == thr0)
      return (thr->thread = mkobj(type(Thread), thr));
    else
      return __FAIL;
  } else
    return __FAIL;
}

#ifdef USE_THREADS

static bool thread_ready;
static pthread_mutex_t thread_ready_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t thread_ready_cond = PTHREAD_COND_INITIALIZER;
static THREAD *nthr;

static void my_mutex_unlock(void *mut)
{
  pthread_mutex_unlock((pthread_mutex_t*)mut);
}

static void thread_canceled_proc(void *arg)
{
  THREAD *thr = (THREAD*)arg;
#ifdef DEBUG
  printf("thread %d (id %d) %s\n", thr-thr0, (int)thr->id,
	 thr->result?"exited":"canceled");
#endif
  thr->canceled = (thr->result == NULL);
  exit_thread(thr-thr0);
  pthread_mutex_lock(&thr->exit_mutex);
  thr->active = 0;
  pthread_cond_broadcast(&thr->exit_cond);
  pthread_mutex_unlock(&thr->exit_mutex);
}

static void *thread_proc(void *arg)
{
  expr x = (expr)arg, y;
  THREAD *thr;
  /* register the new thread with the interpreter */
  int thrid = init_thread();
  /* fill in the thread info */
  pthread_mutex_lock(&thread_ready_mutex);
  if (thrid >= 0) {
    thr = thr0+thrid;
    if ((thr->thread = mkobj(type(Thread), thr))) {
      /* we count a new ref to the thread object here so that it does not get
	 garbage-collected while we already start executing, before the thread
	 function has had a chance to return the object */
      newref(thr->thread);
      thr->arg = x;
      thr->result = NULL;
      thr->id = pthread_self();
#ifdef WIN32
      if (!DuplicateHandle(GetCurrentProcess(), GetCurrentThread(),
			   GetCurrentProcess(), &thr->handle,
			   0, FALSE, DUPLICATE_SAME_ACCESS)) {
	dispose(thr->thread);
	goto errexit;
      }
      thr->pol = 0;
      prio_class[1]++;
#ifdef DEBUG
      printf("set priority class %d\n", max_prio_class());
#endif
      SetPriorityClass(GetCurrentProcess(), max_prio_class());
#endif
      pthread_mutex_init(&thr->exit_mutex, NULL);
      pthread_cond_init(&thr->exit_cond, NULL);
      thr->active = 1; thr->canceled = 0; thr->used = 1;
      if (thr-thr0 > maxused) maxused = thr-thr0;
      nthr = thr;
#ifdef DEBUG
      printf("initialized thread %d (id %d)\n", thr-thr0, (int)thr->id);
#endif
    } else {
    errexit:
      exit_thread(thrid);
      fini_thread(thrid);
#ifdef DEBUG
      printf("error initializing thread %d (id %d), exiting\n", thr-thr0, (int)thr->id);
#endif
    }
  }
#ifdef DEBUG
  else
    printf("error registering thread\n");
#endif
  /* exit if error */
  if (!nthr) {
    thread_ready = 1;
    pthread_cond_signal(&thread_ready_cond);
    pthread_mutex_unlock(&thread_ready_mutex);
    return NULL;
  }
  /* signal that we're ready */
  pthread_cleanup_push(thread_canceled_proc, thr);
  release_lock();
  thread_ready = 1;
  pthread_cond_signal(&thread_ready_cond);
  pthread_mutex_unlock(&thread_ready_mutex);
  /* we're up and running, start evaluating the special argument */
#ifdef DEBUG
  printf("thread %d (id %d) up and running\n", thr-thr0, (int)thr->id);
#endif
  acquire_lock();
  y = newref(eval(x));
  pthread_cleanup_pop(0);
  /* exit from thread */
  thr->result = y;
  exit_thread(thr-thr0);
  /* signal that we've exited */
  pthread_mutex_lock(&thr->exit_mutex);
  thr->active = 0;
  pthread_cond_broadcast(&thr->exit_cond);
  pthread_mutex_unlock(&thr->exit_mutex);
#ifdef WIN32
  CloseHandle(thr->handle);
  thr->handle = INVALID_HANDLE_VALUE;
  prio_class[thr->pol+1]--;
#ifdef DEBUG
  printf("set priority class %d\n", max_prio_class());
#endif
  SetPriorityClass(GetCurrentProcess(), max_prio_class());
#endif
#ifdef DEBUG
  printf("thread %d (id %d) exits\n", thr-thr0, (int)thr->id);
#endif
  return (void*)y;
}
#endif

FUNCTION(clib,thread,argc,argv)
{
#ifdef USE_THREADS
  if (argc == 1) {
    pthread_t id;
    int res;
    /* start the new thread */
    pthread_cleanup_push(my_mutex_unlock, (void*)&thread_ready_mutex);
    pthread_mutex_lock(&thread_ready_mutex);
    thread_ready = 0; nthr = NULL;
#ifdef DEBUG
    printf("starting new thread\n");
#endif
    if ((res = pthread_create(&id, NULL, thread_proc, newref(argv[0]))) == 0) {
      /* wait until the new thread signals that it's up and running */
#ifdef DEBUG
      printf("waiting for new thread\n");
#endif
      release_lock();
      while (!thread_ready)
	pthread_cond_wait(&thread_ready_cond, &thread_ready_mutex);
      acquire_lock();
    }
    pthread_cleanup_pop(1);
    if (res) return __FAIL;
#ifdef DEBUG
    printf("new thread %s\n", nthr?"ready":"aborted");
#endif
    /* return the thread object */
    if (nthr)
      return unref(nthr->thread);
    else
      return __ERROR;
  } else
#endif
    return __FAIL;
}

FUNCTION(clib,return,argc,argv)
{
  if (argc == 1) {
    THREAD *thr = thr0+this_thread();
#ifdef USE_THREADS
    if (thr > thr0) {
      thr->result = newref(argv[0]);
      pthread_exit((void*)thr->result);
    } else
#endif
      return mksym(sym(halt));
  } else
    return __FAIL;
}

FUNCTION(clib,cancel,argc,argv)
{
#ifdef USE_THREADS
  THREAD *thr;
  if (argc == 1 && isobj(argv[0], type(Thread), (void**)&thr)) {
    if (thr == thr0) return __FAIL;
    pthread_cancel(thr->id);
    return mkvoid;
  } else
#endif
    return __FAIL;
}

FUNCTION(clib,result,argc,argv)
{
#ifdef USE_THREADS
  THREAD *thr;
  if (argc == 1 && isobj(argv[0], type(Thread), (void**)&thr)) {
    if (thr == thr0) return __FAIL;
    /* wait for the thread to finish */
    pthread_cleanup_push(my_mutex_unlock, (void*)&thr->exit_mutex);
    pthread_mutex_lock(&thr->exit_mutex);
    release_lock();
    while (thr->active)
      pthread_cond_wait(&thr->exit_cond, &thr->exit_mutex);
    pthread_cleanup_pop(1);
    acquire_lock();
    /* return result, if any */
    if (thr->canceled || !thr->result)
      return __FAIL;
    else
      return thr->result;
  } else
#endif
    return __FAIL;
}

FUNCTION(clib,yield,argc,argv)
{
  if (argc == 0) {
#ifdef USE_THREADS
    release_lock(); acquire_lock();
#endif
    return mkvoid;
  } else
    return __FAIL;
}

FUNCTION(clib,active,argc,argv)
{
  THREAD *thr;
  if (argc == 1 && isobj(argv[0], type(Thread), (void**)&thr))
#ifdef USE_THREADS
    return thr->active?mktrue:mkfalse;
#else
    return mktrue;
#endif
  else
    return __FAIL;
}

FUNCTION(clib,canceled,argc,argv)
{
  THREAD *thr;
  if (argc == 1 && isobj(argv[0], type(Thread), (void**)&thr))
#ifdef USE_THREADS
    return thr->canceled?mktrue:mkfalse;
#else
    return mkfalse;
#endif
  else
    return __FAIL;
}

/* NOTES ON WINDOWS SCHEDULING:

   The win32 pthread functions only allow SCHED_OTHER, so we provide our own
   implementation using the native Windows functions here. This is a bit
   kludgy since Windows does not provide real POSIX-compatible scheduling,
   but it's not too difficult to come up with a reasonable scheme. Here is how
   we chose to implement it:

   The policies 0, 1 and 2 are actually implemented as the process priority
   classes NORMAL_PRIORITY_CLASS, HIGH_PRIORITY_CLASS and
   REALTIME_PRIORITY_CLASS, respectively. There's also an extra policy -1 for
   the IDLE_PRIORITY_CLASS, which is rarely used (who wants to write a
   screensaver in Q?) but it's there anyway.

   Note that the *entire process* will run at higher scheduling priority when
   starting a thread using a policy >0. This means that the effective
   priorities of *all threads* will be beefed up once a realtime thread starts
   running; there's no way around this on Windows. Clib keeps track of the
   policies of all running threads and always chooses the lowest among these
   for the priority class of the process.

   For each of the policies we provide the priority values -3
   (THREAD_PRIORITY_IDLE), -2 (THREAD_PRIORITY_LOWEST), -1
   (THREAD_PRIORITY_BELOW_NORMAL), 0 (THREAD_PRIORITY_NORMAL), 1
   (THREAD_PRIORITY_ABOVE_NORMAL), 2 (THREAD_PRIORITY_HIGHEST) and 3
   (THREAD_PRIORITY_TIME_CRITICAL). Threads executing at priority -3 will only
   be executed when the system is idle. Priority 0 is the "normal"
   priority. The other priority levels provide some amount of control over
   which thread gets the bone first.

   Typically, a policy of 0 with zero priority will be used for ordinary
   threads, a policy of 1 with some positive priority value for a thread with
   moderate realtime requirements, and a policy of 2 if time is very critical
   (the latter should be used with utmost care since a process in the realtime
   class can easily freeze up Windows). This matches the typical usage on
   POSIX systems. */

FUNCTION(clib,setsched,argc,argv)
{
#ifdef USE_THREADS
  THREAD *thr;
  long pol, prio;
  if (argc == 3 && isobj(argv[0], type(Thread), (void**)&thr) &&
      isint(argv[1], &pol) && isint(argv[2], &prio)) {
#ifndef WIN32
    struct sched_param param;
    int actpol;
    switch (pol) {
    case 0: pol = SCHED_OTHER; break;
    case 1: pol = SCHED_RR; break;
    case 2: pol = SCHED_FIFO; break;
    default: return __FAIL;
    }
    if (pthread_getschedparam(thr->id, &actpol, &param))
      return __FAIL;
    else {
      param.sched_priority = prio;
      if (pthread_setschedparam(thr->id, pol, &param))
	return __FAIL;
      else
	return mkvoid;
    }
#else /* WIN32 */
    if (thr->handle == INVALID_HANDLE_VALUE) return __FAIL;
    switch (prio) {
    case -3: prio = THREAD_PRIORITY_IDLE; break;
    case -2: prio = THREAD_PRIORITY_LOWEST; break;
    case -1: prio = THREAD_PRIORITY_BELOW_NORMAL; break;
    case 0: prio = THREAD_PRIORITY_NORMAL; break;
    case 1: prio = THREAD_PRIORITY_ABOVE_NORMAL; break;
    case 2: prio = THREAD_PRIORITY_HIGHEST; break;
    case 3: prio = THREAD_PRIORITY_TIME_CRITICAL; break;
    default: return __FAIL;
    }
    prio_class[thr->pol+1]--;
    prio_class[pol+1]++;
#ifdef DEBUG
    printf("set priority class %d\n", max_prio_class());
#endif
    if (!SetPriorityClass(GetCurrentProcess(), max_prio_class())) {
      prio_class[pol+1]--;
      prio_class[thr->pol+1]++;
      return __FAIL;
    } else {
#ifdef DEBUG
      printf("set priority %d for thread %d (id %d, handle %d)\n", prio,
	     thr-thr0, (int)thr->id, (int)thr->handle);
#endif
      if (SetThreadPriority(thr->handle, prio)) {
	thr->pol = pol;
	return mkvoid;
      } else {
	prio_class[pol+1]--;
	prio_class[thr->pol+1]++;
	SetPriorityClass(GetCurrentProcess(), max_prio_class());
	return __FAIL;
      }
    }
#endif
  } else
#endif
    return __FAIL;
}

FUNCTION(clib,getsched,argc,argv)
{
#ifdef USE_THREADS
  THREAD *thr;
  int pol, prio;
  if (argc == 1 && isobj(argv[0], type(Thread), (void**)&thr)) {
#ifndef WIN32
    struct sched_param param;
    if (pthread_getschedparam(thr->id, &pol, &param))
      return __FAIL;
    else {
      switch(pol) {
      case SCHED_OTHER: pol = 0; break;
      case SCHED_RR: pol = 1; break;
      case SCHED_FIFO: pol = 2; break;
      default: return __FAIL;
      }
      prio = param.sched_priority;
      return mktuplel(2, mkint(pol), mkint(prio));
    }
#else /* WIN32 */
    if (thr->handle == INVALID_HANDLE_VALUE) return __FAIL;
    switch (GetThreadPriority(thr->handle)) {
    case THREAD_PRIORITY_IDLE: prio = -3; break;
    case THREAD_PRIORITY_LOWEST: prio = -2; break;
    case THREAD_PRIORITY_BELOW_NORMAL: prio = -1; break;
    case THREAD_PRIORITY_NORMAL: prio = 0; break;
    case THREAD_PRIORITY_ABOVE_NORMAL: prio = 1; break;
    case THREAD_PRIORITY_HIGHEST: prio = 2; break;
    case THREAD_PRIORITY_TIME_CRITICAL: prio = 3; break;
    default: return __FAIL;
    }
    pol = thr->pol;
    return mktuplel(2, mkint(pol), mkint(prio));
#endif
  } else
#endif
    return __FAIL;
}

#ifdef USE_THREADS

/* mutex data structure */

typedef struct {
  pthread_mutex_t mut;
  pthread_mutexattr_t attr, *attrp;
  int stamp;
} my_mutex_t;

/* condition data structure */

typedef struct {
  pthread_mutex_t mut;
  pthread_cond_t cond;
  bool set;
  int stamp;
} my_cond_t;

/* semaphore data structure */

/* KLUDGE ALERT: MacOS X doesn't have unnamed semaphores, se we use
   named ones instead. FIXME: Maybe we should provide our own
   lightweight semaphore implementation here? */

#ifdef __APPLE__
#define NAMED_SEM
#endif

#ifdef NAMED_SEM
static unsigned long sem_counter = 0;
static char *new_sem_name(void)
{
  static char sem_name[32];
  sprintf(sem_name, "/clib_sem-%d-%4.4d", getpid(), sem_counter++);
  return sem_name;
}
#endif

typedef struct expr_queue_entry {
  expr val;
  struct expr_queue_entry *next;
} expr_queue_entry_t;

typedef struct {
  long size;
  expr_queue_entry_t *head, *tail, *last_tail;
} expr_queue_t;

typedef struct {
  pthread_mutex_t mut;
  sem_t *semp;
#ifndef NAMED_SEM
  sem_t sem;
#endif
  pthread_cond_t cond;
  expr_queue_t queue;
  int stamp;
  long max;
} my_sem_t;

static void init_queue(expr_queue_t *queue)
{
  queue->size = 0;
  queue->head = queue->tail = queue->last_tail = NULL;
}

static void free_queue(expr_queue_t *queue)
{
  expr_queue_entry_t *head = queue->head;
  while (head) {
    expr_queue_entry_t *next = head->next;
    if (head->val) freeref(head->val);
    free(head);
    head = next;
  }
}

static expr_queue_entry_t *enqueue_expr(expr_queue_t *queue, expr val)
{
  expr_queue_entry_t *new_tail = malloc(sizeof(expr_queue_entry_t));
  if (!new_tail) return NULL;
  new_tail->val = newref(val);
  new_tail->next = NULL;
  if (queue->tail) {
    queue->tail->next = new_tail;
    queue->last_tail = queue->tail;
    queue->tail = new_tail;
  } else {
    queue->last_tail = NULL;
    queue->head = queue->tail = new_tail;
  }
  queue->size++;
  return new_tail;
}

static expr_queue_entry_t *unenqueue_expr(expr_queue_t *queue)
{
  if (queue->tail) {
    freeref(queue->tail->val);
    free(queue->tail);
    queue->tail = queue->last_tail;
    if (queue->tail)
      queue->tail->next = NULL;
    else
      queue->head = NULL;
    queue->last_tail = NULL;
    queue->size--;
    return queue->tail;
  } else
    return NULL;
}

static expr dequeue_expr(expr_queue_t *queue)
{
  expr val;
  if (!queue->head) return NULL;
  val = queue->head->val;
  if (queue->head == queue->tail) {
    free(queue->head);
    queue->head = queue->tail = queue->last_tail = NULL;
  } else {
    expr_queue_entry_t *next = queue->head->next;
    if (queue->last_tail == queue->head) queue->last_tail = NULL;
    free(queue->head);
    queue->head = next;
  }
  queue->size--;
  return val;
}

/* clean up after fork */

static check_mut(my_mutex_t *mut)
{
  if (mut && mut->stamp != stamp) {
    pthread_mutex_init(&mut->mut, mut->attrp);
    mut->stamp = stamp;
  }
}

static check_cond(my_cond_t *cond)
{
  if (cond && cond->stamp != stamp) {
    pthread_mutex_init(&cond->mut, NULL);
    pthread_cond_init(&cond->cond, NULL);
    cond->set = 0;
    cond->stamp = stamp;
  }
}

static check_sem(my_sem_t *sem)
{
  if (sem && sem->stamp != stamp) {
    pthread_mutex_init(&sem->mut, NULL);
#ifdef NAMED_SEM
    sem->semp = sem_open(new_sem_name(), O_CREAT, 0600, sem->queue.size);
    if (sem->semp == (sem_t*)SEM_FAILED) sem->semp = NULL;
#else
    sem_init(&sem->sem, 0, sem->queue.size);
    sem->semp = &sem->sem;
#endif
    pthread_cond_init(&sem->cond, NULL);
    sem->stamp = stamp;
  }
}

#endif

/* destructors */

DESTRUCTOR(clib,Mutex,ptr)
{
#ifdef USE_THREADS
  my_mutex_t *mut = (my_mutex_t*)ptr;
  if (mut) {
    pthread_mutex_destroy(&mut->mut);
    if (mut->attrp)
      pthread_mutexattr_destroy(mut->attrp);
    free(mut);
  }
#endif
}

DESTRUCTOR(clib,Condition,ptr)
{
#ifdef USE_THREADS
  my_cond_t *cond = (my_cond_t*)ptr;
  if (cond) {
    pthread_mutex_destroy(&cond->mut);
    pthread_cond_destroy(&cond->cond);
    free(cond);
  }
#endif
}

DESTRUCTOR(clib,Semaphore,ptr)
{
#ifdef USE_THREADS
  my_sem_t *sem = (my_sem_t*)ptr;
  if (sem) {
    pthread_mutex_destroy(&sem->mut);
#ifdef NAMED_SEM
    sem_close(sem->semp);
#else
    sem_destroy(&sem->sem);
#endif
    sem->semp = NULL;
    pthread_cond_destroy(&sem->cond);
    free_queue(&sem->queue);
    free(sem);
  }
#endif
}

/* constructors */

FUNCTION(clib,mutex,argc,argv)
{
#ifdef USE_THREADS
  my_mutex_t *mut = (my_mutex_t*)malloc(sizeof(my_mutex_t));
  if (mut) {
    mut->attrp = NULL;
    pthread_mutex_init(&mut->mut, NULL);
    mut->stamp = stamp;
    return mkobj(type(Mutex), mut);
  } else
    return __ERROR;
#else
  return __FAIL;
#endif
}

FUNCTION(clib,errorchecking_mutex,argc,argv)
{
#ifdef USE_THREADS
  my_mutex_t *mut = (my_mutex_t*)malloc(sizeof(my_mutex_t));
  if (mut) {
    mut->attrp = &mut->attr;
    pthread_mutexattr_init(mut->attrp);
    pthread_mutexattr_settype(mut->attrp, PTHREAD_MUTEX_ERRORCHECK);
    pthread_mutex_init(&mut->mut, mut->attrp);
    mut->stamp = stamp;
    return mkobj(type(Mutex), mut);
  } else
    return __ERROR;
#else
  return __FAIL;
#endif
}

FUNCTION(clib,recursive_mutex,argc,argv)
{
#ifdef USE_THREADS
  my_mutex_t *mut = (my_mutex_t*)malloc(sizeof(my_mutex_t));
  if (mut) {
    mut->attrp = &mut->attr;
    pthread_mutexattr_init(mut->attrp);
    pthread_mutexattr_settype(mut->attrp, PTHREAD_MUTEX_RECURSIVE);
    pthread_mutex_init(&mut->mut, mut->attrp);
    mut->stamp = stamp;
    return mkobj(type(Mutex), mut);
  } else
    return __ERROR;
#else
  return __FAIL;
#endif
}

FUNCTION(clib,condition,argc,argv)
{
#ifdef USE_THREADS
  my_cond_t *cond = (my_cond_t*)malloc(sizeof(my_cond_t));
  if (cond) {
    pthread_mutex_init(&cond->mut, NULL);
    pthread_cond_init(&cond->cond, NULL);
    cond->set = 0;
    cond->stamp = stamp;
    return mkobj(type(Condition), cond);
  } else
    return __ERROR;
#else
  return __FAIL;
#endif
}

FUNCTION(clib,semaphore,argc,argv)
{
#ifdef USE_THREADS
  my_sem_t *sem = (my_sem_t*)malloc(sizeof(my_sem_t));
  if (sem) {
    pthread_mutex_init(&sem->mut, NULL);
#ifdef NAMED_SEM
    sem->semp = sem_open(new_sem_name(), O_CREAT, 0600, 0);
    if (sem->semp == (sem_t*)SEM_FAILED) {
      free(sem);
      return __ERROR;
    }
#else
    sem_init(&sem->sem, 0, 0);
    sem->semp = &sem->sem;
#endif
    pthread_cond_init(&sem->cond, NULL);
    init_queue(&sem->queue);
    sem->stamp = stamp;
    sem->max = 0;
    return mkobj(type(Semaphore), sem);
  } else
    return __ERROR;
#else
  return __FAIL;
#endif
}

FUNCTION(clib,bounded_semaphore,argc,argv)
{
#ifdef USE_THREADS
  long max;
  if (argc == 1 && isint(argv[0], &max) && max > 0) {
    my_sem_t *sem = (my_sem_t*)malloc(sizeof(my_sem_t));
    if (sem) {
      pthread_mutex_init(&sem->mut, NULL);
#ifdef NAMED_SEM
      sem->semp = sem_open(new_sem_name(), O_CREAT, 0600, 0);
      if (sem->semp == (sem_t*)SEM_FAILED) {
	free(sem);
	return __ERROR;
      }
#else
      sem_init(&sem->sem, 0, 0);
      sem->semp = &sem->sem;
#endif
      pthread_cond_init(&sem->cond, NULL);
      init_queue(&sem->queue);
      sem->stamp = stamp;
      sem->max = max;
      return mkobj(type(Semaphore), sem);
    } else
      return __ERROR;
  } else
#endif
    return __FAIL;
}

/* interface ops */

FUNCTION(clib,lock,argc,argv)
{
#ifdef USE_THREADS
  my_mutex_t *mut;
  if (argc == 1 && isobj(argv[0], type(Mutex), (void**)&mut)) {
    int res;
    check_mut(mut);
    release_lock();
    res = pthread_mutex_lock(&mut->mut);
    acquire_lock();
    if (res)
      return __FAIL;
    else
      return mkvoid;
  } else
#endif
    return __FAIL;
}

FUNCTION(clib,unlock,argc,argv)
{
#ifdef USE_THREADS
  my_mutex_t *mut;
  if (argc == 1 && isobj(argv[0], type(Mutex), (void**)&mut)) {
    check_mut(mut);
    if (pthread_mutex_unlock(&mut->mut))
      return __FAIL;
    else
      return mkvoid;
  } else
#endif
    return __FAIL;
}

FUNCTION(clib,try,argc,argv)
{
#ifdef USE_THREADS
  my_mutex_t *mut;
  my_sem_t *sem;
  if (argc == 1) {
    int n;
    expr x = argv[0], *v;
    double t;
    struct timespec tspec;
    bool tset = 0;
    if (istuple(x, &n, &v) && n == 2 &&
	(isfloat(v[1], &t) || ismpz_float(v[1], &t))) {
      double ip, fp;
      unsigned long secs;
      unsigned long nsecs;
      fp = modf(t, &ip);
      if (ip > LONG_MAX) { ip = (double)LONG_MAX; fp = 0.0; }
      secs = (unsigned long)ip;
      nsecs = (unsigned long)(fp*1e9);
      tspec.tv_sec = secs; tspec.tv_nsec = nsecs;
      x = v[0];
      tset = 1;
    }
    if (isobj(x, type(Mutex), (void**)&mut)) {
      int res;
      check_mut(mut);
#ifdef HAVE_PTHREAD_MUTEX_TIMEDLOCK
      if (tset) {
	release_lock();
	res = pthread_mutex_timedlock(&mut->mut, &tspec);
	acquire_lock();
      } else
#endif
	res = pthread_mutex_trylock(&mut->mut);
      if (res)
	return __FAIL;
      else
	return mkvoid;
    } else if (isobj(x, type(Semaphore), (void**)&sem)) {
      int res;
      expr val;
      check_sem(sem);
      release_lock();
#ifdef HAVE_SEM_TIMEDWAIT
      if (tset)
	res = sem_timedwait(sem->semp, &tspec);
      else
#endif
	res = sem_trywait(sem->semp);
      if (!res) {
	pthread_mutex_lock(&sem->mut);
	if (sem->queue.size <= 0)
	  res = -1;
	else {
	  val = dequeue_expr(&sem->queue);
	  if (sem->max) pthread_cond_signal(&sem->cond);
	}
	pthread_mutex_unlock(&sem->mut);
      }
      acquire_lock();
      if (res)
	return __FAIL;
      else
	return unref(val);
    } else
      return __FAIL;
  } else
#endif
    return __FAIL;
}

FUNCTION(clib,signal,argc,argv)
{
#ifdef USE_THREADS
  my_cond_t *cond;
  if (argc == 1 && isobj(argv[0], type(Condition), (void**)&cond)) {
    int res;
    check_cond(cond);
    pthread_mutex_lock(&cond->mut);
    res = pthread_cond_signal(&cond->cond);
    if (!res) cond->set = 1;
    pthread_mutex_unlock(&cond->mut);
    if (res)
      return __FAIL;
    else
      return mkvoid;
  } else
#endif
    return __FAIL;
}

FUNCTION(clib,broadcast,argc,argv)
{
#ifdef USE_THREADS
  my_cond_t *cond;
  if (argc == 1 && isobj(argv[0], type(Condition), (void**)&cond)) {
    int res;
    check_cond(cond);
    pthread_mutex_lock(&cond->mut);
    res = pthread_cond_broadcast(&cond->cond);
    if (!res) cond->set = 1;
    pthread_mutex_unlock(&cond->mut);
    if (res)
      return __FAIL;
    else
      return mkvoid;
  } else
#endif
    return __FAIL;
}

FUNCTION(clib,await,argc,argv)
{
#ifdef USE_THREADS
  my_cond_t *cond;
  int n;
  expr *v;
  double t;
  struct timespec tspec;
  bool tset = 0;

  if (argc == 1 &&
      (isobj(argv[0], type(Condition), (void**)&cond) ||
       istuple(argv[0], &n, &v) && n == 2 && (tset = 1) &&
       isobj(v[0], type(Condition), (void**)&cond) &&
       (isfloat(v[1], &t) || ismpz_float(v[1], &t)))) {
    int res = 0;
    check_cond(cond);
    if (tset) {
      double ip, fp;
      unsigned long secs;
      unsigned long nsecs;
      fp = modf(t, &ip);
      if (ip > LONG_MAX) { ip = (double)LONG_MAX; fp = 0.0; }
      secs = (unsigned long)ip;
      nsecs = (unsigned long)(fp*1e9);
      tspec.tv_sec = secs; tspec.tv_nsec = nsecs;
    }
    pthread_cleanup_push(my_mutex_unlock, (void*)&cond->mut);
    pthread_mutex_lock(&cond->mut);
    release_lock();
    cond->set = 0;
    while (!cond->set && !res)
      if (tset)
	res = pthread_cond_timedwait(&cond->cond, &cond->mut, &tspec);
      else
	res = pthread_cond_wait(&cond->cond, &cond->mut);
    pthread_cleanup_pop(1);
    acquire_lock();
    if (res)
      return __FAIL;
    else
      return mkvoid;
  } else
#endif
    return __FAIL;
}

FUNCTION(clib,post,argc,argv)
{
#ifdef USE_THREADS
  my_sem_t *sem;
  if (argc == 2 && isobj(argv[0], type(Semaphore), (void**)&sem)) {
    int res = 0;
    check_sem(sem);
    if (sem->max) {
      pthread_cleanup_push(my_mutex_unlock, (void*)&sem->mut);
      pthread_mutex_lock(&sem->mut);
      release_lock();
      while (sem->queue.size >= sem->max && !res)
	res = pthread_cond_wait(&sem->cond, &sem->mut);
      if (!res) {
	if (!enqueue_expr(&sem->queue, argv[1]))
	  res = -1;
	if (!res) {
	  res = sem_post(sem->semp);
	  if (res) unenqueue_expr(&sem->queue);
	}
      }
      pthread_cleanup_pop(1);
      acquire_lock();
    } else {
      pthread_mutex_lock(&sem->mut);
      if (!enqueue_expr(&sem->queue, argv[1]))
	res = -1;
      if (!res) {
	res = sem_post(sem->semp);
	if (res) unenqueue_expr(&sem->queue);
      }
      pthread_mutex_unlock(&sem->mut);
    }
    if (res)
      return (res==-1)?__ERROR:__FAIL;
    else
      return mkvoid;
  } else
#endif
    return __FAIL;
}

FUNCTION(clib,get,argc,argv)
{
  expr *x;
  if (argc != 1) return __FAIL;
#ifdef USE_THREADS
  my_sem_t *sem;
  if (isobj(argv[0], type(Ref), (void**)&x))
    return *x;
  else if (isobj(argv[0], type(Semaphore), (void**)&sem)) {
    int res;
    expr val;
    check_sem(sem);
    release_lock();
  retry:
    res = sem_wait(sem->semp);
    if (!res) {
      pthread_mutex_lock(&sem->mut);
      if (sem->queue.size <= 0) {
	pthread_mutex_unlock(&sem->mut);
	goto retry;
      }
      val = dequeue_expr(&sem->queue);
      if (sem->max) pthread_cond_signal(&sem->cond);
      pthread_mutex_unlock(&sem->mut);
    }
    acquire_lock();
    if (res)
      return __FAIL;
    else
      return unref(val);
  } else
#else
  if (isobj(argv[0], type(Ref), (void**)&x))
    return *x;
  else
#endif
    return __FAIL;
}

FUNCTION(clib,get_size,argc,argv)
{
#ifdef USE_THREADS
  my_sem_t *sem;
  if (argc == 1 && isobj(argv[0], type(Semaphore), (void**)&sem)) {
    long size;
    check_sem(sem);
    pthread_mutex_lock(&sem->mut);
    size = sem->queue.size;
    pthread_mutex_unlock(&sem->mut);
    return mkint(size);
  } else
#endif
    return __FAIL;
}

FUNCTION(clib,get_bound,argc,argv)
{
#ifdef USE_THREADS
  my_sem_t *sem;
  if (argc == 1 && isobj(argv[0], type(Semaphore), (void**)&sem)) {
    check_sem(sem);
    return mkint(sem->max);
  } else
#endif
    return __FAIL;
}

/* filename globbing: *****************************************************/

FUNCTION(clib,fnmatch,argc,argv)
{
#ifdef HAVE_FNMATCH
  char *pattern, *s;
  if (argc == 2 && isstr(argv[0], &pattern) && isstr(argv[1], &s)) {
    int res;
    pattern = utf8_to_sys(pattern); s = utf8_to_sys(s);
    if (!pattern || !s) {
      if (pattern) free(pattern); if (s) free(s);
      return __ERROR;
    }
    res = fnmatch(pattern, s, 0);
    free(pattern); free(s);
    if (res)
      return mkfalse;
    else
      return mktrue;
  } else
#endif
    return __FAIL;
}

FUNCTION(clib,glob,argc,argv)
{
#ifdef HAVE_GLOB
  char *pattern;
  if (argc == 1 && isstr(argv[0], &pattern)) {
    glob_t g;
    int res;
    g.gl_offs = 0;
    pattern = utf8_to_sys(pattern);
    if (!pattern) return __ERROR;
    res = glob(pattern, 0, NULL, &g);
    free(pattern);
    if (res == GLOB_NOMATCH)
      return mknil;
    else if (res)
      return __FAIL;
    else {
      expr x = mknil;
      int i = g.gl_pathc;
      while (x && --i >= 0)
	x = mkcons(mkstr(sys_to_utf8(g.gl_pathv[i])), x);
      globfree(&g);
      if (x)
	return x;
      else
	return __ERROR;
    }
  } else
#endif
    return __FAIL;
}

/* regular expression matching: *******************************************/

#ifdef HAVE_REGCOMP

/* regexp stack */

typedef struct {
  unsigned done:1, global:2, matched:1;
  int cflags, eflags;
  regex_t rx;
  regmatch_t *matches;
  char *s, *p, *start;
} regstate_t;

long regalloc = 0;
regstate_t *regstack = NULL, *regp = NULL;
char regmsg[BUFSZ];

#define REGALLOC 50

static int reg_push(void)
{
  if (!regstack)
    if ((regstack = malloc(REGALLOC*sizeof(regstate_t)))) {
      regalloc = REGALLOC;
      regp = regstack;
    } else
      return -1;
  else if (!regp)
    regp = regstack;
  else if (regp-regstack+1 == regalloc) {
    regstate_t *newstack = realloc(regstack,
				   (regalloc+REGALLOC)*sizeof(regstate_t));
    if (newstack) {
      regstack = newstack;
      regp = regstack+regalloc;
      regalloc += REGALLOC;
    } else
      return -1;
  } else
    regp++;
  regp->done = regp->global = regp->matched = 0;
  regp->cflags = regp->eflags = 0;
  regp->matches = NULL;
  regp->s = regp->p = regp->start = NULL;
  return 0;
}

static void reg_pop(void)
{
  if (!regp) return;
  regfree(&regp->rx);
  if (regp->matches) free(regp->matches);
  if (regp->s) free(regp->s);
  if (regp > regstack)
    regp--;
  else
    regp = NULL;
}

/* push a new expression on the stack */

static int reg_add(char *pattern, char *s, int global, int cflags, int eflags)
{
  int ret;
  if (regp && regp->done) reg_pop();
  if (reg_push()) return -1;
  regp->global = global;
  regp->cflags = cflags;
  regp->eflags = eflags;
  ret = regcomp(&regp->rx, pattern, REG_EXTENDED|cflags);
  *regmsg = 0;
  if (ret) {
    regerror(ret, &regp->rx, regmsg, BUFSZ);
    reg_pop();
    return ret;
  }
  if (!(regp->s = strdup(s))) {
    reg_pop();
    return -1;
  }
  regp->p = regp->s; regp->start = NULL;
  if (!(regp->matches = malloc((regp->rx.re_nsub+1)*sizeof(regmatch_t)))) {
    reg_pop();
    return -1;
  }
  return 0;
}

/* search */

static int reg_flags(char *p)
{
  int flags;
  flags = regp->eflags;
  if (p > regp->s)
    if (regp->cflags & REG_NEWLINE)
      if (p[-1] == '\n')
	flags &= ~REG_NOTBOL;
      else
	flags |= REG_NOTBOL;
    else
      flags |= REG_NOTBOL;
  return flags;
}

static int reg_search(void)
{
  int ret;
  char *prev;
  while (regp && regp->done && regp>regstack) reg_pop();
  if (!regp) return -1;
  if (regp->matched)
    /* note the beginning of the previous match */
    prev = regp->start+regp->matches[0].rm_so;
  regp->start = regp->p;
  if (regp->global || !regp->matched) {
    ret = regexec(&regp->rx, regp->p, regp->rx.re_nsub+1, regp->matches,
		  reg_flags(regp->p));
    if (!ret) {
      if (regp->matched)
	if (regp->matches[0].rm_eo == regp->matches[0].rm_so &&
	    regp->p == prev)
	  /* an extra empty match: if not at end of string then advance to the
	     next position and try again; otherwise simply ignore this match
	     and fail */
	  if (*regp->p) {
	    int i;
	    /* this cannot fail since we can always match the empty string */
	    ret = regexec(&regp->rx, regp->p+1, regp->rx.re_nsub+1,
			  regp->matches, reg_flags(regp->p+1));
	    /* translate offsets */
	    for (i = 0; i <= regp->rx.re_nsub; i++) {
	      regp->matches[i].rm_so++;
	      regp->matches[i].rm_eo++;
	    }
	  } else
	    ret = REG_NOMATCH;
      regp->matched = 1;
    }
  } else
    ret = REG_NOMATCH;
  *regmsg = 0;
  if (ret) {
    regp->done = 1;
    regerror(ret, &regp->rx, regmsg, BUFSZ);
  } else if (regp->global == 2 &&
	     regp->matches[0].rm_eo > regp->matches[0].rm_so)
    regp->p += regp->matches[0].rm_so+1;
  else
    regp->p += regp->matches[0].rm_eo;
  return ret;
}

/* stop search */

static void reg_done(void)
{
  if (regp) {
    regp->start = regp->p;
    regp->done = 1;
  }
}

/* return matches */

static size_t reg_nmatches(void)
{
  if (regp)
    return regp->rx.re_nsub;
  else
    return 0;
}

static long reg_start(void)
{
  if (regp && regp->start)
    return regp->start-regp->s;
  else
    return -1;
}

static char *reg_skipstr(void)
{
  if (regp && regp->start)
    return regp->start;
  else
    return NULL;
}

static long reg_pos(int i)
{
  if (regp && regp->start && 0 <= i && i <= regp->rx.re_nsub)
    if (!regp->done && regp->matches[i].rm_so >= 0)
      return regp->start+regp->matches[i].rm_so-regp->s;
    else
      return -1;
  else
    return -1;
}

static long reg_end(int i)
{
  if (regp && regp->start && 0 <= i && i <= regp->rx.re_nsub)
    if (!regp->done && regp->matches[i].rm_eo >= 0)
      return regp->start+regp->matches[i].rm_eo-regp->s;
    else
      return -1;
  else
    return -1;
}

static char *reg_str(int i)
{
  if (regp && regp->start && 0 <= i && i <= regp->rx.re_nsub)
    if (!regp->done && regp->matches[i].rm_so >= 0)
      return regp->start+regp->matches[i].rm_so;
    else
      return NULL;
  else
    return NULL;
}

#endif

/* interface functions */

FUNCTION(clib,regmatch,argc,argv)
{
#ifdef HAVE_REGCOMP
  char *opts, *regex, *s;
  int cflags = 0, eflags = 0, global = 0, ret;
  if (argc != 3 || !isstr(argv[0], &opts) || !isstr(argv[1], &regex) ||
      !isstr(argv[2], &s))
    return __FAIL;
  while (*opts)
    switch (*(opts++)) {
    case 'g':
      if (!global) global = 1;
      break;
    case 'G':
      global = 2;
      break;
    case 'i':
      cflags |= REG_ICASE;
      break;
    case 'n':
      cflags |= REG_NEWLINE;
      break;
    case '^':
      eflags |= REG_NOTBOL;
      break;
    case '$':
      eflags |= REG_NOTEOL;
      break;
    default:
      return __FAIL;
    }
  regex = utf8_to_sys(regex); s = utf8_to_sys(s);
  if (!regex || !s) {
    if (regex) free(regex); if (s) free(s);
    return __ERROR;
  }
  ret = reg_add(regex, s, global, cflags, eflags);
  free(regex); free(s);
  if (ret == -1)
    return __ERROR;
  else if (ret)
    return mkapp(mksym(sym(regerr)), mkstr(sys_to_utf8(regmsg)));
  ret = reg_search();
  if (ret == -1 || ret == REG_NOMATCH)
    return mkfalse;
  else if (ret)
    return mkapp(mksym(sym(regerr)), mkstr(sys_to_utf8(regmsg)));
  else
    return mktrue;
#else
  return __FAIL;
#endif
}

FUNCTION(clib,regnext,argc,argv)
{
#ifdef HAVE_REGCOMP
  int ret;
  if (argc != 0) return __FAIL;
  ret = reg_search();
  if (ret == -1 || ret == REG_NOMATCH)
    return mkfalse;
  else if (ret)
    return mkapp(mksym(sym(regerr)), mkstr(sys_to_utf8(regmsg)));
  else
    return mktrue;
#else
  return __FAIL;
#endif
}

FUNCTION(clib,regdone,argc,argv)
{
#ifdef HAVE_REGCOMP
  if (argc != 0) return __FAIL;
  reg_done();
  return mkvoid;
#else
  return __FAIL;
#endif
}

FUNCTION(clib,regstart,argc,argv)
{
#ifdef HAVE_REGCOMP
  long start;
  if (argc != 0) return __FAIL;
  start = reg_start();
  if (start >= 0)
    return mkint(start);
  else
    return __FAIL;
#else
  return __FAIL;
#endif
}

FUNCTION(clib,regskip,argc,argv)
{
#ifdef HAVE_REGCOMP
  char *skip;
  if (argc != 0) return __FAIL;
  if ((skip = reg_skipstr())) {
    long start = reg_start(), pos = reg_pos(0);
    char *s, *t;
    if (pos >= start)
      s = malloc(pos-start+1);
    else
      s = malloc(strlen(skip)+1);
    if (!s) return __ERROR;
    if (pos >= start) {
      strncpy(s, skip, pos-start);
      s[pos-start] = 0;
    } else
      strcpy(s, skip);
    t = sys_to_utf8(s); free(s);
    return mkstr(t);
  } else
    return __FAIL;
#else
  return __FAIL;
#endif
}

FUNCTION(clib,reg,argc,argv)
{
#ifdef HAVE_REGCOMP
  long i;
  if (argc != 1 || !isint(argv[0], &i) || i < 0 || i > reg_nmatches())
    return __FAIL;
  if (reg_start() >= 0) {
    long pos = reg_pos(i), end = reg_end(i);
    char *s, *t;
    if (pos < 0 || end < 0)
      s = strdup("");
    else if (!(s = malloc(end-pos+1)))
      return __ERROR;
    else {
      strncpy(s, reg_str(i), end-pos);
      s[end-pos] = 0;
    }
    t = sys_to_utf8(s); free(s);
    return mkstr(t);
  } else
    return __FAIL;
#else
  return __FAIL;
#endif
}

FUNCTION(clib,regpos,argc,argv)
{
#ifdef HAVE_REGCOMP
  long i;
  if (argc != 1 || !isint(argv[0], &i) || i < 0 || i > reg_nmatches())
    return __FAIL;
  if (reg_start() >= 0)
    return mkint(reg_pos(i));
  else
    return __FAIL;
#else
  return __FAIL;
#endif
}

FUNCTION(clib,regend,argc,argv)
{
#ifdef HAVE_REGCOMP
  long i;
  if (argc != 1 || !isint(argv[0], &i) || i < 0 || i > reg_nmatches())
    return __FAIL;
  if (reg_start() >= 0)
    return mkint(reg_end(i));
  else
    return __FAIL;
#else
  return __FAIL;
#endif
}

FUNCTION(clib,regs,argc,argv)
{
#ifdef HAVE_REGCOMP
  expr x;
  size_t i;
  if (argc != 0) return __FAIL;
  x = mknil;
  i = reg_nmatches();
  while (x && i > 0) {
    if (reg_pos(i) >= 0 && reg_end(i) >= 0)
      x = mkcons(mkint(i), x);
    i--;
  }
  if (x)
    return x;
  else
    return __ERROR;
#else
  return __FAIL;
#endif
}

/* initialization: ********************************************************/

#ifdef USE_THREADS

/* do necessary cleanup at fork time */

static void atfork_child(void)
{
  THREAD *thr, *this = thr0+this_thread();
  for (thr = thr0; thr <= thr0+maxused; thr++)
    if (thr->used) {
      pthread_mutex_init(&thr->exit_mutex, NULL);
      pthread_cond_init(&thr->exit_cond, NULL);
      if (thr == this)
	thr->id = pthread_self();
      else {
	/* XXX FIXME: experimental stuff here. Second attempt to prevent
	   waiting for dead child threads in thread destructor (see also
	   ChangeLog entry from 2002-08-29). */
#if 0
	thr->active = 0; thr->canceled = 1;
#else
	/* this thread does not exist in the child any more, so we collect its
	   resources */
	thr->active = thr->canceled = thr->used = 0;
	if (thr->arg) {
	  freeref(thr->arg);
	  thr->arg = NULL;
	}
	if (thr->result) {
	  freeref(thr->result);
	  thr->result = NULL;
	}
	if (thr->thread) thr->thread = NULL;
#endif
      }
    }
  stamp++;
#ifdef DEBUG
  printf("thread %d (id %d) forked\n", this-thr0, (int)this->id);
#endif
}

#endif

INIT(clib)
{
  thr0->active = 1; thr0->canceled = 0; thr0->used = 1;
  thr0->result = NULL;
  maxused = 0;
#ifdef USE_THREADS
  thr0->id = pthread_self();
#ifdef WIN32
  thr0->handle = GetCurrentThread();
  DuplicateHandle(GetCurrentProcess(), GetCurrentThread(),
		  GetCurrentProcess(), &thr0->handle,
		  0, FALSE, DUPLICATE_SAME_ACCESS);
  thr0->pol = 0;
#endif
  pthread_mutex_init(&thr0->exit_mutex, NULL);
  pthread_cond_init(&thr0->exit_cond, NULL);
  pthread_mutex_init(&format_mutex, NULL);
  thread_atfork(NULL, NULL, atfork_child);
#ifdef __linux__
  { FILE *fp;
    const char * givertcap = getenv("GIVERTCAP");
    if(!givertcap) givertcap = "/usr/local/bin/givertcap";
    if ((fp = fopen(givertcap, "r"))) {
      fclose(fp);
      system(givertcap);
    }
  }
#endif
#endif
}
