// main45.cc is a part of the PYTHIA event generator.
// Copyright (C) 2020 Torbjorn Sjostrand.
// PYTHIA is licenced under the GNU GPL v2 or later, see COPYING for details.
// Please respect the MCnet Guidelines, see GUIDELINES for details.

// Author: Stefan Prestel <stefan.prestel@thep.lu.se>.

// Keywords: LHE file; hepmc;

// This program (main45.cc) illustrates how a file with HepMC3 events can be
// generated by Pythia8. See main44.cc for how to ouput HepMC2 events instead.
// Note: both main44.cc and main45.cc can use the same main44.cmnd input card.

#include "Pythia8/Pythia.h"
#include "Pythia8Plugins/HepMC3.h"
#include <unistd.h>

using namespace Pythia8;

//==========================================================================

// Example main programm to illustrate merging.

int main( int argc, char* argv[] ){

  // Check that correct number of command-line arguments
  if (argc != 3) {
    cerr << " Unexpected number of command-line arguments ("<<argc<<"). \n"
         << " You are expected to provide the arguments" << endl
         << " 1. Input file for settings" << endl
         << " 2. Output file for HepMC events" << endl
         << " Program stopped. " << endl;
    return 1;
  }

  Pythia pythia;

  // Input parameters:
  pythia.readFile(argv[1],0);

  // Interface for conversion from Pythia8::Event to HepMC one.
  HepMC3::Pythia8ToHepMC3 toHepMC;
  // Specify file where HepMC events will be stored.
  HepMC3::WriterAscii ascii_io(argv[2]);

  // Allow abort of run if many errors.
  int  nAbort  = pythia.mode("Main:timesAllowErrors");
  int  iAbort  = 0;
  bool doAbort = false;

  cout << endl << endl << endl;
  cout << "Start generating events" << endl;

  long nEvent = pythia.settings.mode("Main:numberOfEvents");
  int nRuns = pythia.mode("Main:numberOfSubruns");

  double sigmaTotal(0.), errorTotal(0.);

  bool wroteRunInfo = false;

  // Loop over subruns with varying number of jets.
  for (int iRuns = 0; iRuns < nRuns; ++iRuns) {

    double sigmaSample = 0., errorSample = 0.;

    // Read in name of LHE file for current subrun and initialize.
    pythia.readFile(argv[1], iRuns);

    // Initialise.
    pythia.init();

    // Get the inclusive x-section by summing over all process x-sections.
    double xs = 0.;
    for (int i=0; i < pythia.info.nProcessesLHEF(); ++i)
      xs += pythia.info.sigmaLHEF(i);

    // Start generation loop
    while( pythia.info.nSelected() < nEvent ){

      // Generate next event
      if( !pythia.next() ) {
        if ( pythia.info.atEndOfFile() ) break;
        else if (++iAbort > nAbort) {doAbort = true; break;}
        else continue;
      }

      // Get event weight(s).
      double evtweight         = pythia.info.weight();

      // Do not print zero-weight events.
      if ( evtweight == 0. ) continue;

      // Create a GenRunInfo object with the necessary weight names and write
      // them to the HepMC3 file only once.
      if (!wroteRunInfo) {
        shared_ptr<HepMC3::GenRunInfo> genRunInfo;
        genRunInfo = make_shared<HepMC3::GenRunInfo>();
        vector<string> weight_names = pythia.info.weightNameVector();
        genRunInfo->set_weight_names(weight_names);
        ascii_io.set_run_info(genRunInfo);
        ascii_io.write_run_info();
        wroteRunInfo = true;
      }

      // Construct new empty HepMC event.
      HepMC3::GenEvent hepmcevt;

      // Work with weighted (LHA strategy=-4) events.
      double normhepmc = 1.;
      if (abs(pythia.info.lhaStrategy()) == 4)
        normhepmc = 1. / double(1e9*nEvent);
      // Work with unweighted events.
      else
        normhepmc = xs / double(1e9*nEvent);

      // Set event weight
      //hepmcevt.weights().push_back(evtweight*normhepmc);
      // Fill HepMC event
      toHepMC.fill_next_event( pythia, &hepmcevt );
      // Add the weight of the current event to the cross section.
      sigmaTotal  += evtweight*normhepmc;
      sigmaSample += evtweight*normhepmc;
      errorTotal  += pow2(evtweight*normhepmc);
      errorSample += pow2(evtweight*normhepmc);
      // Report cross section to hepmc
      shared_ptr<HepMC3::GenCrossSection> xsec;
      xsec = make_shared<HepMC3::GenCrossSection>();
      // First add object to event, then set cross section. This order ensures
      // that the lengths of the cross section and the weight vector agree.
      hepmcevt.set_cross_section( xsec );
      xsec->set_cross_section( sigmaTotal*1e9, pythia.info.sigmaErr()*1e9 );
      // Write the HepMC event to file. Done with it.
      ascii_io.write_event(hepmcevt);

    } // end loop over events to generate.
    if (doAbort) break;

    // print cross section, errors
    pythia.stat();

    cout << endl << " Contribution of sample " << iRuns
         << " to the inclusive cross section : "
         << scientific << setprecision(8)
         << sigmaSample << "  +-  " << sqrt(errorSample)  << endl;

  }

  cout << endl << endl << endl;
  if (doAbort)
    cout << " Run was not completed owing to too many aborted events" << endl;
  cout << endl << endl << endl;

  // Done
  return 0;

}
