#
# ltorsions.py 
#
# Copyright (C) 2005  Dr. Stephane Gagne
# the full copyright notice is found in the LICENSE file in this directory
#
# This program is free software; 
# you can redistribute it and/or modify it under the terms of the 
# GNU General Public License as published by the Free Software Foundation; 
# either version 2 of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, 
# but WITHOUT ANY WARRANTY; without even the implied warranty of 
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
# See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License 
# along with this program; if not, write to the Free Software Foundation, Inc., 
# 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#
# Contact:  nmr@rsvs.ulaval.ca
#


from string import *
from math import sqrt, pi, acos
import os.path
import common


#------------------------------------------------------------------------
# OVERVIEW:  read in all relevant data from the pdb file.
# INPUT:    the filename of the pdb file
# OUTPUT:   no output from this routine, but the global variable
#           _mypdb contains the pdb information.
#

def _read_pdb_file(fname):
    global _mypdb, _resnums
    
    _resnums = list()
    _mypdb = dict()

    fullfname = os.path.join(common.workingdir, fname)
    pdbf = open(fullfname, "r")
    try:
        pdblines = pdbf.readlines()
    except:
        print "** ERROR: cannot open file", fullfname
        sys.exit(1)
    pdbf.close()


    for line in pdblines:
        if ((line.find("ATOM")) == -1 and (line.find("HETATM") == -1)):
            continue


        key = rstrip(strip(line[0:6]))
#        anum = rstrip(strip(line[6:11]))
        atom = rstrip(strip(line[12:17]))
        res = rstrip(strip(line[17:21]))
#        chain = rstrip(strip(line[21]))
        resnum = rstrip(strip(line[22:26]))
#        insert = rstrip(strip(line[26]))
        x = rstrip(strip(line[30:38]))
        y = rstrip(strip(line[38:46]))
        z = rstrip(strip(line[46:54]))
#        occ = rstrip(strip(line[54:60]))
#        bval = rstrip(strip(line[60:66]))

#       store unique residue numbers
        try:
            _resnums.index((resnum, res))
        except ValueError:
            _resnums += [(resnum, res)]

        if (key == "ATOM") or (key == "HETATM"):
            if (atom in keepatoms):
                _mypdb[(resnum, atom)] = (float(x), float(y), float(z))

#------------------------------------------------------------------------
# OVERVIEW: calculate the normal to 2 planes, which is
#           defined as the cross products:
#           P1 x P2
#           ----------
#           | P1 x P2 |
#
# INPUT:    P1, P2  the 2 planes
# OUTPUT:   a vector

def _normal(P1, P2):
    normal = [0,0,0]
    normal[0] = P1[1]*P2[2] - P1[2]*P2[1]
    normal[1] = P1[2]*P2[0] - P1[0]*P2[2]
    normal[2] = P1[0]*P2[1] - P1[1]*P2[0]

    return normal

#------------------------------------------------------------------------
# OVERVIEW:  return the angle between 2 vectors
# INPUT:    V1, V2 the two vectors
# OUTPUT:   a real number
# NOTE:     formula taken from mathworld.wolfram.com/DihedralAngle.html

def _get_angle(V1, V2, cdx):

    d1 = sqrt(V1[0]**2 + V1[1]**2 + V1[2]**2)
    d2 = sqrt(V2[0]**2 + V2[1]**2 + V2[2]**2)

    costheta = V1[0]* V2[0] + V1[1] * V2[1] + V1[2] * V2[2]
    costheta = costheta / (d1 * d2)

#   don't think we need these next 2 lines, but just in case...
    if costheta > 1.0: costheta = 1.0
    if costheta < -1.0: costheta = -1.0

    theta = acos(costheta) * 180.0 / pi

#   check for - sign:
#   without this check, the final angle is always positive. 
    check = (cdx[1][0]-cdx[2][0]) * (V1[2] * V2[1] - V1[1] * V2[2]) + \
            (cdx[1][1]-cdx[2][1]) * (V1[0] * V2[2] - V1[2] * V2[0]) + \
            (cdx[1][2]-cdx[2][2]) * (V1[1] * V2[0] - V1[0] * V2[1]) 
    if (check < 0.0): theta = -theta

    return theta


#------------------------------------------------------------------------
# OVERVIEW: given 4 sets of coordinates, returns the torsion 
#           angle between the 4 atoms.
# INPUT:  a list containing 4 tuples of coordinates
#           [(xA, yA, zA), (xB, yB, zB), (xC, yC, zC), (xD, yD, zD)]
# OUTPUT: returns a real number.
#
def _calc_torsion(cdx):

# assume that atoms are called A, B, C, and D.
# calculate 3 vectors.  A<->B    C<->B     C<->D
    AB = (cdx[0][0]-cdx[1][0], cdx[0][1]-cdx[1][1], cdx[0][2]-cdx[1][2])
    BC = (cdx[1][0]-cdx[2][0], cdx[1][1]-cdx[2][1], cdx[1][2]-cdx[2][2])
    CD = (cdx[2][0]-cdx[3][0], cdx[2][1]-cdx[3][1], cdx[2][2]-cdx[3][2])

    N1 = _normal(AB, BC)
    N2 = _normal(BC, CD)

    torsion = _get_angle(N1, N2, cdx)

    return torsion



#------------------------------------------------------------------------
#
# main routine to calculate torsion angles
#
def _do_torsions():
    global _torsions
    _torsions = dict()
    pdbsort = list()
    
#   give me a sorted list of _resnums and atoms.  
#   right now it's in a dictionary
#   _resnums is a sorted list of residue numbers.
    for (resnum, resname) in _resnums:
        for atom in keepatoms:
            try: coords = _mypdb[resnum, atom]
            except: continue
            pdbsort += [(resnum, atom, coords)]


    for line in pdbsort: 
        (resnum, type, angles) = line
        position = pdbsort.index(line)


#       use this atom, plus next 3 in sequence
        try:
            four_atoms = [angles, 
                pdbsort[position+1][2], 
                pdbsort[position+2][2],
                pdbsort[position+3][2]]
        except IndexError:
            continue


#       which angle did we just measure?
#       if we used N, CA, C, N, then it is psi (index 1)
#       if we used CA, C, N, CA then it is omega (index 2)
#       if we used C, N, CA, C, then it is phi (index 0), of residue i+1
        if type == "N": idx = 1
        if type == "CA": idx = 2
        if type == "C": idx = 0; resnum = pdbsort[position+1][0]


        mytors = _calc_torsion(four_atoms)
        _torsions[(resnum, idx)] = mytors
#        print res, resnum, type[idx], "TORSION: ", mytors



#------------------------------------------------------------------------
# OVERVIEW: to print out the results
# INPUT:    none, but global variable _torsions is used
# OUTPUT:   nothing returned, but output written to screen
#
def _output_results():

    print "RESIDUE        PHI      PSI     OMEGA"
    print "-------------------------------------"


    for (resnum, resname) in _resnums:

        try:
            phi = _torsions[(resnum,0)]
        except KeyError:
            phi = 9999.000

        try:
            psi = _torsions[(resnum,1)]
        except KeyError:
            psi = 9999.000

        try:
            omega = _torsions[(resnum,2)]
        except KeyError:
            omega = 9999.000


        printline = '%5d  %3s  %8.3f %8.3f %8.3f' % \
            (int(resnum), resname, phi, psi, omega)
        print printline

#------------------------------------------------------------------------
# OVERVIEW: to calculate _torsions.
# INPUT:    
# OUTPUT:   nothing returned, but output written to screen
#
def get_torsions(fname):
    global keepatoms


#   list of atom types to keep.  order is important.
    keepatoms = ["N", "CA", "C"]

#   read the PDB file in - just the C, N, and CAs.
    _read_pdb_file(fname)

#   call the routine which loops through all pdb entries
    _do_torsions()

    return _torsions


#------------------------------------------------------------------------
# ***  MAIN PROGRAM ***
# if this section is called, then it means that the program was
# evoked from the command line ie. an interactive session, so
# go ahead and write the results to stdout.
#------------------------------------------------------------------------

if __name__ == "__main__": 

    fname = raw_input('>>> Give name of pdb file: ')

    get_torsions(fname)

#   write output
    _output_results()

