#!/usr/bin/python
__author__ = "Romain Gouaty"

import os
import sys
import commands

from optparse import *

from pylal import git_version

usage = """ %prog [options]
"""

parser = OptionParser(usage, version=git_version.verbose_msg)

parser.add_option("-f","--mcmc-file",action="store",type="string",\
    metavar=" FILE",help="list of .txt files (or list of path/files) generated by the MCMC code, used as input (there are several files because there are several chains running in parallel). The files in the list have to be separated by a coma.")

parser.add_option("-r","--plot-routine",action="store",type="string",\
    metavar=" FILE",help="path to the \"mcmcsummary.R\" script")

parser.add_option("-e","--executable",action="store",type="string",\
    metavar=" FILE",help="path to the R executable")

parser.add_option("-b","--burnin",action="store",type="string",\
    metavar=" VALUE",help="number of mcmc iterations to disregard.\
    If this argument is not specified, an autoburnin will be set up.")

parser.add_option("-t","--reference-time",action="store",type="string",\
    metavar=" GPS",help="GPS time to be used as reference")

parser.add_option("-C","--reference-mchirp",action="store",type="string",\
    metavar=" VALUE",help="VALUE chirp mass to be used as reference")

parser.add_option("-E","--reference-eta",action="store",type="string",\
    metavar=" VALUE",help="VALUE eta to be used as reference")

parser.add_option("-d","--reference-distance",action="store",type="string",\
    metavar=" VALUE",help="VALUE distance to be used as reference")

parser.add_option("-P","--reference-phi",action="store",type="string",\
    metavar=" VALUE",help="VALUE phi to be used as reference")

parser.add_option("-a","--reference-a_spin1",action="store",type="string",\
    metavar=" VALUE",help="VALUE a_spin1 to be used as reference")

parser.add_option("-x","--reference-cs_th_sp1",action="store",type="string",\
    metavar=" VALUE",help="VALUE cs_th_sp1 to be used as reference")

parser.add_option("-y","--reference-phi_spin1",action="store",type="string",\
    metavar=" VALUE",help="VALUE phi_spin1 to be used as reference")

parser.add_option("-A","--reference-a_spin2",action="store",type="string",\
    metavar=" VALUE",help="VALUE a_spin2 to be used as reference")

parser.add_option("-X","--reference-cs_th_sp2",action="store",type="string",\
    metavar=" VALUE",help="VALUE cs_th_sp2 to be used as reference")
	
parser.add_option("-Y","--reference-phi_spin2",action="store",type="string",\
    metavar=" VALUE",help="VALUE phi_spin2 to be used as reference")
	
parser.add_option("-o","--output-path",action="store",type="string",\
    metavar=" PATH",help="use output path PATH")

parser.add_option("-i","--identity",action="store",type="string",\
    metavar=" STRING",help="job identity STRING")

command_line = sys.argv[1:]
(opts,args) = parser.parse_args()

#################################
# Sanity check of input arguments

if not opts.mcmc_file:
  print >> sys.stderr, "No mcmc file specified."
  print >> sys.stderr, "Use --mcmc-file FILE to specify location."
  sys.exit(1)

if not opts.plot_routine:
  print >> sys.stderr, "No R routine specified."
  print >> sys.stderr, "Use --plot-routine FILE to specify location."
  sys.exit(1)

if not opts.executable:
  print >> sys.stderr, "No R executable specified."
  print >> sys.stderr, "Use --executable FILE to specify location."
  sys.exit(1)

if not opts.reference_time:
  print >> sys.stderr, "No reference time specified."
  print >> sys.stderr, "Use --reference-time GPS to specify location."
  sys.exit(1)

if not opts.reference_mchirp:
  print >> sys.stderr, "No reference mchirp specified."
  print >> sys.stderr, "Use --reference-mchirp VALUE to specify location."
  sys.exit(1)

if not opts.reference_eta:
  print >> sys.stderr, "No reference eta specified."
  print >> sys.stderr, "Use --reference-eta VALUE to specify location."
  sys.exit(1)

if not opts.reference_distance:
  print >> sys.stderr, "No reference distance specified."
  print >> sys.stderr, "Use --reference-distance VALUE to specify location."
  sys.exit(1)

if not opts.reference_phi:
  print >> sys.stderr, "No reference phi specified."
  print >> sys.stderr, "Use --reference-phi VALUE to specify location."
  sys.exit(1)
  
if not opts.reference_a_spin1:
  print >> sys.stderr, "No reference a_spin1 specified."
  print >> sys.stderr, "Use --reference-a_spin1 VALUE to specify location."
  sys.exit(1)

if not opts.reference_cs_th_sp1:
  print >> sys.stderr, "No reference cs_th_sp1 specified."
  print >> sys.stderr, "Use --reference-cs_th_sp1 VALUE to specify location."
  sys.exit(1)

if not opts.reference_phi_spin1:
  print >> sys.stderr, "No reference phi_spin1 specified."
  print >> sys.stderr, "Use --reference-phi_spin1 VALUE to specify location."
  sys.exit(1)

if not opts.reference_a_spin2:
  print >> sys.stderr, "No reference a_spin2 specified."
  print >> sys.stderr, "Use --reference-a_spin2 VALUE to specify location."
  sys.exit(1)

if not opts.reference_cs_th_sp2:
  print >> sys.stderr, "No reference cs_th_sp2 specified."
  print >> sys.stderr, "Use --reference-cs_th_sp2 VALUE to specify location."
  sys.exit(1)

if not opts.reference_phi_spin2:
  print >> sys.stderr, "No reference phi_spin2 specified."
  print >> sys.stderr, "Use --reference-phi_spin2 VALUE to specify location."
  sys.exit(1)

if not opts.output_path:
  print >> sys.stderr, "No output path specified."
  print >> sys.stderr, "Use --output-path PATH to specify location."
  sys.exit(1)

if not opts.identity:
  print >> sys.stderr, "No identity specified."
  print >> sys.stderr, "Use --identity STRING to specify it."
  sys.exit(1)

#################################
# Main program

file = open(opts.output_path + "/" + opts.identity + ".R",'w')

file.write("# load the \"mcmcsummary\" code:\n")
file.write("source(\"" + opts.plot_routine + "\")\n\n")

file.write("# load the data written in the MCMC txt file\n")
inputlist = opts.mcmc_file.split(',')
for k,inputfile in enumerate(inputlist):
  file.write("input" + str(k+1) + " <- read.table(\"" + inputfile + "\",header=TRUE,sep=\"\",dec=\".\",fill=TRUE,comment.char=\"\",skip=10)\n")

file.write("\n# keep only the tens of seconds for the GPS time. This will allow to display the decimal part of the gps time with a precision of 1.e-5 seconds.\n")
nb_chain = len(inputlist)
for k in range(nb_chain):
  file.write("input" + str(k+1) + "[,\"t_c\"] <- input" + str(k+1) + "[,\"t_c\"] %% 100\n")

file.write("\n# convert the logdistance into distance\n")
for k in range(nb_chain):
  file.write("input" + str(k+1) + "[,\"distance\"] <- exp(input" + str(k+1) + "[,\"log.d.\"])\n")

file.write("\n# Reorganize the input matrix with columns in the following order: chirpmass,massratio,time,phase,distance,logposterior,loglikelihood\n")
for k in range(nb_chain):
  file.write("newinput" + str(k+1) + " <- cbind(input" + str(k+1) + "[,\"Mc\"],input" + str(k+1) + "[,\"eta\"],input" + str(k+1) + "[,\"t_c\"],input" + str(k+1) + "[,\"phi_orb\"],input" + str(k+1) + "[,\"distance\"],input" + str(k+1) + "[,\"a_spin1\"],input" + str(k+1) + "[,\"cs_th_sp1\"],input" + str(k+1) + "[,\"phi_spin1\"],input" + str(k+1) + "[,\"a_spin2\"],input" + str(k+1) + "[,\"cs_th_sp2\"],input" + str(k+1) + "[,\"phi_spin2\"],input" + str(k+1) + "[,\"log_Post.\"],input" + str(k+1) + "[,\"log_Post.\"])\n")

file.write("\n# prepare the input data for the plotting routine\n")
inputdatalist = ""
for k in range(nb_chain):
  inputdatalist += "as.vector(as.matrix(newinput" + str(k+1) + ")),"
file.write("post <- array(c(" + inputdatalist.strip(',') + "), dim=c(nrow(newinput" + str(1) + "),13," + str(nb_chain) + "))\n") #we assume that we are looking at 13 parameters as follows...
file.write("colnames(post) <- c(\"chirpmass\",\"massratio\",\"time\",\"phase\",\"distance\",\"a_spin1\",\"cs_th_sp1\",\"phi_spin1\",\"a_spin2\",\"cs_th_sp2\",\"phi_spin2\",\"logposterior\",\"loglikelihood\")\n")

file.write("\n# enter injected or inspiral parameters\n")
file.write("injpar <- c(\"chirpmass\"=" + opts.reference_mchirp + ",\"massratio\"=" + opts.reference_eta + ",\"tc\"=" + opts.reference_time + ",\"phi\"=" + opts.reference_phi + ",\"dl\"=" + opts.reference_distance + ",\"a_spin1\"=" + opts.reference_a_spin1 + ",\"cs_th_sp1\"=" + opts.reference_cs_th_sp1 + ",\"phi_spin1\"=" + opts.reference_phi_spin1 + ",\"a_spin2\"=" + opts.reference_a_spin2 + ",\"cs_th_sp2\"=" + opts.reference_cs_th_sp2 + ",\"phi_spin2\"=" + opts.reference_phi_spin2 + ",\"logpost\"= NA, \"loglikeli\"= NA" + ")\n")
file.write("injpar[\"tc\"] <- injpar[\"tc\"] %% 100\n\n")

if not opts.burnin:
  file.write("# prepare a matrix of posterior density values. Use in order to determine the burnin for each chain\n")
  posteriorlist = ""
  for k in range(nb_chain):
    posteriorlist += "input" + str(k+1) + "[,\"log_Post.\"],"
  file.write("posteriorvalues <- cbind(" + posteriorlist.strip(',') + ")\n\n")
  file.write("# execute the \"mcmcsummary\" code:\n")
  file.write("mcmcsummary(data=post,targetdirectory=\"" + opts.output_path + "/" + opts.identity + "\",iteration=input1[,1],autoburnin=T,posterior=posteriorvalues,varnames=colnames(post),truevalue=injpar,graphicsformats = c(\"png\"), overwrite = T)")


else:
  burninlist = ""
  for k in range(nb_chain):
    burninlist += opts.burnin + ","  
  file.write("# execute the \"mcmcsummary\" code:\n")
  file.write("mcmcsummary(data=post,targetdirectory=\"" + opts.output_path + "/" + opts.identity + "\",iteration=input1[,1],burnin=c(" + burninlist.strip(',') + "),varnames=colnames(post),truevalue=injpar,graphicsformats = c(\"png\"), overwrite = T)")

file.close()

command = opts.executable + " --slave --vanilla --file=" + opts.output_path + "/" + opts.identity + ".R"

result = commands.getstatusoutput(command)
#result = commands.getoutput(command)

if not result[0] == 0:
  print >> sys.stderr, "status=" + str(result[0])
  print >> sys.stderr, result[1]
  sys.exit(1)

