#Copyright 2009 Diego Duclos
#
#This program is free software: you can redistribute it and/or modify
#it under the terms of the GNU General Public License as published by
#the Free Software Foundation, either version 3 of the License, or
#(at your option) any later version.
#
#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License
#along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os.path
import urllib
import shutil
import ConfigParser
from multiprocessing import Process, Pipe

scheduledUpdaters = set()
    
def scheduleForUpdate(updater):
    scheduledUpdaters.add(updater)

def runAllUpdatersThreaded():
    parent, child = Pipe(False)
    p = Process(target=runAllUpdaters, args=(child,scheduledUpdaters))
    p.start()
    return parent, p

def runAllUpdaters(conn, scheduledUpdaters):
    for updater in scheduledUpdaters:
        updater.setConnection(conn)
        updater.update()
        
    conn.send(("done", None))

def isUpdateNeeded():
    for updater in scheduledUpdaters:
        if not updater.isUpToDate():
            return True
    
    return False

def checkAndRun():
    if isUpdateNeeded():
        runAllUpdaters()

class Updater(object):
    def __init__(self, iniUrl = None, iniPath = None):
        '''
        Constructor
        @param iniUrl: the url to the remote ini file
        @param iniPath: the path to the local ini file
        @param progressReporter: the callback to report progress with [typicaly used to update the progress bar, it should accept 3 arguments: a count of blocks transferred so far, a block size in bytes, and the total size of the file
        @param statusReporter: the callback to report the status with, it should take a single argument: the new status
        '''
        self._iniUrl = iniUrl
        self._iniPath = iniPath
        
    def setConnection(self, conn):
        self.conn = conn
        
    def setIniUrl(self, iniUrl):
        self._iniUrl = iniUrl
        
    def setIniPath(self, iniPath):
        self._iniPath = iniPath
    
    def setStatus(self, status):
        self.conn.send(("status", status))
            
    def setProgress(self, blocks, blockSize, fileSize):
        self.conn.send(("progress", blocks * blockSize / float(fileSize)))
        
    def setSection(self, section):
        self.conn.send(("section", section))
        
    def getIniUrl(self):
        return self._iniUrl
    
    def getIniPath(self):
        return self._iniPath
    
    def isUpToDate(self, sections = "*"):
        '''
        Check if an update is needed or not
        @param sections: What sections of the ini file to check, or "*" to check them all
        '''
        if not os.path.exists(self._iniPath):
            return False

        #Open ini files
        iniNew = urllib.urlopen(self._iniUrl)
        iniOld = open(self._iniPath)
        
        #Get configparser instances for the old and new ini file
        configOld = ConfigParser.ConfigParser()
        configOld.readfp(iniOld)
        
        configNew = ConfigParser.ConfigParser()
        configNew.readfp(iniNew)
        
        upToDate = True
        for section in configNew.sections():
            if sections == "*" or section in sections:
                try:
                    if configNew.get(section, 'version') > configOld.get(section, 'version'):
                        upToDate = False
                        break
                except ConfigParser.NoSectionError:
                    upToDate = False
        
        iniOld.close()
        iniNew.close()     
                    
        return upToDate
            
    def update(self, sections = "*"):
        '''
        Update the stuff
        @param sections: The sections of the ini file to check, or "*" to check them all
        '''
        try:
            #Open ini files
            iniNew = urllib.urlopen(self._iniUrl)
            if os.path.exists(self._iniPath):
                iniOld = open(self._iniPath)
            else: 
                dirPath = os.path.dirname(self._iniPath)
                if not os.path.exists(dirPath):
                    os.makedirs(dirPath)
                
                iniOld = open(self._iniPath, "w+")
                
            #Get configparser instances for the old and new ini file
            configOld = ConfigParser.ConfigParser()
            configOld.readfp(iniOld)
                
            configNew = ConfigParser.ConfigParser()
            configNew.readfp(iniNew)
            
            for section in configNew.sections():
                if sections == "*" or section in sections:
                    self.setSection(section)
                    try:
                        if configNew.get(section, 'version') > configOld.get(section, 'version'):
                            self.updateHandler(section, configNew)
                    except ConfigParser.NoSectionError:
                        self.updateHandler(section, configNew)
    
            urllib.urlretrieve(self._iniUrl, self._iniPath)
        
            return True
        except IOError:
            return False
        finally:
            iniNew.close()
            iniOld.close()
        
        
    def updateHandler(self):
        '''
        This method should be implemented by a child class, it is called by update to do the actual handling
        '''
        raise NotImplementedError(), "This is expected to be implemented by the child class"    