#Copyright 2009 Diego Duclos
#
#This program is free software: you can redistribute it and/or modify
#it under the terms of the GNU General Public License as published by
#the Free Software Foundation, either version 3 of the License, or
#(at your option) any later version.
#
#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License
#along with this program.  If not, see <http://www.gnu.org/licenses/>.

import sql
import graphic
from model.cache import cached_property, cached_query

baseQuery= '''
SELECT
marketGroupID,
parentGroupID,
marketGroupName,
description,
graphicID,
hasTypes,
iconID,
dataID,
types
FROM invmarketgroups
'''
getMarketGroupByID = baseQuery + " WHERE marketGroupID = ?"
getMarketGroupByName = baseQuery + " WHERE marketGroupName = ?"
getMarketGroupByParent = baseQuery + " WHERE parentGroupID = ?"

class marketGroup(object):
    """
    Class used to describe market groups
    """
    @cached_property
    def parent(self): return getMarketGroup(ID=self.parentID)
    
    @cached_property
    def children(self): return getChildren(self)
    
    @cached_property
    def icon(self): return graphic.getIconName(self.iconID)
    
    def __init__(self, (ID, parentID, name, description, graphicID, hasTypes, iconID, dataID, types)):
        '''
        Constructor
        @param ID: the ID of the market group [from database]
        @param parentID: the ID of the market of this market group [from database]
        @param name: the name of this market group [from database]
        @param description: the description of this market group [from database]
        @param graphicID: the graphicID of this market group [from database]
        @param hasTypes: whether this market group has any items in it [from database]
        @param iconID: the iconID of this market group [from database]
        @param dataID: the dataID of this market group [from database]
        @param types: the list of types in this market group [from database]
        @param parent: the parent of this market group [optional]
        @param children: the children of this market group [optional]
        '''
        self.ID = ID
        self.parentID = parentID
        self.name = name
        self.description = description
        self.iconID = iconID
        self.hasTypes = hasTypes     

    def __str__(self, level = 0):
        '''
        Returns a textual representation of this market group
        @param level: the level of indentation
        '''
        formatting = ((level * 2) * " ") + "{0:15} = {1}\n"
        newline = ((level * 2) * " ")  + "\n" + ((level * 2) * " ")
        str = ((level * 2) * " ") + "[marketGroup]\n" + formatting.format("name", self.name) + formatting.format("description", self.description)+ formatting.format("iconID", self.iconID) + formatting.format("hasTypes", self.hasTypes)
        if self.parent != None:
            str = str + newline + "parent:\n" + self.parent.__str__(level + 1) + "\n\n"

        return str
    
    def __eq__(self, other): return type(other) == type(self) and self.ID == other.ID
    def __ne__(self, other): return not self.__eq__(other)

def getChildren(mGroup):
    '''
    Fetch all children of a market group from database
    @param mGroup: the marketgroup to fetch for
    '''
    c = cached_query(getMarketGroupByParent, (mGroup.ID,))
    childrenDict = {}
    for row in c:
        mGroup = marketGroup(row)
        childrenDict[mGroup.name] = mGroup
    
    return childrenDict
    
def getMarketGroup(ID = None, name = None):
    '''
    Fetch a market group from database. Either ID or name must be specified NOT BOTH
    @param ID: the ID of the market group
    @param name: the name of the market group
    '''
    if ID != None and name == None: c = cached_query(getMarketGroupByID, (ID,))
    elif name != None: c = cached_query(getMarketGroupByName, (name,))
    else: return None
    if not c: return None
    row = c[0]
    if row == None: return None
    return marketGroup(row)
    
if __name__ == "__main__":
    print("This module defines the following:\n")
    for name in vars().keys():
        print("{0:20} = {1}".format(name, vars()[name]))