#Copyright 2009 Diego Duclos
#
#This program is free software: you can redistribute it and/or modify
#it under the terms of the GNU General Public License as published by
#the Free Software Foundation, either version 3 of the License, or
#(at your option) any later version.
#
#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License
#along with this program.  If not, see <http://www.gnu.org/licenses/>.
import sql
import dataFolder
import os.path
import overrides
import traceback
from cache import cached_query

buildEffectQueryByTypeID = '''
SELECT
dte.effectID,
dte.isDefault,
de.effectName,
de.effectCategory,
de.preExpression,
de.postExpression,
de.description,
de.guid,
de.graphicID,
de.isOffensive,
de.isAssistance,
de.durationAttributeID,
de.trackingSpeedAttributeID,
de.dischargeAttributeID,
de.rangeAttributeID,
de.falloffAttributeID,
de.published,
de.displayName,
de.isWarpSafe,
de.rangeChance,
de.electronicChance,
de.propulsionChance,
de.distribution,
de.sfxName,
de.npcUsageChanceAttributeID,
de.npcActivationChanceAttributeID,
de.fittingUsageChanceAttributeID,
de.iconID,
de.dataID
FROM dgmtypeeffects dte
INNER JOIN dgmeffects de ON dte.effectID = de.effectID
WHERE dte.typeID = ?
'''

class effect(object):
    """
    This class is used to represent effects in the database
    """
    #Why do the effect database tables have so much columns ? Seriously
    def __init__(self, item, (ID, isDefault, name, category, preExpression, postExpression, description, guid, graphicID, isOffensive, isAssistance, durationAttributeID, trackingSpeedAttributeID, dischargeAttributeID, rangeAttributeID, falloffAttributeID, published, displayName, isWarpSafe, rangeChance, electronicChance, propulsionChance, distribution, sfxName, npcUsageChanceAttributeID, npcActivationChanceAttributeID, fittingUsageChanceAttributeID, iconID, dataID), ignoreOverrides = False):
        '''
        Constructor. The constructor will also try to load the function corresponding to this effect. It will fail silently if it doesn't manage
        @param item: the item this effect belongs to [from database]
        @param ID: the ID of the effect [from database]
        @param isDefault: whether this effect is the default [from database]
        @param name: the name of this effect [from database]
        @param category: the category of this effect [from database]
        @param preExpression: the preExpression of this effect [from database]
        @param postExpression: the postExpression of this effect [from database]
        @param description: the description of this effect [from database]
        @param guid: the guid of this effect [from database]
        @param graphicID: the graphicID of this effect [from database]
        @param isOffensive: whether this effect agresses the target [from database]
        @param isAssistance: whether this effect provides assistance to the target [from database]
        @param durationAttributeID: the attribute that specifies the duration of this effect [from database]
        @param trackingSpeedAttributeID: the attribute that specifies the tracking speed of this effect [from database]
        @param dischargeAttributeID: the attribute that specifies the discharge of this effect [from database]
        @param rangeAttributeID: the attribute that specifies the range of this effect [from database]
        @param falloffAttributeID: the attribute that specifies the faloff of this effect [from database]
        @param published: whether this effect has been published [from database]
        @param displayName: the display name of this effect [from database]
        @param isWarpSafe: if this effect can be used during warp [from database]
        @param rangeChance: unknown [from database]
        @param electronicChance: unknown [from database]
        @param propulsionChance: unknown [from database]
        @param distribution: unknown [from database]
        @param sfxName: unknown [from database]
        @param npcUsageChanceAttributeID: the attribute specifying the chance an NPC uses this effect [from database]
        @param npcActivationChanceAttributeID: the attribute specifying the chance an NPC activates this effect [from database]
        @param fittingUsageChanceAttributeID: unknown [from database]
        @param iconID: icon ID of the effect [from database]
        @param dataID: unknown [from database]
        '''
        self.item = item
        self.ID = ID
        self.isDefault = isDefault
        strippedName = ""
        for char in name:
            if char.isalnum(): strippedName = strippedName + char
        self.name = strippedName
        self.category = category
        self.preExpression = preExpression
        self.postExpression = postExpression
        self.description = description
        self.guid = guid
        self.iconID = iconID
        self.isOffensive = isOffensive
        self.isAssistance = isAssistance
        self.durationAttributeID = durationAttributeID
        self.trackingSpeedAttributeID = trackingSpeedAttributeID
        self.dischargeAttributeID = dischargeAttributeID
        self.rangeAttributeID = rangeAttributeID
        self.falloffAttributeID = falloffAttributeID
        self.published = published
        self.displayName = displayName
        self.isWarpSafe = isWarpSafe
        self.rangeChance = rangeChance
        self.electronicChance = electronicChance
        self.propulsionChance = propulsionChance
        self.distribution = distribution
        self.sfxName = sfxName
        self.npcUsageChanceAttributeID = npcUsageChanceAttributeID
        self.npcActivationChanceAttributeID = npcActivationChanceAttributeID
        self.fittingUsageChanceAttributeID = fittingUsageChanceAttributeID
        self.ignoreOverrides = ignoreOverrides
        if not ignoreOverrides:
            overrideMod = overrides.getOverrideModule(item.name, ("effects", name, "basevalues"))
            if overrideMod:
                for key in dir(overrideMod): setattr(self, key, getattr(overrideMod, key))

    def __str__(self, level = 0): #We're not going to print everything here, that'd be quite useless really, and way too much typing work too (mostly the former)
        '''
        Returns a textual representation of the effect
        @param level: level of indentation to use
        '''
        formatting = ((level * 2) * " ") + "{0:15} = {1}\n"
        newline = ((level * 2) * " ") + "\n"
        try: func = (bool(self.func) and "\033[0;32mImplemented" or "\033[0;33mMissing")
        except: func = "\033[1;31mErroneous"
        str = ((level * 2) * " ") + "[effect]\n" + \
                formatting.format("name", self.name) + \
                formatting.format("ID", self.ID) +  \
                formatting.format("description", self.description) + \
                formatting.format("status", func + "\033[0m")

        return str

    def load(self):
        '''
        Try to load the function that describes this effect and put it in self.func
        '''
        try:
            #See if we got an override first, if so, totaly ignore the builtin
            if not self.ignoreOverrides:
                overrideMod = overrides.getOverrideModule(self.item.name, ("effects", self.name, self.name))
                if overrideMod: effectMod = overrideMod
                else: effectMod = None
            if not effectMod:
                effectMod = __import__('model.effects.' + self.name, fromlist=True)

            self.func = getattr(effectMod, self.name)
            try: self.runTime = getattr(effectMod, "runTime")
            except: self.runTime = None
            try:
                type = getattr(effectMod, "type")
                if type == None: type = "normal"
                if not isinstance(type, tuple): type = (type,)
                self.type = type
            except: self.type = ("normal",)
            try: self.displayName = getattr(effectMod, "displayName")
            except: pass
            try: self.extraData = getattr(effectMod, "extraData")
            except: self.extraData = None
        except ImportError:
            self.func = None
            self.runTime = None
            self.type = ("normal",)
            self.extraData = None
        except Exception, e:
            self.func = 0
            self.runTime = None
            self.type = ("normal",)
            self.extraData = None
            print "Failed to load effect", self.name, "\n"
            traceback.print_exc()

    def __getattr__(self, attr):
        if attr == "func" or attr == "type" or attr == "runTime" or attr == "extraData":
            self.load()
            return self.__dict__[attr]
        else: raise AttributeError(attr)

def buildEffects(item, effectList = None):
    '''
    Build the effects belonging to the passed item
    @param item: the item to build effects for
    @param effectList: the list to put the effects in, a new list is created if this isn't specified
    '''
    if effectList == None: effectList = []
    addedNames = set()
    c = cached_query(buildEffectQueryByTypeID, (item.ID,))
    for row in c:
        f = effect(item, row)
        effectList.append(f)
        addedNames.add(f.name)

    for effectName in overrides.getDirList(item.name, ("effects",)):
        if not effectName in addedNames:
            #ARMY OF NONES STRAIGHT AHEAD, CAPTAIN
            f = effect(item, (None, None, effectName, None, None, None, None, None, None, None,
                              None, None, None, None, None, None, None, None, None, None, None,
                              None, None, None, None, None, None, None, None))
            effectList.append(f)
            addedNames.add(effectName)
    return effectList

def findItemsWithEffect(effect_name):
    query = """
    SELECT it.typeName
    FROM invtypes it
        INNER JOIN dgmtypeeffects dte
            ON dte.typeID = it.typeID
        INNER JOIN dgmeffects de
            ON de.effectID = dte.effectID
    WHERE it.published = 1
        AND de.effectName = ?
    """

    return cached_query(query, (effect_name,))

if __name__ == "__main__":
    print("This module defines the following:\n")
    for name in vars().keys():
        print("{0:20} = {1}".format(name, vars()[name]))
