#Copyright 2009 Diego Duclos
#
#This program is free software: you can redistribute it and/or modify
#it under the terms of the GNU General Public License as published by
#the Free Software Foundation, either version 3 of the License, or
#(at your option) any later version.
#
#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License
#along with this program.  If not, see <http://www.gnu.org/licenses/>.
import gtk, gobject
from gui import pixbufLoader
from gui.itemStatsWindow import itemStatsWindow
from model import fitting, item, group
from model.fitting import module
import copy

def init(self):
    self.selectionKeyPressed = False
    
    #Register our custom events and their handlers
    gobject.signal_new("fitting-changed", self, gobject.SIGNAL_RUN_FIRST,
                       gobject.TYPE_NONE, ())
    self.connect("fitting-changed", self.fittingChanged)
    self.tvwModules.connect("button-press-event", self.tvwModulesClicked)
    self.tvwModules.connect("key-press-event", self.tvwModulesKeyPress)
    self.tvwModules.connect("key-release-event", self.keyReleased)
    #itemIcon, Name, description, PG, cpu, cap, range, ammoIcon, ammo, item
    #Get needed images
    pgImage = gtk.image_new_from_pixbuf(pixbufLoader.getPixbuf("pgSmall"))
    cpuImage = gtk.image_new_from_pixbuf(pixbufLoader.getPixbuf("cpuSmall"))
    capImage = gtk.image_new_from_pixbuf(pixbufLoader.getPixbuf("capacitorSmall"))
    rangeImage = gtk.image_new_from_pixbuf(pixbufLoader.getPixbuf("range"))

    #Call show on all the images, they're very shy
    pgImage.show()
    cpuImage.show()
    capImage.show()
    rangeImage.show()

    #Make and set store
    #To indicate which column is what, here is the line that fills a row, including the column number
    store = gtk.ListStore(  gtk.gdk.Pixbuf, #0 pixbufItem
                            str,            #1 item.name
                            str,            #2 item.description.replace("\\\"", "\"")
                            str,            #3 power
                            str,            #4 cpu
                            str,            #5 cap
                            str,            #6 rangeStr
                            gtk.gdk.Pixbuf, #7 pixbufAmmo
                            str,            #8 nameAmmo
                            object,         #9 mod
                            str,            #10 mod.getSlot()
                            gtk.gdk.Pixbuf, #11 pixbufState
                            int,            #12 i (module number)
                            str,            #13 None
                            bool)           #14 None
    self.tvwModules.set_model(store)

    #Setup sorting
    store.set_sort_func(11, self.sortBySlot)
    store.set_sort_column_id(11, gtk.SORT_ASCENDING)

    #Inserts empty lines between racks and replaces them with separators
    self.tvwModules.set_row_separator_func(insSep, self)

    #Cell renderers are our good old friends LETS MAKE LOTS OF THEM
    itemStateRenderer = gtk.CellRendererPixbuf()
    itemIconRenderer = gtk.CellRendererPixbuf()
    nameRenderer = gtk.CellRendererText()
    descriptionRenderer = gtk.CellRendererText()
    pgRenderer = gtk.CellRendererText()
    cpuRenderer = gtk.CellRendererText()
    capRenderer = gtk.CellRendererText()
    rangeRenderer = gtk.CellRendererText()
    ammoIconRenderer = gtk.CellRendererPixbuf()
    ammoRenderer = gtk.CellRendererText()

    #tree view columns are also our friends LETS INVITE A FEW OVER
    self.itemStateCol = gtk.TreeViewColumn()
    nameCol = gtk.TreeViewColumn("Name")
    pgCol = gtk.TreeViewColumn()
    pgCol.set_widget(pgImage)
    cpuCol = gtk.TreeViewColumn()
    cpuCol.set_widget(cpuImage)
    capCol = gtk.TreeViewColumn()
    capCol.set_widget(capImage)
    rangeCol = gtk.TreeViewColumn()
    rangeCol.set_widget(rangeImage)
    ammoCol = gtk.TreeViewColumn("Ammo")

    #Cell renderers, meet columns. Columns, meet cell renderers
    self.itemStateCol.pack_start(itemStateRenderer, False)
    nameCol.pack_start(itemIconRenderer, False)
    nameCol.pack_start(nameRenderer, True)
    pgCol.pack_start(pgRenderer, False)
    cpuCol.pack_start(cpuRenderer, False)
    capCol.pack_start(capRenderer, False)
    rangeCol.pack_start(rangeRenderer, False)
    ammoCol.pack_start(ammoIconRenderer, False)
    ammoCol.pack_start(ammoRenderer, False)

    #Need to tell those columns they need to be resizable
    #can't do anything themselves these days
    nameCol.set_resizable(True)
    pgCol.set_resizable(True)
    cpuCol.set_resizable(True)
    capCol.set_resizable(True)
    rangeCol.set_resizable(True)
    ammoCol.set_resizable(True)
    
    #Ok, here's your info, NOW LEAVE ME ALONE
    self.itemStateCol.add_attribute(itemStateRenderer, 'pixbuf', 11)
    nameCol.add_attribute(itemIconRenderer, 'pixbuf', 0)
    nameCol.add_attribute(nameRenderer, 'text', 1)
    nameCol.add_attribute(nameRenderer, "foreground-set", 14)
    nameRenderer.set_property("foreground", gtk.gdk.color_parse('red'))
    pgCol.add_attribute(pgRenderer, 'text', 3)
    cpuCol.add_attribute(cpuRenderer, 'text', 4)
    capCol.add_attribute(capRenderer, 'text', 5)
    rangeCol.add_attribute(rangeRenderer, 'text', 6)
    ammoCol.add_attribute(ammoIconRenderer, 'pixbuf', 7)
    ammoCol.add_attribute(ammoRenderer, 'text', 8)

    #Columns, meet treeview. Treeview, meet columns
    tv = self.tvwModules
    tv.append_column(self.itemStateCol)
    tv.append_column(nameCol)
    tv.append_column(pgCol)
    tv.append_column(cpuCol)
    tv.append_column(capCol)
    tv.append_column(rangeCol)
    tv.append_column(ammoCol)

    #Set tooltip with description
    tv.set_tooltip_column(2)

    #Tooltip stuff for lock times
    w = gtk.Window(gtk.WINDOW_POPUP)
    f = gtk.Frame()
    f.set_border_width(1)
    w.set_decorated(False)
    self.lockTimesTable = gtk.Table(1, 2)
    lbl = gtk.Label()
    lbl.set_markup("<b>Lock Times</b>")
    self.lockTimesTable.attach(lbl, 0, 3, 0, 1)
    self.lockTimeLabels = []
    f.add(self.lockTimesTable)
    w.add(f)
    w.set_name('gtk-tooltips')
    f.show_all()

    for widget in ("lblScanResHeader", "boxScanRes"):
        widget = getattr(self, widget)
        widget.set_property("has-tooltip", True)
        widget.set_tooltip_window(w)
        widget.connect("query-tooltip", lambda one, two, three, four, five: self.getActiveFit() != None)

    #Generate the menus and menuitems we need
    self.moduleStatsLabel = gtk.Label("Module Stats")
    self.moduleStatsEntry = self.generateMenuItem(self.moduleStatsLabel, self.fittedModuleStats)
    self.moduleGroupLabel = gtk.Label("Module Market Group")
    self.moduleGroupEntry = self.generateMenuItem(self.moduleGroupLabel, self.fittedModuleGroup)
    self.moduleFavoriteLabel = gtk.Label("Add Module to Favorites")
    self.moduleFavoriteEntry = self.generateMenuItem(self.moduleFavoriteLabel, self.fittedModuleFavorite)
    self.ammoChangeLabel = gtk.Label("Charge")
    self.ammoChangeEntry = self.generateMenuItem(self.ammoChangeLabel, None)
    self.ammoStatsLabel = gtk.Label("Charge Stats")
    self.ammoStatsEntry = self.generateMenuItem(self.ammoStatsLabel, self.fittedAmmoStats)
    self.ammoGroupLabel = gtk.Label("Charge Market Group")
    self.ammoGroupEntry = self.generateMenuItem(self.ammoGroupLabel, self.fittedAmmoGroup)
    self.shipStatsLabel = gtk.Label("Ship Stats")
    self.shipStatsEntry = self.generateMenuItem(self.shipStatsLabel, self.fittedShipStats)
    self.fittedModuleMenu = self.generateMenu({})

    #Set tooltips for other stuff
    self.setTooltips(**{"boxTurretSlots" : "Turret hardpoints",
                        "boxLauncherSlots" : "Launcher hardpoints",
                        "boxCalibration" : "Calibration",
                        "boxCpu" :  "CPU",
                        "boxPower" : "Powergrid",
                        "boxDroneBay" : "Drone bay volume",
                        "boxDroneBandwidth" : "Drone control bandwidth",
                        "imgShield" : "Shield",
                        "imgArmor" : "Armor",
                        "imgHull" : "Hull",
                        "imgEm" : "EM resistance",
                        "imgThermal" : "Thermal resistance",
                        "imgKinetic" : "Kinetic resistance",
                        "imgExplosive" : "Explosive resistance",
                        "lblEhpHeader" : "Effective HP",
                        "imgPassiveShield" : "Shield regeneration peak",
                        "imgActiveShield" : "Active shield tank",
                        "imgActiveArmor" : "Armor tank",
                        "imgActiveHull" : "Hull tank",
                        "imgReinforced" : "Reinforced tank",
                        "imgSustained" : "Sustained tank",
                        "boxShipDps" : "Ship weapon DPS",
                        "boxDroneDps" : "Drone DPS",
                        "lblVolleyHeader" : "Ship weapon volley damage",
                        "lblVolley" : "Ship weapon volley damage"})

    self.tvwModules.get_selection().set_mode(gtk.SELECTION_MULTIPLE)
    self.tvwModules.set_rubber_banding(True)
    self.tvwModules.set_reorderable(True)

    self.tvwModules.enable_model_drag_dest(
        [('text/plain', 0, 0)],
        gtk.gdk.ACTION_DEFAULT | gtk.gdk.ACTION_MOVE
    )
    self.tvwModules.enable_model_drag_source(
        gtk.gdk.BUTTON1_MASK, 
        [('text/plain', 0, 0)],
        gtk.gdk.ACTION_DEFAULT | gtk.gdk.ACTION_MOVE
    )
    self.tvwModules.connect("drag-data-received", self.dragDataRecv)

def insSep(store, iter, self):
    if iter != None:
        slot = store.get_value(iter, 10)
        itern = store.iter_next(iter)
        if itern != None:
            slot_next = store.get_value(itern, 10)
            if slot != slot_next and slot != 'sepSlot' and slot_next != 'sepSlot' and slot != None and slot_next != None:
                iterNew = store.insert_after(iter)
                store.set_value(iterNew, 10, 'sepSlot')
            if slot == None or slot == 'sepSlot': return True
        elif self.fittingChanged:
            self.fittingChanged = False
            self.tvwModules.get_selection().unselect_all()
            for path in self.selected_rows:
                self.tvwModules.get_selection().select_path(path)
        return None

def dragDataRecv(self, widget, drag_context, x, y, selection_data, info, timestamp):
    #Figure where to move to
    drop_info = widget.get_dest_row_at_pos(x, y)
    if drop_info == None or drop_info[0] == None: return
    #Change dest depending on where exactly we need to move
    dest, info = drop_info
    #Figure where the items to move are at
    paths = self.tvwModules.get_selection().get_selected_rows()[1]
    if not paths or dest in paths: return
    if info == gtk.TREE_VIEW_DROP_AFTER: dest = (dest[0] + 1,)
    
    store = self.tvwModules.get_model()
    destIter = store.get_iter(dest)
    #Figure what's at the destination
    destModule, destSlot, destIndex = store.get(destIter, 9, 10, 12)
    activeFit = self.getActiveFit()
    if info == gtk.TREE_VIEW_DROP_INTO_OR_AFTER or info == gtk.TREE_VIEW_DROP_INTO_OR_BEFORE \
     and len(paths) == 1 and \
     store.get_value(store.get_iter(paths[0]), 9).getSlot() == destModule.getSlot():
        path = paths[0]
        iter = store.get_iter(path)
        srcIndex = store.get_value(iter, 12)
        activeFit.modules[srcIndex], activeFit.modules[destIndex] = activeFit.modules[destIndex], activeFit.modules[srcIndex]
    else:
        #Put indice in a list
        sourceIndice = {}
        for i in range(len(activeFit.modules)): sourceIndice[i] = i
        for path in paths:
            #Figure where we need to drop exactly
            iter = store.get_iter(path)
            srcIndex = store.get_value(iter, 12)
            srcMod = activeFit.modules[srcIndex]
            srcSlot = srcMod.getSlot()
            if srcSlot != destSlot:
                refactor = True
                #Slots aren't of the same types
                #figure if a slot type with higher or lower prio was selected
                if srcSlot == 'sepSlot':
                    before = fitting.slotTypes.index(destSlot) + 1
                elif destSlot == 'sepSlot':
                    before = fitting.slotTypes.index(srcSlot) + 1
                else:
                    before = fitting.slotTypes.index(srcSlot) <= fitting.slotTypes.index(destSlot)
                #Figure the valid index of the same slot to add the item at
                i = before and len(activeFit.modules) - 1 or 0
                while before and i > 0 or i < 0:
                    i = before and i - 1 or i + 1
                    if activeFit.modules[i].getSlot() == srcSlot:
                        destIndex = i
                        break
            else: refactor = False
            #Figure where sourceItem moved to (if it moved)
            srcIndex = sourceIndice[srcIndex]
            del activeFit.modules[srcIndex]
            for orig, current in sourceIndice.iteritems():
                if (info == gtk.TREE_VIEW_DROP_INTO_OR_BEFORE or\
                    info == gtk.TREE_VIEW_DROP_INTO_OR_AFTER or \
                    refactor == True) and \
                current == destIndex: continue
                if current > srcIndex: sourceIndice[orig] = current - 1
            
                
            #Figure where the destIndex moved to
            destIndex = sourceIndice[destIndex]
            #Insert it at the correct location now
            activeFit.modules.insert(destIndex, srcMod)
            #Change the indice dict accordingly again
            for orig, current in sourceIndice.iteritems():
                if current >= destIndex: sourceIndice[orig] = current + 1
                
    self.emit("fitting-changed")

def getActiveFit(self):
    store = self.cmbSetup.get_model()
    if store == None: return
    iter = self.cmbSetup.get_active_iter()
    if iter == None: return
    return store.get(iter, 2)[0]

def generateFitMenu(self, callback, parent = None):
    store = self.cmbSetup.get_model()
    iter = store.iter_children(parent)
    if iter == None: return
    menu = gtk.Menu()
    while iter:
        menuItem = gtk.MenuItem()
        sub = self.generateFitMenu(callback, iter)
        if sub:
            pixbuf, name = store.get(iter, 0, 1)
            menuItem.set_submenu(sub)
        else:
            pixbuf = None
            name, fit = store.get(iter, 1, 2)
            menuItem.connect("activate", callback, fit)
            
        box = gtk.HBox()
        box.pack_start(gtk.image_new_from_pixbuf(pixbuf), False)
        lbl = gtk.Label(name)
        lbl.set_alignment(0.0, 0.5)
        box.pack_start(lbl, True)
        menuItem.add(box)
        menuItem.show_all()
        menu.append(menuItem)
        iter = store.iter_next(iter)

    return menu

def getSelectedModule(self):
    store = self.tvwModules.get_model()
    select = self.tvwModules.get_cursor()[0]
    if select == None or select == -1:
        select = self.tvwModules.get_selection().get_selected_rows()[1][0]
    if select == None or select == -1: return
    iter = store.get_iter(select)
    return store.get_value(iter, 9)

def fittedShipStats(self, *stuff):
    fit = self.getActiveFit()
    itemStatsWindow(fit.ship, fit, pixbufLoader.getPixbuf(fit.ship.race))

def fittingChanged(self, *stuff):
    #startLoc = self.tvwModules.get_cursor()[0]
    moduleStore, self.selected_rows = self.tvwModules.get_selection().get_selected_rows()
        
    if moduleStore == None: return True
    moduleStore.clear()

    #Figure the character
    char = self.getActiveCharacter()
    #Save the character
    if char != None:
        self.guiSettings["activeCharacter"] = char.name

    fit = self.getActiveFit()
    if fit == None:
        labels = ["lblTurretSlotsUsed", "lblTurretSlotsMax", "lblLauncherSlotsUsed",
                  "lblLauncherSlotsMax", "lblCalibrationUsed", "lblCalibrationMax",
                  "lblCpuUsed", "lblCpuMax", "lblPowerUsed", "lblPowerMax",
                  "lblDroneBayUsed", "lblDroneBayMax", "lblDroneBandwidthUsed",
                  "lblDroneBandwidthMax", "lblPassiveShieldTank", "lblShieldTank",
                  "lblArmorTank", "lblHullTank", "lblCapCapacity", "lblCapState",
                  "lblSustainedArmorTank", "lblSustainedHullTank", "lblEhp",
                  "lblSustainedShieldTank", "lblCapRecharge", "lblCapDischarge",
                  "lblShipDps", "lblDroneDps", "lblVolley", "lblTargets", "lblRange",
                  "lblScanRes", "lblScanStr", "lblSpeed", "lblAlignTime", "lblCargo",
                  "lblSigRadius", "lblDronesUsed", "lblDronesTotal", "lblTotalDps",]
        for lbl in labels:
            getattr(self, lbl).set_text("0")
        for type in ("Shield", "Armor", "Hull"):
            getattr(self, "lbl" + type + "Ehp").set_text("0")
            for dmg in ("Em", "Thermal", "Kinetic", "Explosive"):
                getattr(self, "lbl" + type + dmg + "Resistance").set_text("0.0")
        for pbar in ("Cpu", "Power", "DroneBay", "DroneBandwidth"):
            bar = getattr(self, "pbar" + pbar)
            bar.set_text("")
            bar.set_fraction(0.0)

        return True

    fit.character = char
    #Start of missing skills stuff

    fit.calculateModifiedAttributes()
    #Check if any mods need disabling
    if(fit.disableImpossibleMods()):
        self.emit("fitting-changed")
        return True
    
    #Remove superfluous dummies/Add missing ones        
    for slotType in fitting.slotTypes:
        left = fit.getSlotsLeft(slotType)
        numDummies = 0
        i = 0
        for mod in fit.modules:
            if mod.getItem().ID == 0 and mod.getSlot() == slotType: numDummies += 1
            
        left = left - numDummies
        if left > 0:
            for i in range(int(left)):
                fit.modules.append(fitting.generateDummy(slotType))
        elif left < 0:
            for i in range(int(abs(left))):
                for y in range(len(fit.modules)):
                    module = fit.modules[y]
                    if module.getItem().ID == 0 and module.getSlot() == slotType:
                        del fit.modules[y]
                        break

    fit.calculateCapUsage()
    #Add modules to the view
    for i in range(len(fit.modules)):
        mod = fit.modules[i]
        item, state, ammo = mod.getItem(), mod.getState(), mod.getAmmo()
        #This item is a dummy
        if item.ID == 0: pixbufItem = pixbufLoader.getPixbuf(mod.getSlot())
        else: pixbufItem = pixbufLoader.getPixbuf(item.icon, True, "16x16")
        if state is not None:
            pixbufState = pixbufLoader.getPixbuf(fitting.modStates[state])
        else:
            pixbufState = None
        if ammo != None:
            pixbufAmmo = pixbufLoader.getPixbuf(ammo.icon, True, "16x16")
            #python has rounding problems with floats (eg 2.3/0.1 is 22.999999999999996 which will be rounded down to 22)
            #so we use multiplied integers here (5 signs is enough for smallest projectile ammo) to show proper value
            maxAmmoSuffix = " (" + str(int(item.getModifiedAttribute("capacity")*10000)/int(ammo.getModifiedAttribute("volume")*10000)) + ")"
            nameAmmo = ammo.name + maxAmmoSuffix
        else:
            pixbufAmmo = None
            nameAmmo = None

        power = item.getModifiedAttribute("power") or 0
        cpu = item.getModifiedAttribute("cpu") or 0
        if int(power) == power:
            power = "%d" % power
        else:
            power = '%.1f' % power
        if int(cpu) == cpu:
            cpu = "%d" % cpu
        else:
            cpu = '%.1f' % cpu

        capUsage = item.getModifiedAttribute("_capUsage")
        capBoost = item.getModifiedAttribute("_capBoost")
        if capUsage:
            cap = "%.1f" % capUsage
        elif capBoost:
            cap = "+%.1f" % capBoost
        else:
            cap = ""

        maxRange = item.getModifiedAttribute("maxRange") or item.getModifiedAttribute("_maxRange")
        falloff = item.getModifiedAttribute("falloff")
        maxMissileRange = item.getModifiedAttribute("_maxMissileRange")
        if maxRange != None:
            maxRange = maxRange / 1000.0
            if falloff != None and falloff != 0:
                falloff = falloff / 1000.0
                rangeStr = '%.1f + %.1fkm' % (maxRange, falloff)
            else: rangeStr = '%.1fkm' % (maxRange)
        elif maxMissileRange != None:
            maxMissileRange = maxMissileRange / 1000.0
            rangeStr = '%s%.1fkm' % ("<", maxMissileRange)
        else:
            rangeStr = None
        moduleStore.append((pixbufItem, item.name, item.description.replace("\\\"", "\""), power, cpu, cap, rangeStr, pixbufAmmo, nameAmmo, mod, mod.getSlot(), pixbufState, i, None, None))

    #Right pane
    #Turret and launcher slots
    for slotType in ("turret", "launcher"):
        used = fit.getNumModulesAttributeValue("_hardpoint", slotType)
        try: total = fit.ship.getModifiedAttribute(slotType + "SlotsLeft") + used
        except Exception: total = 0
        lblUsed = getattr(self, "lbl" + slotType.capitalize() + "SlotsUsed")
        if used > total:
            lblUsed.modify_fg(gtk.STATE_NORMAL, gtk.gdk.color_parse('red'))
        else:
            lblUsed.modify_fg(gtk.STATE_NORMAL, None)

        lblUsed.set_text(str(used))
        getattr(self, "lbl" + slotType.capitalize() + "SlotsMax").set_text('%d' % total)

    #Calibration
    used = fit.getModuleAttributeSum("upgradeCost")
    total = fit.ship.attributes["upgradeCapacity"].getModifiedValue()
    if used > total:
        self.lblCalibrationUsed.modify_fg(gtk.STATE_NORMAL, gtk.gdk.color_parse('red'))
    else:
        self.lblCalibrationUsed.modify_fg(gtk.STATE_NORMAL, None)

    self.lblCalibrationUsed.set_text(str(int(used)))
    self.lblCalibrationMax.set_text(str(int(total)))

    #cpu and power
    for type in ("cpu", "power"):
        used = fit.getModuleAttributeSum(type, fitting.STATE_INACTIVE)
        total = fit.ship.attributes[type + "Output"].getModifiedValue()
        type = type.capitalize()
        lblUsed = getattr(self, "lbl" + type + "Used")
        lblUsed.set_text('%.1f' % used)
        getattr(self, "lbl" + type + "Max").set_text('%.1f' % total)
        if total == 0:
            if used > 0:
                fraction = float("inf")
            else:
                fraction = 0.0
        else:
            fraction = used * 1.0 / total
        pbar = getattr(self, "pbar" + type)
        if fraction > 1.0:
            if fraction == float("inf"):
                self.pbarDroneBay.set_text(u"+\u221E")
            else:
                excess = fraction - 1
                pbar.set_text('+%.2f%%' % (excess * 100))
            #pbar.modify_fg(gtk.STATE_PRELIGHT, gtk.gdk.color_parse('red'))
            lblUsed.modify_fg(gtk.STATE_NORMAL, gtk.gdk.color_parse('red'))
        else:

            pbar.set_text('%.2f%%' % (fraction * 100))
            #pbar.modify_fg(gtk.STATE_PRELIGHT, None)
            lblUsed.modify_fg(gtk.STATE_NORMAL, None)

        pbar.set_fraction(min(fraction, 1.0))

    #Drone bay
    used = fit.getDronesAttributeSum("volume", False)
    total = fit.ship.attributes["droneCapacity"].getModifiedValue()
    self.lblDroneBayUsed.set_text('%.0f' % used)
    self.lblDroneBayMax.set_text('%.0f' % total)
    if total != 0:
        fraction = used * 1.0 / total
    else:
        if used == 0:
            fraction = 0
        else:
            fraction = float("inf")

    if fraction > 1.0:
        if fraction == float("inf"):
            self.pbarDroneBay.set_text(u"+\u221E")
        else:
            excess = fraction - 1
            self.pbarDroneBay.set_text('+%.2f%%' % (excess * 100))

        self.lblDroneBayUsed.modify_fg(gtk.STATE_NORMAL, gtk.gdk.color_parse('red'))
        #self.pbarDroneBay.modify_fg(gtk.STATE_PRELIGHT, gtk.gdk.color_parse('red'))
    else:
        self.pbarDroneBay.set_text('%.2f%%' % (fraction * 100))
        self.lblDroneBayUsed.modify_fg(gtk.STATE_NORMAL, None)
        #self.pbarDroneBay.modify_fg(gtk.STATE_NORMAL, None)

    self.pbarDroneBay.set_fraction(min(fraction, 1.0))

    #Drone bandwidth
    used = fit.getDronesAttributeSum("droneBandwidthUsed", True)
    total = fit.ship.attributes["droneBandwidth"].getModifiedValue()
    self.lblDroneBandwidthUsed.set_text('%.0f' % used)
    self.lblDroneBandwidthMax.set_text('%.0f' % total)
    if total != 0:
        fraction = used * 1.0 / total
    else:
        if used == 0:
            fraction = 0
        else:
            fraction = float("inf")

    if fraction > 1.0:
        if fraction == float("inf"):
            self.pbarDroneBandwidth.set_text(u"+\u221E")
        else:
            excess = fraction - 1
            self.pbarDroneBandwidth.set_text('+%.2f%%' % (excess * 100))

        self.lblDroneBandwidthUsed.modify_fg(gtk.STATE_NORMAL, gtk.gdk.color_parse('red'))
        #self.pbarDroneBandwidth.modify_fg(gtk.STATE_PRELIGHT, gtk.gdk.color_parse('red'))
    else:
        self.pbarDroneBandwidth.set_text('%.2f%%' % (fraction * 100))
        self.lblDroneBandwidthUsed.modify_fg(gtk.STATE_NORMAL, None)
        #self.pbarDroneBandwidth.modify_fg(gtk.STATE_PRELIGHT, None)
    self.pbarDroneBandwidth.set_fraction(min(fraction, 1.0))

    #capacitor
    drain = fit.ship.getModifiedAttribute("_drain") or 0
    capacity = fit.ship.getModifiedAttribute("capacitorCapacity")
    recharge = fit.ship.getModifiedAttribute("_peakCapRecharge")
    discharge = fit.getModuleAttributeSum("_capUsage") + drain
    stable = fit.ship.getModifiedAttribute("_capStable")
    if stable == True:
        capStatePrefix = "Stable at "
        capState = '%.1f' % fit.ship.getModifiedAttribute("_capState")
        capStateSuffix = " %"
    else:
        capStatePrefix = "Lasts "
        capState = fit.ship.getModifiedAttribute("_capState")
        if capState >= 100:
            capState = '%.0fm%.0f' % ((capState / 60) , (capState % 60))
        else:
            capState = '%.0f' % capState

        capStateSuffix = "s"

    self.lblCapCapacity.set_text('%.1f' % capacity)
    self.lblCapRecharge.set_text('%.1f' % recharge)
    self.lblCapDischarge.set_text('%.1f' % -discharge)
    self.lblCapStatePrefix.set_text(capStatePrefix)
    self.lblCapState.set_text(capState)
    self.lblCapStateSuffix.set_text(capStateSuffix)

    #Ship DPS and volley
    fit.calculateVolley()
    fit.calculatePerSecond("_volley", "_dps")
    fit.calculatePerSecond("miningAmount", "_miningYield", True)
    miningYield = fit.getModuleAttributeSum("_miningYield") + fit.getDronesAttributeSum("_miningYield", True)
    shipDps = fit.getModuleAttributeSum("_dps")
    if shipDps or not miningYield:
        volley = fit.getModuleAttributeSum("_volley")
        droneDps = fit.getDronesAttributeSum("_dps", True)
        self.lblFirePowerHeader.set_markup("<b>Firepower</b>")
        self.imgShipDps.set_from_pixbuf(pixbufLoader.getPixbuf("turret"))
        self.lblVolleyHeader.set_text("Ship volley: ")
        for lbl in ("lblShipDpsUnit", "lblDroneDpsUnit"):
            getattr(self, lbl).set_text(" DPS")
        self.lblTotalDpsHeader.set_text("Total DPS: ")
    else: #Switch over to mining yield
        shipDps = fit.getModuleAttributeSum("_miningYield")
        volley = fit.getModuleAttributeSum("miningAmount")
        droneDps = fit.getDronesAttributeSum("_miningYield", True)
        self.lblFirePowerHeader.set_markup("<b>Mining Yield</b>")
        self.imgShipDps.set_from_pixbuf(pixbufLoader.getPixbuf("mining"))
        self.lblVolleyHeader.set_text("Yield: ")
        for lbl in ("lblShipDpsUnit", "lblDroneDpsUnit"):
            getattr(self, lbl).set_text(" MPS")
        self.lblTotalDpsHeader.set_text("Total MPS: ")

    self.lblShipDps.set_text('%.1f' % shipDps)
    self.lblVolley.set_text('%.1f' % volley)

    #Drone DPS
    self.lblDroneDps.set_text('%.1f' % droneDps)

    #total dps
    self.lblTotalDps.set_text('%.1f' % (droneDps + shipDps))

    #Effective hp and resistance
    if self.activePattern == "raw": damagePattern = "raw"
    else: damagePattern = self.damagePatterns[self.activePattern]
    fit.calculateTankEhp(damagePattern)
    for (layer, attrPrefix) in (("Shield", "shield"), ("Armor", "armor"), ("Hull", "")):
        for damageType in ("Kinetic", "Thermal", "Explosive", "Em"):
            lbl = getattr(self, "lbl" + layer + damageType + "Resistance")
            attrName = attrPrefix + damageType + "DamageResonance"
            attrName = attrName[0].lower() + attrName[1:]
            resonance = fit.ship.getModifiedAttribute(attrName)
            resistance = (1 - (resonance or 1)) * 100
            lbl.set_text('%.2f' % resistance)

        ehp = fit.ship.getModifiedAttribute("_" + attrPrefix.capitalize() + "Ehp")
        getattr(self, "lbl" + layer + "Ehp").set_text('%.0f' % ehp)

    self.lblEhp.set_text('%d' % fit.ship.getModifiedAttribute("_totalEhp"))

    #Tank
    fit.calculateTankRegen(damagePattern)
    fit.calculateSustainedTankRegen()
    for tankType in ("passiveShield", "shield", "armor", "hull"):
        for subType in ("", "Sustained"):
            if subType == "Sustained" and tankType == "passiveShield": continue
            lblName = "lbl" + subType + tankType[0].upper() + tankType[1:] + "Tank"
            lbl = getattr(self, lblName)
            attrName = "_" + tankType + subType + "Regen"
            tank = fit.ship.getModifiedAttribute(attrName) or 0
            lbl.set_text('%.1f' % tank)

    #Targetting related info
    #Targetting range
    targetRange = fit.ship.getModifiedAttribute("maxTargetRange") / 1000
    self.lblRange.set_text("%.1f" % targetRange)

    #Max locked targets
    maxTargets = min(fit.ship.getModifiedAttribute("_maxLockedTargetsSkill"),
                     fit.ship.getModifiedAttribute("maxLockedTargets"))
    self.lblTargets.set_text('%.0f' % maxTargets)

    #scan Resolution
    scanResolution = fit.ship.getModifiedAttribute("scanResolution")
    self.lblScanRes.set_text('%.1f' % (scanResolution or 0))

    #Lock times tooltip
    signatures = [("Interceptor", 32),
                  ("Frigate", 46),
                  ("Cruiser", 110),
                  ("Battlecruiser", 272),
                  ("Battleship", 470)]

    self.lockTimesTable.set_property("n-rows", len(signatures) + 1)
    for lbl in self.lockTimeLabels: self.lockTimesTable.remove(lbl)
    del self.lockTimeLabels[:]
    for i in range(len(signatures)):
        shipType, shipSig = signatures[i]
        lockTime = fit.calculateLockTime(shipSig)
        lblShip = gtk.Label()
        self.lockTimeLabels.append(lblShip)
        lblShip.set_markup("%s [%d]: " % (shipType, shipSig))
        lblShip.set_alignment(1.0, 0.5)
        self.lockTimesTable.attach(lblShip, 0, 1, i + 1, i + 2)
        lblTime = gtk.Label("%.1fs" % lockTime)
        self.lockTimeLabels.append(lblTime)
        self.lockTimesTable.attach(lblTime, 2, 3, i + 1, i + 2)

    self.lockTimesTable.show_all()


    #Scan strength, Pick the highest one to display
    scanStr = 0
    t = None
    for scanType in ("Magnetometric", "Ladar", "Radar", "Gravimetric"):
        currScanStr = fit.ship.getModifiedAttribute("scan" + scanType + "Strength")
        if currScanStr > scanStr:
            scanStr = currScanStr
            t = scanType

    self.lblScanStr.set_text('%.0f' % scanStr)
    tt = "Type: %s" % t
    self.lblScanStr.set_tooltip_text(tt)
    self.lblSensorStrHeader.set_tooltip_text(tt)
    
    #Speed
    speed = fit.ship.getModifiedAttribute("maxVelocity")
    if speed > 1000:
        speed = speed / 1000.0
        self.lblSpeedUnit.set_text(" km/s")
        self.lblSpeed.set_text('%.3f' % speed)
    else:
        self.lblSpeedUnit.set_text(" m/s")
        self.lblSpeed.set_text('%.1f' % speed)

    fit.calculateAlignTime()
    #Align time
    alignTime = fit.ship.getModifiedAttribute("_alignTime")
    self.lblAlignTime.set_text("%.1f" % alignTime)

    #Cargo hold
    cargo = fit.ship.getModifiedAttribute("capacity")
    if cargo > 1000:
        cargo = cargo / 1000
        self.lblCargoUnit.set_markup(" k m\302\263")
    else:
        self.lblCargoUnit.set_markup(" m\302\263")

    self.lblCargo.set_text('%.1f' % cargo)

    #Signature radius
    sigRadius = fit.ship.getModifiedAttribute("signatureRadius")
    self.lblSigRadius.set_text('%.0f' % sigRadius)

    #Set the reload time togglebutton correctly
    self.includeReloadTimeToggle.set_active(fit.factorInReload)

    self.fittingChanged = True

def manageRecent(self, item):
    if item.ID in self.recentItems: self.recentItems.remove(item.ID)
    self.recentItems.append(item.ID)
    while len(self.recentItems) > 30: self.recentItems.pop(0)
    
def addItem(self, treeview):
    #Get item being added
    itemStore = treeview.get_model()
    path = treeview.get_cursor()[0]
    if path == -1 or path == None:
        return

    iter = itemStore.get_iter(path)
    it = itemStore.get_value(iter, 3)
    if not isinstance(it, item.item):
        return
    #Get item with all it's info
    it = item.getItem(ID = it.ID)
    #Get current fit
    fit = self.getActiveFit()
    if fit == None: return
    cat = it.group.category.name
    if cat == "Drone":
        fit.addDrone(it)
    elif cat == "Charge":
        #Figure the currently selected module
        moduleStore, pathList = self.tvwModules.get_selection().get_selected_rows()
        for modPath in pathList:
            iter = moduleStore.get_iter(modPath)
            mod = moduleStore.get_value(iter, 9)
            if mod == None: return
            if mod.isValidAmmo(it): mod.setAmmo(it)

    elif cat == "Implant":
        if it.group.name == "Booster": fit.addBooster(it)
        else: fit.addImplant(it)
    elif it.group.name == "Effect Beacon":
        #Directly add to projected
        self.addProjModule()
    else:
        mod = module.module(copy.deepcopy(it), fit)
        if fit.canAddModule(mod):
            mod.setState(fitting.STATE_ACTIVE)
            fit.addModule(mod)

    #Manage recently used
    self.manageRecent(it)
    self.emit("fitting-changed")

def removeItem(self, source):
    #Get item being Removed
    itemStore = self.tvwModules.get_model()

    modPath = self.tvwModules.get_cursor()[0]
    if modPath == -1 or modPath == None: return
    #Get current fit
    fit = self.getActiveFit()
    if fit == None: return
    iter = itemStore.get_iter(modPath)
    i = itemStore.get_value(iter, 12)
    mod = itemStore.get_value(iter, 9)
    if mod.getItem().ID == 0: return
    del fit.modules[i]
    #Manage recently used
    self.manageRecent(mod.getItem())
    self.emit("fitting-changed")

def sortBySlot(self, store, iter1, iter2):
    slot1 = store.get_value(iter1, 10)
    slot2 = store.get_value(iter2, 10)
    if slot1 == 'sepSlot' or slot2 == 'sepSlot':
        return 0
    else:
        if slot1 == None: return 1
        if slot2 == None: return -1
        slotPrio1 = fitting.slotTypes.index(slot1)
        slotPrio2 = fitting.slotTypes.index(slot2)
        if slotPrio1 > slotPrio2:
            return 1
        elif slotPrio1 < slotPrio2:
            return -1
        else:
            index1 = store.get_value(iter1, 12)
            index2 = store.get_value(iter2, 12)
            if index1 == -5 or index2 == -5: return index1 and 1 or -1
            return cmp(index1, index2)

def changeAmmo(self, source, ammo):
    fit = self.getActiveFit()
    if fit == None: return
    store = self.tvwModules.get_model()
    modules = []
    rows = self.tvwModules.get_selection().get_selected_rows()[1]
    for row in rows:
        iter = store.get_iter(row)
        modules.append(store.get_value(iter, 12))

    for index in modules:
        module = fit.modules[index]
        if module.getItem().ID != 0 and module.isValidAmmo(ammo): module.setAmmo(copy.deepcopy(ammo))

    self.emit("fitting-changed")

def tvwModulesClicked(self, widget, event):
    if event.window != widget.get_bin_window(): return
    #if no fit is open, do nothing
    fit = self.getActiveFit()
    if not fit: return True
    clickedRealSlot = False
    clickedRealMod = False
    pathInSelection = False
    clickedObj = widget.get_path_at_pos(int(event.x), int(event.y))
    if clickedObj:
        path, col = clickedObj[0:2]
        #check if separator has been clicked
        store = self.tvwModules.get_model()
        iter = store.get_iter(path)
        if store.get_value(iter,10) != "sepSlot":
            clickedRealSlot = True
    else:
        path, col = None, None
    #if empty area or separator has been clicked - unselect everything
    #we're checking for button press here because clickedObj returns nothing for double and triple clicks when
    #fitting-changed signal has been emitted recently
    if not clickedRealSlot and event.type == gtk.gdk.BUTTON_PRESS:
        widget.get_selection().unselect_all()
        self.selected_rows = []

    event_handlers = [
        #handler for row selection, 2 areas: state column and modules. Items are not reselected when user
        #RMBs modules/state selection (for group context menus and enabling overheat for all selected modules)
        #and when he LMBs state selection (for group state toggling). In all other cases items are reselected,
        #including case when he clicks module in selection - for proper drag'n'drop support
        [
            lambda: event.type == gtk.gdk.BUTTON_PRESS and clickedRealSlot and ((event.button == 1 and \
            col != self.itemStateCol) or (event.button == 1 and  not pathInSelection) or (event.button == 3 and not pathInSelection)),
            lambda i,w,e: widget.do_button_press_event(widget, event) and self.setSelectedRow(path) and False
        ],
        #handler for module deletion, called on double LMB click on module
        [
            lambda: event.type == gtk.gdk._2BUTTON_PRESS and event.button == 1 and col != self.itemStateCol,
            lambda i,w,e: self.removeItem(widget) or True,
        ],
        #handler for state toggling, processed only when user LMBs/RMBs state column of real slot
        [
            lambda: event.type == gtk.gdk.BUTTON_PRESS and clickedRealSlot and col == self.itemStateCol and event.button in [1, 3],
            self.toggleModuleStates,
        ],
        #handler for context menus, called when user RMBs anything outside state column
        [
            lambda: event.type == gtk.gdk.BUTTON_PRESS and event.button == 3 and col != self.itemStateCol,
            self.popupModuleContext,
        ]
    ]

    for handler in event_handlers:
        store, selected_rows = self.tvwModules.get_selection().get_selected_rows()
        pathInSelection      = path in selected_rows
        info                 = [ fit, path, col, store, selected_rows, ]

        if handler[0]():
            if handler[1](info, widget, event):
                break

    return True

def toggleModuleStates(self, info, widget, event):
    fit, path, col, store, selected_rows = info
    #check if state of clicked module is changed
    clickedModChangedState = self._toggleModuleState(event, store, fit, path, None)
    if clickedModChangedState:
        module         = store.get_value(store.get_iter(path), 9)
        proposed_state = module.getState()
        for row in selected_rows:
            #do not change state of the clicked module as we already changed it
            if row != path: self._toggleModuleState(event, store, fit, row, proposed_state)
        self.emit("fitting-changed")
    return clickedModChangedState

def _toggleModuleState(self, event, store, fit, path, proposed_state):
    mod     = store.get_value(store.get_iter(path), 9)
    changed = False
    state   = mod.getState()

    if mod.getItem().ID == 0 or mod.getSlot() in ("rig", "subsystem"): return 0

    if proposed_state:
        state = proposed_state
    elif event.button == 3:
        if "overload" in mod.getItem().getType(): state = fitting.STATE_OVERLOADED
    elif event.button == 1:
        passive_state_map = {
            #  Prev State            : Next State
            fitting.STATE_OFFLINE    : fitting.STATE_ACTIVE,
            fitting.STATE_ACTIVE     : fitting.STATE_OFFLINE,
        }

        active_state_map = {
            #  Prev State            : Next State
            fitting.STATE_OVERLOADED : fitting.STATE_ACTIVE,
            fitting.STATE_ACTIVE     : fitting.STATE_OFFLINE,
            fitting.STATE_INACTIVE   : fitting.STATE_ACTIVE,
            fitting.STATE_OFFLINE    : fitting.STATE_INACTIVE,
        }

        state_map = passive_state_map
        if "active" in mod.getItem().getType():
            state_map = active_state_map
        state = state_map[state]

    if mod.isValidState(state):
        mod.setState(state)
        changed = True

    return changed

def generateAmmoMenu(self, modules, callback):
    ammoTypes = {}
    metaOrder = ("normal", "faction", "deadspace", "officer")
    chargeGroups = set()
    #Cycling through modules to compose list of ammo groups and mark loadable modules
    for mod in modules:
        mod.loadable = False
        it = mod.getItem()
        for i in range(5):
            chargeGroup = it.getModifiedAttribute('chargeGroup%d' % i)
            if chargeGroup != None:
                chargeGroups.add(chargeGroup)
                mod.loadable = True

    for chargeGroup in chargeGroups:
        grp = group.getGroup(ID = chargeGroup)
        items = item.getItemsByGroup(group=grp)
        itemsToAdd = {}
        bases = {}
        for it in items:
            if it.published == 1:
                if it.metaGroupID == 2: base = it
                else: base = item.getBase(it.ID) or it
                skip = False
                for mod in modules:
                    itt = mod.getItem()
                    #Omit modules which cannot load ammo to generate ammo menu even
                    #if non-ammo module is selected alongside with ammo container
                    if mod.loadable == True:
                        t = (itt, base)
                        if not t in bases:
                            bases[t] = not mod.isValidAmmo(base)
                        skip = bases[t]
                        if skip: break

                if skip: continue
                if it.metaGroupID == 4: meta = "faction"
                elif it.metaGroupID == 5: meta = "officer"
                elif it.metaGroupID == 6: meta = "deadspace"
                else: meta = "normal"
                if meta not in itemsToAdd: itemsToAdd[meta] = []
                if it not in itemsToAdd[meta]: itemsToAdd[meta].append(it)

        if len(itemsToAdd) > 0:
            ammoTypes[grp.name] = itemsToAdd

    if len(ammoTypes) > 0:
        menu = gtk.Menu()
        for grpName, metas in ammoTypes.iteritems():
            if grp.icon: pixbuf = pixbufLoader.getPixbuf(grp.icon, True, "16x16")
            else: pixbuf = None
            box = gtk.HBox()
            if pixbuf: box.pack_start(gtk.image_new_from_pixbuf(pixbuf), False)
            lbl = gtk.Label(grpName)
            lbl.set_alignment(0, 0.5)
            box.pack_start(lbl, False)
            grpMenu = gtk.Menu()
            grpMenuItem = gtk.MenuItem()
            grpMenuItem.add(box)
            grpMenuItem.set_submenu(grpMenu)
            menu.append(grpMenuItem)
            for meta in metaOrder:
                if meta in metas:
                    items = metas[meta]
                    if len(metas) > 1:
                        submenu = gtk.Menu()
                        menuItem = gtk.MenuItem()
                        lbl = gtk.Label(meta.capitalize())
                        lbl.set_alignment(0, 0.5)
                        menuItem.add(lbl)
                        menuItem.set_submenu(submenu)
                        grpMenu.append(menuItem)
                    else: submenu = grpMenu
                    items.sort(cmp = lambda one, two: cmp(one.name, two.name))
                    for it in items:
                        if it.icon: pixbuf = pixbufLoader.getPixbuf(it.icon, True, "16x16")
                        else: pixbuf = None
                        box = gtk.HBox()
                        if pixbuf: box.pack_start(gtk.image_new_from_pixbuf(pixbuf), False)
                        lbl = gtk.Label(it.name)
                        lbl.set_alignment(0, 0.5)
                        box.pack_start(lbl, False)
                        itMI = gtk.MenuItem()
                        if callback: itMI.connect("activate", callback, it)
                        itMI.add(box)
                        submenu.append(itMI)
        
        lbl = gtk.Label("None")
        lbl.set_alignment(0, 0.5)
        mi = gtk.MenuItem()
        mi.add(lbl)
        mi.connect("activate", callback, None)
        menu.append(mi)
    else: menu = None
    return menu

def fittedModuleStats(self, *stuff):
    store, selected_rows = self.tvwModules.get_selection().get_selected_rows()
    for row in selected_rows:
        iter = store.get_iter(row)
        module = store.get_value(iter, 9)
        item, ammo = module.getItem(), module.getAmmo()
        #Do not consider empty slots
        if item.ID != 0:
            itemStatsWindow(item, self.getActiveFit(), ammo=ammo)
            break

def fittedModuleGroup(self, *stuff):
    store, selected_rows = self.tvwModules.get_selection().get_selected_rows()
    modules = []
    for row in selected_rows:
        iter = store.get_iter(row)
        module = store.get_value(iter, 9)
        item = module.getItem()
        if item.ID != 0: modules.append(item)
    self.expandToItem(modules)

def fittedModuleFavorite(self, *stuff):
    store, selected_rows = self.tvwModules.get_selection().get_selected_rows()
    for row in selected_rows:
        iter = store.get_iter(row)
        module = store.get_value(iter, 9)
        item = module.getItem()
        if item.ID != 0:
            self.favoriteItems.add(item.ID)
            #Refresh favs list if it's opened
            self.groupSelected(self.tvwGroups)
            break

def fittedAmmoStats(self, *stuff):
    store, selected_rows = self.tvwModules.get_selection().get_selected_rows()
    for row in selected_rows:
        iter = store.get_iter(row)
        module = store.get_value(iter, 9)
        item, ammo = module.getItem(), module.getAmmo()
        if item.ID != 0 and ammo:
            itemStatsWindow(ammo, self.getActiveFit(), None)
            break

def fittedAmmoGroup(self, *stuff):
    store, selected_rows = self.tvwModules.get_selection().get_selected_rows()
    ammos = []
    for row in selected_rows:
        iter = store.get_iter(row)
        mod = store.get_value(iter, 9)
        item, ammo = mod.getItem(), mod.getAmmo()
        if item.ID != 0 and ammo:
            ammos.append(ammo)
    self.expandToItem(ammos)

def popupModuleContext(self, info, widget, event):

    fit, path, col, store, selected_rows = info
    modTypes = set()
    modMktGrps = set()
    modIds = set()
    ammoTypes = set()
    ammoMktGrps = set()
    modules = []

    for row in selected_rows:
        iter = store.get_iter(row)
        itm = store.get_value(iter, 9)
        module, ammo = itm.getItem(), itm.getAmmo()
        if module.ID != 0:
            modTypes.add((module, ammo))
            modMktGrp = self.findMktGroup(module)
            if modMktGrp: modMktGrps.add(modMktGrp)
            modIds.add(module.ID)
            modules.append(itm)
            if ammo:
                ammoTypes.add(ammo)
                ammoMktGrp = self.findMktGroup(ammo)
                if ammoMktGrp: ammoMktGrps.add(ammoMktGrp)

    for it in self.fittedModuleMenu: self.fittedModuleMenu.remove(it)
    
    modCatName = self.getItemCategoryName(self.getSelectedModule().getItem()) or "Module"
    ammoCatName = self.getItemCategoryName(self.getSelectedModule().getAmmo()) or "Charge"
    shipCatName = self.getItemCategoryName(self.getActiveFit().ship) or "Ship"

    moduleMenuGroup = []
    chargeMenuGroup = []
    shipMenuGroup = []

    if len(modTypes) == 1:
        self.moduleStatsLabel.set_text(modCatName + " Stats")
        moduleMenuGroup.append(self.moduleStatsEntry)
    if len(modMktGrps) == 1:
        self.moduleGroupLabel.set_text(modCatName + " Market Group")
        moduleMenuGroup.append(self.moduleGroupEntry)
    if len(modIds) == 1:
        self.moduleFavoriteLabel.set_text("Add " + modCatName + " to Favorites")
        moduleMenuGroup.append(self.moduleFavoriteEntry)

    m = self.generateAmmoMenu(modules, self.changeAmmo)
    if m:
        self.ammoChangeLabel.set_text(ammoCatName)        
        chargeMenuGroup.append(self.ammoChangeEntry)
        self.ammoChangeEntry.set_submenu(m)

    if len(ammoTypes) == 1:
        self.ammoStatsLabel.set_text(ammoCatName + " Stats")
        chargeMenuGroup.append(self.ammoStatsEntry)
    if len(ammoMktGrps) == 1:
        self.ammoGroupLabel.set_text(ammoCatName + " Market Group")
        chargeMenuGroup.append(self.ammoGroupEntry)

    self.shipStatsLabel.set_text(shipCatName + " Stats")
    shipMenuGroup.append(self.shipStatsEntry)

    for menuItem in moduleMenuGroup:
        self.fittedModuleMenu.append(menuItem)

    if moduleMenuGroup and (chargeMenuGroup or shipMenuGroup):
        self.fittedModuleMenu.append(gtk.SeparatorMenuItem())

    for menuItem in chargeMenuGroup:
        self.fittedModuleMenu.append(menuItem)

    if chargeMenuGroup and shipMenuGroup:
        self.fittedModuleMenu.append(gtk.SeparatorMenuItem())

    for menuItem in shipMenuGroup:
        self.fittedModuleMenu.append(menuItem)

    self.fittedModuleMenu.show_all()
    self.fittedModuleMenu.popup(None, None, None, event.button, event.time)

def tvwModulesKeyPress(self, widget, event):
    fit = self.getActiveFit()

    if event.keyval == 65505 or event.keyval == 65507:
        self.selectionKeyPressed = True
    else:
        self.selectionKeyPressed = False
    
    event_handlers = [
        [ lambda: fit and event.keyval == 65535, self.deleteModules ],
    ]

    for handler in event_handlers:
        if handler[0]() and handler[1](fit, widget, event):
            break
    return True

def deleteModules(self, fit, widget, event):
    store = self.tvwModules.get_model()
    rows = self.tvwModules.get_selection().get_selected_rows()[1]
    modules = []
    
    for row in rows:
        iter = store.get_iter(row)
        modules.append(store.get_value(iter, 12))
        
    modules.sort(reverse=True)
    for i in modules: del fit.modules[i]
    self.emit("fitting-changed")
    
    return True
    
def setSelectedRow(self, path):
    self.selected_rows = []
    if self.selectionKeyPressed:
        tmp = self.tvwModules.get_selection().get_selected_rows()[1]
        for i in range(len(tmp)):
            self.selected_rows.append(tmp[i - 1][0])
    elif path != None:
        self.selected_rows = [path[0]]
    else:
        self.selected_rows = []

def keyReleased(self, widget, event):
    self.selectionKeyPressed = False
