#Copyright 2009 Diego Duclos
#
#This program is free software: you can redistribute it and/or modify
#it under the terms of the GNU General Public License as published by
#the Free Software Foundation, either version 3 of the License, or
#(at your option) any later version.
#
#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License
#along with this program.  If not, see <http://www.gnu.org/licenses/>.
from gui.window import window
from gui import pixbufLoader
from model import attribute, item
import gtk
from gui.pluggable import pluggable

class itemStatsWindow(window):
    def __init__(self, module, fit, forcedIcon = None, ammo = None):
        signalsDict = {'exitRequest' : self.exit,
                       'delete' : self.exit}
        
        window.__init__(self, "itemStatsWindow", signalsDict)
        self.initPluggable()
        pb = forcedIcon or pixbufLoader.getPixbuf(module.icon, True, "16x16")
        if pb: self.hboxWindowHeader.pack_start(gtk.image_new_from_pixbuf(pb), False)
        self.hboxWindowHeader.pack_start(gtk.Label(module.name), False)
        if ammo:
            self.hboxWindowHeader.pack_start(gtk.Label(" - "), False, False)
            pb = pixbufLoader.getPixbuf(ammo.icon, True, "16x16")
            if pb: self.hboxWindowHeader.pack_start(gtk.image_new_from_pixbuf(pb), False)
            self.hboxWindowHeader.pack_start(gtk.Label(ammo.name), False, False)
        
        self.hboxWindowHeader.show_all()
        self.initDescriptionView(module)
        self.initAttributeView(module)
        self.initNamesCheckbox()
        self.initSkillView(module)
        self.initEffectView(module)
        self.initAffectedByView(module, fit)
        self.initAllSections()
        self.nbItemStats.set_current_page(self.nbItemStats.page_num(self.vboxAttributes))
        self.window.show()
    
    def itemClicked(self, widget, event):
        widget.do_button_press_event(widget, event)
        stuff = widget.get_path_at_pos(int(event.x), int(event.y))
        if stuff == None: 
            widget.get_selection().unselect_all()
            return
        if event.type == gtk.gdk._2BUTTON_PRESS and event.button == 1:
            store, selected_rows = widget.get_selection().get_selected_rows()
            if selected_rows:
                selected_row = selected_rows[0]
                if widget.row_expanded(selected_row):
                    widget.collapse_row(selected_row)
                else: widget.expand_row(selected_row, False)

    def initDescriptionView(self, module):
        buffer = self.txwDescription.get_buffer()
        buffer.set_text(module.description)
                
    def initEffectView(self, module):
        self.tvwEffects.connect("button-press-event", self.itemClicked)
        store = gtk.ListStore(str, str, str, str, str, gtk.gdk.Color)
        store.set_sort_column_id(0, gtk.SORT_ASCENDING)
        self.tvwEffects.set_model(store)
        self.tvwEffects.set_tooltip_column(4)
        nameRenderer = gtk.CellRendererText()
        implementationRenderer = gtk.CellRendererText()
        implementationRenderer.set_property("foreground-set", True)
        typeRenderer = gtk.CellRendererText()
        runTimeRenderer = gtk.CellRendererText()
        colName = gtk.TreeViewColumn("Name")
        colImplementation = gtk.TreeViewColumn("Implementation")
        colType = gtk.TreeViewColumn("Type")
        colRunTime = gtk.TreeViewColumn("runTime")
        
        colName.pack_start(nameRenderer, True)
        colImplementation.pack_start(implementationRenderer, False)
        colType.pack_start(typeRenderer, False)
        colRunTime.pack_start(runTimeRenderer, False)
        
        colName.add_attribute(nameRenderer, 'text', 0)
        
        colImplementation.add_attribute(implementationRenderer, 'text', 1)
        colImplementation.add_attribute(implementationRenderer, 'foreground-gdk', 5)
        
        colType.add_attribute(typeRenderer, 'text', 2)
        colRunTime.add_attribute(runTimeRenderer, 'text', 3)
        
        self.tvwEffects.append_column(colName)
        self.tvwEffects.append_column(colImplementation)
        self.tvwEffects.append_column(colType)
        self.tvwEffects.append_column(colRunTime)
        
        for effect in module.effects:
            if effect.func:
                implementation = "Implemented"
                color = gtk.gdk.color_parse('green')
            elif effect.func == None:
                implementation = "Missing"
                color = gtk.gdk.color_parse('yellow')
            else:
                implementation = "Erroneous"
                color = gtk.gdk.color_parse('red')
            
            types = ", ".join(effect.type)
            store.append((effect.name, implementation, types, effect.runTime, effect.description, color))
    
    def initNamesCheckbox(self):
        self.tbnRawNames.connect("toggled", self.updateNames)
    
    def updateNames(self, *stuff):
        raw = self.tbnRawNames.get_active()
        store = self.tvwAttributes.get_model()
        iter = store.get_iter_first()
        while iter:
            for i in range(store.iter_n_children(iter)):
                child = store.iter_nth_child(iter, i)
                attr = store.get_value(child, 5)
                store.set_value(child, 2,  raw and attr.name or attr.displayName or attr.name)
                
            iter = store.iter_next(iter)
            
    def initAttributeView(self, module):
        self.tvwAttributes.connect("button-press-event", self.itemClicked)
        store = gtk.TreeStore(gtk.gdk.Pixbuf, str, str, str, str, object)
        store.set_sort_column_id(1, gtk.SORT_ASCENDING)
        self.tvwAttributes.set_model(store)
        self.tvwAttributes.set_tooltip_column(4)
        iconRenderer = gtk.CellRendererPixbuf()
        nameRenderer = gtk.CellRendererText()
        valueRenderer = gtk.CellRendererText()
        colName = gtk.TreeViewColumn("Name")
        colValue = gtk.TreeViewColumn("Value")
        
        colName.pack_start(iconRenderer, False)
        colName.pack_start(nameRenderer, True)
        colValue.pack_start(valueRenderer, False)
        colName.add_attribute(iconRenderer, 'pixbuf', 0)
        colName.add_attribute(nameRenderer, 'text', 2)
        colValue.add_attribute(valueRenderer, 'text', 3)
        
        self.tvwAttributes.append_column(colName)
        self.tvwAttributes.append_column(colValue)
        
        categories = {}
        for attrName, attr in module.attributes.iteritems():
            if attr.name == attrName:
                if isinstance(attr, attribute.attribute):
                    if attr.categoryName == "NULL": catName = "Miscellaneous"
                    else: catName = attr.categoryName or "Miscellaneous"
                else:
                    catName = "Internal"
                    
                if not catName in categories:
                    categories[catName] = []
                    
                categories[catName].append(attr)
                
        for catName, catList in categories.iteritems():
            iter = store.append(None, (None, catName, catName, None, None, None))
            for attr in catList:
                val = attr.getModifiedValue()
                if isinstance(val, int):
                    val = "%.d" % val
                elif isinstance(val, float): 
                    val = "%.5f" % val
                    i = len(val) -1
                    while i >= 0:
                        if val[i] == "0": val = val[:(i)]
                        else: break
                        i -= 1
                    if val[-1:] == ".": val = val[:-1]
                elif isinstance(val, item.item):
                    val = "item: %s" % val.name
                        
                if isinstance(attr, attribute.attribute) and attr.icon:
                    icon = pixbufLoader.getPixbuf(attr.icon, True, "16x16")
                    desc = attr.description
                else:
                    icon = None
                    desc = None
                store.append(iter, (icon, attr.name, attr.displayName or attr.name, val, desc or None, attr))
                
    def initSkillView(self, module):
        self.tvwSkills.connect("button-press-event", self.itemClicked)
        self.skillPixbuf = pixbufLoader.getPixbuf("skill")
        store = gtk.TreeStore(gtk.gdk.Pixbuf, str, str, str)
        store.set_sort_column_id(1, gtk.SORT_ASCENDING)
        self.tvwSkills.set_model(store)
        self.tvwSkills.set_tooltip_column(3)
        iconRenderer = gtk.CellRendererPixbuf()
        nameRenderer = gtk.CellRendererText()
        valueRenderer = gtk.CellRendererText()
        colName = gtk.TreeViewColumn("Skill")
        colValue = gtk.TreeViewColumn("Level")
        
        colName.pack_start(iconRenderer, False)
        colName.pack_start(nameRenderer, True)
        colValue.pack_start(valueRenderer, False)
        colName.add_attribute(iconRenderer, 'pixbuf', 0)
        colName.add_attribute(nameRenderer, 'text', 1)
        colValue.add_attribute(valueRenderer, 'text', 2)
        
        self.tvwSkills.append_column(colName)
        self.tvwSkills.append_column(colValue)
        
        self.addRequiredSkills(module, store)
        self.tvwSkills.expand_all()
    
    def addRequiredSkills(self, module, store, iter = None):
        for skill, level in module.requiredSkills.iteritems():
            newIter = store.append(iter, (self.skillPixbuf, skill.name, '%d' % level.getModifiedValue(), skill.description))
            if skill.requiredSkills:
                self.addRequiredSkills(skill, store, newIter)
                
    def initAffectedByView(self, module, fit):
        #Image, name, (level), (description)
        self.tvwAffectedBy.connect("button-press-event", self.itemClicked)
        store = gtk.TreeStore(gtk.gdk.Pixbuf, str, str, str)
        self.tvwAffectedBy.set_model(store)
        self.tvwAffectedBy.set_tooltip_column(3)
        
        iconRenderer = gtk.CellRendererPixbuf()
        nameRenderer = gtk.CellRendererText()
        levelRenderer = gtk.CellRendererText()
        
        colName = gtk.TreeViewColumn("Name")
        colLevel = gtk.TreeViewColumn("Level")
        
        colName.pack_start(iconRenderer, False)
        colName.pack_start(nameRenderer, True)
        colLevel.pack_start(levelRenderer, False)
        
        colName.add_attribute(iconRenderer, 'pixbuf', 0)
        colName.add_attribute(nameRenderer, 'text', 1)
        colLevel.add_attribute(levelRenderer, 'text', 2)
        
        self.tvwAffectedBy.append_column(colName)
        self.tvwAffectedBy.append_column(colLevel)
        skillIter = store.append(None, (pixbufLoader.getPixbuf("skill"), "Skills", None, None))
        moduleIter = store.append(None, (pixbufLoader.getPixbuf("modules"), "Modules", None, None))
        implantIter = store.append(None, (pixbufLoader.getPixbuf("hardwiring"), "Implants", None, None))
        subIter = store.append(None, (pixbufLoader.getPixbuf("subsystems"), "Subsystems", None, None))
        alreadyIn = set()
        for name, attribute in module.attributes.iteritems():
            #Only consider attributes that haven't been copied over from other items
            #If an attribute comes from another item, we simply consider that item a source too
            if attribute.item == module or attribute.item == None:
                for source in attribute.sources:
                    self.processSource(source, alreadyIn, store, skillIter, moduleIter, implantIter, subIter, fit)
            else:
                self.processSource(attribute.item, alreadyIn, store, skillIter, moduleIter, implantIter, subIter, fit)
                
        #Check for empty groups & nuke them out of orbit
        for iter in (skillIter, moduleIter, implantIter, subIter):
            if store.iter_children(iter) == None: store.remove(iter)
            
        #Expand everything
        self.tvwAffectedBy.expand_all()
        
    def processSource(self, source, alreadyIn, store, skillIter, moduleIter, implantIter, subIter, fit):
        if not source in alreadyIn:
            alreadyIn.add(source)
            if source.group.category.name != "Skill":
                pb = pixbufLoader.getPixbuf(source.icon, True, "16x16")
                level = None
            if source.group.category.name == "Module": iter = moduleIter
            elif source.group.category.name == "Skill":
                iter = skillIter
                skill, level = fit.getCharSkill(source.name)
                level = "%d" % level
                pb = pixbufLoader.getPixbuf("skill")
            elif source.group.category.name == "Ship":
                iter = None
                pb = pixbufLoader.getPixbuf("ships")
            elif source.group.category.name == "Implant": iter = implantIter
            elif source.group.category.name == "Subsystem": iter = subIter
            else: iter = None
            store.append(iter, (pb, source.name, level, source.description))
    
    def exit(self, *stuff):
        self.window.destroy()
        return False           
    
itemStatsWindow = pluggable(itemStatsWindow) 
