# Copyright (C) 2008-2010 Association of Universities for Research in Astronomy (AURA)

# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:

#     1. Redistributions of source code must retain the above copyright
#       notice, this list of conditions and the following disclaimer.

#     2. Redistributions in binary form must reproduce the above
#       copyright notice, this list of conditions and the following
#       disclaimer in the documentation and/or other materials provided
#       with the distribution.

#     3. The name of AURA and its representatives may not be used to
#       endorse or promote products derived from this software without
#       specific prior written permission.

# THIS SOFTWARE IS PROVIDED BY AURA ``AS IS'' AND ANY EXPRESS OR IMPLIED
# WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL AURA BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
# BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
# OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
# ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
# TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
# USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
# DAMAGE.

from __future__ import with_statement

# STDLIB
import optparse
import os
import sys

# VO
from vo import util
from vo.validator import html
from vo.validator import result

def get_urls(dest="results"):
    types = ['good', 'broken', 'incorrect']

    seen = set()
    urls = []
    for type in types:
        with open('urls/cone.%s.dat' % type, 'rb') as fd:
            for url in fd.readlines():
                url = url.strip()
                if url not in seen:
                    with result.Result(url, root=dest) as r:
                        r['expected'] = type
                    urls.append(url)
                seen.add(url)

    return urls

def download(url):
    with result.Result(url, root=dest) as r:
        r.download_xml_content()

def validate(url):
    with result.Result(url, root=dest) as r:
        r.validate_vo()

def votlint_validate(args):
    path_to_stilts_jar, url = args
    with result.Result(url, root=dest) as r:
        if r['network_error'] is None and 'votlint' not in r:
            r.validate_with_votlint(path_to_stilts_jar)

def write_html_result(url):
    with result.Result(url, root=dest) as r:
        html.write_result(r)

def get_write_subindex(total):
    def write_subindex(subset):
        html.write_index_table(dest, *subset, total=total)
    return write_subindex
        
if __name__ == '__main__':
    parser = optparse.OptionParser()
    parser.add_option("-j", action='store_true',
                      help="Run validation in parallel")
    parser.add_option("-d", "--dest", default="results",
                      help="Destination directory")
    parser.add_option("--stilts",
                      help="Path to stilts.jar, required to run votlint")
    (options, args) = parser.parse_args()

    dest = options.dest

    if options.stilts:
        if not os.path.exists(options.stilts):
            print("'%s' does not exist.  It should be the path to stilts.jar")
            sys.exit(1)

    if len(args):
        urls = list(args)
        for url in urls:
            with result.Result(url, root=dest) as r:
                r['expected'] = 'good'
    else:
        util.color_print('green', 'Loading set of URLs')
        urls = get_urls(dest)

    util.color_print('green', 'Downloading VO files')
    util.map_with_progress(download, urls, multiprocess=options.j)

    util.color_print('green', 'Validating VO files')
    util.map_with_progress(validate, urls, multiprocess=options.j)

    if options.stilts:
        util.color_print('green', 'Validating with votlint')
        args = [(options.stilts, x) for x in urls]
        util.map_with_progress(votlint_validate, args, multiprocess=options.j)

    util.color_print('green', 'Generating HTML files')
    util.map_with_progress(write_html_result, urls, multiprocess=options.j)

    util.color_print('green', 'Grouping results')
    subsets = result.get_result_subsets(urls, dest)
    
    util.color_print('green', 'Generating index')
    html.write_index(subsets, urls, dest)

    util.color_print('green', 'Generating subindices')
    util.map_with_progress(
        get_write_subindex(len(urls)), subsets,
        multiprocess=False)
