#!/usr/bin/env python
from __future__ import division, absolute_import, print_function

import os, sys, re

# compiler & distutils setup
# --------------------------

import setuptools

import subprocess
from numpy.distutils.misc_util import Configuration
from distutils.util import get_platform

if not hasattr(sys, 'version_info') or (
        sys.version_info[0] == 2 and sys.version_info < (2,7,0) or
        sys.version_info[0] >= 3 and sys.version_info < (3,3,0)):
    raise SystemExit("Python version >=2.7 or >=3.3 required to build this module.")

# usadel1
# -------

def configuration(parent_package='', top_path=None):
    ext_sources = """
    src/usadel1/_solvercore.pyf

    src/third-party/blas/daxpy.f
    src/third-party/blas/dcopy.f
    src/third-party/blas/ddot.f
    src/third-party/blas/dscal.f
    src/third-party/blas/dswap.f
    src/third-party/blas/idamax.f
    src/third-party/linpack/dgefa.f
    src/third-party/linpack/dgesl.f
    src/third-party/slatec/dintrv.f
    src/third-party/slatec/d1mach.f
    src/third-party/bvp/colnew.f90
    src/third-party/bvp/twpbvpc.f90

    src/usadel1/lazy_alloc.f90
    src/usadel1/miscmath.f90
    src/usadel1/interpolate.f90
    src/usadel1/params.f90
    src/usadel1/error.f90
    src/usadel1/kin_equations.f90
    src/usadel1/kin_solve.f90
    src/usadel1/kin_solve2.f90
    src/usadel1/kin_solve3.f90
    src/usadel1/sp_equations.f90
    src/usadel1/sp_solve.f90
    src/usadel1/sp_solve2.f90
    
    src/usadel1/solvercore.f90
    """.split()

    version = "0.2.dev"
    version = generate_version(version, os.path.dirname(__file__))

    config = Configuration('usadel1', parent_package, top_path,
                           package_path='src/usadel1',
                           install_requires=["six >= 1.5",
                                             "numpy >= 1.9",
                                             "scipy >= 0.9"],
                           version=version,
                           )
    config.add_extension('_solvercore', sources=ext_sources)
    config.add_subpackage('hdf5pickle', 'src/hdf5pickle')
    config.add_data_dir('tests')

    return config


basedir = os.path.abspath(os.path.dirname(__file__))


def get_git_hash():
    """
    Get version from asv/__init__.py and generate asv/_version.py
    """
    # Obtain git revision
    githash = ""
    if os.path.isdir(os.path.join(basedir, '.git')):
        try:
            proc = subprocess.Popen(
                ['git', '-C', basedir, 'rev-parse', 'HEAD'],
                stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            rev, err = proc.communicate()
            if proc.returncode == 0:
                githash = rev.strip().decode('ascii')
        except OSError:
            pass
    return githash


def get_git_revision():
    """
    Get the number of revisions since the last tag.
    """
    revision = "0"
    if os.path.isdir(os.path.join(basedir, '.git')):
        try:
            proc = subprocess.Popen(
                ['git', '-C', basedir, 'rev-list', '--count', 'HEAD'],
                stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            rev, err = proc.communicate()
            if proc.returncode == 0:
                revision = rev.strip().decode('ascii')
        except OSError:
            pass
    return revision



def write_version_file(filename, version, revision):
    # Write revision file (only if it needs to be changed)
    content = '''
__version__ = "{0}"
__githash__ = "{1}"
__release__ = {2}
    '''.format(version, revision, 'dev' not in version)

    old_content = None
    if os.path.isfile(filename):
        with open(filename, 'r') as f:
            old_content = f.read()

        if 'dev' in version and not revision.strip():
            # Dev version and Git revision not available. Probably
            # running from an sdist, so assume the version file is up
            # to date.
            m = re.search(r'__version__ = "([0-9a-z+.-]*)"', old_content)
            if m:
                old_version = m.group(1)
                prefix = version[:version.find('dev')]
                if old_version.startswith(prefix):
                    version = old_version
                    content = old_content

    if content != old_content:
        with open(filename, 'w') as f:
            f.write(content)

    return version

def generate_version(version, top_path):
    version_file = os.path.join(top_path, 'src', 'usadel1', 'version.py')

    git_hash = get_git_hash()

    release = 'dev' not in version

    if not release:
        version = '{0}{1}+{2}'.format(
            version, get_git_revision(), git_hash[:8])

    return write_version_file(version_file, version, git_hash)


# setup
# -----

if __name__ == "__main__":
    from numpy.distutils.core import setup
    cfg = configuration(top_path='').todict()
    setup(**cfg)
