"""	TimeFormat
	--------------------------------------------------------------------
	Copyright (c) 2004 Colin Stewart (http://www.owlfish.com/)
	All rights reserved.
	
	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:
	1. Redistributions of source code must retain the above copyright
	   notice, this list of conditions and the following disclaimer.
	2. Redistributions in binary form must reproduce the above copyright
	   notice, this list of conditions and the following disclaimer in the
	   documentation and/or other materials provided with the distribution.
	3. The name of the author may not be used to endorse or promote products
	   derived from this software without specific prior written permission.
	
	THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
	IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
	OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
	IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
	INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
	NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
	DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
	THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
	THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
"""

import re, time, locale, string, math, os

__version__ = "1.0.0"

# Regex for our date/time format strings.  Format is: %TYPE[\[MODIFIER\]]
# %% is used to escape the %
regex = re.compile ('(%[%abcCdHIjmMnpPrStTuUwWxXyYZz])(\[[^\]]*\])?')
strftime_regex = re.compile ('((?<!%)%[mlnhkjedacbyxzutwprMIHDFACBYXZUTWPSR])')

# Used to index into locale
DAY_WEEK_LONG = [locale.DAY_2, locale.DAY_3, locale.DAY_4, locale.DAY_5, locale.DAY_6, locale.DAY_7, locale.DAY_1]
DAY_WEEK_SHORT = [locale.ABDAY_2, locale.ABDAY_3, locale.ABDAY_4, locale.ABDAY_5, locale.ABDAY_6, locale.ABDAY_7, locale.ABDAY_1]

MONTH_LONG = [locale.MON_1, locale.MON_2, locale.MON_3, locale.MON_4, locale.MON_5, locale.MON_6, locale.MON_7, locale.MON_8
				,locale.MON_9, locale.MON_10, locale.MON_11, locale.MON_12]
MONTH_SHORT = [locale.ABMON_1, locale.ABMON_2, locale.ABMON_3, locale.ABMON_4, locale.ABMON_5, locale.ABMON_6, locale.ABMON_7
				, locale.ABMON_8,locale.ABMON_9, locale.ABMON_10, locale.ABMON_11, locale.ABMON_12]
					
# Default names if locale function nl_langinfo doesn't work
DEFAULT_DAY_WEEK = {locale.DAY_2: 'Monday', locale.DAY_3: 'Tuesday', locale.DAY_4: 'Wednesday', locale.DAY_5: 'Thursday'
					,locale.DAY_6: 'Friday', locale.DAY_7: 'Saturday', locale.DAY_1: 'Sunday'
					,locale.ABDAY_2: 'Mon', locale.ABDAY_3: 'Tue', locale.ABDAY_4: 'Wed', locale.ABDAY_5: 'Thu'
					,locale.ABDAY_6: 'Fri', locale.ABDAY_7: 'Sat', locale.ABDAY_1: 'Sun'}
DEFAULT_MONTH = {locale.MON_1: 'January', locale.MON_2: 'February', locale.MON_3: 'March', locale.MON_4: 'April', locale.MON_5: 'May'
				,locale.MON_6: 'June', locale.MON_7: 'July', locale.MON_8: 'August', locale.MON_9: 'September', locale.MON_10: 'October'
				,locale.MON_11: 'November', locale.MON_12: 'December'
				,locale.ABMON_1: 'Jan', locale.ABMON_2: 'Feb', locale.ABMON_3: 'Mar', locale.ABMON_4: 'Apr', locale.ABMON_5: 'May'
				,locale.ABMON_6: 'Jun', locale.ABMON_7: 'Jul',locale.ABMON_8: 'Aug', locale.ABMON_9: 'Sep', locale.ABMON_10: 'Oct'
				,locale.ABMON_11: 'Nov', locale.ABMON_12: 'Dec'}
					
# Use this if no locale format is available.  Note this is in stfrtime format not TimeFormat
DEFAULT_T_FMT_AMPM = '%I:%M:%S %p'
DEFAULT_D_T_FMT = "%a %b %e %H:%M:%S %Y"
DEFAULT_D_FMT = "%d/%m/%Y"
DEFAULT_T_FMT = "%H:%M:%S"
					
# Used to convert strftime format into TimeFormat format.
STRFMAP = 	{'%a': '%a[SHORT]', '%A': '%a[LONG]', '%b': '%b[SHORT]', '%B': '%b[LONG]', '%c': '%c', '%C': '%C', '%d': '%d[0]'
			,'%D': '%m[0]/%d[0]/%y[0]', '%e': '%d[SP]', '%F': '%Y-%m[0]-%d[0]', '%h': '%b[SHORT]', '%H': '%H[0]', '%I': '%I[0]'
			,'%j': '%j[0]', '%k': '%H[SP]', '%l': '%I[SP]', '%m': '%m[0]', '%M': '%M[0]', '%n': '%n', '%p': '%p', '%P': '%P'
			,'%r': '%r', '%R': '%H[0]:%M[0]', '%S': '%S[0]', '%t': '%t', '%T': '%H[0]:%M[0]:%S[0]', '%u': '%u', '%U': '%U'
			,'%w': '%w', '%W': '%W[0]', '%x': '%x', '%X': '%X', '%y': '%y[0]', '%Y': '%Y', '%z': '%T', '%Z': '%Z'}
				
# NOT IMPLEMENTED: %E, %G, %g, %O, %s, %V

def format (informat, intime = None, utctime = 0):
	""" Python implementation of an alternative strftime format with more options.
		For %c, %p, %P, %U, %x, %X, %W we fall back to the 'C' implementation due to limitations 
		in Python's locale support.
		
		Set utctime to 1 if the time is in UTC rather than local.
	"""
	if (intime is None):
		ourTime = time.localtime()
	else:
		ourTime = intime
		
	resultBuf = []
	
	position = 0
	last = 0
	match = regex.search (informat)
	while (match):
		resultBuf.append (informat [last:match.start()])
		formatType, formatModifier = match.groups()
		if (formatType == '%a'):
			# Weekday in locale
			resultBuf.append (_getWeekday_ (ourTime[6], formatModifier))
		elif (formatType == '%b'):
			# Month in locale
			resultBuf.append (_getMonth_ (ourTime[1], formatModifier))
		elif (formatType == '%d'):
			# Month as a number
			resultBuf.append (_getNumber_ (ourTime[2], formatModifier, '[0]'))
		elif (formatType == '%H'):
			# Hour in 24 hour format 
			resultBuf.append (_getNumber_ (ourTime[3], formatModifier, '[0]'))
		elif (formatType == '%I'):
			# Hour in 12 hour  format
			hour = ourTime [3]
			if (hour == 0):
				resultBuf.append ("12")
			elif (hour > 12):
				resultBuf.append (_getNumber_ (hour - 12, formatModifier, '[0]'))
			else:
				resultBuf.append (_getNumber_ (hour, formatModifier, '[0]'))
		elif (formatType == '%j'):
			# Day of year as a number
			resultBuf.append (_getNumber_ (ourTime[7], formatModifier, '[0]', 3))
		elif (formatType == '%m'):
			# Month of year as a number
			resultBuf.append (_getNumber_ (ourTime[1], formatModifier, '[0]'))
		elif (formatType == '%M'):
			# Minute as a number
			resultBuf.append (_getNumber_ (ourTime[4], formatModifier, '[0]'))
		elif (formatType == '%n'):
			resultBuf.append (os.linesep)
		elif (formatType == '%t'):
			resultBuf.append ('\t')
		elif (formatType == '%r'):
			try:
				ampmFmt = locale.nl_langinfo (locale.T_FMT_AMPM)
			except:
				ampmFmt = DEFAULT_T_FMT_AMPM
			# Now get a translation and expansion of this.
			resultBuf.append (strftime (ampmFmt, ourTime))
		elif (formatType == '%c'):
			try:
				prefFmt = locale.nl_langinfo (locale.D_T_FMT)
			except:
				prefFmt = DEFAULT_D_T_FMT
			# Now get a translation and expansion of this.
			resultBuf.append (strftime (prefFmt, ourTime))
		elif (formatType == '%x'):
			try:
				prefFmt = locale.nl_langinfo (locale.D_FMT)
			except:
				print "WARNING Having to fall back to default code!"
				prefFmt = DEFAULT_D_FMT
			# Now get a translation and expansion of this.
			resultBuf.append (strftime (prefFmt, ourTime))
		elif (formatType == '%X'):
			try:
				prefFmt = locale.nl_langinfo (locale.T_FMT)
			except:
				prefFmt = DEFAULT_T_FMT
			# Now get a translation and expansion of this.
			resultBuf.append (strftime (prefFmt, ourTime))
		elif (formatType == '%S'):
			# Second  as a number
			resultBuf.append (_getNumber_ (ourTime[5], formatModifier, '[0]'))
		elif (formatType == '%w'):
			# Day of week as number, Sunday = 0
			weekDayNum = ourTime [6] + 1
			if (weekDayNum == 7):
				weekDayNum = 0
			resultBuf.append (str (weekDayNum))
		elif (formatType == '%u'):
			# Day of week as number, Monday = 1
			weekDayNum = ourTime [6] + 1
			resultBuf.append (str (weekDayNum))
		elif (formatType == '%y'):
			# 2 digit year
			year = int (str (ourTime [0])[2:])
			resultBuf.append (_getNumber_ (year, formatModifier, '[0]'))
		elif (formatType == '%Y'):
			# 4 digit year
			resultBuf.append (str (ourTime [0]))
		elif (formatType == '%C'):
			# 2 digit century
			resultBuf.append (str (ourTime [0])[0:2])
		elif (formatType == '%p' or formatType == '%P'):
			# Have to fall back on the 'C' version
			resultBuf.append (time.strftime (formatType, ourTime))
		elif (formatType == '%U' or formatType == '%W'):
			# Fall back to 'C' version, but still allow the extra modifier.
			resultBuf.append (_getNumber_ (int (time.strftime (formatType, ourTime)), formatModifier, '[0]'))
		elif (formatType == '%z'):
			# W3C Timezone format
			resultBuf.append (_getTimeZone_ (w3cFormat = 1, utctime = utctime))
		elif (formatType == '%T'):
			# hhmm timezone format
			resultBuf.append (_getTimeZone_ (w3cFormat = 0, utctime = utctime))
		elif (formatType == '%Z'):
			# TLA timezone format
			if (utctime):
				resultBuf.append ('UTC')
			elif (ourTime [8] == 1):
				resultBuf.append (time.tzname[1])
			else:
				resultBuf.append (time.tzname[0])
		elif (formatType == '%%'):
			resultBuf.append ('%')
		else:
			# Silently ignore the error - print as litteral
			if (formatModifier is None):
				resultBuf.append ("%s" % formatType)
			else:
				resultBuf.append ("%s%s" % (formatType, formatModifier))
		last = match.end()
		match = regex.search (informat, last)
	resultBuf.append (informat [last:])
		
	return u"".join (resultBuf)
	
def strftime (informat, intime = None):
	""" Provides a backwards-compatible strftime implementation.
		This converts strftime format codes into TimeFormat codes, and then expands them using format()
	"""
	resultBuf = []
	
	position = 0
	last = 0
	match = strftime_regex.search (informat)
	while (match):
		resultBuf.append (informat [last:match.start()])
		formatType = match.group(1)
		if (STRFMAP.has_key (formatType)):
			resultBuf.append (STRFMAP [formatType])
		else:
			# Silently ignore the error - print as litteral
			resultBuf.append ("%s" % formatType)
		last = match.end()
		match = strftime_regex.search (informat, last)
	resultBuf.append (informat [last:])
	
	# Now expand the TimeFormat string
	return format (u"".join (resultBuf), intime)


def _getWeekday_ (dayOfWeek, formatModifier):
	constantList = DAY_WEEK_LONG
	
	if (formatModifier == '[SHORT]'):
		constantList = DAY_WEEK_SHORT
	
	localeConst = constantList [dayOfWeek]
	
	try:
		weekDay = locale.nl_langinfo (localeConst)
		return weekDay
	except:
		# nl_langinfo not supported
		return DEFAULT_DAY_WEEK [localeConst]
		
def _getMonth_ (monthNum, formatModifier):
	constantList = MONTH_LONG
	
	if (formatModifier == '[SHORT]'):
		constantList = MONTH_SHORT
	
	# Months are 1-12 not 0-11
	localeConst = constantList [monthNum-1]
	
	try:
		monthName = locale.nl_langinfo (localeConst)
		return monthName
	except:
		# nl_langinfo not supported
		return DEFAULT_MONTH [localeConst]
		
def _getNumber_ (theNumber, formatModifier, defaultModifier, cols=2):
	" Returns a positive digit number either as-is, or padded"
	if (formatModifier is None):
		formatModifier = defaultModifier
	
	# By default do no padding
	padding = 0
	
	if (formatModifier == '[NP]'):
		padding = 0
	elif (formatModifier == '[SP]'):
		padding = 1
		padder = " "
	elif (formatModifier == '[0]'):
		padding = 1
		padder = "0"
		
	if (padding == 0):
		return str (theNumber)
	
	ourNum = str (theNumber)
	
	return "%s%s" % (padder * (cols - len (ourNum)), ourNum)
	
def _getTimeZone_ (w3cFormat, utctime):
	if (utctime):
		if (w3cFormat):
			return "Z"
		return "-0000"
	
	# Work out the timezone in +/-HH:MM format.
	if (time.daylight):
		offset = time.altzone
	else:
		offset = time.timezone
	hours = int (math.floor (offset/3600.0))
	mins = int (math.floor ((offset - (hours * 3600))/60.0))
	if (offset > 0):
		thesign = "-"
	else:
		thesging = "+"
	if (w3cFormat):
		return "%s%s:%s" % (thesign, string.zfill (hours,2), string.zfill (mins, 2))
	else:
		return "%s%s%s" % (thesign, string.zfill (hours,2), string.zfill (mins, 2))