"""\
Test cases for ConfigSection
Copyright (C) 2005 Remy Blank

This file is part of SvnReporter.

SvnReporter is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation, version 2.

SvnReporter is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with SvnReporter; if not, write to the Free Software Foundation,
Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
"""

from cStringIO import StringIO
import os
import sys
import unittest

import SvnReporter

sys.path.append(os.path.dirname(__file__))        # To find Imports.py


class DummySource:
    """Dummy data source"""
    revision = 10

    def getRepositoryData(self):
        return SvnReporter.Struct(
            youngest=10,
            uuid="0000-1234-etc"
        )
        
    def getRevisionData(self, revision):
        project = revision % 2
        return SvnReporter.Struct(
            number=revision,
            date=revision,
            summary="Summary for <b>rev=%d</b>" % revision,
            changes={
                "Projects/Project %d/File 1.c" % project: "U ",
                "Projects/Project %d/File 2.h" % project: "U ",
                "Projects/Project %d/File 3.py" % project: "U "
            }
        )


class ConfigSectionTest(unittest.TestCase):
    def setUp(self):
        SvnReporter._open = self.open
        
    def tearDown(self):
        SvnReporter._open = open

    def open(self, fileName, mode="r", bufSize=-1):
        return StringIO({
            "Atom.xml": """\
<?xml version="1.0" encoding="utf-8"?>
<feed xmlns="http://www.w3.org/2005/Atom">
  <id>urn:uuid:%(repository.uuid)s</id>
<?foreach revision?>
  <entry>
    <title>%(entryTitle)s</title>
    <id>tag:%(repository.uuid)s:%(revision.number)d</id>
    <updated>%(inetTime(revision.date))s</updated>
    <summary>
%(escape(revision.summary))s
    </summary>
  </entry>
<?end revision?>
</feed>
""",
            "Mail.elm": """\
From: svn@here
Subject: %(entryTitle)s
Date: %(inetTime(revision.date))s
X-Repos-Uuid: %(repository.uuid)s

%(revision.summary)s
"""
        }[fileName])

    def testCompleteConfig(self):
        """Reading of a complete configuration"""
        config = SvnReporter.Config(StringIO("""\
[DEFAULT]
module = Imports.py
generators =
    Feed("Atom.xml", "dummy destination", 2),
    Mail("Mail.elm", "dummy server", "dummy from", '''%(mailTo)s''')
entryTitle = Revision %(revision.number)d

[All]
match = .*
mailTo = admin@nospam.com

[Project 1]
match = Projects/%(__name__)s/.*
mailTo =
    user1@nospam.com
    user2@nospam.com
"""))
        self.assertEqual("""\
<?xml version="1.0" encoding="utf-8"?>
<feed xmlns="http://www.w3.org/2005/Atom">
  <id>urn:uuid:0000-1234-etc</id>

  <entry>
    <title>Revision 10</title>
    <id>tag:0000-1234-etc:10</id>
    <updated>1970-01-01T00:00:10Z</updated>
    <summary>
Summary for &lt;b&gt;rev=10&lt;/b&gt;
    </summary>
  </entry>

  <entry>
    <title>Revision 9</title>
    <id>tag:0000-1234-etc:9</id>
    <updated>1970-01-01T00:00:09Z</updated>
    <summary>
Summary for &lt;b&gt;rev=9&lt;/b&gt;
    </summary>
  </entry>

</feed>
""", config.sections["All"].toString(DummySource(), 0))
        self.assertEqual("""\
<?xml version="1.0" encoding="utf-8"?>
<feed xmlns="http://www.w3.org/2005/Atom">
  <id>urn:uuid:0000-1234-etc</id>

  <entry>
    <title>Revision 9</title>
    <id>tag:0000-1234-etc:9</id>
    <updated>1970-01-01T00:00:09Z</updated>
    <summary>
Summary for &lt;b&gt;rev=9&lt;/b&gt;
    </summary>
  </entry>

  <entry>
    <title>Revision 7</title>
    <id>tag:0000-1234-etc:7</id>
    <updated>1970-01-01T00:00:07Z</updated>
    <summary>
Summary for &lt;b&gt;rev=7&lt;/b&gt;
    </summary>
  </entry>

</feed>
""", config.sections["Project 1"].toString(DummySource(), 0))
        self.assertEqual("""\
From: svn@here
Subject: Revision 10
Date: 1970-01-01T00:00:10Z
X-Repos-Uuid: 0000-1234-etc

Summary for <b>rev=10</b>
""", config.sections["All"].toString(DummySource(), 1))
